/*
 * key.c
 *   Handle lots of key[board] related stuff - read keys, bind keys, run
 *     keys.
 */

/* Copyright 1992 Craig Durland
 *   Distributed under the terms of the GNU General Public License.
 *   Distributed "as is", without warranties of any kind, but comments,
 *     suggestions and bug reports are welcome.
 */

/* Notes:
 *   The global keymap always exists (it is created at init time).  Never
 *     NULL (might be the null keymap though).
 *   The null keymap always exists and is always empty.  Be careful not to
 *     let the bind routines put anything in it.
 *   The local_keymap can be NULL, the null_keymap or a real keymap.  If it
 *     is NULL, local_keys points to the null_keymap.  These variables are
 *     kept in sync by the buffer change routines.  I don't want to always
 *     install the null_keymap as the local keymap (at buffer create time)
 *     because I wouldn't know when to install a new local keymap (if
 *     (bind-key LOCAL-KEYMAP ...)  is called without creating/installing a
 *     keymap).
 */

#include "me2.h"
#include "bind.h"

extern int pgm_flag, pgm_prefix;			/* in mmaux.c */

#define GLOBAL_KEYMAP	0
#define LOCAL_KEYMAP	1
#define NULL_KEYMAP	2

#define FIRST_REAL_ID	3

typedef struct KeyMap
{
  struct KeyMap *next;
  int16 id;
  KeyTable keytable;
/*  PKey prefix_keys[PKEYS];	/*  */
  int immortal;
} KeyMap;

/* ******************************************************************** */
/* *************************** Prefix Keys **************************** */
/* ******************************************************************** */

EKeyCode pkeys[PKEYS] =
{
  SHIFT,	/* Meta : undefined */
  SHIFT,	/* undefined */
  SHIFT,	/* undefined */
  SHIFT,	/* undefined */
};

/* ******************************************************************** */
/* ******************************* Keys ******************************* */
/* ******************************************************************** */

static KeyTable *global_keys, *local_keys;

/* Read in a key.
 * Do the standard keyboard preprocessing.
 * Convert the keys to the internal character set.
 */
EKeyCode getkey() { return Eget_key(pkeys); }

	/* convert a key string to keycode */
EKeyCode to_keycode(ptr,prefixable) char *ptr;
{ return Eto_keycode(pkeys,ptr,prefixable); }

void xpandkey(buf,keycode) char *buf; EKeyCode keycode;
{ Expand_key(pkeys,keycode,buf); }

UKey *lookupkey(keycode) EKeyCode keycode;	/* lookup user key */
{
  UKey *key;

  if ((key = (UKey *)Efind_key(local_keys, keycode))	||
      (key = (UKey *)Efind_key(global_keys,keycode)))
    return key;

  return NULL;
}

    /* Run whatever is bound to a key.
     * Input:
     *   keycode : The key to run
     *   f : The C-u flag.
     *   n : The C-u count.
     *   status : exit status of the command just run (TRUE, FALSE).
     * Returns:
     *   TRUE  : keycode is bound to something.
     *   FALSE : keycode is not bound.
     */
int run_key(keycode,f,n,status) EKeyCode keycode; int f,n,*status;
{
  UKey *key;

  if ((key = lookupkey(keycode)))
  {
    pgm_flag = f; pgm_prefix = n;
    *status = MMrun_pgm((int)key->index);

    return TRUE;
  }

  return FALSE;		/* ain't nothing I know about */
}

    /* Bind a key in a keymap.
     * Input:
     *   Keymap:  Pointer to a KeyMap
     *   key_name: Name of the key to bind (something like "C-xa" or "z").
     *   pgm_name:  Name of the program to bind to the key (something like
     *     "show-buffer-stats").
     * Returns:
     *   TRUE: Everything went as expected.
     *   FALSE: Unknown program name or out of memory.
     */
static int bind_key(keymap, key_name, pgm_name)
  KeyMap *keymap; char *key_name, *pgm_name;
{
  int index;
  EKeyCode keycode;
  KeyTable *kt;
  UKey *key;

  kt = &keymap->keytable;

  keycode = Eto_keycode(pkeys,key_name,TRUE);

  if (*pgm_name == '\0') { Eunbind_key(kt,keycode); return TRUE; }

  if (-1 == (index = MMpgm_lookup(pgm_name)))
  {
    mlwrite("bind-key: \"%s\" not found.", pgm_name);
    return FALSE;
  }

  if (!(key = (UKey *)(Ekalloc(kt,keycode))))	/* !!!???bitch */
	{ mlwrite("No memory for key table!"); return FALSE; }

  key->keycode = keycode; key->type = 0; key->index = index;

  return TRUE;
}

/* ******************************************************************** */
/* ******************** Gory Details ********************************** */
/* ******************************************************************** */

/* Keymaps are kept in a big linked list.  Keymaps that are not currently
 *   being used are kept in a free pool on the assumption that this will
 *   reduce malloc trashing since keymaps are used and reused often (mostly
 *   because they are part of a buffer).  An attempt is made to further
 *   reduct heap fragmentation by malloc()ing lots of keymaps when some are
 *   needed.
 */

/* Each and every keymap has a unique id.  Because it makes the code easier
 *   (I can recycle keymaps without worrying about ids clashing).
 */

static KeyMap *live_list = NULL, *free_list = NULL;
static int id_seed = FIRST_REAL_ID;

static KeyMap
  *global_keymap = NULL, *local_keymap = NULL, *null_keymap = NULL;

static KeyMap *alloc_keymap();

static KeyMap *id_to_keymap(keymap_id, error_msg)
  int keymap_id; char *error_msg;
{
  switch(keymap_id)
  {
    case GLOBAL_KEYMAP: return global_keymap;
    case LOCAL_KEYMAP:	return local_keymap;
    case NULL_KEYMAP:   return null_keymap;
  }

  {
    register KeyMap *keymap;

    for (keymap = live_list; keymap; keymap = keymap->next)
      if (keymap->id == keymap_id) return keymap;
  }

  if (error_msg) MMbitch(error_msg);

  return NULL;
}

#define MBLOCK    10  /* Number of things to allocate per malloc() */
#define MAX_ID 16384  /* A max (< 32K) to keep ids from wrapping around */

   /* Allocate a keymap.
    * Returns:
    *   Pointer to the allocated keymap.
    *   NULL if can't malloc().
    * Returns:
    *   Pointer to the created keymap.
    *   NULL if no memory or created all the keymaps I'm gonna create and
    *     ain't gonna create no more.
    */
static KeyMap *alloc_keymap(immortal) int immortal;
{
  int j;
  KeyMap *keymap;

  if (free_list)		/* stuff in the free pool */
  {
    keymap = free_list;
    free_list = free_list->next;

    keymap->next = live_list;
    live_list = keymap;

    keymap->immortal = immortal;

    return keymap;
  }

  if (id_seed > MAX_ID) return NULL;	/* I hope this never happens! */

		/* malloc a bunch of objects */
  if (!(keymap = (KeyMap *)malloc(MBLOCK*sizeof(KeyMap))))
	return NULL;
  for (j = MBLOCK; j--; keymap++)
  {
    keymap->id = id_seed++;
    keymap->next = free_list;
    free_list = keymap;

    initialize_dTable(&keymap->keytable,sizeof(UKey));
  }

      /* Since I know there are objects in the free pool */
  return alloc_keymap(immortal);
}

static void clear_keymap(keymap) KeyMap *keymap;
{
  clear_keytable(&keymap->keytable);
}

    /* Notes:
     *   If keymap is a immortal keymap, don't mess with it - this is called
     *     when buffers are freed, in that case, only mortal keymaps need be
     *     freed.
     */
static void free_keymap(keymap) KeyMap *keymap;
{
  if (!keymap) return;
  if (keymap->immortal) return;

  clear_keymap(keymap);		/* !!!??? */

/* !!! what if keymap is in use???  ie Mutt var pointing to it */

/*  free_dTable(&bp->buf_keys.keytable); !!!??? */

  if (live_list == keymap) live_list = keymap->next;
  else
  {
    KeyMap *ptr, *qtr;

    for (ptr = qtr = live_list; ptr; qtr = ptr, ptr = ptr->next)
      if (ptr == keymap)
      {
	qtr->next = keymap->next;
	break;
      }
   }

  keymap->next = free_list;
  free_list = keymap;
}

/* ******************************************************************** */
/* ************************ Buffer Interaction ************************ */
/* ******************************************************************** */

    /* Initialize the keymaps in a newly created buffer.  This must be
     *   called when a buffer is created.
     * Input:
     *   bp : pointer to the buffer that has just been created.
     * Returns:
     *   TRUE:   Everything OK
     *   FALSE:  Probably out of memory but maybe there are too many
     *		 keymaps.
     */
int init_buffer_keys(bp) Buffer *bp;
{
  bp->buf_keys = NULL;

  return TRUE;
}

void free_buffer_keys(bp) Buffer *bp;
{
  free_keymap(bp->buf_keys);
}

    /* Called from use_buffer()
     */
void keys_use_buffer()
{
  local_keymap = (KeyMap *)curbp->buf_keys;
  if (local_keymap)
       local_keys = &local_keymap->keytable;
  else local_keys =  &null_keymap->keytable;
}

static int new_local_keymap()
{
  if (!(curbp->buf_keys = alloc_keymap(FALSE))) return FALSE;
  keys_use_buffer();

  return TRUE;
}

    /* Notes:
     *   If installing the Null keymap, don't actually install it.  This
     *     way, if a program gets the keymap id (NULL if no installed
     *     keymap), installs a keymap and then restores the old keymap, if a
     *     (bind-key) is done, a local keymap will be created instead of a
     *     error (trying to change the Null keymap).
     *   free_keymap() won't change an immortal keymap.
     */
static void install_local_keymap(keymap) KeyMap *keymap;
{
  free_keymap(local_keymap);
  if (keymap == null_keymap) keymap = NULL;
  curbp->buf_keys = keymap;
  keys_use_buffer();
}

/* ******************************************************************** */
/* ************************* Geneneral Stuff ************************** */
/* ******************************************************************** */

   /* If fails: system probably to low on mem to do anything */
init_keys()
{
  if (!(global_keymap = alloc_keymap(TRUE))	||
      !(null_keymap   = alloc_keymap(TRUE))) return FALSE;

  global_keys = &global_keymap->keytable;

  return pad_dTable(global_keys,150);
}

get_keymap_data(id, data, num_keys) UKey **data; int *num_keys;
{
  KeyMap *keymap;

  if (keymap = id_to_keymap(id, (char *)NULL))
  {
    *data = (UKey *)keymap->keytable.table;
    *num_keys = sizeof_dTable(&keymap->keytable);
    return TRUE;
  }
  *num_keys = 0;
  return FALSE;
}

/* ******************************************************************** */
/* ************************** Mutt Interface ************************** */
/* ******************************************************************** */

#include "mm.h"
extern MMDatum RV, TV;				/* in mm.c */

    /* (bind-key keymap-id pgm-name key-name ...)
     *  if pgm-name if "", the key is unbound.
     */
void Mbind_key()
{
  char *pgm_name, *key_name;
  int n, kid;
  KeyMap *keymap;

  get1arg(NUMBER,"bind-key");
  kid = RV.val.num;

  if (kid == NULL_KEYMAP)
	MMbitch("bind-key:  Can't change the Null keymap.");

  if (kid == LOCAL_KEYMAP && !local_keymap && !new_local_keymap())
	MMbitch("bind-key:  Can't alloc keymap.");

  keymap = id_to_keymap(kid, "bind-key:  Bad keymap id.");
  if (keymap == null_keymap)
	MMbitch("bind-key:  Null keymap installed, can't change it.");

  n = 1;
  while (maybearg(n++,STRING,"bind-key"))
  {
    pgm_name = TV.val.str;
    if (maybearg(n++,STRING,"bind-key")) key_name = TV.val.str;
    else MMbitch("bind-key: args must be in pairs!");

    if (!bind_key(keymap, key_name, pgm_name)) MMabort_pgm(0);
  }

  RV.type = VOID;
}

    /* (clear-keymap keymap-id)
     */
void Mclear_keymap()
{
  get1arg(NUMBER,"clear-keymap");
  clear_keymap(
	id_to_keymap((int)RV.val.num, "clear-keymap:  Bad keymap id."));

  RV.type = VOID;
}

    /* (install-keymap keymap-id [where])
     *   where:  global (0), local (1)
     */
void Minstall_keymap()
{
  int keymap_id;
  KeyMap *keymap;

  get1arg(NUMBER,"install-keymap");
  keymap_id = RV.val.num;

  keymap = id_to_keymap(keymap_id, (char *)NULL);

  if (maybearg(1,NUMBER,"install-keymap"))
  {
    if (!keymap) MMbitch("install-keymap:  Bad keymap id.");
    switch(TV.val.num)
    {
      case GLOBAL_KEYMAP:
	global_keymap = keymap;
	global_keys = &keymap->keytable;
	break;
      case LOCAL_KEYMAP:
        install_local_keymap(keymap);
	break;
      default:
	MMbitch("install-keymap:  Bad where.");	/* !!!??? */
    }
  }

  RV.val.num = (keymap ? keymap->id : -1);
  RV.type = NUMBER;
}

    /* (create-keymap [id of keymap to duplicate])
     */
void Mcreate_keymap()
{
  KeyMap *keymap;

  if (!(keymap = alloc_keymap(TRUE))) MMbitch("create-keymap: No memory!");

  if (maybearg(0,NUMBER,"create-keymap"))
  {
    KeyTable *new, *dup;
    int n;

    new = &keymap->keytable;
    dup = &id_to_keymap((int)TV.val.num,
		"create-keymap:  Bad duplicate id")->keytable;
    n = sizeof_dTable(dup);

	/* Memory leak that should never happen */
    if (!xpand_dTable(new, n, 0,10)) MMbitch("create-keymap: No memory!");
    memcpy((char *)new->table, (char *)dup->table, n*sizeof(UKey));
  }

  RV.type = NUMBER;
  RV.val.num = keymap->id;
}

/* ******************************************************************** */
/* ***************************** GC keys ****************************** */
/* ******************************************************************** */

static int dead_pgm_key(key) UKey *key;
	{ return pgmdead((int)key->index); }

	/* Get rid of keys bound to dead pgms.
	 * Assume this doesn't happen very often.
	 */
void pack_keys()
{
  KeyMap *keymap;

  for (keymap = live_list; keymap; keymap = keymap->next)
	Epack_keytable(&keymap->keytable,dead_pgm_key);
}
