/*
 * 	FILE.C MODULE
 *
 * The routines in this file handle the reading and writing of disk files.
 *   All of details about the reading and writing of the disk itself are in
 *   "fileio.c".
 */

/* Copyright 1990, 1991, 1992 Craig Durland
 *   Distributed under the terms of the GNU General Public License.
 *   Distributed "as is", without warranties of any kind, but comments,
 *     suggestions and bug reports are welcome.
 */

#include <const.h>
#include "me2.h"
#include "config.h"

extern char *strcpy();
extern int buffer_is_read_only;		/* in buffer.c */

/*
 * Write the contents of a buffer to a file.
 * Input:
 *   fname : Name of file to write to.
 *   n : pointer to int - number of lines written returned here.
 *   error_code : pointer to int - if FALSE returned, the error code is put
 *     here.  If TRUE, error_code = FIOSUC.
 * Returns:
 *   TRUE if all OK, error_code = FIOSUC.
 *   else FALSE and error_code is set.
 * Notes:
 *   Wildcards are expanded.
 *   Buffer changed flag is cleared.
 */
write_file(bp, fname, n, error_code)
  Buffer *bp; char *fname; int *n, *error_code;
{
  int num_lines, s, s1;
  register Line *lp;

  if (bp->b_flags & BFBAD_READ)
  {
    *error_code = FIOERR;	/* !!! should be fio bad read */
    return FALSE;
  }

#if FILENAME_COMPLETION
  {
    char tmp[NFILEN];
    if (fxpand(fname,TRUE,FALSE,TRUE,tmp,(pfi)NULL))
      { *error_code = FIOBADNAME; return FALSE; }
    fname = tmp;		/* !!!??? is this legal */
  }
#endif

  if ((s = ffwopen(fname)) != FIOSUC)		/* Open writes err message */
    { *error_code = s; return FALSE; }

  num_lines = 0; s = FIOSUC;
  
  for (lp = BUFFER_FIRST_LINE(bp); lp != BUFFER_LAST_LINE(bp);
       lp = lforw(lp))
  {
    if ((s = ffputline(lp->l_text,llength(lp))) != FIOSUC) break;
    num_lines++;
  }
  *n = num_lines;

  set_buffer_modified(bp,FALSE);
/* !!! (error or no error) and close fails s==? */

  if ((s1 = ffclose()) == FIOSUC && s != FIOERR)	/* All OK */
  {
    *error_code = FIOSUC;
    return TRUE;
  }

  *error_code = s;	/* !!!??? or s1??? see below */
  return FALSE;		/* Some sort of error */
}  

/*
 * Insert a file into the current buffer at the dot.
 * Input:
 *   fname : Name of file to insert.
 *   n : pointer to int - number of lines inserted returned here.
 *   error_code : pointer to int - if FALSE returned, the error code is put
 *     here.  If TRUE, error_code = FIOSUC.
 * Returns:
 *   TRUE if all OK, error_code = FIOSUC.
 *   else FALSE and error_code is set.
 * Notes:
 *   Wildcards are expanded.
 *   Don't really need to check buffer_is_read_only (because the line break
 *     routines will fail) but it is nice not to have to worry about "what
 *     ifs".
 *   It is assumed the inserted file ends with a newline, even if it
 *     doesn't.  This may be a bug.
 */
insert_file(fname, n, error_code) char *fname; int *n, *error_code;
{
  char line[FLINE], buf[NFILEN];
  int nbytes, num_lines, s, s1, undo_state;
  int32 total_bytes;
  register Line *lp1, *lp2, *anchorline;

  *n = 0;

  if (buffer_is_read_only)
  {
    *error_code = FIOREAD_ONLY;
    return FALSE;
  }

  if (!arf(fname, buf))		/* bogus file name syntax */
  {
    *error_code = FIOBADNAME;
    return FALSE;
  }

  if ((s = ffropen(buf)) != FIOSUC)
  {
    *error_code = s;
    return FALSE;
  }

  undo_state = do_undo;
  do_undo = FALSE;

  if (!lnewline())		/* split the current line at the cursor */
  {
    *error_code = FIONOMEM;
    do_undo = undo_state;
    return FALSE;
  }
  next_character(-1);			/* leave dot on first line */
  lp2 = the_dot->line;   /* line containing dot. Insert new line after this */
  anchorline = lp2->l_next;  /* Second half of split line. This never moves */
  num_lines = total_bytes = 0;
  while (FIOSUC == (s = ffgetline(line,FLINE)))
  {
    nbytes = strlen(line);
    if (NULL == (lp1 = alloc_line(nbytes,FALSE)))
    {		/* Keep message on the display */
      s = FIONOMEM;	/* ???redundant */
      set_buffer_flags(curbp,
	       (int32)(curbp->b_flags | BFBAD_READ | BFREAD_ONLY));
      total_bytes++;		/* for that lnewline() */
      break;
    }
    lp1->l_prev = lp2; lp2->l_next = lp1;	/* link in new line */
    memcpy((lp2 = lp1)->l_text, line, nbytes);

    total_bytes += (nbytes + 1);		/* remember that newline */
    num_lines++;
  }
  lp2->l_next = anchorline; anchorline->l_prev = lp2;   /* make the last link */
  if (!ldelete((int32)1))	/* rejoin split line to inserted line */
	s = FIONOMEM;

  *n = num_lines;

  s1 = ffclose();
  *error_code = s = (s == FIOEOF) ? s1 : s;

  do_undo = undo_state;
  save_for_undo(BU_INSERT,total_bytes);

	/* leave the dot after the insert */
  the_dot->line = anchorline; the_dot->offset = 0;

  return (s == FIOSUC);
}

   /* Massage a filename:  expand wildcards and canonize.
    * Massaged filename is put on top of fname.
    * Input:
    *   fname : File name to expand.
    *   cname : Pointer to area to put new name.
    * Output:
    *   cname
    * Returns:
    *   TRUE : All went as expected.
    *   FALSE : Bad file name syntax.
    */
arf(fname, cname) char *fname, *cname;
{
  extern char
    *current_directory(),	/* in os.c */
    *canonize();

#if FILENAME_COMPLETION

  char tmp[NFILEN];

  if (fxpand(fname,TRUE,FALSE,TRUE,tmp,(pfi)NULL)) return FALSE;
  return (canonize(tmp,cname, current_directory(TRUE)) != NULL);

#else

  return (canonize(fname,cname, current_directory(TRUE)) != NULL);

#endif
}

/* ******************************************************************** */
/* ******************************************************************** */
/* ******************************************************************** */

/*
 * This command allows the user to modify the file name associated with a
 *   buffer.  It is like the "f" command in UNIX "ed".  The operation is
 *   simple; just zap the name in the Buffer structure and mark the windows
 *   as needing an update.
 * Input:
 *   bp : Buffer to change.
 *   fname :  New name of file attached to buffer.  fname == "" is OK.
 * Returns:
 *   TRUE : All went OK.
 *   FALSE : Invalid file name.
 */
new_file_name(bp,fname) Buffer *bp; char *fname;
{
  char new_name[NFILEN];

  new_name[0] = '\0';
  if (*fname && !arf(fname,new_name)) return FALSE;
  set_dString(&bp->b_fname,new_name);
  fixWB(bp,WFMODE);		/* Update mode lines */

  return TRUE;
}

/*
 * Fabricate a buffer name from a file name.
 * On most systems, this is just <name.ext>.
 * Input:
 *   fname : File name to build buffer name from.
 *   bname : Pointer to string to hold new buffer name.
 *   n : Size of bname.
 * Output:
 *   bname holds buffer name.
 * Notes:
 *   fname should be canonized.
 */
/* Move to os.c!!! */
void create_buffer_name(fname, bname,n) char *fname, *bname; int n;
{
  extern char *nanex();

  register char *ptr;

	/* remember to leave room for the '\0' in bname */

  ptr = nanex(fname);
  while (--n && (*bname++ = *ptr++)) ;
  *bname = '\0';
}

/*
 * Check to see if a file exists.  Useful for Mutt programming.
 */
file_exists(fname) char *fname;
{
  char file_name[NFILEN];

	/* if not a valid name or can't open it, it don't exist */
  if (!arf(fname,file_name) || ffropen(file_name) != FIOSUC) return FALSE;
  ffclose();

  return TRUE;
}
