/*
 * Xkmap.c: X11 key mapping
 * C Durland	Public Domain	??/92, 1/94
 */

static char what[] = "@(#)X11 Keymap";

#include <ed.h>
#include "term.h"
#include <char.h>
#include <const.h>

#include <X11/X.h>
#define XK_MISCELLANY
#include <X11/keysymdef.h>

    /* Notes:
     *   Some ME internals (like shell_command()) look at keyboard input and
     *     expect to see 'r' (0x0D) for return, not C-M.  This is pretty
     *     easy to fix but not worth it because of the next problem.
     *   Mutt programs also expect (get-key) to return 0x0D for return
     *     (because it has always done so).  Don't want to break existing
     *     Mutt programs.  Plus, most programers think of \r as 0x0D.
     *   X allows bunches of modifiers on any key.  I want to return:
     *     Return	  : 0x0D
     *     Control Return : C-M
     *     Meta Return    : M-C-M, not M-^M
     *   Hence, some funky look ups.
     *   I wish I had thought about this more long ago.
     */

typedef struct { KeySym keysym; EKeyCode keycode; } KeyMap;

KeyMap control_key_map[] =
{
  XK_BackSpace,	0x08,		/* CTRL|'H', 0x08 */
  XK_Tab,	0x09,		/* CTRL|'I', 0x09 */
  XK_Linefeed,	0x0A,		/* CTRL|'J', 0x0A */
  XK_Return,	0x0D,		/* CTRL|'M', 0x0D */
  XK_Escape,	0x1B,		/* CTRL|'[', 0x1B */

  NoSymbol,	0,		/* End of map */
};

KeyMap key_map[] =
{
  XK_BackSpace,	CTRL|'H',	/* CTRL|'H', 0x08 */
  XK_Tab,	CTRL|'I',	/* CTRL|'I', 0x09 */
  XK_Linefeed,	CTRL|'J',	/* CTRL|'J', 0x0A */
  XK_Return,	CTRL|'M',	/* CTRL|'M', 0x0D */
  XK_Escape,	CTRL|'[',	/* CTRL|'[', 0x1B */

  XK_Delete,	SOFKEY|'L',	/* CTRL|'?', 0x7F */
  XK_Clear,	SOFKEY|'G',

  XK_Home,	SOFKEY|'A',
  XK_Left,	SOFKEY|'F',	/* Move left, left arrow */
  XK_Up,	SOFKEY|'C',	/* Move up, up arrow */
  XK_Right,	SOFKEY|'E',	/* Move right, right arrow */
  XK_Down,	SOFKEY|'D',	/* Move down, down arrow */
  XK_Prior,	SOFKEY|'I',	/* Prior, previous <> */
  XK_Next,	SOFKEY|'J',	/* Next <> */
  XK_End,	SOFKEY|'B',	/* EoL */
  XK_Begin,	SOFKEY|'A',	/* BoL */

  XK_Select,	'\0',		/* Select, mark */
  XK_Insert,	SOFKEY|'G',	/* Insert, insert here */
  XK_Cancel,	0x7,		/* Cancel, stop, abort, exit */

#ifdef XK_ClearLine	/* HP keysyms */
  XK_ClearLine,  SOFKEY|'K',
  XK_InsertLine, SOFKEY|'M',
  XK_DeleteLine, SOFKEY|'L',
  XK_InsertChar, SOFKEY|'G',
  XK_DeleteChar, SOFKEY|'H',
  XK_BackTab,	 META|'I',
#endif	/* XK_ClearLine */

  XK_F1,	SOFKEY|'1',
  XK_F2,	SOFKEY|'2',
  XK_F3,	SOFKEY|'3',
  XK_F4,	SOFKEY|'4',
  XK_F5,	SOFKEY|'5',
  XK_F6,	SOFKEY|'6',
  XK_F7,	SOFKEY|'7',
  XK_F8,	SOFKEY|'8',
  XK_F9,	SOFKEY|'9',
  XK_F10,	SOFKEY|'0',

  XK_KP_Space,		' ',		/* Number pad space */
  XK_KP_Tab,		0x9,
  XK_KP_Enter,		0xD,		/* Number pad enter */
  XK_KP_Equal,		'=',		/* Number pad equals */
  XK_KP_Multiply,	'*',
  XK_KP_Add,		'+',
  XK_KP_Separator,	',',		/* Number pad separator, often comma */
  XK_KP_Subtract,	'-',
  XK_KP_Decimal,	'.',
  XK_KP_Divide,		'/',

  XK_KP_0,		'0',		/* Number pad 0 */
  XK_KP_1,		'1',		/* Number pad 1 */
  XK_KP_2,		'2',		/* Number pad 2 */
  XK_KP_3,		'3',		/* Number pad 3 */
  XK_KP_4,		'4',		/* Number pad 4 */
  XK_KP_5,		'5',		/* Number pad 5 */
  XK_KP_6,		'6',		/* Number pad 6 */
  XK_KP_7,		'7',		/* Number pad 7 */
  XK_KP_8,		'8',		/* Number pad 8 */
  XK_KP_9,		'9',		/* Number pad 9 */

#ifdef XK_KP_Home	/* Newer X11R5 KeySyms */
  XK_KP_Home,		SOFKEY|'{',	/* Number pad Home */
  XK_KP_End,		SOFKEY|'}',	/* Number pad End */
  XK_KP_Page_Up,	SOFKEY|'[',	/* Number pad Page Up */
  XK_KP_Page_Down,	SOFKEY|']',	/* Number pad Page Down */
  XK_KP_Up,		SOFKEY|'^',	/* Number pad Up Arrow */
  XK_KP_Down,		SOFKEY|'.',	/* Number pad Down Arrow */
  XK_KP_Right,		SOFKEY|'>',	/* Number pad Right Arrow */
  XK_KP_Left,		SOFKEY|'<',	/* Number pad Left Arrow */
  XK_KP_Insert,		SOFKEY|'&',	/* Number pad Insert */
  XK_KP_Delete,		SOFKEY|'#',	/* Number pad Delete */
#endif

  XK_KP_Enter,		SOFKEY|'$',	/* Number pad Enter */
  XK_KP_Multiply,	SOFKEY|'*',	/* Number pad *  */
  XK_KP_Add,		SOFKEY|'+',	/* Number pad +  */
  XK_KP_Subtract,	SOFKEY|'-',	/* Number pad -  */
  XK_KP_Divide,		SOFKEY|'/',	/* Number pad /  */

  NoSymbol,	0,		/* End of map */
};

unsigned int Xmask_to_MEmask(Xmask) unsigned int Xmask;
{
  unsigned int MEmask = 0;

  if (Xmask & Mod1Mask)    MEmask |= META;
  if (Xmask & ControlMask) MEmask |= CTRL;
  if (Xmask & ShiftMask)   MEmask |= SHIFT;

  return MEmask;
}


static int lookup_keysym(keymap, keysym, modifiers, special, keycode)
  KeyMap *keymap;
  KeySym keysym;
  unsigned int modifiers;
  int special;
  EKeyCode *keycode;
{
  if (special && modifiers) return FALSE;

  for (; NoSymbol != keymap->keysym; keymap++)
    if (keysym == keymap->keysym)			/* found it */
    {
      *keycode = (keymap->keycode | Xmask_to_MEmask(modifiers));
      return TRUE;
    }
  return FALSE;
}

    /* Convert X11 keysyms to ME codes */
map_keysym(modifiers,keysym,keycode)
  unsigned int modifiers;
  KeySym keysym;
  EKeyCode *keycode;
{
  EKeyCode kc = 0;

  if (keysym & 0xFF00)	/* non ASCII */
  {
    if (
	!lookup_keysym(control_key_map, keysym, modifiers, TRUE, &kc)	&&
	!lookup_keysym(key_map, keysym, modifiers, FALSE, &kc))
      return FALSE;
  }
  else		/* might be ASCII */
  {
    kc = keysym;
    if (modifiers & (ShiftMask | LockMask | ControlMask | Mod1Mask))
    {
      kc = TOUPPER(kc);
      if (modifiers & ControlMask)
        if (modifiers & Mod1Mask) kc |= CTRL; else kc ^= 0x40;
      if (modifiers & Mod1Mask) kc |= META;
    }
  }
  *keycode = kc;
  return TRUE;
}
