/* -----------------------------------------------------------------
                             SPRITE.CPP
     This module is for the encoding, storing, and drawing of sprites
  to and from canvas objects.

  Author: Michael T. Duffy
  Date: August 6th, 1995

  Copyright 1995 Michael T. Duffy. All Rights Reserved.

   ----------------------------------------------------------------- */

/* Useage Notes:

   * Coordinates for sprites are given as if the origin (0,0) were in the
     top left hand corner of the canvas

*/

// ***********************************************************************
//  Compile Directives
// ***********************************************************************

#pragma pack (1);

// ***********************************************************************
//  Header Files
// ***********************************************************************

// Note: The __IBMC__ define is needed to get the Warp Toolkit to work with
//  Watcom compilers.
#define __IBMC__

// Included for OS/2 type definitions
#include <os2.h>

#include <stdlib.h>
#include <string.h>

#include "newtypes.hpp"
#include "canvas.hpp"
#include "sprite.hpp"

#include "errcodes.hpp"




// ***********************************************************************
//  Code
// ***********************************************************************

//........................................................................
SpriteManager::SpriteManager
//........................................................................
  (
  VOID
  )
{
ULONG              ulIndex;


// clear all variables to 0
pcnvEncode  = NULL;
pcnvDisplay = NULL;

for (ulIndex = 0; ulIndex < MAX_NUMBER_BOOKS; ++ulIndex)
  {
  asprdData [ulIndex] = NULL;
  ausCount  [ulIndex] = 0;
  };
};


//........................................................................
SpriteManager::~SpriteManager
//........................................................................
  (
  VOID
  )
{
};

//........................................................................
ERRORSTATUS SpriteManager::FindEmptySlot
//........................................................................
  (
  BYTE           byBookNumberIn,
  PUSHORT        pusSpriteNumberOut
  )
{
USHORT           usIndex;


// make sure book exists
if (asprdData [byBookNumberIn] == NULL)
  {
  usLastErrorCode = ERR_SPR_BAD_BOOK;
  return (ES_ERROR);
  };

// check each sprite position in the book
for (usIndex = 0; usIndex < ausCount [byBookNumberIn]; ++usIndex)
  {
  // check to see if this position is empty
  if (asprdData [byBookNumberIn] [usIndex].usFlags & SPR_FLG_EMPTY)
    {
    // Found a slot that is empty.  Return it.

    *pusSpriteNumberOut = usIndex;
    return (ES_NO_ERROR);
    };
  };

// no slots open
usLastErrorCode = ERR_SPR_NO_SLOTS;
return (ES_ERROR);
};

//........................................................................
ERRORSTATUS SpriteManager::FindEmptyBook
//........................................................................
  (
  PBYTE            pbyBookNumberOut
  )
{
USHORT             usIndex;


// check each book space in the library to find an empty spot
for (usIndex = 0; usIndex < MAX_NUMBER_BOOKS; ++usIndex)
  {
  if (asprdData [usIndex] == NULL)
    {
    // found an empty spot
    *pbyBookNumberOut = (BYTE) usIndex;
    return (ES_NO_ERROR);
    };
  };
return (ES_ERROR);
};

//........................................................................
ERRORSTATUS SpriteManager::SetSprite
//........................................................................
  (
  BYTE           byBookNumberIn,
  USHORT         usSpriteNumberIn,
  PBYTE          pbyDataIn,
  ULONG          ulDataSizeIn
  )
{
PSPRITEDATA      psprdDataIn;


// remove compiler warning
ulDataSizeIn = ulDataSizeIn;

// make sure book exists
if (asprdData [byBookNumberIn] == NULL)
  {
  usLastErrorCode = ERR_SPR_BAD_BOOK;
  return (ES_ERROR);
  };

// make sure position exists
if (usSpriteNumberIn >= ausCount [byBookNumberIn])
  {
  usLastErrorCode = ERR_SPR_BAD_POSITION;
  return (ES_ERROR);
  };

// In the future, the following code will store the actual encoded data
//  of the sprite.  For now, we are drawing sprites directly from the encode
//  canvas, so we need to store the sprite's position in that canvas.

psprdDataIn = (PSPRITEDATA) pbyDataIn;

asprdData [byBookNumberIn] [usSpriteNumberIn].sXPos   = psprdDataIn->sXPos;
asprdData [byBookNumberIn] [usSpriteNumberIn].sYPos   = psprdDataIn->sYPos;
asprdData [byBookNumberIn] [usSpriteNumberIn].sWidth  = psprdDataIn->sWidth;
asprdData [byBookNumberIn] [usSpriteNumberIn].sHeight = psprdDataIn->sHeight;

// clear the empty flag
asprdData [byBookNumberIn] [usSpriteNumberIn].usFlags = 0;

return (ES_NO_ERROR);
};

//........................................................................
ERRORSTATUS SpriteManager::EraseSprite
//........................................................................
  (
  BYTE           byBookNumberIn,
  USHORT         usSpriteNumberIn
  )
{


// make sure book exists
if (asprdData [byBookNumberIn] == NULL)
  {
  usLastErrorCode = ERR_SPR_BAD_BOOK;
  return (ES_ERROR);
  };

// make sure position exists
if (usSpriteNumberIn >= ausCount [byBookNumberIn])
  {
  usLastErrorCode = ERR_SPR_BAD_POSITION;
  return (ES_ERROR);
  };

// In the future, the following code will erase the actual encoded data.
//  For now, we will just clear our entry, since there is no encoded data
//  to erase.

asprdData [byBookNumberIn] [usSpriteNumberIn].sXPos   = 0;
asprdData [byBookNumberIn] [usSpriteNumberIn].sYPos   = 0;
asprdData [byBookNumberIn] [usSpriteNumberIn].sWidth  = 0;
asprdData [byBookNumberIn] [usSpriteNumberIn].sHeight = 0;

// set the empty flag
asprdData [byBookNumberIn] [usSpriteNumberIn].usFlags = SPR_FLG_EMPTY;

return (ES_NO_ERROR);
};

//........................................................................
ERRORSTATUS SpriteManager::EraseAllSprites
//........................................................................
  (
  BYTE             byBookNumberIn
  )
{
USHORT             usIndex;


for (usIndex = 0; usIndex < ausCount [byBookNumberIn]; ++usIndex)
  {
  if (EraseSprite (byBookNumberIn, usIndex) == ES_ERROR)
    {
    // the last error code has already been set in EraseSprite
    return (ES_ERROR);
    };
  };
return (ES_NO_ERROR);
};

//........................................................................
ERRORSTATUS SpriteManager::DestroyBook
//........................................................................
  (
  BYTE             byBookNumberIn
  )
{


free (asprdData [byBookNumberIn]);

asprdData [byBookNumberIn] = NULL;
ausCount [byBookNumberIn]  = 0;

return (ES_NO_ERROR);
};


//........................................................................
VOID SpriteManager::SetEncodeCanvas
//........................................................................
  (
  PCANVAS        pcnvCanvasIn
  )
{


pcnvEncode = pcnvCanvasIn;
};

//........................................................................
VOID SpriteManager::SetDisplayCanvas
//........................................................................
  (
  PCANVAS        pcnvCanvasIn
  )
{


pcnvDisplay = pcnvCanvasIn;
};

//........................................................................
ERRORSTATUS SpriteManager::EncodeSprite
//........................................................................
  (
  SHORT          sX1In,
  SHORT          sY1In,
  SHORT          sX2In,
  SHORT          sY2In,
  BYTE           byBookNumberIn,
  USHORT         usSpriteNumberIn
  )
{
SPRITEDATA       sprdDataOut;
SHORT            sTemp;


// make sure (sX1In, sY1In) is in the upper left hand corner

if (sX1In > sX2In)
  {
  sTemp = sX1In;
  sX1In = sX2In;
  sX2In = sTemp;
  };

if (sY1In > sY2In)
  {
  sTemp = sY1In;
  sY1In = sY2In;
  sY2In = sTemp;
  };

// In the future we will actually encode the sprite here with RLE encoding.
//  For now, we will pass the location of the sprite in the encode canvas
//  to the storage routine.

sprdDataOut.sXPos   = sX1In;
sprdDataOut.sYPos   = sY1In;
sprdDataOut.sWidth  = (SHORT)(sX2In - sX1In + 1);
sprdDataOut.sHeight = (SHORT)(sY2In - sY1In + 1);
sprdDataOut.usFlags = 0;

return (SetSprite (byBookNumberIn, usSpriteNumberIn,
                   (PBYTE)&sprdDataOut, 0));
};

//........................................................................
ERRORSTATUS SpriteManager::SetNumSprites
//........................................................................
  (
  BYTE           byBookNumberIn,
  USHORT         usNumOfSpritesIn
  )
{
ULONG            ulDataSize;
PSPRITEDATA      asprdNew;
USHORT           usIndex;
USHORT           usNumEntries;


// Allocate memory to hold the new book
ulDataSize = usNumOfSpritesIn * sizeof (SPRITEDATA);

if ((asprdNew = (PSPRITEDATA) malloc (ulDataSize)) == NULL)
  {
  usLastErrorCode = ERR_GEN_ALLOC_FAILURE;
  return (ES_ERROR);
  };

usIndex = 0;

// copy over as many of the old entries as possible, if they exist
if (asprdData [byBookNumberIn] != NULL)
  {
  usNumEntries = min (usNumOfSpritesIn, ausCount [byBookNumberIn]);

  for (;usIndex < usNumEntries; ++usIndex)
    {
    asprdNew [usIndex].sXPos   = asprdData [byBookNumberIn] [usIndex].sXPos;
    asprdNew [usIndex].sYPos   = asprdData [byBookNumberIn] [usIndex].sYPos;
    asprdNew [usIndex].sWidth  = asprdData [byBookNumberIn] [usIndex].sWidth;
    asprdNew [usIndex].sHeight = asprdData [byBookNumberIn] [usIndex].sHeight;
    };
  };

// clear the remaining entries
for (; usIndex < usNumOfSpritesIn; ++usIndex)
  {
  asprdNew [usIndex].sXPos   = 0;
  asprdNew [usIndex].sYPos   = 0;
  asprdNew [usIndex].sWidth  = 0;
  asprdNew [usIndex].sHeight = 0;
  };

// deallocate the old array of sprites
DestroyBook (byBookNumberIn);

// Set the new book in place.

asprdData [byBookNumberIn] = asprdNew;
ausCount  [byBookNumberIn] = usNumOfSpritesIn;
return (ES_NO_ERROR);
};

//........................................................................
USHORT SpriteManager::QueryNumSprites
//........................................................................
  (
  BYTE           byBookNumberIn
  )
{


// bounds checking
if (byBookNumberIn < MAX_NUMBER_BOOKS)
  {
  return (ausCount [byBookNumberIn]);
  }
else
  {
  return (0);
  };
};

//........................................................................
ERRORSTATUS SpriteManager::SpritesFromBlock
//........................................................................
  (
  BYTE           byBookNumberIn,
  PBYTE          pbyBlockIn,
  ULONG          ulBlockSizeIn
  )
{
// This routine will be implemented in the next version of the sprite
//  class, when we get into RLE sprites.

// remove compiler warning
byBookNumberIn = byBookNumberIn;
pbyBlockIn     = pbyBlockIn;
ulBlockSizeIn  = ulBlockSizeIn;
return (ES_NO_ERROR);
};

//........................................................................
ERRORSTATUS SpriteManager::BlockFromSprites
//........................................................................
  (
  BYTE           byBookNumberIn,
  PBYTE *        ppbyBlockOut,
  PULONG         pulBlockSizeOut
  )
{
// This routine will be implemented in the next version of the sprite
//  class, when we get into RLE sprites.

// remove compiler warning
byBookNumberIn  = byBookNumberIn;
ppbyBlockOut    = ppbyBlockOut;
pulBlockSizeOut = pulBlockSizeOut;
return (ES_NO_ERROR);
};

//........................................................................
ERRORSTATUS SpriteManager::DisplayUnclippedSprite
//........................................................................
  (
  USHORT         usXPosIn,
  USHORT         usYPosIn,
  BYTE           byBookNumberIn,
  USHORT         usSpriteNumberIn
  )
{
SHORT            sWidthIndex;
SHORT            sHeightIndex;
PBYTE            pbySource;
PBYTE            pbySourceStart;
LONG             lSourceIncrement;
PBYTE            pbyDest;
PBYTE            pbyDestStart;
LONG             lDestIncrement;
PSPRITEDATA      psprdTheSprite;



// Note:  This method of drawing sprites is great for illustration of what
//  a sprite is and how it works, but it is lousy as far as speed is
//  concerned.  The next version of the sprite class will have improved
//  encoding and display routines that are much, much faster than the one
//  presented here.

// Note that the sprite drawing routine will be called many times, so it
//  is wise to make it as fast as possible.  To speed up routines such as
//  these, it is best to remove error checking code, and just make sure
//  valid data is always passed to the routine.  However during development
//  it is good to have error checking code to track down bugs.  Therefore
//  it is often useful to use conditional compiles (#ifdef.. #endif) to
//  use error checking during development, and removing it for the release
//  version.

#ifdef DEBUG_ON
// make sure book exists
if (asprdData [byBookNumberIn] == NULL)
  {
  usLastErrorCode = ERR_SPR_BAD_BOOK;
  return (ES_ERROR);
  };

// make sure position exists
if (usSpriteNumberIn >= ausCount [byBookNumberIn])
  {
  usLastErrorCode = ERR_SPR_BAD_POSITION;
  return (ES_ERROR);
  };
#endif // DEBUG_ON


// set a pointer to sprite data.
psprdTheSprite = &asprdData [byBookNumberIn] [usSpriteNumberIn];

pbySourceStart = pcnvEncode->QueryRowAddress (psprdTheSprite->sYPos) +
                 psprdTheSprite->sXPos;

// set a pointer to the destination canvas where we will draw the sprite
pbyDestStart = pcnvDisplay->QueryRowAddress (usYPosIn) + usXPosIn;

// bring much used values to local variables for speed
lSourceIncrement = pcnvEncode->QueryRowIncrement  ();
lDestIncrement   = pcnvDisplay->QueryRowIncrement ();


// loop through each row of the sprite
for (sHeightIndex = psprdTheSprite->sHeight; sHeightIndex > 0; --sHeightIndex)
  {
  // loop through each pixel in the row
  pbySource = pbySourceStart;
  pbyDest   = pbyDestStart;

  for (sWidthIndex = psprdTheSprite->sWidth; sWidthIndex > 0; --sWidthIndex)
    {
    // check whether or not this pixel is transparent
    if (*pbySource < 248)
      {
      // pixel is opaque, so draw it
      *pbyDest = *pbySource;
      };
    ++pbySource;
    ++pbyDest;
    };

  // move down to the next row
  pbySourceStart += lSourceIncrement;
  pbyDestStart   += lDestIncrement;
  };
return (ES_NO_ERROR);
};




