/*
**
** problem - a problem database manager
**
** Written in C++ using the termcap library for screen management
** and GDBM as the database library.
**
** problem.C is made by cating together problem1.C and problem2.C
**
** problem1.C problem1.C 1.33   Delta\'d: 17:24:36 11/9/92   Mike Lijewski, CNSF
**
** Copyright \(c\) 1991, 1992 Cornell University
** All rights reserved.
**
** Redistribution and use in source and binary forms are permitted
** provided that: \(1\) source distributions retain this entire copyright
** notice and comment, and \(2\) distributions including binaries display
** the following acknowledgement:  ``This product includes software
** developed by Cornell University\'\' in the documentation or other
** materials provided with the distribution and in all advertising
** materials mentioning features or use of this software. Neither the
** name of the University nor the names of its contributors may be used
** to endorse or promote products derived from this software without
** specific prior written permission.
**
** THIS SOFTWARE IS PROVIDED ``AS IS\'\' AND WITHOUT ANY EXPRESS OR
** IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
** WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
*/

#include <ctype.h>

#ifndef _IBMR2
#include <libc.h>
#endif

#include <fcntl.h>
#include <new.h>
#include <osfcn.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <time.h>
#include <unistd.h>

#ifdef __EMX__
#include <io.h>
#include <process.h>
#else
#include <sys/wait.h>
#endif

#include "classes.h"
#include "display.h"
#include "keys.h"
#include "lister.h"
#include "problem.h"
#include "regexp.h"
#include "utilities.h"
#include "version.h"

// a little POSIX stuff
#ifndef SEEK_SET
#define SEEK_SET 0
#endif
#ifndef SEEK_END
#define SEEK_END 2
#endif

//
// Where problem\'s maillists and the sequence file reside.
//
#ifdef HOMEBASE
const char *HomeBase = HOMEBASE;
#else
#error "must define HOMEBASE"
#endif

//
// Name of file containing valid problem areas relative to HomeBase.
//
const char *const ProblemAreas = "AREAS";

//
// Definition of our GDMB filehandle  -- only one open GDBM file at a time.
//
GDBM_FILE GdbmFile;

//
// Suffix for gdbm files.
//
const char *const GdbmSuffix = ".db";

//
// File containing last problem #, relative to HomeBase.
//
const char *const SequenceFile = "SEQUENCE";

//
// Prefix for maillist files.
//
const char *const MailListSuffix = ".lst";

//
// Our modeline prefix.
//
const char *const ModelinePrefix = "----- Problem: ";

//
// Help message for the message window when displaying help.
//
const char *const HELP_MSG[] = {
    "Space scrolls forward.  Other keys quit.",
    "Space forward, Backspace backward.  Other keys quit.",
    "Backspace scrolls backward.  Other keys quit.",
    "Any key quits."
};

//
// Our fields.
//
static const char *const Fields[] =
{
    "Area",
    "Logger",
    "Reporter",
    "Logged",
    "Updated",
    "Keywords",
    "Summary",
    "Status",
    "Site",
    "Severity",
    "Problem #"
};

inline int NFields() { return sizeof(Fields) / sizeof(Fields[0]); }

//
// The current area - used by setCurrentArea and CurrentArea.
//
String current_area;

void setCurrentArea(const char *area)
{
    current_area    = area;
    const char *tmp = strchr(current_area, '-');

    //
    // strip off any comments
    //
    if (tmp)
        current_area[tmp - (const char *)current_area] = 0;
    else
        tmp = (const char *)current_area + current_area.length();

    //
    // strip off any trailing blanks
    //
    for (--tmp; *tmp == ' ' || *tmp == '\t'; --tmp)
        current_area[tmp - (const char *)current_area] = 0;

    //
    // set any remaining spaces to underscores
    //
    while ((tmp = strchr(current_area, ' ')) != 0)
        current_area[tmp - (const char *)current_area] = '_';
}

//
// our commands - the first character of each string is the unique
//                character identifying that command.
//
static const char *const Commands[] =
{
    "l  -- log new problem",
    "a  -- append to a problem",
    "c  -- close a problem",
    "e  -- examine a problem",
    "v  -- view problem summaries",
    "s  -- subscribe to this problem area",
    "u  -- unsubscribe from this problem area",
    "k  -- keyword search over problem headers",
    "K  -- keyword search over problem headers and data",
    "d  -- delete a problem from the database",
    "r  -- reorganize the database",
    "M  -- modify keyword field",
    "P  -- change priority (severity) of problem",
    "R  -- reopen a closed problem",
    "T  -- transfer problem to another area",
    "q  -- quit"
};

inline int NCommands() { return sizeof(Commands) / sizeof(Commands[0]); }

/*
** exception handler for new - called once in main
*/

static void free_store_exception()
{
    error("exiting, memory exhausted, sorry");
}

/*
** get_problem_areas - read in the valid problem areas. Exits on error.
*/

static const char **get_problem_areas()
{
    static char **Areas;
    if (Areas) return (const char **)Areas;

    String filename(HomeBase);
    filename += "/";
    filename += ProblemAreas;

    FILE *fp = fopen(filename, "r");
    if (!fp)
        error ("file %s, line %d, couldn't fopen() `%s'",
               __FILE__, __LINE__, (const char *)filename);

    const int chunksize = 10;  // average number of problem areas expected
    const int linelen   = 80;  // average length of line in AREAS expected
    Areas = new char*[chunksize];

    if (read_file(fp, Areas, chunksize, linelen) < 0)
        error("file %s, line %d, error reading `%s'.",
              __FILE__, __LINE__, (const char *)filename);
    (void)fclose(fp);

    return (const char **)Areas;
}

/*
** is_area - is area a valid problem area? Returns 1 if true, else 0.
*/

int is_area(const char *area)
{
    const char **areas = get_problem_areas();

    for (int i = 0; areas[i]; i++)
    {
        const char *space = strchr(areas[i],' ');
        int length = space ? space - areas[i] - 1 : (int)strlen(areas[i]);
        if (strncmp(area, areas[i], length) == 0) return 1;
    }

    return 0;
}

/*
** NAreas - the number of areas
*/

static int NAreas()
{
    static int nAreas;
    if (!nAreas)
    {
        const char **areas = get_problem_areas();
        for (int i = 0; areas[i]; i++) ;
        nAreas = i;
    }
    return nAreas;
}

/*
** display_area_screen - displays the screen of available areas.
**                       We assume that the caller will display the modeline.
**                       Needs at least four rows to be useful.
*/

static void display_area_screen()
{
    clear_display_area();

    const char **areas = get_problem_areas();
    enter_standout_mode();

    //
    // If we have enough available screen lines, we display the areas
    // one per line.  Otherwise, we display them in columns of up to
    // column_width characters.
    //
    const int column_width = 25;           // space allowed for each area
    int dvd = columns() / column_width;    // # compacted areas per row
    int rws = rows() - 5;
    int compact = NAreas() <= rws ? 0 : 1; // compact the areas?

    const char *fmt = NAreas() <= rws * dvd ? "The %d Areas:" :
                                              "The First %d Areas:";

    char *msg = new char[strlen(fmt) + 8]; // lots of problem areas

    (void)sprintf(msg, fmt, NAreas() <= rws * dvd ? NAreas() : rws * dvd);

    display_string(msg);

    //
    // Maximum # of digits we must display.
    //
    int digits = int(strlen(msg) - strlen(fmt)) + 2;

    DELETE msg;

    end_standout_mode();
    cursor_wrap();

    for (int i = 0, offset = 0; i < NAreas() && i < rws * dvd; i++)
    {
        if (compact)
        {
            offset = (i / rws) * column_width;
            move_cursor((i % rws) + 2, offset);
        }
        (void)fputs("  ", stdout);  // two spaces for initial indentation
        enter_standout_mode();
        (void)fprintf(stdout, "%*d", digits, i + 1);
        end_standout_mode();
        putchar(' ');               // and one more between number and area
        display_string(areas[i], 0, digits + 3 + offset);
    }
}

/*
** help - display some help lines.  The first two lines of the output
**        consists of the message msg and then a blank line.
**        This is followed by the contents of lines, which has dim
**        elements.
**
*/

static void help(const char *lines[], int dim, const char *msg)
{
    String old_modeline(current_modeline);

    update_modeline("----- HELP");
    clear_display_area();

    int position = 0, offset = 0;
    char key;
    do
    {
        cursor_home();
        if (position == 0)
        {
            offset = 2;
            display_string(msg);
            move_cursor(2, 0);
        }
        for (int i = 0; i + offset < rows()-2 && i + position < dim; i++)
            display_string(lines[i + position]);

        clear_message_line();

        if (position == 0 && dim + offset <= rows() - 2)
            //
            // whole help message fits in initial screen
            //
            (void)fputs(HELP_MSG[3], stdout);
        else if (position == 0)
            //
            // the head of the help message -- it all does not fit
            //
            (void)fputs(HELP_MSG[0], stdout);
        else if (position + rows() - 2 >= dim)
            //
            // the tail of the help message
            //
            (void)fputs(HELP_MSG[2], stdout);
        else
            //
            //  somewhere in between
            //
            (void)fputs(HELP_MSG[1], stdout);
        synch_display();

        if (resumingAfterSuspension ||
#ifdef SIGWINCH
            windowSizeChanged       ||
#endif
            get_key(&key) < 0 // assume fails only when errno == EINTR
            )
        {
#ifdef SIGWINCH
            if (windowSizeChanged)
            {
                windowSizeChanged = 0;
                adjust_window();
            }
#endif
            resumingAfterSuspension = 0;
            update_modeline();
            continue;
        }
        else if (key == KEY_SPC)
        {
            if (position >= dim - 1) break;
            position += (position == 0 ? rows() - 2 - offset : rows() - 2);
        }
        else if (key == *BC)
        {
            if (position == 0) break;
            position -= rows() - 2;
            if (position < 0) position = 0;
        }
        else
            break;  // return to the listing

        offset = 0;
    }
    while (position < dim + offset);

    update_modeline(old_modeline);
}

/*
** redisplay_area_screen - suitable for calling after SIGWINCH and SIGTSTP
*/

static void redisplay_area_screen()
{
    display_area_screen();
    update_modeline();
}

/*
** choose_problem_area - returns a problem area to examine.
**                       Also gives user option to exit.
*/

static const char *choose_problem_area()
{
    const char **areas = get_problem_areas();
    display_area_screen();
    update_modeline(ModelinePrefix, "---- q (quit) H (help)");
    const char *helpmsg = "The Available Problem Areas:";
    char key;
    int index;

    if (NAreas() < 10)
    {
        //
        // The most usual case.  Read digit as typed.
        //
        message("Your Choice --> ");

        while (1)
        {
            if (resumingAfterSuspension ||
#ifdef SIGWINCH
                windowSizeChanged       ||
#endif
                get_key(&key) < 0)  // assume only fails when errno==EINTR
            {
#ifdef SIGWINCH

                if (windowSizeChanged)
                {
                    windowSizeChanged = 0;
                    adjust_window();
                }
#endif
                resumingAfterSuspension = 0;
                redisplay_area_screen();
                message("Your Choice --> ");
                continue;
            }
            else if (key == KEY_q)
                quit();
            else if (key == KEY_H || key == KEY_QM)
            {
                help(areas, NAreas(), helpmsg);
                redisplay_area_screen();
                message("Your Choice --> ");
            }
            else
            {
                index = key - '0';
                if (index > 0 && index <= NAreas()) return areas[index-1];
                ding();
            }
        }
    }
    else
    {
        char *response;
        while (1)
        {
            //
            // prompt takes care of window resizes and resume/suspends
            //
            response = prompt("Your Choice --> ", redisplay_area_screen);

            if (*response == KEY_q)
                quit();
            else if (*response == KEY_H || *response == KEY_QM)
            {
                help(areas, NAreas(), helpmsg);
                redisplay_area_screen();
                message("Your Choice --> ");
                DELETE response;
            }
            else
            {
                index = atoi(response);
                DELETE response;
                if (index > 0 && index <= NAreas()) return areas[index - 1];
                ding();
            }
        }
    }
}

/*
** max_field_length - returns the length of the longest field
*/

static int max_field_length()
{
    static int len;
    if (!len)
        for (int i = 0; i < NFields(); i++)
            if (len < strlen(Fields[i])) len = (int) strlen(Fields[i]);
    return len;
}

/*
** get_severity - prompt for the severity of the problem. Deal with
**                getting SIGTSTP or SIGWINCH.
*/

static char get_severity(void (*redisplay)())
{
    //
    // Compile with -DSEVLEVEL3 if you only want three severity levels.
    //
#ifdef SEVLEVEL3
    const char *msg = "Severity (1 (=highest), 2, or 3 (=lowest)) --> ";
#else
    const char *msg = "Severity (1 (=highest), 2, 3 or 4 (=lowest)) --> ";
#endif

    message(msg);

    char key;
    while (1)
    {
        if (resumingAfterSuspension ||
#ifdef SIGWINCH
            windowSizeChanged       ||
#endif
            get_key(&key) < 0)   // assume only fails when errno == EINTR 
        {
#ifdef SIGWINCH
            if (windowSizeChanged)
            {
                windowSizeChanged = 0;
                adjust_window();
            }
#endif
            resumingAfterSuspension = 0;
            redisplay();
            message(msg);
            continue;
        }
        switch (key)
        {
#ifdef SEVLEVEL3
          case '1': case '2': case '3': return key;
#else
          case '1': case '2': case '3': case '4': return key;
#endif
          default: ding(); break;
        }
    }
}

/*
** filesize - returns size of file or MAXFILESIZE,
**            whichever is smaller. Exits on error.
*/

static int filesize(const char *file)
{
#ifdef MAXFILESIZE
    const int MaxFileSize =  MAXFILESIZE;
#else
    const int MaxFileSize = 16000;
#endif
    struct stat buf;
    if (stat(file, &buf) < 0)
        error("file %s, line %d, fstat() failed", __FILE__, __LINE__);
    return buf.st_size > MaxFileSize ? MaxFileSize : buf.st_size;
}

/*
** sequence_file - returns full pathname of "SequenceFile"
*/

static const char *sequence_file()
{
    static String filename;
    if (filename == "")
    {
        filename  = HomeBase;
        filename += "/";
        filename += SequenceFile;
    }
    return filename;
}

/*
** update_sequence_file - reads the previous problem number from
**                        SequenceFile; increments that number;
**                        writes it back to the file and returns it.
**                        Exits on error. We should have an exclusive
**                        lock on the sequence file before we get here.
**                        This is to guarantee that we only have one
**                        "writer" to the GDBM file.
*/

static const char *update_sequence_file(int fd)
{
    static char buf[10];  // will not have this many problems for a while
    char line[10];

    int bytes = read(fd, line, sizeof(line));
    if (bytes == -1)
        error("file %s, line %d, read() on `%s' failed",
                __FILE__, __LINE__, sequence_file());
    line[bytes] = 0;
    (void)sprintf(buf, "%d", atoi(line) + 1);

#ifdef __EMX__
    if (chsize(fd, 0) < 0)
        error("file %s, line %d, chsize() of `%s' failed",
              __FILE__, __LINE__, sequence_file());
#else
    //
    // Truncate the file.  I would like to use ftruncate here, but that
    // is not as portable as a close\(open\(\)\) scheme.
    //
    if (close(open(sequence_file(), O_RDWR|O_TRUNC)) < 0)
        error("file %s, line %d, close(open()) of `%s' failed",
              __FILE__, __LINE__, sequence_file());
#endif

    //
    // Go to the beginning.
    //
    if (lseek(fd, 0, SEEK_SET) < 0)
        error("file %s, line %d, lseek() on `%s' failed",
              __FILE__, __LINE__, sequence_file());

    //
    // Write the next problem #.
    //
    if (write(fd, buf, (unsigned) strlen(buf)) < 0)
        error("file %s, line %d, write() to `%s' failed",
              __FILE__, __LINE__, sequence_file());

    DELETE line;

    return buf;
}

/*
** mail_list_prefix - returns the full pathname of MailList location
*/

static const char *mail_list_prefix()
{
    static String filename;
    if (filename == "")
    {
        filename  = HomeBase;
        filename += "/";
    }
    return filename;
}

/*
** open_maillist_file - open the file containing the interested parties
**                      maillist for the problem area. Exits on error.
**                      If the file did not previously exist, it will be
**                      created.
*/

static int open_maillist_file()
{
    String file(mail_list_prefix());
    file += CurrentArea();
    file += MailListSuffix;

    int fd = open((const char *)file, O_RDWR|O_CREAT, 0644);
    if (fd < 0)
        error("file %s, line %d, open(%s) failed",
              __FILE__, __LINE__, (const char *)file);
    return fd;
}

//
// The ways that a database can get modified.
//
static const char *const How[] = {
    "logged",
    "appended",
    "closed",
    "deleted",
    "reorganized",
    "keywords modified",
    "reopened",
    "severity modified",
    "transferred"
};

//
// indices into How
//
enum Modified {
    LOGGED,
    APPENDED,
    CLOSED,
    DELETED,
    REORGANIZED,
    KEYWORDMOD,
    REOPENED,
    SEVERITYMOD,
    TRANSFER
};

/*
** update_subscribers - send a mail file about problem to all
**                      those who have subscribed to this area.  If
**                      this is being called after an append,
**                      append is non-zero.  Otherwise, we assume
**                      it is the initial logging.  If we are appending
**                      or closing a problem, we pass the offset of the
**                      new data that needs to be printed.  When logging,
**                      this offset is zero, which we make the default.
*/

static void update_subscribers(const datum data, const char *number,
                               const Modified how, int offset = 0)
{
#ifdef MAILPROG
    const char *mailprog = MAILPROG;
#else
    const char *mailprog = "/bin/mail";
#endif

#ifdef __EMX__
    char *env = getenv("MAILER");
    if (env) mailprog = env;
#else
    //
    // Does mailprog really exist?
    //
    if (!read_and_exec_perm(mailprog))
        error("file %s, line %d, `%s' doesn't appear to be executable",
              __FILE__, __LINE__, mailprog);
#endif

    int mailfd   = open_maillist_file();
    FILE *mailfp = fdopen(mailfd, "r");
    if (!mailfp)
        error("file %s, line %d, fdopen() failed", __FILE__, __LINE__);

    const int namelen   = 10;  // average length of uid expected
    const int chunksize = 20;  // average number of subscribees expected
    char **args = new char*[chunksize];
    args[0] = (char *) mailprog;

#ifdef NOSUBJECT
    String subject("Subject: Problem: ");
#else
    args[1] = "-s";
    String subject("Problem: ");
#endif

    subject += CurrentArea();
    subject += " # ";
    subject += number;
    subject += " ";
    subject += How[how];
    subject += " by `";
    subject += username();
    subject += "'";

#ifdef NOSUBJECT
    subject += "\n\n";
#endif

#ifndef NOSUBJECT
    args[2] = subject;
#endif

    //
    // Are there any subscribers?
    //
#ifdef NOSUBJECT
    int nlines = read_file(mailfp, args, chunksize, namelen, 1);
#else
    int nlines = read_file(mailfp, args, chunksize, namelen, 3);
#endif
    if (nlines < 0)
        error("file %s, line %d, problem reading from maillist file",
              __FILE__, __LINE__);
    (void)close(mailfd);
    (void)fclose(mailfp);
    if (nlines == 0)
    {
        //
        // No subscribers.
        //
#ifdef NOSUBJECT
        for (int i = 1; args[i]; i++) DELETE args[i];
#else
        for (int i = 3; args[i]; i++) DELETE args[i];
#endif
        DELETE args;
        return;
    }

    int fds[2];
    if (pipe(fds) < 0)
        error("file %s, line %d, pipe() failed", __FILE__, __LINE__);

#ifdef __EMX__
    int old0 = dup(0);
    fcntl(old0, F_SETFD, 1);
    if (dup2(fds[0], 0) < 0)
      error("file %s, line %d, dup() failed", __FILE__, __LINE__);
    fcntl(fds[1], F_SETFD, 1);
    fcntl(1, F_SETFD, 1);

    if ( spawnvp(P_NOWAIT, mailprog, (char *const *)args) == -1 )
        error("file %s, line %d, spawn() failed", __FILE__, __LINE__);

    fcntl(1, F_SETFD, 0);
    dup2(old0, 0);
    close(old0);
#else
    switch(fork())
    {
      case -1: // error
        error("file %s, line %d, fork() failed", __FILE__, __LINE__);
      case 0: // in the child
        //
        // Set stdin to the read end of pipe.
        //
        (void)close(0);
        if (dup(fds[0]) < 0)
            error("file %s, line %d, dup() failed", __FILE__, __LINE__);
        (void)close(fds[0]);
        (void)close(fds[1]);
        (void)close(1);
        (void)close(2);

        //
        // We fork again and let the grandchild do the actual
        // mailing.  This way our parent will not have to wait
        // for the mail to be sent.
        //
        switch(fork())
        {
          case -1:
            exit(1);
          case 0:
            execv(mailprog, (char *const *)args);
            exit(1);  // exec failed
          default:
            exit(0);
        }
        break;
      default:  // in the parent
      {
#endif
          (void)close(fds[0]);

#ifdef NOSUBJECT
          //
          // Write Subject to pipe.
          //
          write_to_pipe(fds[1], subject, subject.length());
#endif

          //
          // write the mailfile to the pipe
          //
          switch (how)
          {
            case CLOSED:
            case REOPENED:
            case APPENDED:
            case KEYWORDMOD:
            case SEVERITYMOD:
            case TRANSFER:
            {
                //
                // Write the fields and the new data only.
                //
                char *tail = data.dptr;
                for (int i = 0; i < NFields(); i++)
                {
                    tail = strchr(tail, '\n');
                    tail += 1; // step past the newline
                }
                tail += 1;     // step past the second newline to the
                               // first character past the header
                //
                // write the header
                //
                write_to_pipe(fds[1], data.dptr, tail - data.dptr);
                if (offset <= 0)
                    error("file %s, line %d, offset must be positive",
                          __FILE__, __LINE__);
                write_to_pipe(fds[1], data.dptr + offset, data.dsize-offset-1);
            }
                break;
            case LOGGED:
                write_to_pipe(fds[1], data.dptr, data.dsize - 1);
                break;
            default: error("file %s, line %d, illegal case in switch()",
                           __FILE__, __LINE__);
          }
          (void)close(fds[1]);

#ifdef NOSUBJECT
          for (int i = 1; args[i]; i++) DELETE args[i];
#else
          for (int i = 3; args[i]; i++) DELETE args[i];
#endif
          DELETE args;

          //
          // We are not interested in the return value.  The assumption
          // here is that if something goes wrong with the mailing that
          // the error will eventually get to the problem administrator.
          //
          (void)wait(0);

          return;
#ifndef __EMX__
      }
    }
#endif
}

/*
** invoke_editor - invoke users editor on file
*/

static void invoke_editor(const char *file)
{
    char *editor = getenv("EDITOR");
    if (editor == 0) editor = "vi";

    String argstring(editor);
    argstring += " ";
    argstring += file;

    //
    // We tokenize because the user could have EDITOR
    // set to "emacs -q -nw", or some such thing, which execvp
    // would not recognize as a legal filename.
    //
    const char **args = tokenize(argstring, " \t");

    //
    // We must be careful of that venerable old editor "vi",
    // which has a habit of returning error codes, when nothing
    // went wrong.
    //
    if (!execute(args[0], args) && strcmp(args[0], "vi"))
        error("file %s, line %d, couldn't exec() your editor `%s'",
              __FILE__, __LINE__, editor);
}

/*
** database_exists - checks to see if a database for the current area exists.
**                   This is important since gdbm_open\(GDBM_READER\)
**                   will fail if the database does not already exist.
**                   Returns one if a file of the appropriate name
**                   exists, else zero.  There is no guarantee that we
**                   actually have a database, only an appropriately
**                   named file.
*/

int database_exists()
{
    String filename(HomeBase);
    filename += "/";
    filename += CurrentArea();
    filename += GdbmSuffix;

    int rc = open((const char *)filename, O_RDONLY);
    (void)close(rc);
    return rc < 0 ? 0 : 1;
}

/*
** open_database - opens the GDBM database on the current area.
**                 Exits on error.
*/

void open_database(int mode)
{
    String filename(HomeBase);
    filename += "/";
    filename += CurrentArea();
    filename += GdbmSuffix;

    if ((GdbmFile = gdbm_open(filename, 0, mode, 00644, 0)) == 0)
        error("file %s, line %d, gdbm_open() failed on `%s', errno = %d",
              __FILE__, __LINE__, (const char *)filename, gdbm_errno);
}

/*
** database_directory_exists - does HomeBase exist?
*/

static int database_directory_exists()
{
    int rc = access(HomeBase, 0);
    return rc < 0 ? 0 : 1;
}

/*
** database locking
*/

static int lock_fd = -1;

static void lock_database(void)
{
    if (lock_fd != -1)
        return; /* already locked */
    lock_fd = open(sequence_file(), O_RDWR);
    if (lock_fd < 0)
        error("file %s, line %d, open() on `%s' failed",
              __FILE__, __LINE__, sequence_file());
    lock_file(lock_fd);                  // lock our sequence file
}

static void unlock_database(void)
{
    if (lock_fd == -1)
        return; /* already unlocked */
    unlock_file(lock_fd);
    (void)close(lock_fd);
    lock_fd = -1;
}

/*
** update_database - updates database on the current area with problem_data.
**                   size is total size of data.  This function is
**                   used both to insert new entries and to replace
**                   old entries. If offset is nonzero, it is the
**                   start of problem number field, which we fill in
**                   after getting new problem number.  offset is
**                   nonzero only when initially logging a problem.
*/

static void update_database(datum &key, datum &data, const Modified how,
                            int offset = 0)
{
    block_tstp_and_winch();             // block SIGTSTP and WINCH
    lock_database();
    open_database(GDBM_WRCREAT);        // open database for writing
    if (how != REORGANIZED)
        data.dptr[data.dsize - 1] = 0;  // make sure data is stringified

    switch (how)
    {
      case DELETED:
        if (gdbm_delete(GdbmFile, key))
            error("file %s, line %d, gdbm_delete() failed, errno = %d",
                  __FILE__, __LINE__, gdbm_errno);
        break;
      case REORGANIZED:
        if (gdbm_reorganize(GdbmFile))
            error("file %s, line %d, gdbm_reorganize() failed, errno = %d",
                  __FILE__, __LINE__, gdbm_errno);
        break;
      case TRANSFER:
      {
        //
        // Close original database.
        //
        gdbm_close(GdbmFile);

        //
        // Remember current area before switching to new one.
        //
        String oldArea(CurrentArea());
        char *newArea = data.dptr + max_field_length() + 1;
        char *tmp = strchr(newArea, '\n');
        *tmp = 0;
        setCurrentArea(newArea);
        *tmp = '\n';

        //
        // Open database on the new area and insert the problem.
        //
        open_database(GDBM_WRCREAT);

        if (gdbm_store(GdbmFile, key, data, GDBM_INSERT))
            error("file %s, line %d, gdbm_store() failed, errno = %d",
                  __FILE__, __LINE__, gdbm_errno);

        //
        // Now go back to previous database and delete problem from there.
        //
        gdbm_close(GdbmFile);
        setCurrentArea(oldArea);
        open_database(GDBM_WRCREAT);

        if (gdbm_delete(GdbmFile, key))
            error("file %s, line %d, gdbm_delete() failed, errno = %d",
                  __FILE__, __LINE__, gdbm_errno);
        break;
      }
      case LOGGED:
      {
          //
          // Must fill in key values; we are doing an initial log
          // of the problem.
          //
          key.dptr  = (char *) update_sequence_file(lock_fd);
          key.dsize = (int)strlen(key.dptr) + 1;

          //
          // update problem # field
          //
          for (int i = 0; i < strlen(key.dptr); i++)
              data.dptr[offset + i] = key.dptr[i];
      }
      //
      // Fall through.
      //
      case CLOSED:
      case REOPENED:
      case APPENDED:
      case KEYWORDMOD:
      case SEVERITYMOD:
        if (gdbm_store(GdbmFile, key, data, how == LOGGED ?
                       GDBM_INSERT : GDBM_REPLACE))
            error("file %s, line %d, gdbm_store() failed, errno = %d",
                  __FILE__, __LINE__, gdbm_errno);
        break;
      default:
        error("file %s, line %d, illegal case in switch()",
              __FILE__, __LINE__);
    }

    gdbm_close(GdbmFile);
    unlock_database();
    unblock_tstp_and_winch(); // unblock SIGTSTP and WINCH
}

//
// These variables are shared by build_log_screen and log_new_problem
// so that we can keep track of where we are in case we get a SIGTSTP
// or SIGWINCH.  We have to be careful to nullify them after we are done
// with them so that build_log_screen knows when it needs to prompt
// for fresh data.
//
static char *logged;
static char *reporter;
static char *keywords;
static char *summary;
static char *site;
static char severity;

/*
** build_log_screen - prints the initial screen when logging a problem.
**                    Is also called after a SIGTSTP or SIGWINCH to
**                    redo the screen appropriately.
*/

// forward declaration
static void redisplay_log_screen();

static void build_log_screen()
{
    clear_display_area();

    //
    // Print as many of the fields as will fit.
    // This gets done both on a normal call or on a redisplay.
    //
    enter_standout_mode();
    for (int i = 0; i < NFields() && i < rows() - 2; i++)
        display_string(Fields[i]);
    end_standout_mode();

    int fieldlen = max_field_length() + 1; // plus one accounts for the space

    int currline = 0;  // keep track of where we are on screen

    //
    // Fill in those fields which are obvious.
    //
    if (currline < rows() - 2)
    {
        move_cursor(currline++, fieldlen);
        display_string(CurrentArea(), 0, fieldlen);
    }
    if (currline < rows() - 2)
    {
        move_cursor(currline, fieldlen);
        display_string(username(), 0, fieldlen);
        currline += 2;
    }
    time_t t = time(0);
    logged = ctime(&t);
    if (currline < rows() - 2)
    {
        move_cursor(currline++, fieldlen);
        display_string(logged, 0, fieldlen);
    }
    if (currline < rows() - 2)
    {
        move_cursor(currline, fieldlen);
        display_string(logged, 0, fieldlen);
        currline += 3;
    }
    if (currline < rows() - 2)
    {
        move_cursor(currline, fieldlen);
        display_string("open", 0, fieldlen);
    }

    //
    // Prompt for those that are not.
    //
    const char *const suffix = " --> ";
    static char *line;
    int recursing = 0;  // is this a recursive call?
    if (!line)
        line = new char[fieldlen + strlen(suffix)];
    else
        recursing = 1;
    (void)strcpy(line, Fields[2]);
    (void)strcat(line, suffix);
    if (!reporter)
        if (recursing)
            goto exit;
        else
	    if ((reporter = (char *) fullname()) == NULL)
            reporter = prompt(line, redisplay_log_screen);
    currline = 2;
    if (currline < rows() - 2)
    {
        move_cursor(currline, fieldlen);
        display_string(reporter, 0, fieldlen);
        currline += 3;
    }
    (void)strcpy(line, Fields[5]);
    (void)strcat(line, suffix);
    if (!keywords)
        if (recursing)
            goto exit;
        else
            keywords = prompt(line, redisplay_log_screen);
    if (currline < rows() - 2)
    {
        move_cursor(currline++, fieldlen);
        display_string(keywords, 0, fieldlen);
    }
    (void)strcpy(line, Fields[6]);
    (void)strcat(line, suffix);
    if (!summary)
        if (recursing)
            goto exit;
        else
            summary = prompt(line, redisplay_log_screen);
    if (currline < rows() - 2)
    {
        move_cursor(currline, fieldlen);
        display_string(summary, 0, fieldlen);
        currline += 2;
    }
    (void)strcpy(line, Fields[8]);
    (void)strcat(line, suffix);
    if (!site)
        if (recursing)
            goto exit;
        else
            site = prompt(line, redisplay_log_screen);
    if (currline < rows() - 2)
    {
        move_cursor(currline++, fieldlen);
        display_string(site, 0, fieldlen);
    }
    if (!severity)
        if (recursing)
            goto exit;
        else
            severity = get_severity(redisplay_log_screen);
    if (currline < rows() - 2)
    {
        move_cursor(currline, fieldlen);
        putchar(severity);
    }
    DELETE line;
    line = 0;  // the nullification is important

    //
    // We got here when we have been called recursively due to servicing
    // a SIGTSTP or SIGWINCH.  We do not delete line as we will shortly be
    // back in our original self where we will continue to use it.
    //
  exit:
    return;
}

/*
** redisplay_log_screen - redisplay the log screen.  Can be called at any
**                        point during our building of the log screen to
**                        service a SIGTSTP or SIGWINCH.
*/

static void redisplay_log_screen() { update_modeline(); build_log_screen(); }

/*
** log_new_problem - need at least 4 rows to be useful
*/

static void log_new_problem()
{
    String suffix(CurrentArea());
    suffix += " (logging)";

    update_modeline(ModelinePrefix, suffix);

    build_log_screen();

    message("Invoking your editor ...");

    //
    // Build  tmp file into which the problem will be edited by user.
    //
    const char *file = temporary_file();

    invoke_editor(file);

    //
    // Generate string just large enough to hold the problem.
    // Do not forget to add space for the newlines.
    //
    int flen       = max_field_length() + 1; // plus one accounts for the space
    int fsize      = filesize(file);
    int totalsize  = fsize + NFields() * flen;

    const int PDim = 10;   // spaces reserved for Problem # field
    const int StatDim = 6; // big enough for "open" or "closed"


    totalsize += int(strlen(CurrentArea())) + 1;
    totalsize += int(strlen(username())) + 1;
    totalsize += 50;       // strlen\(ctime\)==25 && already contains newline
    totalsize += StatDim + 1;  // "open" or "closed"
    totalsize += int(strlen(reporter)) + 1;
    totalsize += int(strlen(keywords)) + 1;
    totalsize += int(strlen(summary)) + 1;
    totalsize += int(strlen(site)) + 1;
    totalsize += 2;        // the severity field
    totalsize += PDim + 2; // space reserved for the problem number, its
                           // newline, and an extra one

    datum data;
    data.dsize = totalsize + 1; // do not forget about the null
    data.dptr  = new char[data.dsize];

    //
    // Write the header info to data.
    //
    int pos = 0;  // our position in data
    (void)sprintf(data.dptr, "%-*.*s%s\n", flen, flen, Fields[0], CurrentArea());
    pos += int (flen+strlen(CurrentArea())+1);
    (void)sprintf(&data.dptr[pos], "%-*.*s%s\n", flen, flen, Fields[1],
                  username());
    pos += int (flen+strlen(username())+1);
    (void)sprintf(&data.dptr[pos], "%-*.*s%s\n", flen, flen, Fields[2],
                  reporter);
    pos += int (flen+strlen(reporter)+1);
    (void)sprintf(&data.dptr[pos], "%-*.*s%s", flen, flen, Fields[3], logged);
    pos += flen+25;
    (void)sprintf(&data.dptr[pos], "%-*.*s%s", flen, flen, Fields[4], logged);
    pos += flen+25;
    (void)sprintf(&data.dptr[pos], "%-*.*s%s\n", flen, flen, Fields[5],
                  keywords);
    pos += int (flen+strlen(keywords)+1);
    (void)sprintf(&data.dptr[pos], "%-*.*s%s\n", flen, flen, Fields[6],
                  summary);
    pos += int (flen+strlen(summary)+1);
    (void)sprintf(&data.dptr[pos], "%-*.*s%-*.*s\n", flen, flen, Fields[7],
                  StatDim, StatDim, "open");
    pos += flen+StatDim+1;
    (void)sprintf(&data.dptr[pos], "%-*.*s%s\n", flen, flen, Fields[8], site);
    pos += int (flen+strlen(site)+1);
    (void)sprintf(&data.dptr[pos], "%-*.*s%c\n", flen, flen, Fields[9],
                  severity);
    pos += flen+2;
    (void)sprintf(&data.dptr[pos], "%-*.*s          \n\n", flen, flen,
                  Fields[10]);
    int offset = pos + flen; // data.dptr\[offset\] is where Problem # goes
    pos += flen+PDim+2;      // we output two newlines here as separator

    //
    // Now for the problem itself.  Make sure this read only fails on a
    // real error -- block SIGTSTP and SIGWINCH
    //
    block_tstp_and_winch();

    int fd;
    if ((fd = open(file, O_RDONLY)) < 0)
        error("file %s, line %d, open(%s, O_RDONLY) failed",
              __FILE__, __LINE__, file);

    if (read(fd, &data.dptr[pos], fsize) == -1)
        error("file %s, line %d, read() failed", __FILE__, __LINE__);

    unblock_tstp_and_winch(); // unblock SIGTSTP and WINCH
    (void)close(fd);
    (void)unlink(file);

    if (yes_or_no("Really log this problem (y|n)? ",
                  redisplay_log_screen, Yes, 1))
    {
        datum key;  // empty key to be filled in by update_database
        update_database(key, data, LOGGED, offset);
        update_subscribers(data, key.dptr, LOGGED);
    }

    update_modeline();  // redisplay last modeline
    DELETE data.dptr;

    //
    // We have to both delete these and nullify them so that the next
    // time we call build_log_screen, we prompt for new data.
    //
    DELETE reporter;
    DELETE keywords;
    DELETE summary;
    DELETE site;
    reporter = 0;
    keywords = 0;
    summary  = 0;
    site     = 0;
    severity = 0;  // Just nullify this; it is a static char.
}

/*
** commands_screen - display screen of problem commands. We assume
**                   that the caller will be setting the modeline.
**                   Needs at least five before anything useful is displayed.
*/

static void commands_screen()
{
    clear_display_area();

    enter_standout_mode();
    display_string("Commands");
    end_standout_mode();
    cursor_wrap();

    //
    // Display as many commands as will fit on screen starting in third row.
    //
    for (int i = 0; i < NCommands() && i < rows() - 4; i++)
    {
        (void)fputs("  ", stdout);
        enter_standout_mode();
        putchar(Commands[i][0]);  // first char of command in bold
        end_standout_mode();
        display_string(&Commands[i][1], 0, 3);
    }
}

/*
** redisplay_commands - redisplay the commands screen and modeline.
**                      This gets called on a SIGTSTP or SIGWINCH.
*/

static void redisplay_commands() { commands_screen(); update_modeline(); }

/*
** update_existing_problem - update an existing problem entry.
*/

static void update_existing_problem(datum &key, Modified how, 
				    char *newitem)
{
    message("Invoking your editor ...");

    //
    // Build tmp file into which the data will be edited by user.
    //
    const char *file = temporary_file();

    invoke_editor(file);

    //
    // Merge old data with the new.
    //
    time_t t = time(0);
    char *updated = ctime(&t);

    lock_database(); /* lock refresh in case someone else works in this */
    open_database(GDBM_READER);
    datum data = gdbm_fetch(GdbmFile, key);
    gdbm_close(GdbmFile);

    switch ( how )
    {
        int i;
	datum olddata;

    case KEYWORDMOD:
    
	//
	// Make new data datum.
	//
	olddata = data;
	char *tmp = olddata.dptr;

	//
	// Find old Keyword field -- Fields\[5\].
	//
	for (i = 0; i < 5; i++)
	{
	    tmp  = strchr(tmp, '\n');
	    tmp += 1;		// step past newline
	}
	tmp += max_field_length() + 1; // the Keywords field data
	data.dsize = olddata.dsize + (int)strlen(newitem) -
	  (strchr(tmp, '\n') - tmp);
	data.dptr = new char[data.dsize + 2];
	*tmp++ = 0;		// stringify data.dptr
	(void)strcpy(data.dptr, olddata.dptr); // data preceding Keywords field
	(void)strcat(data.dptr, newitem); // new keywords
	(void)strcat(data.dptr, strchr(tmp, '\n')); // following data
	free(olddata.dptr);

	break;

    case SEVERITYMOD:

	//
	// Find old Severity field -- Fields\[9\].
	//
	tmp = data.dptr;
	for (i = 0; i < 9; i++)
	{
	    tmp  = strchr(tmp, '\n');
	    tmp += 1;                   // step past newline
	}
	tmp += max_field_length() + 1;  // the Severity field data
	*tmp = *newitem;             // set new Severity

	break;

    case TRANSFER:

	//
	// Build new problem header -- must update Area field.
	//
	i = int (strlen(newitem) - strlen(CurrentArea()));
	olddata = data;
	data.dsize = olddata.dsize + i;
	data.dptr  = new char[data.dsize];
	strncpy(data.dptr, olddata.dptr, max_field_length() + 1);
	*(data.dptr + max_field_length() + 1) = 0;
	strcat(data.dptr, newitem);
	strcat(data.dptr, strchr(olddata.dptr, '\n'));
	free(olddata.dptr);

	break;

    }

    String separator("\n****** ");
    separator += How[how];
    separator += " by `";
    separator += username();
    separator += "' on ";
    separator += updated;
    separator += "\n";

    int fsize = filesize(file);

    //
    // Is last character in problem a newline?
    //
    int add_newline = data.dptr[data.dsize-2] != '\n';

    datum newdata;

    newdata.dsize = int(separator.length() + data.dsize + fsize + add_newline);
    newdata.dptr  = new char[newdata.dsize];
    (void)strcpy(newdata.dptr, data.dptr);            // the old data
    if (add_newline) (void)strcat(newdata.dptr, "\n"); // add terminal newline
    (void)strcat(newdata.dptr, separator);            // the separator

    //
    // The new data.  Make sure this read only fails on a real
    // error -- block SIGTSTP and SIGWINCH.
    //
    block_tstp_and_winch();

    int fd;
    if ((fd = open(file, O_RDONLY)) < 0)
        error("file %s, line %d, open(%s, O_RDONLY) failed",
              __FILE__, __LINE__, file);

    if (read(fd,&newdata.dptr[separator.length()+data.dsize-1],fsize) == -1)
        error("file %s, line %d, read() failed", __FILE__, __LINE__);

    unblock_tstp_and_winch();
    (void)close(fd);
    (void)unlink(file);

    //
    // Always update the Updated field -- Fields\[4\].
    //
    char *head = newdata.dptr;
    for (int i = 0; i < 4; i++)
    {
        // want to find head of fifth line
        head = strchr(head, '\n');
        head += 1; // step past the newline
    }
    int flen = max_field_length() + 1;
    head += flen;  // skip to the data in Fields\[4\]
    for (i = 0; i < 25; i++) head[i] = updated[i];

    //
    // Update the Status field only on closes and reopens.
    //
    if (how == CLOSED || how == REOPENED)
    {
        char *field = (how == CLOSED ? "closed" : "open  ");
        for (i = 0; i < 3; i++)
        {
            //
            // Skip three more lines.
            //
            head = strchr(head, '\n');
            head += 1;           // step past the newline
        }
        head += flen;            // step to the data in Fields\[7\]
        for (i = 0; i < 6; i++)  // StatDim == 6 in log_new_problem
            head[i] = field[i];
    }

    String msg("Really do the ");
    switch (how)
    {
      case CLOSED:      msg += "close (y|n)?" ; break;
      case REOPENED:    msg += "reopen (y|n)?"; break;
      case APPENDED:    msg += "append (y|n)?"; break;
      case KEYWORDMOD:  msg += "keyword modification (y|n)?"; break;
      case SEVERITYMOD: msg += "severity modification (y|n)?"; break;
      case TRANSFER:    msg += "problem transfer (y|n)?"; break;
      default:          error("file %s, line %d, illegal case in switch()",
                              __FILE__, __LINE__);
    }

    if (yes_or_no(msg, redisplay_commands, Yes, 1))
    {
        update_database(key, newdata, how);
        update_subscribers(newdata, key.dptr, how, separator.length() +
                           data.dsize - 1);
    }

    free(data.dptr);
    DELETE newdata.dptr;
}
/*
**
** problem - a problem database manager
**
** Written in C++ using the termcap library for screen management
** and GDBM as the database library.
**
** problem.C is made by cating together problem1.C and problem2.C
**
** problem2.C problem2.C 1.30   Delta\'d: 17:54:17 11/9/92   Mike Lijewski, CNSF
**
** Copyright \(c\) 1991, 1992 Cornell University
** All rights reserved.
**
** Redistribution and use in source and binary forms are permitted
** provided that: \(1\) source distributions retain this entire copyright
** notice and comment, and \(2\) distributions including binaries display
** the following acknowledgement:  ``This product includes software
** developed by Cornell University\'\' in the documentation or other
** materials provided with the distribution and in all advertising
** materials mentioning features or use of this software. Neither the
** name of the University nor the names of its contributors may be used
** to endorse or promote products derived from this software without
** specific prior written permission.
**
** THIS SOFTWARE IS PROVIDED ``AS IS\'\' AND WITHOUT ANY EXPRESS OR
** IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
** WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
*/

/*
** append_to_problem - append to an already existing problem.  Returns
**                     false if the problem does not exist.  This indicates
**                     that we do not need to redisplay the command list.
**                     Returns true if we need to redisplay the command
**                     list.  If number, which defaults to zero, is
**                     nonzero, we use that number instead of prompting.
*/

int append_to_problem(const char *number)
{
    datum key;
    key.dptr  = number ? (char *) number : prompt("Problem # --> ",
                                                  redisplay_commands);
    key.dsize = int (strlen(key.dptr) + 1);
    if (!database_exists())
    {
        ding();
        message("There is no database for problem area `%' ", CurrentArea());
        if (!number)
        {
            DELETE key.dptr;
            sleep(2);
        }
        return 0;
    }

    open_database(GDBM_READER);
    datum data = gdbm_fetch(GdbmFile, key);
    gdbm_close(GdbmFile);
    if (!data.dptr)
    {
        ding();
        message("There is no problem # `%' ", key.dptr);
        if (!number)
        {
            DELETE key.dptr;
            sleep(2);
        }
        return 0;  // only the message area has been corrupted
    }

    free(data.dptr);

    //
    // The problem exists.
    //
    update_existing_problem(key, APPENDED, NULL);
    update_modeline();  // redisplay the previous modeline

    if (!number) DELETE key.dptr;

    return 1;     // must refresh the screen
}

/*
** subscribe_to_area - put user on interested parties list for current area.
**                     Exits on error
*/

static void subscribe_to_area()
{
    int mailfd = open_maillist_file();
    const int chunksize = 20;
    const int linelen   = 10;
    char **users = new char*[chunksize];
    FILE *mailfp = fdopen(mailfd, "r+");
    if (!mailfp)
        error("file %s, line %d, fdopen() failed", __FILE__, __LINE__);
    int nusers = read_file(mailfp, users, chunksize, linelen);
    if (nusers < 0)
        error("file %s, line %d, error reading %s maillist",
              __FILE__, __LINE__, CurrentArea());

    //
    // Is user already subscribed?
    //
    const char *user = mailname();
    for (int i = 0, subscribed = 0; i < nusers; i++)
        if (strcmp(users[i], user) == 0)
        {
            subscribed = 1; break;
        }

    for (i = 0; i < nusers; i++) DELETE users[i];
    DELETE users;

    if (subscribed)
    {
        (void)close(mailfd); (void)fclose(mailfp);
        ding();
        message("You already subscribe to the `%' area ", CurrentArea());
        sleep(2);
        return;
    }

    //
    // Lock on sequence file when updating the maillist.
    //
    block_tstp_and_winch(); // block SIGTSTP and WINCH
    lock_database();

    //
    // Seek to end of maillist file.
    //
    if (fseek(mailfp, 0, SEEK_END))
        error("file %s, line %d, fseek() failed", __FILE__, __LINE__);
    (void)fprintf(mailfp, "%s\n", user); // add them

    (void)fclose(mailfp);
    (void) close(mailfd);

    unlock_database();
    unblock_tstp_and_winch();

    message("You now subscribe to the `%' area ", CurrentArea());
    sleep(2);
}

/*
** unsubscribe_from_area - unsubscribe from the current area.  Exits on error.
*/

static void unsubscribe_from_area()
{
    int mailfd   = open_maillist_file();
    FILE *mailfp = fdopen(mailfd, "r+");
    if (!mailfp)
        error("file %s, line %d, fdopen() failed", __FILE__, __LINE__);

    const int chunksize = 20;
    const int linelen   = 10;
    char **users = new char*[chunksize];
    int nusers   = read_file(mailfp, users, chunksize, linelen);
    if (nusers < 0)
        error("file %s, line %d, error reading %s maillist",
              __FILE__, __LINE__, CurrentArea());

    //
    // Are they already subscribed?
    //
    const char *user = mailname();
    for (int i = 0, subscribed = 0; i < nusers; i++)
        if (strcmp(users[i], user) == 0)
        {
            subscribed = 1;
            break;
        }

    for (i = 0; i < nusers; i++) DELETE users[i];
    DELETE users;

    if (!subscribed)
    {
        (void)fclose(mailfp);
        (void)close(mailfd);
        ding();
        message("You're not a subscribee of the `%' area ", CurrentArea());
        sleep(2);
        return;
    }

    //
    // They subscribe - remove them.
    //
    if (fseek(mailfp, 0, SEEK_SET))  // seek to beginning of file
        error("file %s, line %d, fseek() failed", __FILE__, __LINE__);

    //
    // First, build a tmp file for new maillist.
    //
    char *tmpfile = new char[256];
    (void)sprintf(tmpfile, "%s%d.tmp", mail_list_prefix(), getpid());
    int tmpfd = open(tmpfile, O_RDWR|O_CREAT, 0644);

    if (tmpfd < 0)
        error("file %s, line %d, open(%s) failed",
              __FILE__, __LINE__, tmpfile);

    FILE *tmpfp = fdopen(tmpfd, "w+");
    if (!tmpfp)
        error("file %s, line %d, fdopen() failed", __FILE__, __LINE__);

    //
    // We use the sequence_file for locking.
    //
    block_tstp_and_winch(); // block SIGTSTP and WINCH
    lock_database();

    for (char *line = fgetline(mailfp, 10); line; line = fgetline(mailfp, 10))
    {
        if (strcmp(line, user) == 0)
        {
            DELETE line;
            continue;
        }
        (void)fprintf(tmpfp, "%s\n", line);
        DELETE line;
    }

    if (feof(mailfp) && !ferror(mailfp))
    {
        //
        // Alternate maillist correctly constructed.
        //
        (void)fclose(tmpfp);
        (void)fclose(mailfp);
        (void) close(tmpfd);
        (void) close(mailfd);

        //
        // Remove old maillist.
        //
        String maillist(mail_list_prefix());
        maillist += CurrentArea();
        maillist += MailListSuffix;

        if (unlink((const char *)maillist) < 0)
            error("file %s, line %d, unlink(%s) failed",
                  __FILE__, __LINE__, (const char *)maillist);
        if (rename((const char *)tmpfile, (const char *)maillist) < 0)
            error("file %s, line %d, rename(%s, %s) failed", __FILE__,
                  __LINE__, (const char *)tmpfile, (const char *)maillist);

        message("You've been unsubscribed from the `%' area ", CurrentArea());
    }
    else
    {
        (void)fclose(tmpfp);
        (void)fclose(mailfp);
        (void) close(tmpfd);
        (void) close(mailfd);
        ding();
        message("Problem updating maillist -- update not committed ");
    }

    unlock_database();
    unblock_tstp_and_winch();

    DELETE tmpfile;

    sleep(2);
}

/*
** examine_problem - pages through a problem in current area.  Returns one
**                   if the problem exists, otherwise zero.  If number is
**                   nonzero -- it defaults to zero --, we use that number
**                   instead of prompting for one.
*/

int examine_problem(const char *number)
{
    if (!database_exists())
    {
        ding();
        message("There is no database for problem area `%' ", CurrentArea());
        if (!number) sleep(2);
        return 0;
    }

    datum key;
    key.dptr = number ? (char *) number : prompt("Problem # --> ",
                                                 redisplay_commands);
    key.dsize = int(strlen(key.dptr)) + 1;
    open_database(GDBM_READER);
    datum data = gdbm_fetch(GdbmFile, key);
    gdbm_close(GdbmFile);

    if (!data.dptr)
    {
        ding();
        message("There is no problem # % ", key.dptr);
        if (!number)
        {
            DELETE key.dptr;
            sleep(2);
        }
        return 0;
    }

    //
    // Build tmp file to pass to pager.
    //
    const char *file = temporary_file();

    //
    // Write the problem to the file.  Remember that the problem is
    // stored in the database as a string, with the null character counted
    // as part of its length.
    //
    int fd;
    if ((fd = open(file, O_RDWR)) < 0)
        error("file %s, line %d, open(%s, O_RDWR) failed",
              __FILE__, __LINE__, file);
    if (write(fd, data.dptr, data.dsize - 1) != data.dsize - 1)
        error("file %s, line %d, write() failed", __FILE__, __LINE__);

#ifdef NOLESS
    const char *pager = getenv("PAGER");
    if (pager == 0) pager = "more";

    String argstring(pager);
    argstring += " ";
    argstring += file;

    //
    // We tokenize because the user could have PAGER
    // set to "less -i -Q -M", which execvp would not
    // recognize as a legal filename.
    //
    const char **args = tokenize(argstring, " \t");
    pager = args[0];
#else
    //
    // Use "less".
    //
    const char *prefix = "-PProblem #";
    const char *suffix = " -- line %lb?L/%L. byte %bB?s/%s. ?e(END) :?pB%pB\\% ..-- q (quit) H (help)";

    String prompt(prefix);
    prompt += key.dptr;
    prompt += suffix;

    const char *pager = "less";
    const char *args[5];
    args[0] = pager;
    args[1] = "-d";
    args[2] = prompt;
    args[3] = file;
    args[4] = 0;
#endif /*NOLESS*/

#ifdef NOLESS
    //
    // We prompt for a keypress before returning from the pager, so
    // that we do not immediately return when we hit last page.
    // This is because some pagers -- "more" being one -- exit when
    // you get to the last page, which makes perusal of the last page
    // difficult.  We do not have to do this with "less", since it is exited
    // deliberately by typing a q.
    //
    if (!execute(pager, args, 1))
#else
    if (!execute(pager, args))
#endif /*NOLESS*/
    {
        (void)unlink(file);
        error("file %s, line %d, problem running pager: `%s'",
              __FILE__, __LINE__, pager);
    }

    (void)close(fd);
    (void)unlink(file);

    update_modeline();  // redisplay previous modeline

    free(data.dptr);
    if (!number) DELETE key.dptr;

    return 1;           // must update screen
}

/*
** summary_info - returns a string in newd space of the form:
**
**                     prob# status severity last-activity-date summary
**
**                sort_by_date "knows" the format of the lines output
**                by this function, so if you change the format, you need
**                to change sort_by_date also.
*/

char *summary_info(datum &data)
{
    const char *tail = data.dptr;

    //
    // Updated is Fields\[4\]
    //
    for (int i = 0; i < 4; i++)
    {
        tail = strchr(tail, '\n');
        tail += 1;                  // step past the newline
    }
    tail += max_field_length() + 1; // this is the Updated line
    const char *updated = tail + 4; // do not output the day of the week
    tail = strchr(tail, '\n');      // end of Updated line
    int updatedLen = tail - updated;

    //
    // Keywords is Fields\[5\] - just skip over it for now
    //
    tail += 1;                      // step past the newline
    tail = strchr(tail, '\n');      // end of Keywords line

    //
    // Summary is Fields\[6\]
    //
    tail += 1;                      // step past the newline
    tail += max_field_length() + 1; // this is the Summary line
    const char *summary = tail;
    tail = strchr(tail, '\n');      // end of Summary line
    int summaryLen = tail - summary;

    //
    // Status Field\[7\]
    //
    tail += 1;                      // step past the newline
    tail += max_field_length() + 1; // this is the Status line
    const char *status = tail;
    tail = strchr(tail, '\n');      // end of Status line
    int statusLen = tail - status;

    //
    // Severity is Field\[9\]
    //
    tail += 1;                      // step past the newline
    tail = strchr(tail, '\n');      // end of Site line
    tail += 1;                      // step past the newline
    tail += max_field_length() + 1; // this is the Severity line
    const char severity = *tail;
    tail = strchr(tail, '\n');      // end of Severity line

    //
    // Problem # is Fields\[10\]
    //
    tail += 1;                      // step past the newline
    tail += max_field_length() + 1; // this is the prob # line
    const char *number = tail;
    //
    // Find the end of the problem # field - probably contains a space.
    //
    tail = strchr(tail, '\n');
    while (*(tail - 1) == ' ') tail--; // strip off trailing spaces
    int numberLen = tail - number;

    const int numberFieldLen = 5;      // min width of Prob # field in summary
    int len = numberLen > numberFieldLen ? numberLen : numberFieldLen;
    char *line = new char[updatedLen + summaryLen + statusLen + len + 4];


    *line = 0; // stringify

    if (numberLen < numberFieldLen)
    {
        //
        // Pad the length to numberFieldLen by prepending spaces.
        //
        int nSpaces = numberFieldLen - numberLen;
        for (int i = 0; i < nSpaces; i++)
            *(line + i) = ' ';
        *(line + nSpaces) = 0;  // restringify
    }

    (void)strncat(line, number, numberLen);
    line[len] = 0;
    (void)strcat(line, " ");
    (void)strncat(line, status, statusLen);
    line[len += statusLen + 1] = 0;

    //
    // if status == "open", output the severity also
    //
    if (line[len - 1] == ' ') line[len - 1] = severity;

    (void)strcat(line, " ");
    (void)strncat(line, updated, updatedLen);
    line[len += updatedLen + 1] = 0;
    (void)strcat(line, " ");
    (void)strncat(line, summary, summaryLen);
    line[len += summaryLen + 1] = 0;

    return line;
}

/*
** lsign - returns -1, 0 or 1 depending on the sign of the argument
*/

inline int lsign(long arg)
{
    return arg == 0 ? 0 : (arg > 0 ? 1 : -1);
}

/*
** sort_by_date - the comparison function passed to qsort used when
**                sorting listing lines by date.
*/

static int sort_by_date(const void *a, const void *b)
{
    char *tmpa = *(char **)a;
    char *tmpb = *(char **)b;

    while (*tmpa == ' ') tmpa++;        // step over any spaces preceding Prob#
    while (*tmpb == ' ') tmpb++;        // ditto

    tmpa = strchr(tmpa, ' ') + 1;       // step onto first character of Status
    tmpb = strchr(tmpb, ' ') + 1;       // ditto

    if (strncmp(tmpa, tmpb, 4))
        //
        // Sort "open" before "closed".
        //
        return -strncmp(tmpa, tmpb, 4);
    else if (strncmp(tmpa, tmpb, 6))
        //
        // Sort by severity.
        //
        return strncmp(tmpa, tmpb, 6);
    else
    {
        //
        // Lastly, sort by most recent first.
        //
        tmpa += 7;
        tmpb += 7;
        return lsign(seconds_in_date(tmpb) - seconds_in_date(tmpa));
    }
}

/*
** view_summary_lines - page through data from problem headers in current area.
**                      Returns one if the database exists, else zero.
*/

static int view_summary_lines()
{
    if (!database_exists())
    {
        ding();
        message("There is no database for problem area `%' ", CurrentArea());
        sleep(2);
        return 0;
    }

    open_database(GDBM_READER);
    datum key = gdbm_firstkey(GdbmFile);
    if (!key.dptr)
    {
        String msg("Area `");
        msg += CurrentArea();
        msg += "' appears to be empty ";

        ding();
        message(msg);
        sleep(2);

        gdbm_close(GdbmFile);
        return 0;
    }

    DList summaryLines;  // listing of problem summaries
    datum data, tmp;
    const int chunksize = 100;
    int size = chunksize;
    int pos = 0;
    char **lines = new char*[size];

    message("Reading problems ... ");
    while (1)
    {
        data = gdbm_fetch(GdbmFile, key);
        lines[pos++] = summary_info(data);
        if (pos == size)
        {
            //
            // Grow lines.
            //
            char **newspace = new char*[size += chunksize];
            for (int i = 0; i < pos; i++) newspace[i] = lines[i];
            DELETE lines;
            lines = newspace;
        }
        free(data.dptr);
        tmp = gdbm_nextkey(GdbmFile, key);
        free(key.dptr);
        key = tmp;
        if (!key.dptr) break;
    }
    gdbm_close(GdbmFile);
    message("Reading problems ... done");

    //
    // Sort lines "most recently updated" first.
    //
    qsort(lines, pos, sizeof(char**), sort_by_date);
    for (int i = 0; i < pos; i++)
        summaryLines.add(new DLink((char **)&lines[i]));

    DELETE lines;

    initialize_lister(&summaryLines);
    initial_listing(&summaryLines);

    String suffix(CurrentArea());
    suffix += " ---- q (quit) H (help)";

    update_modeline(ModelinePrefix, suffix);

    summaryLines.saveYXPos(0, goal_column(&summaryLines));
    if (summaryLines.currLine()->length() > columns())
        leftshift_current_line(&summaryLines);
    else
        move_cursor(summaryLines.savedYPos(), summaryLines.savedXPos());
    synch_display();

    lister_cmd_loop(&summaryLines);

    return 1;
}

/*
** close_problem - close an existing problem in current area.  Returns one if
**                 we did the close, zero otherwise.  Only a database
**                 administrator or the original logger can close a problem.
**                 number, which defaults to null, is used if not null,
**                 instead of prompting for the number.  Also, only an open
**                 problem can be closed.
*/

int close_problem(const char *number)
{
    if (!database_exists())
    {
        ding();
        message("There is no database for problem area `%' ", CurrentArea());
        if (!number) sleep(2);
        return 0;
    }
    datum key;
    key.dptr  = number ? (char *) number : prompt("Problem # --> ",
                                                  redisplay_commands);
    key.dsize = int(strlen(key.dptr)) + 1;

    open_database(GDBM_READER);
    datum data = gdbm_fetch(GdbmFile, key);
    gdbm_close(GdbmFile);

    if (!data.dptr)
    {
        ding();
        message("There is no problem # % ", key.dptr);
        if (!number)
        {
            DELETE key.dptr;
            sleep(2);
        }
        return 0;
    }

    //
    // Are we authorized to close the problem?
    //
    char *tmp = strchr(data.dptr, '\n');  // logger is Fields\[1\]
    tmp += 1;                             // step past the newline
    tmp += max_field_length() + 1;        // the logger
    int llen  = strchr(tmp, '\n') - tmp;  // # of chars in logger
    const char *user = username();

    if ((llen != strlen(user) || strncmp(user, tmp, llen)) &&
        getuid() != geteuid())
    {
        ding();
        message("You're not authorized to close problem # % ", key.dptr);
        if (!number)
        {
            DELETE key.dptr;
            sleep(2);
        }
        free(data.dptr);
        return 0;
    }

    //
    // Is the problem open?
    //
    for (int i = 0; i < 6; i++)
    {
        // status is Fields\[7\]
        tmp = strchr(tmp, '\n');
        tmp += 1;  // step past newline
    }
    if (strncmp(tmp + max_field_length() + 1, "open", 4))
    {
        ding();
        message("Only open problems can be closed ");
        if (!number)
        {
            DELETE key.dptr;
            sleep(2);
        }
        free(data.dptr);
        return 0;
    }

    update_existing_problem(key, CLOSED, NULL);
    update_modeline();  // redisplay previous modeline

    free(data.dptr);
    if (!number) DELETE key.dptr;

    return 1;
}

/*
** reopen_problem - reopen a closed problem in current area.  Returns one if
**                  we did the reopen, zero otherwise.  Only a database
**                  administrator or the original logger can reopen a problem.
**                  number, which defaults to null, is used if not null,
**                  instead of prompting for the number.
*/

int reopen_problem(const char *number)
{
    if (!database_exists())
    {
        ding();
        message("There is no database for problem area `%' ", CurrentArea());
        if (!number) sleep(2);
        return 0;
    }

    datum key;
    key.dptr  = number ? (char *) number : prompt("Problem # --> ",
                                                  redisplay_commands);
    key.dsize = int(strlen(key.dptr)) + 1;

    open_database(GDBM_READER);
    datum data = gdbm_fetch(GdbmFile, key);
    gdbm_close(GdbmFile);

    if (!data.dptr)
    {
        ding();
        message("There is no problem # % ", key.dptr);
        if (!number)
        {
            DELETE key.dptr;
            sleep(2);
        }
        return 0;
    }

    //
    // Are we authorized to reopen the problem?
    //
    char *tmp = strchr(data.dptr, '\n');  // logger is Fields\[1\]
    tmp += 1;                             // step past the newline
    tmp += max_field_length() + 1;        // the logger
    int llen  = strchr(tmp, '\n') - tmp;  // # of chars in logger
    const char *user = username();

    if ((llen != strlen(user) || strncmp(user, tmp, llen)) &&
        getuid() != geteuid())
    {
        ding();
        message("You're not authorized to close problem # % ", key.dptr);
        if (!number)
        {
            DELETE key.dptr;
            sleep(2);
        }
        free(data.dptr);
        return 0;
    }

    //
    // Only closed problems can be opened.
    //
    for (int i = 0; i < 6; i++)
    {
        // status is Fields\[7\]
        tmp = strchr(tmp, '\n');
        tmp += 1;  // step past newline
    }
    if (strncmp(tmp + max_field_length() + 1, "closed", 6))
    {
        ding();
        message("Only closed problems can be reopened ");
        if (!number)
        {
            DELETE key.dptr;
            sleep(2);
        }
        free(data.dptr);
        return 0;
    }

    update_existing_problem(key, REOPENED, NULL);
    update_modeline();  // redisplay previous modeline

    free(data.dptr);
    if (!number) DELETE key.dptr;

    return 1;
}

/*
** delete_problem - delete the problem from current area.
**                  This is strictly for the database administrators.
**                  If number is non-null, use it instead of prompting
**                  for the problem number.  Returns 1 if the problem
**                  was deleted, 0 otherwise.
*/

int delete_problem(const char *number)
{
    int del = 0;  // was delete successful?

    if (getuid() != geteuid())
    {
        ding();
        message("Only database administrators can to delete problems ");
        if (!number) sleep(2);
        return 0;
    }

    if (!database_exists())
    {
        ding();
        message("There is no database for problem area `%' ", CurrentArea());
        if (!number) sleep(2);
        return 0;
    }

    datum key;
    key.dptr  = number ? (char *) number : prompt("Problem # --> ",
                                                  redisplay_commands);
    key.dsize = int(strlen(key.dptr)) + 1;

    open_database(GDBM_READER);
    datum data = gdbm_fetch(GdbmFile, key);
    gdbm_close(GdbmFile);

    if (!data.dptr)
    {
        ding();
        message("There is no problem # % ", key.dptr);
        if (!number)
        {
            DELETE key.dptr;
            sleep(2);
        }
        return 0;
    }

    //
    // The problem exists; delete it.
    //
    const char *msg = "Do you really want to delete this problem (n|y)? ";
    if (del = yes_or_no(msg, redisplay_commands, No, 0))
        update_database(key, data, DELETED);
    free(data.dptr);
    if (!number) DELETE key.dptr;

    return del;
}

/*
** reorganize_database - reorganize the database for current area
*/

void reorganize_database(int dodelay)
{
    if (getuid() != geteuid())
    {
        ding();
        message("Only database administrators can reorganize the database ");
        if (dodelay) sleep(2);
        return;
    }
    if (!database_exists())
    {
        ding();
        message("There is no database for problem area `%' ", CurrentArea());
        if (dodelay) sleep(2);
        return;
    }

    const char *msg = "Do you really want to reorganize this database (n|y)? ";
    if (yes_or_no(msg, redisplay_commands, No, 0))
    {
        datum key, data;  // just placeholders here
        update_database(key, data, REORGANIZED);
    }
}

/*
** search - prompt for pattern -- regexp -- and display the matches.
**          Returns zero if we only need to redisplay
**          the prompt, else returns one.
*/

//
// Ways to search -- over full problem text or just the header
//
enum SearchMethod { FULLTEXT, HEADER };

static int search(const SearchMethod how)
{
    if (!database_exists())
    {
        ding();
        message("There is no database for problem area `%' ", CurrentArea());
        sleep(2);
        return 0;
    }

    char *keywords = prompt("Search for --> ", redisplay_commands);
    const char **list = tokenize(keywords, " ,\t");
    DELETE keywords;

    if (!list[0]) return 0;  // no words to search for

    //
    // Build a regular expression to search for.
    // We want to build a pattern of the form:
    //
    //   word1|word2|word3|word4 ...
    //
    int len = 0;                          // total length of all words in list
    for (int i = 0; list[i]; i++)
        len += (int) strlen(list[i]) + 1; // plus one for the |
    char *line = new char[len + 1];
    line[0] = 0;                          // make it into a valid string

    //
    // Are each of the words individually a valid regexp?
    //
    regexp *regex;
    for (i = 0; list[i]; i++)
    {
        if ((regex = regcomp(list[i])) == 0)
        {
            String msg("`");
            msg += list[i];
            msg += "' isn't a valid regex: ";
            msg += REerror;
            msg +=  ", skipping ... ";

            ding();
            message(msg);
            sleep(2);
            DELETE line;
        }
        else
        {
            (void)strcat(line, list[i]);
            (void)strcat(line, "|");
        }
        DELETE (char *) regex;
    }

    //
    // Remove any trailing |s.
    //
    if (line[strlen(line) - 1] == '|') line[strlen(line) - 1] = 0;
    if (strlen(line) == 0)
    {
        ding();
        message("No valid regular expressions among your keywords ");
        sleep(2);
        DELETE line;
        return 0;
    }
    if ((regex = regcomp(line)) == 0)
    {
        ding();
        message("regcomp() failed: %", REerror);
        sleep(2);
        DELETE line;
        return 0;
    }

    open_database(GDBM_READER);
    datum key = gdbm_firstkey(GdbmFile);
    if (!key.dptr)
    {
        String msg("Area `");
        msg += CurrentArea();
        msg += "' appears to be empty ";

        ding();
        message(msg);
        sleep(2);
        DELETE line;
        DELETE (char *) regex;
        return 0;
    }

    DList summaryLines;  // listing of problem summaries
    datum data, tmp;
    const int chunksize = 100;
    int size = chunksize, pos = 0;
    char **lines = new char*[size];

    message("Reading problems ... ");

    while (1)
    {
        data = gdbm_fetch(GdbmFile, key);
        switch (how)
        {
          case HEADER:
          {
              //
              // Search only the problem header.
              //
              char *tail = data.dptr;
              for (int i = 0; i < NFields(); i++)
              {
                  tail = strchr(tail, '\n');
                  tail += 1; // step past the newline
              }
              *tail = 0;     // treat the header as one long string
              if (regexec(regex, data.dptr)) lines[pos++] = summary_info(data);
          }
              break;
          case FULLTEXT:
              //
              // Search over full problem text.
              //
              if (regexec(regex, data.dptr)) lines[pos++] = summary_info(data);
              break;
          default: error("file %s, line %d, illegal case in switch()",
                           __FILE__, __LINE__);
        }
        if (pos == size)
        {
            //
            // Grow lines.
            //
            char **newspace = new char*[size += chunksize];
            for (int i = 0; i < pos; i++) newspace[i] = lines[i];
            DELETE lines;
            lines = newspace;
        }
        free(data.dptr);
        tmp = gdbm_nextkey(GdbmFile, key);
        free(key.dptr);
        key = tmp;
        if (!key.dptr) break;
    }

    gdbm_close(GdbmFile);
    message("Reading problems ... done");
    DELETE (char *) regex;

    //
    // Sort lines.
    //
    qsort(lines, pos, sizeof(char**), sort_by_date);
    for (i = 0; i < pos; i++)
        summaryLines.add(new DLink((char **)&lines[i]));
    DELETE lines;

    //
    // Are there any problem summaries to peruse?
    //
    if (!summaryLines.nelems())
    {
         ding();
         message("No matches for regex `%' ", line);
         sleep(2);
         DELETE line;
         return 0;
     }

    initialize_lister(&summaryLines);
    initial_listing(&summaryLines);

    String suffix(CurrentArea());
    suffix += " (regex: ";
    suffix += line;
    suffix += ") ---- q (quit) H (help)";

    update_modeline(ModelinePrefix, suffix);
    DELETE line;

    summaryLines.saveYXPos(0, goal_column(&summaryLines));
    if (summaryLines.currLine()->length() > columns())
        leftshift_current_line(&summaryLines);
    else
        move_cursor(summaryLines.savedYPos(), summaryLines.savedXPos());
    synch_display();

    lister_cmd_loop(&summaryLines);

    return 1;
}

//
// the header data -- needs to be shared between display_header and
//                    modify_keywords so that we can call prompt using
//                    display_header as its second argument.
//
static datum header_data;

/*
** display_header - put up the header of the problem in data.
*/

static void display_header()
{
    clear_display_area();

    enter_standout_mode();
    for (int i = 0; i < NFields() && i < rows() - 2; i++)
        display_string(Fields[i]);
    end_standout_mode();

    int flen   = max_field_length() + 1;
    char *tmp1 = header_data.dptr, *tmp2;

    for (i = 0; i < NFields() && i < rows() - 2; i++)
    {
        tmp2  = strchr(tmp1, '\n');
        *tmp2 = 0;        // stringify
        move_cursor(i, flen);
        display_string(tmp1 + flen);
        *tmp2 = '\n';     // unstringify
        tmp1  = tmp2 + 1; // step past newline to next Field
    }
}

/*
** modify_keywords - allows the problem owner or the administrator
**                   to change the Keyword field.
*/

int modify_keywords(const char *number)
{
    if (!database_exists())
    {
        ding();
        message("There is no database for problem area `%' ", CurrentArea());
        if (!number) sleep(2);
        return 0;
    }

    datum key;
    key.dptr  = number ? (char *) number : prompt("Problem # --> ",
                                                  redisplay_commands);
    key.dsize = int(strlen(key.dptr)) + 1;

    open_database(GDBM_READER);
    datum data = gdbm_fetch(GdbmFile, key);
    gdbm_close(GdbmFile);

    if (!data.dptr)
    {
        ding();
        message("There is no problem # % ", key.dptr);
        if (!number)
        {
            DELETE key.dptr;
            sleep(2);
        }
        return 0;
    }

    header_data = data;

    //
    // Are we authorized to modify the problem\'s keywords?
    //
    char *tmp = strchr(data.dptr, '\n');  // logger is Fields\[1\]
    tmp += 1;                             // step past the newline
    tmp += max_field_length() + 1;        // the logger
    int llen  = strchr(tmp, '\n') - tmp;  // # of chars in logger
    const char *user = username();

    if ((llen != strlen(user) || strncmp(user, tmp, llen)) &&
        getuid() != geteuid())
    {
        ding();
        message("You're not authorized to modify problem # %'s keywords ",
                key.dptr);
        if (!number)
        {
            DELETE key.dptr;
            sleep(2);
        }
        return 0;
    }

    //
    // Put up problem header.
    //
    String suffix(CurrentArea());
    suffix += "  # ";
    suffix += key.dptr;
    suffix += " (modifying keywords)";

    String old_modeline(current_modeline);

    update_modeline(ModelinePrefix, suffix);
    display_header();

    char *newkeywords = prompt("New keyword field --> ", display_header);
    free(data.dptr);

    update_existing_problem(key, KEYWORDMOD, newkeywords);
    update_modeline();                        // force comlete redisplay
    update_modeline(old_modeline);            // update to old version

    if (!number) DELETE key.dptr;
    DELETE newkeywords;

    return 1;
}

/*
** modify_severity - allows the problem owner or the administrator
**                   to change the Severity field.
*/

int modify_severity(const char *number)
{
    if (!database_exists())
    {
        ding();
        message("There is no database for problem area `%' ", CurrentArea());
        if (!number) sleep(2);
        return 0;
    }

    datum key;
    key.dptr  = number ? (char *) number : prompt("Problem # --> ",
                                                  redisplay_commands);
    key.dsize = int(strlen(key.dptr)) + 1;

    open_database(GDBM_READER);
    datum data = gdbm_fetch(GdbmFile, key);
    gdbm_close(GdbmFile);

    if (!data.dptr)
    {
        ding();
        message("There is no problem # % ", key.dptr);
        if (!number)
        {
            DELETE key.dptr;
            sleep(2);
        }
        return 0;
    }

    //
    // Are we authorized to modify the problem\'s severity?
    //
    char *tmp = strchr(data.dptr, '\n');  // logger is Fields\[1\]
    tmp += 1;                             // step past the newline
    tmp += max_field_length() + 1;        // the logger
    int llen  = strchr(tmp, '\n') - tmp;  // # of chars in logger
    const char *user = username();

    if ((llen != strlen(user) || strncmp(user, tmp, llen)) &&
        getuid() != geteuid())
    {
        ding();
        message("You're not authorized to modify problem # %'s severity ",
                key.dptr);
        if (!number)
        {
            DELETE key.dptr;
            sleep(2);
        }
        return 0;
    }

    char newSeverity = get_severity(display_header);

    //
    // Find old Severity field -- Fields\[9\].
    //
    tmp = data.dptr;
    for (int i = 0; i < 9; i++)
    {
        tmp  = strchr(tmp, '\n');
        tmp += 1;                   // step past newline
    }
    tmp += max_field_length() + 1;  // the Severity field data


    //
    // Are the severities actually different?
    //
    if (*tmp == newSeverity)
    {
        ding();
        message("The old and new severities are the same");
        if (!number)
        {
            DELETE key.dptr;
            sleep(2);
        }
        return 0;
    }

    *tmp = newSeverity;             // set new Severity
    free(data.dptr);

    update_existing_problem(key, SEVERITYMOD, &newSeverity);
    update_modeline();

    if (!number) DELETE key.dptr;

    return 1;
}

/*
** transfer_problem - allows the problem owner or the administrator
**                    to move the problem to another area.
*/

int transfer_problem(const char *number, char *area)
{
    if (!database_exists())
    {
        ding();
        message("There is no database for problem area `%' ", CurrentArea());
        if (!number) sleep(2);
        return 0;
    }

    datum key;
    key.dptr  = number ? (char *) number : prompt("Problem # --> ",
                                                  redisplay_commands);
    key.dsize = int(strlen(key.dptr)) + 1;

    open_database(GDBM_READER);
    datum data = gdbm_fetch(GdbmFile, key);
    gdbm_close(GdbmFile);

    if (!data.dptr)
    {
        ding();
        message("There is no problem # % ", key.dptr);
        if (!number)
        {
            DELETE key.dptr;
            sleep(2);
        }
        return 0;
    }

    header_data = data;

    //
    // Are we authorized to transfer the problem to another area?
    //
    char *tmp = strchr(data.dptr, '\n');  // logger is Fields\[1\]
    tmp += 1;                             // step past the newline
    tmp += max_field_length() + 1;        // the logger
    int llen  = strchr(tmp, '\n') - tmp;  // # of chars in logger
    const char *user = username();

    if ((llen != strlen(user) || strncmp(user, tmp, llen)) &&
        getuid() != geteuid())
    {
        ding();
        message("You're not authorized to transfer problem # %", key.dptr);
        if (!number)
        {
            DELETE key.dptr;
            free(data.dptr);
            sleep(2);
        }
        return 0;
    }

    //
    // Area is null when called from the command display.
    //
    if (!area) area = prompt("New Area --> ", redisplay_commands);

    //
    // Is area a valid area?
    //
    // This is guaranteed to be true if we get here when called
    // from the view window.
    //
    if (!is_area(area))
    {
        ding();
        message("`%' isn't a valid problem area", area);
        if (!number)
        {
            DELETE key.dptr;
            DELETE area;
            free(data.dptr);
            sleep(2);
        }
        return 0;
    }

    //
    // Is area really a new problem area?
    //
    // This is guaranteed to be true if we get here when called
    // from the view window.
    //
    if (strcmp(area, CurrentArea()) == 0)
    {
        ding();
        message("`%' is the same as the current area", area);
        if (!number)
        {
            DELETE key.dptr;
            DELETE area;
            free(data.dptr);
            sleep(2);
        }
        return 0;
    }

    update_existing_problem(key, TRANSFER, area);
    update_modeline(); // completely redisplay modeline

    if (!number)
    {
        DELETE key.dptr;
        DELETE area;
    }

    return 1;
}

/*
** problem - examine the area in problem
*/

static void problem(const char *area)
{
    const char *the_area = is_area(area) ? area : choose_problem_area();
    const char *helpmsg  = "The Available Commands:";
    setCurrentArea(the_area);
    commands_screen();
    String suffix(CurrentArea());
    suffix += " ---- q (quit) H (help)";

    update_modeline(ModelinePrefix, suffix);
    message("Your Choice --> ");

    char key;
    char refresh = 1; // need to redisplay command list?
    char redomsg = 1; // need to redisplay the message?
    while (1)
    {
        if (resumingAfterSuspension ||
#ifdef SIGWINCH
            windowSizeChanged       ||
#endif
            get_key(&key) < 0    || // assume only fails when errno==EINTR 
            key == KEY_CTL_L)
        {
#ifdef SIGWINCH
            if (windowSizeChanged)
            {
                windowSizeChanged = 0;
                adjust_window();
            }
#endif
            resumingAfterSuspension = 0;
            redisplay_commands();
            refresh = 0;
        }
        else
            switch(key)
            {
              case KEY_l:
                log_new_problem(); refresh = 1; break;
              case KEY_e:
                refresh = examine_problem(); break;
              case KEY_v:
                refresh = view_summary_lines(); break;
              case KEY_a:
                refresh = append_to_problem(); break;
              case KEY_s:
                subscribe_to_area(); refresh = 0; break;
              case KEY_u:
                unsubscribe_from_area(); refresh = 0; break;
              case KEY_c:
                refresh = close_problem(); break;
              case KEY_k:
                refresh = search(HEADER); break;
              case KEY_K:
                refresh = search(FULLTEXT); break;
              case KEY_M:
                refresh = modify_keywords(); break;
              case KEY_R:
                refresh = reopen_problem(); break;
              case KEY_P:
                refresh = modify_severity(); break;
              case KEY_T:
                refresh = transfer_problem(); break;
              case KEY_d:
                (void)delete_problem(); refresh = 0; break;
              case KEY_r:
                reorganize_database(); refresh = 0; break;
              case KEY_q:
                return;
              case KEY_H: case KEY_QM:
                help((const char **)&Commands[0], NCommands(), helpmsg);
                refresh = 1; break;
              default:
                ding(); redomsg = refresh = 0; break;
            }
        if (refresh)
        {
            commands_screen();
            update_modeline(ModelinePrefix, suffix);
        }
        if (redomsg) message("Your Choice --> ");
        redomsg = 1;
    }
}

main(int argc, char *argv[])
{
    char *LocalBase = getenv("PROBLEMBASE");
    
    if (LocalBase)
        HomeBase = LocalBase;

    setvbuf(stdout, NULL, _IOFBF, BUFSIZ);

    if (!isatty(0) || !isatty(1))
    {
        (void)fputs("stdin & stdout must be terminals\n", stderr);
        exit(1);
    }

    //
    // Usage: problem \[-v\] \[-d\] \[area1\] \[area2\] \[...\]
    //
    // We only accept the -d flag if it is the first argument.
    // It directs us to use a different directory as our HomeBase.
    // Regarding problem areas, we just ignore invalid ones.
    // The -v option just prints out the version string and exits.
    //
    argc--;
    argv++;
    if (argc && strcmp(*argv, "-v") == 0)
    {
        fputs(Version, stdout);
        exit(0);
    }
    else if (argc && strcmp(*argv, "-d") == 0)
    {
        argc--; argv++;
        if (!argc)
        {
            fputs("Ignoring `-d' flag - no corresponding directory.", stdout);
            sleep(2);
        }
        else
        {
            HomeBase = *argv;
            argc--;
            argv++;
        }
    }

    //
    // If you do not have SIGINTERRUPT then signals almost surely interrupt
    // read.  If you do have it, you will need this to ensure that signals
    // interrupt slow system calls -- we are only interested in read.
    //
#ifdef SIGINTERRUPT
#ifdef SIGTSTP
    if (siginterrupt(SIGTSTP, 1) < 0)
    {
        perror("siginterrupt(SIGTSTP)");
        exit(1);
    }
#endif
#ifdef SIGWINCH
    if (siginterrupt(SIGWINCH, 1) < 0)
    {
        perror("siginterrupt(SIGWINCH)");
        exit(1);
    }
#endif
#endif

    set_new_handler(free_store_exception);
    initialize();
    set_signals();

    if (!database_directory_exists())
        error("Database directory `%s' isn't accessible.", HomeBase);

    while (1) problem(*argv ? *argv++ : choose_problem_area());
}
