/*
** classes.h - the declarations of the classes used in problem
**
** classes.h classes.h 1.12   Delta'd: 18:12:47 11/8/92   Mike Lijewski, CNSF
**
** Copyright (c) 1991, 1992 Cornell University
** All rights reserved.
**
** Redistribution and use in source and binary forms are permitted
** provided that: (1) source distributions retain this entire copyright
** notice and comment, and (2) distributions including binaries display
** the following acknowledgement:  ``This product includes software
** developed by Cornell University'' in the documentation or other
** materials provided with the distribution and in all advertising
** materials mentioning features or use of this software. Neither the
** name of the University nor the names of its contributors may be used
** to endorse or promote products derived from this software without
** specific prior written permission.
**
** THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
** IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
** WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
*/

#ifndef __CLASSES_H
#define __CLASSES_H

#include <string.h>

//
// Deal with old and new types of delete
//
#ifdef  OLDDELETE
#define DELETE delete
#else
#define DELETE delete []
#endif

/////////////////////////////////////////////////////////////////////////////
// A Simple reference counted string class.  It is implemented as an
// Envelope-Letter abstaction with String being the envelope and StringRep
// being the letter.
/////////////////////////////////////////////////////////////////////////////

class String;
class SBHelper;

class StringRep {
  public:
    friend class String;
    friend class SBHelper;

    StringRep();
    StringRep(const char *s);
    StringRep(const char *s, int slen);
    StringRep(char** r, size_t slen) { rep = *r; len = slen; count = 1; }
    ~StringRep()                     { DELETE rep;                      }

    enum { chunksize = 50 };     // # of StringReps to allocate at a time
    static StringRep *freeList;  // we manage our own storage
    void *operator new(size_t size);
    void operator delete(void *object);

    int operator!=(const char *rhs) const;
    int operator==(const char *rhs) const;
    int operator!=(const StringRep& rhs) const;
    int operator==(const StringRep& rhs) const;

    String operator+(const String& s) const;

    size_t length() const { return len; }
  private:
    //
    // Disable these two methods
    //
    StringRep(const StringRep&);
    StringRep& operator=(const StringRep &);
    union {
        char *rep;
        StringRep *next;
    };
    size_t len;
    int count;
};

class String {
  public:
    friend class StringRep;
    friend class SBHelper;

    String()                       { p = new StringRep();                }
    String(const String& s)        { p = s.p; p->count++;                }
    String(const char *s)          { p = new StringRep(s);               }
    String(const char *s, int slen){ p = new StringRep(s, slen);         }
    String(char **s)               { p = new StringRep(s, ::strlen(*s)); }
    String(char** s, size_t slen)  { p = new StringRep(s, slen);         }
    ~String();

    String& operator=(const String& rhs);

    int operator==(const char *rhs)   const;
    int operator==(const String& rhs) const;
    int operator!=(const char *rhs)   const;
    int operator!=(const String& rhs) const;

    String operator+(const String &rhs) const   { return *p + rhs;      }
    friend String operator+(const char *lhs, const String& rhs)
                                            { return rhs + String(lhs); }

    void operator+=(const String &rhs);
    void operator+=(const char *rhs);

    operator const char *() const { return p->rep; }
    SBHelper operator[](int index);
    size_t length() const { return p->len; }
    void range_error(int index);
  private:
    StringRep *p;
};

/////////////////////////////////////////////////////////////////////////////
// This class is a helper class used by String::operator[] to distinguish
// between applications of operator[] on the lhs and rhs of "=" signs.
/////////////////////////////////////////////////////////////////////////////

class SBHelper {
  public:
    SBHelper(String &s, int i);
    SBHelper(const SBHelper& s);
    char operator=(char c);
    operator char() { return str.p->rep[index]; }
  private:
    void operator=(const SBHelper&);  // disallow this method
    String &str;
    int index;
};

///////////////////////////////////////////////////////////////////////////////
// DLink - Class modeling a link in a doubly-linked list of strings.
//         We also maintain the length of the string.
///////////////////////////////////////////////////////////////////////////////

class DLink {
    friend class DList;
  public:
    DLink(char **);
    ~DLink() { }

    static DLink *freeList;    // we manage our own storage for DLinks
    enum { chunksize = 50 };   // size blocks of DLinks we allocate
    void *operator new(size_t size);
    void operator delete(void *object);

    const char *line()  const { return _line;          }
    int length()        const { return _line.length(); }
    DLink *next()       const { return _next;          }
    DLink *prev()       const { return _prev;          }
    void update(char **);
  private:
    String            _line;
    DLink            *_next;
    DLink            *_prev;
    //
    // Disallow these operations by not providing definitions.
    // Also keep compiler from generating default versions of these.
    //
    DLink();
    DLink(const DLink &);
    DLink &operator=(const DLink &);
};

///////////////////////////////////////////////////////////////////////////////
// DList - Class modeling a doubly-linked list of DLinks.
//         It also maintains our current notion of what
//         is and isn't visible in the window.
///////////////////////////////////////////////////////////////////////////////

class DList {
  public:
    DList();
    ~DList();

    DLink *head()              const { return _head;                   }
    DLink *tail()              const { return _tail;                   }
    DLink *firstLine()         const { return _firstLine;              }
    DLink *lastLine()          const { return _lastLine;               }
    DLink *currLine()          const { return _currLine;               }
    DList *next()              const { return _next;                   }
    DList *prev()              const { return _prev;                   }

    int savedXPos()            const { return _saved_x;                }
    int savedYPos()            const { return _saved_y;                }

    void setFirst(DLink *e)          { _firstLine = e;                 }
    void setLast (DLink *e)          { _lastLine  = e;                 }
    void setCurrLine (DLink *ln)     { _currLine = ln;                 }

    void setNext (DList *l)          { _next = l;                      }
    void setPrev (DList *l)          { _prev = l;                      }

    int nelems()                 const { return _nelems;               }
    void saveYXPos(int y, int x)       { _saved_x = (short)x;
                                         _saved_y = (short)y;          }

    int atBegOfList()            const { return _currLine == _head;    }
    int atEndOfList()            const { return _currLine == _tail;    }

    int atWindowTop()            const { return _currLine == _firstLine; }
    int atWindowBot()            const { return _currLine == _lastLine;  }

    void add(DLink *);
    void deleteLine();
  private:
    DLink      *_head;
    DLink      *_tail;
    int         _nelems;
    short       _saved_x;     // saved x cursor position
    short       _saved_y;     // saved y cursor position
    DLink      *_firstLine;   // first viewable DLink in window
    DLink      *_lastLine;    // last  viewable DLink in window
    DLink      *_currLine;    // line cursor is on in window
    DList      *_next;
    DList      *_prev;
    //
    // Disallow these operations by not providing definitions.
    // Also keep compiler from generating default versions of these.
    //
    DList(const DList &);
    DList &operator=(const DList &);
};

inline SBHelper::SBHelper(String& s, int i) : str(s), index(i) { };

inline SBHelper::SBHelper(const SBHelper& s) : str(s.str), index(s.index) { };

inline int StringRep::operator!=(const char *rhs) const
{
    return strcmp(rep, rhs);
}

inline int StringRep::operator==(const char *rhs) const
{
    return strcmp(rep, rhs) == 0;
}

inline int StringRep::operator!=(const StringRep& rhs) const
{
    return strcmp(rep, rhs.rep);
}

inline int StringRep::operator==(const StringRep& rhs) const
{
    return strcmp(rep, rhs.rep) == 0;
}

inline int String::operator==(const char *rhs) const
{
    return *p == rhs;
}

inline int String::operator==(const String& rhs) const
{
    return *p == *(rhs.p);
}

inline int String::operator!=(const char *rhs) const
{
    return *p != rhs;
}

inline int String::operator!=(const String& rhs) const
{
    return *p != *(rhs.p);
}

#endif /* __CLASSES_H */
