/*********************************************************************
 *                                                                   *
 * MODULE NAME :  drag.c                 AUTHOR:  Rick Fishman       *
 * DATE WRITTEN:  07-16-93                                           *
 *                                                                   *
 * MODULE DESCRIPTION:                                               *
 *                                                                   *
 *  Part of the 'DRGTHRND' drag/drop sample program.                 *
 *                                                                   *
 *  This module handles all the Drag/Drop processing for the         *
 *  DRGTHRND.EXE sample program. The rendering code is split into    *
 *  srcrendr.c and trgrendr.c for the source and target respectively.*
 *                                                                   *
 * NOTES:                                                            *
 *                                                                   *
 * FUNCTIONS AVALABLE TO OTHER MODULES:                              *
 *                                                                   *
 *   dragInit                                                        *
 *   dragOver                                                        *
 *   dragDrop                                                        *
 *   dragTargetCleanup                                               *
 *   dragSourceCleanup                                               *
 *                                                                   *
 * HISTORY:                                                          *
 *                                                                   *
 *  07-16-93 - Program coded.                                        *
 *                                                                   *
 *  Rick Fishman                                                     *
 *  Code Blazers, Inc.                                               *
 *  4113 Apricot                                                     *
 *  Irvine, CA. 92720                                                *
 *  CIS ID: 72251,750                                                *
 *                                                                   *
 *********************************************************************/

#pragma strings(readonly)   // used for debug version of memory mgmt routines

/*********************************************************************/
/*------- Include relevant sections of the OS/2 header files --------*/
/*********************************************************************/

#define  INCL_DOSERRORS
#define  INCL_DOSPROCESS
#define  INCL_SHLERRORS
#define  INCL_WINDIALOGS
#define  INCL_WINERRORS
#define  INCL_WINFRAMEMGR
#define  INCL_WININPUT
#define  INCL_WINMENUS
#define  INCL_WINSTDCNR
#define  INCL_WINSTDDRAG
#define  INCL_WINWINDOWMGR

/**********************************************************************/
/*----------------------------- INCLUDES -----------------------------*/
/**********************************************************************/

#include <os2.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "drgthrnd.h"

/*********************************************************************/
/*------------------- APPLICATION DEFINITIONS -----------------------*/
/*********************************************************************/


/**********************************************************************/
/*---------------------------- STRUCTURES ----------------------------*/
/**********************************************************************/


/**********************************************************************/
/*----------------------- FUNCTION PROTOTYPES ------------------------*/
/**********************************************************************/

int  CountSelectedRecs   ( HWND hwndFrame, PCNRREC pCnrRecUnderMouse,
                           PBOOL pfUseSelectedRecs );
void SetSelectedDragItems( HWND hwndFrame, int cRecs, PDRAGINFO pDragInfo,
                           PDRAGIMAGE pDragImage );
void SetOneDragItem      ( HWND hwndFrame, PCNRREC pCnrRecUnderMouse,
                           PDRAGINFO pDragInfo, PDRAGIMAGE pDragImage,
                           int iOffset );
void RemoveSourceEmphasis( HWND hwndFrame );

/**********************************************************************/
/*------------------------ GLOBAL VARIABLES --------------------------*/
/**********************************************************************/


/**********************************************************************/
/*--------------------------- dragMessage ----------------------------*/
/*                                                                    */
/*  A DM_ MESSAGE WAS RECEIVED BY THE FRAME WINDOW PROCEDURE.         */
/*                                                                    */
/*  PARMS: frame window handle,                                       */
/*         message id,                                                */
/*         mp1 of the message                                         */
/*                                                                    */
/*  NOTES:                                                            */
/*                                                                    */
/*  RETURNS: return code of message                                   */
/*                                                                    */
/*--------------------------------------------------------------------*/
/**********************************************************************/
MRESULT dragMessage( HWND hwndFrame, ULONG msg, MPARAM mp1 )
{
    switch( msg )
    {
        // The target sends this to the source before it sends the DM_RENDER
        // message if the source specified DC_PREPARE in DRAGITEM.fsControl.

        case DM_RENDERPREPARE:

            // In srcrendr.c

            return srcRenderPrepare( hwndFrame, (PDRAGTRANSFER) mp1 );
    }

    return 0;
}

/**********************************************************************/
/*----------------------------- dragInit -----------------------------*/
/*                                                                    */
/*  PROCESS CN_INITDRAG NOTIFY MESSAGE.                               */
/*                                                                    */
/*  PARMS: frame window handle,                                       */
/*         pointer to the CNRDRAGINIT structure                       */
/*                                                                    */
/*  NOTES: the CN_INITDRAG message is equivalent to the WM_BEGINDRAG  */
/*         message. The reason I mention this is because if the source*/
/*         window in the drag is not a container you will get the     */
/*         WM_BEGINDRAG message.                                      */
/*                                                                    */
/*         WM_BEGINDRAG is sent to windows to allow them to know that */
/*         the user is requesting a drag from their window. This      */
/*         message contains only the x,y coordinates of the mouse at  */
/*         the time of WM_BEGINDRAG.                                  */
/*                                                                    */
/*         The reason for CN_INITDRAG is first so that the container  */
/*         can notify its owner when it gets a WM_BEGINDRAG message   */
/*         and second so that the container can give its owner more   */
/*         info, like what container record the mouse pointer is over.*/
/*         To accomplish this, the container packages up this informa-*/
/*         tion in a CNRDRAGINIT structure and sends that structure   */
/*         along with the CN_INITDRAG message.                        */
/*                                                                    */
/*  RETURNS: nothing                                                  */
/*                                                                    */
/*--------------------------------------------------------------------*/
/**********************************************************************/
void dragInit( HWND hwndFrame, PCNRDRAGINIT pcdi )
{
    PCNRREC     pCnrRecUnderMouse = (PCNRREC) pcdi->pRecord;
    PDRAGIMAGE  pDragImage = NULL;
    PDRAGINFO   pDragInfo = NULL;
    BOOL        fUseSelectedRecs;
    PINSTANCE   pi = INSTDATA( hwndFrame );
    int         cRecs;

    if( !pi )
    {
        Msg( "dragInit cant get Inst data RC(%X)", HWNDERR( hwndFrame ) );
        return;
    }

    // pSavedDragInfo is used during drop processing. When the drop is complete
    // (the source gets a DM_ENDCONVERSATION message from the target), the
    // source will free the DRAGINFO structure and set this field to NULL.
    // So pSavedDragInfo is non-NULL if a drop has not yet completed. In this
    // case we make it easy on ourselves by not allowing another drag. The
    // reason for this is that if we allow another drag to take place we will
    // need to overwrite this pSavedDragInfo field in which case the drop
    // processing would not free the right DRAGINFO structure. Obviously in a
    // commercial app you'd want to use a different mechanism for storing the
    // DRAGINFO structure, like a linked list, so you could start another drag
    // while the drop is in progress.

    if( pi->pSavedDragInfo )
    {
        WinAlarm( HWND_DESKTOP, WA_WARNING );
        return;
    }

    // Count the records that have CRA_SELECTED emphasis. Also return whether
    // or not we should process the CRA_SELECTED records. If the container
    // record under the mouse does not have this emphasis, we shouldn't. In that
    // case we would just process the record under the mouse.


    cRecs = CountSelectedRecs( hwndFrame, pCnrRecUnderMouse, &fUseSelectedRecs);

    if( cRecs )
    {
        int iDragImageArraySize = cRecs * sizeof( DRAGIMAGE );

        // Allocate an array of DRAGIMAGE structures. Each structure contains
        // info about an image that will be under the mouse pointer during the
        // drag. This image will represent a container record being dragged.

        pDragImage = (PDRAGIMAGE) malloc( iDragImageArraySize );

        if( pDragImage )
        {
            memset( pDragImage, 0, iDragImageArraySize );

            // Let PM allocate enough memory for a DRAGINFO structure as well
            // as a DRAGITEM structure for each record being dragged. It will
            // allocate shared memory so other processes can participate in the
            // drag/drop.

            pDragInfo = DrgAllocDraginfo( cRecs );

            if( pDragInfo )
            {
                pi->pSavedDragInfo = pDragInfo;
                pi->cDragItems = cRecs;
            }
            else
                Msg( "DrgAllocDraginfo failed. RC(%X)", HWNDERR( hwndFrame ) );
        }
        else
            Msg( "Out of memory in dragInit" );
    }

    if( cRecs && pDragInfo && pDragImage )
    {
        // Set the data from the container records into the DRAGITEM and
        // DRAGIMAGE structures. If we are to process CRA_SELECTED container
        // records, do them all in one function. If not, pass a pointer to the
        // container record under the mouse to a different function that will
        // fill in just one DRAGITEM / DRAGIMAGE structure.

        if( fUseSelectedRecs )
            SetSelectedDragItems( hwndFrame, cRecs, pDragInfo, pDragImage );
        else
            SetOneDragItem( hwndFrame, pCnrRecUnderMouse, pDragInfo,
                            pDragImage, 0 );

        // If DrgDrag returns NULLHANDLE, that means the user hit Esc or F1
        // while the drag was going on so the target didn't have a chance to
        // delete the string handles. So it is up to the source window to do
        // it. Unfortunately there doesn't seem to be a way to determine
        // whether the NULLHANDLE means Esc was pressed as opposed to there
        // being an error in the drag operation. So we don't attempt to figure
        // that out. To us, a NULLHANDLE means Esc was pressed...

        if( !DrgDrag( hwndFrame, pDragInfo, pDragImage, cRecs, VK_ENDDRAG,
                      NULL ) )
        {
            if( !DrgDeleteDraginfoStrHandles( pDragInfo ) )
                Msg( "dragInit DrgDeleteDraginfoStrHandles RC(%X)",
                     HWNDERR( hwndFrame ) );

            if( !DrgFreeDraginfo( pDragInfo ) )
                Msg( "dragInit DrgFreeDraginfo RC(%X)", HWNDERR( hwndFrame ) );

            pi->pSavedDragInfo = NULL;
        }

        // Take off source emphasis from the records that were dragged

        RemoveSourceEmphasis( hwndFrame );
    }

    if( pDragImage )
        free( pDragImage );
}

/**********************************************************************/
/*----------------------------- dragOver -----------------------------*/
/*                                                                    */
/*  PROCESS CN_DRAGOVER NOTIFY MESSAGE.                               */
/*                                                                    */
/*  PARMS: frame window handle,                                       */
/*         pointer to the CNRDRAGINFO structure                       */
/*                                                                    */
/*  NOTES: The container sends the CN_DRAGOVER message to its owner   */
/*         when it gets a DM_DRAGOVER message. The container takes    */
/*         the info it gets on the DM_DRAGOVER message and combines   */
/*         it with other container-specific dragover info into a      */
/*         CNRDRAGINFO structure, then passes a pointer to that       */
/*         structure to its owner in the CN_DRAGOVER message.         */
/*                                                                    */
/*         In this program, all we care about is that the other       */
/*         window can handle the DRM_FISHMAN protocol and that the    */
/*         window is not trying to drop on itself. Strangely enough,  */
/*         the DRM_FISHMAN protocol is not in widespread use <g>.     */
/*                                                                    */
/*  RETURNS: return value from CN_DRAGOVER processing                 */
/*                                                                    */
/*--------------------------------------------------------------------*/
/**********************************************************************/
MRESULT dragOver( HWND hwndFrame, PCNRDRAGINFO pcdi )
{
    USHORT    usDrop, usDefaultOp;
    PDRAGINFO pDragInfo = pcdi->pDragInfo;
    PDRAGITEM pDragItem;

    if( !DrgAccessDraginfo( pDragInfo ) )
    {
        Msg( "dragOver DrgAccessDraginfo RC(%X)", HWNDERR( hwndFrame ) );
        return MRFROM2SHORT( DOR_NEVERDROP, 0 );
    }

    // Don't allow a window to drop on itself

    if( pDragInfo->hwndSource == hwndFrame )
        return MRFROM2SHORT( DOR_NEVERDROP, 0 );

    // We're only concerned with the first item being dragged. We "assume" that
    // if the first item is using the DRM_FISHMAN protocol then all others are.
    // In a real-world program you would want to check all dragitems.

    pDragItem = DrgQueryDragitemPtr( pDragInfo, 0 );

    if( pDragItem )
    {
        // We will only allow DRM_FISHMAN items to be dropped on us

        if( DrgVerifyRMF( pDragItem, "DRM_FISHMAN", NULL ) )
        {
            usDrop = DOR_DROP;

            // We only allow a logical 'copy' operation to take place, so this
            // will change the pointer to be a 'copy' (halftoned) pointer
            // during the drag when the pointer is over our window.

            usDefaultOp = DO_COPY;
        }
        else
        {
            usDrop = DOR_NEVERDROP;
            usDefaultOp = 0;
        }
    }
    else
        Msg( "dragOver DrgQueryDragitemPtr RC(%X)", HWNDERR( hwndFrame ) );

    // Free our handle to the shared memory if the source window is not in our
    // process.

    if( !DrgFreeDraginfo( pDragInfo ) &&
        PMERR_SOURCE_SAME_AS_TARGET != HWNDERR( hwndFrame ) )
        Msg( "dragOver DrgFreeDraginfo RC(%X)", HWNDERR( hwndFrame ) );

    return MRFROM2SHORT( usDrop, usDefaultOp );
}

/**********************************************************************/
/*---------------------------- dragDrop ------------------------------*/
/*                                                                    */
/*  PROCESS CN_DROP NOTIFY MESSAGE.                                   */
/*                                                                    */
/*  PARMS: frame window handle,                                       */
/*         pointer to the CNRDRAGINFO structure                       */
/*                                                                    */
/*  NOTES: The container sends the CN_DROP message to its owner when  */
/*         it gets a DM_DROP message. The container takes the info it */
/*         gets on the DM_DROP message and combines it with other     */
/*         container-specific dragover info into a CNRDRAGINFO        */
/*         structure, then passes a pointer to that structure to its  */
/*         owner in the CN_DROP message.                              */
/*                                                                    */
/*  RETURNS: nothing                                                  */
/*                                                                    */
/*--------------------------------------------------------------------*/
/**********************************************************************/
void dragDrop( HWND hwndFrame, PCNRDRAGINFO pcdi )
{
    PDRAGINFO pDragInfo = pcdi->pDragInfo;
    PINSTANCE pi = INSTDATA( hwndFrame );

    if( !pi )
    {
        Msg( "dragDrop cant get Inst data RC(%X)", HWNDERR( hwndFrame ) );
        return;
    }

    // Save this info so it can be used on the last DM_RENDERCOMPLETE message.

    pi->pSavedDragInfo = pDragInfo;
    pi->cDragItems     = pDragInfo->cditem;
    pi->hwndRender     = targCreateWindow( hwndFrame );

    // Post a message to the 'render' window which is in another thread. This
    // will free up the user interface while the rendering is going on.

    if( pi->hwndRender )
        WinPostMsg( pi->hwndRender, UM_DO_THE_DROP, MPFROMP( pDragInfo ),
                    NULL );
}

/**********************************************************************/
/*------------------------- dragTargetCleanup ------------------------*/
/*                                                                    */
/*  FREE THE RESOURCES USED BY DRAG/DROP PROCESSING FOR THE TARGET    */
/*                                                                    */
/*  PARMS: frame window handle                                        */
/*                                                                    */
/*  NOTES:                                                            */
/*                                                                    */
/*  RETURNS: nothing                                                  */
/*                                                                    */
/*--------------------------------------------------------------------*/
/**********************************************************************/
void dragTargetCleanup( HWND hwndFrame )
{
    PINSTANCE pi = INSTDATA( hwndFrame );

    if( !pi )
    {
        Msg( "dragTargetCleanup cant get Inst data RC(%X)", HWNDERR(hwndFrame));
        return;
    }

    // It is the target's responsibility to delete the string resources. This
    // one API frees all the strings in all DRAGITEM structures.

    if( !DrgDeleteDraginfoStrHandles( pi->pSavedDragInfo ) )
        Msg( "dragTargetCleanup DrgDeleteDraginfoStrHandles RC(%X)",
             HWNDERR( hwndFrame ) );

    if( !DrgFreeDraginfo( pi->pSavedDragInfo ) &&
        PMERR_SOURCE_SAME_AS_TARGET != HWNDERR( hwndFrame ) )
        Msg( "dragTargetCleanup DrgFreeDraginfo RC(%X)", HWNDERR( hwndFrame ) );

    // Terminate the Render window's thread since rendering is complete.

    if( pi->hwndRender )
    {
        WinPostMsg( pi->hwndRender, WM_QUIT, NULL, NULL );
        pi->hwndRender = NULLHANDLE;
    }

    pi->pSavedDragInfo = NULL;
    pi->cDragItems     = 0;

    // Enable the Close option off the system menu that was disabled when
    // rendering began.

    WinSendDlgItemMsg( hwndFrame, FID_SYSMENU, MM_SETITEMATTR,
                       MPFROM2SHORT( SC_CLOSE, TRUE ),
                       MPFROM2SHORT( MIA_DISABLED, 0 ) );
}

/**********************************************************************/
/*------------------------- dragSourceCleanup ------------------------*/
/*                                                                    */
/*  FREE THE RESOURCES USED BY DRAG/DROP PROCESSING FOR THE SOURCE    */
/*                                                                    */
/*  PARMS: frame window handle                                        */
/*                                                                    */
/*  NOTES:                                                            */
/*                                                                    */
/*  RETURNS: nothing                                                  */
/*                                                                    */
/*--------------------------------------------------------------------*/
/**********************************************************************/
void dragSourceCleanup( HWND hwndFrame )
{
    PINSTANCE pi = INSTDATA( hwndFrame );

    if( !pi )
    {
        Msg( "dragSourceCleanup cant get Inst data RC(%X)", HWNDERR(hwndFrame));
        return;
    }

    // Free the shared memory we got access to using DrgAccessDragInfo. If
    // the source process is the same as the target process, we get the
    // PMERR_SOURCE_SAME_AS_TARGET message. It's ok to get that - it just
    // means that we don't need to free the structure because the target
    // process already freed it.

    if( !DrgFreeDraginfo( pi->pSavedDragInfo ) &&
        PMERR_SOURCE_SAME_AS_TARGET != HWNDERR( hwndFrame ) )
        Msg( "dragSourceCleanup DrgFreeDraginfo RC(%X)", HWNDERR( hwndFrame ) );

    // Terminate the Render window's thread since rendering is complete.

    if( pi->hwndRender )
    {
        WinPostMsg( pi->hwndRender, WM_QUIT, NULL, NULL );
        pi->hwndRender = NULLHANDLE;
    }

    // This is important because the NULL-ness of pSavedDragInfo lets the main
    // thread know that another drag is now possible.

    pi->pSavedDragInfo = NULL;
    pi->cDragItems     = 0;
}

/**********************************************************************/
/*------------------------ CountSelectedRecs -------------------------*/
/*                                                                    */
/*  COUNT THE NUMBER OF RECORDS THAT ARE CURRENTLY SELECTED.          */
/*                                                                    */
/*  PARMS: frame window handle,                                       */
/*         pointer to the record that was under the pointer,          */
/*         address of BOOL - should we process selected records?      */
/*                                                                    */
/*  NOTES:                                                            */
/*                                                                    */
/*  RETURNS: number of records to process                             */
/*                                                                    */
/*--------------------------------------------------------------------*/
/**********************************************************************/
int CountSelectedRecs( HWND hwndFrame, PCNRREC pCnrRecUnderMouse,
                       PBOOL pfUseSelectedRecs )
{
    int cRecs = 0;

    *pfUseSelectedRecs = FALSE;

    // If the record under the mouse is NULL, we must be over whitespace, in
    // which case we don't want to drag any records.

    if( pCnrRecUnderMouse )
    {
        PCNRREC pCnrRec = (PCNRREC) CMA_FIRST;

        // Count the records with 'selection' emphasis. These are the records
        // we want to drag, unless the container record under the mouse does
        // not have selection emphasis. If that is the case, we only want to
        // process that one.

        while( pCnrRec )
        {
            pCnrRec = (PCNRREC) WinSendDlgItemMsg( hwndFrame, FID_CLIENT,
                                                   CM_QUERYRECORDEMPHASIS,
                                                   MPFROMP( pCnrRec ),
                                                   MPFROMSHORT( CRA_SELECTED ));

            if( pCnrRec == (PCNRREC) -1 )
                Msg( "CountSelectedRecs..CM_QUERYRECORDEMPHASIS RC(%X)",
                     HWNDERR( hwndFrame ) );
            else if( pCnrRec )
            {
                if( pCnrRec == pCnrRecUnderMouse )
                    *pfUseSelectedRecs = TRUE;

                cRecs++;
            }
        }

        if( !(*pfUseSelectedRecs) )
            cRecs = 1;
    }

    return cRecs;
}

/**********************************************************************/
/*----------------------- SetSelectedDragItems -----------------------*/
/*                                                                    */
/*  FILL THE DRAGINFO STRUCT WITH DRAGITEM STRUCTS FOR SELECTED RECS. */
/*                                                                    */
/*  PARMS: frame window handle,                                       */
/*         count of selected records,                                 */
/*         pointer to allocated DRAGINFO struct,                      */
/*         pointer to allocated DRAGIMAGE array                       */
/*                                                                    */
/*  NOTES:                                                            */
/*                                                                    */
/*  RETURNS: nothing                                                  */
/*                                                                    */
/*--------------------------------------------------------------------*/
/**********************************************************************/
void SetSelectedDragItems( HWND hwndFrame, int cRecs, PDRAGINFO pDragInfo,
                           PDRAGIMAGE pDragImage )
{
    PCNRREC pCnrRec = (PCNRREC) CMA_FIRST;
    int     i;

    for( i = 0; i < cRecs; i++, pDragImage++ )
    {
        pCnrRec = (PCNRREC) WinSendDlgItemMsg( hwndFrame, FID_CLIENT,
                                               CM_QUERYRECORDEMPHASIS,
                                               MPFROMP( pCnrRec ),
                                               MPFROMSHORT( CRA_SELECTED ) );

        if( pCnrRec == (PCNRREC) -1 )
            Msg( "SetSelectedDragItems..CM_QUERYRECORDEMPHASIS RC(%X)",
                 HWNDERR( hwndFrame ) );
        else
            SetOneDragItem( hwndFrame, pCnrRec, pDragInfo, pDragImage, i );
    }
}

/**********************************************************************/
/*-------------------------- SetOneDragItem --------------------------*/
/*                                                                    */
/*  SET ONE DRAGITEM STRUCT INTO A DRAGINFO STRUCT.                   */
/*                                                                    */
/*  PARMS: frame window handle,                                       */
/*         pointer to CNRREC that contains current container record,  */
/*         pointer to allocated DRAGINFO struct,                      */
/*         pointer to allocated DRAGIMAGE array,                      */
/*         record offset into DRAGINFO struct to place DRAGITEM       */
/*                                                                    */
/*  NOTES:                                                            */
/*                                                                    */
/*  RETURNS: nothing                                                  */
/*                                                                    */
/*--------------------------------------------------------------------*/
/**********************************************************************/
void SetOneDragItem( HWND hwndFrame, PCNRREC pCnrRec, PDRAGINFO pDragInfo,
                     PDRAGIMAGE pDragImage, int iOffset )
{
    DRAGITEM DragItem;
    PCH      pchColon = NULL;

    memset( &DragItem, 0, sizeof DragItem );

    // hwndDragItem is the window that will get the DM_RENDER and
    // DM_RENDERCOMPLETE messages.

    DragItem.hwndItem = hwndFrame;

    // ulItemID is used to store information that can be used at drop time. Here
    // we store the container record pointer.

    DragItem.ulItemID = (ULONG) pCnrRec;

    // hstrType identifies 'types' when it is necessary to differentiate. A
    // good example is if you are dragging file names (DRM_OS2FILE) and need to
    // pass the file type to the target (i.e. DRT_BITMAP would mean the file
    // contained a bitmap, DRT_TEXT is an ascii file, etc. )

    DragItem.hstrType = DrgAddStrHandle( DRT_UNKNOWN );

    // This is the rendering mechanism/format. This establishes the protocol
    // that we will be using during the Drag/Drop operation. In this program we
    // only use DRM_FISHMAN which means rendering will take place after the
    // drop using a predefined and agreed-upon mechanism. Any target objects
    // that OK this type of protocol must know how to handle themselves after
    // the drop.

    DragItem.hstrRMF = DrgAddStrHandle( DRAG_RMF );

    // This will contain our 'database file' name. We have multiple 'database'
    // files, each with their own set of 'tables'. This field identifies the
    // 'database' name. The text for each container record has been set (in the
    // InsertRecords function in drgthrnd.c) to "database:table". So we
    // temporarily set the colon to a null-terminator for the database name so
    // we can create a stringhandle for the database name.

    pchColon = strchr( pCnrRec->szTableName, ':' );
    if( pchColon )
        *pchColon = 0;
    else                  // This should never happen! Play it safe though.
        pchColon = pCnrRec->szTableName;

    DragItem.hstrContainerName = DrgAddStrHandle( pCnrRec->szTableName );

    // This will contain the table name (minus the database name)

    DragItem.hstrSourceName = DrgAddStrHandle( pchColon + 1 );

    if( pchColon != pCnrRec->szTableName )
        *pchColon = ':';

    // Suggested target name is the same as the source name. We *could* ask the
    // target to call it something else but we don't need that functionality
    // in this program.

    DragItem.hstrTargetName = DragItem.hstrSourceName;

    // Allow the user to only 'copy' this item. 'Move doesn't make sense since
    // the records will always stay in the source container.

    DragItem.fsSupportedOps = DO_COPYABLE;

    // This tells the target to send a DC_RENDERPREPARE message before it
    // sends a DM_RENDER message.

    DragItem.fsControl = DC_PREPARE;

    // Set the DRAGITEM struct into the memory allocated by
    // DrgAllocDraginfo()

    DrgSetDragitem( pDragInfo, &DragItem, sizeof DragItem, iOffset );

    // Fill in the DRAGIMAGE structure

    pDragImage->cb       = sizeof( DRAGIMAGE );
    pDragImage->hImage   = pCnrRec->mrc.hptrIcon; // DragImage under mouse
    pDragImage->fl       = DRG_ICON;              // hImage is an HPOINTER
    pDragImage->cxOffset = 5 * iOffset;           // Image offset from mouse ptr
    pDragImage->cyOffset = 5 * iOffset;           // Image offset from mouse ptr

    // Set source emphasis for this container record

    if( !WinSendDlgItemMsg( hwndFrame, FID_CLIENT, CM_SETRECORDEMPHASIS,
                        MPFROMP( pCnrRec ), MPFROM2SHORT( TRUE, CRA_SOURCE ) ) )
        Msg( "SetOneDragItem..CM_SETRECORDEMPHASIS RC(%X)",
             HWNDERR( hwndFrame ) );
}

/**********************************************************************/
/*----------------------- RemoveSourceEmphasis -----------------------*/
/*                                                                    */
/*  REMOVE SOURCE EMPHASIS FROM THE DRAGGED RECORDS.                  */
/*                                                                    */
/*  PARMS: frame window handle                                        */
/*                                                                    */
/*  NOTES:                                                            */
/*                                                                    */
/*  RETURNS: nothing                                                  */
/*                                                                    */
/*--------------------------------------------------------------------*/
/**********************************************************************/
void RemoveSourceEmphasis( HWND hwndFrame )
{
    PCNRREC pCnrRec = (PCNRREC) CMA_FIRST;

    // For every record with source emphasis, remove it.

    while( pCnrRec )
    {
        pCnrRec = (PCNRREC) WinSendDlgItemMsg( hwndFrame, FID_CLIENT,
                                            CM_QUERYRECORDEMPHASIS,
                                            MPFROMP( pCnrRec ),
                                            MPFROMSHORT( CRA_SOURCE ) );

        if( pCnrRec == (PCNRREC) -1 )
            Msg( "RemoveSourceEmphasis..CM_QUERYRECORDEMPHASIS RC(%X)",
                 HWNDERR( hwndFrame ) );
        else if( pCnrRec )
            if( !WinSendDlgItemMsg( hwndFrame, FID_CLIENT,
                                    CM_SETRECORDEMPHASIS, MPFROMP( pCnrRec ),
                                    MPFROM2SHORT( FALSE, CRA_SOURCE ) ) )
                Msg( "RemoveSourceEmphasis..CM_SETRECORDEMPHASIS RC(%X)",
                     HWNDERR( hwndFrame ) );
    }
}

/*************************************************************************
 *                     E N D     O F     S O U R C E                     *
 *************************************************************************/
