/*********************************************************************
 *                                                                   *
 * MODULE NAME :  drgrendr.c             AUTHOR:  Rick Fishman       *
 * DATE WRITTEN:  07-15-93                                           *
 *                                                                   *
 * HOW TO RUN THIS PROGRAM:                                          *
 *                                                                   *
 *  Just enter DRGRENDR on the command line.                         *
 *                                                                   *
 * MODULE DESCRIPTION:                                               *
 *                                                                   *
 *  Root module for DRGRENDR.EXE, a program that demonstrates using  *
 *  source rendering on the drop during a Drag/Drop operation. This  *
 *  program creates 2 windows, a 'drag' window and a 'drop' window.  *
 *                                                                   *
 *  Each window is actually a frame window with a container control  *
 *  as its client window.                                            *
 *                                                                   *
 *  The drag window contains icons that each represent a table in    *
 *  a pseudo-database (actually ASCII files that simulate a          *
 *  database).                                                       *
 *                                                                   *
 *  The drop window starts out empty and allows you to drag an icon  *
 *  from the 'drag' window to it. If you do, rendering is done       *
 *  during the drop that writes the rows in the selected table to a  *
 *  temporary file and adds an icon representing that temporary file *
 *  to the drop window. You can then double-click on the icon in the *
 *  drop window to display the contents of the temporary file.       *
 *                                                                   *
 *  The source window (the drag window) is the one that does the     *
 *  actual rendering. By rendering, I mean it 'renders' the contents *
 *  of a table in the database to a file.                            *
 *                                                                   *
 *  The target window (the drop window) is the one that adds the icon*
 *  to its container when the source window indicates that the       *
 *  rendering is complete.                                           *
 *                                                                   *
 *  All rendering is done on thread 1. To see an example where the   *
 *  drop and rendering takes place in separate threads, have a look  *
 *  at my DRGTHRND.EXE sample.
 *                                                                   *
 *  When the drag window is created, it goes thru all database files *
 *  and inserts an icon into the container for each 'table' that it  *
 *  finds.                                                           *
 *                                                                   *
 *  To get an idea of how these 'database' files are structured,     *
 *  have a look at the dbase_?.db files provided with this sample.   *
 *                                                                   *
 * OTHER MODULES:                                                    *
 *                                                                   *
 *  dbaccess.c - contains the code used to access the 'database'.    *
 *  drag.c     - contains all drag/drop processing code.             *
 *                                                                   *
 * NOTES:                                                            *
 *                                                                   *
 *  I hope this code proves useful for other PM programmers. The     *
 *  more of us the better!                                           *
 *                                                                   *
 * HISTORY:                                                          *
 *                                                                   *
 *  07-15-93 - Program coding started.                               *
 *                                                                   *
 *  Rick Fishman                                                     *
 *  Code Blazers, Inc.                                               *
 *  4113 Apricot                                                     *
 *  Irvine, CA. 92720                                                *
 *  CIS ID: 72251,750                                                *
 *                                                                   *
 *********************************************************************/

#pragma strings(readonly)   // used for debug version of memory mgmt routines

/*********************************************************************/
/*------- Include relevant sections of the OS/2 header files --------*/
/*********************************************************************/

#define  INCL_WINERRORS
#define  INCL_WINDIALOGS
#define  INCL_WINFRAMEMGR
#define  INCL_WINLISTBOXES
#define  INCL_WINPOINTERS
#define  INCL_WINSTDCNR
#define  INCL_WINSTDDRAG
#define  INCL_WINSYS
#define  INCL_WINWINDOWMGR

#define  GLOBALS_DEFINED    // extern globals instantiated

/**********************************************************************/
/*----------------------------- INCLUDES -----------------------------*/
/**********************************************************************/

#include <os2.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "drgrendr.h"

/*********************************************************************/
/*------------------- APPLICATION DEFINITIONS -----------------------*/
/*********************************************************************/

#define PROGRAM_TITLE               "Drag/Drop Rendering Sample"

#define MESSAGE_SIZE                1024

#define TITLE_FOR_DROPCNR_FRAME     "'Drop' Container"
#define TITLE_FOR_DRAGCNR_FRAME     "'Drag' Container"

#define FRAME_FLAGS                 (FCF_TASKLIST   | FCF_TITLEBAR | \
                                     FCF_SYSMENU    | FCF_MINMAX   | \
                                     FCF_SIZEBORDER | FCF_ICON     | \
                                     FCF_ACCELTABLE)

#define CONTAINER_STYLES            (CCS_EXTENDSEL | CCS_MINIRECORDCORE | \
                                     CCS_AUTOPOSITION)

/**********************************************************************/
/*---------------------------- STRUCTURES ----------------------------*/
/**********************************************************************/

typedef struct _LBFRAMEDATA         // LISTBOX FRAME'S WINDOW WORD DATA
{
    HWND    hwndCnr;                // Container window handle
    PCNRREC pCnrRec;                // Record whose table is being displayed in
} LBFRAMEDATA, *PLBFRAMEDATA;       //    the listbox

/**********************************************************************/
/*----------------------- FUNCTION PROTOTYPES ------------------------*/
/**********************************************************************/

int  main             ( void );
void GetCurrentPath   ( void );
BOOL CreateWindows    ( HAB hab );
HWND CreateWindow     ( HAB hab, PFRAMECDATA pfcdata, ULONG idWindow,
                        PSZ pszWindow, PSZ pszCnrTitle );
BOOL InsertRecords    ( HWND hwndCnr );
void DisplayTable     ( HWND hwndCnr, PCNRREC pCnrRec );
void DestroyWindows   ( void );
void DeleteTempFiles  ( void );

FNWP wpFrame, wpLBFrame;

/**********************************************************************/
/*------------------------ GLOBAL VARIABLES --------------------------*/
/**********************************************************************/

PFNWP pfnwpFrame, pfnwpLBFrame;

HWND hwndDrag,           // Window that is the source of the drag
     hwndDrop;           // Window that is the target of the drop

char szDragCnrTitle[] = "Take a 'table' and drop it on the other window";
char szDropCnrTitle[] = "Go ahead. Drag one of those tables on me!";

/**********************************************************************/
/*------------------------------ MAIN --------------------------------*/
/*                                                                    */
/*  PROGRAM ENTRYPOINT                                                */
/*                                                                    */
/*  PARMS: nothing                                                    */
/*                                                                    */
/*  NOTES:                                                            */
/*                                                                    */
/*  RETURNS: return code                                              */
/*                                                                    */
/*--------------------------------------------------------------------*/
/**********************************************************************/
int main( void )
{
    HAB  hab;
    HMQ  hmq = NULLHANDLE;
    QMSG qmsg;

    // This macro is defined for the debug version of the C Set/2 Memory
    // Management routines. Since the debug version writes to stderr, we
    // send all stderr output to a debuginfo file.

#ifdef __DEBUG_ALLOC__
    freopen( DEBUG_FILENAME, "w", stderr );
#endif

    GetCurrentPath();

    hab = WinInitialize( 0 );

    if( hab )
        hmq = WinCreateMsgQueue( hab, 0 );
    else
    {
        DosBeep( 1000, 100 );
        fprintf( stderr, "WinInitialize failed!" );
    }

    if( hmq )
    {
        if( CreateWindows( hab ) )
        {
            while( WinGetMsg( hab, &qmsg, NULLHANDLE, 0, 0 ) )
                WinDispatchMsg( hab, &qmsg );

            DestroyWindows();
        }
    }
    else if( hab )
        Msg( "WinCreateMsgQueue RC(%X)", HABERR( hab ) );

    if( hmq )
        WinDestroyMsgQueue( hmq );

    if( hab )
        WinTerminate( hab );

    DeleteTempFiles();

#ifdef __DEBUG_ALLOC__
    _dump_allocated( -1 );
#endif

    return 0;
}

/**********************************************************************/
/*------------------------- GetCurrentPath ---------------------------*/
/*                                                                    */
/*  STORE THE CURRENT DRIVE/DIRECTORY.                                */
/*                                                                    */
/*  PARMS: nothing                                                    */
/*                                                                    */
/*  NOTES: This stores the current drive:\directory\  that is used    */
/*         to create temporary files in.                              */
/*                                                                    */
/*  RETURNS: nothing                                                  */
/*                                                                    */
/*--------------------------------------------------------------------*/
/**********************************************************************/
void GetCurrentPath()
{
    PBYTE  pbCurrent = szCurrentPath;
    INT    cbBuf = sizeof szCurrentPath, cbUsed;
    ULONG  ulDrive, ulCurrDriveNo, ulDriveMap, cbPath;
    APIRET rc;

    // Fill in the drive letter, colon, and backslash

    rc = DosQueryCurrentDisk( &ulCurrDriveNo, &ulDriveMap );

    if( !rc )                                // Use 'current' drive
    {
        *(pbCurrent++) = (BYTE) (ulCurrDriveNo + ('A' - 1));
        *(pbCurrent++) = ':';
        *(pbCurrent++) = '\\';
    }
    else
    {                                        // API failed - use drive C:
        strcpy( pbCurrent, "C:\\" );
        pbCurrent += 3;                      // Incr our place in the buffer
    }

    cbUsed = pbCurrent - szCurrentPath;      // How many bytes left?

    // Fill in the current directory

    ulDrive = *szCurrentPath - 'A' + 1;      // Get drive number from letter
    cbPath = cbBuf - cbUsed;                 // How many bytes left?

    rc = DosQueryCurrentDir( ulDrive, pbCurrent, &cbPath );
                                             // Get 'current' directory
    if( szCurrentPath[ strlen( szCurrentPath ) - 1 ] != '\\' )
        strcat( szCurrentPath, "\\" );       // Add trailing backslash
}

/**********************************************************************/
/*------------------------- CreateWindows ----------------------------*/
/*                                                                    */
/*  CREATE BOTH THE SOURCE AND TARGET WINDOWS.                        */
/*                                                                    */
/*  PARMS: anchor block handle                                        */
/*                                                                    */
/*  NOTES:                                                            */
/*                                                                    */
/*  RETURNS: TRUE or FALSE if successful or not                       */
/*                                                                    */
/*--------------------------------------------------------------------*/
/**********************************************************************/
BOOL CreateWindows( HAB hab )
{
    BOOL       fSuccess = TRUE;
    FRAMECDATA fcdata;

    memset( &fcdata, 0, sizeof fcdata );
    fcdata.cb            = sizeof( FRAMECDATA );
    fcdata.flCreateFlags = FRAME_FLAGS;

    // Create 2 windows. One will act as the 'drag' window, the other as the
    // 'drop' window. The user will then be able to drag the windows from the
    // 'drag' window to the 'drop' window.

    hwndDrag = CreateWindow( hab, &fcdata, ID_DRAGCNR, TITLE_FOR_DRAGCNR_FRAME,
                             szDragCnrTitle );
    if( hwndDrag )
    {
        hwndDrop = CreateWindow( hab, &fcdata, ID_DROPCNR,
                                 TITLE_FOR_DROPCNR_FRAME, szDropCnrTitle );
        if( !hwndDrop )
            fSuccess = FALSE;
    }
    else
        fSuccess = FALSE;

    // Load the icons that will be used for the container records.

    hptrDragMe = WinLoadPointer( HWND_DESKTOP, 0, ID_DRAGCNR );
    hptrOpenMe = WinLoadPointer( HWND_DESKTOP, 0, ID_DROPCNR );

    // Insert the records into the 'drag' container.

    if( fSuccess )
        fSuccess = InsertRecords( WinWindowFromID( hwndDrag, FID_CLIENT ) );

    if( fSuccess )
    {
        LONG cxDesktop = WinQuerySysValue( HWND_DESKTOP, SV_CXSCREEN );
        LONG cyDesktop = WinQuerySysValue( HWND_DESKTOP, SV_CYSCREEN );

        // Set the windows up so they split the bottom half of the display. The
        // left window will be the 'drag' window, the right one will be the
        // 'drop' window.

        if( !WinSetWindowPos( hwndDrag, HWND_TOP, 0, 0,
                              cxDesktop /2, cyDesktop / 2,
                  SWP_SIZE | SWP_ZORDER | SWP_MOVE | SWP_SHOW | SWP_ACTIVATE ) )
        {
            fSuccess = FALSE;
            Msg( "WinSetWindowPos( hwndDrag ) RC(%X)", HABERR( hab ) );
        }

        if( fSuccess &&
            !WinSetWindowPos( hwndDrop, HWND_TOP, cxDesktop / 2, 0,
                              cxDesktop /2, cyDesktop / 2,
                              SWP_SIZE | SWP_ZORDER | SWP_MOVE | SWP_SHOW ) )
        {
            fSuccess = FALSE;
            Msg( "WinSetWindowPos( hwndDrop ) RC(%X)", HABERR( hab ) );
        }

        if( fSuccess )
        {
            // The container was set up as the client window of the frame. We
            // need to set focus to it - otherwise it will not accept keystrokes
            // right away.

            WinSetFocus( HWND_DESKTOP,
                         WinWindowFromID( hwndDrag, FID_CLIENT ) );
        }
    }

    if( !fSuccess )
    {
        if( hwndDrag )
        {
            WinDestroyWindow( hwndDrag );
            hwndDrag = NULLHANDLE;
        }

        if( hwndDrop )
        {
            WinDestroyWindow( hwndDrop );
            hwndDrop = NULLHANDLE;
        }
    }

    return fSuccess;
}

/**********************************************************************/
/*-------------------------- CreateWindow ----------------------------*/
/*                                                                    */
/*  CREATE A FRAME WINDOW WITH A CONTAINER AS ITS CLIENT WINDOW       */
/*                                                                    */
/*  PARMS: anchor block handle,                                       */
/*         pointer to frame control data,                             */
/*         id of the frame window,                                    */
/*         window title of the frame window,                          */
/*         container title                                            */
/*                                                                    */
/*  NOTES:                                                            */
/*                                                                    */
/*  RETURNS: frame window handle or NULLHANDLE if not successful      */
/*                                                                    */
/*--------------------------------------------------------------------*/
/**********************************************************************/
HWND CreateWindow( HAB hab, PFRAMECDATA pfcdata, ULONG idWindow, PSZ pszWindow,
                   PSZ pszCnrTitle )
{
    HWND hwndFrame = NULLHANDLE, hwndCnr = NULLHANDLE;

    pfcdata->idResources = idWindow;  // Let the frame know its icon id

    // Create the container as the client window of the frame. There is no
    // need for a normal client window. Because of this we must subclass the
    // frame window so we can catch the messages that the container sends to
    // its owner.

    hwndFrame = WinCreateWindow( HWND_DESKTOP, WC_FRAME, NULL,
                                 FS_NOBYTEALIGN | WS_CLIPCHILDREN,
                                 0, 0, 0, 0, NULLHANDLE, HWND_TOP,
                                 idWindow, pfcdata, NULL );
    if( hwndFrame )
    {
        pfnwpFrame = WinSubclassWindow( hwndFrame, wpFrame );

        if( pfnwpFrame )
        {
            hwndCnr = WinCreateWindow( hwndFrame, WC_CONTAINER, NULL,
                                       WS_VISIBLE | CONTAINER_STYLES,
                                       0, 0, 0, 0, hwndFrame, HWND_TOP,
                                       FID_CLIENT, NULL, NULL );
            if( hwndCnr )
                WinSetWindowText( hwndFrame, pszWindow );
            else
            {
                WinDestroyWindow( hwndFrame );
                hwndFrame = NULLHANDLE;
                Msg( "WinCreateWindow(hwndCnr,%s) RC(%X)", pszWindow,
                     HABERR( hab ) );
            }
        }
        else
        {
            WinDestroyWindow( hwndFrame );
            hwndFrame = NULLHANDLE;
            Msg( "WinSubclassWindow(%s) RC(%X)", pszWindow, HABERR( hab ) );
        }
    }
    else
        Msg( "WinCreateWindow(%s) RC(%X)", pszWindow, HABERR( hab ) );

    if( hwndFrame )
    {
        CNRINFO cnri;

        // Set container into Icon view and give it a read-only title

        cnri.cb           = sizeof( CNRINFO );
        cnri.pszCnrTitle  = pszCnrTitle;
        cnri.flWindowAttr = CV_ICON | CA_CONTAINERTITLE | CA_TITLESEPARATOR |
                            CA_TITLEREADONLY;

        if( !WinSendMsg( hwndCnr, CM_SETCNRINFO, MPFROMP( &cnri ),
                         MPFROMLONG( CMA_FLWINDOWATTR | CMA_CNRTITLE ) ) )
        {
            WinDestroyWindow( hwndFrame );
            hwndFrame = NULLHANDLE;
            Msg( "CM_SETCNRINFO(%S) RC(%X)", pszWindow, HABERR( hab ) );
        }
    }

    if( hwndFrame )
    {
        PINSTANCE pi;

        // Allocate memory for the instance data and set it into the
        // Frame window's QWL_USER window word.

        pi = (PINSTANCE) malloc( sizeof *pi );
        if( pi )
        {
            memset( pi, 0, sizeof *pi );
            WinSetWindowPtr( hwndFrame, QWL_USER, pi );
        }
        else
        {
            WinDestroyWindow( hwndFrame );
            hwndFrame = NULLHANDLE;
            Msg( "Out of memory in CreateWindow!" );
        }
    }

    return hwndFrame;
}

/**********************************************************************/
/*-------------------------- InsertRecords ---------------------------*/
/*                                                                    */
/*  INSERT RECORDS INTO A CONTAINER                                   */
/*                                                                    */
/*  PARMS: container window handle                                    */
/*                                                                    */
/*  NOTES: We use the enumeration functions in dbaccess.c to enumerate*/
/*         all the tables in all the database files (dbase_?.db).     */
/*         We insert a record into the container for each table that  */
/*         we find.                                                   */
/*                                                                    */
/*  RETURNS: TRUE if successful, FALSE if not                         */
/*                                                                    */
/*--------------------------------------------------------------------*/
/**********************************************************************/
BOOL InsertRecords( HWND hwndCnr )
{
    HENUMTABLES  henum = dbBeginEnumTables();
    BOOL         fSuccess = TRUE;
    RECORDINSERT ri;
    PCNRREC      pCnrRec;
    char         szTableName[ CCHMAXPATH ];

    memset( &ri, 0, sizeof( RECORDINSERT ) );
    ri.cb                 = sizeof( RECORDINSERT );
    ri.pRecordOrder       = (PRECORDCORE) CMA_END;
    ri.pRecordParent      = (PRECORDCORE) NULL;
    ri.zOrder             = (USHORT) CMA_TOP;
    ri.cRecordsInsert     = 1;
    ri.fInvalidateRecord  = FALSE;

    if( henum )
    {
        while( fSuccess )
        {
            if( dbGetNextTable( henum, szTableName, sizeof szTableName ) )
            {
                pCnrRec = WinSendMsg( hwndCnr, CM_ALLOCRECORD,
                                      MPFROMLONG( EXTRA_BYTES ),
                                      MPFROMLONG( 1 ) );

                if( pCnrRec )
                {
                    strcpy( pCnrRec->szTableName, szTableName );

                    pCnrRec->flAttr       = RECATTR_DRAGGABLE;
                    pCnrRec->mrc.pszIcon  = (PSZ) &pCnrRec->szTableName;
                    pCnrRec->mrc.hptrIcon = hptrDragMe;

                    if( !WinSendMsg( hwndCnr, CM_INSERTRECORD,
                                     MPFROMP( pCnrRec ), MPFROMP( &ri ) ) )
                        Msg( "InsertRecords CM_INSERTRECORD RC(%X)",
                             HWNDERR( hwndCnr ) );
                }
                else
                    Msg( "InsertRecords CM_ALLOCRECORD RC(%X)",
                         HWNDERR( hwndCnr ) );
            }
            else
                break;
        }

        dbEndEnumTables( henum );

        // By setting ri.fInvalidateRecord to FALSE, we are saying not to
        // display the records yet. We display them only when they are all
        // inserted by waiting until we get here and do an invalidate of all
        // the records at once.

        if( !WinSendMsg( hwndCnr, CM_INVALIDATERECORD, NULL, NULL ) )
            Msg( "InsertRecords CM_INVALIDATERECORD RC(%X)", HWNDERR(hwndCnr) );
    }

    return fSuccess;
}

/**********************************************************************/
/*----------------------------- wpFrame ------------------------------*/
/*                                                                    */
/*  SUBCLASSED FRAME WINDOW PROCEDURE.                                */
/*                                                                    */
/*  PARMS: normal winproc parms                                       */
/*                                                                    */
/*  NOTES:                                                            */
/*                                                                    */
/*  RETURNS: MRESULT value                                            */
/*                                                                    */
/*--------------------------------------------------------------------*/
/**********************************************************************/
MRESULT EXPENTRY wpFrame( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 )
{
    // Route all DM_ messages to drag.c

    if( msg == UM_DO_THE_RENDERING ||
        (msg >= WM_DRAGFIRST && msg <= WM_DRAGLAST) )
        return dragMessage( hwnd, msg, mp1, mp2 );

    switch( msg )
    {
        // Don't let the message queue be destroyed until the user has closed
        // *both* windows. If SC_CLOSE gets processed by the default window
        // proc, a WM_QUIT message will get posted to the queue which will in
        // effect end this program.

        case WM_SYSCOMMAND:
            if( SHORT1FROMMP( mp1 ) == SC_CLOSE )
                if( (hwnd == hwndDrag) && hwndDrop )
                {
                    WinDestroyWindow( hwndDrag );
                    hwndDrag = NULLHANDLE;
                    WinSetActiveWindow( HWND_DESKTOP, hwndDrop );
                    return 0;
                }
                else if( (hwnd == hwndDrop) && hwndDrag )
                {
                    WinDestroyWindow( hwndDrop );
                    hwndDrop = NULLHANDLE;
                    WinSetActiveWindow( HWND_DESKTOP, hwndDrag );
                    return 0;
                }

            break;

        // From the F3 accelerator key
        case WM_COMMAND:
            if( SHORT1FROMMP( mp1 ) == IDM_EXIT )
            {
                WinPostMsg( hwnd, WM_SYSCOMMAND, MPFROMSHORT( SC_CLOSE ),
                            MPFROM2SHORT( CMDSRC_ACCELERATOR, FALSE ) );
                return 0;
            }

            break;

        case WM_DESTROY:
        {
            PINSTANCE pi = INSTDATA( hwnd );

            // Free the window word memory

            if( pi )
                free( pi );

            // Free all container resources associated with any records that
            // were inserted (note that the container is actually the client
            // window.

            WinSendDlgItemMsg( hwnd, FID_CLIENT, CM_REMOVERECORD, NULL,
                               MPFROM2SHORT( 0, CMA_FREE ) );
            break;
        }

        case WM_CONTROL:
            if( SHORT1FROMMP( mp1 ) == FID_CLIENT )
                switch( SHORT2FROMMP( mp1 ) )
                {
                    case CN_ENTER:
                    {
                        PNOTIFYRECORDENTER pnre = (PNOTIFYRECORDENTER) mp2;
                        if( pnre->pRecord &&
                            (((PCNRREC)pnre->pRecord)->flAttr &
                                                    RECATTR_OPENABLE ))
                            DisplayTable( WinWindowFromID( hwnd, FID_CLIENT ),
                                          (PCNRREC) pnre->pRecord );
                        return 0;
                    }

                    case CN_INITDRAG:
                    {
                        PCNRDRAGINIT pcdi = (PCNRDRAGINIT) mp2;
                        if( pcdi->pRecord &&
                            (((PCNRREC)pcdi->pRecord)->flAttr &
                                                    RECATTR_DRAGGABLE ))
                            dragInit( hwnd, pcdi );
                        return 0;
                    }

                    case CN_DRAGOVER:
                        return dragOver( hwnd, (PCNRDRAGINFO) mp2 );

                    case CN_DROP:
                        dragDrop( hwnd, (PCNRDRAGINFO) mp2 );
                        return 0;
                }

            break;
    }

    return pfnwpFrame( hwnd, msg, mp1, mp2 );
}

/**********************************************************************/
/*-------------------------- DisplayTable ----------------------------*/
/*                                                                    */
/*  DISPLAY A FILE (TABLE DATA) IN A LISTBOX WITHIN A FRAME WINDOW    */
/*                                                                    */
/*  PARMS: container window handle,                                   */
/*         pointer to container record that has info for table        */
/*                                                                    */
/*  NOTES: We create a frame window with a listbox as its client      */
/*         window and insert all records found in a file into the     */
/*         listbox. This acts as a simple file viewer.                */
/*                                                                    */
/*         The files being displayed are files created during the     */
/*         rendering operation that occurred as a result of dragging  */
/*         an icon from the drag container to the drop container.     */
/*                                                                    */
/*  RETURNS: nothing                                                  */
/*                                                                    */
/*--------------------------------------------------------------------*/
/**********************************************************************/
void DisplayTable( HWND hwndCnr, PCNRREC pCnrRec )
{
    FRAMECDATA  fcdata;
    HWND        hwndFrame, hwndLB;

    // If the container record that the user double-clicked on is currently
    // being displayed, set focus to that window and exit.

    if( pCnrRec->hwndDisplay )
    {
        WinSetActiveWindow( HWND_DESKTOP, pCnrRec->hwndDisplay );
        return;
    }

    (void) memset( &fcdata, 0, sizeof( FRAMECDATA ) );
    fcdata.cb            = sizeof( FRAMECDATA );
    fcdata.flCreateFlags = FCF_TITLEBAR | FCF_SYSMENU | FCF_SIZEBORDER |
                           FCF_MINMAX;

    hwndFrame = WinCreateWindow( HWND_DESKTOP, WC_FRAME, NULL,
                                 WS_CLIPCHILDREN, 100, 100, 400, 200,
                                 NULLHANDLE, HWND_TOP, 1, &fcdata, NULL );
    if( hwndFrame )
    {
        WinSetWindowPtr( hwndFrame, QWL_USER, pCnrRec );

        hwndLB = WinCreateWindow( hwndFrame, WC_LISTBOX, NULL,
                                  WS_VISIBLE | LS_HORZSCROLL | LS_NOADJUSTPOS,
                                  0, 0, 0, 0, hwndFrame, HWND_TOP, FID_CLIENT,
                                  NULL, NULL );
        if( hwndLB )
        {
            pfnwpLBFrame = WinSubclassWindow( hwndFrame, wpLBFrame );

            if( pfnwpLBFrame )
            {
                FILE *fh;
                PLBFRAMEDATA plbfd = (PLBFRAMEDATA) malloc( sizeof *plbfd );

                if( plbfd )
                {
                    plbfd->hwndCnr = hwndCnr;
                    plbfd->pCnrRec = pCnrRec;
                    WinSetWindowPtr( hwndFrame, QWL_USER, plbfd );

                    WinSendMsg( hwndFrame, WM_UPDATEFRAME, NULL, NULL );
                    WinSetWindowText( hwndFrame, pCnrRec->szTableName );

                    fh = fopen( pCnrRec->szRenderedFileName, "r" );
                    if( fh )
                    {
                        char szLine[ CCHMAXPATH ];

                        while( fgets( szLine, sizeof szLine, fh ) )
                        {
                            if( szLine[ strlen( szLine ) - 1 ] == '\n' )
                                szLine[ strlen( szLine ) - 1 ] = 0;

                            WinSendMsg( hwndLB, LM_INSERTITEM,
                                        MPFROMSHORT(LIT_END), MPFROMP(szLine) );
                        }

                        fclose( fh );
                    }
                    else
                        WinSendMsg( hwndLB, LM_INSERTITEM, MPFROMSHORT(LIT_END),
                                    "Could not open file!" );

                    pCnrRec->hwndDisplay = hwndFrame;

                    // Set on the In-Use emphasis to the container record that
                    // represents the table so the user knows that a listbox
                    // is on his desktop that represents the table.

                    WinSendMsg( hwndCnr, CM_SETRECORDEMPHASIS,
                                MPFROMP( pCnrRec ),
                                MPFROM2SHORT( TRUE, CRA_INUSE ) );

                    WinSetWindowPos( hwndFrame, HWND_TOP, 0, 0, 0, 0,
                                     SWP_SHOW | SWP_ACTIVATE | SWP_ZORDER );
                }
                else
                    WinDestroyWindow( hwndFrame );
            }
            else
                WinDestroyWindow( hwndFrame );
        }
        else
            WinDestroyWindow( hwndFrame );
    }
}

MRESULT EXPENTRY wpLBFrame( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 )
{
    switch( msg )
    {
        case WM_SYSCOMMAND:
            if( SHORT1FROMMP( mp1 ) == SC_CLOSE )
                WinDestroyWindow( hwnd );

            return 0;

        case WM_DESTROY:
        {
            PLBFRAMEDATA plbfd =
                    (PLBFRAMEDATA) WinQueryWindowPtr( hwnd, QWL_USER );

            if( plbfd )
            {
                // Take off the In-Use emphasis that was set on for the
                // container record when this listbox-frame was created.

                WinSendMsg( plbfd->hwndCnr, CM_SETRECORDEMPHASIS,
                            MPFROMP( plbfd->pCnrRec ),
                            MPFROM2SHORT( FALSE, CRA_INUSE ) );
                plbfd->pCnrRec->hwndDisplay = NULLHANDLE;
                free( plbfd );
            }
        }
    }

    return pfnwpLBFrame( hwnd, msg, mp1, mp2 );
}

/**********************************************************************/
/*------------------------- DeleteTempFiles --------------------------*/
/*                                                                    */
/*  DELETE THE TEMPORARY FILES USED BY THIS PROGRAM.                  */
/*                                                                    */
/*  PARMS: nothing                                                    */
/*                                                                    */
/*  NOTES: A temporary file is created in the current directory each  */
/*         time an icon is dragged from the 'drag' container to the   */
/*         'drop' container. Here we delete all those temporary files.*/
/*                                                                    */
/*  RETURNS: nothing                                                  */
/*                                                                    */
/*--------------------------------------------------------------------*/
/**********************************************************************/
void DeleteTempFiles()
{
    FILEFINDBUF3 ffb;
    HDIR         hdir = HDIR_SYSTEM;
    ULONG        cFiles = 1;
    char         szTempFileSpec[ CCHMAXPATH ];
    APIRET       rc;

    strcpy( szTempFileSpec, szCurrentPath );
    strcat( szTempFileSpec, BASE_TEMPFILE_NAME );
    strcat( szTempFileSpec, ".*" );

    rc = DosFindFirst( szTempFileSpec, &hdir, FILE_NORMAL,
                       &ffb, sizeof ffb, &cFiles, FIL_STANDARD );
    while( !rc )
    {
        DosDelete( ffb.achName );
        rc = DosFindNext( hdir, &ffb, sizeof ffb, &cFiles );
    }
}

/**********************************************************************/
/*------------------------- DestroyWindows ---------------------------*/
/*                                                                    */
/*  DESTROY THE WINDOWS AND RESOURCES USED IN THIS PROGRAM.           */
/*                                                                    */
/*  PARMS: nothing                                                    */
/*                                                                    */
/*  NOTES:                                                            */
/*                                                                    */
/*  RETURNS: nothing                                                  */
/*                                                                    */
/*--------------------------------------------------------------------*/
/**********************************************************************/
void DestroyWindows()
{
    WinDestroyWindow( hwndDrag );
    WinDestroyWindow( hwndDrop );

    WinDestroyPointer( hptrDragMe );
    WinDestroyPointer( hptrOpenMe );
}

/**********************************************************************/
/*------------------------------- Msg --------------------------------*/
/*                                                                    */
/*  DISPLAY A MESSAGE TO THE USER.                                    */
/*                                                                    */
/*  PARMS: a message in printf format with its parms                  */
/*                                                                    */
/*  NOTES:                                                            */
/*                                                                    */
/*  RETURNS: nothing                                                  */
/*                                                                    */
/*--------------------------------------------------------------------*/
/**********************************************************************/
void Msg( PSZ szFormat,... )
{
    PSZ     szMsg;
    va_list argptr;

    szMsg = (PSZ) malloc( MESSAGE_SIZE );
    if( szMsg )
    {
        va_start( argptr, szFormat );
        vsprintf( szMsg, szFormat, argptr );
        va_end( argptr );

        szMsg[ MESSAGE_SIZE - 1 ] = 0;

        WinAlarm( HWND_DESKTOP, WA_WARNING );
        WinMessageBox(  HWND_DESKTOP, HWND_DESKTOP, szMsg,
                        "Container Drag/Drop Sample Program", 1,
                        MB_OK | MB_MOVEABLE );
        free( szMsg );
    }
    else
    {
        DosBeep( 1000, 1000 );
        return;
    }
}

/*************************************************************************
 *                     E N D     O F     S O U R C E                     *
 *************************************************************************/
