/*********************************************************************
 *                                                                   *
 * MODULE NAME :  trgwin.c               AUTHOR:  Rick Fishman       *
 * DATE WRITTEN:  07-28-93                                           *
 *                                                                   *
 * MODULE DESCRIPTION:                                               *
 *                                                                   *
 *  Part of the 'DRGFILES' drag/drop sample program.                 *
 *                                                                   *
 *  This module takes care of all the code relating to the target    *
 *  window (the window that will be dropped on)                      *
 *                                                                   *
 * NOTES:                                                            *
 *                                                                   *
 *  Some GPI is done to create a 'bullseye' and to display text that *
 *  shows where to drop the icons. This code is not commented very   *
 *  well because it doesn't apply to the problem at hand - Drag/Drop.*
 *  (Plus I'm running out of time to do these samples <g>).          *
 *                                                                   *
 * FUNCTIONS AVALABLE TO OTHER MODULES:                              *
 *                                                                   *
 *   targCreateWindow                                                *
 *                                                                   *
 *                                                                   *
 * HISTORY:                                                          *
 *                                                                   *
 *  07-28-93 - Program coded.                                        *
 *                                                                   *
 *  Rick Fishman                                                     *
 *  Code Blazers, Inc.                                               *
 *  4113 Apricot                                                     *
 *  Irvine, CA. 92720                                                *
 *  CIS ID: 72251,750                                                *
 *                                                                   *
 *********************************************************************/

#pragma strings(readonly)   // used for debug version of memory mgmt routines

/*********************************************************************/
/*------- Include relevant sections of the OS/2 header files --------*/
/*********************************************************************/

#define  INCL_GPICONTROL
#define  INCL_GPILCIDS
#define  INCL_GPIPRIMITIVES
#define  INCL_GPIREGIONS
#define  INCL_WINERRORS
#define  INCL_WINDIALOGS
#define  INCL_WINFRAMEMGR
#define  INCL_WINPOINTERS
#define  INCL_WINRECTANGLES
#define  INCL_WINSHELLDATA
#define  INCL_WINSTDCNR
#define  INCL_WINSTDDRAG
#define  INCL_WINSYS
#define  INCL_WINWINDOWMGR

/**********************************************************************/
/*----------------------------- INCLUDES -----------------------------*/
/**********************************************************************/

#include <os2.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "drgfiles.h"

/*********************************************************************/
/*------------------- APPLICATION DEFINITIONS -----------------------*/
/*********************************************************************/

#define ID_FRAME           1
#define CNR_DROP           10

#define WINDOW_TITLE       "'Drop' Window"

#define CLIENT_CLASS       "DragDropTargetWindow"

#define FRAME_FLAGS        (FCF_TASKLIST   | FCF_TITLEBAR   | FCF_SYSMENU | \
                            FCF_MINMAX     | FCF_SIZEBORDER | FCF_ICON    | \
                            FCF_ACCELTABLE)

#define CONTAINER_STYLES   (CCS_EXTENDSEL | CCS_MINIRECORDCORE | \
                            CCS_AUTOPOSITION)

#define FONTFACE_NAME      "Helv"

/**********************************************************************/
/*---------------------------- STRUCTURES ----------------------------*/
/**********************************************************************/


/**********************************************************************/
/*----------------------- FUNCTION PROTOTYPES ------------------------*/
/**********************************************************************/

void FileDropped     ( HWND hwndClient, HSTR hstrFileName, USHORT fsDrop );
BOOL FillFattrs      ( HWND hwndClient );
BOOL FillFattrsStruct( HPS hps, HWND hwndClient, PFONTMETRICS pfm );
INT  GetFontCount    ( HPS hps, HWND hwndClient, PBOOL pfUseOurFont );
void PaintClient     ( HWND hwndClient );
void PaintText       ( HPS hps, HWND hwndClient, PRECTL prcl );
void DrawBullsEye    ( HPS hps, HWND hwndClient, PRECTL prcl );

FNWP wpClient;

/**********************************************************************/
/*------------------------ GLOBAL VARIABLES --------------------------*/
/**********************************************************************/

char szDropCnrTitle[] = "Dropped Icons";

PSZ szTextLine[] = { "DROP", "IT", "HERE!" };
#define TEXT_LINES (sizeof( szTextLine ) / sizeof( PSZ ))

FATTRS fattrs;
LONG   lcid = 1;

/**********************************************************************/
/*------------------------- targCreateWindow -------------------------*/
/*                                                                    */
/*  CREATE THE TARGET WINDOW                                          */
/*                                                                    */
/*  PARMS: anchor block handle                                        */
/*                                                                    */
/*  NOTES:                                                            */
/*                                                                    */
/*  RETURNS: frame window handle                                      */
/*                                                                    */
/*--------------------------------------------------------------------*/
/**********************************************************************/
HWND targCreateWindow( HAB hab )
{
    HWND       hwndFrame = NULLHANDLE, hwndClient, hwndCnr;
    FRAMECDATA fcdata;

    memset( &fcdata, 0, sizeof fcdata );
    fcdata.cb            = sizeof( FRAMECDATA );
    fcdata.idResources   = ID_RESOURCES;
    fcdata.flCreateFlags = FRAME_FLAGS;

    // Create a frame/client window and create a container which will be
    // placed on the right half of the client window.

    hwndFrame = WinCreateWindow( HWND_DESKTOP, WC_FRAME, NULL,
                                 FS_NOBYTEALIGN | WS_CLIPCHILDREN,
                                 0, 0, 0, 0, NULLHANDLE, HWND_TOP,
                                 ID_FRAME, &fcdata, NULL );
    if( hwndFrame )
    {
        WinRegisterClass( hab, CLIENT_CLASS, wpClient, CS_SIZEREDRAW, 0 );

        hwndClient = WinCreateWindow( hwndFrame, CLIENT_CLASS, NULL, 0, 0, 0, 0,
                                      0, NULLHANDLE, HWND_TOP, FID_CLIENT, NULL,
                                      NULL );
        if( hwndClient )
        {
            hwndCnr = WinCreateWindow( hwndClient, WC_CONTAINER, NULL,
                                       WS_VISIBLE | CONTAINER_STYLES,
                                       0, 0, 0, 0, hwndClient, HWND_TOP,
                                       CNR_DROP, NULL, NULL );
            if( hwndCnr )
                WinSetWindowText( hwndFrame, WINDOW_TITLE );
            else
            {
                WinDestroyWindow( hwndFrame );
                hwndFrame = NULLHANDLE;
                Msg( "WinCreateWindow(hwndCnr,%s) RC(%X)", WINDOW_TITLE,
                     HABERR( hab ) );
            }
        }
        else
        {
            WinDestroyWindow( hwndFrame );
            hwndFrame = NULLHANDLE;
            Msg( "WinCreateWindow(hwndClient,%s) RC(%X)", WINDOW_TITLE,
                 HABERR( hab ) );
        }
    }
    else
        Msg( "WinCreateWindow(%s) RC(%X)", WINDOW_TITLE, HABERR( hab ) );

    if( hwndFrame )
    {
        CNRINFO cnri;

        // Set container into Icon view and give it a read-only title

        cnri.cb           = sizeof( CNRINFO );
        cnri.pszCnrTitle  = szDropCnrTitle;
        cnri.flWindowAttr = CV_ICON | CA_CONTAINERTITLE | CA_TITLESEPARATOR |
                            CA_TITLEREADONLY;

        if( !WinSendMsg( hwndCnr, CM_SETCNRINFO, MPFROMP( &cnri ),
                         MPFROMLONG( CMA_FLWINDOWATTR | CMA_CNRTITLE ) ) )
        {
            WinDestroyWindow( hwndFrame );
            hwndFrame = NULLHANDLE;
            Msg( "CM_SETCNRINFO(%S) RC(%X)", WINDOW_TITLE, HABERR( hab ) );
        }
    }

    // Fill the FATTRS structure that will be used for text drawing
    if( hwndFrame )
        if( !FillFattrs( hwndClient ) )
            lcid = 0;

    // This allows PM to do all the dirty work involved in being the target of
    // a drop.

    if( hwndFrame )
        if( !DrgAcceptDroppedFiles( hwndClient, szTempDir, NULL, DO_COPY,0))
        {
            WinDestroyWindow( hwndFrame );
            hwndFrame = NULLHANDLE;
            Msg( "DrgAcceptDroppedFiles RC(%X)", HABERR( hab ) );
        }

    return hwndFrame;
}

/**********************************************************************/
/*---------------------------- wpClient ------------------------------*/
/*                                                                    */
/*  CLIENT WINDOW PROCEDURE.                                          */
/*                                                                    */
/*  PARMS: normal winproc parms                                       */
/*                                                                    */
/*  NOTES:                                                            */
/*                                                                    */
/*  RETURNS: MRESULT value                                            */
/*                                                                    */
/*--------------------------------------------------------------------*/
/**********************************************************************/
MRESULT EXPENTRY wpClient( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 )
{
    switch( msg )
    {
        case WM_SIZE:

            // Place the container in the right half of the client window.

            if( !WinSetWindowPos( WinWindowFromID( hwnd, CNR_DROP ), HWND_TOP,
                                  SHORT1FROMMP( mp2 ) / 2, 0,
                                  SHORT1FROMMP( mp2 ) / 2, SHORT2FROMMP( mp2 ),
                                  SWP_MOVE | SWP_SIZE ) )
                Msg( "wpClient WM_SIZE of Container RC(%X)", HWNDERR( hwnd ) );

            break;

        case WM_PAINT:
            PaintClient( hwnd );
            break;

        case WM_CLOSE:

            // Don't let the message queue be destroyed until the user has
            // closed *both* windows. If WM_CLOSE gets processed by the
            // default window proc, a WM_QUIT message will get posted to the
            // queue which will in effect end this program.

            if( hwndDrag )
            {
                WinDestroyWindow( PARENT( hwnd ) );
                hwndDrop = NULLHANDLE;
                WinSetActiveWindow( HWND_DESKTOP, hwndDrag );
                return 0;
            }

            break;

        case WM_COMMAND:

            // From the F3 accelerator key

            if( SHORT1FROMMP( mp1 ) == IDM_EXIT )
            {
                WinPostMsg( hwnd, WM_CLOSE, NULL, NULL );
                return 0;
            }

            break;

        case WM_DESTROY:

            // Free all container resources associated with any records that
            // were dropped on our window.

            WinSendDlgItemMsg( hwnd, CNR_DROP, CM_REMOVERECORD, NULL,
                               MPFROM2SHORT( 0, CMA_FREE ) );
            break;


        // These are the only messages that we need to process if we use the
        // DrgAcceptDroppedFiles API.

        case DM_DRAGERROR:     // In drgfiles.c
            return DragError( SHORT1FROMMP( mp1 ), SHORT2FROMMP( mp1 ),
                              (HSTR) mp2 );

        case DM_DRAGFILECOMPLETE:
            FileDropped( hwnd, (HSTR) mp1, SHORT1FROMMP( mp2 ) );
            return 0;
    }

    return WinDefWindowProc( hwnd, msg, mp1, mp2 );
}

/**********************************************************************/
/*--------------------------- FileDropped ----------------------------*/
/*                                                                    */
/*  PROCESS A DM_DRAGFILECOMPLETE MESSAGE.                            */
/*                                                                    */
/*  PARMS: client window handle,                                      */
/*         HSTR that contains the dropped filename,                   */
/*         flags that indicate what happened on the drop              */
/*                                                                    */
/*  NOTES: PM sends this to the target for each file that was dropped.*/
/*                                                                    */
/*  RETURNS: nothing                                                  */
/*                                                                    */
/*--------------------------------------------------------------------*/
/**********************************************************************/
void FileDropped( HWND hwndClient, HSTR hstrFileName, USHORT fsDrop )
{
    PCNRREC      pCnrRec;
    RECORDINSERT ri;
    HWND         hwndCnr = WinWindowFromID( hwndClient, CNR_DROP );

    if( !(fsDrop & DF_SUCCESSFUL) )
        return;

    // Insert a record into the container that represents the file that was
    // dropped.

    memset( &ri, 0, sizeof ri );
    ri.cb                 = sizeof( RECORDINSERT );
    ri.pRecordOrder       = (PRECORDCORE) CMA_END;
    ri.pRecordParent      = (PRECORDCORE) NULL;
    ri.zOrder             = (USHORT) CMA_TOP;
    ri.cRecordsInsert     = 1;
    ri.fInvalidateRecord  = TRUE;

    pCnrRec = WinSendMsg( hwndCnr, CM_ALLOCRECORD, MPFROMLONG( EXTRA_BYTES ),
                          MPFROMLONG( 1 ) );

    if( pCnrRec )
    {
        DrgQueryStrName( hstrFileName, sizeof pCnrRec->szFileName,
                         pCnrRec->szFileName );

        strcpy( pCnrRec->szFullFileName, szTempDir );
        strcat( pCnrRec->szFullFileName, "\\" );
        strcat( pCnrRec->szFullFileName, pCnrRec->szFileName );

        pCnrRec->mrc.hptrIcon = WinLoadFileIcon( pCnrRec->szFullFileName,
                                                 FALSE );
        if( !pCnrRec->mrc.hptrIcon )
            pCnrRec->mrc.hptrIcon =
                WinQuerySysPointer( HWND_DESKTOP, SPTR_QUESICON, FALSE );

        pCnrRec->mrc.pszIcon  = (PSZ) &pCnrRec->szFileName;

        if( !WinSendMsg( hwndCnr, CM_INSERTRECORD,
                         MPFROMLONG( pCnrRec ), MPFROMP( &ri ) ) )
            Msg( "FileDropped CM_INSERTRECORD RC(%X)", HWNDERR( hwndClient ) );
    }
    else
        Msg( "FileDropped CM_ALLOCRECORD RC(%X)", HWNDERR( hwndClient ) );
}

/**********************************************************************/
/*---------------------------- FillFattrs ----------------------------*/
/*                                                                    */
/*  DO EVERYTHING NECESSARY TO FILL IN THE FATTRS STRUCT FOR OUR FONT */
/*                                                                    */
/*  PARMS: client window handle                                       */
/*                                                                    */
/*  NOTES: We fill in a FATTRS structure for an outline font          */
/*                                                                    */
/*  RETURNS: TRUE or FALSE if successful or not                       */
/*                                                                    */
/*--------------------------------------------------------------------*/
/**********************************************************************/
BOOL FillFattrs( HWND hwndClient )
{
    BOOL         fUseOurFont, fSuccess = FALSE;
    PFONTMETRICS afm;
    HDC          hdc;
    LONG         cFonts;
    INT          i;
    HPS          hps = WinGetPS( hwndClient );

    if( hps )
        hdc = GpiQueryDevice( hps );
    else
        Msg( "FillFattrs WinGetPS RC(%X)", HWNDERR( hwndClient ) );

    if( hdc && hdc != HDC_ERROR )
    {
        cFonts = GetFontCount( hps, hwndClient, &fUseOurFont );

        afm = (PFONTMETRICS) malloc( cFonts * sizeof( FONTMETRICS ) );

        if( afm )
        {
            if( GPI_ALTERROR != GpiQueryFonts( hps, QF_PUBLIC,
                                        fUseOurFont ? FONTFACE_NAME : NULL,
                                        &cFonts, sizeof( FONTMETRICS ), afm ) )
            {
                for( i = 0; i < cFonts; i++)
                {
                    if( afm[ i ].fsDefn & FM_DEFN_OUTLINE )
                    {
                        if( FillFattrsStruct( hps, hwndClient, &afm[ i ] ) )
                            fSuccess = TRUE;

                        break;
                    }
                }
            }
            else
                Msg( "FillFattrs GpiQueryFonts RC(%X)", HWNDERR( hwndClient ) );

            free( afm );
        }
        else
            Msg( "Out of memory in FillFattrs!" );
    }
    else
        Msg( "FillFattrs GpiQueryDevice RC(%X)", HWNDERR( hwndClient ) );

    if( hps )
        WinReleasePS( hps );

    return fSuccess;
}

/**********************************************************************/
/*------------------------- FillFattrsStruct -------------------------*/
/*                                                                    */
/*  FILL THE FATTRS STRUCTURE FOR THE FONT THAT WE WILL USE.          */
/*                                                                    */
/*  PARMS: presentation space handle,                                 */
/*         client window handle,                                      */
/*         pointer to FONTMETRICS struct for our font                 */
/*                                                                    */
/*  NOTES:                                                            */
/*                                                                    */
/*  RETURNS: TRUE or FALSE if successful or not                       */
/*                                                                    */
/*--------------------------------------------------------------------*/
/**********************************************************************/
BOOL FillFattrsStruct( HPS hps, HWND hwndClient, PFONTMETRICS pfm )
{
    BOOL fSuccess = TRUE;

    fattrs.usCodePage = GpiQueryCp( hps );

    if( fattrs.usCodePage != GPI_ERROR )
    {
        (void) strcpy( fattrs.szFacename, pfm->szFacename );

        fattrs.usRecordLength = sizeof( FATTRS );
        fattrs.idRegistry = pfm->idRegistry;
        fattrs.fsFontUse = FATTR_FONTUSE_OUTLINE;
    }
    else
    {
        fSuccess = FALSE;
        Msg( "FillFattrsStruct GpiQueryCp RC(%X)", HWNDERR( hwndClient ) );
    }

    return fSuccess;
}

/**********************************************************************/
/*--------------------------- GetFontCount ---------------------------*/
/*                                                                    */
/*  GET A COUNT OF THE FONTS AVAILABLE.                               */
/*                                                                    */
/*  PARMS: presentation space handle,                                 */
/*         client window handle,                                      */
/*         pointer to bool that returns if we are using the original  */
/*           font requested                                           */
/*                                                                    */
/*  NOTES:                                                            */
/*                                                                    */
/*  RETURNS: count of fonts                                           */
/*                                                                    */
/*--------------------------------------------------------------------*/
/**********************************************************************/
int GetFontCount( HPS hps, HWND hwndClient, PBOOL pfUseOurFont )
{
    LONG cFonts, cFontsRequested = 0;

    *pfUseOurFont = TRUE;

    cFonts = GpiQueryFonts( hps, QF_PUBLIC, FONTFACE_NAME, &cFontsRequested, 0,
                            NULL );

    if( cFonts == GPI_ALTERROR )
    {
        cFonts = 0;

        Msg( "GetFontCount GpiQueryFonts RC(%X)", HWNDERR( hwndClient ) );
    }
    else if( !cFonts )
    {
        *pfUseOurFont = FALSE;

        cFontsRequested  = 0;

        cFonts = GpiQueryFonts( hps, QF_PUBLIC, NULL, &cFontsRequested, 0,
                                NULL );

        if( !cFonts )
            Msg( "GetFontCount GpiQueryFonts found NO FONTS in your system!" );
        else if( cFonts == GPI_ALTERROR )
        {
            cFonts = 0;

            Msg( "GetFontCount GpiQueryFonts RC(%X)", HWNDERR( hwndClient ) );
        }
    }

    return cFonts;
}

/**********************************************************************/
/*--------------------------- PaintClient ----------------------------*/
/*                                                                    */
/*  PROCESS THE CLIENT'S WM_PAINT MESSAGE                             */
/*                                                                    */
/*  PARMS: client window handle                                       */
/*                                                                    */
/*  NOTES:                                                            */
/*                                                                    */
/*  RETURNS: nothing                                                  */
/*                                                                    */
/*--------------------------------------------------------------------*/
/**********************************************************************/
void PaintClient( HWND hwndClient )
{
    POINTL ptl;
    HRGN   hrgn;
    RECTL  rcl;
    SWP    swpCnr;
    HPS    hps = WinBeginPaint( hwndClient, NULLHANDLE, NULL );

    WinQueryWindowPos( WinWindowFromID( hwndClient, CNR_DROP ), &swpCnr );
    WinQueryWindowRect( hwndClient, &rcl );

    GpiErase( hps );

    // Put a vertical separator between the left-side of the client and the
    // right side where the container is (the container does not have a border
    // so we have to draw it).

    GpiSetColor( hps, SYSCLR_WINDOWFRAME );
    ptl.x = swpCnr.x - 1;
    ptl.y = 0;
    GpiMove( hps, &ptl );
    ptl.y = rcl.yTop;
    GpiLine( hps, &ptl );

    // Set up a clipping region for the text (the left half of the client area)

    rcl.xRight = ptl.x;
    hrgn = GpiCreateRegion( hps, 1, &rcl );

    if( hrgn )
    {
        HRGN hrgnOld;
        LONG lComplexity = GpiSetClipRegion( hps, hrgn, &hrgnOld );

        if( lComplexity == RGN_ERROR )
            Msg( "GpiSetClipRegion RC(%X)", HWNDERR( hwndClient ) );
        else
        {
            GpiSetColor( hps, SYSCLR_WINDOWFRAME );
            GpiSetMix( hps, FM_OVERPAINT );
            GpiSetBackColor( hps, SYSCLR_WINDOWFRAME );
            GpiSetBackMix( hps, BM_LEAVEALONE );
            DrawBullsEye( hps, hwndClient, &rcl );

            GpiSetColor( hps, CLR_RED );
            PaintText( hps, hwndClient, &rcl );

            if( hrgnOld )
                GpiDestroyRegion( hps, hrgnOld );
        }

        GpiDestroyRegion( hps, hrgn );
    }
    else
        Msg( "GpiCreateRegion RC(%X)", HWNDERR( hwndClient ) );

    WinEndPaint( hps );
}

/**********************************************************************/
/*--------------------------- PaintText ------------------------------*/
/*                                                                    */
/*  PAINT TEXT IN THE CLIENT AREA                                     */
/*                                                                    */
/*  PARMS: presentation space handle,                                 */
/*         client window handle,                                      */
/*         pointer to rectangle to draw into                          */
/*                                                                    */
/*  1.                                                                */
/*                                                                    */
/*  OUTPUT: nothing                                                   */
/*                                                                    */
/*--------------------------------------------------------------------*/
/**********************************************************************/
void PaintText( HPS hps, HWND hwndClient, PRECTL prcl )
{
    RECTL rclText;

    if( !WinCopyRect( ANCHOR( hwndClient ), &rclText, prcl ) )
    {
        Msg( "PaintText WinCopyRect RC(%X)", HWNDERR( hwndClient ) );
        return;
    }

    if( GPI_ERROR == GpiCreateLogFont( hps, NULL, lcid, &fattrs ) )
        Msg( "PaintText GpiCreateLogFont RC(%X)", HWNDERR( hwndClient ) );
    else
    {
        if( GpiSetCharSet( hps, lcid ) )
        {
            int   cbLongestString = 0;
            ULONG cxChar, cyChar;
            ULONG cxDraw = (rclText.xRight - rclText.xLeft) + 1;
            SIZEF sizf;
            int   i;

            for( i = 0; i < TEXT_LINES; i++ )
                if( strlen( szTextLine[ i ] ) > cbLongestString )
                    cbLongestString = strlen( szTextLine[ i ] );

            cxChar = ((rclText.xRight - rclText.xLeft) + 1) / cbLongestString;
            cyChar = ((rclText.yTop - rclText.yBottom) + 1) / TEXT_LINES;

            sizf.cx = MAKEFIXED( cxChar, 0 );
            sizf.cy = MAKEFIXED( cyChar, 0 );

            if( GpiSetCharBox( hps, &sizf ) )
            {
                POINTL aptl[ TXTBOX_COUNT ];

                rclText.yBottom = rclText.yTop;

                for( i = 0; i < TEXT_LINES; i++ )
                {
                    // Center string horizontally

                    GpiQueryTextBox( hps, strlen( szTextLine[ i ] ),
                                     szTextLine[ i ], TXTBOX_COUNT, aptl );
                    rclText.xLeft = (cxDraw - aptl[ TXTBOX_CONCAT ].x) / 2;

                    rclText.yBottom -= cyChar;

                    if( GPI_ERROR == GpiCharStringPosAt( hps,
                                                (PPOINTL) &rclText, NULL, 0,
                                                strlen( szTextLine[ i ] ),
                                                szTextLine[ i ], NULL ) )
                        Msg( "PaintText GpiCharStringPosAt RC(%X)",
                            HWNDERR( hwndClient ) );
                }
            }
            else
                Msg( "PaintText GpiSetCharBox RC(%X)", HWNDERR( hwndClient ) );
        }
        else
            Msg( "PaintText GpiSetCharSet RC(%X)", HWNDERR( hwndClient ) );
    }
}

/**********************************************************************/
/*-------------------------- DrawBullsEye ----------------------------*/
/*                                                                    */
/*  DRAW A BULLSEYE IN THE CLIENT AREA                                */
/*                                                                    */
/*  PARMS: presentation space handle,                                 */
/*         client window handle,                                      */
/*         pointer to rectangle to draw into                          */
/*                                                                    */
/*  1.                                                                */
/*                                                                    */
/*  OUTPUT: nothing                                                   */
/*                                                                    */
/*--------------------------------------------------------------------*/
/**********************************************************************/
void DrawBullsEye( HPS hps, HWND hwndClient, PRECTL prcl )
{
    ARCPARAMS arcParams;
    POINTL    ptlCenter;
    FIXED     fixedMult;
    ULONG     cxDraw, cyDraw;
    int       i;

    cxDraw = (prcl->xRight - prcl->xLeft) + 1;
    cyDraw = (prcl->yTop - prcl->yBottom) + 1;

    ptlCenter.x = cxDraw / 2;
    ptlCenter.y = cyDraw / 2;

    GpiMove( hps, &ptlCenter );

    arcParams.lP = cxDraw / 10;
    arcParams.lQ = cyDraw / 10;
    arcParams.lR = 0;
    arcParams.lS = 0;

    GpiSetArcParams( hps, &arcParams );

    for( i = 1; i < 11; i++ )
    {
        fixedMult = MAKEFIXED( i, 0 );

        if( GPI_ERROR == GpiFullArc( hps, DRO_OUTLINE, fixedMult ) )
            Msg( "DrawBullsEye GpiFullArc RC(%X)", HWNDERR( hwndClient ) );
    }
}

/*************************************************************************
 *                     E N D     O F     S O U R C E                     *
 *************************************************************************/
