

#ifndef _visualobj_h_
#define _visualobj_h_





/*
 *
 *          Copyright (C) 1995, M. A. Sridhar
 *  
 *
 *     This software is Copyright M. A. Sridhar, 1995. You are free
 *     to copy, modify or distribute this software  as you see fit,
 *     and to use  it  for  any  purpose, provided   this copyright
 *     notice and the following   disclaimer are included  with all
 *     copies.
 *
 *                        DISCLAIMER
 *
 *     The author makes no warranties, either expressed or implied,
 *     with respect  to  this  software, its  quality, performance,
 *     merchantability, or fitness for any particular purpose. This
 *     software is distributed  AS IS.  The  user of this  software
 *     assumes all risks  as to its quality  and performance. In no
 *     event shall the author be liable for any direct, indirect or
 *     consequential damages, even if the  author has been  advised
 *     as to the possibility of such damages.
 *
 */



// VisualObject is the root of the inheritance hierarchy for the GUI
// objects of YACL. It encapsulates the common features of a view (or a
// window). A view is any visible area on the desktop available to the 
// user for input, output or both. VisualObject is an abstract class with a
// pure virtual destructor.
//
// Every VisualObject created by an application, with the exception of the
// main window, has a parent VisualObject. This parent-child relationship
// imposes a tree structure called the {\it view tree,} which is maintained
// by the Controller.  When creating a VisualObject, the application must
// specify its parent, shape (a Rectangle whose units are pixels in the
// parent's co-ordinate space) and view id (a long integer).
// The coordinates of the shape Rectangle are always interpreted relative to 
// the parent's top left corner.
//
// VisualObjects respond to {\it events}. Each event has a corresponding
// handler method declared in the VisualObject protocol.
//
// The VisualObject protocol also includes view-related methods that allow
// querying and manipulating its shape and title.
// There are essentially three categories of VisualObject methods:
// view-related, event related and model-related.
//
//
// YACL does not allow views as C++ static objects. Nor can a VisualObject 
// be explicitly destroyed. That is, {\it never\/} do a {\tt delete} of a
// VisualObject or any of its derivatives! In order to destroy a
// VisualObject, it must
// be sent an {\small\tt Event_Destroy} via the controller. Before it is
// destroyed, 
// the method WantToQuit() is invoked; if the latter returns TRUE, then the
// controller destroys the view and all its children. This
// method may be overriden in derived classes.
//
// Every VisualObject inherits two static protected instance variables {\tt
// _Application} and {\tt _Controller}, which are respectively the
// Application and Controller pointers for this application.
// 
// Of special interest is the method {\tt HandleEvent (UI_Event*)}.
// All events originating in a VisualObject are passed to the HandleEvent
// method on that VisualObject which provides default interpretation
// to these events by invoking the corresponding method outlined in
// the protected section. Being virtual, all these methods may be overwrriten
// in derived classes so that individual VisualObjects may interpret the
// same event differently.
//
// Every visual object exports a method \\
//          {\small\tt    UI_Font& UI_VisualObject::Font() }\\
// which returns a reference to the font used by the object. By default,
// each visual object uses the same font as its parent; the root of the
// window hierarchy creates a font when it is created, and all its
// descendants inherit this font. When a client program modifies the
// return value of the {\tt Font()} method of a particular VisualObject {\tt
// v}, this font change is propagated to all the descendants of {\tt v} in
// the view tree; but the fonts used by {\tt v}'s ancestors are not affected.



#if defined(__GNUC__)
#pragma interface
#endif

#if defined(__X_YACL__)
#include "ui/yaclxcls.h"
#endif

#include "ui/uidefs.h"   // Include this first!!

#include "base/map.h"
#include "base/clntset.h"

#include "ui/rectangl.h"
#include "ui/event.h"
#include "ui/cursor.h"
#include "ui/color.h"
#include "ui/evtbind.h"


class YACL_UI UI_Dialog;
class YACL_UI UI_Font;
class YACL_UI UI_VObjCollection;
class YACL_UI UI_CompositeVObject;
class YACL_UI UI_Controller;
class YACL_UI UI_Application;
class YACL_UI UI_DisplaySurface;

class YACL_UI YVE_Window;

//[ Global
enum UI_MouseButton { UIM_None = 0, UIM_Left, UIM_Middle, UIM_Right };
//] Global


class YACL_UI  UI_VisualObject: public CL_Object {

public:

    // ----------------- Querying and setting attributes ----------------

    virtual CL_Object& Model ();
    // Return the model used by this VisualObject. Although the signature
    // declares a return type of Object, the actual return type can be a
    // class derived from Object, and depends upon the class derived from
    // VisualObject to which this method is applied.

    virtual CL_String& Title ();
    // Return the title. The return type is a reference, so the application
    // may apply of the String methods to it, possibly modifying the title.
    // Such a modification is communicated via notification to the
    // VisualObject, which then updates the display.

    virtual UI_Rectangle& Shape ();
    // Return the client area of the VisualObject. Note that assigning to
    // the return value causes the object to resize and/or reposition itself.


    UI_VisualObject* Parent () const;
    // Return the parent of this VisualObject in the view tree. Return NULL
    // if this is the root of the visual object hierarchy.

    UI_ViewID ViewID () const {return _id;} ;
    // Returns   the integer  ID  that  is  passed  to   the view during
    // construction.  Unlike the handle, the  ID may be used to identify
    // objects uniquely between  runs of the  same application. If no ID
    // is passed during to the  constructor, then the  handle and id are
    // the same. In this case, the id is thus assigned at run time.

    virtual bool WantToQuit ();
    // Return TRUE if this object can be closed down. This is a hook method
    // meant for overriding by derived classes. The
    // default implementation returns TRUE. 

    
#if defined(__MS_WINDOWS__) || defined(__MS_WIN32__)
    virtual bool CreatedViaResource ();
    // [MS-Windows-specific] Return TRUE if this object was created via a
    // resource name. The default implementation returns TRUE if the parent
    // was created via a resource.

    virtual bool Has3DLook () const;
    // [MS-Windows-specific] Return TRUE if this VisualObject has the 3-d
    // look obtained using Microsoft's {\tt ctl3d.dll}.

#elif defined(__X_MOTIF__)  || defined(__X_YACL__)    
    virtual CL_String InstanceName ();
    // [X-windows-specific] Used to specify the instance name for the
    // widget when creating it. The default implementation returns the value
    // returned by the Application's {\tt InstanceName} method.

#endif
    // ------------------- View-related methods ---------------
    
    virtual bool Enable ();
    // Allow the VisualObject to respond to events.
    // By default, every VisualObject is enabled at creation.

    virtual bool Disable ();
    // Prevent view from responding to events.

    bool IsEnabled () { return _enabled;};
    // Return whether the VisualObject is enabled.

    bool ToggleEnabledState () { return IsEnabled() ? Disable() : Enable();};
    // Toggle the enabled state; i.e., disable it if it is enabled, and
    // enable it otherwise. The implementation uses the {\tt Enable} and
    // {\tt Disable} methods.
    
    bool SetEnabledState (bool enabled)
        { return enabled ? Enable() : Disable();};
    // Set the enabled state explicitly; i.e., if {\tt enabled} is TRUE,
    // enable this object, otherwise disable it.  The implementation uses
    // the {\tt Enable} and {\tt Disable} methods.

    virtual void MakeInvisible ();
    // Make the VisualObject invisible.
    
    virtual void MakeVisible ();
    // Make the VisualObject visible.

    bool IsVisible () {return _visible;};
    // Return whether the VisualObject is currently visible.

    void SetVisibility (bool visible);
    // Set the visibility explicitly; i.e., if {\tt visible} is TRUE, make
    // this object visible, otherwise make it invisible. The implementation
    // uses the {\tt MakeVisible} and {\tt MakeInvisible} methods.

    void ToggleVisibility ();
    // Make this VisualObject visible if it is currently invisible, and make
    // it invisible if it is visible. The implementation uses the {\tt
    // MakeVisible} and {\tt MakeInvisible} methods.

    virtual void Invalidate ();
    // Invalidate the client area of this VisualObject, so that the platform
    // sends it a Paint event.

    virtual void Invalidate (const UI_Rectangle&);
    // Invalidate the specified region of the client area.
    
    virtual void ScrollView (const UI_Rectangle& scrollArea, short xAmount,
                             short yAmount);
    // Scroll the {\tt scrollArea} rectangle within the client area of this
    // VisualObject by {\tt xAmount} and {\tt yAmount} pixels. These amounts
    // can be positive or negative. Note that {\tt scrollArea} must be
    // specified with respect to this VisualObject's window.
    
    bool ShowBorder ();
    // Show a border around this VisualObject. The exact look of the border
    // is platform-dependent.

    bool HideBorder ();
    // Hide the border around this VisualObject. The exact look of the border
    // is platform-dependent.

    bool BorderShown () const {return _borderShown;};

    virtual bool IsTabStop () const {return _isTabStop;};
    // Are we a tab stop in our parent composite?

    virtual void TabStop (bool b) {_isTabStop = b;} ;
    // Change our tab-stop status to the given value.
    

    virtual bool IsIconified();
    // Return TRUE if this VisualObject, or the Composite containing it, is
    // currently iconified.
    
    virtual bool TakeFocus ();
    // Tell this VisualObject to acquire focus. Return TRUE if successful,
    // FALSE otherwise.

    
    // ------------------ Display resource methods -------------------
    
    virtual UI_DisplaySurface* DisplaySurface ();
    // Return the display surface for this visual object. The returned
    // pointer is to an object owned by this VisualObject. The return value
    // is NULL if this VisualObject has not created a DisplaySurface for
    // itself.

    UI_Cursor& Cursor ();
    // Return the cursor displayed when the mouse is on this
    // object. Whenever the mouse enters the client area of a particular
    // VisualObject, the controller sets the current cursor to be the one
    // returned by this method. (The only exception to this is when the
    // controller is currently in "wait" state, displaying a wait cursor.)
    
    virtual UI_Font&   Font ();
    // Return the Font used by this object. The return value can be modified.

    virtual UI_Color& Foreground(); 
    // Return the foreground color of the visual object. This is the color
    // in which text is drawn on the VisualObject.
    
    void Foreground (const UI_Color &c);
    // Set the foreground color of the visual object to the specified
    // color. This is the color in which text is drawn on the VisualObject.
    
    virtual UI_Color& Background();
    // The background color used by the visual object.
    
    void Background (const UI_Color& c);
    // Set the background color of the visual object to the specified
    // color.
    
    // 
    // -------------------- Display surface methods -------------------
    // 

    // A display surface does not exist until a call to
    // CreateDisplaySurface(). One thus created stays alive until a call to
    // DestroyDisplaySurface() or till the VisualObject terminates.

    virtual UI_DisplaySurface& CreateDisplaySurface ();
    // Create a DisplaySurface for this VisualObject and return a reference
    // to it.
    
    virtual void DestroyDisplaySurface ();
    // Destroy this VisualObject's DisplaySurface if it has one.
    

    // ------------------ Dependents ------------------------------
    bool AddEventDependent (UI_EventType eType,
                            const UI_AbstractEventBinding& bind);
    // Add an event dependent to this VisualObject. After each event of
    // type {\tt eType} is
    // handled, the event dependent binding is executed; the 
    // parameter of the binding method will be the event.
    //
    // The return value of this method is TRUE if the binding was added, and
    // FALSE if this binding was already present or else memory allocation
    // failure occurred.

    bool RemoveEventDependent (UI_EventType eType,
                               const UI_AbstractEventBinding& bind);
    // Remove the given binding from the set of event dependents. The return
    // value of this method is TRUE if the binding was removed, and FALSE if
    // this binding was already absent or else memory allocation failure
    // occurred.
    
    // ------------------ Basic methods ----------------------------
    
    
    virtual UI_WindowClass WindowClass () const;
    // Return the platform-specific window class for this VisualObject. This
    // method is used by {\tt MakeVisualElement}.

    const char* ClassName () const { return "UI_VisualObject";};




protected:
    // 
    // ------------------Construction---------------------
    // 

    UI_VisualObject (UI_VisualObject* parent, 
                     const UI_Rectangle& shape, UI_ViewID id,
                     long style = -1);
    // Create a VisualObject with the given rectangle as the canvas.
    // All Visualobjects are enabled at creation. The id can be
    // used to uniquely identify the object between runs.

#if defined(__MS_WINDOWS__) || defined(__MS_WIN32__)
    UI_VisualObject (UI_VObjCollection* parent, UI_ViewID id,
                     UI_ViewHandle handle = 0);
    // [MS-Windows-specific, for YACL internal use only] For resource-based
    // construction.

#endif
    
    virtual ~UI_VisualObject () = 0;
    // [Pure virtual] Destructor.

    // ---------------- Initialization and termination -------------

    virtual void Initialize () {};
    // Called by the Controller, just after the visual element
    // of this object has been created. This is a hook,
    // meant to be overridden by derived classes. Its default implementation
    // does nothing.

    virtual void Finalize () {};
    // Called just before the visual element is destroyed. This is a hook,
    // meant to be overridden by derived classes. Its default implementation
    // does nothing.
    //
    // Warning: if you override this method, remember that the overriding
    // code must not access any children of this VisualObject; those
    // children will have been destroyed by the time this method is
    // called. If you want to access children, override the {\tt WantToQuit}
    // method, not the {\tt Finalize} method.
    
    friend UI_Controller;



    // ------------------Event Handling------------------

    // The following event-handling methods are intended only for being
    // overridden by derived classes; they are never (and should never be)
    // called by any object except the Controller.
    
    virtual bool HandleEvent (UI_Event* anEvent);
    // HandleEvent: the ``gateway'' through which this object receives
    // events from the controller. This method is called by the controller on
    // receiving an event on this object (the event destination). This
    // virtual method simply calls the non-virtual method
    // {\tt ProcessEvent (UI_Event* e)} that actually process all events.
    // Therefore, the {\tt HandleEvent} method may be overridden in derived 
    // classes to capture special events and deal with them. However,
    // the overriding method {\it must\/} invoke {\tt ProcessEvent(anEvent)}
    // to deal with events not handled by it. A sample implementation
    // of this method in a derived class may be:
    // \par{\small\begin{verbatim}
    //      bool UI_DerivedClass::HandleEvent (UI_Event *e)
    //      {
    //           if(e satisfies some condition) {
    //               do the special processing;
    //           
    //           } else
    //               return ProcessEvent(e);
    //      } 
    // \end{verbatim}
    // }\par

     
    bool ProcessEvent (UI_Event *e);
    // ProcessEvent: A non-virtual method that provides default
    // implementation for all events that occur on every object. Special
    // events, specific to a class, may be taken care of by the
    // HandleEvent() method.
 
    // The following comprise the entire set of events that are
    // recognised by any visual object.
    // Each event returns a bool response --- TRUE if the event is
    // recognized and processed by it, FALSE otherwise. Each of the
    // following methods gets invoked by the ProcessEvent() method when
    // the corresponding event occurs on the object. If the method returns
    // FALSE, the controller sends the event up the view tree to the parent,
    // grandparent and so forth. If the method returns TRUE, the event is
    // not sent up the view tree.
    // The default implementations provided in VisualObject, for each
    // of the methods, do nothing and return FALSE.
         

    // 
    // --------------------Hard Events------------------
    // 
 
    // 
    // Mouse Events:
    // 
   
   
    virtual bool ButtonDown (const UI_Point& pt, UI_MouseButton btn,
                              bool shiftKey, bool ctrlKey); 
    // Inform the VisualObject that the mouse button {\tt btn} was pressed
    // on it. The parameters are the position of the mouse, the mouse
    // button, and whether the shift or control key was depressed at the
    // time of click.

    virtual bool ButtonUp (const UI_Point& pt, UI_MouseButton btn);
    // Inform the VisualObject that the mouse button {\tt btn} was released
    // on it at position {\tt pt}.
  
    virtual bool DoubleClick (const UI_Point& pt, UI_MouseButton btn);
    // Inform the VisualObject that the mouse button {\tt btn} was
    // double-clicked on it at position {\tt pt}.

    virtual bool ViewEnter (const UI_Point& p);
    // Informs the VisualObject that the mouse has just moved into its
    // client area. (This has nothing to do with focus.)
    // The parameter is the point of entry.
 
    virtual bool ViewLeave (const UI_Point& p);
    // Inform the VisualObject that the mouse has just moved out of its
    // client area.  The parameter is the point of exit.

    virtual bool MouseMove (const UI_Point& cursorPos);
    // Inform the VisualObject that the mouse has moved on its client area.
 
    // 
    // KeyBoard Events: 
    // 

    virtual bool GetFocus ();
    // Inform the VisualObject that it has acquired focus.

    virtual bool LoseFocus ();
    // Inform the VisualObject that it has lost focus.

    virtual bool KeyTyped (char aKey);
    // Inform the VisualObject with current keyboard control that a key  has
    // been pressed. The paremeter is the ASCII value of the key. This event
    // is only sent if the VisualObject has focus.


    // 
    // Motion and Sizing Events:
    // 

    virtual bool Reconfigure (const UI_Rectangle& new_rect);
    // Inform the VisualObject that it has been either resized or moved or
    // both. The parameter specifies the new shape rectangle.

    // 
    // ----------------Soft Events------------------------
    // 


    // 
    // Display:
    // 

    virtual bool Paint (const UI_Rectangle& invalidRect);
    // Inform the VisualObject to repaint the given rectangle in its view
    // area. The behavior of the default implementation of this method is
    // system dependent. Under Windows, it fills the invalid rectangle with
    // COLOR_WINDOW if this VisualObject's window class is the YACL class.
    // Under OS/2, it fills it with white if this VisualObject's window
    // class is the YACL class. Under X, it does nothing. In all cases, it
    // returns FALSE.

    virtual bool Select () { return FALSE; };
    // Inform the VisualObject that it received an {\tt Event_Select}. This
    // means that this VisualObject was somehow ``selected'' -- exact
    // semantics if this is left to derived classes.

    virtual bool Iconify () {return FALSE; };
    // Tell this VisualObject that it (or its parent Composite) has been
    // iconified.

    
    virtual bool Deiconify () {return FALSE; };
    // Tell this VisualObject that it (or its parent Composite) has been
    // de-iconified.

    
    // 
    // ------------- Creating and destroying the visual element -----
    //

    virtual bool MakeVisualElement ();
    // Create the visual element for this VisualObject.  The
    // implementation creates a window whose class is specified by the
    // virtual {\tt WindowClass} method. Derived classes may override either 
    // this method or {\tt WindowClass} to determine the kind of visual
    // element created.

    virtual bool DestroyVisualElement ();
    // Destroy the visual element. This method is called by the Controller
    // just before the VisualObject itself is destroyed.

    virtual void CloseDown  ();
    // Called in response to a CloseDown event. The default
    // implementation calls {\tt WantToQuit}, and if the latter returns
    // TRUE, calls {\tt Application->Destroy} on this object.


#if defined(__X_MOTIF__)
    virtual void _SetupStyle (void* arg, short& argn);
    // [X-Motif-specific, YACL internal use only] Set the resources for
    // this widget. Called by MakeVisualElement. The first parameter will be
    // a pointer to an {\tt Arg} array.

    virtual bool _CursorChanged();
    // [X-Motif-specific, YACL internal use only] Called when the
    // application changes this VisualObject's cursor.
    
#elif defined(__X_YACL__)
    virtual void DrawVisualElement ();
    // [X-YACL-specific, internal use only] Draw the YACL visual element for
    // this object.
    
#endif
    
    // ------------------- Miscellaneous methods -----------------------

    void _SetShapeRectangle (const UI_Rectangle& r);
    // Set our shape rectangle to {\tt r}. This method is to ensure that
    // when we assign to our shape rectangle, the shape doesn't notify us
    // that we changed it.
    
    void _SetTitle (const char* newTitle);
    // Set our title. This method is to ensure that when we assign
    // to our title, it doesn't notify us that we changed
    // it.
    
    void _MakeNewFont ();
    // _MakeNewFont: called to create a new font for this VisualObject.

    UI_Font* OurFont ();
    // [For YACL interal use only] return the font used by this VisualObject.

    virtual bool _ShapeRectChanged ();
    // The application has invoked a method on our shape rectangle that
    // modified it; so our shape rectangle is telling us that it's modified.

    virtual bool _TitleChanged ();
    // TitleChanged: called by our title string to notify us that it has
    // changed. 

    virtual bool _FontChanged ();
    // The method that our Font object uses to notify that someone
    // modified our font.

    virtual bool _ModelChanged () {return TRUE;};
    // The method that our model uses to notify us that its value has
    // changed.
    
    void _SetModelValue (const CL_Object& m);
    // Assign {\tt m} to the model for this VisualObject. This method first
    // removes this object from the model's dependent set, performs the
    // assignment, and puts the dependency back.  This is done so that this
    // VisualObject does not receive a notification of change when the
    // assignment takes place.
    
    virtual void _PrivateInitialize ();
    // [For YACL internal use only] The real initialization method.

    void _FillWithBgColor (const UI_Rectangle&);
    // [For YACL internal use only] fill the given rectangle with this
    // VisualObject's background color. Must be called {\it only\/} in
    // response to a Paint event.
    
    void _PrivateFinalize ();
    // [For YACL internal use only] The real finalization method, called by
    // the Controller before destroying this object.
    
    
    bool _SetPlatformFont (UI_ViewHandle h);
    // Utility function: tell the platform to set this VisualObject's font
    // according to its  {\tt _font}  value.
    
    virtual bool _DoShowBorder (bool);
    // [For YACL internal use only] The actual border display code.
    
    virtual void AddChild    (UI_VisualObject* child);
    // [YACL internal use only] Called when a new child has been created.

    virtual UI_VisualObject* RemoveChild (UI_VisualObject* view);
    // [YACL internal use only] Called when a child is being destroyed.

public:


    // -------------------- For YACL internal use only -------------------

    virtual UI_ViewHandle ViewHandle () const;
    // [YACL internal use only.]
    // Returns the platform-specific handle that identifies this
    // VisualObject. A handle is assigned by the platform API at run time
    // and may thus change values between runs.

    virtual bool PropagateFontChange (UI_VisualObject* initiator);
    // [YACL Internal use only] Called to tell this VisualObject that its
    // font, or perhaps its parent's font, has changed. The parameter is the
    // origin of the font change propagation wave. The default
    // implementation sets the font used for this object in the underlying
    // window system. The method returns TRUE on success, FALSE on failure.
    // Child classes override this method according to their implementation.

    virtual bool AdjustPosition ();
    // [YACL internal use only]
    // This method is called by this VisualObject's parent, to tell it that
    // the parent is a CompositeVObject with a menu bar. This call is made
    // so that this VisualObject may reposition itself to account for the
    // menu bar's height.
  
#if defined(__OS2__)
    ulong& Style ()  {return _style;};
    // [OS/2-specific; internal use only.]

#elif defined(__MS_WINDOWS__) || defined(__MS_WIN32__)
    ulong& Style ()  {return _style;};
    // [MS/Windows-specific; internal use only.]

    bool BgColorChanged () const {return _bgColorChanged;};
    // [MS/Windows-specific; internal use only.]
    
    bool FgColorChanged () const {return _fgColorChanged;};
    // [MS/Windows-specific; internal use only.]
    
    virtual long WindowProcHook (HWND h, UINT msg, WPARAM wParam, LPARAM
                                 lParam);
    // [MS Windows specific, YACL internal use only] Used for processing MS
    // Windows messages that don't directly correspond to YACL events.

    virtual LRESULT DefaultProc (HWND h, UINT msg, WPARAM wParam,
                                 LPARAM lParam);
    // [MS Windows specific, YACL internal use only] Default processing.
    
#elif defined (__X_MOTIF__)

    // -------------------- For YACL internal use only ------------------

    operator _WidgetRec* () { return _xwidget; };
    // [X-windows-specific; for YACL Internal use only]
    // Return the Widget corresponding to the visual object for the X
    // implementation.

#endif
    
    
protected:
    // 
    // ------------------Instance variables------------------
    // 
  
    UI_VisualObject*     _parent;
    UI_Rectangle         _shape;
    CL_String            _title;
    CL_IntPtrMap*        _eventDependents;
    
#if defined(__MS_WINDOWS__) || defined(__MS_WIN32__)
    UI_ViewHandle        _handle;
    ulong                _style;
#elif defined(__OS2__)
    UI_ViewHandle        _handle;
    UI_ViewHandle        _borderHandle;
    ulong                _style;
#elif defined(__X_MOTIF__) 
    UI_ViewHandle        _xwidget;
    CL_String            _instanceName;
#elif defined(__X_YACL__)
    UI_ViewHandle        _window;
    CL_String            _instanceName;
    YVE_Window*          _visualElement;
#endif
    UI_ViewID            _id;              // An assigned ID used to uniquely
                                           // identify a view
    UI_DisplaySurface*   _displaySurface;  // Not initialized until call to
                                           // CreateDisplaySurface
    UI_Cursor            _cursor;
    UI_Font*             _font;
    UI_Color             _bgColor;
    UI_Color             _fgColor;
    
    CL_Object*           _model  ; 


    bool                 _isTabStop;
    bool                 _ownFont;         // Do we own our font object?
    bool                 _created;         // Has the interface element for
                                           // this  object been created?
                                           // (Set by Initialize)
    bool                 _visible;         // Are we currently visible?
    bool                 _enabled;         // Are we currently enabled?
    bool                 _borderShown;

    bool                 _bgColorChanged;  // Set if the app set this
                                           // object's background before
                                           // the window is displayed
    bool                 _fgColorChanged;
    
    //  Static instance variables:
    static UI_Controller*  _Controller;
    static UI_Application* _Application;
    
    bool _PrivateHandleEvent (UI_Event* e);
    // The real event gateway.


private:
    virtual bool _PrivateHandleChildEvent (UI_Event& e);
    // The real child event gateway.
    
    void _Init (UI_VisualObject* parent, UI_ViewID id);
    // Contains all the initialization code. Called from the constructors.

    virtual void _DoSetBackground ();
    // Set the background color.
    
    virtual void _DoSetForeground ();
    // Set the foreground color.
    
    UI_VisualObject (const UI_VisualObject&) {};
    // Private copy constructor to prevent copying.

    // void UI_VisualObject::operator= (const CL_Object&) {};
    // Private assignment operator to prevent assignment.

#if defined(__X_MOTIF__) || defined(__X_YACL__)
    bool _hasBeenInited; // This is a terrible hack, intended to ensure that
                         // the VObj is initialized at the first paint event

#endif
    

};


inline CL_Object& UI_VisualObject::Model()
{
    return *_model;
}


inline UI_ViewHandle UI_VisualObject::ViewHandle() const
{
#if defined (__MS_WINDOWS__) || defined(__MS_WIN32__)
    return _handle;
#elif defined(__OS2__)
    return _handle;
#elif defined(__X_YACL__)
    return _window;
#elif defined (__X_MOTIF__)
    return (UI_ViewHandle) (_xwidget);
#endif
}




inline UI_DisplaySurface* UI_VisualObject::DisplaySurface ()
{
    return _displaySurface;
}

inline UI_Cursor& UI_VisualObject::Cursor ()
{
    return _cursor;
}


inline UI_Color& UI_VisualObject::Foreground ()
{
    return _fgColor;
}

inline UI_Color& UI_VisualObject::Background ()
{
    return _bgColor;
}


inline UI_VisualObject* UI_VisualObject::Parent() const
{
    return _parent;
}

inline UI_Rectangle& UI_VisualObject::Shape()
{
    return _shape;
}

inline CL_String& UI_VisualObject::Title() 
{
    return _title;
}

inline void UI_VisualObject::SetVisibility (bool visible)
{
    if (visible)
        MakeVisible();
    else
        MakeInvisible();
}

inline void UI_VisualObject::ToggleVisibility ()
{
    if (IsVisible())
        MakeInvisible();
    else
        MakeVisible();
}

#if defined(__OS2__)
long _YACLWindowHeight (UI_ViewHandle h);
#endif


#endif
