




/*
 *
 *          Copyright (C) 1995, M. A. Sridhar
 *  
 *
 *     This software is Copyright M. A. Sridhar, 1995. You are free
 *     to copy, modify or distribute this software  as you see fit,
 *     and to use  it  for  any  purpose, provided   this copyright
 *     notice and the following   disclaimer are included  with all
 *     copies.
 *
 *                        DISCLAIMER
 *
 *     The author makes no warranties, either expressed or implied,
 *     with respect  to  this  software, its  quality, performance,
 *     merchantability, or fitness for any particular purpose. This
 *     software is distributed  AS IS.  The  user of this  software
 *     assumes all risks  as to its quality  and performance. In no
 *     event shall the author be liable for any direct, indirect or
 *     consequential damages, even if the  author has been  advised
 *     as to the possibility of such damages.
 *
 */



#if defined(__GNUC__)
#pragma implementation
#endif


#include "ui/scrolbar.h"
#include "ui/cntroler.h"
#include "ui/interval.h"

#if defined(__MS_WINDOWS__) || defined(__MS_WIN32__)
#    include <windows.h>

#elif defined(__X_MOTIF__)
#    if defined(__GNUC__) && __GNUC_MINOR__ >= 7
#        include <string.h>  // Without this, the X includes barf
#    endif
#    include <Xm/ScrollBar.h> 
#    include <iostream.h> // DEBUG
#elif defined(__X_YACL__)
#    include <iostream.h> // DEBUG
#    include <stdlib.h>
#    include "ui/support/x_yacl/window.h"
#    include "ui/shadorec.h"
#    include "ui/shadotrg.h"
#    include "ui/dsplsurf.h"
#    include "ui/dsinmem.h"
#    define MIN_SLIDER_SIZE 8  // Slider is at least this big
#endif

#if defined(__GNUC__) && __GNUC_MINOR__ >= 6
template class CL_Binding0<UI_ScrollBar>;
#elif defined(_MSC_VER)
template CL_Binding0<UI_ScrollBar>;
#endif


/*------------------ ScrollBar Object ---------------------------*/

typedef CL_Binding0<UI_ScrollBar> Bind;

UI_ScrollBar::UI_ScrollBar (UI_VisualObject* p,
                            const UI_Rectangle& shape,
                            UI_ViewID id, long bstyle, bool vert)
:  UI_SimpleVObject (p, shape, id, bstyle), _range (0, 100),
   _vertical(vert)
#if defined(__X_YACL__)
    , _gutterColor (0.62, 0.58, 0.62)
#endif
   
{ 
    _model = new CL_Interval (0,0);
    _ownModel = TRUE;
    Bind bind (this, &UI_ScrollBar::_RangeChanged);
    _range.AddDependent (bind); // Second parameter unused
    _lineAmount = 1;
    _pageAmount = 10;
    _smoothScroll = TRUE;
#if defined(__X_YACL__)
    _window        = 0;
    _visualElement = NULL;
    _sliderPos     = 0;
    _sliderSize    = 40;
    _mouseIsDown   = FALSE;
    _memDS         = NULL;
#endif
}


UI_ScrollBar::~UI_ScrollBar()
{
}



UI_WindowClass UI_ScrollBar::WindowClass () const
{
#if defined(__MS_WINDOWS__) || defined(__MS_WIN32__)
    return "scrollbar";
#elif defined(__OS2__)
    return WC_SCROLLBAR;
#elif defined(__X_MOTIF__)
    return xmScrollBarWidgetClass;
#endif
}


void UI_ScrollBar::_PrivateInitialize()
{
    UI_SimpleVObject::_PrivateInitialize();

#if defined(__MS_WINDOWS__) || defined(__MS_WIN32__)
    if (_handle) {
        _SetRange ();
        CL_Interval& interval = *(CL_Interval*) _model;
        _SetPosition (interval.Low());
    }
#elif defined(__OS2__)
    _RangeChanged (); // Force an initialization of range
    _ModelChanged (); // Force initialization of model
#elif defined(__X_MOTIF__)
    Arg args[1];
    XtSetArg    (args[0], XmNorientation,
                 _vertical ? XmVERTICAL : XmHORIZONTAL);
    XtSetValues (_xwidget, args, 1);

    _RangeChanged (); // Force an initialization of range
    _ModelChanged (); // Force initialization of model
    XtAddCallback (_xwidget, XmNdragCallback, 
                   &UI_ScrollBar::ScrollBarCallback, (XtPointer) this);
    XtAddCallback (_xwidget, XmNincrementCallback, 
                   &UI_ScrollBar::ScrollBarCallback, (XtPointer) this);
    XtAddCallback (_xwidget, XmNpageIncrementCallback, 
                   &UI_ScrollBar::ScrollBarCallback, (XtPointer) this);
    XtAddCallback (_xwidget, XmNdecrementCallback, 
                   &UI_ScrollBar::ScrollBarCallback, (XtPointer) this);
    XtAddCallback (_xwidget, XmNpageDecrementCallback, 
                   &UI_ScrollBar::ScrollBarCallback, (XtPointer) this);
    XtAddCallback (_xwidget, XmNtoBottomCallback, 
                   &UI_ScrollBar::ScrollBarCallback, (XtPointer) this);
    XtAddCallback (_xwidget, XmNtoTopCallback, 
                   &UI_ScrollBar::ScrollBarCallback, (XtPointer) this);
    XtAddCallback (_xwidget, XmNvalueChangedCallback, 
                   &UI_ScrollBar::ScrollBarCallback, (XtPointer) this);
#endif
}


#if defined(__X_MOTIF__)
void UI_ScrollBar::ScrollBarCallback (Widget , void* client, void* call)
{
    UI_ScrollBar* bar = (UI_ScrollBar *) client;
    UI_EventType type;
    XmScrollBarCallbackStruct* cb = (XmScrollBarCallbackStruct*) call;
    switch (cb->reason) {
    case XmCR_DRAG:
        type = Event_Scroll;
        break;
        
    case XmCR_DECREMENT:
        type = Event_ScrollBackwardLine;
        break;
        
    case XmCR_INCREMENT:
        type = Event_ScrollForwardLine;
        break;
        
    case XmCR_PAGE_DECREMENT:
        type = Event_ScrollBackwardPage;
        break;
        
    case XmCR_PAGE_INCREMENT:
        type = Event_ScrollForwardPage;
        break;

    case XmCR_TO_TOP:
        type = Event_ScrollToBegin;
        break;

    case XmCR_TO_BOTTOM:
        type = Event_ScrollToEnd;
        break;

    case XmCR_VALUE_CHANGED:
        type = Event_FinishScroll;
        break;

    default:
        CL_Error::Warning ("UI_ScrollBar callback: unknown reason");
        break;
    };
    int val, slsize, inc, page_inc;
    XmScrollBarGetValues (bar->_xwidget, &val, &slsize, &inc, &page_inc);
    bar->_SetModelValue (CL_Interval (val, val+slsize-1));
    UI_Event* evt = new UI_Event (type, bar, bar);
    evt->ScrollAmount() = val;
    _Controller->AddEvent (evt);
}

#endif // __X_MOTIF__


#if defined(__MS_WINDOWS__) || defined(__MS_WIN32__)
void UI_ScrollBar::_SetPosition (long pos)
{
    if (_handle)
        SetScrollPos (_handle, SB_CTL, pos, TRUE);
}
void UI_ScrollBar::_SetRange ()
{
    if (_handle) {
        CL_Interval& interval = *(CL_Interval*) _model;
        SetScrollRange ((HWND) _handle, SB_CTL, _range.Low(), 
                        _range.High() - interval.Length () + 1,
                        TRUE);
    }
}
#endif


#if !defined(__X_YACL__)
bool UI_ScrollBar::HandleEvent (UI_Event* e)
{
    UI_EventType type = e->Type();
    if (type < Event_Scroll || type > Event_ScrollToPosition) {
        return ProcessEvent (e);
    }

    CL_Interval& value = *(CL_Interval*) _model;
    long pos = value.Low();
    switch (type) {
    case Event_ScrollToBegin:
        pos = _range.Low();
        break;
        
    case Event_ScrollToEnd:
        pos = _range.High() - value.Length() + 1;
        break;
        
    case Event_ScrollForwardLine:
        pos = minl (_range.High(), pos + _lineAmount);
        break;
        
    case Event_ScrollBackwardLine:
        pos = maxl (_range.Low(), pos - _lineAmount);
        break;

    case Event_ScrollForwardPage:
        pos = minl (_range.High(), pos + _pageAmount);
        break;
        
    case Event_ScrollBackwardPage:
        pos = maxl (_range.Low (), pos - _pageAmount);
        break;

    case Event_ScrollToPosition:
    case Event_Scroll:
        pos = e->ScrollAmount();
        break;

    default:
        break;
    }        
#if defined(__MS_WINDOWS__) || defined(__MS_WIN32__)
    if (type != Event_FinishScroll) {
        long lo, hi, maxLo = _range.High() - value.Length() + 1;
        if (pos <= maxLo) {
            lo = pos;
            hi = pos + value.Length() - 1;
        }
        else {
            lo = maxLo;
            hi = _range.High();
        }
        _SetModelValue (CL_Interval (lo, hi));
        _SetPosition (pos);
    }
#elif defined(__OS2__)
    long lo, hi, maxLo = _range.High() - value.Length() + 1;
    if (pos <= maxLo) {
        lo = pos;
        hi = pos + value.Length() - 1;
    }
    else {
        lo = maxLo;
        hi = _range.High();
    }
    _SetModelValue (CL_Interval (lo, hi));
    if (type != Event_FinishScroll)
        WinSendMsg (_handle, SBM_SETPOS, MPFROMSHORT (pos), 0);
#endif
    if (type != Event_Scroll || _smoothScroll) {
        _clientSet.NotifyAll (*e);
    }
    return FALSE; // So the parent gets to see the event
}
#endif // Not X_YACL

CL_Interval& UI_ScrollBar::Range ()
{
    return _range;
}



#if defined(__X_MOTIF__)
static void _SetResources (Widget scroll, const CL_Interval& model,
                           const CL_Interval& range, short line, short page)
{
    if (model.Low () < range.Low() || model.High() > range.High())
        return; // Refuse the change
    Arg args[2];
    XtSetArg    (args[0] , XmNminimum, range.Low  ());
    XtSetArg    (args[1] , XmNmaximum, range.High () /* + model.Length() */);
    XtSetValues (scroll, args, 2);
    int sliderPos  = maxl (range.Low(), minl (model.Low(), range.High()));
    int sliderSize = maxl (1, minl (model.Length(), range.Length()));
    sliderSize = minl (sliderSize, range.High() - range.Low());
    XmScrollBarSetValues  (scroll, sliderPos, sliderSize,  line, page,
                          FALSE);
//     UI_ScrollBar* b = (UI_ScrollBar*) YACLApp()->Controller()[scroll];
//     if (b)
//         cout << b->InstanceName () << " ";
//     cout << "sliderPos " << sliderPos << " sliderSize "
//          << sliderSize 
//          << " model " << model << " range " << range << endl; // DEBUG
}

#endif

bool UI_ScrollBar::_ModelChanged ()
{
#if defined(__MS_WINDOWS__) || defined(__MS_WIN32__)
    _SetPosition (((CL_Interval*) _model)->Low());
#elif defined(__OS2__)
    if (_handle) {
        CL_Interval& value = *(CL_Interval*) _model;
        short range = _range.High() - _range.Low() + 1 - value.Length();
        WinSendMsg (_handle, SBM_SETSCROLLBAR, (MPARAM) value.Low(),
                    MPFROM2SHORT (0, range));
        WinSendMsg (_handle, SBM_SETTHUMBSIZE,
                    MPFROM2SHORT (value.Length(), _range.Length()), 0);
    }
#elif defined(__X_MOTIF__)
    if (_xwidget) {
        CL_Interval& value = *(CL_Interval*) _model;
        _SetResources (_xwidget, value, _range, _lineAmount, _pageAmount);
    }
#elif defined(__X_YACL__)
    _AdjustModel ();
    _ConfigureSlider();
    Invalidate ();
#endif
    return TRUE;
}

bool UI_ScrollBar::_RangeChanged ()
{
#if defined(__MS_WINDOWS__) || defined(__MS_WIN32__)
    _SetRange ();
#elif defined(__OS2__)
    if (_handle) {
        CL_Interval& value = *(CL_Interval*) _model;
        short range = _range.High() - _range.Low() - 1 - value.Length();
        WinSendMsg (_handle, SBM_SETSCROLLBAR, (MPARAM) value.Low(),
                    MPFROM2SHORT (0, range));
        WinSendMsg (_handle, SBM_SETTHUMBSIZE,
                    MPFROM2SHORT (value.Length(), _range.Length()), 0);
    }
                    
#elif defined(__X_MOTIF__)
    if (_xwidget) {
        CL_Interval& value = *(CL_Interval*) _model;
        _SetResources (_xwidget, value, _range, _lineAmount, _pageAmount);
    }
#elif defined(__X_YACL__)
    _AdjustModel ();
    _ConfigureSlider();
    Invalidate ();
#endif
    return TRUE;
}


#if defined(__X_YACL__)
// Define hard-wired constants describing the scroll bar's looks.

static const short
    GutterShadow        = 1,  // The gutter shadow is this many pixels wide
    SliderShadow        = 1,
    SliderSideGap       = 1,  // Gap between slider edge and gutter
    EndButtonSize       = 15; // The height  (or width) of the triangular
                              // buttons at the ends of the scroll bar




void UI_ScrollBar::_AdjustModel ()
{
    CL_Interval& model = *(CL_Interval*) _model;
    long len = minl (model.Length(), _range.Length());
    long low = maxl (_range.Low(), minl (_range.High() - len + 1,
                                         model.Low ()));
    CL_Interval newModel = CL_Interval (low, low + len - 1);
    _SetModelValue (newModel);
}

void UI_ScrollBar::_DrawSlider (UI_DrawingSurface& sfc, const UI_Point& p =
                                UI_Point (0, 0))
{
    UI_ShadowRectangle::Style st = _mouseIsDown
        ? UI_ShadowRectangle::Recessed : UI_ShadowRectangle::Raised;
    UI_ShadowRectangle r (_sliderRect, st, 1, FALSE);
    r.Origin (p);
    r.DrawOn (sfc, UI_Point (0,0));
}

void UI_ScrollBar::_Draw (UI_DrawingSurface& sfc)
{
    if (!_visible)
        return;
    sfc.ColorDisplay (_bgColor);
    long w = _shape.Width(), h = _shape.Height();
    UI_ShadowRectangle gutter (0, 0, w, h,
                               UI_ShadowRectangle::Recessed, 1, FALSE,
                               _gutterColor);
    gutter.DrawOn (sfc);
    if (_vertical) {
        UI_ShadowedTriangle north
            (UI_Rectangle (3, GutterShadow+1, w-6, EndButtonSize),
             UI_ShadowedTriangle::North,
             UIRelief_Raised);
        north.DrawOn (sfc);
        UI_ShadowedTriangle south
            (UI_Rectangle (3, h - EndButtonSize - GutterShadow - 1,
                           w-6, EndButtonSize),
             UI_ShadowedTriangle::South, UIRelief_Raised);
        south.DrawOn (sfc);
        _DrawSlider (sfc, _sliderRect.Origin());
    }
    else {
        UI_ShadowedTriangle west
            (UI_Rectangle (GutterShadow+1, 3, EndButtonSize, h-6),
             UI_ShadowedTriangle::West, UIRelief_Raised);
        west.DrawOn (sfc);
        UI_ShadowedTriangle east
            (UI_Rectangle (w - EndButtonSize - GutterShadow - 1, 3,
                           EndButtonSize, h-6),
             UI_ShadowedTriangle::East, UIRelief_Raised);
        east.DrawOn (sfc);
        _DrawSlider (sfc, _sliderRect.Origin());
    }
}


void UI_ScrollBar::_AddToModel (short amount)
{
    CL_Interval model = *(CL_Interval*) _model;
    long len = model.Length();
    long newLo = maxl (0, minl (model.Low() + amount,
                                _range.High() - len + 1));
    CL_Interval newModel (newLo, newLo + len - 1);
    _SetModelValue (newModel);
}

void UI_ScrollBar::_RedrawSlider ()
{
    // Warning: Call this method only after setting up the display surface
    // and memDS.
    
    // Erase the slider:
    UI_DisplaySurface* sfc = DisplaySurface();
    long w = _shape.Width(), h = _shape.Height();
    UI_Rectangle innerGutter;
    if (_vertical)
        innerGutter = UI_Rectangle
            (2, EndButtonSize+2, w-4, h-4-2*EndButtonSize);
    else
        innerGutter = UI_Rectangle
            (EndButtonSize+2, 2, w-4-2*EndButtonSize, h-4);
    sfc->ColorRectangle (innerGutter, _gutterColor);

    // Move and redraw it:
    _ConfigureSlider();
    _memDS->CopyTo (*sfc, _sliderRect.Origin());
}

bool UI_ScrollBar::HandleEvent (UI_Event* e)
{
    if (!e)
        return FALSE;
    if (!_enabled)
        return FALSE;
    UI_EventType type = e->Type();
    if (type >= Event_Scroll && type <= Event_ScrollToPosition) {
        // It's a scroll event that was enqueued by the handler code below
        if (type != Event_Scroll || _smoothScroll)
            _clientSet.NotifyAll (*e);
        return FALSE;
    }
    UI_EventType scrollEvtType = Event_None;
    switch (type) {
    case Event_LButtonPress: {
        _memDS = new UI_DwgSurfaceInMemory
            (_sliderRect.Width(), _sliderRect.Height());
        _memDS->ColorDisplay (_gutterColor);
        UI_DisplaySurface& sfc = CreateDisplaySurface();

        // Check if the click hit the slider
        UI_Point clickPt = e->Position();
        long clickX = clickPt.XCoord(), clickY = clickPt.YCoord();
        long compareWith = _vertical ? clickY : clickX;
        short sPos = _sliderPos + EndButtonSize;
        if (compareWith < sPos || compareWith >= sPos + _sliderSize) {
            // It didn't hit the slider. It might be a page-up or page-down,
            // but we'll take care of those in the button-release event.
            _DrawSlider (*_memDS, UI_Point (0, 0));
            break;
        }
        _mouseIsDown = TRUE;

        // Create a memory DS for the slider
        _DrawSlider (*_memDS, UI_Point (0, 0));
        _memDS->CopyTo (sfc, _sliderRect.Origin());
        _offset = _vertical ? e->Position().YCoord() - _sliderRect.Top()
            :  e->Position().XCoord() - _sliderRect.Left();
        _Controller->GiveFocusTo (this);

        // Now don't dispatch this event unless there are no more Scroll
        // events in the Controller's queue.
        if (!_Controller->EventsPending (Event_MouseMove, this))
            scrollEvtType = Event_Scroll;
        break;
    }

    case Event_MouseMove:
        if (_mouseIsDown) {
            short newPos;
            if (_vertical) {
                newPos = e->Position().YCoord() - _offset;
                newPos = minl
                    (_shape.Height() - _sliderRect.Height() - EndButtonSize -
                     GutterShadow,
                     maxl (EndButtonSize + GutterShadow, newPos));
            }
            else {
                newPos = e->Position().XCoord() - _offset;
                newPos = minl
                    (_shape.Width() - _sliderRect.Width() - EndButtonSize -
                     GutterShadow,
                     maxl (EndButtonSize + GutterShadow, newPos));
            }
            newPos -= EndButtonSize + GutterShadow;
            CL_Interval& model = *(CL_Interval*) _model;
            long lo = (long) (newPos * _scaleFactorNumerator) /
                _scaleFactorDenominator;
            if (abs (lo - model.Low()) >= 1){
                // Don't dipatch small mouse moves
                Display* d = _Controller->AppDisplay();
                XEvent xevt;
                if (!XCheckTypedWindowEvent (d, _window, MotionNotify,
                                            &xevt)) {
                    CL_Interval newModel (lo, lo - 1 + model.Length());
                    _SetModelValue (newModel);
                    _RedrawSlider ();
                    scrollEvtType = Event_Scroll;
                }
            }
        }
        break;

    case Event_LButtonRelease:
        if (!_mouseIsDown) {
            // The click must not have hit the slider, so it must be a
            // page-down or page-up event.
            // Check what kind of event this has to be
            UI_Point clickPt = e->Position();
            long clickX = clickPt.XCoord(), clickY = clickPt.YCoord();
            long pos, size, sliderLow, sliderHi;
            if (_vertical) {
                pos  =  clickY;
                size = _shape.Height();
                sliderLow = _sliderRect.Top();
                sliderHi  = _sliderRect.Bottom();
            }
            else {
                pos  =  clickX;
                size = _shape.Width();
                sliderLow = _sliderRect.Left();
                sliderHi  = _sliderRect.Right();
            }
            if (pos >= 0 && pos < EndButtonSize) { // Hit the top or left
                                                   // triangle
                scrollEvtType = Event_ScrollBackwardLine;
                _AddToModel (-_lineAmount);
            }
            else if (pos > EndButtonSize && pos < sliderLow) {
                scrollEvtType = Event_ScrollBackwardPage;
                _AddToModel (-_pageAmount);
            }
            else if (pos > sliderHi && pos < size - EndButtonSize) {
                scrollEvtType = Event_ScrollForwardPage;
                _AddToModel (_pageAmount);
            }
            else {
                scrollEvtType = Event_ScrollForwardLine;
                _AddToModel (_lineAmount);
            }
            _RedrawSlider();
            break;
        }
        else {

            // None of the above, so this must be the end of a drag.
            _mouseIsDown = FALSE;
            _Controller->ReleaseMouse();
            _Draw (*DisplaySurface());
            scrollEvtType = Event_FinishScroll;
        }
        DestroyDisplaySurface();
        delete _memDS;
        _memDS = NULL;
        break;
        
    case Event_Paint: {
        UI_DisplaySurface& sfc = CreateDisplaySurface();
        _Draw (sfc);
        DestroyDisplaySurface();
        break;
    }

    default:
        break;
    }
    if (scrollEvtType != Event_None) {
//         UI_Event se (scrollEvtType, this, this);
//         _Controller->DispatchEvent (&se);
        UI_Event* se = new UI_Event (scrollEvtType, this, this);
        _Controller->AddEvent (se);
    }
    return ProcessEvent (e);
}

bool UI_ScrollBar::MakeVisualElement ()
{
    _ConfigureSlider ();
    _visualElement = new YVE_Window (this);
    if (_visualElement)
        _window = _visualElement->Handle();
    return _visualElement && _visualElement->Handle() ? TRUE : FALSE;
}

void UI_ScrollBar::_ConfigureSlider ()
{
    if (_range.Length() <= 0) 
        _scaleFactorNumerator = _scaleFactorDenominator = 1;

    CL_Interval& model = *(CL_Interval*) _model;
    long x, y;
    if (_vertical) {
        _scaleFactorNumerator   =  _range.Length();
        _scaleFactorDenominator =  maxl
            (2, _shape.Height() - 2 * (GutterShadow + EndButtonSize));
        _sliderPos  =  (short) (model.Low() * _scaleFactorDenominator) /
            _scaleFactorNumerator;
        _sliderSize =  maxl
            (MIN_SLIDER_SIZE,
             model.Length() * _scaleFactorDenominator / _scaleFactorNumerator);
        x = GutterShadow + 1;
        y = _sliderPos + GutterShadow + EndButtonSize;
        _sliderRect = UI_Rectangle (x, y, _shape.Width() - 2*GutterShadow - 2,
                                    _sliderSize);
    }
    else {
        _scaleFactorNumerator = _range.Length();
        _scaleFactorDenominator =
            maxl (2, _shape.Width() - 2 - 2 * EndButtonSize);
        double d = ((double) _scaleFactorNumerator) / _scaleFactorDenominator;
        _sliderPos  =  (model.Low() * _scaleFactorDenominator
                        / _scaleFactorNumerator);
        _sliderSize =  maxl (8, (short)
                             (model.Length() * _scaleFactorDenominator
                              / _scaleFactorNumerator));
        x = _sliderPos + GutterShadow + EndButtonSize;
        y = GutterShadow + 1;
        _sliderRect = UI_Rectangle (x, y, _sliderSize,
                                    _shape.Height() - 2 * GutterShadow - 2);
    }
}

#endif





















/*---------------------VScrollBar------------------------------------*/

UI_VScrollBar::UI_VScrollBar
    (UI_VisualObject* p, const UI_Rectangle& shape,
     UI_ViewID id, long bstyle)
: UI_ScrollBar (p, shape, id, bstyle, TRUE)
{
#if defined (__MS_WINDOWS__) || defined(__MS_WIN32__)
    if (_style == 0xffffffff)
        _style = WS_VISIBLE | SBS_VERT | WS_CHILD;
#elif defined (__OS2__)
    _style = SBS_VERT | WS_VISIBLE;
#endif
}




/*-------------------------HScrollBar----------------------------*/

UI_HScrollBar::UI_HScrollBar
    (UI_VisualObject* p, const UI_Rectangle& shape,
     UI_ViewID id, long bstyle)
: UI_ScrollBar (p, shape, id, bstyle, FALSE)
{
#if defined(__MS_WINDOWS__) || defined(__MS_WIN32__)
    if (_style == 0xffffffff)
        _style = WS_VISIBLE | SBS_HORZ | WS_CHILD;
#elif defined (__OS2__)
    _style = SBS_HORZ | WS_VISIBLE;
#endif
}
