




/*
 *
 *          Copyright (C) 1995, M. A. Sridhar
 *  
 *
 *     This software is Copyright M. A. Sridhar, 1995. You are free
 *     to copy, modify or distribute this software  as you see fit,
 *     and to use  it  for  any  purpose, provided   this copyright
 *     notice and the following   disclaimer are included  with all
 *     copies.
 *
 *                        DISCLAIMER
 *
 *     The author makes no warranties, either expressed or implied,
 *     with respect  to  this  software, its  quality, performance,
 *     merchantability, or fitness for any particular purpose. This
 *     software is distributed  AS IS.  The  user of this  software
 *     assumes all risks  as to its quality  and performance. In no
 *     event shall the author be liable for any direct, indirect or
 *     consequential damages, even if the  author has been  advised
 *     as to the possibility of such damages.
 *
 */



#if defined(__GNUC__)
#pragma implementation
#endif



#include "ui/rectangl.h"
#include "ui/dwgsurf.h"


#if defined(__MS_WINDOWS__) || defined(__MS_WIN32__)
#include <windows.h>
#endif

#if defined(__GNUC__)
#include "base/basicops.cxx"
template class CL_Basics<UI_Rectangle>;
#endif



UI_Rectangle::UI_Rectangle(const UI_Point& origin, long width, long height)
{
    _origin = origin;
    if (width < 0) {
        width = -width;
        _origin = _origin - UI_Point (width-1, 0); // Shift origin to left end
    }
    if (height < 0) {
        height = -height;
        _origin = _origin - UI_Point (0, height-1); 
    }
    _height = height;
    _width  = width;
}

UI_Rectangle::UI_Rectangle (long x, long y, long width, long height)
{
    _origin = UI_Point (x, y);
    if (width < 0) {
        width = -width;
        _origin = _origin - UI_Point (width-1, 0); // Shift origin to left end
    }
    if (height < 0) {
        height = -height;
        _origin = _origin - UI_Point (0, height-1); 
    }
    _height = height;
    _width  = width;
}


UI_Rectangle::UI_Rectangle()
{
   _height = 0;
   _width  = 0;
}


UI_Rectangle::UI_Rectangle (const UI_RectangleStruct& r)
: _origin (r.x, r.y)
{
    _width = r.w;
    _height = r.h;
}


UI_Rectangle::UI_Rectangle(const UI_Rectangle& r)
{
    _origin = r._origin;
    _height = r. _height;
    _width = r._width;
}


UI_Rectangle::~UI_Rectangle()
{
}

    

UI_Rectangle UI_Rectangle::operator+ (const UI_Vector& p) const
{
    UI_Rectangle q (*this);
    q._origin += p;
    return q;
}



void UI_Rectangle::operator= (const CL_Object& r)
{
    if (!IsA (r))
        return;
    *this = (const UI_Rectangle&) r;
}


UI_Rectangle& UI_Rectangle::operator= (const UI_Rectangle& r)
{
    if (this == &r  || *this == r || !PrepareToChange())
        return *this;
    _origin = r._origin; 
    _height = r._height;
    _width  = r._width;
    Notify();
    return *this;
}


bool UI_Rectangle::operator== (const CL_Object& r) const
{
    if (!IsA (r))
        return FALSE;
    return (*this == (const UI_Rectangle&) r);
}


bool UI_Rectangle::operator== (const UI_Rectangle& r) const
{
    if (_origin != r._origin || _width != r._width || _height != r._height)
        return FALSE;
    return TRUE;
}


CL_String UI_Rectangle::AsString() const
{
    CL_String s;
    s.AssignWithFormat ("(%ld, %ld, %ld, %ld)", _origin.XCoord(),
                        _origin.YCoord(), _width, _height);
    return s;
}




short UI_Rectangle::Compare (const UI_Rectangle& p) const
{
    short r1 = _origin.Compare (p._origin);
    if (r1 != 0)
        return r1;
    if (_width < p._width)
        return -1;
    if (_height < p._height)
        return -1;
    return 0;
}


short UI_Rectangle::Compare (const CL_Object& o) const
{
    if (o.IsA (*this))
        return Compare ((const UI_Rectangle&) o);
    return this < ((const UI_Rectangle*) &o) ? -1 : 1; // Should really
                                                       // throw an exception
                                                       // here?
}


bool UI_Rectangle::Origin  (const UI_Point& o)
{
    if (!PrepareToChange())
        return FALSE;
    _origin = o;
    Notify();
    return TRUE;
}



bool UI_Rectangle::AddToHeight (long incr)
{
    long new_val = _height + incr;
    if (new_val < 0 || !PrepareToChange())
        return FALSE;
    _height = new_val;
    Notify();
    return TRUE;
}



bool UI_Rectangle::AddToWidth (long incr)
{
    long new_val = _width + incr;
    if (new_val < 0 || !PrepareToChange())
        return FALSE;
    _width = new_val;
    Notify();
    return TRUE;
}



bool UI_Rectangle::DrawOn (UI_DrawingSurface& sfc, const UI_Point& p) const
{
    UI_Rectangle r (_origin + p, _width, _height);
    sfc.DrawRectangle (r, UID_Outline);
    return TRUE;
}


bool UI_Rectangle::Fill (UI_DrawingSurface& s) const
{
    s.DrawRectangle (*this, UID_Fill);
    return TRUE;
}



bool UI_Rectangle::Includes(const UI_Point& p) const
{
    long x, y, x1, x2, y1, y2;
    x = p.XCoord();
    y = p.YCoord();

    x1 = _origin.XCoord();
    y1 = _origin.YCoord();
 
    x2 = x1 + _width;
    y2 = y1 + _height;

    return (x >= x1 && x <= x2 && y >= y1 && y <= y2);
}

bool UI_Rectangle::OnBoundary (const UI_Point& p) const
{
    long px = p.XCoord(), py = p.YCoord();
    long left    = _origin.XCoord();
    long right   = left + _width;
    long top     = _origin.YCoord();
    long bottom  = top + _height;
    return ((px == left || px == right)  && py >= top  && py <= bottom)
               // on left or right side
        || ((py == top || py == bottom)  && px >= left && px <= right);
               // on top or bottom side
}


UI_HitTest UI_Rectangle::HitTest (const UI_Point& p) const
{
    return OnBoundary (p) ? UIHit_Boundary
        : (Includes (p) ? UIHit_Inside : UIHit_Outside);
}


bool UI_Rectangle::SetHeight (long newHeight)
{
    if (!PrepareToChange() || newHeight < 0)
        return FALSE;
    _height = newHeight;
    Notify();
    return TRUE;
}


bool UI_Rectangle::SetWidth (long newWidth)
{
    if (!PrepareToChange() || newWidth < 0)
        return FALSE;
    _width = newWidth;
    Notify();
    return TRUE;
}


static bool Intersects (const UI_Point& p1, long l1,
                        const UI_Point& p2, long l2)
{
    // Determine if the vertical line segment at p1 of length l1
    // intersects the horizontal line segment at p2 of length l2
    long x1 = p1.XCoord(), y1 = p1.YCoord();
    long x2 = p2.XCoord(), y2 = p2.YCoord();
    return y1 <= y2 && y1 + l1 > y2 && x1 >= x2 && x1 < x2 + l2;
}

bool UI_Rectangle::IntersectsBoundary (const UI_Rectangle& rect) const
{
//     UI_Point tl = TopLeft(),    tr = TopRight();
//     UI_Point bl = BottomLeft(), br = BottomRight();
//     bool b1 = rect.Includes (tl),  b2 = rect.Includes (bl);
//     bool b2 = rect.Includes (tr),  b4 = rect.Includes (br);
//     if ((b1 || b2 || b3 || b4) && ! (b1 && b2 && b3 && b4))
//         return TRUE;
//     tl = rect.TopLeft(); tr = rect.TopRight();
//     bl = rect.BottomLeft(), br = rect.BottomRight();
//     b1 = Includes (tl),  b2 = Includes (bl);
//     b2 = Includes (tr),  b4 = Includes (br);
//     if ((b1 || b2 || b3 || b4) && ! (b1 && b2 && b3 && b4))
//         return TRUE;
    long h = rect._height, w = rect._width;
    UI_Point mtl = TopLeft();
    UI_Point mtr = UI_Point (mtl.XCoord() + _width - 1, mtl.YCoord());
    UI_Point mbl = UI_Point (mtl.XCoord(), mtl.YCoord() + _height - 1);
    UI_Point mbr = UI_Point (mtr.XCoord(), mtr.YCoord() + _height - 1);

    UI_Point rtl = TopLeft();
    UI_Point rtr = UI_Point (mtl.XCoord() + w - 1, mtl.YCoord());
    UI_Point rbl = UI_Point (mtl.XCoord(), mtl.YCoord() + h - 1);
    UI_Point rbr = UI_Point (mtr.XCoord(), mtr.YCoord() + h - 1);
    if (Intersects (rtl, h, mtl, _width) ||
        Intersects (rtr, h, mtl, _width) ||
        Intersects (rtl, h, mbl, _width) ||
        Intersects (rtr, h, mbl, _width) ||
        Intersects (mtl, _height, rtl, w) ||
        Intersects (mtr, _height, rtl, w) ||
        Intersects (mtl, _height, rbl, w) ||
        Intersects (mtr, _height, rbl, w)
        )
        return TRUE;
    return FALSE;
}


UI_Point UI_Rectangle::Center () const
{
    return UI_Point (_origin.XCoord() + maxl (1, _width)/2 - 1,
                     _origin.YCoord() + maxl (1, _height)/2 - 1);
}


UI_PointSequence UI_Rectangle::EndPoints () const
{
    UI_PointSequence sq(4);
    long x = _origin.XCoord(), y = _origin.YCoord();
    sq(0) = _origin;
    sq(1) = UI_Point (x + _width - 1, y);
    sq(2) = UI_Point (x + _width - 1, y + _height - 1);
    sq(3) = UI_Point (x, y + _height - 1);
    return sq;
}

bool UI_Rectangle::ReshapeTo (const UI_Point& p1, const UI_Point& p2)
{
    if (!PrepareToChange())
        return FALSE;
    UI_Rectangle r (p1, p2.XCoord() - p1.XCoord() + 1, p2.YCoord() -
                    p1.YCoord() + 1);
    _origin = r._origin;
    _width  = r._width;
    _height = r._height;
    return TRUE;
}

bool UI_Rectangle::MoveTo (const UI_Point& p)
{
    if (!PrepareToChange())
        return FALSE;
    UI_Point center = Center();
    _origin += UI_Vector (p.XCoord() - center.XCoord(),
                          p.YCoord() - center.YCoord());
    Notify();
    return TRUE;
}

bool UI_Rectangle::IsContainedIn (const UI_Rectangle& r) const
{
    long mx = _origin.XCoord(), my = _origin.YCoord();
    long rx = r._origin.XCoord(), ry = r._origin.YCoord();
    long mw = _width, mh = _height;
    long rw = r._width, rh = r._height;
    // Check if top left corner is inside r
    if (mx < rx || mx > rx + rw - 1)
        return FALSE;
    if (my < ry || my > ry + rh - 1)
        return FALSE;
    // Check if bottom right corner is inside r
    long right = mx + mw - 1;
    if (right < rx || right > rx + rw - 1)
        return FALSE;
    long bot = my + mh - 1;
    if (bot < ry || bot > ry + rh - 1)
        return FALSE;
    // Ok, return TRUE
    return TRUE;
}


CL_Sequence<UI_Point> UI_Rectangle::RowMajor (long m, long n) const
{
    if (m <= 0 || n <= 0) {
        CL_Error::Warning ("UI_Rectangle::RowMajor: invalid parameters"
                           " %ld %ld", m, n);
        CL_Sequence<UI_Point> junk;
        return junk;
    }
    long ncols = _width/m;
    long nrows = _height/n;
    if (nrows <= 0 || ncols <= 0) {
        CL_Sequence<UI_Point> junk;
        return junk;
    }
    CL_Sequence<UI_Point> seq (nrows * ncols);
    long x = _origin.XCoord();
    long y = _origin.YCoord();
    for (long i = 0; i < nrows; i++) {
        long xpt = x;
        long ypt = y + i * n;
        for (long j = 0; j < ncols; j++) {
            seq(i * ncols + j) = UI_Point (xpt, ypt);
            xpt += m;
        }
    }
    return seq;
}



#if defined(__MS_WINDOWS__) || defined(__MS_WIN32__)
tagRECT UI_Rectangle::AsMSRect () const
{
    RECT rect;
    rect.left   = Left ();  
    rect.top    = Top ();
    rect.right  = Right();
    rect.bottom = Bottom();
    return rect;
}

#endif

