

#ifndef _graphic_h_ /* Fri Oct 28 17:18:04 1994 */
#define _graphic_h_



/*
 *
 *          Copyright (C) 1995, M. A. Sridhar
 *  
 *
 *     This software is Copyright M. A. Sridhar, 1995. You are free
 *     to copy, modify or distribute this software  as you see fit,
 *     and to use  it  for  any  purpose, provided   this copyright
 *     notice and the following   disclaimer are included  with all
 *     copies.
 *
 *                        DISCLAIMER
 *
 *     The author makes no warranties, either expressed or implied,
 *     with respect  to  this  software, its  quality, performance,
 *     merchantability, or fitness for any particular purpose. This
 *     software is distributed  AS IS.  The  user of this  software
 *     assumes all risks  as to its quality  and performance. In no
 *     event shall the author be liable for any direct, indirect or
 *     consequential damages, even if the  author has been  advised
 *     as to the possibility of such damages.
 *
 */


// GraphicObject is an abstract class representing a graphic object that can
// be drawn on a DrawingSurface. The GraphicObject exists in an abstract
// co-ordinate space unrelated to any particular device such as screen or
// printer. This class  contains three pure virtual methods:
// \par\begin{itemize}
// \item  {\tt DrawOn}, the method to draw this GraphicObject on a given
// DrawingSurface at a given point;
// \item  {\tt BoundingRectangle}, which must return the bounding rectangle
// of this GraphicObject;
// \item {\tt operator+=} which must translate this GraphicObject by the
// given vector.
// \end{itemize}
// There are a few other virtual methods with default implementations.


#if defined(__GNUC__)
#pragma interface
#endif

#include "base/object.h"
#include "ui/vector.h"

class YACL_UI UI_DrawingSurface;
class YACL_UI UI_Rectangle;
class YACL_UI UI_Point;


//[ Global
enum UI_HitTest {
    UIHit_Boundary, UIHit_Inside, UIHit_Outside
};
// This enumeration is used by the {\tt HitTest} method.
//] Global

class YACL_UI UI_GraphicObject: public CL_Object {

public:

    ~UI_GraphicObject() {};
    // The destructor does nothing. It is provided because some compilers
    // complain if it's absent, since Object's destructor is pure virtual.
    
    virtual UI_Rectangle BoundingRectangle () const = 0;
    // Return the bounding rectangle for this graphic. [Pure virtual]

    virtual UI_Point Center () const;
    // Return the center of this GraphicObject. The default implementation
    // returns the center of the Rectangle returned by the {\tt
    // BoundingRectangle} method.
    
    virtual bool DrawOn (UI_DrawingSurface& sfc,
                         const UI_Point& p) const = 0;
    // [Pure virtual] Draw the graphic on the given surface,
    // translated with respect to {\tt p} (i.e., treating {\tt p} as the
    // origin).

    virtual bool ReshapeTo (const UI_Point& p1, const UI_Point& p2);
    // Grow or shrink the graphic so that the two given points become the
    // diagonally-opposing points of its bounding box. Return TRUE if
    // successful. The default implementation returns FALSE unconditionally.

    virtual bool Fill   (UI_DrawingSurface& s) const;
    // Fill the interior of the graphic on the given surface with the current
    // foreground color of the given surface. This operation makes sense
    // only for closed figures. The default implementation returns FALSE.

    virtual UI_HitTest HitTest (const UI_Point& p) const;
    // Tell whether the given point is inside, outside or on the boundary of
    // this GraphicObject. The default implementation uses {\tt HitTest} on the
    // return value of the {\tt BoundingRectangle} method.

    virtual bool IntersectsBoundary (const UI_Rectangle& r) const;
    // Tell whether the given rectangle {\tt r} intersects the boundary of this
    // GraphicObject. The default implementation returns TRUE if {\tt r}
    // intersects the return value of the {\tt BoundingRectangle} method.

    virtual bool IsContainedIn (const UI_Rectangle& r) const;
    // Tell whether the given rectangle entirely contains this
    // GraphicObject. The default implementation returns whether this
    // object's bounding rectangle is contained in {\tt r}.

    virtual void operator+= (const UI_Vector& p) = 0;
    // Translate this graphic object by the given vector. [Pure virtual.]

    void operator-= (const UI_Vector& p)
        {*this += UI_Vector (0, 0) - p;};
    // Translate this graphic backwards by the given vector.
    
    virtual bool MoveTo (const UI_Point& p);
    // Move this object to the given point, and return TRUE if successful.
    // The semantics of the movement is left to subclasses. The default
    // implementation returns FALSE unconditionally.
    
    
};


inline bool UI_GraphicObject::MoveTo (const UI_Point&)
{
    return FALSE;
}

#endif /* _graphic_h_ */
