#ifndef _font_h_
#define _font_h_





/*
 *
 *          Copyright (C) 1995, M. A. Sridhar
 *  
 *
 *     This software is Copyright M. A. Sridhar, 1995. You are free
 *     to copy, modify or distribute this software  as you see fit,
 *     and to use  it  for  any  purpose, provided   this copyright
 *     notice and the following   disclaimer are included  with all
 *     copies.
 *
 *                        DISCLAIMER
 *
 *     The author makes no warranties, either expressed or implied,
 *     with respect  to  this  software, its  quality, performance,
 *     merchantability, or fitness for any particular purpose. This
 *     software is distributed  AS IS.  The  user of this  software
 *     assumes all risks  as to its quality  and performance. In no
 *     event shall the author be liable for any direct, indirect or
 *     consequential damages, even if the  author has been  advised
 *     as to the possibility of such damages.
 *
 */



/* Authors:   M. A. Sridhar
             N. Bhowmik

   Modified:  R. Sampath, Dec 1994
*/


// The Font class represents a single font of the underlying windowing
// system that has been selected for use by a DrawingSurface or a
// VisualObject.  The Font instance contains the point size, type style and
// other attributes of the font, as well as the font handle obtained when
// the Font is bound to its DrawingSurface or VisualObject.
//
// An application program does not create an instance of Font. Applications
// merely apply the methods of this class to the return value of the {\tt
// Font} method of the VisualObject
// or DrawingSurface to set the font attributes. Thus every YACL Font
// instance is owned by a corresponding VisualObject or DrawingSurface
// instance.
//
// Applying a Font method can have unexpected results when the desired font
// is not available. For example, if you try to set the font size to a
// negative number or use an unknown font name, the consequences depend on
// the system. Under Microsoft Windows, the system runs a font-mapping
// algorithm; under X windows, the font remains unchanged.


#if defined(__GNUC__)
#pragma interface
#endif



#include "base/string.h"
#include "base/sequence.h"

#include "ui/dsplrsrc.h"
#include "ui/fontdesc.h"


#if defined(__MS_WINDOWS__) || defined(__MS_WIN32__)
    typedef ulong UI_NativeFontRep;

    // Definitions of three common font names:
#   define UIFont_Times      "Times New Roman"
#   define UIFont_Courier    "Courier"
#   define UIFont_Helvetica  "Arial"

#elif defined (__X_MOTIF__)  || defined(__X_YACL__)
    typedef CL_String UI_NativeFontRep;

    // Definitions of three common font names:
#   define UIFont_Times      "times"
#   define UIFont_Courier    "courier"
#   define UIFont_Helvetica  "helvetica"

#elif defined (__OS2__)
    typedef ulong UI_NativeFontRep;

    // Definitions of three common font names:
#   define UIFont_Times      "Times New Roman"
#   define UIFont_Courier    "Courier"
#   define UIFont_Helvetica  "Helvetica"

#endif

class YACL_UI UI_VisualObject;

class YACL_UI UI_Font: public UI_DisplayResource {

public:

    long Height () const;
    // Return the height of each character, in pixels.

    long Width  () const;
    // Return the average width of a character, in pixels.

    long Width (const char* string) const;
    // Return the width, in pixels, of the box containing {\tt string}, if
    // it were to be drawn on the DrawingSurface or VisualObject that owns
    // this Font.

    UI_FontDesc Descriptor () const
        {return UI_FontDesc (_typeFace, _ptSize, _style);};
    // Return this Font's contents as a descriptor.
    
    short PointSize() const;
    // Return the point size of this font, i.e., the character height in
    // points (each point is 1/72 of an inch). Point size is largely
    // determined by the platform window system.

    CL_String TypeFace() const;
    // Return the name of the type face.

    bool IsFixed () const;
    // Return whether this is a fixed-width font (i.e., all characters have
    // the same width). For example, Courier is a fixed-width font but Times
    // Roman is not.

    ulong Style () const {return _style; };
    // Return an unsigned long value containing the bitwise {\sc or} of the
    // style parameters for this Font. (See {\tt fontdesc.h} for
    // the definitions of style parameters.)
    
    
    bool Italic() const;
    // Return whether this is an italic font.

    bool Underline() const;
    // Return whether this font has the underline attribute.

    bool BoldFace() const;
    // Return whether this is a boldface font.

    bool StrikeOut() const;
    // Return whether this font has the strike-out attribute.

    short Ascent ()  const;
    // Return the ascent of this font, in pixels. The ascent is the number
    // of pixels from the baseline to the top of the tallest character in
    // the font.

    short Descent () const;
    // Return the descent of this font, in pixels. The value is the number
    // of pixels from the baseline to the bottom of the  character with most
    // descent in the font.

    bool IsScalable () const;
    // Return whether this font is scalable to arbitrary point sizes. For
    // example, under Windows, this method returns TRUE if the font is a
    // TrueType font.
    
    void PointSize (short ptSize);
    // Set the point size of this font to the given size.

    void TypeFace  (const char* new_type_face);
    // Set the type face name of this font.

    void Italic    (bool );
    // Set the italic attribute for this font.

    void Underline (bool );
    // Set the underline attribute for this font.

    void BoldFace  (bool );
    // Set the boldface attribute for this font.

    void StrikeOut (bool );
    // Set the strike-out attribute for this font.

    void Style (ulong newStyle);
    // Set all the style parameters according to {\tt newStyle}, which must
    // be a bitwise or of the font style constants described in {\tt
    // fontdesc.h}.
    
    void operator= (const UI_FontDesc& desc);
    // Modify this Font according to the contents of the given descriptor.

    void operator= (const UI_Font& f)
        { operator= (f.Descriptor());};
    // Modify this Font according to the contents of the given Font.
    
    void operator= (const CL_Object& o);
    // The parameter is required to be of type {\tt UI_FontDesc}. This
    // method is provided so that the virtual assignment operator does not
    // become hidden.

#if defined(__X_MOTIF__) || defined(__X_YACL__)
    void operator= (const char* native_name);
    // [X-windows-specific] Use the font whose fully-qualified name is
    // {\tt native_name}.

#endif
    
    static const CL_Sequence<UI_FontInfo>& AvailableFonts ();
    // Return a reference to a sequence containing all available fonts.
    
    const char* ClassName() const { return "UI_Font";};
    // Override the method inherited from {\tt CL_Object}.

protected:

    void    _Setup ();
    // This method is called when any of the Font's attributes changes. It
    // sets up the new font handle.

    void _Init (UI_VisualObject* v = NULL);

    
#if defined(__MS_WINDOWS__) || defined(__MS_WIN32__) || defined(__X_YACL__) || defined(__X_MOTIF__)
    void    _Setup (UI_NativeFontRep nativename);
    // [YACL internal use only]
    
#endif

    short             _ptSize;
    CL_String         _typeFace;
    UI_NativeFontRep  _nativeName;
    ulong             _style;
    bool              _stockFont;
    long              _height;
    long              _width;

    void* _entry;

    static CL_Sequence<UI_FontInfo> _allFonts;

#if defined(__MS_WINDOWS__) || defined(__MS_WIN32__)
private:
    static long _ppi; // Pixels per inch: used during font enumeration
#endif
    
public:
#if defined(__MS_WINDOWS__) || defined(__MS_WIN32__)
    static void _AddFont (void*, void*, int);
    // [MS-Windows-specific; YACL internal use only]

#elif defined(__X_MOTIF__) || defined(__X_YACL__)
    long TextWidth (const char*) const;
    // X-Windows-specific. Return the width of the given character string
    // in the given font.
    
#endif

    void*  NativeFontStruct () const;
    // [YACL internal use only] Return the native font structure of the
    // font; specifically, this returns a pointer to the {\tt TEXTMETRIC}
    // structure under MS Windows, the {\tt FONTMETRICS} structure under
    // OS/2 and the {\tt XFontStruct} structure under X windows.
    
    UI_Font (UI_VisualObject* client = NULL);
    // [Internal use only] Create the platform's default font for the given
    // client, if any.

    UI_Font (const char* typeface, short ptSize, ulong style = 0);
    // [Internal use only] Create a font with the given attributes. The last
    // parameter is a disjunction of {\tt UIFont_Style} elements.
    
    UI_Font (UI_DrawingSurface*);
    // [Internal use only] Create a font for the given surface.

    UI_Font (const UI_Font&);
    // [Internal use only] Copy constructor.

    ~UI_Font ();
    // [Internal use only] Destructor.

    void UseClient (UI_DrawingSurface* sfc);
    // [Internal use only]
    
};




#endif



