




/*
 *
 *          Copyright (C) 1995, M. A. Sridhar
 *  
 *
 *     This software is Copyright M. A. Sridhar, 1995. You are free
 *     to copy, modify or distribute this software  as you see fit,
 *     and to use  it  for  any  purpose, provided   this copyright
 *     notice and the following   disclaimer are included  with all
 *     copies.
 *
 *                        DISCLAIMER
 *
 *     The author makes no warranties, either expressed or implied,
 *     with respect  to  this  software, its  quality, performance,
 *     merchantability, or fitness for any particular purpose. This
 *     software is distributed  AS IS.  The  user of this  software
 *     assumes all risks  as to its quality  and performance. In no
 *     event shall the author be liable for any direct, indirect or
 *     consequential damages, even if the  author has been  advised
 *     as to the possibility of such damages.
 *
 */


#if defined(__GNUC__)
#pragma implementation
#endif


#include "ui/dwgsurf.h"
#include "ui/visualob.h"
#include "ui/bitmap.h"
#include "ui/arc.h"
#include "ui/chord.h"
#include "ui/ellipse.h"
#include "ui/piewedge.h"
#include "ui/applic.h"
#include "ui/font.h"

#if defined(__MS_WINDOWS__) || defined(__MS_WIN32__)
#    include <windows.h>
#    include <string.h>

// static HRGN getPolyRgnMS (UI_Point par[], short numpts);

#elif defined(__OS2__)
#    include <string.h>
#    include "ui/applic.h"
#    include "ui/cntroler.h"

#elif defined(__X_MOTIF__) || defined(__X_YACL__)
#    if defined(__GNUC__) && __GNUC_MINOR__ >= 7
#        include <string.h>  // Without this, the X includes barf
#    endif
#    include <X11/Xatom.h>
#    include <X11/Xlib.h>
#    include <X11/Intrinsic.h>
#    include <X11/StringDefs.h>
#    include <iostream.h> // DEBUG
#    include "base/strgseq.h"
#endif







UI_DrawingSurface::UI_DrawingSurface ()
{
    _pen        = NULL;
    _font       = NULL;
    _brush      = NULL;
    _horzPixels = 0;
    _vertPixels = 0;
}



UI_Rectangle UI_DrawingSurface::DrawingAreaInMM () const
{
#if defined(__MS_WINDOWS__) || defined(__MS_WIN32__)
    UI_Rectangle r = DrawingArea ();
    return UI_Rectangle (0, 0, r.Width()/ GetDeviceCaps (_handle, LOGPIXELSX),
                         r.Height()/GetDeviceCaps (_handle, LOGPIXELSY));
#elif defined(__OS2__)
    HDC hdc = GpiQueryDevice (_handle);
    LONG w, h; // w and h will be in pixels per meter
    DevQueryCaps (hdc, CAPS_HORIZONTAL_RESOLUTION, 1L, &w);
    DevQueryCaps (hdc, CAPS_VERTICAL_RESOLUTION,   1L, &h);
    UI_Rectangle r = DrawingArea();
    return UI_Rectangle (0, 0, r.Width() * 100 / w,  r.Height() * 100 / h);
#else
    NotImplemented ("DrawingAreaInMM");
    return UI_Rectangle (0, 0, 0, 0);
#endif
}



void UI_DrawingSurface::ColorDisplay (const UI_Color& cs)
{
    UI_Rectangle r = DrawingArea ();
    r.Origin (UI_Point (0, 0));
    ColorRectangle (r, cs);
}


void UI_DrawingSurface::ColorRectangle (const UI_Rectangle& r,
                                        const UI_Color& cs)
{
#if defined(__MS_WINDOWS__) || defined(__MS_WIN32__)
    HBRUSH hbr = CreateSolidBrush (cs.NativeForm ());
    RECT rect = r.AsMSRect();
    rect.right++; rect.bottom++; // Correct for Windows' omission of right
                                 // and bottom border
    FillRect (_handle, &rect, hbr);
//     HBRUSH old = SelectObject (_handle, hbr);
//     HRGN   hregion = getRectRgnMS (r);
//     PaintRgn(_handle, hregion);
//     SelectObject (_handle, old);
//     DeleteObject (hregion);
    DeleteObject (hbr);
#elif defined (__OS2__)
    UI_Rectangle area = DrawingArea ();
    RECTL rect;
    rect.xLeft    = r.Left();
    rect.yBottom  = area.Height() - r.Bottom () - 1;
    rect.xRight   = r.Right();
    rect.yTop     = area.Height() - r.Top () - 1;
    UI_NativeColorRep color = cs.NativeForm();
    //    GpiSavePS (_handle);
    if (SupportsColor())
        GpiSetColor (_handle, color);
    GpiMove (_handle, (PPOINTL) &rect);
    GpiBox  (_handle, DRO_FILL, ((PPOINTL) &rect) + 1, 0, 0);
    //    GpiRestorePS (_handle, -1);
#elif defined(__X_MOTIF__) || defined(__X_YACL__)
    Display *dpy = YACLApp()->AppDisplay();
    XSetForeground (dpy, _handle, cs.XPixel());
    XFillRectangle (dpy, _Drawable(), _handle,
                    r.Origin ().XCoord (), r.Origin ().YCoord (),
                    r.Width (), r.Height ());
#endif 
}



void UI_DrawingSurface::DrawEllipse (const UI_Rectangle& r,
                                     ushort opt)
{
    if (!opt)
        return;
#if defined(__MS_WINDOWS__) || defined(__MS_WIN32__)
    HBRUSH hbr, old;
    if (opt & UID_Fill) {
        old = SelectObject (_handle, _brush->Handle());
    }
    else {
        hbr   = GetStockObject (HOLLOW_BRUSH);
        old   = SelectObject (_handle, hbr);
    }
    if (! (opt & UID_Outline))
        SelectObject (_handle, GetStockObject (NULL_PEN));
    Ellipse (_handle, r.Left (), r.Top (), r.Right (), r.Bottom ());
    if (! (opt & UID_Outline))
        SelectObject (_handle, _pen->Handle());
    SelectObject (_handle, old);
#elif defined(__OS2__)
    long width  = r.Width();
    long height = r.Height ();
    POINTL ptLB, ptTR;
    ptLB.x = r.Left();
    ptLB.y = DrawingArea().Height() - r.Bottom() - 1;
    ptTR.x = r.Right();
    ptTR.y = ptLB.y + r.Height() - 1;
    GpiSavePS (_handle);
    if ((opt & UID_Fill) && (_brush->Pattern() != UIBrush_Hollow)) {
        if (SupportsColor())
            GpiSetColor (_handle, _brush->Color().NativeForm());
        GpiMove (_handle, &ptLB);
        GpiBox  (_handle, DRO_FILL, &ptTR, width, height);
    }
    if (opt & UID_Outline) {
        if (SupportsColor())
            GpiSetColor (_handle, _pen->Color().NativeForm());
        GpiMove (_handle, &ptLB);
        GpiBox  (_handle, DRO_OUTLINE, &ptTR, width, height);
    }
    GpiRestorePS (_handle, -1);
#elif defined(__X_MOTIF__) || defined(__X_YACL__)
    Display* dpy = YACLApp()->AppDisplay();
    if (opt & UID_Fill) {
        XSetForeground (dpy, _handle, _brush->Color().XPixel());
        XFillArc (dpy, _Drawable(), _handle,  r.Origin ().XCoord (),
                  r.Origin ().YCoord (), r.Width (), r.Height (), 0,
                  64*360);
    }
    if (opt & UID_Outline) {
        XSetForeground (dpy, _handle, _pen->Color().XPixel());
        XDrawArc (dpy, _Drawable(), _handle,
                  r.Origin().XCoord (), r.Origin().YCoord (), r.Width (),
                  r.Height (), 0, 64*360);
    }
#endif
}



void UI_DrawingSurface::DrawArc (const UI_Arc& arc)
{
    UI_Rectangle  rect  = arc.Ellipse().BoundingRectangle();
#if defined (__MS_WINDOWS__) || defined(__MS_WIN32__)
    UI_Point pa, pb;

    UI_PointPair pp = arc.EndPoints();
    pa = pp.p1;
    pb = pp.p2;

    Arc (Handle(),
         rect.Left(),  rect.Top(),
         rect.Right(), rect.Bottom(),
         pa.XCoord(),  pa.YCoord(),
         pb.XCoord(),  pb.YCoord());
#elif defined(__OS2__)
    long width  = rect.Width  ();
    long height = rect.Height ();
    ARCPARAMS arcParams = {width/2, height/2, 0, 0};
    GpiSetArcParams (_handle, &arcParams);
    POINTL center;
    UI_Point centerPt = rect.Center();
    center.x = centerPt.XCoord();
    center.y = DrawingArea().Height() - centerPt.YCoord() - 1;
    long start = arc.StartAngle(), sweep = arc.SubtendedAngle();
    GpiSetLineType (_handle, LINETYPE_INVISIBLE);
    GpiPartialArc (_handle, &center, MAKEFIXED(1,0),
                   MAKEFIXED (start / 64, start % 64),
                   MAKEFIXED (0, 0));
    if (_pen)
        GpiSetLineType (_handle, _pen->OS2Pattern());
    GpiPartialArc (_handle, &center, MAKEFIXED(1,0),
                   MAKEFIXED (start / 64, start % 64),
                   MAKEFIXED (sweep / 64, sweep % 64));
#elif defined (__X_MOTIF__) || defined(__X_YACL__)
    long startAngle     = arc.StartAngle();
    long subtAngle      = arc.SubtendedAngle();
    Display *dpy        = YACLApp()->AppDisplay();
    XSetForeground (dpy, _handle, _pen->Color().XPixel());
    XDrawArc (dpy, _Drawable(), Handle(), rect.Left(),
              rect.Top(), rect.Width(), rect.Height(), startAngle,
              subtAngle);
#else
    NotImplemented ("DrawArc");
#endif
}






void UI_DrawingSurface::DrawChord (const UI_Chord& chord,
                                   ushort opt)
{
    UI_Rectangle  rect  = chord.Ellipse().BoundingRectangle();
#if defined (__MS_WINDOWS__) || defined(__MS_WIN32__)
    HBRUSH hbr, old;
    if (opt & UID_Fill) {
        old = SelectObject (_handle, _brush->Handle());
    }
    else {
        hbr   = GetStockObject (HOLLOW_BRUSH);
        old   = SelectObject (_handle, hbr);
    }
    if (! (opt & UID_Outline))
        SelectObject (_handle, GetStockObject (NULL_PEN));
    UI_PointPair pp = chord.EndPoints();
    Chord (Handle(),
           rect.Left(),  rect.Top(),
           rect.Right(), rect.Bottom(),
           pp.p1.XCoord(),  pp.p1.YCoord(),
           pp.p2.XCoord(),  pp.p2.YCoord());
    if (! (opt & UID_Outline))
        SelectObject (_handle, _pen->Handle());
    SelectObject (_handle, old);
#elif defined(__OS2__)
    long width  = rect.Width  ();
    long height = rect.Height ();
    ARCPARAMS arcParams = {width/2, height/2, 0, 0};
    GpiSetArcParams (_handle, &arcParams);
    POINTL center;
    UI_Point centerPt = rect.Center();
    center.x = centerPt.XCoord();
    center.y = DrawingArea().Height() - centerPt.YCoord() - 1;
    long start = chord.StartAngle(), sweep = chord.SubtendedAngle();

    GpiSetLineType (_handle, LINETYPE_INVISIBLE);
    GpiPartialArc (_handle, &center, MAKEFIXED(1,0),
                   MAKEFIXED (start / 64, start % 64),
                   MAKEFIXED (0, 0));
    POINTL startPt;
    GpiQueryCurrentPosition (_handle, &startPt);
    if ((opt & UID_Fill) && (_brush->Pattern() != UIBrush_Hollow)) {
        if (SupportsColor())
            GpiSetColor (_handle, _brush->Color().NativeForm());
        GpiBeginArea (_handle, 0L);
        GpiPartialArc (_handle, &center, MAKEFIXED(1,0),
                       MAKEFIXED (start / 64, start % 64),
                       MAKEFIXED (sweep / 64, sweep % 64));
        GpiEndArea (_handle);
    }
    if (opt & UID_Outline) {
        if (_pen) {
            if (SupportsColor())
                GpiSetColor (_handle, _pen->Color().NativeForm());
            GpiSetLineType (_handle, _pen->OS2Pattern());
        }
        GpiPartialArc (_handle, &center, MAKEFIXED(1,0),
                       MAKEFIXED (start / 64, start % 64),
                       MAKEFIXED (sweep / 64, sweep % 64));
        GpiLine (_handle, &startPt);
    }
#elif defined (__X_MOTIF__) || defined(__X_YACL__)
    long startAngle     = chord.StartAngle();
    long subtAngle      = chord.SubtendedAngle();
    Display *dpy        = YACLApp()->AppDisplay();

    if (opt & UID_Fill) {
        XGCValues v;
        v.arc_mode = ArcChord;
        XChangeGC (dpy, Handle(), GCArcMode, &v);
        XSetForeground (dpy, _handle, _brush->Color().XPixel());
        XFillArc (dpy, _Drawable(), Handle(), rect.Left(),
                  rect.Top(), rect.Width(), rect.Height(), startAngle,
                  subtAngle);
    }
    if (opt & UID_Outline) {
        UI_PointPair pp     = chord.EndPoints();

        XSetForeground (dpy, _handle, _pen->Color().XPixel());
        XDrawArc (dpy, _Drawable(), Handle(), rect.Left(),
                  rect.Top(), rect.Width(), rect.Height(), startAngle,
                  subtAngle);
        XDrawLine (dpy, _Drawable(), Handle(),
                   pp.p1.XCoord(), pp.p1.YCoord(),
                   pp.p2.XCoord(), pp.p2.YCoord());
    }
#else
    NotImplemented ("DrawChord");
    
#endif
}




void UI_DrawingSurface::DrawPieWedge (const UI_PieWedge& pie,
                                      ushort opt)
{
    UI_Rectangle  rect  = pie.Ellipse().BoundingRectangle();
#if defined (__MS_WINDOWS__) || defined(__MS_WIN32__)
    HBRUSH hbr, old;
    if (opt & UID_Fill) {
        old = SelectObject (_handle, _brush->Handle());
    }
    else {
        hbr   = GetStockObject (HOLLOW_BRUSH);
        old   = SelectObject (_handle, hbr);
    }
    if (! (opt & UID_Outline))
        SelectObject (_handle, GetStockObject (NULL_PEN));
    UI_PointPair pp = pie.EndPoints();
    Pie (Handle(),
         rect.Left(),  rect.Top(),
         rect.Right(), rect.Bottom(),
         pp.p1.XCoord(),  pp.p1.YCoord(),
         pp.p2.XCoord(),  pp.p2.YCoord());
    if (! (opt & UID_Outline))
        SelectObject (_handle, _pen->Handle());
    SelectObject (_handle, old);
#elif defined(__OS2__)
    long width  = rect.Width  ();
    long height = rect.Height ();
    ARCPARAMS arcParams = {width/2, height/2, 0, 0};
    GpiSetArcParams (_handle, &arcParams);
    POINTL center;
    UI_Point centerPt = rect.Center();
    center.x = centerPt.XCoord();
    center.y = DrawingArea().Height() - centerPt.YCoord() - 1;
    long start = pie.StartAngle(), sweep = pie.SubtendedAngle();
    GpiMove (_handle, &center);
    if ((opt & UID_Fill) && (_brush->Pattern() != UIBrush_Hollow)) {
        if (SupportsColor())
            GpiSetColor (_handle, _brush->Color().NativeForm());
        GpiBeginArea (_handle, 0L);
        GpiPartialArc (_handle, &center, MAKEFIXED(1,0),
                       MAKEFIXED (start / 64, start % 64),
                       MAKEFIXED (sweep / 64, sweep % 64));
        GpiLine (_handle, &center);
        GpiEndArea (_handle);
    }
    if (opt & UID_Outline) {
        if (_pen) {
            if (SupportsColor())
                GpiSetColor (_handle, _pen->Color().NativeForm());
            GpiSetLineType (_handle, _pen->OS2Pattern());
        }
        GpiPartialArc (_handle, &center, MAKEFIXED(1,0),
                       MAKEFIXED (start / 64, start % 64),
                       MAKEFIXED (sweep / 64, sweep % 64));
        GpiLine (_handle, &center);
    }
#elif defined (__X_MOTIF__) || defined(__X_YACL__)
    long startAngle     = pie.StartAngle();
    long subtAngle      = pie.SubtendedAngle();
    Display *dpy        = YACLApp()->AppDisplay();

    if (opt & UID_Fill) {
        XGCValues v;
        v.arc_mode = ArcPieSlice;
        XChangeGC (dpy, Handle(), GCArcMode, &v);
        XSetForeground (dpy, _handle, _brush->Color().XPixel());
        XFillArc (dpy, _Drawable(), Handle(), rect.Left(),
                  rect.Top(), rect.Width(), rect.Height(), startAngle, 
                  subtAngle);
    }
    if (opt & UID_Outline) {
	UI_Rectangle rect  = pie.Ellipse().BoundingRectangle();
	UI_PointPair pp    = pie.EndPoints();

        XSetForeground (dpy, _handle, _pen->Color().XPixel());
	XDrawArc (dpy, _Drawable(), Handle(),
		  rect.Left(), rect.Top(), rect.Width(),
		  rect.Height(), startAngle, subtAngle);
	XDrawLine (dpy, _Drawable(), Handle(), pp.p1.XCoord(),
                   pp.p1.YCoord(), rect.Left()+(rect.Width()/2),
		   rect.Top()+(rect.Height()/2));
	XDrawLine (dpy, _Drawable(), Handle(),
                   pp.p2.XCoord(), pp.p2.YCoord(),
		   rect.Left()+(rect.Width()/2),
		   rect.Top()+(rect.Height()/2));
    }
#else
    NotImplemented ("DrawPieWedge");
    
#endif
}


void UI_DrawingSurface::DrawPolygon
     (UI_Point point[], short num_pts, ushort opt)
{
    if (num_pts < 2 || !opt || !_handle)
        return;
#if defined(__MS_WINDOWS__) || defined(__MS_WIN32__)
    POINT* pts = new POINT [num_pts];
    if (!pts)
        return; // No memory
    for (short i = 0; i < num_pts; i++) {
        pts[i].x = point[i].XCoord();
        pts[i].y = point[i].YCoord();
    }
    if ((opt & UID_Outline) && (opt & UID_Fill))
        Polygon (_handle, pts, num_pts); // Both filling and outlining
    else if (opt & UID_Outline) {
        // Outline only, no filling
        HBRUSH oldBrush = SelectObject (_handle, GetStockObject (NULL_BRUSH));
        Polygon (_handle, pts, num_pts);
        SelectObject (_handle, oldBrush);
    }
    else if (opt & UID_Fill) {
        // No outline
        HPEN oldPen = SelectObject (_handle, GetStockObject (NULL_PEN));
        Polygon (_handle, pts, num_pts);
        SelectObject (_handle, oldPen);
    }
    delete [] pts;
#elif defined (__X_MOTIF__) || defined(__X_YACL__) 
    Display *dpy = YACLApp()->AppDisplay();
    if (num_pts <= 1) return;
    if (opt & UID_Fill) {
        XPoint* xp = new XPoint [num_pts];
        for (long i = 0; i < num_pts; i++) {
            xp [i].x = point [i].XCoord ();
            xp [i].y = point [i].YCoord ();
        }
        XSetForeground (dpy, _handle, _brush->Color().XPixel());
        XFillPolygon (dpy, _Drawable(), _handle, xp, num_pts,
                      Convex, CoordModeOrigin);
        delete [] xp;
    }
    if (opt & UID_Outline) {
        DrawPolyLine (point, num_pts);
        DrawLine (point [num_pts-1], point [0]);
    }
#elif defined(__OS2__)
    POINTL* pts = new POINTL [num_pts + 1];
    if (!pts)
        return; // No memory
    long height = DrawingArea().Height();
    for (short i = 0; i < num_pts; i++) {
        pts[i].x = point[i].XCoord();
        pts[i].y = height - point[i].YCoord() - 1;
    }
    pts[num_pts] = pts[0];
    if (opt & UID_Fill) {
        GpiSetColor (_handle, _brush->Color().NativeForm());
        GpiBeginPath  (_handle, 1);
        GpiMove (_handle, &pts[0]);
        GpiPolyLine (_handle, num_pts+1, pts);
        GpiEndPath (_handle);
        GpiFillPath (_handle, 1L, FPATH_ALTERNATE);
    }
    if (opt & UID_Outline) {
        GpiSetColor (_handle, _pen->Color().NativeForm());
        GpiBeginPath  (_handle, 1);
        GpiMove (_handle, &pts[0]);
        GpiPolyLine (_handle, num_pts+1, pts);
        GpiEndPath (_handle);
        GpiOutlinePath (_handle, 1L, 0);
    }
    delete [] pts;
#else
        NotImplemented ("DrawPolygon");
#endif
}






void UI_DrawingSurface::DrawPolyLine (UI_Point point[], short num_pts)
{
    if (num_pts <= 1)
        return;
#if defined(__MS_WINDOWS__) || defined(__MS_WIN32__)
    for (short i = 1; i < num_pts; i++)
        DrawLine (point[i-1], point[i]);
#elif defined(__OS2__)
    POINTL* pts = new POINTL [num_pts-1];
    if (!pts)
        return; // No memory
    long height = DrawingArea().Height();
    pts[0].x = point[0].XCoord();
    pts[0].y = height - point[0].YCoord();
    GpiSavePS (_handle);
    GpiMove (_handle, pts);
    for (short i = 1; i < num_pts; i++) {
        pts[i-1].x = point[i].XCoord();
        pts[i-1].y = height - point[i].YCoord() - 1;
    }
    GpiPolyLine (_handle, num_pts-1, pts);
    GpiRestorePS (_handle, -1);
    delete [] pts;
#elif defined(__X_MOTIF__) || defined(__X_YACL__)
    if (num_pts <= 1)
        return;
    Display *dpy = YACLApp()->AppDisplay();
    for (long i = 1; i < num_pts; i++)
        DrawLine (point [i-1], point [i]);
#else
    NotImplemented ("DrawPolyLine");
#endif
}


bool UI_DrawingSurface::DrawLine (const UI_Point& p, const UI_Point& q)
{
#if defined(__MS_WINDOWS__) || defined(__MS_WIN32__)
    POINT array[2];
    array[0].x = p.XCoord();
    array[0].y = p.YCoord();
    array[1].x = q.XCoord();
    array[1].y = q.YCoord();
    if (Polyline (Handle(), array, 2)) {
        return TRUE;
    }
    return FALSE;
#elif defined(__OS2__)
    if (!_handle)
        return FALSE;
    long height = DrawingArea().Height();
    POINTL pt;
    pt.x = p.XCoord();
    pt.y = height - p.YCoord() - 1;
    GpiSavePS (_handle);
    GpiMove (_handle, &pt);
    if (SupportsColor())
        GpiSetColor (_handle, _pen->Color().NativeForm());
    pt.x = q.XCoord();
    pt.y = height - q.YCoord() - 1;
    GpiLine (_handle, &pt);
    GpiRestorePS (_handle, -1);
    return TRUE;
#elif defined (__X_MOTIF__) || defined(__X_YACL__)
    Display* dpy = YACLApp()->AppDisplay();
    XSetForeground (dpy, _handle, _pen->Color().XPixel());
    XDrawLine (dpy, _Drawable(), Handle (),
               p.XCoord(), p.YCoord(), q.XCoord(), q.YCoord());
    return TRUE;
#else
    NotImplemented ("DrawLine");
    return FALSE;
#endif
}




void UI_DrawingSurface::DrawRectangle (const UI_Rectangle& r,
                                       ushort opt)
{
    if (!opt)
        return;
#if defined(__MS_WINDOWS__) || defined(__MS_WIN32__)
    if (!_handle)
        return;
    if (r.Width() <= 0 && r.Height() <= 0)
        return;
    if (r.Width() <= 1 && r.Height() <= 1) {
        SetPixel (_handle, r.Left(), r.Top(), _pen->Color().NativeForm());
        return;
    }
    if (opt & UID_Fill && _handle) {
        RECT rect = r.AsMSRect ();
        rect.right++; rect.bottom++; // Padding because Windows does not
                                     // fill the right and bottom borders
        FillRect (_handle, &rect, _brush->Handle());
    }
    if (opt & UID_Outline) {
//         UI_Point pts[5];
//         pts[0] = r.TopLeft();
//         pts[1] = r.TopRight();
//         pts[2] = r.BottomRight();
//         pts[3] = r.BottomLeft();
//         pts[4] = pts[0];
//         COLORREF clr = _pen->Color().NativeForm();
//         for (short i = 0; i < 4; i++) {
//             DrawLine (pts[i], pts[i+1]);
//             SetPixel (_handle, pts[i].XCoord(), pts[i].YCoord(), clr);
//         }
        POINT pts[5];
        pts[0].x = r.Left(), pts[0].y = r.Top ();
        pts[1].x = r.Right (), pts[1].y = pts[0].y;
        pts[2].x = pts[1].x,   pts[2].y = r.Bottom();
        pts[3].x = pts[0].x,   pts[3].y = pts[2].y;
        pts[4] = pts[0];
        Polyline (_handle, pts, 5);
        
    }
#elif defined(__OS2__)
    if (r.Width() <= 0 && r.Height() <= 0)
        return;
    if (r.Width() <= 1 && r.Height() <= 1) {
        POINTL pt;
        pt.x = r.Left();
        pt.y = DrawingArea().Height() - r.Bottom() - 1;
        GpiSetPel (_handle, &pt);
        return;
    }
    POINTL pt[4], ptLB, ptTR;
    ptLB.x = r.Left();
    ptLB.y = DrawingArea().Height() - r.Bottom() - 1;

    pt[0].x = ptLB.x;
    pt[0].y = ptLB.y + r.Height() - 1;

    pt[1].x = pt[0].x + r.Width() - 1;
    pt[1].y = pt[0].y;

    pt[2].x = r.Right();
    pt[2].y = ptLB.y;

    pt[3] = ptLB;
    
    ptTR.x = r.Right();
    ptTR.y = ptLB.y + r.Height() - 1;
    GpiSavePS (_handle);
    if ((opt & UID_Fill) && (_brush->Pattern() != UIBrush_Hollow)) {
        if (SupportsColor())
            GpiSetColor (_handle, _brush->Color().NativeForm());
        GpiMove (_handle, &ptLB);
        GpiBox  (_handle, DRO_FILL, &ptTR, 0, 0);
    }
    if (opt & UID_Outline) {
        if (SupportsColor())
            GpiSetColor (_handle, _pen->Color().NativeForm());
        GpiMove (_handle, &ptLB);
        //    GpiBox  (_handle, DRO_OUTLINE, &ptTR, 0, 0);
        GpiPolyLine (_handle, 4, pt);
    }
    GpiRestorePS (_handle, -1);
#elif defined(__X_MOTIF__) || defined(__X_YACL__)
    Display* dpy = YACLApp()->AppDisplay();
    if (opt & UID_Fill) {
        XSetForeground (dpy, _handle, _brush->Color().XPixel());
        XFillRectangle (dpy, _Drawable(), _handle,  r.Left (),
                        r.Top (), r.Width (), r.Height ());
    }
    if (opt & UID_Outline) {
        XSetForeground (dpy, _handle, _pen->Color().XPixel());
        XDrawRectangle (dpy, _Drawable(), _handle,
                        r.Left (), r.Top (), r.Width (), r.Height ());
    }
#endif
}




#if defined(__X_MOTIF__) || defined(__X_YACL__)
static Region _MakeRectRegion (const UI_Rectangle& r)
{
    Region region;
    XPoint points[5];

    long left   = r.Left ();
    long right  = r.Right ();
    long top    = r.Top  ();
    long bottom = r.Bottom ();
    points[0].x = left;
    points[0].y = top;

    points[1].x = right;
    points[1].y = top;

    points[2].x = right;
    points[2].y = bottom;

    points[3].x = left;
    points[3].y = bottom;

    points[4].x = left;
    points[4].y = top;

    region = XPolygonRegion (points, 5, WindingRule);
    return region;
}

#endif


bool UI_DrawingSurface::InvertRectangle (const UI_Rectangle& r)
{
#if defined(__MS_WINDOWS__) || defined(__MS_WIN32__)
    RECT rect =  r.AsMSRect ();
    InvertRect (_handle, &rect);
    return TRUE;
#elif defined(__OS2__)
    long height = DrawingArea ().Height();
    RECTL rect;
    rect.xLeft    = r.Left();
    rect.yBottom  = height - r.Bottom () - 1;
    rect.xRight   = r.Right();
    rect.yTop     = height - r.Top ();
    return WinInvertRect (_handle, &rect);
        // There must be a Gpi function for this, but I don't know which.
#elif defined(__X_MOTIF__) || defined(__X_YACL__)
    Display *dpy  = YACLApp()->AppDisplay();
    Drawable win  = _Drawable();
    long width  = r.Width();
    long height = r.Height();

        // This code is courtesy of Rajesh Chandran, Jan 25, 1995
    XGCValues values;
    XtGCMask mask = GCForeground|GCBackground|GCFunction|GCSubwindowMode;
    GC gc;


    gc = _handle;
    XGCValues oldValues;
    XGetGCValues   (dpy, gc, mask, &oldValues);
//     values.foreground     = 1;
//     values.background     = 0;
//     values.foreground     = BlackPixel (dpy, scr);
//     values.background     = WhitePixel (dpy, scr);
    values.function       = GXxor;
    values.subwindow_mode = IncludeInferiors;
    values.foreground     = oldValues.background;
    values.background     = oldValues.foreground;
    Region region = _MakeRectRegion (r);
    XChangeGC      (dpy, gc, mask, &values);
    XSetRegion     (dpy, gc, region);
    XDestroyRegion (region);
    XFillRectangle (dpy, win, gc, r.Left(), r.Top(), width, height);
    XChangeGC      (dpy, gc, mask, &oldValues);
    XSetClipMask   (dpy, gc, None);
    
// The following code uses a more elaborate approach of inverting the
// image pixel by pixel. It is much slower, but illustrates the idea, so
// is retained here but commented out.
//     int format  = DefaultDepth (dpy, scr) == 1 ? XYPixmap : ZPixmap;
//     XImage* handle = XGetImage (dpy, win, r.Left(), r.Top(), width, height,
//                                 AllPlanes, format);
//     // Now iterate over the image and complement all its cells
// 
//     CL_IntIntMap tmpColorMap;
//     // This map is used to speed up the loops below, avoiding XQueryColor
//     // and XAllocColor calls if we already know what the pixel values are.
//     for (long i = 0; i < width; i++) {
//         for (long j = 0; j < height; j++) {
//             XColor xcolor;
//             xcolor.pixel = XGetPixel (handle, i, j);
//             long outputPixel = tmpColorMap[xcolor.pixel] - 1;
//             // -----------------------------------------^^^^
//             // We subtract one here, and add 1 below, because we want the
//             // values of the map to be positive and nonzero. Otherwise a
//             // lookup failure (which returns a 0) would be indistinguishable
//             // from a zero pixel value for outputPixel. Of course, we could
//             // use the Map's IncludesKey method, but that would call for two
//             // map lookups instead of one.
//             if (outputPixel < 0) { // Lookup failed
//                 long inputPixel = xcolor.pixel;
//                 XQueryColor (dpy, DefaultColormap (dpy, scr), &xcolor);
//                 xcolor.green = 65535L - xcolor.green;
//                 xcolor.blue  = 65535L - xcolor.blue;
//                 xcolor.red   = 65535L - xcolor.red;
//                 XAllocColor (dpy, DefaultColormap (dpy, scr), &xcolor);
//                 outputPixel = xcolor.pixel;
//                 tmpColorMap.Add (inputPixel, outputPixel + 1);
//                 // --------------------------------------^^^ see above
//                 //                                           comment
//             }
//             XPutPixel (handle, i, j, outputPixel);
//         }
//     }
// 
//     // Finally, output the image
//     XPutImage (dpy, win, _handle, handle, 0, 0, r.Left(), r.Top(),
//                width, height);

    return TRUE;
#else
    NotImplemented ("InvertRectangle");
    return FALSE;
#endif
}


void UI_DrawingSurface::DrawPoint (const UI_Point& p)
{
#if defined(__MS_WINDOWS__) || defined(__MS_WIN32__)
    SetPixel (_handle, p.XCoord(), p.YCoord(), _brush->Color().NativeForm());
#elif defined(__OS2__)
    GpiSavePS (_handle);
    if (SupportsColor())
        GpiSetColor (_handle, _brush->Color().NativeForm());
    POINTL pt;
    pt.x = p.XCoord();
    pt.y = DrawingArea().Height() - p.YCoord() - 1;
    GpiSetPel (_handle, &pt);
    GpiRestorePS (_handle, -1);
#elif defined (__X_MOTIF__) || defined(__X_YACL__)
    Display* dpy = YACLApp()->AppDisplay();
    XSetForeground (dpy, _handle, _pen->Color().XPixel());
    XDrawPoint (dpy, _Drawable(), _handle, p.XCoord(), p.YCoord());
#else
    NotImplemented ("DrawPoint");
#endif
}





void UI_DrawingSurface::WriteString (const char* s,
                                     const UI_Rectangle& tRect,
                                     UI_TextStyle tStyle)
{
#if defined(__MS_WINDOWS__) || defined(__MS_WIN32__)
    UI_Point p = tRect.Origin();
    CL_String str (s);
    long style;
    TEXTMETRIC tm;
    RECT rect = tRect.AsMSRect();
    rect.right++; rect.bottom++; // Fix for off-by-one
    GetTextMetrics (_handle, &tm);
    style = (tStyle == UIText_Left) ? DT_LEFT
        : (tStyle == UIText_Center ? DT_CENTER : DT_RIGHT);
    int oldMode = SetBkMode (_handle, TRANSPARENT);
    DrawText(_handle, str.AsPtr(), str.Size(), &rect, style);
    SetBkMode (_handle, oldMode);
#elif defined(__OS2__)
    CL_String str (s);
    if (SupportsColor())
        GpiSetColor (_handle, _pen->Color().NativeForm());
    GpiSetBackMix (_handle, BM_LEAVEALONE);
    UI_Font& fnt = Font();
    short fontDescent = fnt.Descent(), fontAscent = fnt.Ascent();
    long ht =  DrawingArea().Height();
    // Now adjust for the base line of the font  
    // (See p. 109 of Petzold's book)    
    POINTL stringBase;
    stringBase.x = tRect.Left();
    stringBase.y = ht - tRect.Top() - fontAscent - 1;
    RECTL clipRect;
    clipRect.xLeft   = tRect.Left();
    if (tStyle == UIText_Center)
        stringBase.x += (tRect.Width() - TextWidth (str))/2;
    else if (tStyle == UIText_Right)
        stringBase.x += tRect.Width() - TextWidth (str);
    clipRect.yBottom = stringBase.y - fontDescent;
    clipRect.xRight  = tRect.Right();
    clipRect.yTop    = clipRect.yBottom + fontAscent + fontDescent - 1;
    HRGN hrgn = GpiCreateRegion (_handle, 1, &clipRect);
    HRGN oldRegion;
    GpiSetClipRegion (_handle, hrgn, &oldRegion);
    if (GpiCharStringAt (_handle, (PPOINTL) &stringBase, str.Length(),
                         (char*) str.AsPtr()) == GPI_ERROR)
        UI_Application::PMError();
    GpiSetClipRegion (_handle, NULLHANDLE, &oldRegion);
#elif defined(__X_YACL__) || defined(__X_MOTIF__)
    if (!_font)
        return;
    Display* dpy = YACLApp()->AppDisplay();
    Drawable   d = _Drawable();
    XFontStruct* fstr = (XFontStruct*) _font->NativeFontStruct ();
    if (!fstr)
        return; // Failed

    // Now get the lines in the string, and draw them one at a time
    CL_StringSequence lines = CL_String(s).Split ('\n');
    long nLines = lines.Size();
    short ht    = _font->Height();
    long tRectWidth = tRect.Width(), tRectHeight = tRect.Height(),
        tRectTop = tRect.Top(), tRectLeft = tRect.Left();
    for (long i = 0, y = tRectTop; i < nLines; i++, y += ht) {
        short l = TextWidth (lines(i));
        short x;
        x =  (l >= tRectWidth || tStyle == UIText_Left)
            ? 0
            : (tStyle == UIText_Center
               ? (tRectWidth - l)/2
               : (tRectWidth - l));
        x += tRectLeft;
        short w = minl (l, tRectWidth);
        UI_Rectangle lineRect (x, y, w, ht);
        Region region = _MakeRectRegion (lineRect);
        XSetRegion     (dpy, _handle, region);
        XDestroyRegion (region);
        XSetForeground (dpy, _handle, _pen->Color().XPixel());
        CL_String& strg = lines(i);
        UI_Point p = lineRect.Origin();
        XDrawString (dpy, d, _handle, x, y + fstr->ascent,
                     (const char*) strg, strg.Length ());
        if (_font->Underline ()) {
            // XGetFontProperty (fstr, XA_UNDERLINE_POSITION, &val); :-(
            // Since the XA_UNDERLINE_POSITION property is not reliable, we
            // resort to ad-hoc measures.
            UI_Point leftEnd  = p + UI_Point
                (0, fstr->ascent + fstr->descent - 2);
            UI_Point rightEnd = leftEnd + UI_Point (lineRect.Width(), 0);
            _DrawFontLine (leftEnd, rightEnd);
        }
        if (_font->StrikeOut ()) {
            int direction, ascent, descent;
            XCharStruct cstr;
            XTextExtents (fstr, "g", 1, &direction, &ascent, &descent, &cstr);

            UI_Point leftEnd = p + UI_Point (0, fstr->ascent -  cstr.ascent/2);
            UI_Point rightEnd = leftEnd + UI_Point (lineRect.Width(), 0);
            _DrawFontLine (leftEnd, rightEnd);
        }
        XSetClipMask   (dpy, _handle, None); // Must fix this if we want to 
                                             // allow the user to set clip
                                             // masks
    }

#else
    NotImplemented ("WriteString");
#endif
}




short UI_DrawingSurface::TextWidth (const char* strg) const
{
#if defined (__MS_WINDOWS__)
    return GetTextExtent (_handle, strg, lstrlen(strg));
#elif defined(__MS_WIN32__)
    SIZE sz;
    BOOL b = GetTextExtentPoint (_handle, strg, lstrlen(strg), &sz);
    return b ? sz.cx : 0;
#elif defined(__OS2__)
    POINTL textBox[TXTBOX_COUNT];
    GpiQueryTextBox (_handle, strlen (strg), (char*) strg,
                     TXTBOX_COUNT, textBox);
    return textBox[TXTBOX_CONCAT].x;
#elif defined (__X_MOTIF__) || defined(__X_YACL__)
    return _font ? _font->Width (strg) : 0;
#else
    NotImplemented ("TextWidth"); // Implemented in DisplaySurface
    return 0;
#endif
}





#if defined(__MS_WINDOWS__) || defined(__MS_WIN32__)
long UI_DrawingSurface::_BitBltROP () const
{
    return SRCCOPY;
}
#elif defined(__OS2__)
long UI_DrawingSurface::_BitBltROP () const
{
    return ROP_SRCCOPY;
}
#endif


bool UI_DrawingSurface::DrawBitmap (const UI_Bitmap& b, const UI_Point& p)
{
#if defined (__MS_WINDOWS__) || defined(__MS_WIN32__)
    UI_BitmapHandle bmpHandle = b.Handle();
    if (!bmpHandle)
        return FALSE;
    HDC destdc = Handle ();
    if (!destdc)
        return FALSE;
    HDC hdc = CreateCompatibleDC (destdc);
    if (!hdc)
        return FALSE;
    HANDLE oldh = SelectObject (hdc, bmpHandle);
    SetMapMode (hdc, GetMapMode (destdc));

    BITMAP bm;
    POINT ptsize, ptorg;

    GetObject (bmpHandle, sizeof (BITMAP), (LPSTR) &bm);
    ptsize.x = bm.bmWidth;
    ptsize.y = bm.bmHeight;
    DPtoLP (destdc, &ptsize, 1);

    ptorg.x = 0; ptorg.y = 0;
    DPtoLP (hdc, &ptorg, 1);
                
    BitBlt (destdc, p.XCoord(), p.YCoord(), b.Width(), b.Height(), hdc, 0, 0,
            _BitBltROP());
    SelectObject (hdc, oldh);
    DeleteDC (hdc);
    return TRUE;

#elif defined(__OS2__)
    POINTL pt[4];
    pt[0].x = p.XCoord();
    pt[0].y = DrawingArea().Height() - p.YCoord() - b.Height() + 1;
    pt[1].x = pt[0].x + b.Width ()  - 1;
    pt[1].y = pt[0].y + b.Height () - 1;
    pt[2].x = pt[2].y = 0;
    pt[3].x = b.Width ()  - 1;
    pt[3].y = b.Height () - 1;
    GpiWCBitBlt (_handle, b.Handle(), 4, pt, ROP_SRCCOPY, BBO_AND);
//     extern CL_String PrintForm (RECTL*);
//     CL_Error::Warning ("DwgSurface DrawBitmap: dest %s src %s\n",
//                        PrintForm ((RECTL*) &pt).AsPtr(),
//                        PrintForm ((RECTL*) &pt[2]).AsPtr()); //DEBUG
    
    return TRUE;
#elif defined (__X_MOTIF__) || defined(__X_YACL__)
    Display* dpy  = YACLApp()->AppDisplay();
    Drawable   d  = _Drawable ();
    XPutImage (dpy, d, _handle, b.Handle(), 0, 0, p.XCoord(), p.YCoord(),
               b.Width(), b.Height());
    return TRUE;
#else
    NotImplemented ("DrawBitmap");
    return FALSE;
#endif
}





bool UI_DrawingSurface::_FontWillChange ()
{
#if defined (__MS_WINDOWS__) || defined(__MS_WIN32__)
    SelectObject (_handle, GetStockObject (SYSTEM_FONT));
#endif
    return TRUE;
}


bool UI_DrawingSurface::_FontChanged ()
{
#if defined (__MS_WINDOWS__) || defined(__MS_WIN32__)
    SelectObject (_handle, _font->Handle ());
#elif defined(__OS2__)
    GpiSetCharSet (_handle, _font->Handle());
#elif defined (__X_MOTIF__) || defined(__X_YACL__)
    UI_ResourceHandle h = _font->Handle();
    if (h)
        XSetFont (YACLApp()->AppDisplay(), _handle, h);
#endif
    return TRUE;
}








struct StateStruct {

public:
    StateStruct (const UI_PenDescriptor& pen,
                 const UI_BrushDescriptor& brush,
                 const UI_FontDesc& font)
        : penState (pen), brushState (brush), fontState (font) {};
    UI_PenDescriptor   penState;
    UI_BrushDescriptor brushState;
    UI_FontDesc        fontState;
};
    

bool UI_DrawingSurface::SaveState ()
{
    if (!_handle || !_pen || !_brush || !_font)
        return FALSE;
    StateStruct* state = new StateStruct (_pen->Descriptor(),
                                          _brush->Descriptor(),
                                          _font->Descriptor());
    _stateStack.Add (state);
    return TRUE;
}


bool UI_DrawingSurface::RestoreState ()
{
    if (!_handle || !_stateStack.Size() || !_pen || !_brush || !_font)
        return FALSE;
    StateStruct* state = (StateStruct*) _stateStack.ExtractRightmost();
    if (state) {
        *_pen    = state->penState;
        *_brush  = state->brushState;
        *_font   = state->fontState;
    }
    delete state;
    return TRUE;
// #if defined(__MS_WINDOWS__) || defined(__MS_WIN32__)
//     return RestoreDC (_handle, -1) != 0 ? TRUE : FALSE;
// #elif defined(__OS2__)
//     return GpiRestorePS (_handle, -1);
// #elif defined(__X_MOTIF__) || defined(__X_YACL__)
//     if (_stateStack.Size() <= 0)
//         return FALSE;
//     XGCValues* v = (XGCValues*) _stateStack.ExtractRightmost();
//     if (!v)
//         return FALSE;
//     XChangeGC (YACLApp()->AppDisplay(), _handle, (ulong) -1, v);
//     delete v;
//     return TRUE;
// #endif
}


bool UI_DrawingSurface::DrawMaskedBitmap (const UI_Bitmap& b, uchar *andMask,
                                          const UI_Point& p)
{
#if defined (__MS_WINDOWS__) || defined(__MS_WIN32__) 
    HDC destdc = Handle ();

    if (!destdc)
        return FALSE;
    HDC memDC = CreateCompatibleDC (destdc);
    if (!memDC)
        return FALSE;

    HANDLE hloc      = LocalAlloc (LMEM_ZEROINIT | LMEM_MOVEABLE,
                                   sizeof (BITMAPINFOHEADER)+(sizeof (RGBQUAD)
                                   * 16));
    PBITMAPINFO pbmi = (PBITMAPINFO) LocalLock (hloc);
    RGBQUAD argbq [] = { {0  , 0  , 0  , 0},
                         {255, 255, 255, 0}
                       };

    pbmi->bmiHeader.biSize        = sizeof (BITMAPINFOHEADER);
    pbmi->bmiHeader.biWidth       = b.Width();
    pbmi->bmiHeader.biHeight      = b.Height();
    pbmi->bmiHeader.biPlanes      = 1;
    pbmi->bmiHeader.biBitCount    = 1;
    pbmi->bmiHeader.biCompression = 0;
    memcpy (pbmi->bmiColors, argbq, sizeof (RGBQUAD)*2);

    HBITMAP hMaskBM = CreateDIBitmap (memDC, (BITMAPINFOHEADER FAR *) pbmi,
                                      CBM_INIT, andMask, pbmi, DIB_RGB_COLORS);
    LocalFree (hloc);
    HANDLE oldh = SelectObject (memDC, hMaskBM);
    SetMapMode (memDC, GetMapMode (destdc));

    BITMAP bm;
    POINT ptsize, ptorg;

    GetObject (hMaskBM, sizeof (BITMAP), (LPSTR) &bm);
    ptsize.x  = bm.bmWidth;
    ptsize.y  = bm.bmHeight;
    DPtoLP (destdc, &ptsize, 1);

    ptorg.x = 0; ptorg.y = 0;
    DPtoLP (memDC, &ptorg, 1);
                
    BitBlt (destdc, p.XCoord(), p.YCoord(), b.Width(), b.Height(), memDC,
            0, 0, SRCAND);
    UI_BitmapHandle bmpHandle = b.Handle();
    oldh      = SelectObject (memDC, bmpHandle);
    SetMapMode (memDC, GetMapMode (destdc));
    GetObject (bmpHandle, sizeof (BITMAP), (LPSTR) &bm);
    BitBlt (destdc, p.XCoord(), p.YCoord(), b.Width(), b.Height(),
            memDC, 0, 0, SRCINVERT);
    SelectObject (memDC, oldh);
    DeleteObject (hMaskBM);
    DeleteDC (memDC);

    return TRUE;

#elif defined (__OS2__)
    BITMAPINFO2       *pbmi;
    BITMAPINFOHEADER2  bmi ;

    bmi.cbFix     = sizeof (BITMAPINFOHEADER2); 
    bmi.cx        = b.Width();
    bmi.cy        = b.Height();
    bmi.cPlanes   = 1;
    bmi.cBitCount = 1;

    pbmi = (BITMAPINFO2 *) new uchar [sizeof (BITMAPINFO2) +
                                      (2*sizeof (RGB2))];
    memset (pbmi, 0, sizeof (BITMAPINFOHEADER2));
    pbmi->cbFix     = sizeof (BITMAPINFOHEADER2);
    pbmi->cx        = b.Width();
    pbmi->cy        = b.Height();
    pbmi->cPlanes   = 1;
    pbmi->cBitCount = 1;
    pbmi->argbColor [0].bBlue  = 0;
    pbmi->argbColor [0].bGreen = 0;
    pbmi->argbColor [0].bRed   = 0;
    pbmi->argbColor [1].bBlue  = 0xff;
    pbmi->argbColor [1].bGreen = 0xff;
    pbmi->argbColor [1].bRed   = 0xff;

    HPS     hps = Handle();
    HBITMAP hbm = GpiCreateBitmap (hps, &bmi, CBM_INIT, (char*) andMask, pbmi);

    POINTL pt[4];
    pt[0].x = p.XCoord();
    pt[0].y = DrawingArea().Height() - p.YCoord() - b.Height();
    pt[1].x = pt[0].x + b.Width ()  - 1;
    pt[1].y = pt[0].y + b.Height () - 1;
    pt[2].x = pt[2].y = 0;
    pt[3].x = b.Width ()  - 1;
    pt[3].y = b.Height () - 1;
    LONG rop = ROP_MERGEPAINT;
    GpiWCBitBlt     (_handle, hbm,        4, pt, rop,           BBO_AND);
    GpiWCBitBlt     (_handle, b.Handle(), 4, pt, ROP_SRCINVERT, BBO_AND);
    GpiDeleteBitmap (hbm);
    return TRUE;

#elif defined (__X_MOTIF__) || defined(__X_YACL__)
    Display *dpy      = YACLApp()->AppDisplay();
    short screen_num  = DefaultScreen (dpy);
    Visual  *v        = DefaultVisual (dpy, screen_num);
    int depth         = DefaultDepth (dpy, screen_num);
    Drawable d        = _Drawable ();
    int format        = (depth == 1 ? XYPixmap: ZPixmap);
    int pad           = (depth == 1 ? 32: (depth > 8) ? 32: 8);
    ushort wp         = WhitePixel (dpy, screen_num);
    ushort bp         = BlackPixel (dpy, screen_num);

    // Because X treats white and black pixels differently than Windows
    // special processing was needed to be done to get the pixels on the
    // screen and individually apply the masks on them. A Black Pixel
    // seemed to be 1 and a White Pixel is 0 in X, but its vice versa in
    // Windows.

    ushort width  = b.Width();
    ushort height = b.Height();
    
    if (width + p.XCoord() > DrawingArea().Width())
        return FALSE;
    if (height + p.YCoord() > DrawingArea().Height())
        return FALSE;
    
    char *bits  = new char [b.Width()*b.Height()];
    XImage *dim = XGetImage (dpy, d, p.XCoord(), p.YCoord(), width,
                             height, AllPlanes, format);
    XImage *im  = XCreateImage (dpy, v, depth, format, 0, bits, b.Width(),
                                b.Height(), pad, 0);
    uchar shift = 8;
    ushort result = 0;

    for (short i = 1; i <= height; ++i) {
        for (short j = 0; j < width; ++j) {
            shift --;
            ushort pixel = ((((*andMask) >> shift) & 0x1) ? 1: 0);
            if (shift == 0) { andMask++; shift = 8; }
            short xorp   = XGetPixel (b.Handle(), j, height - i);
            short existp = XGetPixel (dim, j, height - i);
            if (existp == bp) existp = wp;
            else if (existp == wp) existp = bp;
            result = (existp & pixel) ^ xorp;
            XPutPixel (im, j, height - i, result);
        }
    }
    XPutImage (dpy, d, Handle(), im, 0, 0, (int) p.XCoord(),
               (int) p.YCoord(), b.Width(), b.Height());
    XDestroyImage ( im);
    XDestroyImage (dim);
    return TRUE;
#endif
}


#if defined(__X_MOTIF__) || defined(__X_YACL__)
void UI_DrawingSurface::_DrawFontLine (const UI_Point& p1, const UI_Point& p2)
{
    short width = _pen->Thickness ();
    _pen->Thickness (2);
    DrawLine (p1, p2);
    _pen->Thickness (width);
}

#endif



// bool UI_DrawingSurface::BeginMaskMode (short w, short h, uchar* mask)
// {
// #if defined(__X_YACL__) || defined(__X_MOTIF__)
//     if (_maskPixmap)
//         return FALSE;  // Already in masked mode
//     Display* d = YACLApp()->AppDisplay();
//     _maskPixmap = XCreatePixmapFromBitmapData (d, _Drawable(), (char*) mask,
//                                                w, h, 1, 0, 1);
//     XGCValues val;
//     val.stipple = _maskPixmap;
//     val.fill_style = FillStippled;
//     XChangeGC (d, _handle, GCFillStyle|GCStipple, &val);
//     return TRUE;
// #else
//     return TRUE; // For now
// #endif
// }



// bool UI_DrawingSurface::EndMaskMode ()
// {
// #if defined(__X_YACL__) || defined(__X_MOTIF__)
//     if (_maskPixmap) {
//         XGCValues val;
//         Display* d = YACLApp()->AppDisplay();
//         val.stipple = 0;
//         val.fill_style = 0;
//         XChangeGC (d, _handle, GCFillStyle | GCStipple, &val);
//         XFreePixmap (d, _maskPixmap);
//         _maskPixmap = 0;
//         return TRUE;
//     }
//     return FALSE;
// #else
//     return TRUE; // For now
// #endif
// }



