
#include "ui/dsplsurf.h"
#include "ui/bitmap.h"
#include "ui/stddlg.h"
#include "ui/applic.h"
#include "ui/cntroler.h"
#include "appwin.h"
#include "sysvals.h"

#include <iostream.h> // DEBUG
#if defined(__GNUC__)
template class CL_Binding0 <AppWindow>;
#endif

typedef CL_Binding0<AppWindow> Bind;

// scrollbars' view IDs
const UI_ViewID ID_VERTBAR = 102;
const UI_ViewID ID_HORZBAR = 103;

SysValues sVals;




AppWindow::AppWindow()
: UI_CompositeVObject (NULL, NULL, FALSE, UI_Rectangle (150, 150, 350, 250))
{
    Title () = "Bitmap Viewer";
    VScroller = NULL;
    HScroller = NULL;
    _bitMap = NULL;
#if defined(__OS2__)
    _style  = WS_CLIPCHILDREN;
#endif
}


AppWindow::~AppWindow ()
{
    DestroyDisplaySurface();    
    if (_bitMap)
        delete _bitMap;
}


void AppWindow::Initialize ()
{
    UI_CompositeVObject::Initialize();
    CreateDisplaySurface ();
    
    // create the scrollbars at the edges of the main window,
    // initially hidden.
    VScroller = new UI_VScrollBar(this,
                                  UI_Rectangle(Shape().Width() - sVals.scrollBarWidth, 
                                               sVals.menuBarHeight + 1,
                                               sVals.scrollBarWidth, 
                                               Shape().Height() - sVals.scrollBarHeight
                                               - sVals.menuBarHeight),
                                  ID_VERTBAR);
    HScroller = new UI_HScrollBar(this,
                                  UI_Rectangle(0, 
                                               Shape().Height() - sVals.scrollBarHeight,
                                               Shape().Width() - sVals.scrollBarWidth,
                                               sVals.scrollBarHeight),                                               
                                  ID_HORZBAR);    

    VScroller->SmoothScroll() = FALSE;
    HScroller->SmoothScroll() = FALSE;
// #if 0
//     Bind vscrollBind (this, &AppWindow::_ScrollVert);
//     VScroller->ClientSet().Add (vscrollBind);
//     Bind hscrollBind (this, &AppWindow::_ScrollHorz);
//     HScroller->ClientSet().Add (hscrollBind);
// #endif
    HScroller->LineAmount() = 5;    
    VScroller->LineAmount() = 5;
    HScroller->SetVisibility(FALSE);
    VScroller->SetVisibility(FALSE);
    _oldvpos = 0;
    _oldhpos = 0;
}




bool AppWindow::Draw (const CL_String& s)
{
// load and draw the bitmap file s
    
    if (!_bitMap)
        _bitMap = new UI_Bitmap();
    
    UI_DisplaySurface* sfc = DisplaySurface ();
    if (sfc) {
        if (!sfc->Handle()) {
            CL_Error::Warning("No handle to DS!");
            return FALSE;
        }
            
        _Controller->BeginWait();
        if (_bitMap->BuildFrom(s.AsPtr(), sfc)) {
            if (!_bitMap->Handle()) {
                _Controller->EndWait();
                CL_Error::Warning("No bitmap handle!");
                return FALSE;
            }
            _Controller->EndWait();
            // reset the scrollbars' models -- mainly to reset the
            // thumb positions back to 0 for displaying the new bitmap
            ((CL_Interval&) VScroller->Model()) = CL_Interval(0, 0);
            ((CL_Interval&) HScroller->Model()) = CL_Interval(0, 0);
            _oldhpos = _oldvpos = 0;
            
            AdjustWindow(_bitMap->Width(), _bitMap->Height());
            Title() = s + " " + CL_String(_bitMap->Width()) + "x" + 
                                CL_String(_bitMap->Height()) + "x" +
                                CL_String(1 << _bitMap->BitsPerPixel());
        }
        else {
            CL_Error::Warning("Can't build bitmap!");
            _Controller->EndWait();
            return FALSE;
        }
    }
    else {
        CL_Error::Warning("No DisplaySurface!");
        return FALSE;
    }
    return TRUE;
}


bool AppWindow::Reconfigure (const UI_Rectangle& r)
{
// the main window is resized, so we must
// adjust the scrollbars to be at the edges of the main window
    
    if (HScroller)
        HScroller->Shape() = UI_Rectangle(0,
                                          r.Height() - sVals.scrollBarHeight + 1,
                                          r.Width() - sVals.scrollBarWidth,
                                          sVals.scrollBarHeight);
    if (VScroller)
        VScroller->Shape() = UI_Rectangle(r.Width() - sVals.scrollBarWidth,
                                          0,
                                          sVals.scrollBarWidth,
                                          r.Height() - sVals.scrollBarHeight + 1);
    AdjustScrollers(r.Width(), r.Height());
    return TRUE;
}


bool AppWindow::Paint (const UI_Rectangle& invalidRect)
{
// paint the bitmap
    
    if (!_bitMap)
        return FALSE;
    UI_DisplaySurface* sfc = DisplaySurface (); 
    if (sfc) {
        if (!sfc->Handle()) {
            CL_Error::Warning("No handle to DS!");
            return FALSE;
        }
        UI_BitmapHandle hbm = _bitMap->Handle();
        if (hbm) {
            CL_Interval& vModel = (CL_Interval&) VScroller->Model();
            CL_Interval& hModel = (CL_Interval&) HScroller->Model();
            long hlow = minl (hModel.Low(), _bitMap->Width()
                              - hModel.Length() + 1);
            long vlow = minl (vModel.Low(), _bitMap->Height()
                              - vModel.Length() + 1);
            if (!_bitMap->DrawOn (*sfc, UI_Point(-hlow, -vlow)))
                CL_Error::Warning("Can't draw bitmap!");
        }
    }
    return TRUE;
}


bool AppWindow::HandleChildEvent (const UI_Event& e)
{
    UI_EventType etype = e.Type();
    if (etype == Event_ScrollForwardLine ||
        etype == Event_ScrollForwardPage ||
        etype == Event_ScrollBackwardLine ||
        etype == Event_ScrollBackwardPage ||
        etype == Event_ScrollToPosition) {
        UI_ViewID viewID = e.Origin()->ViewID();
        if (viewID == ID_VERTBAR)
            return _ScrollVert();
        else if (viewID == ID_HORZBAR)
            return _ScrollHorz();
    }

    // For default processing
    return FALSE;
}


bool AppWindow::HandleEvent (UI_Event* e)
{
    if (e->Type() == Event_Other) {
#if defined(__OS2__)    
        // check for RealizePalette message
        QMSG* qmsg = (QMSG*) e->NativeEvent();
        if (qmsg->msg == WM_REALIZEPALETTE)
            return RealizePalette();
#elif defined(__MS_WINDOWS__)
        // check for changed/new palette messages
        MSG* msg = (MSG*) e->NativeEvent();
        if (msg->message == WM_PALETTECHANGED ||
            msg->message == WM_PALETTEISCHANGING ||
            msg->message == WM_QUERYNEWPALETTE)
            return RealizePalette();
#endif        
    }    
        
    // Default event processing
    return ProcessEvent(e);
}


bool AppWindow::RealizePalette()
{
// realize the bitmap's palette to the display surface
    
    if (!_bitMap)
        return FALSE;
        
    UI_DisplaySurface* sfc = DisplaySurface ();
    if (sfc) {
#if defined(__OS2__)        
        if (_bitMap && _bitMap->PaletteHandle())
            Invalidate ();
#elif defined(__MS_WINDOWS__)
        HDC hdc = sfc->Handle();
        if (_bitMap->Handle()) {        
            if (_bitMap->PaletteHandle()) {            
                HPALETTE hOldPal = SelectPalette
                    (hdc, _bitMap->PaletteHandle(), FALSE);
                int nColorsChanged = ::RealizePalette(hdc);
                if (nColorsChanged)
                    InvalidateRect(_handle, NULL, FALSE);
                if (hOldPal)
                    SelectPalette(hdc, hOldPal, FALSE);
            }
        }
#endif            
    }
    return TRUE;
}


bool AppWindow::AdjustWindow(const long w, const long h)
{
// resize the window to fit the bitmap width w and height h,
// but the window should not extend beyond the physical screen edges.
    
    long new_width, new_height;
    static UI_Rectangle screen_rect = (UI_Rectangle)YACLApp()->ScreenRect();
    long screen_width = screen_rect.Width() - sVals.sizeBorderWidth;
    long screen_height = screen_rect.Height() - sVals.sizeBorderHeight;
    UI_Rectangle& client_area = Shape();
    long x = client_area.Left();
    long y = client_area.Top();
    if ((x + w) <= screen_width) {
        new_width = w;
        if ((y + h) <= screen_height)
            new_height = h;
        else {
            new_height = screen_height - y;
            // readjust the new_width to include the vert scrollbar
            if ((new_width + sVals.scrollBarWidth) <= screen_width)
                new_width += sVals.scrollBarWidth;
            else
                new_width = screen_width - x;
        }
    }
    else {
        new_width = screen_width - x;
        // the new_height should include the horz scrollbar also
        if ((y + h + sVals.scrollBarHeight) <= screen_height)
            new_height = h + sVals.scrollBarHeight;
        else
            new_height = screen_height - y;
    }
    client_area = UI_Rectangle(x, y, new_width, new_height);
    return TRUE;
}


bool AppWindow::AdjustScrollers (long w, long h)
{
// set the scrollbars' parameters (models and ranges),
// given the window width w and height h
    
    if (!_bitMap)
        return FALSE;
    
    int bmWidth = _bitMap->Width();
    int bmHeight = _bitMap->Height();
    int iHscrollMax = maxl(0, bmWidth - w);    
    int iVscrollMax = maxl(0, bmHeight - h);    
    // readjust the visible window's width and height
    // when the scrollbars' width/height are taken into account if they
    // are visible, as they reduce the window's area
    if (iHscrollMax) {
        // the horz scrollbar is visible...
        if (iVscrollMax) {
            // both scrollbars are visible,
            // and they reduce the window's area
            w -= sVals.scrollBarWidth;
            iHscrollMax = maxl(0, bmWidth - w);
            h -= sVals.scrollBarHeight;
            iVscrollMax = maxl(0, bmHeight - h);
        }
        else if (bmHeight > (h - sVals.scrollBarHeight)) {
            // although only the horz scrollbar is visible here, but its
            // presence reduces the window's visible height and obscures
            // part of the bitmap.  We'll force the vert scrollbar to be
            // visible also, and thus its presence also affects the
            // window's visible width
            w -= sVals.scrollBarWidth;
            iHscrollMax = maxl(0, bmWidth - w);
            h -= sVals.scrollBarHeight;
            iVscrollMax = maxl(0, bmHeight - h);            
        }
    }
    else {
        // the horz scrollbar is hidden...
        if (iVscrollMax &&
           (bmWidth > (w - sVals.scrollBarWidth))) {
            // although only the vert scrollbar is visible here, but its
            // presence reduces the window's visible width and obscures
            // part of the bitmap.  We'll force the horz scrollbar to be
            // visible also, and thus its presence also affects the
            // window's visible height
            w -= sVals.scrollBarWidth;
            iHscrollMax = maxl(0, bmWidth - w);
            h -= sVals.scrollBarHeight;
            iVscrollMax = maxl(0, bmHeight - h);            
        }        
    }
    
    if (HScroller) {
        CL_Interval& scrollInterval = (CL_Interval&) HScroller->Model();
        long low = minl (scrollInterval.Low(), bmWidth - 1);
        scrollInterval = CL_Interval (low, low + w - 1);
        //        HScroller->PageAmount() = w;
        //        HScroller->Range() = CL_Interval(0, iHscrollMax);
        HScroller->PageAmount() = scrollInterval.Length();
        HScroller->Range() = CL_Interval(0, bmWidth-1);
        if (iHscrollMax > 0) {
            HScroller->SetVisibility(TRUE);
        }
        else {
            HScroller->SetVisibility(FALSE);
            ((CL_Interval&) HScroller->Model()) = CL_Interval(0, 0);
            _oldhpos = 0;
        }
    }

    if (VScroller) {
        CL_Interval& scrollInterval = (CL_Interval&) VScroller->Model();
        long low = minl (scrollInterval.Low(), bmHeight - 1);
        scrollInterval = CL_Interval (low, low + h - 1);
        //        VScroller->PageAmount() = h;
        //        VScroller->Range() = CL_Interval(0, iVscrollMax);
        VScroller->PageAmount() = scrollInterval.Length();
        VScroller->Range() = CL_Interval(0, bmHeight-1);
        if (iVscrollMax) {
            VScroller->SetVisibility(TRUE);
            if (!HScroller->IsVisible()) {
                // horz scrollbar is hidden,
                // extend the vert scrollbar to reach the bottom
                // of the window
                UI_Rectangle& shape = VScroller->Shape();
                if (shape.Height() < h)
                    shape.AddToHeight(sVals.scrollBarHeight);
            }
        }
        else {
            VScroller->SetVisibility(FALSE);
            ((CL_Interval&) VScroller->Model()) = CL_Interval(0, 0);
            _oldvpos = 0;
            if (HScroller->IsVisible()) {
                // vert scrollbar is hidden,
                // extend the horz scrollbar to reach the right edge
                // of the window
                UI_Rectangle& shape = HScroller->Shape();
                if (shape.Width() < w)
                    shape.AddToWidth(sVals.scrollBarWidth);
            }
        }            
    }
    
    return TRUE;
}


bool AppWindow::_ScrollVert()
{
// scroll the window vertically
    
    long low = ((CL_Interval&) VScroller->Model()).Low();
    long windowHeight = Shape().Height();
    // take into account the horz scrollbar's height
    if (HScroller->IsVisible())
        windowHeight -= sVals.scrollBarHeight;
    long scrollAmount = windowHeight >= _bitMap->Height() ? 0 : low - _oldvpos;
    if (ViewHandle()) {
//#if defined(__OS2__)
//        WinScrollWindow (_handle, 0, scrollAmount, NULL, NULL, NULLHANDLE,
//                         NULL, SW_INVALIDATERGN);
//#elif defined(__MS_WINDOWS__)
        // ScrollWindow(_handle, 0, -scrollAmount, NULL, NULL);
        // don't scroll the window under Windows; as the scrolling also
        // moves the scrollbars along!  Repaint instead
        Invalidate();
//#endif;
        _oldvpos = minl (low, _bitMap->Height() - windowHeight);
    }
    return TRUE;
}


bool AppWindow::_ScrollHorz()
{
// scroll the window horizontally
    
    long low = ((CL_Interval&) HScroller->Model()).Low();
    long windowWidth = Shape().Width();
    // take into account the vert scrollbar's width
    if (VScroller->IsVisible())
        windowWidth -= sVals.scrollBarWidth;
    long scrollAmount = windowWidth >= _bitMap->Width() ? 0 : low - _oldhpos;
    if (ViewHandle()) {
//#if defined(__OS2__)
//        WinScrollWindow (_handle, -scrollAmount, 0, NULL, NULL, NULLHANDLE,
//                         NULL, SW_INVALIDATERGN);
//#elif defined(__MS_WINDOWS__)
        // ScrollWindow(_handle, -scrollAmount, 0, NULL, NULL);
        // don't scroll the window under Windows; as the scrolling also
        // moves the scrollbars along!  Repaint instead        
        Invalidate();
//#endif;
        _oldhpos = low;
    }
    return TRUE;
}
