/* 
 * tclOS2Init.c --
 *
 *	Contains the OS/2-specific interpreter initialization functions.
 *
 * Copyright (c) 1994-1996 Sun Microsystems, Inc.
 * Copyright (c) 1996-1998 Illya Vaes
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

#include "tclOS2Int.h"

/* Global PM variables, necessary because of event loop and thus console */
HAB hab;
HMQ hmq;
/* Other global variables */
ULONG maxPath;
APIRET rc;
BOOL usePm = TRUE;

/*
 * The following arrays contain the human readable strings for the OS/2
 * version values.
 */

static char* processors[] = { "intel", "ppc" };
static const int numProcessors = sizeof(processors);

#ifndef PROCESSOR_ARCHITECTURE_INTEL
#define PROCESSOR_ARCHITECTURE_INTEL 0
#endif
#ifndef PROCESSOR_ARCHITECTURE_PPC
#define PROCESSOR_ARCHITECTURE_PPC   1
#endif
#ifndef PROCESSOR_ARCHITECTURE_UNKNOWN
#define PROCESSOR_ARCHITECTURE_UNKNOWN 0xFFFF
#endif


/*
 * The following string is the startup script executed in new
 * interpreters.  It looks on disk in several different directories
 * for a script "init.tcl" that is compatible with this version
 * of Tcl.  The init.tcl script does all of the real work of
 * initialization.
 */

static char *initScript =
"proc init {} {\n\
    global tcl_library tcl_version tcl_patchLevel env\n\
    rename init {}\n\
    set dirs {}\n\
    if [info exists env(TCL_LIBRARY)] {\n\
        lappend dirs $env(TCL_LIBRARY)\n\
    }\n\
    lappend dirs $tcl_library\n\
    lappend dirs [file join [ file dirname [file dirname [info nameofexecutable]]] lib/tcl$tcl_version]\n\
    if [string match {*[ab]*} $tcl_patchLevel] {\n\
        set lib tcl$tcl_patchLevel\n\
    } else {\n\
        set lib tcl$tcl_version\n\
    }\n\
    lappend dirs [file join [file dirname [file dirname [pwd]]] $lib/library]\n\
    lappend dirs [file join [file dirname [pwd]] library]\n\
    foreach i $dirs {\n\
        set tcl_library $i\n\
        if ![catch {uplevel #0 source [list [file join $i init.tcl]]}] {\n\
            return\n\
        }\n\
    }\n\
    set msg \"Can't find a usable init.tcl in the following directories: \n\"\n\
    append msg \"    $dirs\n\"\n\
    append msg \"This probably means that Tcl wasn't installed properly.\n\"\n\
    error $msg\n\
}\n\
init";

/*
 *----------------------------------------------------------------------
 *
 * TclPlatformInit --
 *
 *	Performs OS/2-specific interpreter initialization related to the
 *	tcl_library variable.  Also sets up the HOME environment variable
 *	if it is not already set.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Sets "tcl_library" and "env(HOME)" Tcl variables
 *
 *----------------------------------------------------------------------
 */

void
TclPlatformInit(interp)
    Tcl_Interp *interp;
{
    char *ptr;
    char buffer[13];
    Tcl_DString ds, key;
    ULONG sysInfo[QSV_MAX];   /* System Information Data Buffer */
    int cpu = PROCESSOR_ARCHITECTURE_INTEL;
    
    tclPlatform = TCL_PLATFORM_OS2;

    _control87(MCW_EM,
      EM_INVALID|EM_DENORMAL|EM_ZERODIVIDE|EM_OVERFLOW|EM_UNDERFLOW|EM_INEXACT);

    Tcl_DStringInit(&ds);

    /*
     * Initialize the tcl_library variable from the user profile (OS2.INI).
     * Environment overrides if set.
     */
    ptr = Tcl_GetVar2(interp, "env", "TCL_LIBRARY", TCL_GLOBAL_ONLY);
    if (ptr == NULL) {
        /* Not in environment */
        ULONG size;

        Tcl_DStringInit(&key);
        Tcl_DStringAppend(&key, "Tcl ", -1);
        Tcl_DStringAppend(&key, TCL_VERSION, -1);
        Tcl_DStringSetLength(&ds, CCHMAXPATH);
        rc = PrfQueryProfileData(HINI_USERPROFILE, Tcl_DStringValue(&key),
                                 "TCL_LIBRARY", Tcl_DStringValue(&ds), &size);
        Tcl_DStringFree(&key);
        if (rc == TRUE) {
            Tcl_DStringSetLength(&ds, size);
        } else {
            Tcl_DStringSetLength(&ds, 0);
        }
    } else {
        Tcl_DStringSetLength(&ds, 0);
        Tcl_DStringAppend(&ds, ptr, -1);
    }
    Tcl_SetVar(interp, "tcl_library", Tcl_DStringValue(&ds), TCL_GLOBAL_ONLY);
    if (Tcl_DStringLength(&ds) > 0) {
        char *argv[3];
        argv[0] = Tcl_GetVar(interp, "tcl_library", TCL_GLOBAL_ONLY);
        argv[1] = "lib";
        argv[2] = NULL;
        Tcl_DStringSetLength(&ds, 0);
        Tcl_SetVar(interp, "tcl_pkgPath", Tcl_JoinPath(2, argv, &ds),
                TCL_GLOBAL_ONLY);
        argv[1] = "lib/tcl" TCL_VERSION;
        Tcl_DStringSetLength(&ds, 0);
        Tcl_SetVar(interp, "tcl_library", Tcl_JoinPath(2, argv, &ds),
                TCL_GLOBAL_ONLY);
    }

    /* Request all available system information */
    rc= DosQuerySysInfo (1L, QSV_MAX, (PVOID)sysInfo, sizeof(ULONG)*QSV_MAX);
    maxPath = sysInfo[QSV_MAX_PATH_LENGTH - 1];

    /*
     * Define the tcl_platform array.
     */

    Tcl_SetVar2(interp, "tcl_platform", "platform", "os2", TCL_GLOBAL_ONLY);
    Tcl_SetVar2(interp, "tcl_platform", "os", "OS/2", TCL_GLOBAL_ONLY);
    /*
     * Hack for LX-versions above 2.11
     *  OS/2 version    MAJOR MINOR
     *  2.0             20    0
     *  2.1             20    10
     *  2.11            20    11
     *  3.0             20    30
     *  4.0             20    40
     */
    if (sysInfo[QSV_VERSION_MAJOR-1]==20 && sysInfo[QSV_VERSION_MINOR-1] > 11) {
        int major = (int) (sysInfo[QSV_VERSION_MINOR - 1] / 10);
        sprintf(buffer, "%d.%d", major,
                (int) sysInfo[QSV_VERSION_MINOR - 1] - major * 10);
    } else {
        sprintf(buffer, "%d.%d", (int) (sysInfo[QSV_VERSION_MAJOR - 1] / 10),
                (int)sysInfo[QSV_VERSION_MINOR - 1]);
    }
    Tcl_SetVar2(interp, "tcl_platform", "osVersion", buffer, TCL_GLOBAL_ONLY);
    /* No API for determining processor (yet) */
    Tcl_SetVar2(interp, "tcl_platform", "machine", processors[cpu],
                TCL_GLOBAL_ONLY);

    /*
     * Set up the HOME environment variable from the HOMEDRIVE & HOMEPATH
     * environment variables, if necessary.
     */

    ptr = Tcl_GetVar2(interp, "env", "HOME", TCL_GLOBAL_ONLY);
    if (ptr == NULL) {
	Tcl_DStringSetLength(&ds, 0);
	ptr = Tcl_GetVar2(interp, "env", "HOMEDRIVE", TCL_GLOBAL_ONLY);
	if (ptr != NULL) {
	    Tcl_DStringAppend(&ds, ptr, -1);
	}
	ptr = Tcl_GetVar2(interp, "env", "HOMEPATH", TCL_GLOBAL_ONLY);
	if (ptr != NULL) {
	    Tcl_DStringAppend(&ds, ptr, -1);
	}
	if (Tcl_DStringLength(&ds) > 0) {
	    Tcl_SetVar2(interp, "env", "HOME", Tcl_DStringValue(&ds),
		    TCL_GLOBAL_ONLY);
	} else {
	    Tcl_SetVar2(interp, "env", "HOME", "c:/", TCL_GLOBAL_ONLY);
	}
    }

    Tcl_DStringFree(&ds);

}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_Init --
 *
 *	This procedure is typically invoked by Tcl_AppInit procedures
 *	to perform additional initialization for a Tcl interpreter,
 *	such as sourcing the "init.tcl" script.
 *
 * Results:
 *	Returns a standard Tcl completion code and sets interp->result
 *	if there is an error.
 *
 * Side effects:
 *	Depends on what's in the init.tcl script.
 *
 *----------------------------------------------------------------------
 */

int
Tcl_Init(interp)
    Tcl_Interp *interp;		/* Interpreter to initialize. */
{
    return Tcl_Eval(interp, initScript);
}

/*
 *----------------------------------------------------------------------
 *
 * TclOS2GetPlatform --
 *
 *      This is a kludge that allows the test library to get access
 *      the internal tclPlatform variable.
 *
 * Results:
 *      Returns a pointer to the tclPlatform variable.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

TclPlatformType *
TclOS2GetPlatform()
{
    return &tclPlatform;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_SourceRCFile --
 *
 *      This procedure is typically invoked by Tcl_Main of Tk_Main
 *      procedure to source an application specific rc file into the
 *      interpreter at startup time.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      Depends on what's in the rc script.
 *
 *----------------------------------------------------------------------
 */

void
Tcl_SourceRCFile(interp)
    Tcl_Interp *interp;         /* Interpreter to source rc file into. */
{
    Tcl_DString temp;
    char *fileName;
    Tcl_Channel errChannel;

    fileName = Tcl_GetVar(interp, "tcl_rcFileName", TCL_GLOBAL_ONLY);

    if (fileName != NULL) {
        Tcl_Channel c;
        char *fullName;

        Tcl_DStringInit(&temp);
        fullName = Tcl_TranslateFileName(interp, fileName, &temp);
        if (fullName == NULL) {
            errChannel = Tcl_GetStdChannel(TCL_STDERR);
            if (errChannel) {
                Tcl_Write(errChannel, interp->result, -1);
                Tcl_Write(errChannel, "\n", 1);
            }
        } else {

            /*
             * Test for the existence of the rc file before trying to read it.
             */
            c = Tcl_OpenFileChannel(NULL, fullName, "r", 0);
            if (c != (Tcl_Channel) NULL) {
                Tcl_Close(NULL, c);
                if (Tcl_EvalFile(interp, fullName) != TCL_OK) {
#ifndef CLI_VERSION
                    char cbuf[1000];
                    sprintf(cbuf, "%s\n", interp->result);
                    WinMessageBox(HWND_DESKTOP, NULLHANDLE, cbuf, "Tclsh", 0,
                                  MB_OK | MB_ICONEXCLAMATION | MB_APPLMODAL);
#else
                    errChannel = Tcl_GetStdChannel(TCL_STDERR);
                    if (errChannel) {
                        Tcl_Write(errChannel, interp->result, -1);
                        Tcl_Write(errChannel, "\n", 1);
                    }
#endif
                }
            }
        }
        Tcl_DStringFree(&temp);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * TclOS2PMInitialize --
 *
 *	Performs OS/2-specific initialization.
 *
 * Results:
 *	True or false depending on intialization.
 *
 * Side effects:
 *	Opens the "PM connection"
 *
 *----------------------------------------------------------------------
 */

BOOL
TclOS2PMInitialize(void)
{
    if (TclOS2GetUsePm()) {
        /* Initialize PM */
        hab = WinInitialize (0);
        if (hab == NULLHANDLE) return FALSE;
        /* Create message queue, increased size from 10 */
        hmq= WinCreateMsgQueue (hab, 64);
        if (hmq == NULLHANDLE) {
            WinTerminate(hab);
            hab= (HAB)0;
            return FALSE;
        }
    }
    return TRUE;
}

/*
 *----------------------------------------------------------------------
 *
 * TclOS2PMShutdown --
 *
 *	Performs OS/2-specific cleanup.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Closes the "PM connection"
 *
 *----------------------------------------------------------------------
 */

void
TclOS2PMShutdown(void)
{
    BOOL rc;

    if (TclOS2GetUsePm()) {
        /* Reset pointer to arrow */
        rc = WinSetPointer(HWND_DESKTOP,
                           WinQuerySysPointer(HWND_DESKTOP, SPTR_ARROW, FALSE));
        WinDestroyMsgQueue(hmq);
        WinTerminate(hab);
        hmq= (HMQ)0;
        hab= (HAB)0;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * TclOS2GetHAB --
 *
 *	Get the handle to the anchor block.
 *
 * Results:
 *	HAB or NULLHANDLE.
 *
 * Side effects:
 *	None
 *
 *----------------------------------------------------------------------
 */

HAB
TclOS2GetHAB(void)
{
    return hab;
}

/*
 *----------------------------------------------------------------------
 *
 * TclPlatformExit --
 *
 *	Cleanup and exit on OS/2.
 *
 * Results:
 *	None. This procedure never returns (it exits the process when
 *	it's done).
 *
 * Side effects:
 *	This procedure terminates all relations with PM.
 *
 *----------------------------------------------------------------------
 */

void
TclPlatformExit(status)
    int status;				/* Status to exit with */
{
    if (usePm) {
        /*
         * Set focus to Desktop to force the Terminal edit window to reinstate
         * the system pointer.
         */
        WinSetFocus(HWND_DESKTOP, HWND_DESKTOP);
        WinDestroyMsgQueue(hmq);
        WinTerminate(hab);
        hmq= (HAB)0;
        hab= (HMQ)0;
    }
    exit(status);
}

/*
 *----------------------------------------------------------------------
 *
 * TclOS2GetUsePm --
 *
 *	Get the value of the DLL's usePm value
 *
 * Results:
 *	Value of usePm (Bool).
 *
 * Side effects:
 *	None
 *
 *----------------------------------------------------------------------
 */

BOOL
TclOS2GetUsePm(void)
{
    return usePm;
}

/*
 *----------------------------------------------------------------------
 *
 * TclOS2SetUsePm --
 *
 *	Set the value of the DLL's usePm value
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Sets the DLL's usePm variable.
 *
 *----------------------------------------------------------------------
 */

void
TclOS2SetUsePm(value)
    BOOL value;
{
    usePm = value;
    return;
}
