
package sun.beanbox;

/**
 * This class manages hookups between properties, so that a
 * bound property change on object X turns into a property
 * set on a related property on object Y.
 * <P>
 * We do this by associating a PropertyHookup adaptor with each
 * source object that we are interested in.  As part of the adaptor
 * we keep track of which target setter methods to call when a given
 * property changes.
 */

import java.lang.reflect.*;
import java.beans.*;
import java.io.*;
import java.util.Hashtable;
import java.util.Vector;

class PropertyHookup implements PropertyChangeListener, Serializable {

    static final long serialVersionUID = 85103197332680806L;

    /**
     * Create a property hookup, so that a change to the named bound
     * property on the source object turns into a call on the "setter"
     * method of the given target object.
     */

    public synchronized static void attach(Wrapper sourceWrapper,
			String propertyName, Method getter,
			Wrapper targetWrapper, Method setter) {

	Object source = sourceWrapper.getBean();
	Object targetObject = targetWrapper.getBean();

	PropertyHookup hook = (PropertyHookup) instances.get(source);
	if (hook == null) {
	    // This is the first property hookup on this source object.
	    // Push a PropertyHookup adaptor onto the source.
	    hook = new PropertyHookup(source);
	    instances.put(source, hook);
	    // Register our listener object with the source Wrapper.
	    sourceWrapper.addEventTarget("propertyChange", null, hook);
	}
	Vector targets = (Vector) hook.targetsByPropertyName.get(propertyName);
	if (targets == null) {
	    targets = new Vector();
	    hook.targetsByPropertyName.put(propertyName, targets);
	}
	PropertyHookupTarget target;
	for (int i = 0; i < targets.size(); i++) {
	    target = (PropertyHookupTarget) targets.elementAt(i);
	    if (target.setter == setter && target.object == targetObject) {
		// We've already got this hookup.  Just return.
		return;
	    }
	}
	targets.addElement(new PropertyHookupTarget(targetObject,setter));

	// propagate the initial value.
	try {
	    Object args1[] = { };
	    Object value = getter.invoke(source, args1);
	    Object args2[] = { value };
	    setter.invoke(targetObject, args2);
        } catch (Exception ex) {
	    System.err.println("Property propagation failed");
	    ex.printStackTrace();
	}
    }

    /**
     * Constructor for a new property hookup adaptor.
     */

    private PropertyHookup(Object source) {
	this.source = source;	
	targetsByPropertyName = new Hashtable();
    }

    /**
     * This is the method that gets called when a bound property
     * changes on the source object.
     * We map the proeprty name to a list of targtes and then
     * call each of the target "setter" methods.
     */

    synchronized public void propertyChange(PropertyChangeEvent evt) {
	String propertyName = evt.getPropertyName();
	Vector targets = (Vector) targetsByPropertyName.get(propertyName);
	if (targets == null) {
	    return;
	}
	Object args[] = { evt.getNewValue() };
	for (int i = 0; i < targets.size(); i++) {
	    PropertyHookupTarget target = (PropertyHookupTarget)targets.elementAt(i);
	    if (target.active) {
		// We're already propagating this change.
		continue;
	    }
	    target.active = true;
	    try {
	        target.setter.invoke(target.object, args);
	    } catch (InvocationTargetException ex) {
	 	System.err.println("Property set failed");
		ex.printStackTrace();
	    } catch (Exception ex) {
	 	System.err.println("Unexpected Property set exception");
		ex.printStackTrace();
	    }
	    target.active = false;
	}
    }

    public void vetoablePropertyChange(PropertyChangeEvent evt)
					throws PropertyVetoException {
	propertyChange(evt);
    }

    // Event source
    Object source;

    // Table that maps from property names to a vector of PropertyHookupTargets
    Hashtable targetsByPropertyName;

    // This table maps from event sources to PropertyHookup objects.
    private static Hashtable instances = new Hashtable();

    private static Class listenerClass = java.beans.PropertyChangeListener.class;
}


// Information for an event delivery target.
class PropertyHookupTarget implements Serializable {

    static final long serialVersionUID = -8352305996623495352L;

    private static int ourVersion = 1;
    Object object;
    Method setter;
    boolean active;

    PropertyHookupTarget(Object object, Method setter) {
	this.object = object;
	this.setter = setter;
    }

    private void writeObject(ObjectOutputStream s)
 				   throws IOException {
	// Because Method objects aren't serializable, we 
        // serialize the name of the setter method.
	s.writeInt(ourVersion);
	s.writeObject(setter.toString());
	s.writeObject(object);
	s.writeBoolean(active);
    }
 
 
    private void readObject(ObjectInputStream s)
  		  throws ClassNotFoundException, IOException {
	int version = s.readInt();
	String setterName = (String)s.readObject();
	object = s.readObject();
	active = s.readBoolean();
	// We do a rather expensive search for a setter method
	// matching the given setterName.
	setter = null;
	Method methods[] = object.getClass().getMethods();
	for (int i = 0; i < methods.length; i++) {
	    if (methods[i].toString().equals(setterName)) {
		setter = methods[i];
		break;
	    }
	}
	if (setter == null) {
	    throw new IOException("PropertyHookupTarget : no suitable setter" +
			"\n    " + setterName + 
			"\n    in class " + object.getClass());
	}
    }
}

