/*----------------------------------------------------------------------*
 * Bounds Checking for GCC.						*
 * Copyright (C) 1995 Richard W.M. Jones <rjones@orchestream.com>.	*
 *----------------------------------------------------------------------*
 * This program is free software; you can redistribute it and/or modify	*
 * it under the terms of the GNU General Public License as published by	*
 * the Free Software Foundation; either version 2 of the License, or	*
 * (at your option) any later version.					*
 *									*
 * This program is distributed in the hope that it will be useful,	*
 * but WITHOUT ANY WARRANTY; without even the implied warranty of	*
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the	*
 * GNU General Public License for more details.				*
 *									*
 * You should have received a copy of the GNU General Public License	*
 * along with this program; if not, write to the Free Software		*
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.		*
 *----------------------------------------------------------------------*
 * File:
 *	lib/functions.c
 * Summary:
 *	Deal with pushing and popping of function contexts and parameters
 *	of those function contexts.
 * Other notes:
 *	
 * Author      	Date		Notes
 * RWMJ		6/3/95		Initial implementation.
 * RWMJ		3/4/95		Some optimizations.
 *----------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>

#include "bounds-lib.h"

#if defined(__BOUNDS_CHECKING_ON)
#error "This file must not be compiled with bounds checking enabled."
#endif

typedef struct _cur_context {
    /* The current function context.
     */
    function_context *current_context;
    /* The current position in the linked list of function contexts.
     */
    function_context_list *current_context_list;
    int args_allocated;
    int args_used;
    object **args;
} cur_context;

cur_context *__bounds_function_context = NULL;

/*----------------------------------------------------------------------
 *	Helper functions that allocate and deallocate function contexts.
 *	We used to allocate and free them immediately, but that was
 *	quite inefficient, so now we allocate blocks of function contexts
 *	and never free them. (RWMJ, 3/4/95).
 *----------------------------------------------------------------------*/

static inline cur_context *
get_context(void)
{
  if (__bounds_function_context == NULL) {
	__bounds_function_context = __bounds_malloc(sizeof(cur_context));
	if (!__bounds_function_context)
	    __bounds_internal_error ("out of memory allocating a block of function contexts",
				 __FILE__, __LINE__);
	__bounds_function_context->current_context = NULL;
	__bounds_function_context->current_context_list = NULL;
	__bounds_function_context->args_allocated = 0;
	__bounds_function_context->args_used = 0;
	__bounds_function_context->args = NULL;
  }
  return __bounds_function_context;
}

/* This constant sets how many function contexts we allocate at once.
 */
#define ALLOCATION_UNIT 64

static inline function_context *
allocate_new_context (void)
{
  function_context *nc;
  cur_context *cc = get_context();

  if (cc->current_context_list != NULL
      && cc->current_context_list->used < ALLOCATION_UNIT)
    {
      /* Some free contexts are allocated already. Return one of those. */
      nc = cc->current_context_list->fc_array + (cc->current_context_list->used ++);
    }
  else if (cc->current_context_list == NULL
	   || (cc->current_context_list->used == ALLOCATION_UNIT
	       && cc->current_context_list->next == NULL))
    {
      /* Need to allocate a new block of function contexts. */
      function_context_list *new_list;

      new_list = __bounds_malloc (sizeof (function_context_list));
      if (new_list)
	new_list->fc_array = __bounds_malloc (sizeof (function_context)
					      * ALLOCATION_UNIT);
      if (!new_list || !new_list->fc_array)
	__bounds_internal_error ("out of memory allocating a block of function contexts",
				 __FILE__, __LINE__);
      new_list->used = 1;
      new_list->next = NULL;
      new_list->prev = cc->current_context_list;
      if (cc->current_context_list) cc->current_context_list->next = new_list;
      cc->current_context_list = new_list;
      nc = new_list->fc_array;
    }
  else
    {
      /* Need to move onto the next block of contexts that was allocated
       * some time previously.
       */
      cc->current_context_list = cc->current_context_list->next;
      nc = cc->current_context_list->fc_array + (cc->current_context_list->used ++);
    }

  /* `nc' is the new context. Return it. */
  return nc;
}

static inline int
may_pop_function_stack (void)
{
  cur_context *cc = get_context();
  return cc->current_context != NULL
/*    && cc->current_context_list != NULL
    && (cc->current_context_list->used > 0
	|| cc->current_context_list->prev != NULL) */
    ;
}

static inline function_context *
remove_this_context (void)
{
  /* may_pop_function_stack has been called before this ... */

  cur_context *cc = get_context();
  cc->current_context_list->used --;
  if (cc->current_context_list->used > 0)
    {
      /* Go back to the previous context in the current block. */
      return cc->current_context_list->fc_array + (cc->current_context_list->used - 1);
    }
  else if (cc->current_context_list->prev != NULL)
    {
      /* Go back to the previous block of contexts. Keep this block, we may
       * use it in the future.
       */
      cc->current_context_list = cc->current_context_list->prev;
      return cc->current_context_list->fc_array + (cc->current_context_list->used - 1);
    }
  else
    {
      /* Popped back to the outer-most level. */
      return NULL;
    }
}

/*----------------------------------------------------------------------
 *	Pointers to function arguments are also allocated dynamically.
 *	These two functions allocate and deallocate a stack of object
 *	pointers. We never actually deallocate the memory used.
 *----------------------------------------------------------------------*/

/* This constant controls the number of new object pointers that we allocate
 * when we need more.
 */
#define ARGS_ALLOCATION_UNIT	256


static inline int
allocate_new_args (int nr_args)
{
  int new_args;

  cur_context *cc = get_context();
  /* Allocate more space, if needbe. */
  if (cc->args_allocated - cc->args_used < nr_args)
    {
#if 1
      /* Do this with '__bounds_realloc' here, now it is implemented.
       */
      while (cc->args_allocated - cc->args_used < nr_args)
        cc->args_allocated += ARGS_ALLOCATION_UNIT;
      cc->args = __bounds_realloc (cc->args, sizeof (object *) * cc->args_allocated);
      if (cc->args == NULL)
	__bounds_internal_error ("out of memory allocating arguments for a function context",
				 __FILE__, __LINE__);
#else
      /* Do it by hand with '__bounds_malloc' and '__bounds_free', though
       * this is pretty slow.
       */
      object **args2;
      int new_allocated = cc->args_allocated + ARGS_ALLOCATION_UNIT;

      while (new_allocated - cc->args_used < nr_args)
	new_allocated += ARGS_ALLOCATION_UNIT;

      args2 = __bounds_malloc (sizeof (object *)
			       * new_allocated);
      if (args2 == NULL)
	__bounds_internal_error ("out of memory allocating arguments for a function context",
				 __FILE__, __LINE__);
      __bounds_memcpy (args2, cc->args, sizeof (object *) * cc->args_allocated);
      __bounds_free (cc->args);
      cc->args_allocated = new_allocated;
      cc->args = args2;
#endif
    }

  /* Return a pointer to the allocated arguments. We don't bother setting
   * the memory to zero, since it's not necessary for the function that
   * calls this.
   */
  new_args = cc->args_used;
  cc->args_used += nr_args;
  return new_args;
}

static inline void
remove_these_args (int nr_args)
{
  cur_context *cc = get_context();
  cc->args_used -= nr_args;
}

/*----------------------------------------------------------------------
 *	Push into a function context.
 *----------------------------------------------------------------------*/

void
__bounds_push_function (char *name, int nr_args, int in_main,
			char *filename, int line)
{
  cur_context *cc = get_context();
#if COLLECT_STATS
  ++__bounds_stats_push_function;
#endif
#if DEBUG_FEATURES
  if (__bounds_debug_print_calls)
    printf ("__bounds_push_function(name=\"%s\", args=%d, main=%d, file=\"%s\", ln=%d)\n",
	    name, nr_args, in_main, filename, line);
#endif

  if (in_main) {
    static int count_main = 0;

    /* This function is supposedly `main'. `main' should occur once and only
     * once. ANSI doesn't allow `main' to be called recursively. Notice that
     * `main' is not usually the first function to be called. That is usually
     * one of our global constructor functions.
     */
    count_main ++;

    if (count_main > 1)
      {
	in_main = 0;		/* Not really in main, if called recursively.*/
	if (count_main == 2)
	  __bounds_warning (filename, line, NULL,
			    "ANSI does not allow main() to be called recursively");
      }
  }

  /* Allocate space for the function_context structure plus all the
   * arguments we're going to get.
   */
  cc->current_context = allocate_new_context ();
  cc->current_context->first_arg = allocate_new_args (nr_args);

  /* Set the context variables as appropriate.
   */
  cc->current_context->in_main = in_main;
  cc->current_context->no_checking = 0;
  cc->current_context->nr_args = nr_args;
  cc->current_context->nr_args_had = 0;
  cc->current_context->name = name;
  cc->current_context->filename = filename;
  cc->current_context->line = line;
  cc->current_context->alloca_stack = NULL;

#if DEBUG_FEATURES
  /* Switch on checking for this function by default. It may be switched off
   * by a call to `__bounds_no_checking_in_this_function'.
   */
  __bounds_debug_no_checking = 0;
#endif
}

/*----------------------------------------------------------------------
 *	Pop out of a function context.
 *----------------------------------------------------------------------*/

void
__bounds_pop_function (char *name)
{
  int i;
  cur_context *cc = get_context();
#if 0
  function_context *old_context;
#endif
#if COLLECT_STATS
  ++__bounds_stats_pop_function;
#endif


#if DEBUG_FEATURES
  if (__bounds_debug_print_calls)
    printf ("__bounds_pop_function(name=\"%s\")\n",
	    name);
#endif

  if (!may_pop_function_stack ())
    __bounds_internal_error ("too many function context pops",
			     __FILE__, __LINE__);

  /* Check that the name passed matches the name pushed into. If not, this
   * is probably a compiler error, else longjmp was used.
   */
  if (cc->current_context->name != name &&
      __bounds_strcmp(cc->current_context->name,name) != 0)
    __bounds_internal_error ("popped out of a mismatched function context",
			     __FILE__, __LINE__);

#if 0
  /* Give a compiler error if we didn't see enough arguments here. This
   * indicates a fault in bounds_build_args().
   * Note (8/6/95): If fact, we may now optimize out these calls, so this
   * is no longer an error.
   */
  if (cc->current_context->nr_args != cc->current_context->nr_args_had)
    __bounds_internal_error ("__bounds_add_param_object called too few times in this function context",
			     __FILE__, __LINE__);
#endif

  /* Delete all the parameters to this function.
   */
  for (i = 0; i < cc->current_context->nr_args_had; ++i)
    {
      __bounds_internal_delete_stack_object (cc->args[cc->current_context->first_arg+i]);
    }

  /* Call the alloca module to (if necessary) delete items from the
   * alloca stack. Only do this if `alloca_stack' is non-NULL to try
   * to avoid the function call.
   */
  if (cc->current_context->alloca_stack != NULL)
    __bounds_alloca_free (cc->current_context->alloca_stack);

  /* Delete the function_context structure.
   */
#if 0
  old_context = cc->current_context;
  cc->current_context = cc->current_context->next;
  __bounds_free (old_context);
#endif
  remove_these_args (cc->current_context->nr_args);
  cc->current_context = remove_this_context ();

#if DEBUG_FEATURES
  /* If we aren't checking bounds in this function context, switch off
   * bounds checking now.
   */
  if (cc->current_context != NULL)
    __bounds_debug_no_checking = cc->current_context->no_checking;
  else
    __bounds_debug_no_checking = 0;
#endif
}

/*----------------------------------------------------------------------
 *	Add a parameter. This function may be called repeatedly as we
 *	enter a function context.
 *----------------------------------------------------------------------*/

void
__bounds_add_param_object (void *ptr, size_t size, size_t align,
			   char *filename, int line, char *name)
{
  cur_context *cc = get_context();
#if COLLECT_STATS
  ++__bounds_stats_param_function;
#endif

#if DEBUG_FEATURES
  if (__bounds_debug_print_calls)
    printf ("__bounds_add_param_object(p=%p, sz=%u, align=%u, file=\"%s\", ln=%d, name=\"%s\")\n",
	    ptr, size, align, filename, line, name);
#endif

  if (cc->current_context == NULL)
    __bounds_internal_error ("__bounds_add_param_object called when not in a function context",
			     __FILE__, __LINE__);

  if (cc->current_context->nr_args_had >= cc->current_context->nr_args)
    __bounds_internal_error ("__bounds_add_param_object called too many times in this function context",
			     __FILE__, __LINE__);

  /* Add the stack object, and note it down in the function_context, so
   * we can automatically delete it later. We set the `no padding' flag
   * in the object since padding isn't added between parameters for
   * compatibility reasons.
   */
  cc->args[cc->current_context->first_arg + (cc->current_context->nr_args_had)++]
    = __bounds_internal_add_stack_object (ptr, size, align,
					  filename, line, name,
					  1);

  /* If this is `main' and we have the first two arguments, then note the
   * positions of the command line arguments here.
   */
  if (cc->current_context->in_main
      && cc->current_context->nr_args_had == 2)
    {
      void *arg0_ptr, *arg1_ptr;
      int argc;
      char **argv;

      arg0_ptr = cc->args[cc->current_context->first_arg+0]->base;/* points to `argc' */
      arg1_ptr = cc->args[cc->current_context->first_arg+1]->base;/* points to `argv' */

      argc = *(int *) arg0_ptr;			/* get them */
      argv = *(char ***) arg1_ptr;

      __bounds_note_main_args (argc, argv);
    }
}

/*----------------------------------------------------------------------
 *	If you don't want to check bounds in the currently active function,
 *	then call this somewhere near the beginning of the function. This
 *	feature only exists if 'DEBUG_FEATURES' is defined in 'bounds-lib.h'
 *	This causes execution of checked code to slow down slightly.
 *----------------------------------------------------------------------*/

#if DEBUG_FEATURES
void
__bounds_no_checking_in_this_function (void)
{
  cur_context *cc = get_context();
  if (cc->current_context != NULL)
    {
      cc->current_context->no_checking = 1;
      __bounds_debug_no_checking = 1;
    }
}
#endif

/*----------------------------------------------------------------------
 *	Print a stack trace. This function is of little interest, since
 *	(a) it misses out all unchecked functions, and
 *	(b) GDB does a better stack trace already.
 *----------------------------------------------------------------------*/

void
__bounds_debug_stack_trace (void)
{
  function_context *context;
  function_context_list *list;
  int i, j;
  cur_context *cc = get_context();

  if (cc->current_context_list == NULL)
    {
      printf ("No stack.\n");
      return;
    }

  /* Rewind to the beginning of the context list. */
  for (list = cc->current_context_list; list->prev; list = list->prev)
    ;

  printf ("Stack trace:\n");
  for (; list; list = list->next)
    {
      for (j = 0; j < list->used; ++j)
	{
	  context = &list->fc_array[j];
	  printf ("- %s (", context->name);
	  for (i = 0; i < context->nr_args_had; ++i)
	    {
	      if (i != 0) printf (", ");
	      printf ("%s = ", cc->args[context->first_arg+i]->name);
	      /* We can only print the argument value if it's an int or a
	       * pointer.
	       */
	      if (cc->args[context->first_arg+i]->size == sizeof (int)
		  && cc->args[context->first_arg+i]->align == sizeof (int))
		{
		  printf ("%u", *(unsigned *)(cc->args[context->first_arg+i]->base));
		}
	      else
		{
		  printf ("?");
		}
	    }
	  printf (") at %s:%d\n",
		  context->filename,
		  context->line);
	}
    }
}

/*----------------------------------------------------------------------
 *	The following function is called from the `alloca.c' code to
 *	update the alloca chain in the current function context. Placing
 *	this function here adds some modularity at the expense of some
 *	speed. Returns the current function name.
 *----------------------------------------------------------------------*/

char *
__bounds_add_alloca_chain (alloca_item *item)
{
  cur_context *cc = get_context();
  if (cc->current_context != NULL)
    {
      item->next = cc->current_context->alloca_stack;
      cc->current_context->alloca_stack = item;
      return cc->current_context->name;
    }
  else
    /* Allow this because we can have mixed objects from bounds checked code
       and non bounds checked code. This also means that the item will never
       be freed. */
#if 0
    __bounds_internal_error ("added item to alloca stack in NULL function context",
			     __FILE__, __LINE__);
#else
    return NULL;
#endif
}
