/*----------------------------------------------------------------------*
 * Bounds Checking for GCC.						*
 * Copyright (C) 1995 Richard W.M. Jones <rjones@orchestream.com>.	*
 *----------------------------------------------------------------------*
 * This program is free software; you can redistribute it and/or modify	*
 * it under the terms of the GNU General Public License as published by	*
 * the Free Software Foundation; either version 2 of the License, or	*
 * (at your option) any later version.					*
 *									*
 * This program is distributed in the hope that it will be useful,	*
 * but WITHOUT ANY WARRANTY; without even the implied warranty of	*
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the	*
 * GNU General Public License for more details.				*
 *									*
 * You should have received a copy of the GNU General Public License	*
 * along with this program; if not, write to the Free Software		*
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.		*
 *----------------------------------------------------------------------*
 * File:
 *	lib/exstatics.c
 * Summary:
 *	Add a lot of external statics to the object tree in one go, making
 *	suitable checks along the way.
 * Other notes:
 *	Use this in conjunction with the grab-statics util. in the tools
 *	directory.
 * Author      	Date		Notes
 * RWMJ		5/9/95		Initial implementation.
 *----------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>

#include "bounds-lib.h"

#if defined(__BOUNDS_CHECKING_ON)
#error "This file must not be compiled with bounds checking enabled."
#endif

#define DUMP 0		/* Dump a list of the objects in order (for dbg). */

static int
compare_entries (const void *p1, const void *p2)
{
  external_statics_table *entry1 = *(external_statics_table **)p1;
  external_statics_table *entry2 = *(external_statics_table **)p2;

  return entry1->address - entry2->address;
}

void
__bounds_add_static_objects_table (external_statics_table *table)
{
  int len, i;
  external_statics_table **ptrs;
  void *last_address;

  /* Compute the length of the table. */
  for (len = 0; table[len].address != NULL; ++len)
    ;
  if (len == 0) return;

  /* Allocate memory for an array of pointers into the table. We will sort
   * the table by swapping these pointers, rather than swapping actual
   * entries in the table. This is more efficient & doesn't cause problems
   * if the table was allocated in r/o memory.
   * N.B. Deliberately allocate a _checked_ heap object here. There is
   * no danger of recursion.
   */
  ptrs = malloc (sizeof (external_statics_table *) * len);
  if (ptrs == NULL)
    __bounds_internal_error ("out of memory allocating table",
			     __FILE__, __LINE__);
  for (i = 0; i < len; ++i)
    ptrs[i] = &table[i];

  /* Sort the table into address order. */
  qsort (ptrs, len, sizeof (external_statics_table *), compare_entries);

#if DUMP
  /* Dump the objects for debugging. */
  for (i = 0; i < len; ++i)
    printf ("%p - %p: %s (at %s:%d)\n",
	    ptrs[i]->address,
	    (char *) (ptrs[i]->address) + ptrs[i]->size - 1,
	    ptrs[i]->name, ptrs[i]->filename, ptrs[i]->line);
#endif

  /* Check that the objects in the table are disjoint. */
  last_address = ptrs[0]->address + ptrs[0]->size;
  for (i = 1; i < len; ++i)
    {
      if (last_address > ptrs[i]->address)
	{
	  __bounds_errorf (__FILE__, __LINE__,
			   NULL, NULL,
			   "overlapping objects `%s:%s' and `%s:%s' in __bounds_add_static_objects_table",
			   ptrs[i-1]->filename,
			   ptrs[i-1]->name,
			   ptrs[i]->filename,
			   ptrs[i]->name);
	  ABORT ();
	}
      last_address = ptrs[i]->address + ptrs[i]->size;
    }

  /* Add the objects to the splay tree. */
  for (i = 0; i < len; ++i)
    {
      external_statics_table *p = ptrs[i];

      __bounds_note_constructed_object (p->address, p->size, p->align,
					p->filename, p->line, p->name);
    }

  /* Free up the table. */
  free (ptrs);
}
