/*----------------------------------------------------------------------*
 * Bounds Checking for GCC.						*
 * Copyright (C) 1995 Richard W.M. Jones <rjones@orchestream.com>.	*
 *----------------------------------------------------------------------*
 * This program is free software; you can redistribute it and/or modify	*
 * it under the terms of the GNU General Public License as published by	*
 * the Free Software Foundation; either version 2 of the License, or	*
 * (at your option) any later version.					*
 *									*
 * This program is distributed in the hope that it will be useful,	*
 * but WITHOUT ANY WARRANTY; without even the implied warranty of	*
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the	*
 * GNU General Public License for more details.				*
 *									*
 * You should have received a copy of the GNU General Public License	*
 * along with this program; if not, write to the Free Software		*
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.		*
 *----------------------------------------------------------------------*
 * File:
 *	lib/error.c
 * Summary:
 *	Error reporting functions for the bounds checking library.
 * Other notes:
 *	
 * Author      	Date		Notes
 * RWMJ		5/12/94		Initial implementation.
 * RWMJ		10/1/96		New format for errors with `file:line'
 *----------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdarg.h>

#include "bounds-lib.h"

#if defined(__BOUNDS_CHECKING_ON)
#error "This file must not be compiled with bounds checking enabled."
#endif

/*----------------------------------------------------------------------
 *	Print something about this object (if obj != NULL).
 *----------------------------------------------------------------------*/

static inline char *
sclass_name (object *obj)
{
  switch (obj->sclass) {
  case obj_sclass_static:
    return "static";
  case obj_sclass_external:
    return "external";
  case obj_sclass_stack:
    return "stack";
  case obj_sclass_heap:
    return "heap";
  }
  return NULL;
}

static void
print_object (char *filename, int line, object *obj)
{
  if (!obj) return;

  printf ("%s:%d:  Object `%s':\n"
	  "%s:%d:    Address in memory:    %p .. %p\n"
	  "%s:%d:    Size:                 %u bytes\n"
	  "%s:%d:    Element size:         %u bytes\n"
	  "%s:%d:    Number of elements:   %d\n",
	  filename, line, obj->name ? obj->name : "(unnamed)",
	  filename, line, obj->base, obj->extent - 1,
	  filename, line, obj->size,
	  filename, line, obj->align,
	  filename, line, obj->size / obj->align);
  if (obj->filename && obj->line > 0)
    printf ("%s:%d:    Created at:           %s, line %d\n",
	    filename, line, obj->filename, obj->line);
  if (obj->alloca_function)
    printf ("%s:%d:    Created by alloca in: %s\n",
	    filename, line, obj->alloca_function);
  printf ("%s:%d:    Storage class:        %s\n",
	  filename, line, sclass_name (obj));
}

/*----------------------------------------------------------------------
 *	Breakpoint for use with debuggers such as GDB. If you place
 *	a breakpoint here, then GDB will stop whenever a bounds error
 *	occurs. You can use 'print __bounds_debug_memory(0,0)' to list
 *	out the known memory objects, and 'where' to give a stack trace.
 *----------------------------------------------------------------------*/

void
__bounds_breakpoint (void)
{
  return;
}

/*----------------------------------------------------------------------
 *	Report errors involving single pointers.
 *----------------------------------------------------------------------*/

void
__bounds_error (char *message, char *filename, int line,
		void *pointer, object *obj)
{
  if (filename == NULL)
    {
      filename = "<unknown>";
      line = 0;
    }
  printf ("%s:%d:Bounds error: %s.\n", filename, line, message);
  printf ("%s:%d:  Pointer value: %p\n", filename, line, pointer);
  print_object (filename, line, obj);

  __bounds_breakpoint ();
  if (!__bounds_never_fatal) ABORT ();
}

/*----------------------------------------------------------------------
 *	Report errors involving two pointers.
 *----------------------------------------------------------------------*/

void
__bounds_error2 (char *message, char *filename, int line,
		 void *pointer1, void *pointer2,
		 object *obj1, object *obj2)
{
  if (filename == NULL)
    {
      filename = "<unknown>";
      line = 0;
    }
  printf ("%s:%d:Bounds error: %s.\n", filename, line, message);
  printf ("%s:%d:  Left pointer value: %p\n", filename, line, pointer1);
  print_object (filename, line, obj1);
  printf ("%s:%d:  Right pointer value: %p\n", filename, line, pointer2);
  print_object (filename, line, obj2);

  __bounds_breakpoint ();
  if (!__bounds_never_fatal) ABORT ();
}

/*----------------------------------------------------------------------
 *	Report errors with optional format string.
 *----------------------------------------------------------------------*/

void
__bounds_errorf (char *filename, int line,
		 void *pointer, object *obj,
		 char *format, ...)
{
  va_list args;

  if (filename == NULL)
    {
      filename = "<unknown>";
      line = 0;
    }
  printf ("%s:%d:Bounds error: ", filename, line);
  va_start (args, format);
  vprintf (format, args);
  va_end (args);
  printf (".\n");
  printf ("%s:%d:  Pointer value: %p\n", filename, line, pointer);
  print_object (filename, line, obj);

  __bounds_breakpoint ();
  if (!__bounds_never_fatal) ABORT ();
}

/*----------------------------------------------------------------------
 *	Function for reporting internal errors in the library.
 *----------------------------------------------------------------------*/

void
__bounds_internal_error (char *message, char *filename, int line)
{
  if (filename == NULL)
    {
      filename = "<unknown>";
      line = 0;
    }
  printf ("%s:%d:Internal error: %s.\n", filename, line, message);

  ABORT ();
}

/*----------------------------------------------------------------------
 * Give a warning message as appropriate.
 *----------------------------------------------------------------------*/

void
__bounds_warning (char *filename, int line, char *function, char *format, ...)
{
  va_list args;

  if (filename == NULL)
    {
      filename = "<unknown>";
      line = 0;
    }
  if (function)
    printf ("%s:%d:In function %s,\n", filename, line, function);
  printf ("%s:%d:Bounds warning: ", filename, line);
  va_start (args, format);
  vprintf (format, args);
  va_end (args);
  printf (".\n");
}

/*----------------------------------------------------------------------
 *	Report warnings involving single pointers.
 *----------------------------------------------------------------------*/

void
__bounds_warn1 (char *message, char *filename, int line,
		void *pointer, object *obj)
{
  if (filename == NULL)
    {
      filename = "<unknown>";
      line = 0;
    }
  printf ("%s:%d:Bounds warning: %s.\n", filename, line, message);
  printf ("%s:%d:  Pointer value: %p\n", filename, line, pointer);
  print_object (filename, line, obj);

  __bounds_breakpoint ();
}

/*----------------------------------------------------------------------
 *	Report warnings with optional format string.
 *----------------------------------------------------------------------*/

void
__bounds_warnf (char *filename, int line,
		 void *pointer, object *obj,
		 char *format, ...)
{
  va_list args;

  if (filename == NULL)
    {
      filename = "<unknown>";
      line = 0;
    }
  printf ("%s:%d:Bounds warning: ", filename, line);
  va_start (args, format);
  vprintf (format, args);
  va_end (args);
  printf (".\n");
  printf ("%s:%d:  Pointer value: %p\n", filename, line, pointer);
  print_object (filename, line, obj);

  __bounds_breakpoint ();
}
