/*----------------------------------------------------------------------*
 * Bounds Checking for GCC.						*
 * Copyright (C) 1995 Richard W.M. Jones <rjones@orchestream.com>.	*
 *----------------------------------------------------------------------*
 * This program is free software; you can redistribute it and/or modify	*
 * it under the terms of the GNU General Public License as published by	*
 * the Free Software Foundation; either version 2 of the License, or	*
 * (at your option) any later version.					*
 *									*
 * This program is distributed in the hope that it will be useful,	*
 * but WITHOUT ANY WARRANTY; without even the implied warranty of	*
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the	*
 * GNU General Public License for more details.				*
 *									*
 * You should have received a copy of the GNU General Public License	*
 * along with this program; if not, write to the Free Software		*
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.		*
 *----------------------------------------------------------------------*
 * File:
 *	lib/check.c
 * Summary:
 *	This file contains all the functions that are used to check pointers
 *	at run time, and to manage the list of memory objects.
 * Other notes:
 *	
 * Author      	Date		Notes
 * RWMJ		5/12/94		Initial implementation.
 *----------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include "bounds-lib.h"
#include "check.h"

#if defined(__BOUNDS_CHECKING_ON)
#error "This file must not be compiled with bounds checking enabled."
#endif

/* Return whether a pointer is aligned with respect to an object.
 */
static inline int
pointer_not_aligned (void *pointer, object *obj)
{
  unsigned long offset = PTR_TO_UNSIGNED (pointer)
                       - PTR_TO_UNSIGNED (obj->base);
  return obj->align_mask ? (offset & (obj->align-1)) != 0
			 : (offset % obj->align) != 0;
}

/*----------------------------------------------------------------------
 *	__bounds_check_ptr_plus_int is generated by cc1 whenever the
 *	expression (pointer + offset*size) needs to be evaluated. If the
 *	arithmetic produces an illegal pointer (ie. object pointed to is
 *	under/over-run or misaligned), we return an ILLEGAL pointer that
 *	will be caught if it is dereferenced or used in further arith-
 *	metic.
 *----------------------------------------------------------------------*/

void *
__bounds_check_ptr_plus_int (void *pointer, int offset, size_t size,
			     int is_plus, char *filename, int line)
{
  object *obj;
  void *new_pointer;

#if COLLECT_STATS
  ++__bounds_stats_ptr_plus_int;
#endif

#if DEBUG_FEATURES
  if (__bounds_debug_no_checking) {
    return is_plus ? pointer + size*offset : pointer - size*offset;
  }
  if (__bounds_debug_print_calls) {
    printf ("__bounds_check_ptr_plus_int(p=%p, off=%d, sz=%u, %s, file=\"%s\", ln=%d)\n",
	    pointer, offset, size, is_plus ? "'+'" : "'-'", filename, line);
  }
#endif

  if (pointer == NULL || pointer == ILLEGAL) {
    /* NULL and ILLEGAL pointers clearly mustn't be used in pointer arithmetic.
     */
    __bounds_error ("NULL or ILLEGAL pointer used in pointer arithmetic",
		    filename, line, pointer, NULL);
    return is_plus ? pointer + size*offset : pointer - size*offset;
  }
  if ((obj = __bounds_find_object (pointer)) == NULL) {
    if (maybe_is_unchecked_static (pointer, filename, line, NULL)
	|| maybe_is_unchecked_stack (pointer, filename, line, NULL)) {
      /* Unchecked pointer used. Allow the arithmetic to go ahead here.
       */
      return is_plus ? pointer + size*offset : pointer - size*offset;
    }
    /* Unknown pointer used in further arithmetic. This arithmetic
     * must be wrong.
     */
    __bounds_error ("illegal pointer used in pointer arithmetic",
		    filename, line, pointer, NULL);
    return is_plus ? pointer + size*offset : pointer - size*offset;
  }

  /* Check to see if this operation is OK for this type of object. If so,
   * we return the result as usual. Otherwise, we return an ILLEGAL pointer
   * that will be caught some time in the future (if it is used).
   */
  if (is_plus)
    new_pointer = pointer + size*offset;
  else
    new_pointer = pointer - size*offset;
  if (new_pointer < obj->base ||
      new_pointer > obj->extent) {
    if (__bounds_warn_illegal)
      __bounds_warn1 ("created an ILLEGAL pointer in pointer arithmetic",
		      filename, line, new_pointer, obj);
    return ILLEGAL;
  }
  if (__bounds_warn_unaligned && pointer_not_aligned (new_pointer, obj)) {
    /* Note: We might return ILLEGAL here, but I think generating a non-
     * aligned pointer at any time must be an error.
     */
    __bounds_error ("created an unaligned pointer in pointer arithmetic",
		    filename, line, new_pointer, obj);
    return new_pointer;
  }

  return new_pointer;
}

/*----------------------------------------------------------------------
 *	__bounds_check_array_reference is generated by cc1 whenever the
 *	expression (pointer + offset*size) needs to be evaluated in an
 *	array reference. It is almost equivalent to the `ptr_plus_int'
 *	case, except that the pointer returned must always be correct,
 *	so we will never return ILLEGAL here, but always fail.
 *----------------------------------------------------------------------*/

int
__bounds_check_array_reference (void *pointer, int offset, size_t size,
				size_t array_size, char *filename, int line)
{
  object *obj;
  void *new_pointer;

#if COLLECT_STATS
  ++__bounds_stats_array_reference;
#endif

#if DEBUG_FEATURES
  if (__bounds_debug_no_checking) return offset;
  if (__bounds_debug_print_calls) {
    printf ("__bounds_check_array_reference(p=%p, off=%d, sz=%u, ar_sz=%u, file=\"%s\", ln=%d)\n",
	    pointer, offset, size, array_size, filename, line);
  }
#endif

  if (pointer == NULL || pointer == ILLEGAL) {
    /* NULL or ILLEGAL pointers clearly mustn't be used in pointer arithmetic.
     */
    __bounds_error ("NULL or ILLEGAL pointer used in array reference",
		    filename, line, pointer, NULL);
    return offset;
  }

  /* HtB has implemented the following hack which will correctly check the
   * bounds of 2D arrays. This is turned on by default.
   */
  if (__bounds_array_index_check &&
      (offset < 0 || (array_size > size && (size*offset) >= array_size))) {
    new_pointer = pointer + size*offset;
    if ((obj = __bounds_find_object (pointer)) == NULL) {
      __bounds_note_constructed_object(pointer,-array_size,size,
                                       filename,line,NULL);
      obj = __bounds_find_object (pointer);
      __bounds_error ("array reference outside bounds of the array",
                      filename, line, new_pointer, obj);
      return offset;
    }
    else {
      obj->base = pointer;
      obj->extent = pointer + array_size;
      obj->size = array_size;
      obj->align = size;
      __bounds_error ("array reference outside bounds of the array",
                      filename, line, new_pointer, obj);
      return offset;
    }
  }

  if ((obj = __bounds_find_object (pointer)) == NULL) {
    if (maybe_is_unchecked_static (pointer, filename, line, NULL)
	|| maybe_is_unchecked_stack (pointer, filename, line, NULL)) {
      /* Unchecked pointer. Allow the arithmetic to go ahead.
       */
      return offset;
    }
    /* Illegal pointer used in further arithmetic. This arithmetic
     * must be wrong.
     */
    __bounds_error ("illegal pointer used in array reference",
		    filename, line, pointer, NULL);
    return offset;
  }

  /* Check to see if this operation is OK for this type of object. If so,
   * we return the result as usual, which will be immediately be referenced,
   * so it had better be correct.
   */
  new_pointer = pointer + size*offset;
  if (new_pointer <  obj->base ||
      new_pointer >= obj->extent) {
    __bounds_error ("array reference outside bounds of the array",
		    filename, line, new_pointer, obj);
    return offset;
  }
  if (__bounds_warn_unaligned && pointer_not_aligned (new_pointer, obj)) {
    /* I'm sure this must be a compiler error if it ever happens, unless
     * the programmer has done some nasty casting.
     */
    __bounds_error ("created an unaligned pointer in array reference",
		    filename, line, new_pointer, obj);
    return offset;
  }

  return offset;
}

/*----------------------------------------------------------------------
 *	__bounds_check_component_reference
 *	Check expressions of the form 'p->e' correctly at run time.
 *----------------------------------------------------------------------*/

void *
__bounds_check_component_reference (void *pointer, int offset, int size,
				    char *filename, int line)
{
  object *obj;

#if COLLECT_STATS
  ++__bounds_stats_component_reference;
#endif

#if DEBUG_FEATURES
  if (__bounds_debug_no_checking) return pointer;
  if (__bounds_debug_print_calls)
    printf ("__bounds_check_component_reference(p=%p, off=%d, sz=%d, file=\"%s\", ln=%d)\n",
	    pointer, offset, size, filename, line);
#endif

  /* Make some internal checks. */
  if (offset < 0 || size <= 0)
    __bounds_internal_error ("incorrect offset or size for '->' operator",
			     filename, line);

  /* Check for NULL and ILLEGAL pointers passed here. Pointer arithmetic on
   * these is certainly wrong.
   */
  if (pointer == NULL || pointer == ILLEGAL) {
    __bounds_error ("NULL or ILLEGAL pointer used in pointer arithmetic",
		    filename, line, pointer, NULL);
    return pointer;
  }
  if ((obj = __bounds_find_object (pointer)) == NULL)
    {
      if (maybe_is_unchecked_static (pointer, filename, line, NULL)
	  || maybe_is_unchecked_stack (pointer, filename, line, NULL))
	return pointer;
      __bounds_error ("illegal pointer used in pointer arithmetic",
		      filename, line, pointer, NULL);
      return pointer;
    }

  /* Check that the operation will not overrun the end of the object. Check
   * that the field is aligned.
   */
  if ((char *) pointer + offset + size > (char *) (obj->extent)) {
    __bounds_error ("pointer arithmetic would overrun the end of the object",
		    filename, line, pointer + offset, obj);
    return pointer;
  }

  return pointer;
}

/*----------------------------------------------------------------------
 *	__bounds_check_ptr_diff is generated by cc1 whenever we need
 *	to evaluate an expression of the form (pointer - pointer). This
 *	object is valid only if the two pointers are pointing to the
 *	same object.
 *----------------------------------------------------------------------*/

int
__bounds_check_ptr_diff (void *pointer1, void *pointer2, size_t size,
			 char *filename, int line)
{
  object *obj1, *obj2;

#if COLLECT_STATS
  ++__bounds_stats_ptr_diff;
#endif

#if DEBUG_FEATURES
  if (__bounds_debug_no_checking) return (pointer1 - pointer2) / size;
  if (__bounds_debug_print_calls) {
    printf ("__bounds_check_ptr_diff(p1=%p, p2=%p, sz=%u, file=\"%s\", ln=%d)\n",
	    pointer1, pointer2, size, filename, line);
  }
#endif

  /* Check for NULL and ILLEGAL pointers passed here. Pointer arithmetic on
   * these is wrong.
   */
  if (pointer1 == NULL || pointer2 == NULL
      || pointer1 == ILLEGAL || pointer2 == ILLEGAL) {
    if (pointer1 == NULL && pointer2 == NULL)
      return 0;
    __bounds_error2 ("NULL or ILLEGAL pointer used in pointer arithmetic",
		    filename, line, pointer1, pointer2, NULL, NULL);
    return (pointer1 - pointer2) / size;
  }
  obj1 = __bounds_find_object(pointer1);
  obj2 = __bounds_find_object(pointer2);
  if (obj1 == NULL || obj2 == NULL) {
    if (maybe_is_unchecked_static (pointer1, filename, line, NULL) &&
	maybe_is_unchecked_static (pointer2, filename, line, NULL)) {
      /* Two unchecked static objects. Allow the arithmetic to go ahead.
       */
      return (pointer1 - pointer2) / size;
    }
    if (maybe_is_unchecked_stack (pointer1, filename, line, NULL) &&
	maybe_is_unchecked_stack (pointer2, filename, line, NULL)) {
      /* Two unchecked stack objects. Allow the arithmetic to go ahead.
       */
      return (pointer1 - pointer2) / size;
    }
    __bounds_error2 ("illegal pointer used in pointer arithmetic",
		    filename, line, pointer1, pointer2, obj1, obj2);
    return (pointer1 - pointer2) / size;
  }

  /* Check that the objects pointed to by the pointers are the same. Since
   * object structures are unique, checking obj1 == obj2 is sufficient.
   */
  if (obj1 != obj2) {
    __bounds_error2 ("in pointer difference, pointers point to different objects",
		     filename, line, pointer1, pointer2, obj1, obj2);
    return (pointer1 - pointer2) / size;
  }

  /* Check that the size of the pointers (size, passed to this function) is
   * a whole multiple of the size of the object's elements.
   */
  if (__bounds_warn_unaligned &&
      (obj1->align_mask
       ? (size & (obj1->align-1)) != 0
       : (size % obj1->align) != 0)) {
    __bounds_errorf (filename, line, pointer1, obj1,
		     "size of pointer (%d bytes) not a whole multiple of size of elements (%d bytes) in pointer difference",
		     size, obj1->align);
    return (pointer1 - pointer2) / size;
  }

  /* Check that both pointers are aligned w.r.t. the object they point to.
   */
  if (__bounds_warn_unaligned && pointer_not_aligned (pointer1, obj1)) {
    __bounds_error ("unaligned pointer used in pointer arithmetic",
		    filename, line, pointer1, obj1);
  }
  if (__bounds_warn_unaligned && pointer_not_aligned (pointer2, obj1)) {
    __bounds_error ("unaligned pointer used in pointer arithmetic",
		    filename, line, pointer2, obj1);
  }

  /* Compute the difference and return it.
   */
  return (pointer1 - pointer2) / size;
}

/*----------------------------------------------------------------------
 *	__bounds_check_reference is generated by cc1 whenever a reference
 *	is about to be made to a pointer. This function does not actually
 *	do the reference, but checks that it can proceed correctly.
 *----------------------------------------------------------------------*/

void*
__bounds_check_reference (void *pointer, size_t size,
			  char *filename, int line)
{
  object *obj;

#if COLLECT_STATS
  ++__bounds_stats_reference;
#endif

#if DEBUG_FEATURES
  if (__bounds_debug_no_checking) return pointer;
  if (__bounds_debug_print_calls) {
    printf ("__bounds_check_reference(p=%p, sz=%u, file=\"%s\", ln=%d)\n",
	    pointer, size, filename, line);
  }
#endif

  /* Check for NULL or ILLEGAL pointers, or pointers that don't seem to
   * point to known memory objects.
   */
  if (pointer == NULL || pointer == ILLEGAL) {
    __bounds_error ("attempt to reference a NULL or ILLEGAL pointer",
		    filename, line, pointer, NULL);
    return pointer;
  }
  if ((obj = __bounds_find_object (pointer)) == NULL) {
    if (maybe_is_unchecked_static (pointer, filename, line, NULL) ||
	maybe_is_unchecked_stack (pointer, filename, line, NULL)) {
      /* Allow unchecked references to go through.
       */
      return pointer;
    }
    __bounds_error ("attempt to reference an illegal pointer",
		    filename, line, pointer, NULL);
    return pointer;
  }

  /* Check that the size is a whole multiple of the element size, and that
   * the pointer is aligned.
   */
  if (__bounds_warn_unaligned &&
      (obj->align_mask
       ? (size & (obj->align-1)) != 0
       : (size % obj->align) != 0)) {
    __bounds_errorf (filename, line, pointer, obj,
		     "size of pointer (%d bytes) not a whole multiple of size of elements (%d bytes) in pointer reference",
		     size, obj->align);
    return pointer;
  }
  if (__bounds_warn_unaligned && pointer_not_aligned (pointer, obj))
    __bounds_error ("unaligned pointer used in pointer reference",
		    filename, line, pointer, obj);

  /* Check that reference will not overrun the end of the object.
   */
  if ((char *) pointer + size > (char *) (obj->extent))
    __bounds_error ("attempt to reference memory overrunning the end of an object",
		    filename, line, pointer, obj);

  return pointer;
}

/*----------------------------------------------------------------------
 *	__bounds_check_ptr_true and __bounds_check_ptr_false. These are
 *	generated by cc1 whenever we meet truthvalue conversion of a
 *	pointer or '! pointer', ie.
 *		if (pointer) { ... }	or	if (!pointer) { ... }
 *	They both check the same thing - namely that the pointer is
 *	NULL or pointing to valid memory. Invalid pointers or ILLEGAL
 *	pointers fail this test.
 *----------------------------------------------------------------------*/

static inline void
check_true_or_false (void *pointer, char *filename, int line)
{
  if (pointer == NULL)
    return;
  if (pointer == ILLEGAL)
    {
      __bounds_error ("attempt to find truthvalue of an ILLEGAL pointer",
		      filename, line, pointer, NULL);
      return;
    }
  if (__bounds_find_object (pointer) == NULL)
    {
      /* Second chance for possible unchecked pointers.
       */
      if (maybe_is_unchecked_static (pointer, filename, line, NULL)
	  || maybe_is_unchecked_stack (pointer, filename, line, NULL))
	return;
      __bounds_error ("attempt to find truthvalue of an illegal pointer",
		      filename, line, pointer, NULL);
    }
}

bool
__bounds_check_ptr_true (void *pointer, char *filename, int line)
{
#if COLLECT_STATS
  ++__bounds_stats_ptr_true;
#endif

#if DEBUG_FEATURES
  if (__bounds_debug_no_checking) return pointer != 0;
  if (__bounds_debug_print_calls)
    printf ("__bounds_check_ptr_true(p=%p, file=\"%s\", ln=%d)\n",
			pointer, filename, line);
#endif

  check_true_or_false (pointer, filename, line);

  return pointer != 0;
}

bool
__bounds_check_ptr_false (void *pointer, char *filename, int line)
{
#if COLLECT_STATS
  ++__bounds_stats_ptr_false;
#endif

#if DEBUG_FEATURES
  if (__bounds_debug_no_checking) return pointer == 0;
  if (__bounds_debug_print_calls)
    printf ("__bounds_check_ptr_false(p=%p, file=\"%s\", ln=%d)\n",
			pointer, filename, line);
#endif

  check_true_or_false (pointer, filename, line);

  return pointer == 0;
}

/*----------------------------------------------------------------------
 *	__bounds_check_ptr_OP_ptr, where 'OP' is le, lt, ge, gt are
 *	generated by cc1 when we meet comparison operations <=, <, >=, >
 *	respectively.
 *----------------------------------------------------------------------*/

static inline int
check_inequality (int result,
		  void *pointer1, void *pointer2, char *filename, int line)
{
  object *obj1, *obj2;

  /* Doing an inequality operation against a NULL pointer is a strange thing
   * to want to do. Against an ILLEGAL pointer, it's definitely wrong.
   */
  if (pointer1 == NULL || pointer2 == NULL ||
      pointer1 == ILLEGAL || pointer2 == ILLEGAL) {
   if (pointer1 == NULL && pointer2 == NULL)
     return result;
   __bounds_error2 ("NULL or ILLEGAL pointer used in <, >, <= or >= of pointers",
		    filename, line, pointer1, pointer2, NULL, NULL);
   return result;
  }
  if ((obj1 = __bounds_find_object (pointer1)) == NULL
      || (obj2 = __bounds_find_object (pointer2)) == NULL) {
    if (maybe_is_unchecked_static (pointer1, filename, line, NULL) &&
	maybe_is_unchecked_static (pointer2, filename, line, NULL)) {
      /* Two unchecked static objects. Allow the arithmetic to go ahead.
       */
      return result;
    }
    if (maybe_is_unchecked_stack (pointer1, filename, line, NULL) &&
	maybe_is_unchecked_stack (pointer2, filename, line, NULL)) {
      /* Two unchecked stack objects. Allow the arithmetic to go ahead.
       */
      return result;
    }
    __bounds_error2 ("illegal pointer used in <, >, <= or >= of pointers",
		     filename, line, pointer1, pointer2, NULL, NULL);
    return result;
  }

  /* Check that the objects pointed to by the pointers are the same. Since
   * object structures are unique, checking obj1 == obj2 is sufficient.
   */
  if (obj1 != obj2) {
    __bounds_error2 ("in <, >, <= or >= of pointers, pointers point to different objects",
		     filename, line, pointer1, pointer2, obj1, obj2);
    return result;
  }

  /* Return the result here. */
  return result;
}

bool
__bounds_check_ptr_le_ptr (void *pointer1, void *pointer2,
			   char *filename, int line)
{
  int result = pointer1 <= pointer2;

#if COLLECT_STATS
  ++__bounds_stats_ptr_le_ptr;
#endif

#if DEBUG_FEATURES
  if (__bounds_debug_no_checking) return result;
  if (__bounds_debug_print_calls) {
    printf ("__bounds_check_ptr_le_ptr(p1=%p, p2=%p, file=\"%s\", ln=%d)\n",
	    pointer1, pointer2, filename, line);
  }
#endif

  return check_inequality (result, pointer1, pointer2, filename, line);
}

bool
__bounds_check_ptr_lt_ptr (void *pointer1, void *pointer2,
			   char *filename, int line)
{
  int result = pointer1 < pointer2;

#if COLLECT_STATS
  ++__bounds_stats_ptr_lt_ptr;
#endif

#if DEBUG_FEATURES
  if (__bounds_debug_no_checking) return result;
  if (__bounds_debug_print_calls) {
    printf ("__bounds_check_ptr_lt_ptr(p1=%p, p2=%p, file=\"%s\", ln=%d)\n",
	    pointer1, pointer2, filename, line);
  }
#endif

  return check_inequality (result, pointer1, pointer2, filename, line);
}

bool
__bounds_check_ptr_ge_ptr (void *pointer1, void *pointer2,
			   char *filename, int line)
{
  int result = pointer1 >= pointer2;

#if COLLECT_STATS
  ++__bounds_stats_ptr_ge_ptr;
#endif

#if DEBUG_FEATURES
  if (__bounds_debug_no_checking) return result;
  if (__bounds_debug_print_calls) {
    printf ("__bounds_check_ptr_ge_ptr(p1=%p, p2=%p, file=\"%s\", ln=%d)\n",
	    pointer1, pointer2, filename, line);
  }
#endif

  return check_inequality (result, pointer1, pointer2, filename, line);
}

bool
__bounds_check_ptr_gt_ptr (void *pointer1, void *pointer2,
			   char *filename, int line)
{
  int result = pointer1 > pointer2;

#if COLLECT_STATS
  ++__bounds_stats_ptr_gt_ptr;
#endif

#if DEBUG_FEATURES
  if (__bounds_debug_no_checking) return result;
  if (__bounds_debug_print_calls) {
    printf ("__bounds_check_ptr_gt_ptr(p1=%p, p2=%p, file=\"%s\", ln=%d)\n",
	    pointer1, pointer2, filename, line);
  }
#endif

  return check_inequality (result, pointer1, pointer2, filename, line);
}

/*----------------------------------------------------------------------
 *	__bounds_check_ptr_ne_ptr and __bounds_check_ptr_eq_ptr are
 * 	generated by cc1 when we get pointer != pointer or pointer ==
 *	pointer respectively. We check that the pointers used aren't
 *	ILLEGAL.
 *	Note (5/3/95): Stale pointers used in == and != are OK, because
 *	of the following piece of code:
 *	  oldPtr = ptr;
 *	  ptr = realloc (ptr, n);
 *	  if (ptr != oldPtr) {
 *	    ** execute code if the realloc'd memory moved. **
 *	    ...
 *	  }
 *	Note (31/5/95): NULL pointers are OK.
 *----------------------------------------------------------------------*/

static inline int
check_equality (int result,
		void *pointer1, void *pointer2,
		char *filename, int line)
{
#if 0
  object *obj1 = NULL, *obj2 = NULL;
#endif

  if (pointer1 == ILLEGAL || pointer2 == ILLEGAL) {
    __bounds_error2 ("ILLEGAL pointer used in == or != comparison of pointers",
		     filename, line, pointer1, pointer2, NULL, NULL);
    return result;
  }
#if 0 /* (See note 5/3/95 above) */
  /* Make sure we know about each pointer, or the pointer points to an
   * unchecked object.
   */
  if ((pointer1 == NULL
       || (obj1 = __bounds_find_object (pointer1)) != NULL
       || maybe_is_unchecked_static (pointer1, filename, line, NULL)
       || maybe_is_unchecked_stack (pointer1, filename, line, NULL))
      &&
      (pointer2 == NULL
       || (obj2 = __bounds_find_object (pointer2)) != NULL
       || maybe_is_unchecked_static (pointer2, filename, line, NULL)
       || maybe_is_unchecked_stack (pointer2, filename, line, NULL)))
    {
      return result;	/* OK => go ahead. */
    }
  else
    {
      /* One of the pointers is invalid. */
      __bounds_error2 ("invalid pointer used in == or != comparison of pointers",
		       filename, line, pointer1, pointer2, obj1, obj2);
      return result;
    }
#else /* 1 */
  return result;
#endif
}

bool
__bounds_check_ptr_ne_ptr (void *pointer1, void *pointer2,
			   char *filename, int line)
{
  int result = pointer1 != pointer2;

#if COLLECT_STATS
  ++__bounds_stats_ptr_ne_ptr;
#endif

#if DEBUG_FEATURES
  if (__bounds_debug_no_checking) return result;
  if (__bounds_debug_print_calls) {
    printf ("__bounds_check_ptr_ne_ptr(p1=%p, p2=%p, file=\"%s\", ln=%d)\n",
	    pointer1, pointer2, filename, line);
  }
#endif

  return check_equality (result, pointer1, pointer2, filename, line);
}

bool
__bounds_check_ptr_eq_ptr (void *pointer1, void *pointer2,
			   char *filename, int line)
{
  int result = pointer1 == pointer2;

#if COLLECT_STATS
  ++__bounds_stats_ptr_eq_ptr;
#endif

#if DEBUG_FEATURES
  if (__bounds_debug_no_checking) return result;
  if (__bounds_debug_print_calls) {
    printf ("__bounds_check_ptr_eq_ptr(p1=%p, p2=%p, file=\"%s\", ln=%d)\n",
	    pointer1, pointer2, filename, line);
  }
#endif

  return check_equality (result, pointer1, pointer2, filename, line);
}

/*----------------------------------------------------------------------
 *	Post- and pre-increment and decrement functions. These functions
 *	change the value of the pointer through the 'void **' argument.
 *	They return the value of the pointer either before or after the
 *	pointer was changed, as appropriate.
 *----------------------------------------------------------------------*/

static inline void *
check_inc_or_dec (void **ptr_to_ptr, int inc, int is_post,
		  char *operation, char *filename, int line)
{
  void *old_ptr = *ptr_to_ptr;		/* Current value of the pointer. */
  void *new_ptr;			/* New value after increment. */
  object *obj;				/* Object pointed to. */

  /* Evaluate the new pointer, assuming the operation will be able to go
   * ahead.
   */
  new_ptr = (void *) ((char *)old_ptr + inc);

  /* Check we are not using a NULL or ILLEGAL pointer in arithmetic.
   */
  if (old_ptr == NULL || old_ptr == ILLEGAL) {
    __bounds_errorf (filename, line, old_ptr, NULL,
		     "NULL or ILLEGAL pointer used in %s",
		     operation);
    goto end;
  }

  /* See if we can find what the pointer points to, or if it is an unchecked
   * pointer.
   */
  if ((obj = __bounds_find_object (old_ptr)) == NULL) {
    if (maybe_is_unchecked_static (old_ptr, filename, line, NULL)
	|| maybe_is_unchecked_stack (old_ptr, filename, line, NULL)) {
      *ptr_to_ptr = new_ptr;
      return is_post ? old_ptr : new_ptr;
    }
    __bounds_errorf (filename, line, old_ptr, NULL,
		     "invalid pointer used in %s",
		     operation);
    goto end;
  }

  /* Check the alignment of the pointer and increment size.
   */
  if (__bounds_warn_unaligned && pointer_not_aligned (old_ptr, obj))
    __bounds_errorf (filename, line, old_ptr, obj,
		     "unaligned pointer used in %s",
		     operation);
  if (__bounds_warn_unaligned &&
      (obj->align_mask
       ? (inc & (obj->align-1)) != 0
       : (inc % obj->align) != 0))
    __bounds_errorf (filename, line, old_ptr, obj,
		     "%srement size (%d bytes) not a whole multiple of size of elements (%d bytes) in %s",
		     inc >= 0 ? "inc" : "dec", abs (inc), obj->align,
		     operation);
  
  /* Check the increment or decrement will not overrun the object.
   */
  if (new_ptr < obj->base || new_ptr > obj->extent) {
    if (__bounds_warn_illegal)
      __bounds_warnf (filename, line, new_ptr, obj,
		      "created an ILLEGAL pointer in pointer %s", operation);
    new_ptr = ILLEGAL;		/* Catch this error later. */
  }

  /* Set the pointer value to the new pointer value, and return the required
   * pointer.
   */
end:
  *ptr_to_ptr = new_ptr;
  return is_post ? old_ptr : new_ptr;
}

void *
__bounds_check_ptr_postinc (void **ptr_to_ptr, int inc,
			    char *filename, int line)
{
#if COLLECT_STATS
  ++__bounds_stats_ptr_postinc;
#endif

#if DEBUG_FEATURES
  if (__bounds_debug_no_checking) {
    void *result = *ptr_to_ptr;
    ((char *) (*ptr_to_ptr)) += inc;
    return result;
  }
  if (__bounds_debug_print_calls) {
    printf ("__bounds_check_ptr_postinc(p=%p, inc=%d, file=\"%s\", ln=%d)\n",
	    *ptr_to_ptr, inc, filename, line);
  }
#endif

  return check_inc_or_dec (ptr_to_ptr, inc, 1, "pointer postincrement (p++)",
			   filename, line);
}

void *
__bounds_check_ptr_preinc (void **ptr_to_ptr, int inc,
			   char *filename, int line)
{
#if COLLECT_STATS
  ++__bounds_stats_ptr_preinc;
#endif

#if DEBUG_FEATURES
  if (__bounds_debug_no_checking) {
    return (*ptr_to_ptr) += inc;
  }
  if (__bounds_debug_print_calls) {
    printf ("__bounds_check_ptr_preinc(p=%p, inc=%d, file=\"%s\", ln=%d)\n",
	    *ptr_to_ptr, inc, filename, line);
  }
#endif

  return check_inc_or_dec (ptr_to_ptr, inc, 0, "pointer preincrement (++p)",
			   filename, line);
}

void *
__bounds_check_ptr_postdec (void **ptr_to_ptr, int inc,
			    char *filename, int line)
{
#if COLLECT_STATS
  ++__bounds_stats_ptr_postdec;
#endif

#if DEBUG_FEATURES
  if (__bounds_debug_no_checking) {
    void *result = *ptr_to_ptr;
    ((char *) (*ptr_to_ptr)) -= inc;
    return result;
  }
  if (__bounds_debug_print_calls) {
    printf ("__bounds_check_ptr_postdec(p=%p, inc=%d, file=\"%s\", ln=%d)\n",
	    *ptr_to_ptr, inc, filename, line);
  }
#endif

  return check_inc_or_dec (ptr_to_ptr, -inc, 1, "pointer postdecrement (p--)",
			   filename, line);
}

void *
__bounds_check_ptr_predec (void **ptr_to_ptr, int inc,
			   char *filename, int line)
{
#if COLLECT_STATS
  ++__bounds_stats_ptr_predec;
#endif

#if DEBUG_FEATURES
  if (__bounds_debug_no_checking) {
    return (*ptr_to_ptr) -= inc;
  }
  if (__bounds_debug_print_calls) {
    printf ("__bounds_check_ptr_predec(p=%p, inc=%d, file=\"%s\", ln=%d)\n",
	    *ptr_to_ptr, inc, filename, line);
  }
#endif

  return check_inc_or_dec (ptr_to_ptr, -inc, 0, "pointer predecrement (--p)",
			   filename, line);
}
