#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

/* control codes */

#define NV      "\033[0m" /* normal video */
#define IV      "\033[7m" /* inverse video */

static void parseCommandLine(int argc, char *argv[]);
static void setupLatin1Tokens(void);
static void setup7BitTokens(void);
static void readHeader(void);
static int processFullLine(void);
static void skipLine(void);
static void processLine(void);
static void processQuotedString(void);
static void processRemStatement(void);

typedef enum {
    DOS,
    LATIN1,
    SEVENBIT
} CharSetType;

static CharSetType charSet = LATIN1;
static int listFrom = 0;
static int listTo = 65535;
static FILE *infile = stdin;
static char *undefToken = IV"~"NV;

static char *tokens[256] = {
    /* 00 */  NULL,       NULL,     NULL,     NULL,      NULL,     NULL,
    /* 06 */  NULL,       NULL,     NULL,     NULL,      NULL,     NULL,
    /* 0c */  NULL,       NULL,     NULL,     NULL,      NULL,     NULL,
    /* 12 */  NULL,       NULL,     NULL,     NULL,      NULL,     NULL,
    /* 18 */  NULL,       NULL,     NULL,     NULL,      NULL,     NULL,
    /* 1e */  NULL,       NULL,     " ",      "!",       "\"",     "#",
    /* 24 */  "$",        "%",      "&",      "'",       "(",      ")",
    /* 2a */  "*",        "+",      ",",      "-",       ".",      "/",
    /* 30 */  "0",        "1",      "2",      "3",       "4",      "5",
    /* 36 */  "6",        "7",      "8",      "9",       ":",      ";",
    /* 3c */  "<",        "=",      ">",      "?",       "@",      "a",
    /* 42 */  "b",        "c",      "d",      "e",       "f",      "g", 
    /* 48 */  "h",        "i",      "j",      "k",       "l",      "m", 
    /* 4e */  "n",        "o",      "p",      "q",       "r",      "s", 
    /* 54 */  "t",        "u",      "v",      "w",       "x",      "y", 
    /* 5a */  "z",        "[",      "\x9c",   "]",       "\x18",   "\x1b",
    /* 60 */  "\xc4",     "A",      "B",      "C",       "D",      "E",
    /* 66 */  "F",        "G",      "H",      "I",       "J",      "K",
    /* 6c */  "L",        "M",      "N",      "O",       "P",      "Q",
    /* 72 */  "R",        "S",      "T",      "U",       "V",      "W",
    /* 78 */  "X",        "Y",      "Z",      "\xc5",    "\xb1",   "\xb3",
    /* 7e */  "\xb1",     "\xb0",

    /* 80 */  "end",      "for",    "next",   "data",    "input#", "input",
    /* 86 */  "dim",      "read",   "let",    "goto",    "run",    "if",
    /* 8c */  "restore",  "gosub",  "return", "rem",     "stop",   "on",
    /* 92 */  "wait",     "load",   "save",   "verify",  "def",    "poke",
    /* 98 */  "print#",   "print",  "cont",   "list",    "clr",    "cmd",
    /* 9e */  "sys",      "open",   "close",  "get",     "new",    "tab(",
    /* a4 */  "to",       "fn",     "spc(",   "then",    "not",    "step",
    /* aa */  "+",        "-",      "*",      "/",       "\x18",   "and",
    /* b0 */  "or",       ">",      "=",      "<",       "sgn",    "int",
    /* b6 */  "abs",      "usr",    "fre",    "pos",     "sqr",    "rnd",
    /* bc */  "log",      "exp",    "cos",    "sin",     "tan",    "atn",
    /* c2 */  "peek",     "len",    "str$",   "val",     "asc",    "chr$",
    /* c8 */  "left$",    "right$", "mid$",   "go",      "concat", "dopen",
    /* ce */  "dclose",   "record", "header", "collect", "backup", "copy",
    /* d4 */  "append",   "dsave",  "dload",  "catalog", "rename", "scratch",
    /* da */  "directory"
};

static char *quotedTokens[256] = {
    /* 00 */  IV"@"NV,    IV"a"NV,    IV"b"NV,    IV"c"NV,
    /* 04 */  IV"d"NV,    IV"e"NV,    IV"f"NV,    IV"g"NV,
    /* 08 */  IV"h"NV,    IV"i"NV,    IV"j"NV,    IV"k"NV,
    /* 0c */  IV"l"NV,    IV"m"NV,    IV"n"NV,    IV"o"NV,
    /* 10 */  IV"p"NV,    IV"q"NV,    IV"r"NV,    IV"s"NV,
    /* 14 */  IV"t"NV,    IV"u"NV,    IV"v"NV,    IV"w"NV,
    /* 18 */  IV"x"NV,    IV"y"NV,    IV"z"NV,    IV"["NV,
    /* 1c */  IV"\x9c"NV, IV"]"NV,    IV"\x18"NV, IV"\x1b"NV,
    /* 20 */  " ",        "!",        "\"",       "#",
    /* 24 */  "$",        "%",        "&",        "'",
    /* 28 */  "(",        ")",        "*",        "+",
    /* 2c */  ",",        "-",        ".",        "/",
    /* 30 */  "0",        "1",        "2",        "3",
    /* 34 */  "4",        "5",        "6",        "7",
    /* 38 */  "8",        "9",        ":",        ";",
    /* 3c */  "<",        "=",        ">",        "?",
    /* 40 */  "@",        "a",        "b",        "c",
    /* 44 */  "d",        "e",        "f",        "g", 
    /* 48 */  "h",        "i",        "j",        "k",
    /* 4c */  "l",        "m",        "n",        "o",
    /* 50 */  "p",        "q",        "r",        "s", 
    /* 54 */  "t",        "u",        "v",        "w",
    /* 58 */  "x",        "y",        "z",        "[",
    /* 5c */  "\x9c",     "]",        "\x18",     "\x1b",
    /* 60 */  "\xc4",     "A",        "B",        "C",
    /* 64 */  "D",        "E",        "F",        "G",
    /* 68 */  "H",        "I",        "J",        "K",
    /* 6c */  "L",        "M",        "N",        "O",
    /* 70 */  "P",        "Q",        "R",        "S",
    /* 74 */  "T",        "U",        "V",        "W",
    /* 78 */  "X",        "Y",        "Z",        "\xc5",
    /* 7c */  "\xb1",     "\xb3",     "\xb1",     "\xb0",
    /* 80 */  IV"\xc4"NV, IV"A"NV,    IV"B"NV,    IV"C"NV,
    /* 84 */  IV"D"NV,    IV"E"NV,    IV"F"NV,    IV"G"NV,
    /* 88 */  IV"H"NV,    IV"I"NV,    IV"J"NV,    IV"K"NV,
    /* 8c */  IV"L"NV,    IV"M"NV,    IV"N"NV,    IV"O"NV,
    /* 90 */  IV"P"NV,    IV"Q"NV,    IV"R"NV,    IV"S"NV,
    /* 94 */  IV"T"NV,    IV"U"NV,    IV"V"NV,    IV"W"NV,
    /* 98 */  IV"X"NV,    IV"Y"NV,    IV"Z"NV,    IV"\xc5"NV,
    /* 9c */  "\xb1",     IV"\xb3"NV, IV"\xb1"NV, "\xb0",
    /* a0 */  " ",        "\xdd",     "\xdc",     "\xc4",
    /* a4 */  "\xc4",     "\xb3",     IV"\xb1"NV, "\xb3",
    /* a8 */  "\xb1",     "\xb0",     "\xb3",     "\xc3",
    /* ac */  "\xdc",     "\xc0",     "\xbf",     "\xc4",
    /* b0 */  "\xda",     "\xc1",     "\xc2",     "\xb4",
    /* b4 */  "\xb3",     "\xdd",     "\xde",     "\xc4",
    /* b8 */  "\xdf",     "\xdc",     "\xfb",     "\xdc",
    /* bc */  "\xdf",     "\xd9",     "\xdf",     "\xb2",
    /* c0 */  "\xc4",     "A",        "B",        "C",
    /* c4 */  "D",        "E",        "F",        "G", 
    /* c8 */  "H",        "I",        "J",        "K",
    /* cc */  "L",        "M",        "N",        "O",
    /* d0 */  "P",        "Q",        "R",        "S", 
    /* d4 */  "T",        "U",        "V",        "W",
    /* d8 */  "X",        "Y",        "Z",        "\xc5",
    /* dc */  "\xb1",     "\xb3",     "\xb1",     "\xb0",
    /* e0 */  " ",        "\xdd",     "\xdc",     "\xc4",
    /* e4 */  "\xc4",     "\xb3",     IV"\xb1"NV, "\xb3",
    /* e8 */  "\xb1",     "\xb0",     "\xb3",     "\xc3",
    /* ec */  "\xb1",     "\xb3",     "\xb1",     "\xb0",
    /* f0 */  " ",        "\xdd",     "\xdc",     "\xc4",
    /* f4 */  "\xb3",     "\xdd",     "\xde",     "\xc4",
    /* f8 */  "\xdf",     "\xdc",     "\xfb",     "\xdc",
    /* fc */  "\xdf",     "\xd9",     "\xdf",     "\xb1"
};

/****************************************************************************/

int
main(int argc, char *argv[]) {
    parseCommandLine(argc, argv);

    switch (charSet) {
        case LATIN1:
            setupLatin1Tokens();
            break;
        case SEVENBIT:
            setup7BitTokens();
            break;
        default:
            /* Already set up for DOS. */
            break;
    }
    
    readHeader();

    while (processFullLine())
        ;

    if (infile != stdin) fclose(infile);

    return 0;
}

/****************************************************************************/

static void
checkForHelp(int argc, char *argv[]) {
    if ( argc > 1 && ( strncmp(argv[1], "-h", 2) == 0 ||
                       strcmp(argv[1], "-?") == 0 ) ) {
        printf("Usage: %s [-charset dos|latin1|7bit] [#|[#]-[#]] "
               "[<infile>]\n\n", argv[0]);
        printf("Lists the source code of Commodore 64 BASIC programs.\n");
        printf("Written by Keith Pomakis in May, 1997.\n\n");
        printf(
"Options are:\n\n"
"    -help\n"
"        Displays this help information and exits.\n\n"
"    -charset dos|latin1|7bit\n"
"       Outputs in the specified character set, where:\n"
"           dos    = 8-bit DOS characters with VT100 inverse video\n"
"           latin1 = 8-bit ISO8859-1 characters with VT100 inverse video\n"
"           7bit   = standard 7-bit ASCII characters\n\n"
"    #|[#]-[#]\n"
"       Lists only the specified range of line numbers.\n"
"       e.g., \"%s 100\"     lists only line 100,\n"
"             \"%s -100\"    lists up to and including line 100,\n"
"             \"%s 100-\"    lists from line 100 onwards, and\n"
"             \"%s 100-200\" lists from line 100 to line 200 inclusive\n\n"
"Notes:\n\n"
"    - All command-line options can be abbreviated.\n\n"
"    - If no filename is specified, input is taken from stdin.\n\n"
"    - If you need to specify a filename that looks like a range, then\n"
"      specify a range of \"-\".  e.g., \"%s - 10-20\" lists the file\n"
"      \"10-20\".  Basically, only the first thing on the command line that\n"
"      looks like a range is taken as the range.\n",
            argv[0], argv[0], argv[0], argv[0], argv[0]);

        exit(0);
    }
}

static int
isRange(const char *s) {
    int i;
    int numOfDashes = 0;

    for (i=0; s[i] != 0; i++) {
        if (!isdigit(s[i]) && s[i] != '-')
            return 0;
        if (s[i] == '-')
            if (++numOfDashes > 1)
                return 0;
    }

    return 1;
}

static void
setRange(const char *s) {
    char *dashPtr = strchr(s, '-');

    if (dashPtr == NULL)
        listFrom = listTo = atoi(s);
    else if (dashPtr == s) {
        if (*(s+1))
            listTo = atoi(s+1);
    }
    else if (*(dashPtr+1) == '\0')
        listFrom = atoi(s);
    else {
        listFrom = atoi(s);
        listTo = atoi(dashPtr+1);
    }
}

static int
setCharSet(const char *s) {
    int len = strlen(s);

    if (strncmp(s, "7bit", len) == 0) {
        charSet = SEVENBIT;
        return 1;
    }
    else if (strncmp(s, "latin1", len) == 0) {
        charSet = LATIN1;
        return 1;
    }
    else if (strncmp(s, "dos", len) == 0) {
        charSet = DOS;
        return 1;
    }
    else
        return 0;
}

static void
parseCommandLine(int argc, char *argv[]) {
    int i = 1;
    int isRangeSet = 0;
    int isCharSetSet = 0;
    int isFileNameSet = 0;

    checkForHelp(argc, argv);

    while (i < argc) {
        int len = strlen(argv[i]);

        if (!isRangeSet && isRange(argv[i])) {
            setRange(argv[i]);
            isRangeSet = 1;
        }

        else if (len > 1 && !isCharSetSet &&
                 strncmp(argv[i], "-charset", len) == 0) {
            if (++i == argc || !setCharSet(argv[i])) {
                fprintf(stderr, "%s: one of 7bit, latin1 or dos must "
                        "follow -charset option\n", argv[0]);
                exit(1);
            }
            isCharSetSet = 1;
        }

        else {
            if (isFileNameSet) {
                fprintf(stderr, "%s: too many filenames\n", argv[0]);
                fclose(infile);
                exit(1);
            }

            infile = fopen(argv[i], "r");
            if (!infile) {
                fprintf(stderr, "%s: error opening file %s\n", argv[0],
                        argv[i]);
                exit(1);
            }

            isFileNameSet = 1;
        }

        i++;
    }
}

/****************************************************************************/

static void
setupLatin1Tokens(void) {
    /* Assumes that the tokens are currently set up for DOS. */

    int i;

    tokens[0x5c] = "\xa3";
    tokens[0x5e] = "^";
    tokens[0x5f] = NULL;
    tokens[0x60] = NULL;

    for (i=0x7b; i<0x80; i++)
        tokens[i] = NULL;

    tokens[0xae] = "^";

    quotedTokens[0x1c] = IV"\xa3"NV;
    quotedTokens[0x1e] = IV"^"NV;
    quotedTokens[0x1f] = NULL;
    quotedTokens[0x5c] = "\xa3";
    quotedTokens[0x5e] = "^";
    quotedTokens[0x5f] = NULL;
    quotedTokens[0x60] = NULL;

    for (i=0x7b; i<0x81; i++)
        quotedTokens[i] = NULL;

    for (i=0x9b; i<0xa0; i++)
        quotedTokens[i] = NULL;

    quotedTokens[0xa1] = NULL;
    quotedTokens[0xa2] = NULL;
    quotedTokens[0xa3] = "\xaf";
    quotedTokens[0xa4] = "\x5f";

    for (i=0xa5; i<0xc1; i++)
        quotedTokens[i] = NULL;

    for (i=0xdb; i<0xe0; i++)
        quotedTokens[i] = NULL;

    quotedTokens[0xe1] = NULL;
    quotedTokens[0xe2] = NULL;
    quotedTokens[0xe3] = "\xaf";
    quotedTokens[0xe4] = "\x5f";

    for (i=0xe5; i<0x100; i++)
        quotedTokens[i] = NULL;
}

static void
setup7BitTokens(void) {
    /* Assumes that the tokens are currently set up for DOS. */

    int i;

    setupLatin1Tokens();

    undefToken = "~";
    tokens[0x5c] = NULL;

    for (i=0x00; i<0x20; i++)
        quotedTokens[i] = NULL;

    quotedTokens[0x5c] = NULL;

    for (i=0x81; i<0xa0; i++)
        quotedTokens[i] = NULL;

    quotedTokens[0xa3] = NULL;
    quotedTokens[0xe3] = NULL;
}

/****************************************************************************/

static void
readHeader(void) {
    if (getc(infile) != 0x01 || getc(infile) != 0x08) {
        fprintf(stderr, "Not a Commodore 64 BASIC file.\n");
        exit(1);
    }
}

static int
processFullLine(void) {
    int lobyte, hibyte;

    lobyte = getc(infile);
    hibyte = getc(infile);

    if (lobyte == 0 && hibyte == 0)
        return 0;

    lobyte = getc(infile);
    hibyte = getc(infile);

    if (hibyte == EOF)
        return 0;
    else {
        int lineNumber = hibyte*256 + lobyte;

        if (lineNumber > listTo)
            return 0;
        else {
            if (lineNumber >= listFrom) {
                printf("%d ", lineNumber);
                processLine();
            }
            else
                skipLine();
            return 1;
        }
    }
}

static void
skipLine(void) {
    int byte;

    byte = getc(infile);
    while (byte != 0 && byte != EOF) {
        byte = getc(infile);
    }

    if (byte == EOF) {
        printf("Premature EOF\n");
        exit(1);
    }
}

static void
processLine(void) {
    int byte;

    byte = getc(infile);
    while (byte != 0 && byte != EOF) {
        if (byte == '\"')
            processQuotedString();
        else if (byte == 0x8f) {
                processRemStatement();
                return;
        }
        else {
            char *token = tokens[byte];
            printf("%s", token? token : undefToken);
        }

        byte = getc(infile);
    }

    if (byte == EOF) {
        printf("Premature EOF\n");
        exit(1);
    }

    putchar('\n');
}

static void
processQuotedString(void) {
    int byte;

    putchar('\"');

    byte = getc(infile);
    while (byte != '\"' && byte != 0 && byte != EOF) {
        char *token = quotedTokens[byte];
        printf("%s", token? token : undefToken);
        byte = getc(infile);
    }

    if (byte == EOF) {
        printf("Premature EOF\n");
        exit(1);
    }

    if (byte == '\"')
        putchar('\"');
    else
        ungetc(byte, infile);
}

static void
processRemStatement(void) {
    int byte;

    printf("%s", tokens[0x8f]);

    byte = getc(infile);
    while (byte != 0 && byte != EOF) {
        char *token = quotedTokens[byte];
        printf("%s", token? token : undefToken);
        byte = getc(infile);
    }

    if (byte == EOF) {
        printf("Premature EOF\n");
        exit(1);
    }

    putchar('\n');
}
