#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

static void parseCommandLine(int argc, char *argv[]);
static int getBlockNum(int track, int sector);
static void bufferDisk(void);
static void loadBlock(int track, int sector);
static void dumpBlock(void);
static void dumpDirectory(void);
static void dumpFile(void);
static void *emalloc(size_t numOfBytes);

typedef enum {
    NONE,
    BLOCK,
    DIRECTORY,
    LOAD
} CommandType;

typedef unsigned char BYTE;

typedef struct {
    BYTE bytes[256];
} DiskBlock;

typedef struct {
    BYTE nextTrack;  /* Only valid for first entry of block. */
    BYTE nextSector;
    BYTE type;
    BYTE firstTrack;
    BYTE firstSector;
    BYTE name[16];
    BYTE firstRelTrack;
    BYTE firstRelSector;
    BYTE relSize;
    BYTE unused[4];
    BYTE firstRelSideTrack;
    BYTE firstRelSideSector;
    BYTE sizeLow;
    BYTE sizeHigh;
} FileEntry;

typedef struct {
    /* First two bytes of first fileEntry represent */
    /* pointer to next track and sector. */
    FileEntry fileEntries[8];
} DirectoryBlock;

typedef struct {
    BYTE sectorsFree;
    BYTE sectorsFreeBitmap[3];
} BamEntry;

typedef struct {
    BYTE firstDirTrack;
    BYTE firstDirSector;
    BYTE dosVersion;
    BYTE unused1;
    BamEntry bamEntries[35];
    BYTE diskName[16];
    BYTE unused2[2];
    BYTE diskID[2];
    BYTE unused3;
    BYTE dosType[2];
    BYTE unused4[89];
} BamBlock;

typedef struct {
    BYTE type;
    BYTE firstTrack;
    BYTE firstSector;
    char name[17];
    int size;
} FileInfo;

typedef struct {
    BYTE nextTrack;
    BYTE nextSector;
    BYTE data[254];
} DataBlock;

static char *progName;
static CommandType command = NONE;
static char *fileToLoad = NULL;
static int track, sector;
static FILE *infile = stdin;
static int numOfFiles = 0;
static FileInfo *directory = NULL;
static int bufferedBlocks = 0;
static DiskBlock *blockBuffer = NULL;
static DiskBlock *singleBlock = NULL;
static DiskBlock *thisBlock = NULL;

static char tokens[256] =
    "~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ !\"#$%&'()*+,-./0123456789:;<=>?"
    "@abcdefghijklmnopqrstuvwxyz[~]^~~ABCDEFGHIJKLMNOPQRSTUVWXYZ~~~~~"
    "~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ ~~~_~~~~~~~~~~~~~~~~~~~~~~~~~~~"
    "~ABCDEFGHIJKLMNOPQRSTUVWXYZ~~~~~ ~~~_~~~~~~~~~~~~~~~~~~~~~~~~~~~";

/****************************************************************************/

int
main(int argc, char *argv[]) {
    progName = argv[0];
    parseCommandLine(argc, argv);

    if (infile == stdin)
        bufferDisk();
    else
        singleBlock = (DiskBlock *) emalloc(sizeof(DiskBlock));

    /* Do everything here... */
    switch (command) {
        case BLOCK:
            dumpBlock();
            break;
        case DIRECTORY:
            dumpDirectory();
            break;
        case LOAD:
            dumpFile();
            break;
        default:
            fprintf(stderr, "%s: must specify one of -block, -directory or "
                    "-load\n", progName);
            exit(1);
            break;
    }

    if (infile != stdin) fclose(infile);

    return 0;
}

/****************************************************************************/

static void
checkForHelp(int argc, char *argv[]) {
    if ( argc > 1 && ( strncmp(argv[1], "-h", 2) == 0 ||
                       strcmp(argv[1], "-?") == 0 ) ) {
        printf("Usage: %s [-block <track>:<sector>] [-dir] [-load <filename>] "
               "[<infile>]\n\n", progName);
        printf("Extracts information from Commodore 64 D64 disk images.\n");
        printf("Written by Keith Pomakis in June, 1997.\n\n");
        printf(
"Options are mutually exclusive and are as follows:\n\n"
"    -help\n"
"        Displays this help information.\n\n"
"    -block <track>:<sector>\n"
"        Dumps the specified block of the disk image to stdout.\n\n"
"    -dir\n"
"        Dumps the directory of the disk image to stdout.\n\n"
"    -load <filename>\n"
"        Dumps the contents of the specified file to stdout.\n"
"        Standard 1541 wildcard symbols are recognized.\n\n"
"Notes:\n"
"    - All command-line options can be abbreviated.\n"
"    - If no filename is specified, input is taken from stdin.\n");

        exit(0);
    }
}

static int
setTrackAndSector(const char *ts) {
    int i = 0;
    char *colonPtr = strchr(ts, ':');

    if (colonPtr == NULL || colonPtr == ts || *(colonPtr+1) == '\0' ||
                            strchr(colonPtr+1, ':') != NULL)
        return 0;

    while (ts[i] != '\0') {
        if (!isdigit(ts[i]) && ts[i] != ':')
            return 0;
        i++;
    }

    track = atoi(ts);
    sector = atoi(colonPtr+1);

    if (getBlockNum(track, sector) < 0)
        return 0;
    else
        return 1;
}

static void
parseCommandLine(int argc, char *argv[]) {
    int i = 1;
    int isFileNameSet = 0;

    checkForHelp(argc, argv);

    while (i < argc) {
        int len = strlen(argv[i]);

        if (len > 1 && command == NONE &&
                 strncmp(argv[i], "-block", len) == 0) {
            if (++i == argc || !setTrackAndSector(argv[i])) {
                fprintf(stderr, "%s: a valid <track>:<sector> must "
                        "follow -block option\n", progName);
                exit(1);
            }
            else
                command = BLOCK;
        }

        else if (len > 1 && command == NONE &&
                 strncmp(argv[i], "-load", len) == 0) {
            if (++i == argc) {
                fprintf(stderr, "%s: a filename must follow -load option\n",
                        progName);
                exit(1);
            }
            else {
                command = LOAD;
                fileToLoad = argv[i];
            }
        }

        else if (len > 1 && command == NONE &&
                 strncmp(argv[i], "-directory", len) == 0) {
            command = DIRECTORY;
        }

        else {
            if (isFileNameSet) {
                fprintf(stderr, "%s: too many filenames\n", progName);
                fclose(infile);
                exit(1);
            }

            infile = fopen(argv[i], "r");
            if (!infile) {
                fprintf(stderr, "%s: error opening file %s\n", progName,
                        argv[i]);
                exit(1);
            }

            isFileNameSet = 1;
        }

        i++;
    }
}

/****************************************************************************/

static int
getBlockNum(int track, int sector) {
    if (track < 1 || track > 35 || sector < 0)
        return -1;
    else if (track > 30) {
        if (sector > 16)
            return -1;
        else
            return 598 + (track-31)*17 + sector;
    }
    else if (track > 24) {
        if (sector > 17)
            return -1;
        else
            return 490 + (track-25)*18 + sector;
    }
    else if (track > 17) {
        if (sector > 18)
            return -1;
        else
            return 357 + (track-18)*19 + sector;
    }
    else {
        if (sector > 20)
            return -1;
        else
            return (track-1)*21 + sector;
    }
}

static void
bufferDisk(void) {
    blockBuffer = (DiskBlock *) emalloc(683 * sizeof(DiskBlock));
    bufferedBlocks = fread(blockBuffer, sizeof(DiskBlock), 683, infile);
}

static void
loadBlock(int track, int sector) {
    int blockNum = getBlockNum(track, sector);

    if (blockNum < 0) {
        fprintf(stderr, "%s: 66, illegal track or sector,%d,%d\n",
                progName, track, sector);
        exit(1);
    }
    else if (infile == stdin && blockNum < bufferedBlocks)
        thisBlock = &blockBuffer[blockNum];
    else if (infile == stdin &&
             fread(&blockBuffer[bufferedBlocks], sizeof(DiskBlock),
                   blockNum-bufferedBlocks, infile) ==
                   blockNum-bufferedBlocks) {
        bufferedBlocks=blockNum + 1;
        thisBlock = &blockBuffer[blockNum];
    }
    else if (infile != stdin &&
             fseek(infile, blockNum*sizeof(DiskBlock), SEEK_SET) == 0 &&
             fread(singleBlock, sizeof(DiskBlock), 1, infile) == 1)
        thisBlock = singleBlock;
    else {
        fprintf(stderr, "%s: error reading track %d, sector %d - "
                "incomplete disk image\n", progName, track, sector);
        exit(1);
    }
}

static void
addFiletoDirectory(FileEntry *fileEntry)
{
    int i;
    FileInfo *fileInfo = &directory[numOfFiles++];

    fileInfo->type = fileEntry->type;
    fileInfo->firstTrack = fileEntry->firstTrack;
    fileInfo->firstSector = fileEntry->firstSector;
    fileInfo->size = fileEntry->sizeHigh*256 + fileEntry->sizeLow;

    for (i=0; i<16 && fileEntry->name[i] != 0xA0; i++)
        fileInfo->name[i] = tokens[fileEntry->name[i]];
    fileInfo->name[i] = '\0';
}

static void
loadDirectory(void)
{
    int i;
    int track = 18, sector = 1;

    directory = (FileInfo *) emalloc(sizeof(FileEntry)*144);

    while (track > 0 && numOfFiles < 144) {
        loadBlock(track, sector);

        for (i=0; i<8; i++) {
            FileEntry *fileEntry =
                &((DirectoryBlock *) thisBlock)->fileEntries[i];

            if (fileEntry->type != 0)
                addFiletoDirectory(fileEntry);
        }

        track = ((DirectoryBlock *) thisBlock)->fileEntries[0].nextTrack;
        sector = ((DirectoryBlock *) thisBlock)->fileEntries[0].nextSector;
    }
}

static const char *
typeString(int type)
{
    switch(type) {
        case 0x01: return "*seq";
        case 0x02: return "*prg";
        case 0x03: return "*usr";
        case 0x04: return "*rel";
        case 0xa0:
        case 0x80: return " del";
        case 0xa1:
        case 0x81: return " seq";
        case 0xa2:
        case 0x82: return " prg";
        case 0xa3:
        case 0x83: return " usr";
        case 0xa4:
        case 0x84: return " rel<";
        case 0xc0: return " del<";
        case 0xc1: return " seq<";
        case 0xc2: return " prg<";
        case 0xc3: return " usr<";
        case 0xc4: return " rel<";
        default:   return " ???";
    }
}

static int
matchFileName(const char *filename, const char *pattern)
{
    int i;

    for (i=0; filename[i] != '\0' && pattern[i] != '\0'; i++)
        if (pattern[i] == '*')
            return 1;
        else if (pattern[i] != '?' && filename[i] != pattern[i])
            return 0;

    if (filename[i] == '\0' && (pattern[i] == '\0' || pattern[i] == '*'))
        return 1;
    else
        return 0;
}

/****************************************************************************/

static void
dumpBlock(void)
{
    int i;
    loadBlock(track, sector);

    for (i=0; i<sizeof(DiskBlock); i++)
        putchar(thisBlock->bytes[i]);
}

static void
dumpDirectory(void)
{
    int i, blocksFree = 0;
    BamBlock *bamBlock;
    const char *spaces = "                ";

    loadBlock(18, 0);
    bamBlock = (BamBlock *) thisBlock;
    printf("0 \"");

    for (i=0; i<16 && bamBlock->diskName[i] != 0xA0; i++)
        putchar(tokens[bamBlock->diskName[i]]);
    printf("%s\" %c%c %c%c\n", &spaces[i-1],
          tokens[bamBlock->diskID[0]], tokens[bamBlock->diskID[1]],
          tokens[bamBlock->dosType[0]], tokens[bamBlock->dosType[1]]);

    for (i=0; i<35; i++)
        blocksFree += bamBlock->bamEntries[i].sectorsFree;
    blocksFree -= bamBlock->bamEntries[17].sectorsFree;

    loadDirectory();
    for (i=0; i<numOfFiles; i++)
        printf("%-5d \"%s\"%s%s\n", directory[i].size, directory[i].name,
                        &spaces[strlen(directory[i].name)],
                        typeString(directory[i].type));

    printf("%d blocks free.\n", blocksFree);
}

static void
dumpFile(void)
{
    int i;

    if (strcmp(fileToLoad, "$") == 0) {
        dumpDirectory();
        return;
    }

    loadDirectory();
    for (i=0; i<numOfFiles; i++)
        if (matchFileName(directory[i].name, fileToLoad))
            break;

    if (i == numOfFiles) {
        fprintf(stderr, "%s: 62, file not found,00,00\n", progName);
        exit(1);
    }
    else if (directory[i].size > 0) {
        int track = directory[i].firstTrack;
        int sector = directory[i].firstSector;

        while (track > 0) {
            DataBlock *dataBlock;
            int bytes;
            loadBlock(track, sector);
            dataBlock = (DataBlock *) thisBlock;
            bytes = dataBlock->nextTrack? 254 : dataBlock->nextSector-1;
            if (bytes >= 0)
                fwrite(&dataBlock->data, 1, bytes, stdout);
            track = dataBlock->nextTrack;
            sector = dataBlock->nextSector;
        };
    }
}

/****************************************************************************/

static void *
emalloc(size_t numOfBytes) {
    void *result = malloc(numOfBytes);
    if (result == NULL) {
        fprintf(stderr, "%s: error allocating memory\n", progName);
        exit(1);
    }
    else
        return result;
}
