/*
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version
 *	2 of the License, or (at your option) any later version.
 *
 *  Copyright 1999 Michael Klein <michael.klein@puffin.lb.shuttle.de>
*/

#include "cbm.h"

#include <ctype.h>
#include <errno.h>
#include <error.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <fcntl.h>

typedef int (*mainfunc)(int fd, char *argv[]);

static int do_reset(int fd, char *argv[])
{
    return cbm_reset(fd);
}

static int do_listen(int fd, char *argv[])
{
    return cbm_listen(fd, atoi(argv[0]), atoi(argv[1]));
}

static int do_talk(int fd, char *argv[])
{
    return cbm_talk(fd, atoi(argv[0]), atoi(argv[1]));
}

static int do_unlisten(int fd, char *argv[])
{
    return cbm_unlisten(fd);
}

static int do_untalk(int fd, char *argv[])
{
    return cbm_untalk(fd);
}

static int do_open(int fd, char *argv[])
{
    int rv;
    
    rv = cbm_open(fd, atoi(argv[0]), atoi(argv[1]));
    write(fd, argv[2], strlen(argv[2]));
    cbm_unlisten(fd);
    
    return rv;
}

static int do_close(int fd, char *argv[])
{
    return cbm_close(fd, atoi(argv[0]), atoi(argv[1]));
}

static int do_status(int fd, char *argv[])
{
    char c;
    int  rv;

    rv = cbm_talk(fd, atoi(argv[0]), 15);
    if(rv == 0) {
        while(read(fd, &c, 1) == 1) {
            putchar(c == 0x0d ? 0x0a : tolower(c));
        }
        rv = cbm_untalk(fd);
    }
    return rv;
}

static int do_command(int fd, char *argv[])
{
    int  rv;

    rv = cbm_listen(fd, atoi(argv[0]), 15);
    if(rv == 0) {
        write(fd, argv[1], strlen(argv[1]));
        rv = cbm_unlisten(fd);
    }
    return rv;
}

struct prog {
    char    *name;
    mainfunc prog;
    int      req_args;
    char    *arglist;
};

static struct prog prog_table[] = {
    {"listen"  , do_listen  , 2, "<device> <secadr>"           },
    {"talk"    , do_talk    , 2, "<device> <secadr>"           },
    {"unlisten", do_unlisten, 0, ""                            },
    {"untalk"  , do_untalk  , 0, ""                            },
    {"open"    , do_open    , 3, "<device> <secadr> <filename>"},
    {"close"   , do_close   , 2, "<device> <secadr>"           },
    {"status"  , do_status  , 1, "<device>"                    },
    {"command" , do_command , 2, "<device> <cmdstr>"           },
    {"reset"   , do_reset   , 0, ""                            },
    {NULL,NULL}
};

static struct prog *find_main(char *name)
{
    int i;

    for(i=0; prog_table[i].name; i++) {
        if(strcmp(name, prog_table[i].name) == 0) {
            return &prog_table[i];
        }
    }
    return NULL;
}

int main(int argc, char *argv[])
{
    struct prog *p;
    int i;

    p = argc < 2 ? NULL : find_main(argv[1]);
    if(p) {
        if(p->req_args == argc-2) {

            int fd = open(cbm_dev, O_RDWR);
            int rv = 1;

            if(fd != -1) {
                rv = p->prog(fd, &argv[2]) != 0;
                close(fd);
            }
            if(rv) {
                error(0, errno, "%s", cbm_dev);
            }
            return rv;
        } else {
            fprintf(stderr, "wrong number of arguments:\n\n  %s %s %s\n",
                        argv[0], argv[1], p->arglist);
        }
    } else {
        fputs("invalid command, available ones are:\n\n", stderr);
        for(i=0; prog_table[i].prog; i++) {
            fprintf(stderr, "  %s %s\n", prog_table[i].name,
                                         prog_table[i].arglist);
        }
    }
    return 2;
}
