/*
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version
 *	2 of the License, or (at your option) any later version.
 *
 *  Copyright 1999 Michael Klein <michael.klein@puffin.lb.shuttle.de>
*/

#include "cbm.h"
#include "d64copy.h"

#include <errno.h>
#include <error.h>
#include <fcntl.h>
#include <getopt.h>
#include <signal.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

char sector_map[36] =
{ 0,
  21, 21, 21, 21, 21, 21, 21, 21, 21, 21,
  21, 21, 21, 21, 21, 21, 21, 19, 19, 19,
  19, 19, 19, 19, 18, 18, 18, 18, 18, 18,
  17, 17, 17, 17, 17
};

int fd_cbm, cbm_drive;

static unsigned char warp_read[] = {
#include "warpread.inc"
};

static unsigned char warp_write[] = {
#include "warpwrite.inc"
};

static unsigned char turbo_read[] = {
#include "turboread.inc"
};

static unsigned char turbo_write[] = {
#include "turbowrite.inc"
};

static int send_turbo(int fd, int drv, int write, int warp)
{
    unsigned char *prog;
    int size;

    if(write) {
        if(warp) {
            prog = warp_write;
            size = sizeof(warp_write);
        } else {
            prog = turbo_write;
            size = sizeof(turbo_write);
        }
    } else {
        if(warp) {
            prog = warp_read;
            size = sizeof(warp_read);
        } else {
            prog = turbo_read;
            size = sizeof(turbo_read);
        }
    }
    return cbm_upload(fd, drv, 0x500, prog, size) != size;
}

static int gcr_decode(unsigned char *gcr, unsigned char *decoded)
{
    static unsigned char lo[] = {
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0x08, 0x00, 0x01, 0xff, 0x0c, 0x04, 0x05,
        0xff, 0xff, 0x02, 0x03, 0xff, 0x0f, 0x06, 0x07,
        0xff, 0x09, 0x0a, 0x0b, 0xff, 0x0d, 0x0e, 0xff
    };

    static unsigned char hi[] = {
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0x80, 0x00, 0x10, 0xff, 0xc0, 0x40, 0x50,
        0xff, 0xff, 0x20, 0x30, 0xff, 0xf0, 0x60, 0x70,
        0xff, 0x90, 0xa0, 0xb0, 0xff, 0xd0, 0xe0, 0xff
    };

    unsigned char ref, chk = 0, hdr, *c = NULL;
    register unsigned char g0, g1, g2, g3, g4;
    int i;

    g0 = *(gcr++);
    g1 = *(gcr++);

    hdr = hi[(g0 >> 3) & 0x1f] |
          lo[((g0 & 0x07) << 2) | ((g1 >> 6) & 0x03)];

    if(hdr != 0x07) {
        return 4;
    }

    for(i = 0; i < BLOCKSIZE; i+=4) {
        g2 = *(gcr++);
        g3 = *(gcr++);
        g4 = *(gcr++);
        if(c) {
            chk ^= *c++ = hi[(g0 >> 3) & 0x1f] |
                          lo[((g0 & 0x07) << 2) | ((g1 >> 6) & 0x03)];
        } else {
            c = decoded;
        }
        chk ^= *c++ = hi[(g1 >> 1) & 0x1f] |
                      lo[((g1 & 0x01) << 4) | ((g2 >> 4) & 0x0f)];
        chk ^= *c++ = hi[((g2 & 0x0f) << 1) | ((g3 >> 7) & 0x01)] |
                      lo[(g3 >> 2) & 0x1f];
        chk ^= *c++ = hi[((g3 & 0x03) << 3) | ((g4 >> 5) & 0x07)] |
                      lo[g4 & 0x1f];
        g0 = *(gcr++);
        g1 = *(gcr++);
    }
    chk ^= *c = hi[(g0 >> 3) & 0x1f] |
                lo[((g0 & 0x07) << 2) | ((g1 >> 6) & 0x03)];
    g2 = *gcr;
    ref = hi[(g1 >> 1) & 0x1f] |
          lo[((g1 & 0x01) << 4) | ((g2 >> 4) & 0x0f)];

    return (chk != ref) ? 5 : 0;
}

static int gcr_encode(unsigned char *block, unsigned char *encoded)
{
    register unsigned char c = 0x07, h, l, g0, g1, g2, g3, g4;
    unsigned char chk = 0;
    int i;

    static unsigned char gcr[] = {
        0x0a, 0x0b, 0x12, 0x13, 0x0e, 0x0f, 0x16, 0x17,
        0x09, 0x19, 0x1a, 0x1b, 0x0d, 0x1d, 0x1e, 0x15
    };

    static unsigned char empty[] = {
        0x00, 0x00, 0x00, 0x00
    };

    for(i = 0; i <= BLOCKSIZE; i+=4) {
        if(i == BLOCKSIZE) {
            block    = empty;
            empty[0] = chk;
        }
        h    = gcr[c >> 4]; l = gcr[c & 0x0f];
        g0   = (h << 3) | (l >> 2);
        g1   = (l & 0x03) << 6;
        chk ^= c = *(block++);
        h    = gcr[c >> 4]; l = gcr[c & 0x0f];
        g1  |= (h << 1) | (l >> 4);
        g2   = (l & 0x0f) << 4;
        chk ^= c = *(block++);
        h    = gcr[c >> 4]; l = gcr[c & 0x0f];
        g2  |= (h >> 1);
        g3   = ((h & 0x01) << 7) | (l << 2);
        chk ^= c = *(block++);
        h    = gcr[c >> 4]; l = gcr[c & 0x0f];
        g3  |= (h >> 3);
        g4   = ((h & 0x07) << 5) | (l);

        *(encoded++) = g0;
        *(encoded++) = g1;
        *(encoded++) = g2;
        *(encoded++) = g3;
        *(encoded++) = g4;

        chk ^= c = *(block++);
    }
    return 0;
}

static int is_cbm(char *name)
{
    return((strcmp(name, "8" ) == 0) ||
           (strcmp(name, "9" ) == 0) ||
           (strcmp(name, "10") == 0) ||
           (strcmp(name, "11") == 0)   );
}

#define TM_FILESYSTEM 0
#define TM_ORIGINAL   1
#define TM_PARALLEL   2
#define TM_SERIAL1    3
#define TM_SERIAL2    4

extern struct transfer_funcs fs_transfer,
                             std_transfer,
                             pp_transfer,
                             s1_transfer,
                             s2_transfer;

static void help()
{
    printf(
"Usage: d64copy [OPTION]... [SOURCE] [TARGET]\n\
Copy .d64 disk images to a CBM-1541 or compatible drive and vice versa\n\
\n\
  -h                display this help and exit\n\
  -q                more quiet output\n\
\n\
  -e TRACK          set start track\n\
  -s TRACK          set end track\n\
                    this version supports only 35 track disks resp. images;\n\
                    so 1<=TRACK<=35 and start<=end.\n\
\n\
  -t TRANSFERMODE   set transfermode; valid modes are:\n\
                      original   (slowest)\n\
                      serial1\n\
                      serial2\n\
                      parallel   (fastest)\n\
                    `original' and `serial1' should work in any case;\n\
                    `serial2' won't work if more than one device is\n\
                    connected to the IEC bus;\n\
                    `parallel' needs a XP1541 cable\n\
\n\
  -i INTERLEAVE     set interleave value; ignored when reading with warp\n\
                    mode; default values are:\n\
\n\
                      original     16
\n\
                                turbo r/w   warp write\n\
\n\
                      serial1       3            5\n\
                      serial2      12           11\n\
                      parallel      6            3\n\
\n\
                    INTERLEAVE is ignored when reading with warp mode;\n\
                    if data transfer is very slow, increasing this value\n\
                    may help.
\n\
  -w                enable warp mode; this is not possible when\n\
                    TRANSFERMODE is set to `original'\n\
\n\
");
}

static void hint(char *s)
{
    fprintf(stderr, "Try `%s' -h for more information.\n", s);
}

static void reset(int dummy)
{
    fprintf(stderr, "\nSIGINT caught X-(  Resetting IEC bus...\n");
    sleep(1);
    cbm_reset(fd_cbm);
    close(fd_cbm);
    exit(1);
}

int main(int argc, char *argv[])
{
    unsigned char block[BLOCKSIZE];
    unsigned char gcr[GCRBUFSIZE];
    int rv = 1, transfer_mode = TM_ORIGINAL, quiet = 0, warp = 0;
    int interleave, s_tr=1, e_tr=NR_TRACKS, src_is_cbm, dst_is_cbm;
    char c, *src_arg, *dst_arg, *tm = NULL;
    struct transfer_funcs *src, *dst;
    static struct transfer_funcs *transfer[] = { &fs_transfer,
                                                 &std_transfer,
                                                 &pp_transfer,
                                                 &s1_transfer,
                                                 &s2_transfer};

    static int default_interleave[] = { 0, 16, 6, 3, 12 };
    static int warp_write_interleave[] = { 0, 0, 3, 5, 11 };

    interleave = -1;

    while((c=getopt(argc, argv, "hwqt:i:s:e:")) != -1) {
        switch(c) {
            case 'h': help();
                      return 0;
            case 'w': warp = 1;
                      break;
            case 'q': quiet = 1;
                      break;
            case 'i': interleave = atoi(optarg);
                      break;
            case 's': s_tr = atoi(optarg);
                      break;
            case 'e': e_tr = atoi(optarg);
                      break;
            case 't': tm = optarg;
                      break;
            default : hint(argv[0]);
                      return 1;
        }
    }

    if(tm) {
        if(strcmp(tm, "original") == 0) {
            transfer_mode = TM_ORIGINAL;
        } else {
            if(strcmp(tm, "parallel") == 0) {
                transfer_mode = TM_PARALLEL;
            } else {
                if(strcmp(tm, "serial1") == 0) {
                    transfer_mode = TM_SERIAL1;
                } else {
                    if(strcmp(tm, "serial2") == 0) {
                        transfer_mode = TM_SERIAL2;
                    } else {
                        fprintf(stderr, "unknown transfer mode: `%s'", tm);
                        return 1;
                    }
                }
            }
        }
    }

    if((interleave != -1) && (interleave < 1 || interleave > 17)) {
        fprintf(stderr, "invalid value (%d) for interleave\n", interleave);
        return 1;
    }

    if(s_tr < 1 || s_tr > NR_TRACKS) {
        fprintf(stderr, "invalid value (%d) for start track\n", s_tr);
        return 1;
    }

    if(e_tr < s_tr || e_tr > NR_TRACKS) {
        fprintf(stderr, "invalid value (%d) for end track\n", e_tr);
        return 1;
    }

    if(optind + 2 == argc) {
        src_arg = argv[optind];
        dst_arg = argv[optind+1];

        src_is_cbm = is_cbm(src_arg);
        dst_is_cbm = is_cbm(dst_arg);

        if(!(src_is_cbm ^ dst_is_cbm)) {
            fprintf(stderr, "either source or target must be a CBM drive\n");
            return 1;
        }

        fd_cbm = open(cbm_dev, O_RDWR);
        if(fd_cbm != -1) {

            signal(SIGINT, reset);

            if(src_is_cbm) {
                cbm_drive = atoi(src_arg);
                src = transfer[transfer_mode];
                dst = transfer[TM_FILESYSTEM];
                if(interleave == -1) {
                    interleave = default_interleave[transfer_mode];
                }
            } else {
                cbm_drive = atoi(dst_arg);
                src = transfer[TM_FILESYSTEM];
                dst = transfer[transfer_mode];
                if(interleave == -1) {
                    interleave = warp ? warp_write_interleave[transfer_mode] :
                                        default_interleave[transfer_mode];
                }
            }
            cbm_exec_command(fd_cbm, cbm_drive, "I0:", 0);
            rv = cbm_device_status(fd_cbm, cbm_drive, block, sizeof(block));
            fprintf(stderr, "drive %02d: %s\n", cbm_drive, block);

            if(rv) {
                close(fd_cbm);
                return 1;
            }

            if(warp && (transfer[transfer_mode]->read_gcr_block == NULL)) {
                fprintf(stderr, "warning: `-w' ignored\n");
                warp = 0;
            }
            if(transfer[transfer_mode]->needs_turbo) {
                send_turbo(fd_cbm, cbm_drive, dst_is_cbm, warp);
            }

            if(src->open_disk(src_arg, 0) == 0) {
                if(dst->open_disk(dst_arg, 1) == 0) {

                    int tr, se, st, cnt = 0, scnt = 0;
                    char trackmap[MAX_SECTORS+1];

                    fprintf(stderr, "copying tracks %d-%d...", s_tr, e_tr);
                    for(tr = s_tr; tr <= e_tr; tr++) {
                        scnt = sector_map[tr];
                        memset(trackmap, '-', scnt);
                        trackmap[(int)sector_map[tr]] = '\0';
                        if(!quiet) {
                            printf("\ntrack %02d: %s", tr, trackmap);
                            fflush(stdout);
                        }
                        if(warp && src_is_cbm) {
                            src->send_track_map(tr, trackmap, scnt);
                        } else {
                            se = 0;
                        }
                        while (scnt) {
                            if(warp && src_is_cbm) {
                                st = src->read_gcr_block(&se, gcr);
                                trackmap[se] = '*';
                                if(st == 0) {
                                    st = gcr_decode(gcr, block);
                                } else {
                                    src->send_track_map(tr, trackmap, scnt);
                                }
                            } else {
                                while(trackmap[se] != '-') {
                                    if(++se >= sector_map[tr]) se = 0;
                                }
                                st = src->read_block(tr, se, block);
                                trackmap[se] = '*';
                            }
                            if(st) {
                                fprintf(stderr,
                                 "read error: %02x/%02x: %d", tr, se, st);
                            }
                            if(warp && dst_is_cbm) {
                                gcr_encode(block, gcr);
                                st = dst->write_block(tr, se, gcr,
                                                      GCRBUFSIZE-1);
                            } else {
                                st = dst->write_block(tr, se, block,
                                                          BLOCKSIZE);
                            }
                            if(st) {
                                fprintf(stderr,
                                 "write error: %02x/%02x: %d", tr, se, st);
                            }
                            cnt++;
                            scnt--;
                            if(!quiet) {
                                printf("\rtrack %02d: %s", tr, trackmap);
                                fflush(stdout);
                            }
                            if(dst_is_cbm || !warp) {
                                se += interleave;
                                if(se >= sector_map[tr]) se -= sector_map[tr];
                            }
                        }
                    }
                    printf("\n%d blocks copied.\n", cnt);
                    dst->close_disk();
                } else {
                    fprintf(stderr, "can't open destination\n");
                }
                src->close_disk();
            } else {
                fprintf(stderr, "can't open source\n");
            }
            close(fd_cbm);
        } else {
            error(0, errno, "%s", cbm_dev);
        }
        return rv;
    } else {
        fprintf(stderr, "Usage: %s [OPTION]... [SOURCE] [TARGET]\n", argv[0]);
        hint(argv[0]);
        return 1;
    }
}
