/*
 * screenshot.h - Create a screenshot.
 *
 * Written by
 *  Andreas Boose <boose@linux.rz.fh-hannover.de>
 *
 * This file is part of VICE, the Versatile Commodore Emulator.
 * See README for copyright notice.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 *  02111-1307  USA.
 *
 */

#ifndef _SCREENSHOT_H
#define _SCREENSHOT_H

#include "types.h"

struct video_frame_buffer_s;
struct palette_s;
struct canvas_s;

typedef struct screenshot_s {
    struct video_frame_buffer_s *frame_buffer;
    struct palette_s *palette;
    PIXEL *pixel_table_sing;
    PIXEL *color_map;

    /* Screen size we actually want to save.  */
    unsigned int width;
    unsigned int height;

    /* Overall screen size.  */
    unsigned int max_width;
    unsigned int max_height;

    /* First and last displayed line.  */
    unsigned int first_displayed_line;
    unsigned int last_displayed_line;

    /* Offset to the overall screen.  */
    unsigned int x_offset;
    unsigned int y_offset;

    /* Pixel size.  */
    unsigned int size_width;
    unsigned int size_height;
} screenshot_t;

typedef struct screendrv_s {
    const char *name;
    const char *default_extension;
    int (*save)(struct screenshot_s *, const char *);
} screendrv_t;

#define SCREENSHOT_MODE_PALETTE 0
#define SCREENSHOT_MODE_RGB32   1

/* Functions called by external emulator code.  */
extern int screenshot_init(void);
extern int screenshot_save(const char *drvname, const char *filename,
                           unsigned int window_number);
extern int screenshot_canvas_save(const char *drvname, const char *filename,
                           struct canvas_s *canvas);

/* Functions called by screenshot modules.  */
extern int screenshot_register(screendrv_t *drv);
extern void screenshot_line_data(screenshot_t *screenshot, BYTE *data,
                                 unsigned int line, unsigned int mode);

extern int screenshot_num_drivers(void);
extern screendrv_t *screenshot_drivers_iter_init(void);
extern screendrv_t *screenshot_drivers_iter_next(void);
extern screendrv_t *screenshot_get_driver(const char *drvname);

/* Initialization prototypes */
extern void screenshot_init_bmp(void);
extern void screenshot_init_png(void);

#endif

