(********************************************************)
(* cdirectory.pas    			    1995-07-25  *)
(*                                                      *)
(* Accessing the directory for CTOOLS      		*)
(*                                                      *)
(* Copyright (C) 1995  Jochen Metzinger			*)
(*                                                      *)
(* This file is distributed WITHOUT ANY WARRANTY;	*)
(* without even the implied warranty of MERCHANTABILITY *)
(* or FITNESS FOR A PARTICULAR PURPOSE.        		*)
(********************************************************)

UNIT Cdirectory;

INTERFACE

USES Cdevice;

CONST
 REC_SIZE   =  128; (* CP/M record size *)
 REC_SECTOR =    8; (* records in a sector *)

TYPE
 dir_search =
  RECORD
   pattern: STRING[13];
   dirpos: BYTE;
   found: BOOLEAN;
   name: STRING[13];
  END; (* dir_search *)

 DIRENT =
  RECORD
   dr: BYTE;
    f: ARRAY [1..8] OF CHAR;
    t: ARRAY [1..3] OF CHAR;
   ex: BYTE;
   s1,
   s2: BYTE;
   rc: BYTE;
    d: ARRAY [0..15] OF BYTE;
  END; (* DIRENT *)

VAR
 cpm_dir: ARRAY [1..DIR_ENTRIES] OF DIRENT;
 cpm_free: ARRAY [0..NR_SECTOR-1] OF BOOLEAN;
 stat_duse, stat_fnr, stat_recnr, stat_clu: WORD;

PROCEDURE dir_load;
(* load cpm_dir *)
PROCEDURE dir_save;
(* save cpm_dir *)
PROCEDURE dir_first(fn: STRING; VAR sr: dir_search);
(* search 1st fn in cpm_dir *)
PROCEDURE dir_next(VAR sr: dir_search);
(* search next in cpm_dir *)
FUNCTION  dir_new_entry(VAR fn: STRING): BYTE;
(* get/allocate a dir entry *)
FUNCTION  dir_new_sector: BYTE;
(* get/allocate a sector *)
PROCEDURE dir_print(pat: STRING);
(* output directory *)
PROCEDURE dir_erase(fn: STRING);
(* erase files *)
PROCEDURE dir_rename(op, np: STRING);
(* rename files *)

IMPLEMENTATION

USES Cerrors, C1541;

PROCEDURE CheckFollowDirEntry(i: BYTE);
 VAR j, k: BYTE;
BEGIN
 j := i + 1;
 IF j > DIR_ENTRIES THEN
  FATAL(E_DIR);
 IF cpm_dir[i].dr <> cpm_dir[j].dr THEN
  FATAL(E_DIR);
 FOR k := 1 TO 8 DO
  IF cpm_dir[i].f[k] <> cpm_dir[j].f[k] THEN
   FATAL(E_DIR);
 FOR k := 1 TO 3 DO
  IF cpm_dir[i].t[k] <> cpm_dir[j].t[k] THEN
   FATAL(E_DIR);
END; (* CheckFollowDirEntry *)

PROCEDURE dir_check;
 VAR i, j, next_ex: BYTE;
BEGIN
 stat_duse := 0;
 stat_fnr := 0;
 stat_recnr := 0;
 FOR i := 0 TO NR_SECTOR-1 DO
  cpm_free[i] := TRUE;
 FOR i := 0 TO DIR_SECTOR-1 DO
  cpm_free[i] := FALSE;
 next_ex := 0;
 FOR i := 1 TO DIR_ENTRIES DO
  WITH cpm_dir[i] DO
   IF dr <> $E5 THEN BEGIN
    Inc(stat_duse);
    IF ex <> next_ex THEN
     FATAL(E_DIR);
    IF rc > $80 THEN
     FATAL(E_DIR);
    IF ex = 0 THEN Inc(stat_fnr);
    IF rc = $80 THEN
     BEGIN
      CheckFollowDirEntry(i);
      Inc(stat_recnr, 16*REC_SECTOR);
      FOR j := 0 TO 15 DO
       cpm_free[d[j]] := FALSE;
      INC(next_ex);
     END
    ELSE (* rc < $80 *)
     BEGIN
      Inc(stat_recnr,rc);
      IF rc > 0 THEN
       FOR j := 0 TO rc DIV REC_SECTOR DO
        cpm_free[d[j]] := FALSE;
      next_ex := 0;
     END; (* else *)
   END; (* if *)
 WITH cpm_dir[DIR_ENTRIES] DO
  IF (dr <> $E5) AND (rc = $80) THEN
   FATAL(E_DIR);
 stat_clu := 0;
 FOR i := 0 TO NR_SECTOR-1 DO
  IF NOT cpm_free[i] THEN
   Inc(stat_clu);
END; (* dir_check *)

PROCEDURE dir_load;
 VAR i: WORD;
  both: ARRAY [1..DIR_SECTOR] OF SECTOR ABSOLUTE cpm_dir;
BEGIN
 FOR i := 1 TO DIR_SECTOR DO BEGIN
  dev_read(i-1,both[i]);
  err_stop;
 END; (* for *)
 dir_check;
END; (* dir_load *)

PROCEDURE dir_save;
 VAR i: WORD;
  both: ARRAY [1..DIR_SECTOR] OF SECTOR ABSOLUTE cpm_dir;
BEGIN
 dir_check;
 FOR i := 1 TO DIR_SECTOR DO BEGIN
  dev_write(i-1,both[i]);
  err_stop;
 END; (* for *)
END; (* dir_save *)

PROCEDURE TransferFileName(VAR fn: STRING);
 VAR s, i: BYTE; ch: CHAR;
  st: STRING;
BEGIN
 s := 8;
 st := fn+'..'; fn := '';
 FOR i := 1 TO Length(st) DO BEGIN
  ch := st[i];
  CASE ch OF
   '.': BEGIN
     WHILE Length(fn) < s DO
      fn := fn + ' ';
     IF s = 12 THEN EXIT;
     fn := fn + '.';
     s := 12;
    END;
   '*':
     WHILE Length(fn) < s DO
      fn := fn + '?';
   ELSE
    IF Length(fn) <= s THEN
     fn := fn + UpCase(ch)
    ELSE IF s = 12 THEN
     EXIT;
  END; (* case *)
 END; (* while *)
END; (* TransferFileName *)

FUNCTION DirName(i: BYTE): STRING;
 VAR j: BYTE; dn: STRING;
BEGIN
 dn := '';
 WITH cpm_dir[i] DO BEGIN
  FOR j := 1 TO 8 DO
   dn := dn + CHR(ORD(f[j]) AND $7F);
  dn := dn + '.';
  FOR j := 1 TO 3 DO
   dn := dn + CHR(ORD(t[j]) AND $7F);
 END; (* with *)
 DirName := dn;
END; (* DirName *)

FUNCTION LIKE(fn, pat: STRING): BOOLEAN;
 VAR i: BYTE;
BEGIN
 LIKE := FALSE;
 FOR i := 1 TO Length(fn) DO
  IF pat[i] <> '?' THEN
   IF pat[i] <> fn[i] THEN
    EXIT;
 LIKE := TRUE;
END; (* LIKE *)

PROCEDURE dir_next(VAR sr: dir_search);
BEGIN
 WITH sr DO BEGIN
  Inc(dirpos);
  WHILE dirpos <= DIR_ENTRIES DO BEGIN
   IF cpm_dir[dirpos].dr <> $E5 THEN
    IF cpm_dir[dirpos].ex = 0 THEN BEGIN
     name := DirName(dirpos);
     found := LIKE(name, pattern);
     IF found THEN EXIT;
    END; (* if *)
   Inc(dirpos);
  END; (* while *)
  found := FALSE;
  name := '';
 END; (* with *)
END; (* dir_next *)

PROCEDURE dir_first(fn: STRING; VAR sr: dir_search);
BEGIN
 TransferFileName(fn);
 WITH sr DO BEGIN
  pattern := fn;
  dirpos := 0;
  found := FALSE;
  name := '';
 END; (* with *)
 dir_next(sr);
END; (* dir_first *)

FUNCTION dir_new_entry(VAR fn: STRING): BYTE;
 VAR i, next: BYTE;
BEGIN
 next := 1;
 FOR i := 1 TO DIR_ENTRIES DO
  IF cpm_dir[i].dr <> $E5 THEN BEGIN
   cpm_dir[next] := cpm_dir[i];
   Inc(next);
  END; (* if *)
 IF next > DIR_ENTRIES THEN
  FATAL(E_DIRFULL);
 TransferFileName(fn);
 dir_new_entry := next;
 WITH cpm_dir[next] DO BEGIN
  dr := 0;
  FOR i := 1 TO  8 DO f[i] := fn[i];
  FOR i := 1 TO  3 DO t[i] := fn[9+i];
  ex := 0;
  s1 := 0; s2 := 0;
  rc := 0;
  FOR i := 0 TO 15 DO d[i] := 0;
 END; (* with *)
 Inc(next);
 WHILE next <= DIR_ENTRIES DO
  WITH CPM_dir[next] DO BEGIN
   dr := $E5;
    f := #$E5#$E5#$E5#$E5#$E5#$E5#$E5#$E5;
    t := #$E5#$E5#$E5;
   ex := $E5;
   s1 := $E5;
   s2 := $E5;
   rc := $E5;
   FOR i := 0 TO 15 DO d[i] := $E5;
   Inc(next);
  END; (* with *)
END; (* dir_new_entry *)

FUNCTION dir_new_sector: BYTE;
 VAR i: BYTE;
BEGIN
 FOR i := 0 TO NR_SECTOR DO
  IF cpm_free[i] THEN BEGIN
   cpm_free[i] := FALSE;
   dir_new_sector := i;
   EXIT;
  END; (* if *)
 FATAL(E_FULL);
END; (* dir_new_sector *)

PROCEDURE dir_print(pat: STRING);
 VAR sr: dir_search;
  recs: WORD;
  this_fnr, this_recnr, this_clu: WORD;
BEGIN
 IF pat = '' THEN pat := '*.*';
 WriteLn('    Name     Bytes   Recs ');
 WriteLn('------------ ----- -------');
 this_fnr := 0;
 this_recnr := 0;
 this_clu := 0;
 dir_first(pat, sr);
 err_stop;
 WHILE sr.found DO
  WITH sr DO BEGIN
   INC(this_fnr);
   Write(name,' ');
   recs := 0;
   WHILE cpm_dir[dirpos].rc = $80 DO BEGIN
    Inc(recs, 16*REC_SECTOR);
    Inc(dirpos);
   END; (* while *)
   Inc(recs, cpm_dir[dirpos].rc);
   Inc(this_recnr, recs);
   Inc(this_clu, (recs+REC_SECTOR-1) DIV REC_SECTOR);
   WriteLn((recs+REC_SECTOR-1) DIV REC_SECTOR : 5,'k', recs:7);
   dir_next(sr);
  END; (* with *)
 WriteLn;
 Write('Total Records  = ',this_recnr,'/',stat_recnr,'  ');
 WriteLn('Files Found ',this_fnr,'/',stat_fnr);
 Write('Total Clusters = ', this_clu,'/',stat_clu,'/',NR_SECTOR,'  ');
 WriteLn('Used/Max Dir Entries  ',stat_duse,'/',DIR_ENTRIES);
END; (* dir_print *)

PROCEDURE dir_erase(fn: STRING);
 VAR sr: dir_search;
BEGIN
 IF fn = '' THEN BEGIN
  err_p1 := 1;
  FATAL(E_PARAM);
 END; (* if *)
 dir_first(fn,sr);
 IF NOT sr.found THEN FATAL(E_NOFILE);
 WHILE sr.found DO
  WITH sr DO BEGIN
   WHILE cpm_dir[dirpos].rc = $80 DO BEGIN
    cpm_dir[dirpos].dr := $E5;
    Inc(dirpos);
   END; (* while *)
   cpm_dir[dirpos].dr := $E5;
   dir_next(sr);
  END; (* with *)
 dir_save;
 err_stop;
END; (* dir_erase *)

FUNCTION NewName(o,p: STRING): STRING;
 VAR i: BYTE;
BEGIN
 FOR i := 1 TO 13 DO
  IF p[i] = '?' THEN
   p[i] := o[i];
 NewName := p;
END; (* NewName *)

PROCEDURE dir_rename(op, np: STRING);
 VAR sr, new_sr: dir_search;
  ofn, nfn: STRING; i: BYTE;
  last: BOOLEAN;
BEGIN
 IF op = '' THEN BEGIN
  err_p1 := 2;
  FATAL(E_PARAM);
 END; (* if *)
 TransferFileName(op);
 IF np = '' THEN BEGIN
  err_p1 := 2;
  FATAL(E_PARAM);
 END; (* if *)
 TransferFileName(np);
 dir_first(op,sr);
 IF NOT sr.found THEN FATAL(E_NOFILE);
 WHILE sr.found DO BEGIN
  ofn := sr.name;
  nfn := NewName(ofn,np);
  WriteLn('RENAMING ',ofn,' TO ', nfn);
  dir_first(nfn, new_sr);
  IF new_sr.found THEN FATAL(E_FILE);
  REPEAT
   WITH cpm_dir[sr.dirpos] DO BEGIN
    FOR i := 1 TO  8 DO f[i] := nfn[i];
    FOR i := 1 TO  3 DO t[i] := nfn[9+i];
    last := rc <> $80;
   END; (* with *)
   INC(sr.dirpos);
  UNTIL last;
  Dec(sr.dirpos);
  dir_next(sr);
 END; (* while *)
 dir_save;
 err_stop;
END; (* dir_rename *)

END. (* Cdirectory *)