/*
 * ACME - a crossassembler for producing 6502/65c02/65816 code.
 * Copyright (C) 1998 Marco Baye
 * Have a look at "acme.c" for further info
 */

/*
 * Input/output
 */

#include "stream.h"

/*
 * Get byte from file and convert newline characters to zero.
 */
static char Stream_ConvertNL() {
  char old;
  int    b;

  /* Exchange newline characters by zero */
  /* If reading LF of CRLF, read another byte */
  do {
    old = Context[nContext].OldFileByte;
    if(EndReason == RENDOFFILE) {
      Context[nContext].OldFileByte = 0;/* Fake end-of-statement */
    } else {
      /* Get byte from file */
      b = fgetc(Context[nContext].u.hFile);
      if(b == EOF) {
        /* Act upon end of file */
        EndReason = RENDOFFILE;
        Context[nContext].OldFileByte = 0;/* Fake end-of-statement */
      } else {
        Context[nContext].OldFileByte = (char) b;
      }
    }
  } while((old == 13) && (Context[nContext].OldFileByte == 10));/* CRLF test */
/* Check for LF and CR separately. Note that setting
 * "Context[nContext].OldFileByte" to zero would confuse the
 * "old/Context[nContext].OldFileByte" stuff above.*/
  if(Context[nContext].OldFileByte == 10) return(0);
  if(Context[nContext].OldFileByte == 13) return(0);
  return(Context[nContext].OldFileByte);
}

/*
 * Get byte (low level)
 */
static char Stream_GetRawByte() {
  char old = Context[nContext].OldRawByte;

  if(old == 0) Context[nContext].nLines++;/* Increase line counter */
  switch(Context[nContext].hByteSource) {

    case BYTESOURCE_FILE:
    /* File handler */
    if(QuoteChar != NO_QUOTE_CHAR) {
      /* Inside quotes, don't process. Only complain about zero */
      Context[nContext].OldRawByte = Stream_ConvertNL();
      if(Context[nContext].OldRawByte == 0) {
        Message(pseOpenQuotes, EERROR);
        QuoteChar = NO_QUOTE_CHAR;
      }
    } else {
      /* Outside quotes, process */
      do {
        Context[nContext].OldRawByte = Stream_ConvertNL();
        /* convert TAB to space, kill multiple spaces */
        if(Context[nContext].OldRawByte == 9)
           Context[nContext].OldRawByte = ' ';
      } while((Context[nContext].OldRawByte == ' ') && ((old == 0)
        || (old == ' ') || (old == ',')
        || (old == ':') || (old == '{')));
      /* After semicolon, skip remainder of line */
      if(Context[nContext].OldRawByte == ';')
        while((Context[nContext].OldRawByte = Stream_ConvertNL()));
    }
    break;

    case BYTESOURCE_RAM:
    /* RAM handler */
    Context[nContext].OldRawByte = *(Context[nContext].u.pRAM++);
    if(Context[nContext].OldRawByte == SEPARATOR) {
      EndReason = RENDOFBLOCK;
      Context[nContext].OldRawByte = 0;/* Fake end-of-statement */
    }
  }
  /* Check quoting */
  if(QuoteChar != NO_QUOTE_CHAR) {
    if(Context[nContext].OldRawByte == QuoteChar) QuoteChar = NO_QUOTE_CHAR;
  } else {
    if(Context[nContext].OldRawByte == '"')  QuoteChar = '"';
    if(Context[nContext].OldRawByte == '\'') QuoteChar = '\'';
  }
  return(Context[nContext].OldRawByte);
}

/*
 * Get byte (high level)
 */
static char GetByte() {
  GotByte = Stream_GetRawByte();
  if(QuoteChar == NO_QUOTE_CHAR) {
    /* Check for braces */
    if(GotByte == '}') {
      EndReason = RRIGHTBRACE;
      GotByte = 0;
    } else {
      /* Check for colon */
      if(GotByte == ':') GotByte = 0;
    }
  }
#ifdef FDEBUG
  fputc(GotByte, filter);
#endif
  return(GotByte);
}

/*
 * This routine reads in characters and stores them in memory, until an
 * illegal character is read or too many bytes have been received.
 * The second case produces a "string too long" error.
 * The routine zero-terminates the string and returns its length. Zero lengths
 * will produce a "missing string" error.
 */
static int Stream_ReadKeyword(char *p, int MaxLen, int fNext) {
  int c = 0;

  if(fNext) GetByte();
  do {
    p[c] = GotByte;
    /* now: c = number of bytes present minus one */
    if(pFlagTable[(u_char) GotByte] & MBYTEILLEGAL) break;
    GetByte();
  } while(c++ < MaxLen);
  if(c > MaxLen) {
    Message(pseTooLong, EERROR);
    c--;
  }
  p[c] = 0;/* terminate by overwriting illegal character */
  if(c == 0) Message(pseMissingString, EERROR);
  return(c);
}

/*
 * Try to read a filename. If the flag "fAllowLib" is TRUE, library access by
 * using <...> quoting is possible as well. The file name given in the
 * assembler source code is converted from UNIX style to system style.
 */
static int Stream_ReadFilename(char *p, int MaxLen, int fAllowLib) {
  int c = 0,
      d = 0;

  SKIPSPACE;
  if(fAllowLib && (GotByte == '<')) {
    if(psLibPath) {
      strcpy(p, psLibPath);
      c = strlen(p);
      QuoteChar = '>';
    } else {
      Message(pseNoVarFound, EERROR);
      SkipRest();
    }
  }
  if((QuoteChar == '"') || (QuoteChar == '>')) {
    d = Stream_FinishQuoted(p, MaxLen, c);
    PLATFORM_CONVERTPATH(p + c);/* Convert to system style */
  } else Message(pseMissingString, EERROR);
  return(d);
}

/*
 * Reads string and stores it in memory until closing quote is reached. Does
 * not store quote. Zero-terminates string and returns its length.
 * If the string is too long or there are no closing quotes, an error will be
 * produced. "cAlready" is the number of chars already present. Storing will
 * take place after them, their number is added when comparing to MaxLen.
 */
static int Stream_FinishQuoted(char *p, int MaxLen, int cAlready) {
  do {
    p[cAlready++] = GetByte();
  } while((cAlready < MaxLen) && GotByte && (QuoteChar != NO_QUOTE_CHAR));
  if(cAlready >= MaxLen) {
    Message(pseTooLong, EERROR);
    cAlready = MaxLen;/* shrink back to limit */
  }
  p[--cAlready] = 0;/* terminate, overwriting last character (closing quote) */
  GetByte();/* proceed with next char */
  SKIPSPACE;
  return(cAlready);
}

/*
 * Try to read a comma, skipping spaces before and after. Return TRUE if comma
 * found, otherwise FALSE.
 */
static int Stream_Comma() {
  SKIPSPACE;
  if(GotByte != ',') return(FALSE);
  NEXTANDSKIPSPACE;
  return(TRUE);
}

/*
 * Check whether the given code byte runs on the current CPU and output it.
 */
static void Stream_PutCodeByte(u_char byte) {
  if(pFlagTable[(u_char) byte] & (1 << hCPU_Now)) {
    Stream_PutByte((Value) byte);
  } else {
    Message(pseWrongCPU, EERROR);
    SkipRest();
  }
}

/*
 * Output 8-bit value with range check
 */
static void Stream_Put8(Value v) {
  if((v <= 0xff) && (v >= -0x80)) {
    Stream_PutByte(v);
  } else Message(pseTooBig, EERROR);
}

/*
 * Output 16-bit values with range check
 */
static void Stream_Put16(Value v) {
  if((v <= 0xffff) && (v >= -0x8000)) {
    Stream_PutByte(v);
    Stream_PutByte(v >> 8);
  } else Message(pseTooBig, EERROR);
}

/*
 * Output 24-bit values with range check
 */
static void Stream_Put24(Value v) {
  if((v <= 0xffffff) && (v >= -0x800000)) {
    Stream_PutByte(v);
    Stream_PutByte(v >> 8);
    Stream_PutByte(v >> 16);
  } else Message(pseTooBig, EERROR);
}


/*
 * Output 32-bit values (without range check)
 */
static void Stream_Put32(Value v) {
/*  if((v <= 0x7fffffff) && (v >= -0x80000000)) {*/
    Stream_PutByte(v);
    Stream_PutByte(v >> 8);
    Stream_PutByte(v >> 16);
    Stream_PutByte(v >> 24);
/*  } else Message(pseTooBig, EERROR);*/
}


/*
 * Send low byte to output file, automatically increasing the program counter
 */
static void Stream_PutByte(Value byte) {
  int offset;

  if(fPCdefined == FALSE) {
    Message(pseNoPC, EERROR);
    fPCdefined = TRUE;
  }
  offset = PC_Mem + PC_inc;
  if(offset == SegmentMax+1) {
    if(offset == OUTBUFFERSIZE) {
      Message(pseTooMuch, ESERIOUS);
    } else {
      Message(pswSegReached, EWARNING);
      if(ffPass & FPASSDOONCE) Segment_FindMax(offset+1);
    }
  }
  OutputBuffer[offset] = byte & 255;
  PC_inc++;
}

/*
 * Try to open top level source file
 */
static int Stream_OpenTop() {
  if(ffOpenFiles & FOPENED_TOP) {
    Message(pseNo3rdFile, EERROR);
    return(FALSE);
  }
  hfTop = fopen(pnfTop, "rb");/* open file */
  if(hfTop) {
    ffOpenFiles |= FOPENED_TOP;
    return(TRUE);
  } else {
    return(FALSE);
  }
}

/*
 * Close top level source file, if open
 */
static void Stream_CloseTop() {
  if(ffOpenFiles & FOPENED_TOP) {
    fclose(hfTop);
    ffOpenFiles &= ~FOPENED_TOP;
  }
}

/*
 * Try to open sub level file (source/binary)
 */
static int Stream_OpenSub(char *pnfSub) {
  if(ffOpenFiles & FOPENED_SUB) {
    Message(pseNo3rdFile, EERROR);
    return(FALSE);
  }
  hfSub = fopen(pnfSub, "rb");/* open file */
  if(hfSub) {
    ffOpenFiles |= FOPENED_SUB;
    return(TRUE);
  } else {
    Message(pseCannotOpenSubFile, EERROR);
    return(FALSE);
  }
}

/*
 * Close sub level file, if open
 */
static void Stream_CloseSub() {
  if(ffOpenFiles & FOPENED_SUB) {
    fclose(hfSub);
    ffOpenFiles &= ~FOPENED_SUB;
  }
}

/*
 * Dump global labels into dump file
 */
static void Stream_Dump() {
  FILE     *hfDump;
  ListItem *p;
  Value     v;
  char      c1, c2;
  int       a,
            ff,
            cGlobal,
            sGlobal = 0,
            cLabel,
            sLabel = 0,
            cAll,
            sAll = 0;

  if(ffRepeat & FREPEAT_DOLABELDUMP) {
    ffRepeat &= ~FREPEAT_DOLABELDUMP;
    hfDump = fopen(pnfDump, "w");/* try to open dump file */
    if(hfDump) {
      for(a = 255; a >= 0; a--) {
        cGlobal = 0;
        cLabel = 0;
        cAll = 0;
        p = (ListItem *) &(pPointerTable[a]);/* point to pointer */
        while(p->Next) {
          cAll++;
          p = p->Next;
          if(p->Type == HTYPE_LABEL) {
            cLabel++;
            /* list item is label, so prepare for zone check */
            c1 = p->String[0];
            c2 = p->String[1];
            if((c1 == 0) && (c2 == 0)) {
              cGlobal++;
              /* label is global, so dump it */
              fprintf(hfDump, "%s", &(p->String[2]));
              ff = p->Data.Label.Flags;
              v  = p->Data.Label.Value;
              switch(ff & MVALUE_FORCEBITS) {

                case MVALUE_FORCE16:
                fprintf(hfDump, "+2=");
                break;

                case MVALUE_FORCE16 | MVALUE_FORCE24:
                /*FALLTHROUGH*/

                case MVALUE_FORCE24:
                fprintf(hfDump, "+3=");
                break;

                default:
                fprintf(hfDump, "  =");

              }
              if(ff & MVALUE_DEFINED) fprintf(hfDump, "$%x", (unsigned int) v);
              else                    fprintf(hfDump, " ?");
              if(ff & MVALUE_UNSURE)  fprintf(hfDump, "; ?\n");
              else                    fprintf(hfDump, "\n");
#ifdef SLCHECK
            } else {
              fprintf(hfDump, "Local %s\n", &(p->String[2]));
#endif
            }
#ifdef SLCHECK
          } else {
            fprintf(hfDump, "Struc %x ", p->Type);
            if(p->Type == HTYPE_MACRO) fprintf(hfDump, "%s\n", &(p->String[2]));
            else                       fprintf(hfDump, "%s\n", &(p->String[0]));
#endif
          }
        }
#ifdef SLCHECK
        fprintf(hfDump, ";$%x %d/%d/%d\n", a, cGlobal, cLabel, cAll);
#endif
        sGlobal += cGlobal;
        sLabel += cLabel;
        sAll += cAll;
      }
      fprintf(hfDump, ";%d/%d/%d\n", sGlobal, sLabel, sAll);
      fclose(hfDump);/* close dump file */
      PLATFORM_SETFILETYPE(pnfDump, FILETYPE_TEXT);
    } else printf(pseCannotOpenDumpFile);/* Message() would be endless loop */
  }
}

/*
 * Dump code from buffer into output file
 */
static void Stream_Output(Value Start, Value End) {
  FILE  *hfOut;
  Value  a;

  if(lVerbosity) printf(psvOutput, Start, End);
  /* Try to open output file */
  hfOut = fopen(pnfOut, "wb");
  if(hfOut) {
    /* !!! always produces cbm files !!! */
    PLATFORM_SETFILETYPE(pnfOut, FILETYPE_CBM);
    fputc(Start & 255, hfOut);/* load address, low  byte */
    fputc(Start >>  8, hfOut);/* load address, high byte */
    for(a = Start; a < End; a++) {
      fputc(OutputBuffer[a], hfOut);
    }
    fclose(hfOut);/* close output file */
  } else Message(pseCannotOpenOutFile, ESERIOUS);
}
