/*
 *      This file contains IDA built-in function declarations
 *      and internal bit definitions.
 *      Each byte of the program has 32-bit flags
 *      (low 8 bits keep the byte value).
 *      These 32 bits are used in GetFlags/SetFlags functions.
 *      You may freely examine these bits using GetFlags()
 *      but I strongly discourage using SetFlags() function.
 *
 *      This file is subject to change without any notice.
 *      Future versions of IDA may use other definitions.
 */

#ifndef _IDC_IDC
#define _IDC_IDC

//--------------------------------------------------------------------------
#define BADADDR         0xFFFFFFFFL     // Not allowed address value
#define MAXADDR         0xFF000000L     // Max allowed address in IDA

//
//      Flag bit definitions (for GetFlags())
//

#define MS_VAL  0x000000FFL             // Mask for byte value
#define FF_IVL  0x00000100L             // Byte has value ?


// Do flags contain byte value? (i.e. has the byte a value?)
// if not, the byte is uninitialized.

#define hasValue(F)     ((F & FF_IVL) != 0)     // any defined value?


// Get byte value from flags
// Get value of byte provided that the byte is initialized.
// This macro works ok only for 8-bit byte machines.

#define byteValue(F)    (F & MS_VAL)    // quick replacement for Byte()


// Is the byte initialized?

#define isLoaded(ea)    hasValue(GetFlags(ea))  // any defined value?

#define MS_CLS  0x00000600L             // Mask for typing
#define FF_CODE 0x00000600L             // Code ?
#define FF_DATA 0x00000400L             // Data ?
#define FF_TAIL 0x00000200L             // Tail ?
#define FF_UNK  0x00000000L             // Unknown ?


#define isCode(F)       ((F & MS_CLS) == FF_CODE) // is code byte?
#define isData(F)       ((F & MS_CLS) == FF_DATA) // is data byte?
#define isTail(F)       ((F & MS_CLS) == FF_TAIL) // is tail byte?
#define isUnknown(F)    ((F & MS_CLS) == FF_UNK)  // is unexplored byte?
#define isHead(F)       ((F & FF_DATA) != 0)      // is start of code/data?

//
//      Common bits
//

#define MS_COMM 0x000FF800L             // Mask of common bits
#define FF_COMM 0x00000800L             // Has comment?
#define FF_REF  0x00001000L             // has references?
#define FF_LINE 0x00002000L             // Has next or prev cmt lines ?
#define FF_NAME 0x00004000L             // Has user-defined name ?
#define FF_LABL 0x00008000L             // Has dummy name?
#define FF_FLOW 0x00010000L             // Exec flow from prev instruction?
#define FF_VAR  0x00080000L             // Is byte variable ?

#define isFlow(F)       ((F & FF_FLOW) != 0)
#define isVar(F)        ((F & FF_VAR ) != 0)
#define isExtra(F)      ((F & FF_LINE) != 0)
#define isRef(F)        ((F & FF_REF)  != 0)
#define hasName(F)      ((F & FF_NAME) != 0)

#define MS_0TYPE 0x00F00000L            // Mask for 1st arg typing
#define FF_0VOID 0x00000000L            // Void (unknown)?
#define FF_0NUMH 0x00100000L            // Hexadecimal number?
#define FF_0NUMD 0x00200000L            // Decimal number?
#define FF_0CHAR 0x00300000L            // Char ('x')?
#define FF_0SEG  0x00400000L            // Segment?
#define FF_0OFF  0x00500000L            // Offset?
#define FF_0NUMB 0x00600000L            // Binary number?
#define FF_0NUMO 0x00700000L            // Octal number?
#define FF_0ENUM 0x00800000L            // Enumeration?
#define FF_0FOP  0x00900000L            // Forced operand?
#define FF_0STRO 0x00A00000L            // Struct offset?
#define FF_0STK  0x00B00000L            // Stack variable?

#define MS_1TYPE 0x0F000000L            // Mask for 2nd arg typing
#define FF_1VOID 0x00000000L            // Void (unknown)?
#define FF_1NUMH 0x01000000L            // Hexadecimal number?
#define FF_1NUMD 0x02000000L            // Decimal number?
#define FF_1CHAR 0x03000000L            // Char ('x')?
#define FF_1SEG  0x04000000L            // Segment?
#define FF_1OFF  0x05000000L            // Offset?
#define FF_1NUMB 0x06000000L            // Binary number?
#define FF_1NUMO 0x07000000L            // Octal number?
#define FF_1ENUM 0x08000000L            // Enumeration?
#define FF_1FOP  0x09000000L            // Forced operand?
#define FF_1STRO 0x0A000000L            // Struct offset?
#define FF_1STK  0x0B000000L            // Stack variable?

// The following macros answer questions like
//   'is the 1st (or 2nd) operand of instruction or data of the given type'?
// Please note that data items use only the 1st operand type (is...0)

#define isDefArg0(F)    ((F & MS_0TYPE) != FF_0VOID)
#define isDefArg1(F)    ((F & MS_1TYPE) != FF_1VOID)
#define isDec0(F)       ((F & MS_0TYPE) == FF_0NUMD)
#define isDec1(F)       ((F & MS_1TYPE) == FF_1NUMD)
#define isHex0(F)       ((F & MS_0TYPE) == FF_0NUMH)
#define isHex1(F)       ((F & MS_1TYPE) == FF_1NUMH)
#define isOct0(F)       ((F & MS_0TYPE) == FF_0NUMO)
#define isOct1(F)       ((F & MS_1TYPE) == FF_1NUMO)
#define isBin0(F)       ((F & MS_0TYPE) == FF_0NUMB)
#define isBin1(F)       ((F & MS_1TYPE) == FF_1NUMB)
#define isOff0(F)       ((F & MS_0TYPE) == FF_0OFF)
#define isOff1(F)       ((F & MS_1TYPE) == FF_1OFF)
#define isChar0(F)      ((F & MS_0TYPE) == FF_0CHAR)
#define isChar1(F)      ((F & MS_1TYPE) == FF_1CHAR)
#define isSeg0(F)       ((F & MS_0TYPE) == FF_0SEG)
#define isSeg1(F)       ((F & MS_1TYPE) == FF_1SEG)
#define isEnum0(F)      ((F & MS_0TYPE) == FF_0ENUM)
#define isEnum1(F)      ((F & MS_1TYPE) == FF_1ENUM)
#define isFop0(F)       ((F & MS_0TYPE) == FF_0FOP)
#define isFop1(F)       ((F & MS_1TYPE) == FF_1FOP)
#define isStroff0(F)    ((F & MS_0TYPE) == FF_0STRO)
#define isStroff1(F)    ((F & MS_1TYPE) == FF_1STRO)
#define isStkvar0(F)    ((F & MS_0TYPE) == FF_0STK)
#define isStkvar1(F)    ((F & MS_1TYPE) == FF_1STK)

//
//      Bits for DATA bytes
//

#define DT_TYPE 0xF0000000L             // Mask for DATA typing

#define FF_BYTE 0x00000000L             // byte
#define FF_WORD 0x10000000L             // word
#define FF_DWRD 0x20000000L             // dword
#define FF_QWRD 0x30000000L             // qword
#define FF_TBYT 0x40000000L             // tbyte
#define FF_ASCI 0x50000000L             // ASCII ?
#define FF_STRU 0x60000000L             // Struct ?
#define FF_OWRD 0x70000000L             // octaword (16 bytes)
#define FF_FLOAT 0x80000000L            // float
#define FF_DOUBLE 0x90000000L           // double
#define FF_PACKREAL 0xA0000000L         // packed decimal real
#define FF_ALIGN    0xB0000000L         // alignment directive

//
//      Bits for CODE bytes
//

#define MS_CODE 0xF0000000L
#define FF_FUNC 0x10000000L             // function start?
#define FF_IMMD 0x40000000L             // Has Immediate value ?
#define FF_JUMP 0x80000000L             // Has jump table

#undef _notdefinedsymbol
#ifdef _notdefinedsymbol // There aren't declarations in IDC, so comment them

        List of built-in functions
        --------------------------

The following conventions are used in this list:
  'ea' is a linear address
  'success' is 0 if a function failed, 1 otherwise
  'void' means that function returns no meaningful value (always 0)

 All function parameter conversions are made automatically.


                      M I S C E L L A N E O U S


// Return value of expression: ((seg<<4) + off)

long    MK_FP           (long seg,long off);    // the same as [ seg, off ]
                                                // i.e: ((seg<<4)+off)

// Return a formatted string.
//      format - printf-style format string.
//               %a - means address expression.
//               floating point values are output only in one format
//                regardless of the character specified (f,e,g,E,G)
//               %p is not supported.
// The resulting string must be less than 255 characters

char    form            (char format,...);      // works as sprintf
                                                // The resulting string should
                                                // be less than 255 characters.

// Return substring of a string
//      str - input string
//      x1  - starting index (0..n)
//      x2  - ending index. If x2 == -1, then return substring
//            from x1 to the end of string.

char    substr          (char str,long x1,long x2); // substring [x1..x2-1]
                                                // if x2 == -1, then till end of line

// Search a substring in a string
//      str    - input string
//      substr - substring to search
// returns: 0..n - index in the 'str' where the substring starts
//          -1   - if the substring is not found

long    strstr          (char str,char substr); // find a substring, -1 - not found


// Return length of a string in bytes
//      str - input string
// Returns: length (0..n)

long    strlen          (char str);             // calculate length


// Convert ascii string to a binary number.
// (this function is the same as hexadecimal 'strtoul' from C library)

long    xtol            (char str);             // ascii hex -> number
                                                // (use long() for atol)


// Convert address value to a string

char    atoa            (long ea);              // returns address in
                                                // the form 'seg000:1234'
                                                // (the same as in line prefixes)


// Convert a number to a string.
//      n - number
//      radix - number base (2,8,10,16)

char    ltoa            (long n,long radix);    // convert to ascii string


// Convert ascii string to a number
//      str - a decimal representation of a number
// returns: a binary number

long    atol            (char str);             // convert ascii decimal to long


// Add hotkey for IDC function
//      hotkey  - hotkey name ('a', "Alt-A", etc)
//      idcfunc - IDC function name
// returns:
#endif
#define IDCHK_OK        0       // ok
#define IDCHK_ARG       -1      // bad argument(s)
#define IDCHK_KEY       -2      // bad hotkey name
#define IDCHK_MAX       -3      // too many IDC hotkeys
#ifdef _notdefinedsymbol

long AddHotkey(char hotkey,char idcfunc);


// Delete IDC function hotkey

success DelHotkey(char hotkey);


// Move cursor to the specifed linear address
//      ea - linear address

success Jump            (long ea);              // move cursor to ea
                                                // screen is refreshed at
                                                // the end of IDC execution

// Wait for the end of autoanalysis
// This function will suspend execution of IDC program
// till the autoanalysis queue is empty.

void    Wait            ();                     // Process all entries in the
                                                // autoanalysis queue

// Compile an IDC file.
// The file being compiled should not contain functions that are
// currently executing - otherwise the behaviour of the replaced
// functions is undefined.
//      filename - name of file to compile
// returns: "" - ok, otherwise it returns an error message.

char    Compile         (char filename);        // Compile an IDC file.
                                                // returns error message.

// Stop execution of IDC program, close the database and exit to OS
//      code - code to exit with.

void    Exit            (long code);            // Exit to OS


// Execute an OS command.
// IDA will wait for the started program to finish.
// In order to start the command in parallel, use OS methods.
// For example, you may start another program in parallel using "start" command.
//      command - command line to execute
// returns: error code from OS

long    Exec            (char command);         // Execute OS command



      C H A N G E   P R O G R A M   R E P R E S E N T A T I O N


// Delete all segments, instructions, comments, i.e. everything
// except values of bytes.

void    DeleteAll       ();                     // delete ALL information
                                                // about the program

// Create an instruction at the specified address
//      ea - linear address
// returns: 0 - can't create an instruction (no such opcode, the instruction would
//              overlap with existing items, etc)
//          otherwise returns length of the instruction in bytes

long    MakeCode        (long ea);              // convert to instruction
                                                // returns number of bytes
                                                // occupied by the instruction


// Perform full analysis of the area
//      sEA - starting linear address
//      eEA - ending linear address (excluded)
// returns: 1-ok, 0-Ctrl-Break was pressed.

long    AnalyseArea     (long sEA,long eEA);    // analyse area and try to
                                                // convert to code all bytes
                                                // Returns 1-ok,0-CtrlBreak pressed

// Rename a byte
//      ea - linear address
//      name - new name of address. If name == "", then delete old name
// returns: 1-ok, 0-failure

success MakeName        (long ea,char name);    // assign a name to a location


// Set an indented regular comment of an item
//      ea      - linear address
//      comment - comment string

success MakeComm        (long ea,char comment); // give a comment


// Set an indented repeatable comment of an item
//      ea      - linear address
//      comment - comment string

success MakeRptCmt      (long ea,char comment); // give a repeatable comment


// Create an array.
//      ea      - linear address
//      nitems  - size of array in items
// This function will create an array of the items with the same type as the
// type of the item at 'ea'. If the byte at 'ea' is undefined, then this
// function will create an array of bytes.

success MakeArray       (long ea,long nitems);  // convert to an array


// Create a string.
// This function creates a string (the style is determinted by the value
// of GetLongPrm(INF_STRTYPE), see below).
//      ea - linear address
//      endea - ending address of the string (excluded)
//              if endea == BADADDR, then length of string will be calculated
//              by the kernel
// returns: 1-ok, 0-failure

success MakeStr         (long ea,long endea);   // convert to ASCII string


// Convert the current item to a byte
//      ea - linear address
// returns: 1-ok, 0-failure

success MakeByte        (long ea);              // convert to byte


// Convert the current item to a word (2 bytes)
//      ea - linear address
// returns: 1-ok, 0-failure

success MakeWord        (long ea);              // convert to word


// Convert the current item to a double word (4 bytes)
//      ea - linear address
// returns: 1-ok, 0-failure

success MakeDword       (long ea);              // convert to double-word


// Convert the current item to a quadro word (8 bytes)
//      ea - linear address
// returns: 1-ok, 0-failure

success MakeQword       (long ea);              // convert to quadro-word


// Convert the current item to a octa word (16 bytes)
//      ea - linear address
// returns: 1-ok, 0-failure

success MakeOword       (long ea);              // convert to octa-word


// Convert the current item to a floating point (4 bytes)
//      ea - linear address
// returns: 1-ok, 0-failure

success MakeFloat       (long ea);              // convert to float


// Convert the current item to a double floating point (8 bytes)
//      ea - linear address
// returns: 1-ok, 0-failure

success MakeDouble      (long ea);              // convert to double


// Convert the current item to a packed real (10 or 12 bytes)
//      ea - linear address
// returns: 1-ok, 0-failure

success MakePackReal    (long ea);              // convert to packed real


// Convert the current item to a tbyte (10 or 12 bytes)
//      ea - linear address
// returns: 1-ok, 0-failure

success MakeTbyte       (long ea);              // convert to 10 bytes (tbyte)


// Convert the current item to a structure instance
//      ea      - linear address
//      strname - name of a structure type
// returns: 1-ok, 0-failure

success MakeStruct      (long ea,char strname); // convert to structure instance


// Convert the current item to an alignment directive
//      ea      - linear address
//      count   - number of bytes to convert
//      align   - 0 or 1..32
//                if it is 0, the correct alignment will be calculated
//                by the kernel
// returns: 1-ok, 0-failure

success MakeAlign       (long ea,long count,long align);


// Create a local variable
//      start,end - range of addresses for the local variable
//                  The current version doesn't use 'end' address
//                  and creates a stack variable for the whole function
//                  If there is no function at 'start' then this function
//                  will fail.
//      location  - variable location in the form "[bp+xx]" where xx is
//                  a hexadecimal offset or a register name
//      name      - name of the local variable
// returns: 1-ok, 0-failure

success MakeLocal(long start,long end,char location,char name);


// Convert the current item to an explored item
//      ea     - linear address
//      expand - 0: just undefine the current item
//               1: undefine other instructions if the removal of the
//                  current instruction removes all references to them.
//                  (note: functions will not be undefined even if they
//                         have no references to them)

void    MakeUnkn        (long ea,long expand);  // convert to 'unknown'
                                                // expand!=0 => undefine consequent
                                                // instructions too


// Convert an operand of the item (instruction or data) to a binary number
//      ea - linear address
/       n  - number of operand
//              0 - the first operand
//              1 - the second, third and all other operands
//              -1 - all operands
// Note: the data items use only the type of the first operand
// Returns: 1-ok, 0-failure

success OpBinary        (long ea,int n);        // make operand binary
                                                // n=0 - first operand
                                                // n=1 - second, third etc. operands
                                                // n=-1 - all operands

// Convert an operand of the item (instruction or data) to an octal number
// (see explanation to Opbinary functions)

success OpOctal         (long ea,int n);

// Convert operand to decimal,hex,char (see OpBinary() for explanations)

success OpDecimal       (long ea,int n);
success OpHex           (long ea,int n);
success OpChr           (long ea,int n);


// Convert operand to an offset
// (for the explanations of 'ea' and 'n' please see OpBinary())
//      base - base of the offset as a linear address
//             If base == BADADDR then the current operand becomes non-offset
// Example:
//  seg000:2000 dw      1234h
// and there is a segment at paragraph 0x1000 and there is a data item
// within the segment at 0x1234:
//  seg000:1234 MyString        db 'Hello, world!',0
// Then you need to specify a linear address of the segment base to
// create a proper offset:
//      OpOffset(["seg000",0x2000],0,0x10000);
// and you will have:
//  seg000:2000 dw      offset MyString
// Motorola 680x0 processor have a concept of "outer offsets".
// If you want to create an outer offset, you need to combine number
// of the operand with the following bit:
#endif
#define OPND_OUTER      0x80                    // outer offset base
#ifdef _notdefinedsymbol
// Please note that the outer offsets are meaningful only for
// Motorola 680x0.

success OpOff           (long ea,int n,long base);


// Convert operand to a complex offset expression
// This is a more powerful version of OpOff() function.
// It allows to explicitly specify the reference type (off8,off16, etc)
// and the expression target with a possible target delta.
// The complex expressions are represented by IDA in the following form:
//
//         target + tdelta - base
//
// If the target is not present, then it will be calculated using
//         target = operand_value - tdelta + base
// The target must be present for LOW.. and HIGH.. reference types
//      ea      - linear address of the instruction/data
//      n       - number of operand to convert (the same as in OpOff)
//      reftype - one of REF_... constants
//      target  - an explicitly specified expression target. if you don't
//                want to specify it, use -1. Please note that LOW... and
//                HIGH... reference type requre the target.
//      base    - the offset base (a linear address)
//      tdelta  - a displacement from the target which will be displayed
//                in the expression.


success OpOffEx(long ea,int n,long reftype,long target,long base,long tdelta);

#endif
#define REF_OFF8    0 // 8bit full offset
#define REF_OFF16   1 // 16bit full offset
#define REF_OFF32   2 // 32bit full offset
#define REF_LOW8    3 // low 8bits of 16bit offset
#define REF_LOW16   4 // low 16bits of 32bit offset
#define REF_HIGH8   5 // high 8bits of 16bit offset
#define REF_HIGH16  6 // high 16bits of 32bit offset
#define REF_HIGH22  7 // high 22bits of 32bit offset (SPARC)
#define REF_LOW10   8 // low 10bits of 32bit offset (SPARC)
#ifdef _notdefinedsymbol

// Convert operand to a segment expression
// (for the explanations of 'ea' and 'n' please see OpBinary())

success OpSeg           (long ea,int n);

// Convert operand to a number (with default number base, radix)
// (for the explanations of 'ea' and 'n' please see OpBinary())

success OpNumber        (long ea,int n);


// Specify operand represenation manually.
// (for the explanations of 'ea' and 'n' please see OpBinary())
//      str - a string represenation of the operand
// IDA will not check the specified operand, it will simply display
// it instead of the orginal representation of the operand.

success OpAlt           (long ea,long n,char str);// manually enter n-th operand


// Change sign of the operand.
// (for the explanations of 'ea' and 'n' please see OpBinary())

success OpSign          (long ea,int n);        // change operand sign


// Toggle the bitwise not operator for the operand
// (for the explanations of 'ea' and 'n' please see OpBinary())

success OpNot           (long ea,int n);


// Convert operand to a symbolic constant
// (for the explanations of 'ea' and 'n' please see OpBinary())
//      enumid - id of enumeration type
//      serial - serial number of the constant in the enumeration
//               The serial numbers are used if there are more than
//               one symbolic constant with the same value in the
//               enumeration. In this case the first defined constant
//               get the serial number 0, then second 1, etc.
//               There could be 256 symbolic constants with the same
//               value in the enumeration.

success OpEnumEx(long ea,int n,long enumid,long serial);


// Convert operand to an offset in a structure
// (for the explanations of 'ea' and 'n' please see OpBinary())
//      strid - id of a structure type
//      delta - struct offset delta. usually 0. denotes the difference
//              between the structure base and the pointer into the structure.

success OpStroffEx      (long ea,int n,long strid, long delta); // make operand a struct offset


// Convert operand to a stack variable
// (for the explanations of 'ea' and 'n' please see OpBinary())

success OpStkvar        (long ea,int n);        // make operand a stack variable


// Convert operand to a high offset
// High offset is the upper 16bits of an offset.
// This type is used by TMS320C6 processors (and probably by other
// RISC processors too)
// (for the explanations of 'ea' and 'n' please see OpBinary())
//      target - the full value (all 32bits) of the offset

success OpHigh          (long ea,int n,long target);


// Mark the location as "variable"
// Note: All that IDA does is to mark the location as "variable". Nothing else,
// no additional analysis is performed.
// This function may disappear in the future.

void    MakeVar         (long ea);              // the location is 'variable'


// Specify an additional line to display before the generated ones.
//      ea   - linear address
//      n    - number of anterior additioal line (0..MAX_ITEM_LINES)
//      line - the line to display
// IDA displays additional lines from number 0 up to the first unexisting
// additional line. So, if you specify additional line #150 and there is no
// additional line #149, your line will not be displayed.
// MAX_ITEM_LINES is defined in IDA.CFG

void    ExtLinA         (long ea,long n,char line); // insert an additional line before the generated ones


// Specify an additional line to display after the generated ones.
//      ea   - linear address
//      n    - number of posterior additioal line (0..MAX_ITEM_LINES)
//      line - the line to display
// IDA displays additional lines from number 0 up to the first unexisting
// additional line. So, if you specify additional line #150 and there is no
// additional line #149, your line will not be displayed.
// MAX_ITEM_LINES is defined in IDA.CFG

void    ExtLinB         (long ea,long n,char line); // insert an additional line after the generated ones


// Delete an additional anterior line
//      ea   - linear address
//      n    - number of anterior additioal line (0..500)

void    DelExtLnA       (long ea,long n);       // delete an additional line before the generated ones


// Delete an additional posterior line
//      ea   - linear address
//      n    - number of posterior additioal line (0..500)

void    DelExtLnB       (long ea,long n);       // delete an additional line aftr  the generated ones


// Specify instruction represenation manually.
//      ea   - linear address
//      insn - a string represenation of the operand
// IDA will not check the specified instruction, it will simply display
// it instead of the orginal representation.

void    SetManualInsn   (long ea, char insn);


// Get manual representation of instruction
//      ea   - linear address
// This function returns value set by SetManualInsn earlier.

char    GetManualInsn   (long ea);



// Change value of a program byte
//      ea    - linear address
//      value - new value of the byte

void    PatchByte       (long ea,long value);   // change a byte


// Change value of a program word (2 bytes)
//      ea    - linear address
//      value - new value of the word

void    PatchWord       (long ea,long value);   // change a word (2 bytes)


// Change value of a double word
//      ea    - linear address
//      value - new value of the double word

void    PatchDword      (long ea,long value);   // change a dword (4 bytes)


// Set new value of flags
// This function should not used be used directly if possible.
// It changes properties of a program byte and if misused, may lead to
// very-very strange results.

void    SetFlags        (long ea,long flags);   // change internal flags for ea


// Set value of a segment register.
//      ea - linear address
//      reg - name of a register, like "cs", "ds", "es", etc.
//      value - new value of the segment register.
// IDA keeps tracks of all the points where segment register change their
// values. This function allows you to specify the correct value of a segment
// register if IDA is not able to find the corrent value.

success SetReg          (long ea,char reg,long value); // set value of segment register


// Plan to perform an action in the future.
// This function will put your request to a special autoanalysis queue.
// Later IDA will retrieve the request from the queue and process
// it. There are several autoanalysis queue types. IDA will process all
// queries from the first queue and then switch to the second queue, etc.

// plan/unplan range of addresses
void    AutoMark2       (long start,long end,long queuetype);
void    AutoUnmark      (long start,long end,long queuetype);

#endif

// plan to analyse an address
#define AutoMark(ea,qtype)      AutoMark2(ea,ea+1,qtype)

#define AU_UNK  10      // make unknown
#define AU_CODE 20      // convert to instruction
#define AU_PROC 30      // make function
#define AU_USED 40      // reanalyse
#define AU_LIBF 60      // apply a flirt signature (the current signature!)
#define AU_FINAL 200    // coagulate unexplored items

#ifdef _notdefinedsymbol


               P R O D U C E   O U T P U T   F I L E S



// Generate an output file
//      type  - type of output file. One of OFILE_... symbols. See below.
//      fp    - the output file handle
//      ea1   - start address. For some file types this argument is ignored
//      ea2   - end address. For some file types this argument is ignored
//      flags - bit combination of GENFLG_...
// returns: number of the generated lines.
//          -1 if an error occured
//          OFILE_EXE: 0-can't generate exe file, 1-ok

int GenerateFile(long type, long file_handle, long ea1, long ea2, long flags);
#endif

// output file types:

#define OFILE_MAP  0
#define OFILE_EXE  1
#define OFILE_IDC  2
#define OFILE_LST  3
#define OFILE_ASM  4
#define OFILE_DIF  5

// output control flags:

#define GENFLG_MAPSEGS 0x0001          // map: generate map of segments
#define GENFLG_MAPNAME 0x0002          // map: include dummy names
#define GENFLG_MAPDMNG 0x0004          // map: demangle names
#define GENFLG_IDCTYPE 0x0008          // idc: gen only information about types
#define GENFLG_ASMTYPE 0x0010          // asm&lst: gen information about types too

#ifdef _notdefinedsymbol


                 C O M M O N   I N F O R M A T I O N


// Get IDA directory
// This function returns the directory where IDA.EXE resides

char    GetIdaDirectory ();


// Get input file name
// This function returns name of the file being disassembled

char    GetInputFile    ();             // only the file name
char    GetInputFilePath();             // full path


// Get IDB full path
// This function returns full path of the current IDB database

char    GetIdbPath();


// Get internal flags
//      ea - linear address
// returns: 32-bit value of internal flags. See start of IDC.IDC file
// for explanations.

long    GetFlags        (long ea);              // get internal flags for ea


// Get value of program byte
//      ea - linear address
// returns: value of byte. If byte has not a value then returns 0xFF
// To check if a byte has a value, use functions hasValue(GetFlags(ea))

long    Byte            (long ea);              // get a byte at ea


// Get value of program word (2 bytes)
//      ea - linear address
// returns: value of word. If word has not a value then returns 0xFF

long    Word            (long ea);              // get a word (2 bytes) at ea


// Get value of program double word (4 bytes)
//      ea - linear address
// returns: value of double word. If double word has not a value
// then returns 0xFF

long    Dword           (long ea);              // get a double-word (4 bytes) at ea


// Get linear address of a name
//      from - the referring address.
//             Allows to retrieve local label addresses in functions.
//             If a local name is not found, then address of a global name is returned.
//      name - name of program byte
// returns: address of the name
//          BADADDR - no such name

long    LocByName       (char name);              // BADADDR - no such name
long    LocByNameEx     (long from, char name);   // BADADDR - no such name


// Get segment by segment base
//      base - segment base paragraph or selector
// returns: linear address of the start of the segment
//          BADADDR - no such segment

long    SegByBase       (long base);            // BADADDR - no such segment


// Get linear address of cursor

long    ScreenEA        ();                     // the current screen ea


// Get the disassembly line at the cursor

char    GetCurrentLine  ();


// Get start address of the selected area
// returns BADADDR - the user has not selected an area

long    SelStart        ();                     // the selected area start ea
                                                // BADADDR - no selected area


// Get end address of the selected area
// returns BADADDR - the user has not selected an area

long    SelEnd          ();                     // the selected area end ea
                                                // BADADDR - no selected area


// Get value of segment register at the specified address
//      ea - linear address
//      reg - name of segment register
// returns: 16bit value of the segment register. The segment registers in
// 32bit program usually contain selectors, so to get paragraph pointed by
// the segment register you need to call AskSelector() function.

long    GetReg          (long ea,char reg);     // get segment register value
                                                // 0xFFFF - undefined or error
                                                // (selector, use AskSelector() to
                                                //  get its mapping)


// Get next addresss in the program
//      ea - linear address
// returns: BADADDR - the specified address in the last used address

long    NextAddr        (long ea);              // returns next defined address
                                                // BADADDR if no such address exists


// Get previous addresss in the program
//      ea - linear address
// returns: BADADDR - the specified address in the first address

long    PrevAddr        (long ea);              // returns prev defined address
                                                // BADADDR if no such address exists


// Get next defined item (instruction or data) in the program
//      ea    - linear address to start search from
//      maxea - the search will stop at the address
//              maxea is not included in the search range
// returns: BADADDR - no (more) defined items

long    NextHead        (long ea, long maxea);  // returns next defined item address
                                                // BADADDR if no such address exists


// Get previous defined item (instruction or data) in the program
//      ea    - linear address to start search from
//      minea - the search will stop at the address
//              minea is included in the search range
// returns: BADADDR - no (more) defined items

long    PrevHead        (long ea, long minea);  // returns prev defined item address
                                                // BADADDR if no such address exists


// Get next not-tail address in the program
// This function searches for the next displayable address in the program.
// The tail bytes of instructions and data are not displayable.
//      ea - linear address
// returns: BADADDR - no (more) not-tail addresses

long    NextNotTail     (long ea);              // returns next not tail address
                                                // BADADDR if no such address exists


// Get previous not-tail address in the program
// This function searches for the previous displayable address in the program.
// The tail bytes of instructions and data are not displayable.
//      ea - linear address
// returns: BADADDR - no (more) not-tail addresses

long    PrevNotTail     (long ea);              // returns prev not tail address
                                                // BADADDR if no such address exists



// Get address of the end of the item (instruction or data)
//      ea - linear address
// returns: address past end of the item at 'ea'

long    ItemEnd         (long ea);              // returns address past end of
                                                // the item


// Get size of instruction or data item in bytes
//      ea - linear address
// returns: 1..n

long    ItemSize        (long ea);              // returns item size, min answer=1


// Get visible name of program byte
// This function returns name of byte as it is displayed on the screen.
// If a name contains illegal characters, IDA replaces them by the substitution
// character during displaying. See IDA.CFG for the definition of the
// substitution character.
//      from - the referring address.
//             Allows to retrieve local label addresses in functions.
//             If a local name is not found, then a global name is returned.
//      ea   - linear address
// returns: "" - byte has no name

char    Name            (long ea);              // get visible name of the byte
char    NameEx          (long from, long ea);   // get visible name of the byte


// Get true name of program byte
// This function returns name of byte as is without any replacements.
//      from - the referring address.
//             Allows to retrieve local label addresses in functions.
//             If a local name is not found, then a global name is returned.
//      ea   - linear address
// returns: "" - byte has no name

char    GetTrueName     (long ea);              // get true name of the byte
char    GetTrueNameEx   (long from, long ea);   // get true name of the byte


// Demangle a name
//      name - name to demangle
//      disable_mask - a mask that tells how to demangle the name
//                     it is a good idea to get this mask using
//                     GetLongPrm(INF_SHORT_DN) or GetLongPrm(INF_LONG_DN)
// Returns: a demangled name
// If the input name cannot be demangled, returns 0

char    Demangle        (char name, long disable_mask);


// Get instruction mnemonics
//      ea - linear address of instruction
// returns: "" - no instruction at the specified location
// note: this function may not return exactly the same mnemonics
// as you see on the screen.

char    GetMnem         (long ea);              // get instruction name


// Get operand of an instruction
//      ea - linear address of instruction
//      n  - number of operand:
//              0 - the first operand
//              1 - the second operand
// returns: the current text representation of operand

char    GetOpnd         (long ea,long n);       // get instruction operand
                                                // n=0 - first operand


// Get type of instruction operand
//      ea - linear address of instruction
//      n  - number of operand:
//              0 - the first operand
//              1 - the second operand
// returns:
//      -1      bad operand number passed
//      0       None
//      1       General Register
//      2       Memory Reference
//      3       Base + Index
//      4       Base + Index + Displacement
//      5       Immediate
//      6       Immediate Far Address (with a Segment Selector)
//      7       Immediate Near Address
//  PC:
//      8       386 Trace register
//      9       386 Debug register
//      10      386 Control register
//      11      FPP register
//      12      MMX register
// 8051:
//      8       bit
//      9       /bit
//      10      bit
// 80196:
//      8       [intmem]
//      9       [intmem]+
//      10      offset[intmem]
//      11      bit
// ARM:
//      8       shifted register
//      9       MLA operands
//      10      register list (for LDM/STM)
//      11      coprocessor register list (for CDP)
//      12      coprocessor register (for LDC/STC)
// PPC:
//      8       SPR
//      9       2 FPRs
//      10      SH & MB & ME
//      11      CR field
//      12      CR bit
// TMS320C5:
//      8       bit
//      9       bit not
//      10      condition
// TMS320C6:
//      8       register pair (A1:A0..B15:B14)
// Z8:
//      8       @intmem
//      9       @Rx
// Z80:
//      8       condition

long    GetOpType       (long ea,long n);       // get operand type


// Get number used in the operand
// This function returns an immediate number used in the operand
//      ea - linear address of instruction
//      n  - number of operand:
//              0 - the first operand
//              1 - the second operand
// If the operand doesn't contain a number, it returns -1.

long    GetOperandValue (long ea,long n);       // get instruction operand value


// Get anterior line
//      ea - linear address
//      num - number of anterior line (0..MAX_ITEM_LINES)
//            MAX_ITEM_LINES is defined in IDA.CFG

char    LineA           (long ea,long num);     // get additional line before generated ones


// Get posterior line
//      ea - linear address
//      num - number of posterior line (0..MAX_ITEM_LINES)

char    LineB           (long ea,long num);     // get additional line after generated ones


// Get regular indented comment
//      ea - linear address

char    Comment         (long ea);              // get comment


// Get repeatable indented comment
//      ea - linear address

char    RptCmt          (long ea);              // get repeatable comment

// Get manually entered operand string
//      ea - linear address
//      n  - number of operand:
//              0 - the first operand
//              1 - the second operand

char    AltOp           (long ea,long n);       // get manually entered operand

//
//      The following functions search for the specified byte
//          ea - address to start from
//          flag is combination of the following bits:
#endif
#define SEARCH_DOWN     0x01            // search forward
#define SEARCH_NEXT     0x02            // search next occurence
#define SEARCH_CASE     0x04            // search case-sensitive
                                        // (only for bin&txt search)
#define SEARCH_REGEX    0x08            // enable regular expressions
#define SEARCH_NOBRK    0x10            // don't test ctrl-break
#define SEARCH_NOSHOW   0x20            // don't display the search progress
#ifdef _notdefinedsymbol

//      returns BADADDR - not found
//
long    FindVoid        (long ea,long flag);
long    FindCode        (long ea,long flag);
long    FindData        (long ea,long flag);
long    FindUnexplored  (long ea,long flag);
long    FindExplored    (long ea,long flag);
long    FindImmediate   (long ea,long flag,long value);
long    FindText        (long ea,long flag,long y,long x,char str);
                // y - number of text line at ea to start from (0..MAX_ITEM_LINES)
                // x - x coordinate in this line
long    FindBinary      (long ea,long flag,char str);
                // str - a string as a user enters it for Search Text in Core
                //      example:  "41 42" - find 2 bytes 41h,42h
                // The default radix depends on the current IDP module
                // (radix for ibm pc is 16)


       G L O B A L   S E T T I N G S   M A N I P U L A T I O N


// The following functions allow you to set/get common parameters.
// Please note that not all parameters can be set directly.

long    GetLongPrm (long offset);
long    GetShortPrm(long offset);
long    GetCharPrm (long offset);
success SetLongPrm (long offset,long value);
success SetShortPrm(long offset,long value);
success SetCharPrm (long offset,long value);

#endif
// 'offset' may be one of the following:

#define INF_VERSION     3               // short;   Version of database
#define INF_PROCNAME    5               // char[8]; Name of current processor
#define INF_LFLAGS      13              // char;    IDP-dependent flags
#define         LFLG_PC_FPP     0x01    //              decode floating point processor
                                        //              instructions?
#define         LFLG_PC_FLAT    0x02    //              Flat model?
#define INF_DEMNAMES    14              // char;    display demangled names as:
#define         DEMNAM_CMNT  0          //              comments
#define         DEMNAM_NAME  1          //              regular names
#define         DEMNAM_NONE  2          //              don't display
#define INF_FILETYPE    15              // short;   type of input file (see ida.hpp)
#define         FT_EXE_OLD      0       //              MS DOS EXE File (obsolete)
#define         FT_COM_OLD      1       //              MS DOS COM File (obsolete)
#define         FT_BIN          2       //              Binary File
#define         FT_DRV          3       //              MS DOS Driver
#define         FT_WIN          4       //              New Executable (NE)
#define         FT_HEX          5       //              Intel Hex Object File
#define         FT_MEX          6       //              MOS Technology Hex Object File
#define         FT_LX           7       //              Linear Executable (LX)
#define         FT_LE           8       //              Linear Executable (LE)
#define         FT_NLM          9       //              Netware Loadable Module (NLM)
#define         FT_COFF         10      //              Common Object File Format (COFF)
#define         FT_PE           11      //              Portable Executable (PE)
#define         FT_OMF          12      //              Object Module Format
#define         FT_SREC         13      //              R-records
#define         FT_ZIP          14      //              ZIP file (this file is never loaded to IDA database)
#define         FT_OMFLIB       15      //              Library of OMF Modules
#define         FT_AR           16      //              ar library
#define         FT_LOADER       17      //              file is loaded using LOADER DLL
#define         FT_ELF          18      //              Executable and Linkable Format (ELF)
#define         FT_W32RUN       19      //              Watcom DOS32 Extender (W32RUN)
#define         FT_AOUT         20      //              Linux a.out (AOUT)
#define         FT_PRC          21      //              PalmPilot program file
#define         FT_EXE          22      //              MS DOS EXE File
#define         FT_COM          23      //              MS DOS COM File
#define         FT_AIXAR        24      //              AIX ar library
#define INF_OSTYPE      25              // short;   FLIRT: OS type the program is for
#define         OSTYPE_MSDOS 0x0001
#define         OSTYPE_WIN   0x0002
#define         OSTYPE_OS2   0x0004
#define         OSTYPE_NETW  0x0008
#define INF_APPTYPE     27              // short;   FLIRT: Application type
#define         APPT_CONSOLE 0x0001     //              console
#define         APPT_GRAPHIC 0x0002     //              graphics
#define         APPT_PROGRAM 0x0004     //              EXE
#define         APPT_LIBRARY 0x0008     //              DLL
#define         APPT_DRIVER  0x0010     //              DRIVER
#define         APPT_1THREAD 0x0020     //              Singlethread
#define         APPT_MTHREAD 0x0040     //              Multithread
#define         APPT_16BIT   0x0080     //              16 bit application
#define         APPT_32BIT   0x0100     //              32 bit application
#define INF_START_SP    29              // long;    SP register value at the start of
                                        //          program execution
#define INF_START_AF    33              // short;   Analysis flags:
#define         AF_FIXUP        0x0001  //              Create offsets and segments using fixup info
#define         AF_MARKCODE     0x0002  //              Mark typical code sequences as code
#define         AF_UNK          0x0004  //              Delete instructions with no xrefs
#define         AF_CODE         0x0008  //              Trace execution flow
#define         AF_PROC         0x0010  //              Create functions if call is present
#define         AF_USED         0x0020  //              Analyse and create all xrefs
#define         AF_FLIRT        0x0040  //              Use flirt signatures
#define         AF_PROCPTR      0x0080  //              Create function if data xref data->code32 exists
#define         AF_JFUNC        0x0100  //              Rename jump functions as j_...
#define         AF_NULLSUB      0x0200  //              Rename empty functions as nullsub_...
#define         AF_LVAR         0x0400  //              Create stack variables
#define         AF_TRACE        0x0800  //              Trace stack pointer
#define         AF_ASCII        0x1000  //              Create ascii string if data xref exists
#define         AF_IMMOFF       0x2000  //              Convert 32bit instruction operand to offset
#define         AF_DREFOFF      0x4000  //              Create offset if data xref to seg32 exists
#define         AF_FINAL        0x8000  //              Final pass of analysis
#define INF_START_IP    35              // long;    IP register value at the start of
                                        //          program execution
#define INF_BEGIN_EA    39              // long;    Linear address of program entry point
#define INF_MIN_EA      43              // long;    The lowest address used
                                        //          in the program
#define INF_MAX_EA      47              // long;    The highest address used
                                        //          in the program - 1
#define INF_LOW_OFF     59              // long;    low limit of voids
#define INF_HIGH_OFF    63              // long;    high limit of voids
#define INF_MAXREF      67              // long;    max xref depth
#define INF_ASCII_BREAK 71              // char;    ASCII line break symbol
#define INF_INDENT      73              // char;    Indention for instructions
#define INF_COMMENT     74              // char;    Indention for comments
#define INF_XREFNUM     75              // char;    Number of references to generate
                                        //          0 - xrefs won't be generated at all
#define INF_ENTAB       76              // char;    Use '\t' chars in the output file?
#define INF_VOIDS       78              // char;    Display void marks?
#define INF_SHOWAUTO    80              // char;    Display autoanalysis indicator?
#define INF_AUTO        81              // char;    Autoanalysis is enabled?
#define INF_BORDER      82              // char;    Generate borders?
#define INF_NULL        83              // char;    Generate empty lines?
#define INF_SHOWPREF    85              // char;    Show line prefixes?
#define INF_PREFSEG     86              // char;    line prefixes with segment name?
#define INF_ASMTYPE     87              // char;    target assembler number (0..n)
#define INF_BASEADDR    88              // long;    base paragraph of the program
#define INF_XREFS       92              // char;    xrefs representation:
#define         SW_SEGXRF       0x01    //              show segments in xrefs?
#define         SW_XRFMRK       0x02    //              show xref type marks?
#define         SW_XRFFNC       0x04    //              show function offsets?
#define         SW_XRFVAL       0x08    //              show xref values? (otherwise-"...")
#define INF_BINPREF     93              // short;   # of instruction bytes to show
                                        //          in line prefix
#define INF_CMTFLAG     95              // char;    comments:
#define         SW_RPTCMT       0x01    //              show repeatable comments?
#define         SW_ALLCMT       0x02    //              comment all lines?
#define         SW_NOCMT        0x04    //              no comments at all
#define         SW_LINNUM       0x08    //              show source line numbers
#define         SW_MICRO        0x10    //              show microcode (if implemented)
#define INF_NAMETYPE    96              // char;    dummy names represenation type
#define         NM_REL_OFF      0
#define         NM_PTR_OFF      1
#define         NM_NAM_OFF      2
#define         NM_REL_EA       3
#define         NM_PTR_EA       4
#define         NM_NAM_EA       5
#define         NM_EA           6
#define         NM_EA4          7
#define         NM_EA8          8
#define         NM_SHORT        9
#define         NM_SERIAL       10
#define INF_SHOWBADS    97              // char;    show bad instructions?
                                        //          an instruction is bad if it appears
                                        //          in the ash.badworks array

#define INF_PREFFLAG    98              // char;    line prefix type:
#define         PREF_SEGADR     0x01    //              show segment addresses?
#define         PREF_FNCOFF     0x02    //              show function offsets?
#define         PREF_STACK      0x04    //              show stack pointer?

#define INF_PACKBASE    99              // char;    pack database?

#define INF_ASCIIFLAGS  100             // uchar;   ascii flags
#define         ASCF_GEN        0x01    //              generate ASCII names?
#define         ASCF_AUTO       0x02    //              ASCII names have 'autogenerated' bit?
#define         ASCF_SERIAL     0x04    //              generate serial names?
#define         ASCF_COMMENT    0x10    //              generate auto comment for ascii references?
#define         ASCF_SAVECASE   0x20    //              preserve case of ascii strings for identifiers

#define INF_LISTNAMES   101             // uchar;   What names should be included in the list?
#define         LN_NORMAL       0x01    //              normal names
#define         LN_PUBLIC       0x02    //              public names
#define         LN_AUTO         0x04    //              autogenerated names
#define         LN_WEAK         0x08    //              weak names

#define INF_ASCIIPREF   102             // char[16];ASCII names prefix
#define INF_ASCIISERNUM 118             // ulong;   serial number
#define INF_ASCIIZEROES 122             // char;    leading zeroes
#define INF_MF          126             // uchar;   Byte order: 1==MSB first
#define INF_ORG         127             // char;    Generate 'org' directives?
#define INF_ASSUME      128             // char;    Generate 'assume' directives?
#define INF_CHECKARG    129             // char;    Check manual operands?
#define INF_START_SS    130             // long;    value of SS at the start
#define INF_START_CS    134             // long;    value of CS at the start
#define INF_MAIN        138             // long;    address of main()
#define INF_SHORT_DN    142             // long;    short form of demangled names
#define INF_LONG_DN     146             // long;    long form of demangled names
                                        //          see demangle.h for definitions
#define INF_DATATYPES   150             // long;    data types allowed in data carousel
#define INF_STRTYPE     154             // long;    current ascii string type
                                        //          is considered as several bytes:
                                        //      low byte:
#define         ASCSTR_TERMCHR  0       //              Character-terminated ASCII string
#define         ASCSTR_PASCAL   1       //              Pascal-style ASCII string (length byte)
#define         ASCSTR_LEN2     2       //              Pascal-style, length is 2 bytes
#define         ASCSTR_UNICODE  3       //              Unicode string
#define         ASCSTR_LEN4     4       //              Delphi string, length is 4 bytes
#define         ASCSTR_ULEN2    5       //              Pascal-style Unicode, length is 2 bytes
#define         ASCSTR_ULEN4    6       //              Pascal-style Unicode, length is 4 bytes
                                        //      2nd byte - termination chracters for ASCSTR_TERMCHR:
#define         STRTERM1(strtype)       ((strtype>>8)&0xFF)
                                        //      3d byte:
#define         STRTERM2(strtype)       ((strtype>>16)&0xFF)
                                        //              The termination characters are kept in
                                        //              the 2nd and 3d bytes of string type
                                        //              if the second termination character is
                                        //              '\0', then it is ignored.
#define INF_AF2         158             // ushort;  Analysis flags 2
#define         AF2_JUMPTBL     0x0001  //              Locate and create jump tables
#define         AF2_DODATA      0x0002  //              Coagulate data segs in final pass
#define         AF2_HFLIRT      0x0004  //              Automatically hide library functions
#define         AF2_STKARG      0x0008  //              Propagate stack argument information
#define         AF2_CHKUNI      0x0010  //              Check for unicode strings
#define INF_NAMELEN     160             // ushort;  max name length (without zero byte)
#define INF_MARGIN      162             // ushort;  max length of data lines
#define INF_LENXREF     164             // ushort;  max length of line with xrefs
#define INF_LPREFIX     166             // char[16];prefix of local names
                                        //          if a new name has this prefix,
                                        //          it will be automatically converted to a local name
#define INF_LPREFIXLEN  182             // uchar;   length of the lprefix
#define INF_COMPILER    183             // uchar;   compiler
#define INF_MODEL       184             // uchar;   memory model & calling convention
#define INF_SIZEOF_INT  185             // uchar;   sizeof(int)
#define INF_SIZEOF_BOOL 186             // uchar;   sizeof(bool)
#define INF_SIZEOF_ENUM 187             // uchar;   sizeof(enum)
#define INF_SIZEOF_ALGN 188             // uchar;   default alignment

#ifdef _notdefinedsymbol

-------------------------------------------------

// Change current processor
//      processor - name of processor in short form.
//                  run 'ida ?' to get list of allowed processor types

success SetPrcsr        (char processor);       // set processor type

// Enable/disable batch mode of operation
//      batch:  0 - ida will display dialog boxes and wait for the user input
//              1 - ida will not display dialog boxes, warnings, etc.
// returns: old balue of batch flag

long    Batch           (long batch);           // enable/disable batch mode
                                                // returns old value


          I N T E R A C T I O N   W I T H   T H E   U S E R


char    AskStr          (char defval,char prompt); // ask a string
char    AskFile         (long forsave,char mask,char prompt);   // ask a file name
long    AskAddr         (long defval,char prompt); // BADADDR - no or bad input
long    AskLong         (long defval,char prompt); // -1 - no or bad input
long    AskSeg          (long defval,char prompt); // BADSEL - no or bad input
                                                   // returns the segment selector (16bit)
char    AskIdent        (char defval,char prompt);
long    AskYN           (long defval,char prompt); // -1:cancel,0-no,1-ok
void    Message         (char format,...);      // show a message in messages window
void    Warning         (char format,...);      // show a warning a dialog box
void    Fatal           (char format,...);      // exit IDA immediately

// Change IDA indicator.
// Returns the previous status.
long    SetStatus       (long status);

#endif
#define IDA_STATUS_READY    0 // READY     IDA is idle
#define IDA_STATUS_THINKING 1 // THINKING  Analyzing but the user may press keys
#define IDA_STATUS_WAITING  2 // WAITING   Waiting for the user input
#define IDA_STATUS_WORK     3 // BUSY      IDA is busy
#ifdef _notdefinedsymbol


                       S E G M E N T A T I O N


***********************************************
** get a selector value
        arguments:      sel - the selector number (16bit value)
        returns:        selector value if found
                        otherwise the input value (sel)
        note:           selector values are always in paragraphs

long    AskSelector     (long sel);     // returns paragraph

***********************************************
** find a selector which has the specifed value
        arguments:      val - value to search for
        returns:        16bit selector number if found
                        otherwise the input value (val & 0xFFFF)
        note:           selector values are always in paragraphs

long    FindSelector    (long val);

***********************************************
** set a selector value
        arguments:      sel - 16bit selector number
                        (should be less than 0xFFFF)
                        val - value of selector
        returns:        nothing
        note:           ida supports up to 4096 selectors.
                        if 'sel' == 'val' then the
                        selector is destroyed because
                        it has no significance

void    SetSelector     (long sel,long value);

***********************************************
** delete a selector
        arguments:      sel - the selector number to delete
        returns:        nothing
        note:           if the selector is found, it will
                        be deleted

void    DelSelector     (long sel);

***********************************************
** SEGMENT FUNCTIONS

// Get first segment
// returns: linear address of the start of the first segment
// BADADDR - no segments are defined

long    FirstSeg        ();                     // returns start of the first
                                                // segment, BADADDR - no segments


// Get next segment
//      ea - linear address
// returns: start of the next segment
//          BADADDR - no next segment

long    NextSeg         (long ea);              // returns start of the next
                                                // segment, BADADDR - no more segs


// Get start address of a segment
//      ea - any address in the segment
// returns: start of segment
//          BADADDR - the specified address doesn't belong to any segment

long    SegStart        (long ea);              // returns start of the segment
                                                // BADADDR if bad address passed


// Get end address of a segment
//      ea - any address in the segment
// returns: end of segment (an address past end of the segment)
//          BADADDR - the specified address doesn't belong to any segment

long    SegEnd          (long ea);              // return end of the segment
                                                // this address doesn't belong
                                                // to the segment
                                                // BADADDR if bad address passed

// Get name of a segment
//      ea - any address in the segment
// returns: "" - no segment at the specified address

char    SegName         (long ea);              // returns name of the segment
                                                // "" if bad address passed


// Create a new segment
//      startea  - linear address of the start of the segment
//      endea    - linear address of the end of the segment
//                 this address will not belong to the segment
//                 'endea' should be higher than 'startea'
//      base     - base paragraph or selector of the segment.
//                 a paragraph is 16byte memory chunk.
//                 If a selector value is specified, the selector should be
//                 already defined.
//      use32    - 0: 16bit segment, 1: 32bit segment
//      align    - segment alignment. see below for alignment values
//      comb     - segment combination. see below for combination values.
// returns: 0-failed, 1-ok

success SegCreate(long startea,long endea,long base,
                                           long use32,long align,long comb);


// Delete a segment
//   ea      - any address in the segment
//   disable - 1: discard all bytes of the segment from the disassembled text
//             0: retain byte values

success SegDelete       (long ea,long disable);


// Change segment boundaries
//   ea      - any address in the segment
//   startea - new start address of the segment
//   endea   - new end address of the segment
//   disable - discard bytes that go out of the segment

success SegBounds       (long ea,long startea,long endea,long disable);


// Change name of the segment
//   ea      - any address in the segment
//   name    - new name of the segment

success SegRename       (long ea,char name);


// Change class of the segment
//   ea      - any address in the segment
//   class   - new class of the segment

success SegClass        (long ea,char class);


// Change alignment of the segment
//   ea      - any address in the segment
//   align   - new alignment of the segment

success SegAlign        (long ea,long alignment);

#endif
        #define saAbs      0    // Absolute segment.
        #define saRelByte  1    // Relocatable, byte aligned.
        #define saRelWord  2    // Relocatable, word (2-byte, 16-bit) aligned.
        #define saRelPara  3    // Relocatable, paragraph (16-byte) aligned.
        #define saRelPage  4    // Relocatable, aligned on 256-byte boundary (a "page"
                                // in the original Intel specification).
        #define saRelDble  5    // Relocatable, aligned on a double word (4-byte)
                                // boundary. This value is used by the PharLap OMF for
                                // the same alignment.
        #define saRel4K    6    // This value is used by the PharLap OMF for page (4K)
                                // alignment. It is not supported by LINK.
        #define saGroup    7    // Segment group
        #define saRel32Bytes 8  // 32 bytes
        #define saRel64Bytes 9  // 64 bytes
        #define saRelQword 10   // 8 bytes
#ifdef _notdefinedsymbol


// Change combination of the segment
//   ea      - any address in the segment
//   comb    - new combination of the segment

success SegComb         (long segea,long comb);
#endif
        #define scPriv     0    // Private. Do not combine with any other program
                                // segment.
        #define scPub      2    // Public. Combine by appending at an offset that meets
                                // the alignment requirement.
        #define scPub2     4    // As defined by Microsoft, same as C=2 (public).
        #define scStack    5    // Stack. Combine as for C=2. This combine type forces
                                // byte alignment.
        #define scCommon   6    // Common. Combine by overlay using maximum size.
        #define scPub3     7    // As defined by Microsoft, same as C=2 (public).
#ifdef _notdefinedsymbol


// Change segment addressing
//   ea      - any address in the segment
//   use32   - 0: 16bit, 1: 32bit

success SegAddrng       (long ea,long use32);


// Get segment by name
//      segname - name of segment
// returns: segment base address or BADADDR

long    SegByName       (char segname);         // returns segment base


// Set default segment register value for a segment
//   ea      - any address in the segment
//             if no segment is present at the specified address
//             then all segments will be affected
//   reg     - name of segment register
//   value   - default value of segment register. -1-undefined.

success SegDefReg       (long ea,char reg,long value);

***********************************************
** set segment type
        arguments:      segea - any address within segment
                        type  - new segment type:
#endif
#define SEG_NORM        0
#define SEG_XTRN        1       // * segment with 'extern' definitions
                                //   no instructions are allowed
#define SEG_CODE        2       // pure code segment
#define SEG_DATA        3       // pure data segment
#define SEG_IMP         4       // implementation segment
#define SEG_GRP         6       // * group of segments
                                //   no instructions are allowed
#define SEG_NULL        7       // zero-length segment
#define SEG_UNDF        8       // undefined segment type
#define SEG_BSS         9       // uninitialized segment
#define SEG_ABSSYM     10       // * segment with definitions of absolute symbols
                                //   no instructions are allowed
#define SEG_COMM       11       // * segment with communal definitions
                                //   no instructions are allowed
#ifdef _notdefinedsymbol
        returns:        !=0 - ok

success SetSegmentType  (long segea,long type);


***********************************************
** get segment attribute
        arguments:      segea - any address within segment
                        attr  - one of SEGATTR_... constants

long    GetSegmentAttr  (long segea,long attr);

#endif
#define  SEGATTR_ALIGN  20      // alignment
#define  SEGATTR_COMB   21      // combination
#define  SEGATTR_PERM   22      // permissions
#define  SEGATTR_USE32  23      // use32 (32-bit segment?)
#define  SEGATTR_FLAGS  24      // segment flags
#define  SEGATTR_SEL    26      // segment selector
#define  SEGATTR_DEF_ES 28      // default ES value
#define  SEGATTR_DEF_CS 30      // default CS value
#define  SEGATTR_DEF_SS 32      // default SS value
#define  SEGATTR_DEF_DS 34      // default DS value
#define  SEGATTR_DEF_FS 36      // default FS value
#define  SEGATTR_DEF_GS 38      // default GS value
#define  SEGATTR_TYPE   40      // segment type
#ifdef _notdefinedsymbol



                   C R O S S   R E F E R E N C E S


//      Flow types:
#endif
#define fl_CF   16              // Call Far
#define fl_CN   17              // Call Near
#define fl_JF   18              // Jump Far
#define fl_JN   19              // Jump Near
#define fl_US   20              // User specified
#define fl_F    21              // Ordinary flow
#ifdef _notdefinedsymbol
                                        // Mark exec flow 'from' 'to'
void    AddCodeXref(long From,long To,long flowtype);
long    DelCodeXref(long From,long To,int undef);// Unmark exec flow 'from' 'to'
                                        // undef - make 'To' undefined if no
                                        //        more references to it
                                        // returns 1 - planned to be
                                        // made undefined

// The following functions include the ordinary flows:
// (the ordinary flow references are returned first)
long    Rfirst  (long From);            // Get first xref from 'From'
long    Rnext   (long From,long current);// Get next xref from
long    RfirstB (long To);              // Get first xref to 'To'
long    RnextB  (long To,long current); // Get next xref to 'To'

// The following functions don't take into account the ordinary flows:
long    Rfirst0 (long From);
long    Rnext0  (long From,long current);
long    RfirstB0(long To);
long    RnextB0 (long To,long current);

//      Data reference types:
#endif
#define dr_O    1                       // Offset
#define dr_W    2                       // Write
#define dr_R    3                       // Read
#define dr_T    4                       // Text (names in manual operands)
#ifdef _notdefinedsymbol

void    add_dref(long From,long To,long drefType);      // Create Data Ref
void    del_dref(long From,long To);    // Unmark Data Ref

long    Dfirst  (long From);            // Get first refered address
long    Dnext   (long From,long current);
long    DfirstB (long To);              // Get first referee address
long    DnextB  (long To,long current);

long    XrefType(void);                 // returns type of the last xref
                                        // obtained by [RD]first/next[B0]
                                        // functions. Return values
                                        // are fl_... or dr_...

                           F I L E   I / O


***********************************************
** open a file
        arguments: similiar to C fopen()
        returns:        0 -error
                        otherwise a file handle

long    fopen           (char file,char mode);

***********************************************
** close a file
        arguments:      file handle
        returns:        nothing

void    fclose          (long handle);

***********************************************
** get file length
        arguments:      file handle
        returns:        -1 - error
                        otherwise file length in bytes

long    filelength      (long handle);

***********************************************
** set cursor position in the file
        arguments:      handle  - file handle
                        offset  - offset from origin
                        origin  - 0 = from the start of file
                                  1 = from the current cursor position
                                  2 = from the end of file
        returns:        0 - ok
                        otherwise error

long    fseek           (long handle,long offset,long origin);

***********************************************
** get cursor position in the file
        arguments:      file handle
        returns:        -1 - error
                        otherwise current cursor position

long    ftell           (long handle);

***********************************************
** load file into IDA database
        arguments:      handle  - file handle
                        pos     - position in the file
                        ea      - linear address to load
                        size    - number of bytes to load
        returns:        0 - error
                        1 - ok

success loadfile        (long handle,long pos,long ea,long size);

***********************************************
** save from IDA database to file
        arguments:      handle  - file handle
                        pos     - position in the file
                        ea      - linear address to save from
                        size    - number of bytes to save
        returns:        0 - error
                        1 - ok

success savefile        (long handle,long pos,long ea,long size);

***********************************************
** read one byte from file
        arguments:      handle  - file handle
        returns:        -1 - error
                        otherwise a byte read.

long    fgetc           (long handle);

***********************************************
** write one byte to file
        arguments:      handle  - file handle
                        byte    - byte to write
        returns:        0 - ok
                        -1 - error

long    fputc           (long byte,long handle);

***********************************************
** fprintf
        arguments:      handle  - file handle
                        format  - format string
        returns:        0 - ok
                        -1 - error

long    fprintf         (long handle,char format,...);

***********************************************
** read 2 bytes from file
        arguments:      handle  - file hanlde
                        mostfirst 0 - least significant byte is first (intel)
                                  1 - most  significant byte is first
        returns:        -1 - error
                        otherwise: a 16-bit value

long    readshort       (long handle,long mostfirst);

***********************************************
** read 4 bytes from file
        arguments:      handle  - file hanlde
                        mostfirst 0 - least significant byte is first (intel)
                                  1 - most  significant byte is first
        returns:        a 32-bit value

long    readlong        (long handle,long mostfirst);

***********************************************
** write 2 bytes to file
        arguments:      handle  - file hanlde
                        word    - a 16-bit value to write
                        mostfirst 0 - least significant byte is first (intel)
                                  1 - most  significant byte is first
        returns:        0 - ok

long    writeshort      (long handle,long word,long mostfirst);

***********************************************
** write 4 bytes to file
        arguments:      handle  - file hanlde
                        dword   - a 32-bit value to write
                        mostfirst 0 - least significant byte is first (intel)
                                  1 - most  significant byte is first
        returns:        0 - ok

long    writelong       (long handle,long dword,long mostfirst);

***********************************************
** read a string from file
        arguments:      handle  - file hanlde
        returns:        a string
                        on EOF, returns -1

char    readstr         (long handle);

***********************************************
** write a string to file
        arguments:      handle  - file hanlde
                        str     - string to write
        returns:        0 - ok

long    writestr        (long handle,char str);


                          F U N C T I O N S


***********************************************
** create a function
        arguments:      start,end - function bounds
                        If the function end address is BADADDR, then
                        IDA will try to determine the function bounds
                        automatically. IDA will define all necessary
                        instructions to determine the function bounds.
        returns:        !=0 - ok

success MakeFunction(long start,long end);

***********************************************
** delete a function
        arguments:      ea - any address belonging to the function
        returns:        !=0 - ok

success DelFunction(long ea);

***********************************************
** change function end address
        arguments:      ea - any address belonging to the function
                        end - new function end address
        returns:        !=0 - ok

success SetFunctionEnd(long ea,long end);

***********************************************
** find next function
        arguments:      ea - any address belonging to the function
        returns:        -1 - no more functions
                        otherwise returns the next function start address

long NextFunction(long ea);

***********************************************
** find previous function
        arguments:      ea - any address belonging to the function
        returns:        -1 - no more functions
                        otherwise returns the previous function start address

long PrevFunction(long ea)

***********************************************
** retrieve function flags
        arguments:      ea - any address belonging to the function
        returns:        -1 - function doesn't exist
                        otherwise returns the flags:
#endif
#define FUNC_NORET      0x00000001L     // function doesn't return
#define FUNC_FAR        0x00000002L     // far function
#define FUNC_LIB        0x00000004L     // library function
#define FUNC_STATIC     0x00000008L     // static function
#define FUNC_FRAME      0x00000010L     // function uses frame pointer (BP)
#define FUNC_USERFAR    0x00000020L     // user has specified far-ness
                                        // of the function
#define FUNC_HIDDEN     0x00000040L     // a hidden function
#ifdef _notdefinedsymbol

long GetFunctionFlags(long ea);

***********************************************
** change function flags
        arguments:      ea - any address belonging to the function
                        flags - see GetFunctionFlags() for explanations
        returns:        !=0 - ok

success SetFunctionFlags(long ea,long flags);

***********************************************
** retrieve function name
        arguments:      ea - any address belonging to the function
        returns:        null string - function doesn't exist
                        otherwise returns function name

char GetFunctionName(long ea);

***********************************************
** retrieve function comment
        arguments:      ea - any address belonging to the function
                        repeatable - 1: get repeatable comment
                                     0: get regular comment
        returns:        function comment string

char GetFunctionCmt(long ea, long repeatable);

***********************************************
** set function comment
        arguments:      ea - any address belonging to the function
                        cmt - a function comment line
                        repeatable - 1: get repeatable comment
                                     0: get regular comment

void SetFunctionCmt(long ea, char cmt, long repeatable);

***********************************************
** ask the user to select a function
        arguments:      title - title of the dialog box
        returns:        -1 - user refused to select a function
                        otherwise returns the selected function start address

long ChooseFunction(char title);

***********************************************
** convert address to 'funcname+offset' string
        arguments:      ea - address to convert
        returns:        if the address belongs to a function then
                          return a string formed as 'name+offset'
                          where 'name' is a function name
                          'offset' is offset within the function
                        else
                          return null string

char GetFuncOffset(long ea);

***********************************************
** Determine a new function boundaries
**
        arguments:      ea  - starting address of a new function
        returns:        if a function already exists, then return
                        its end address.
                        if a function end cannot be determined,
                        the return BADADDR
                        otherwise return the end address of the new function

long FindFuncEnd(long ea);

***********************************************
** Get ID of function frame structure
**
        arguments:      ea - any address belonging to the function
        returns:        ID of function frame or -1
                        In order to access stack variables you need to use
                        structure member manipulaion functions with the
                        obtained ID.

long GetFrame(long ea);

***********************************************
** Get size of local variables in function frame
**
        arguments:      ea - any address belonging to the function
        returns:        Size of local variables in bytes.
                        If the function doesn't have a frame, return 0
                        If the function does't exist, return -1

long GetFrameLvarSize(long ea);

***********************************************
** Get size of saved registers in function frame
**
        arguments:      ea - any address belonging to the function
        returns:        Size of saved registers in bytes.
                        If the function doesn't have a frame, return 0
                        This value is used as offset for BP
                        (if FUNC_FRAME is set)
                        If the function does't exist, return -1

long GetFrameRegsSize(long ea);

***********************************************
** Get size of arguments in function frame which are purged upon return
**
        arguments:      ea - any address belonging to the function
        returns:        Size of function arguments in bytes.
                        If the function doesn't have a frame, return 0
                        If the function does't exist, return -1

long GetFrameArgsSize(long ea);

***********************************************
** Get full size of function frame
**
        arguments:      ea - any address belonging to the function
        returns:        Size of function frame in bytes.
                        This function takes into account size of local
                        variables + size of saved registers + size of
                        return address + size of function arguments
                        If the function doesn't have a frame, return size of
                        function return address in the stack.
                        If the function does't exist, return 0

long GetFrameSize(long ea);

***********************************************
** Make function frame
**
        arguments:      ea      - any address belonging to the function
                        lvsize  - size of function local variables
                        frregs  - size of saved registers
                        argsize - size of function arguments
        returns:        ID of function frame or -1
                        If the function did not have a frame, the frame
                        will be created. Otherwise the frame will be
                        modified

long MakeFrame(long ea,long lvsize,long frregs,long argsize);

***********************************************
** Get current delta for the stack pointer
**
        arguments:      ea      - end address of the instruction
                                  i.e.the last address of the instruction+1
        returns:        The difference between the original SP upon
                        entering the function and SP for
                        the specified address

long GetSpd(long ea);

***********************************************
** Get modification of SP made by the instruction
**
        arguments:      ea      - end address of the instruction
                                  i.e.the last address of the instruction+1
        returns:        Get modification of SP made at the specified location
                        If the specified location doesn't contain a SP
                        change point, return 0
                        Otherwise return delta of SP modification

long GetSpDiff(long ea);

***********************************************
** Setup modification of SP made by the instruction
**
        arguments:      ea      - end address of the instruction
                                  i.e.the last address of the instruction+1
                        delta   - the difference made by the current
                                  instruction.
        returns:        1-ok, 0-failed

success SetSpDiff(long ea,long delta);


                       E N T R Y   P O I N T S


***********************************************
** retrieve number of entry points
        arguments:      none
        returns:        number of entry points

long GetEntryPointQty(void);

***********************************************
** add entry point
        arguments:      ordinal  - entry point number
                                   if entry point doesn't have an ordinal
                                   number, 'ordinal' should be equal to 'ea'
                        ea       - address of the entry point
                        name     - name of the entry point. If null string,
                                   the entry point won't be renamed.
                        makecode - if 1 then this entry point is a start
                                   of a function. Otherwise it denotes data
                                   bytes.
        returns:        0 - entry point with the specifed ordinal already
                                exists
                        1 - ok

success AddEntryPoint(long ordinal,long ea,char name,long makecode);

***********************************************
** retrieve entry point ordinal number
        arguments:      index - 0..GetEntryPointQty()-1
        returns:        0 if entry point doesn't exist
                        otherwise entry point ordinal

long GetEntryOrdinal(long index);

***********************************************
** retrieve entry point address
        arguments:      ordinal - entry point number
                                  it is returned by GetEntryPointOrdinal()
        returns:        -1 if entry point doesn't exist
                        otherwise entry point address.
                        If entry point address is equal to its ordinal
                        number, then the entry point has no ordinal.

long GetEntryPoint(long ordinal);

***********************************************
** rename entry point
        arguments:      ordinal - entry point number
                        name    - new name
        returns:        !=0 - ok

success RenameEntryPoint(long ordinal,char name);



                             F I X U P S


***********************************************
** find next address with fixup information
        arguments:      ea - current address
        returns:        -1 - no more fixups
                        otherwise returns the next address with
                                                fixup information

long GetNextFixupEA(long ea);

***********************************************
** find previous address with fixup information
        arguments:      ea - current address
        returns:        -1 - no more fixups
                        otherwise returns the previous address with
                                                fixup information

long GetPrevFixupEA(long ea);

***********************************************
** get fixup target type
        arguments:      ea - address to get information about
        returns:        -1 - no fixup at the specified address
                        otherwise returns fixup target type:
#endif
#define FIXUP_MASK      0xF
#define FIXUP_BYTE      FIXUP_OFF8 // 8-bit offset.
#define FIXUP_OFF8      0       // 8-bit offset.
#define FIXUP_OFF16     1       // 16-bit offset.
#define FIXUP_SEG16     2       // 16-bit base--logical segment base (selector).
#define FIXUP_PTR32     3       // 32-bit long pointer (16-bit base:16-bit
                                // offset).
#define FIXUP_OFF32     4       // 32-bit offset.
#define FIXUP_PTR48     5       // 48-bit pointer (16-bit base:32-bit offset).
#define FIXUP_HI8       6       // high  8 bits of 16bit offset
#define FIXUP_HI16      7       // high 16 bits of 32bit offset
#define FIXUP_LOW8      8       // low   8 bits of 16bit offset
#define FIXUP_LOW16     9       // low  16 bits of 32bit offset
#define FIXUP_REL       0x10    // fixup is relative to the linear address
                                // specified in the 3d parameter to set_fixup()
#define FIXUP_SELFREL   0x0     // self-relative?
                                //   - disallows the kernel to convert operands
                                //      in the first pass
                                //   - this fixup is used during output
                                // This type of fixups is not used anymore.
                                // Anyway you can use it for commenting purposes
                                // in the loader modules
#define FIXUP_EXTDEF    0x20    // target is a location (otherwise - segment)
#define FIXUP_UNUSED    0x40    // fixup is ignored by IDA
                                //   - disallows the kernel to convert operands
                                //   - this fixup is not used during output
#define FIXUP_CREATED   0x80    // fixup was not present in the input file
#ifdef _notdefinedsymbol

long GetFixupTgtType(long ea);

***********************************************
** get fixup target selector
        arguments:      ea - address to get information about
        returns:        -1 - no fixup at the specified address
                        otherwise returns fixup target selector

long GetFixupTgtSel(long ea);

***********************************************
** get fixup target offset
        arguments:      ea - address to get information about
        returns:        -1 - no fixup at the specified address
                        otherwise returns fixup target offset

long GetFixupTgtOff(long ea);

***********************************************
** get fixup target displacement
        arguments:      ea - address to get information about
        returns:        -1 - no fixup at the specified address
                        otherwise returns fixup target displacement

long GetFixupTgtDispl(long ea);

***********************************************
** set fixup information
        arguments:      ea        - address to set fixup information about
                        type      - fixup type. see GetFixupTgtType()
                                    for possible fixup types.
                        targetsel - target selector
                        targetoff - target offset
                        displ     - displacement
        returns:        none

void SetFixup(long ea,long type,long targetsel,long targetoff,long displ);

***********************************************
** delete fixup information
        arguments:      ea - address to delete fixup information about
        returns:        none

void DelFixup(long ea);


                   M A R K E D   P O S I T I O N S


***********************************************
** mark position
        arguments:      ea      - address to mark
                        lnnum   - number of generated line for the 'ea'
                        x       - x coordinate of cursor
                        y       - y coordinate of cursor
                        slot    - slot number: 1..20
                                  if the specifed value is not within the
                                  range, IDA will ask the user to select slot.
                        comment - description of the mark.
                                  Should be not empty.
        returns:        none

void MarkPosition(long ea,long lnnum,long x,long y,long slot,char comment);

***********************************************
** get marked position
        arguments:      slot    - slot number: 1..20
                                  if the specifed value is <= 0
                                  range, IDA will ask the user to select slot.
        returns:        -1 - the slot doesn't contain a marked address
                        otherwise returns the marked address

long GetMarkedPos(long slot);

***********************************************
** get marked position comment
        arguments:      slot    - slot number: 1..20
        returns:        null string if the slot doesn't contain
                                        a marked address
                        otherwise returns the marked address comment

char GetMarkComment(long slot);


                         S T R U C T U R E S


***********************************************
** get number of defined structure types
        arguments:      none
        returns:        number of structure types

long GetStrucQty(void);

***********************************************
** get index of first structure type
        arguments:      none
        returns:        -1 if no structure type is defined
                        index of first structure type.
                        Each structure type has an index and ID.
                        INDEX determines position of structure definition
                         in the list of structure definitions. Index 1
                         is listed first, after index 2 and so on.
                         The index of a structure type can be changed any
                         time, leading to movement of the structure definition
                         in the list of structure definitions.
                        ID uniquely denotes a structure type. A structure
                         gets a unique ID at the creation time and this ID
                         can't be changed. Even when the structure type gets
                         deleted, its ID won't be resued in the future.

long GetFirstStrucIdx(void);

***********************************************
** get index of last structure type
        arguments:      none
        returns:        -1 if no structure type is defined
                        index of last structure type.
                        See GetFirstStrucIdx() for the explanation of
                        structure indices and IDs.

long GetLastStrucIdx(void);

***********************************************
** get index of next structure type
        arguments:      current structure index
        returns:        -1 if no (more) structure type is defined
                        index of the next structure type.
                        See GetFirstStrucIdx() for the explanation of
                        structure indices and IDs.

long GetNextStrucIdx(long index);

***********************************************
** get index of previous structure type
        arguments:      current structure index
        returns:        -1 if no (more) structure type is defined
                        index of the presiouvs structure type.
                        See GetFirstStrucIdx() for the explanation of
                        structure indices and IDs.

long GetPrevStrucIdx(long index);

***********************************************
** get structure index by structure ID
        arguments:      structure ID
        returns:        -1 if bad structure ID is passed
                        otherwise returns structure index.
                        See GetFirstStrucIdx() for the explanation of
                        structure indices and IDs.

long GetStrucIdx(long id);

***********************************************
** get structure ID by structure index
        arguments:      structure index
        returns:        -1 if bad structure index is passed
                        otherwise returns structure ID.
                        See GetFirstStrucIdx() for the explanation of
                        structure indices and IDs.

long GetStrucId(long index);

***********************************************
** get structure ID by structure name
        arguments:      structure type name
        returns:        -1 if bad structure type name is passed
                        otherwise returns structure ID.

long GetStrucIdByName(char name);


***********************************************
** get structure type name
        arguments:      structure type ID
        returns:        -1 if bad structure type ID is passed
                        otherwise returns structure type name.

char GetStrucName(long id);

***********************************************
** get structure type comment
        arguments:      id         - structure type ID
                        repeatable - 1: get repeatable comment
                                     0: get regular comment
        returns:        null string if bad structure type ID is passed
                        otherwise returns comment.

char GetStrucComment(long id,long repeatable);

***********************************************
** get size of a structure
        arguments:      id         - structure type ID
        returns:        -1 if bad structure type ID is passed
                        otherwise returns size of structure in bytes.

long GetStrucSize(long id);

***********************************************
** get number of members of a structure
        arguments:      id         - structure type ID
        returns:        -1 if bad structure type ID is passed
                        otherwise returns number of members.

long GetMemberQty(long id);

***********************************************
** get previous offset in a structure
        arguments:      id     - structure type ID
                        offset - current offset
        returns:        -1 if bad structure type ID is passed
                           or no (more) offsets in the structure
                        otherwise returns previous offset in a structure.
                        NOTE: IDA allows 'holes' between members of a
                              structure. It treats these 'holes'
                              as unnamed arrays of bytes.
                        This function returns a member offset or a hole offset.
                        It will return size of the structure if input
                        'offset' is bigger than the structure size.

long GetStrucPrevOff(long id,long offset);

***********************************************
** get next offset in a structure
        arguments:      id     - structure type ID
                        offset - current offset
        returns:        -1 if bad structure type ID is passed
                           or no (more) offsets in the structure
                        otherwise returns next offset in a structure.
                        NOTE: IDA allows 'holes' between members of a
                              structure. It treats these 'holes'
                              as unnamed arrays of bytes.
                        This function returns a member offset or a hole offset.
                        It will return size of the structure if input
                        'offset' belongs to the last member of the structure.

long GetStrucNextOff(long id,long offset);

***********************************************
** get offset of the first member of a structure
        arguments:      id            - structure type ID
        returns:        -1 if bad structure type ID is passed
                           or structure has no members
                        otherwise returns offset of the first member.
                        NOTE: IDA allows 'holes' between members of a
                              structure. It treats these 'holes'
                              as unnamed arrays of bytes.

long GetFirstMember(long id);

***********************************************
** get offset of the last member of a structure
        arguments:      id            - structure type ID
        returns:        -1 if bad structure type ID is passed
                           or structure has no members
                        otherwise returns offset of the last member.
                        NOTE: IDA allows 'holes' between members of a
                              structure. It treats these 'holes'
                              as unnamed arrays of bytes.

long GetLastMember(long id);

***********************************************
** get offset of a member of a structure by the member name
        arguments:      id            - structure type ID
                        member_name   - name of structure member
        returns:        -1 if bad structure type ID is passed
                           or no such member in the structure
                        otherwise returns offset of the specified member.

long GetMemberOffset(long id,char member_name);

***********************************************
** get name of a member of a structure
        arguments:      id            - structure type ID
                        member_offset - member offset. The offset can be
                                        any offset in the member. For example,
                                        is a member is 4 bytes long and starts
                                        at offset 2, then 2,3,4,5 denote
                                        the same structure member.
        returns:        -1 if bad structure type ID is passed
                           or no such member in the structure
                        otherwise returns name of the specified member.

char GetMemberName(long id,long member_offset);

***********************************************
** get comment of a member
        arguments:      id            - structure type ID
                        member_offset - member offset. The offset can be
                                        any offset in the member. For example,
                                        is a member is 4 bytes long and starts
                                        at offset 2, then 2,3,4,5 denote
                                        the same structure member.
                        repeatable - 1: get repeatable comment
                                     0: get regular comment
        returns:        null string if bad structure type ID is passed
                           or no such member in the structure
                        otherwise returns comment of the specified member.

char GetMemberComment(long id,long member_offset,long repeatable);

***********************************************
** get size of a member
        arguments:      id            - structure type ID
                        member_offset - member offset. The offset can be
                                        any offset in the member. For example,
                                        is a member is 4 bytes long and starts
                                        at offset 2, then 2,3,4,5 denote
                                        the same structure member.
        returns:        -1 if bad structure type ID is passed
                           or no such member in the structure
                        otherwise returns size of the specified
                                          member in bytes.

long GetMemberSize(long id,long member_offset);

***********************************************
** get type of a member
        arguments:      id            - structure type ID
                        member_offset - member offset. The offset can be
                                        any offset in the member. For example,
                                        is a member is 4 bytes long and starts
                                        at offset 2, then 2,3,4,5 denote
                                        the same structure member.
        returns:        -1 if bad structure type ID is passed
                           or no such member in the structure
                        otherwise returns type of the member, see bit
                        definitions above. If the member type is a structure
                        then function GetMemberStrid() should be used to
                        get the structure type id.

long GetMemberFlag(long id,long member_offset);

***********************************************
** get structure id of a member
        arguments:      id            - structure type ID
                        member_offset - member offset. The offset can be
                                        any offset in the member. For example,
                                        is a member is 4 bytes long and starts
                                        at offset 2, then 2,3,4,5 denote
                                        the same structure member.
        returns:        -1 if bad structure type ID is passed
                           or no such member in the structure
                        otherwise returns structure id of the member.
                        If the current member is not a structure, returns -1.

long GetMemberStrId(long id,long member_offset);

***********************************************
** is a structure a union?
	arguments:	id            - structure type ID
        returns:        1: yes, this is a union id
                        0: no

                        Unions are a special kind of structures

long IsUnion(long id);

***********************************************
** define a new structure type
        arguments:      index         - index of new structure type
                        If another structure has the specified index,
                        then index of that structure and all other
                        structures will be increentedfreeing the specifed
                        index. If index is == -1, then the biggest index
                        number will be used.
                        See GetFirstStrucIdx() for the explanation of
                        structure indices and IDs.

                        name - name of the new structure type.

                        is_union - 0: structure
                                   1: union

        returns:        -1 if can't define structure type because of
                        bad structure name: the name is ill-formed or is
                        already used in the program.
                        otherwise returns ID of the new structure type

long AddStrucEx(long index,char name,long is_union);

***********************************************
** delete a structure type
        arguments:      id            - structure type ID
        returns:        0 if bad structure type ID is passed
                        1 otherwise the structure type is deleted. All data
                        and other structure types referencing to the
                        deleted structure type will be displayed as array
                        of bytes.

success DelStruc(long id);

***********************************************
** change structure index
        arguments:      id      - structure type ID
                        index   - new index of the structure
                        See GetFirstStrucIdx() for the explanation of
                        structure indices and IDs.
        returns:        !=0 - ok

long SetStrucIdx(long id,long index);

***********************************************
** change structure name
        arguments:      id      - structure type ID
                        name    - new name of the structure
        returns:        !=0 - ok

long SetStrucName(long id,char name);

***********************************************
** change structure comment
        arguments:      id      - structure type ID
                        comment - new comment of the structure
                        repeatable - 1: change repeatable comment
                                     0: change regular comment
        returns:        !=0 - ok

long SetStrucComment(long id,char comment,long repeatable);

***********************************************
** add structure member
        arguments:      id      - structure type ID
                        name    - name of the new member
                        offset  - offset of the new member
                                  -1 means to add at the end of the structure
                        flag    - type of the new member. Should be one of
                                  FF_BYTE..FF_PACKREAL (see above)
                                  combined with FF_DATA
                        typeid  - structure id if 'flag' == FF_STRU
                                  Denotes type of the member if the member
                                  itself is a structure.
                                  if isOff0(flag) then typeid specifies
                                  the offset base.
                                  if isASCII(flag) then typeid specifies
                                  the string type (ASCSTR_...).
                                  if isStroff(flag) then typeid specifies
                                  the structure id
                                  Otherwise should be -1.
                        nbytes  - number of bytes in the new member
        returns:        0 - ok, otherwise error code:
#endif
#define STRUC_ERROR_MEMBER_NAME    (-1) // already has member with this name (bad name)
#define STRUC_ERROR_MEMBER_OFFSET  (-2) // already has member at this offset
#define STRUC_ERROR_MEMBER_SIZE    (-3) // bad number of bytes or bad sizeof(type)
#define STRUC_ERROR_MEMBER_TINFO   (-4) // bad typeid parameter
#define STRUC_ERROR_MEMBER_STRUCT  (-5) // bad struct id (the 1st argument)
#define STRUC_ERROR_MEMBER_UNIVAR  (-6) // unions can't have variable sized members
#define STRUC_ERROR_MEMBER_VARLAST (-7) // variable sized member should be the last member in the structure
#ifdef _notdefinedsymbol

long AddStrucMember(long id,char name,long offset,long flag,
                    long typeid,long nbytes);

***********************************************
** delete structure member
        arguments:      id            - structure type ID
                        member_offset - offset of the member
        returns:        !=0 - ok.
                        NOTE: IDA allows 'holes' between members of a
                              structure. It treats these 'holes'
                              as unnamed arrays of bytes.

long DelStrucMember(long id,long member_offset);

***********************************************
** change structure member name
        arguments:      id            - structure type ID
                        member_offset - offset of the member
                        name          - new name of the member
        returns:        !=0 - ok.

long SetMemberName(long id,long member_offset,char name);

***********************************************
** change structure member type
        arguments:      id            - structure type ID
                        member_offset - offset of the member
                        flag    - new type of the member. Should be one of
                                  FF_BYTE..FF_PACKREAL (see above)
                                  combined with FF_DATA
                        typeid  - structure id if 'flag' == FF_STRU
                                  Denotes type of the member is the member
                                  itself is a structure. Otherwise should be
                                  -1.
                                  if isOff0(flag) then typeid specifies
                                  the offset base.
                                  if isASCII(flag) then typeid specifies
                                  the string type (ASCSTR_...).
                        nitems  - number of items in the member
        returns:        !=0 - ok.

long SetMemberType(long id,long member_offset,long flag,long typeid,long nitems);

***********************************************
** change structure member comment
        arguments:      id      - structure type ID
                        member_offset - offset of the member
                        comment - new comment of the structure member
                        repeatable - 1: change repeatable comment
                                     0: change regular comment
        returns:        !=0 - ok

long SetMemberComment(long id,long member_offset,char comment,long repeatable);


                         E N U M S


***********************************************
** get number of enum types
        arguments:      none
        returns:        number of enumerations

long GetEnumQty(void);

***********************************************
** get ID of the specified enum by its serial number
        arguments:      idx - number of enum (0..GetEnumQty()-1)
        returns:        ID of enum or -1 if error

long GetnEnum(long idx);


***********************************************
** get serial number of enum by its ID
        arguments:      enum_id - ID of enum
        returns:        (0..GetEnumQty()-1) or -1 if error

long GetEnumIdx(long enum_id);

***********************************************
** get enum ID by the name of enum
        arguments:      name - name of enum
        returns:        ID of enum or -1 if no such enum exists

long GetEnum(char name);

***********************************************
** get name of enum
        arguments:      enum_id - ID of enum
        returns:        name of enum or empty string

char GetEnumName(long enum_id);

***********************************************
** get comment of enum
        arguments:      enum_id - ID of enum
                        repeatable - 0:get regular comment
                                     1:get repeatable comment
        returns:        comment of enum

char GetEnumCmt(long enum_id,long repeatable);

***********************************************
** get size of enum
        arguments:      enum_id - ID of enum
        returns:        number of constants in the enum
                        Returns 0 if enum_id is bad.

long GetEnumSize(long enum_id);

***********************************************
** get flag of enum
        arguments:      enum_id - ID of enum
        returns:        flags of enum. These flags determine representation
                        of numeric constants (binary,octal,decimal,hex)
                        in the enum definition. See start of this file for
                        more information about flags.
                        Returns 0 if enum_id is bad.

long GetEnumFlag(long enum_id);


***********************************************
** get member of enum - a symbolic constant ID
        arguments:      name - name of symbolic constant
        returns:        ID of constant or -1

long GetConstByName(char name);

***********************************************
** get value of symbolic constant
        arguments:      const_id - id of symbolic constant
        returns:        value of constant or 0

long GetConstValue(long const_id);

***********************************************
** get bit mask of symbolic constant
        arguments:      const_id - id of symbolic constant
        returns:        bitmask of constant or 0
                        ordinary enums have bitmask = -1

long GetConstBmask(long const_id);

***********************************************
** get id of enum by id of constant
        arguments:      const_id - id of symbolic constant
        returns:        id of enum the constant belongs to.
                        -1 if const_id is bad.

long GetConstEnum(long const_id);

***********************************************
** get id of constant
        arguments:      enum_id - id of enum
                        value   - value of constant
                        serial  - serial number of the constant in the
                                  enumeration. See OpEnumEx() for
                                  for details.
                        bmask   - bitmask of the constant
                                  ordinary enums accept only -1 as a bitmask
        returns:        id of constant or -1 if error

long GetConstEx(long enum_id,long value,long serial,long bmask);


***********************************************
** get first bitmask in the enum (bitfield)
        arguments:      enum_id - id of enum (bitfield)
        returns:        the smallest bitmask of constant or -1
                        no bitmasks are defined yet
                        All bitmasks are sorted by their values
                        as unsigned longs.

long GetFirstBmask(long enum_id);

***********************************************
** get last bitmask in the enum (bitfield)
        arguments:      enum_id - id of enum
        returns:        the biggest bitmask or -1 no bitmasks are defined yet
                        All bitmasks are sorted by their values
                        as unsigned longs.

long GetLastBmask(long enum_id);

***********************************************
** get next bitmask in the enum (bitfield)
        arguments:      enum_id - id of enum
                        bmask   - value of the current bitmask
        returns:        value of a bitmask with value higher than the specified
                        value. -1 if no such bitmasks exist.
                        All bitmasks are sorted by their values
                        as unsigned longs.

long GetNextBmask(long enum_id,long value);

***********************************************
** get prev bitmask in the enum (bitfield)
        arguments:      enum_id - id of enum
                        value   - value of the current bitmask
        returns:        value of a bitmask with value lower than the specified
                        value. -1 no such bitmasks exist.
                        All bitmasks are sorted by their values
                        as unsigned longs.

long GetPrevBmask(long enum_id,long value);

***********************************************
** get bitmask name (only for bitfields)
        arguments:      enum_id - id of enum
                        bmask   - bitmask of the constant
        returns:        name of bitmask if it exists. otherwise returns 0.

long GetBmaskName(long enum_id,long bmask);

***********************************************
** get bitmask comment (only for bitfields)
        arguments:      enum_id - id of enum
                        bmask   - bitmask of the constant
                        repeatable - type of comment, 0-regular, 1-repeatable
        returns:        comment attached to bitmask if it exists.
                        otherwise returns 0.

long GetBmaskCmt(long enum_id,long bmask,long repeatable);

***********************************************
** set bitmask name (only for bitfields)
        arguments:      enum_id - id of enum
                        bmask   - bitmask of the constant
                        name    - name of bitmask
        returns:        1-ok, 0-failed

success SetBmaskName(long enum_id,long bmask,char name);

***********************************************
** set bitmask comment (only for bitfields)
        arguments:      enum_id - id of enum
                        bmask   - bitmask of the constant
                        cmt     - comment
                        repeatable - type of comment, 0-regular, 1-repeatable
        returns:        1-ok, 0-failed

long SetBmaskCmt(long enum_id,long bmask,char cmt,long repeatable);

***********************************************
** get first constant in the enum
        arguments:      enum_id - id of enum
                        bmask   - bitmask of the constant
                                  ordinary enums accept only -1 as a bitmask
        returns:        value of constant or -1 no constants are defined
                        All constants are sorted by their values
                        as unsigned longs.

long GetFirstConst(long enum_id,long bmask);

***********************************************
** get last constant in the enum
        arguments:      enum_id - id of enum
                        bmask   - bitmask of the constant
                                  ordinary enums accept only -1 as a bitmask
        returns:        value of constant or -1 no constants are defined
                        All constants are sorted by their values
                        as unsigned longs.

long GetLastConst(long enum_id,long bmask);

***********************************************
** get next constant in the enum
        arguments:      enum_id - id of enum
                        bmask   - bitmask of the constant
                                  ordinary enums accept only -1 as a bitmask
                        value   - value of the current constant
        returns:        value of a constant with value higher than the specified
                        value. -1 no such constants exist.
                        All constants are sorted by their values
                        as unsigned longs.

long GetNextConst(long enum_id,long value,long bmask);

***********************************************
** get prev constant in the enum
        arguments:      enum_id - id of enum
                        bmask   - bitmask of the constant
                                  ordinary enums accept only -1 as a bitmask
                        value   - value of the current constant
        returns:        value of a constant with value lower than the specified
                        value. -1 no such constants exist.
                        All constants are sorted by their values
                        as unsigned longs.

long GetPrevConst(long enum_id,long value,long bmask);

***********************************************
** get name of a constant
        arguments:      const_id - id of const
        returns:        name of constant

char GetConstName(long const_id);

***********************************************
** get comment of a constant
        arguments:      const_id - id of const
                        repeatable - 0:get regular comment
                                     1:get repeatable comment
        returns:        comment string

char GetConstCmt(long const_id,long repeatable);

***********************************************
** add a new enum type
        arguments:      idx - serial number of the new enum.
                              If another enum with the same serial number
                              exists, then all enums with serial
                              numbers >= the specified idx get their
                              serial numbers incremented (in other words,
                              the new enum is put in the middle of the list
                              of enums).
                              If idx >= GetEnumQty() or idx == -1
                              then the new enum is created at the end of
                              the list of enums.
                        name - name of the enum.
                        flag - flags for representation of numeric constants
                               in the definition of enum.
        returns:        id of new enum or -1.

long AddEnum(long idx,char name,long flag);

***********************************************
** delete enum type
        arguments:      enum_id - id of enum

void DelEnum(long enum_id);

***********************************************
** give another serial number to a enum
        arguments:      enum_id - id of enum
                        idx     - new serial number.
                              If another enum with the same serial number
                              exists, then all enums with serial
                              numbers >= the specified idx get their
                              serial numbers incremented (in other words,
                              the new enum is put in the middle of the list
                              of enums).
                              If idx >= GetEnumQty() then the enum is
                              moved to the end of the list of enums.
        returns:        comment string

success SetEnumIdx(long enum_id,long idx);

***********************************************
** rename enum
        arguments:      enum_id - id of enum
                        name    - new name of enum
        returns:        1-ok,0-failed

success SetEnumName(long enum_id,char name);

***********************************************
** set comment of enum
        arguments:      enum_id - id of enum
                        cmt     - new comment for the enum
                        repeatable - 0:set regular comment
                                     1:set repeatable comment
        returns:        1-ok,0-failed

success SetEnumCmt(long enum_id,char cmt,long repeatable);

***********************************************
** set flag of enum
        arguments:      enum_id - id of enum
                        flag - flags for representation of numeric constants
                               in the definition of enum.
        returns:        1-ok,0-failed

success SetEnumFlag(long enum_id,long flag);

***********************************************
** set bitfield property of enum
        arguments:      enum_id - id of enum
                        flag - 1: convert to bitfield
                               0: convert to ordinary enum
        returns:        1-ok,0-failed

success SetEnumBf(long enum_id,long flag);

***********************************************
** is enum a bitfield?
        arguments:      enum_id - id of enum
        returns:        1-yes, 0-no, ordinary enum

success IsBitfield(long enum_id);

***********************************************
** add a member of enum - a symbolic constant
        arguments:      enum_id - id of enum
                        name    - name of symbolic constant. Must be unique
                                  in the program.
                        value   - value of symbolic constant.
                        bmask   - bitmask of the constant
                                  ordinary enums accept only -1 as a bitmask
                                  all bits set in value should be set in bmask too
        returns:        0-ok, otherwise error code:
#endif
#define CONST_ERROR_NAME  1     // already have member with this name (bad name)
#define CONST_ERROR_VALUE 2     // already have member with this value
#define CONST_ERROR_ENUM  3     // bad enum id
#define CONST_ERROR_MASK  4     // bad bmask
#define CONST_ERROR_ILLV  5     // bad bmask and value combination (~bmask & value != 0)
#ifdef _notdefinedsymbol

long AddConstEx(long enum_id,char name,long value,long bmask);

***********************************************
** delete a member of enum - a symbolic constant
        arguments:      enum_id - id of enum
                        value   - value of symbolic constant.
                        serial  - serial number of the constant in the
                                  enumeration. See OpEnumEx() for
                                  for details.
                        bmask   - bitmask of the constant
                                  ordinary enums accept only -1 as a bitmask
        returns:        1-ok,0-failed

success DelConstEx(long enum_id,long value,long serial,long bmask);

***********************************************
** rename a member of enum - a symbolic constant
        arguments:      const_id - id of const
                        name     - new name of constant
        returns:        1-ok,0-failed

success SetConstName(long const_id,char name);

***********************************************
** set a comment of a symbolic constant
        arguments:      const_id - id of const
                        cmt     - new comment for the constant
                        repeatable - 0:set regular comment
                                     1:set repeatable comment
        returns:        1-ok,0-failed

success SetConstCmt(long const_id,char cmt,long repeatable);


                         A R R A Y S  I N  I D C


The following functions allow you to manipulate arrays in IDC.
They have nothing to do with arrays in the disassembled program.
The IDC arrays are persistent and are kept in the database.
They remain until you explicitly delete them using DeleteArray().

The arrays are virtual. IDA allocates space for and keeps only the specified
elements of an array. The array index is 32-bit long. Actually, each array
may keep a set of strings and a set of long(32bit) values.

***********************************************
** create array
        arguments:      name - name of array. There are no restrictions
                               on the name (its length should be less than
                               120 characters, though)
        returns:        -1 - can't create array (it already exists)
                        otherwise returns id of the array

long CreateArray(char name);


***********************************************
** get array id by its name
        arguments:      name - name of existing array.
        returns:        -1 - no such array
                        otherwise returns id of the array

long GetArrayId(char name);


***********************************************
** rename array
        arguments:      id      - array id returned by CreateArray() or
                                  GetArrayId()
                        newname - new name of array. There are no
                                  restrictions on the name (its length should
                                  be less than 120 characters, though)
        returns:        1-ok, 0-failed

success RenameArray(long id,char newname);


***********************************************
** delete array
   This function deletes all elements of the array.
        arguments:      id      - array id

void DeleteArray(long id);


***********************************************
** set 32bit value of array element
        arguments:      id      - array id
                        idx     - index of an element
                        value   - 32bit value to store in the array
        returns:        1-ok, 0-failed

success SetArrayLong(long id,long idx,long value);


***********************************************
** set string value of array element
        arguments:      id      - array id
                        idx     - index of an element
                        str     - string to store in array element
        returns:        1-ok, 0-failed

success SetArrayString(long id,long idx,char str);


***********************************************
** get value of array element
        arguments:      tag     - tag of array, specifies one of two
                                  array types:
#endif
#define AR_LONG 'A'     // array of longs
#define AR_STR  'S'     // array of strings
#ifdef _notdefinedsymbol
                        id      - array id
                        idx     - index of an element
        returns:        value of the specified array element.
                        note that this function may return char or long
                        result. Unexistent array elements give zero as
                        a result.

char or long GetArrayElement(long tag,long id,long idx);


***********************************************
** delete an array element
        arguments:      tag     - tag of array (AR_LONG or AR_STR)
                        id      - array id
                        idx     - index of an element
        returns:        1-ok, 0-failed

success DelArrayElement(long tag,long id,long idx);


***********************************************
** get index of the first existing array element
        arguments:      tag     - tag of array (AR_LONG or AR_STR)
                        id      - array id
        returns:        -1 - array is empty
                        otherwise returns index of the first array element

long GetFirstIndex(long tag,long id);


***********************************************
** get index of the last existing array element
        arguments:      tag     - tag of array (AR_LONG or AR_STR)
                        id      - array id
        returns:        -1 - array is empty
                        otherwise returns index of the last array element

long GetLastIndex(long tag,long id);


***********************************************
** get index of the next existing array element
        arguments:      tag     - tag of array (AR_LONG or AR_STR)
                        id      - array id
                        idx     - index of the current element
        returns:        -1 - no more array elements
                        otherwise returns index of the next array element

long GetNextIndex(long tag,long id,long idx);


***********************************************
** get index of the previous existing array element
        arguments:      tag     - tag of array (AR_LONG or AR_STR)
                        id      - array id
                        idx     - index of the current element
        returns:        -1 - no more array elements
                        otherwise returns index of the previous array element

long GetPrevIndex(long tag,long id,long idx);


***********************************************
** associative arrays (the same as hashes in Perl)
** to create a hash, use CreateArray() function
** you can use the following function with hashes:
**      GetArrayId(), RenameArray(), DeleteArray()
** The following additional functions are defined:

success SetHashLong(long id,char idx,long value);
success SetHashString(long id,char idx,char value);
long    GetHashLong(long id,char idx);
char    GetHashString(long id,char idx);
success DelHashElement(long id,char idx);
char    GetFirstHashKey(long id);
char    GetNextHashKey(long id,char idx);
char    GetLastHashKey(long id);
char    GetPrevHashKey(long id,char idx);


                S O U R C E   F I L E / L I N E   N U M B E R S


  IDA can keep information about source files used to create the program.
  Each source file is represented by a range of addresses.
  A source file may contains several address ranges.

***********************************************
** Mark a range of address as belonging to a source file
   An address range may belong only to one source file.
   A source file may be represented by several address ranges.
        ea1     - linear address of start of the address range
        ea2     - linear address of end of the address range
        filename- name of source file.
   returns: 1-ok, 0-failed.

success AddSourceFile(long ea1,ulong ea2,char filename);


***********************************************
** Get name of source file occupying the given address
        ea - linear address
   returns: NULL - source file information is not found
            otherwise returns pointer to file name

char GetSourceFile(long ea);


***********************************************
** Delete information about the source file
        ea - linear address belonging to the source file
   returns: NULL - source file information is not found
            otherwise returns pointer to file name

success DelSourceFile(long ea);


***********************************************
** set source line number
        arguments:      ea      - linear address
                        lnnum   - number of line in the source file
        returns:        nothing

void SetLineNumber(long ea,long lnnum);


***********************************************
** get source line number
        arguments:      ea      - linear address
        returns:        number of line in the source file or -1

long GetLineNumber(long ea);


***********************************************
** delete information about source line number
        arguments:      ea      - linear address
        returns:        nothing

void DelLineNumber(long ea);



                T Y P E  L I B R A R I E S


***********************************************
** Load a type library
        name - name of type library.
   returns: 1-ok, 0-failed.

success LoadTil(char name);


***********************************************
** Copy information from type library to database
   Copy structure, union, or enum definition from the type library
   to the IDA database.
        idx       - the position of the new type in the list of
                    types (structures or enums)
                    -1 means at the end of the list
        type_name - name of type to copy
   returns: BADNODE-failed, otherwise the type id
                (structure id or enum id)

long Til2Idb(long idx, char type_name);



#endif // _notdefinedsymbol
//--------------------------------------------------------------------------
// Compatibility macros:

#define OpOffset(ea,base)       OpOff(ea,-1,base)
#define OpNum(ea)               OpNumber(ea,-1)
#define OpChar(ea)              OpChr(ea,-1)
#define OpSegment(ea)           OpSeg(ea,-1)
#define OpDec(ea)               OpDecimal(ea,-1)
#define OpAlt1(ea,str)          OpAlt(ea,0,str)
#define OpAlt2(ea,str)          OpAlt(ea,1,str)
#define StringStp(x)            SetCharPrm(INF_ASCII_BREAK,x)
#define LowVoids(x)             SetLongPrm(INF_LOW_OFF,x)
#define HighVoids(x)            SetLongPrm(INF_HIGH_OFF,x)
#define TailDepth(x)            SetLongPrm(INF_MAXREF,x)
#define Analysis(x)             SetCharPrm(INF_AUTO,x)
#define Tabs(x)                 SetCharPrm(INF_ENTAB,x)
#define Comments(x)             SetCharPrm(INF_CMTFLAG,((x) ? (SW_ALLCMT|GetCharPrm(INF_CMTFLAG)) : (~SW_ALLCMT&GetCharPrm(INF_CMTFLAG))))
#define Voids(x)                SetCharPrm(INF_VOIDS,x)
#define XrefShow(x)             SetCharPrm(INF_XREFNUM,x)
#define Indent(x)               SetCharPrm(INF_INDENT,x)
#define CmtIndent(x)            SetCharPrm(INF_COMMENT,x)
#define AutoShow(x)             SetCharPrm(INF_SHOWAUTO,x)
#define MinEA()                 GetLongPrm(INF_MIN_EA)
#define MaxEA()                 GetLongPrm(INF_MAX_EA)
#define BeginEA()               GetLongPrm(INF_BEGIN_EA)
#define set_start_cs(x)         SetLongPrm(INF_START_CS,x)
#define set_start_ip(x)         SetLongPrm(INF_START_IP,x)
#define WriteMap(file) \
        GenerateFile(OFILE_MAP, fopen(file,"w"), 0, BADADDR, \
        GENFLG_MAPSEGS|GENFLG_MAPNAME)
#define WriteTxt(file,ea1,ea2) \
        GenerateFile(OFILE_ASM,fopen(file,"w"), ea1, ea2, 0)
#define WriteExe(file) \
        GenerateFile(OFILE_EXE,fopen(file,"wb"), 0, BADADDR, 0)
#define Direction(x)            // obsolete function
#define AddConst(enum_id,name,value) AddConstEx(enum_id,name,value,-1)
#define AddStruc(index,name)         AddStrucEx(index,name,0)
#define AddUnion(index,name)         AddStrucEx(index,name,1)
#define OpStroff(ea,n,strid)         OpStroffEx(ea,n,strid,0)
#define OpEnum(ea,n,enumid)          OpEnumEx(ea,n,enumid,0)
#define DelConst(id,v,mask)          DelConstEx(id,v,0,mask)
#define GetConst(id,v,mask)          GetConstEx(id,v,0,mask)

#define here                    ScreenEA()


#endif // _IDC_IDC
