/* nc100em, a VGA NC100 emulator for Linux.
 * Copyright (C) 1994 Ian Collier. nc100em changes (C) 1996 Russell Marks.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* This also compiles as a serial-only text version if TEXT_VER is
 * defined.
 */

#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <fcntl.h>
#include <sys/time.h>
#include <sys/time.h>
#include <sys/types.h>
#include <time.h>
#include <unistd.h>
#ifndef TEXT_VER
#include <vga.h>
#include <rawkey.h>
#endif
#include "z80.h"

/* the memory is 256k ROM, 64k RAM, and 1024k PCMCIA memory card */
unsigned char mem[(256+64+1024)*1024];
unsigned char *dfile;
/* boots with all ROM0 */
unsigned char *memptr[4]={mem,mem,mem,mem};
int kports[8];
unsigned long tstates=0,tsmax=60000;

int memattr[4]={0,0,0,0};	/* all ROM at boot */

int screen_dirty;
int hsize=480,vsize=64;
volatile int interrupted=0;
int input_wait=0;
int scrn_freq=10;

int card_size=0;		/* size of pcmcia card, or 0 if none */
int card_status=0;		/* as read from port 0xA0 */
int irq_status=0xff;		/* as read from port 0x90 */
int irq_mask=0;			/* as set via port 0x60 */
int scrnaddr=0;
int do_nmi=0;
int count_and=0x7ff;

struct tm *rtc_tm;

#define SCRN_XOFS	80		/* x offset of display */
#define SCRN_YOFS	200		/* y offset */

#ifdef PTY_SERIAL
int pty_fd;
unsigned char seroutbuf[16384];		/* won't need this many - defensive */
int seroutpos=0;

#ifdef TEXT_VER
#define PTY_NAME	"/dev/tty"
#else
#ifndef SERIAL_MOUSE
/* XXX this is a revolting kludge */
#define PTY_NAME	"/dev/ptysf"
#else
#define PTY_NAME	"/dev/ttyS0"
#endif
#endif

#endif	/* PTY_SERIAL */

unsigned char keyports[10]={0,0,0,0,0, 0,0,0,0,0};
int scancode[128];

#ifdef TEXT_VER
struct autokey_t
  {
  /* up to two keys to press this int */
  int port1,mask1;
  int port2,mask2;
  };

struct autokey_t autokeys[]=
  {
  /* hold down both shifts for a while, to make sure we cold-boot */
  {0,1, 0,2},
  {0,1, 0,2},
  {0,1, 0,2},
  {0,1, 0,2},
  {0,1, 0,2},
  {0,1, 0,2},
  {0,1, 0,2},
  {0,1, 0,2},
  {0,1, 0,2},
  {0,1, 0,2},
  {0,1, 0,2},
  {0,1, 0,2},
  {0,1, 0,2},
  {0,1, 0,2},
  {0,1, 0,2},
  {0,1, 0,2},
  {0,1, 0,2},
  {0,1, 0,2},
  {0,1, 0,2},
  {0,1, 0,2},
  
  /* now do |, s, enter. This should get a serial console going.
   * note that the speed/time we do this doesn't matter - ZCN will
   * buffer the keypresses anyway.
   */
  {0,1, 7,0x04},	/* | */
  {3,0x40, 0,0},	/* s */
  {0,0x10, 0,0},	/* enter */
  
  /* this stops the auto-keypressing */
  {0,0, 0,0}
  };
#endif


void sighandler(a)
int a;
{
interrupted=1;
}


void dontpanic(a)
int a;
{
#ifndef TEXT_VER
rawmode_exit();
vga_setmode(TEXT);
#endif
writecard(mem+CARD_START);
exit(1);
}


void dontpanic_nmi()
{
#ifndef TEXT_VER
rawmode_exit();
vga_setmode(TEXT);
#endif
writeram(mem+RAM_START);
writecard(mem+CARD_START);
exit(1);
}



#ifndef TEXT_VER
void screenon()
{
vga_setmode(G640x480x2);
}


void screenoff()
{
vga_setmode(TEXT);
}
#endif


main(argc,argv)
int argc;
char **argv;
{
struct sigaction sa;
struct itimerval itv;
int tmp=1000/100;	/* 100 ints/sec */
int f;

#ifndef TEXT_VER
vga_init();
#else
press_autokeys(0);	/* so shifts are held down when we boot */
#endif

loadrom(mem);
loadram(mem+RAM_START);
loadcard(mem+CARD_START);
set_card_status();

#ifdef PTY_SERIAL
serial_init();
#endif

#ifndef TEXT_VER
screenon();
rawmode_init();
set_switch_functions(screenoff,screenon);
allow_switch(1);

for(f=32;f<127;f++) scancode[f]=scancode_trans(f);

if(argc==2) scrn_freq=atoi(argv[1])*2;
if(scrn_freq!=0) argc--,argv++;
if(scrn_freq<4) scrn_freq=4;
if(scrn_freq>100) scrn_freq=100;
#endif

if(argc>=2)
  {
  FILE *in;
  
  if((in=fopen(argv[1],"rb"))==NULL)
    printf("Couldn't open file.\n");
  else
    {
    if(argc==3)
      fread(mem+RAM_START+atoi(argv[2]),1,49152,in);
    else
      fread(mem+RAM_START+0x100,1,49152,in);
    fclose(in);
    }
  }

sa.sa_handler=dontpanic;
sa.sa_mask=0;
sa.sa_flags=SA_ONESHOT;

sigaction(SIGINT, &sa,NULL);
sigaction(SIGHUP, &sa,NULL);
sigaction(SIGILL, &sa,NULL);
sigaction(SIGTERM,&sa,NULL);
sigaction(SIGQUIT,&sa,NULL);
sigaction(SIGSEGV,&sa,NULL);

#ifndef NO_SPEED_CONTROL
sa.sa_handler=sighandler;
sa.sa_mask=0;
sa.sa_flags=SA_RESTART;

sigaction(SIGALRM,&sa,NULL);

itv.it_value.tv_sec=  0;
itv.it_value.tv_usec= 100000;
itv.it_interval.tv_sec=  tmp/1000;
itv.it_interval.tv_usec=(tmp%1000)*1000;
setitimer(ITIMER_REAL,&itv,NULL);
#endif

mainloop();

#ifndef TEXT_VER
rawmode_exit();
vga_setmode(TEXT);
#endif
writeram(mem+RAM_START);
writecard(mem+CARD_START);
exit(0);
}


#ifdef TEXT_VER
press_autokeys(pos)
int pos;
{
keyports[autokeys[pos].port1]|=autokeys[pos].mask1;
keyports[autokeys[pos].port2]|=autokeys[pos].mask2;
}

unpress_autokeys()
{
int f;

/* just unpress the lot */
for(f=0;f<10;f++) keyports[f]=0;
}
#endif


/* the card_status flag never changes after this */
set_card_status()
{
card_status=0x33; /* no batteries low, and parallel always ACK and !BUSY */
if(card_size==0) card_status|=0x80;	/* is card present? */
}


loadrom(x)
unsigned char *x;
{
int i;
FILE *in;

if((in=fopen("nc100.rom","rb"))!=NULL)
  {
  fread(x,1024,256,in);
  fclose(in);
  }
else
  {
  printf("Couldn't load ROM.\n");
  exit(1);
  }
}


loadram(x)
unsigned char *x;
{
int i;
FILE *in;

if((in=fopen("nc100.ram","rb"))!=NULL)
  {
  fread(x,1024,64,in);
  fclose(in);
  }
else
  {
  printf("Couldn't load RAM.\n");
  /* not fatal - blank it and carry on */
  memset(x,0xff,64*1024);
  }
}


loadcard(x)
unsigned char *x;
{
int i;
FILE *in;

if((in=fopen("nc100.card","rb"))!=NULL)
  {
  card_size=fread(x,1024,1024,in);
  fclose(in);
  }
else
  {
  printf("Couldn't load PCMCIA memory card.\n");
  /* also not fatal, blank it */
  memset(x,0,1024*1024);
  }
}


writeram(x)
unsigned char *x;
{
FILE *out;

if((out=fopen("nc100.ram","wb"))==NULL)
  printf("Couldn't write RAM.\n");
else
  {
  fwrite(x,1024,64,out);
  fclose(out);
  }
}


writecard(x)
unsigned char *x;
{
FILE *out;

if(card_size>0)
  {
  if((out=fopen("nc100.card","wb"))==NULL)
    printf("Couldn't write PCMCIA memory card.\n");
  else
    {
    fwrite(x,1024,card_size,out);
    fclose(out);
    }
  }
}


/* I/O, quoting from nciospec.doc:
[we ignore some]
D0-DF                   RTC (TC8521)            R/W	(ignore for now)
C0-C1                   UART (uPD71051)         R/W	(ignore for now)
B0-B9                   Key data in             R
A0                      Card Status etc.        R
90                      IRQ request status      R/W
70                      Power on/off control    W
60                      IRQ Mask                W
50-53                   Speaker frequency       W	(ignore)
40                      Parallel port data      W	(ignore)
30                      Baud rate etc.          W	(ignore)
20                      Card wait control       W	(ignore)
10-13                   Memory management       R/W
00                      Display memory start    W
*/


/* simulate reads from paging ports.
 * this is slow and cack but these are almost never read, just written.
 */
int check_paging(page)
int page;
{
int ofs,ret;

if(page<0 || page>3)
  {
  printf("can't happen");
  return(0);
  }

ret=0;	/* default to ROM */
ofs=memptr[page]-mem;
if(ofs>=RAM_START && ofs<CARD_START)
  ret|=0x40,ofs-=RAM_START;
else
  if(ofs>=CARD_START)
    ret|=0x80,ofs-=CARD_START;

ret|=(ofs/16384);

return(ret);
}


/* now this *is* used a bit, but fortunately it's dead quick :-) */
do_paging(page,n)
int page,n;
{
switch(n&0xc0)
  {
  case 0x00:	/* ROM */
    memptr[page]=mem+16384*n;
    memattr[page]=0;
    break;
  case 0x40:	/* RAM */
    memptr[page]=mem+RAM_START+16384*(n&0x0f);
    memattr[page]=1;
    break;
  case 0x80:	/* card ram */
    memptr[page]=mem+CARD_START+16384*(n&0x3f);
    memattr[page]=1;
    /* one way of detecting the size of a card requires
     * that writes which are off-card must fail, so...
     */
    if(16*(n&0x3f)>=card_size) memattr[page]=0;
    break;
  }
}



unsigned int in(h,l)
int h,l;
{
static int ts=(13<<8);	/* num. t-states for this out, times 256 */

/* h is ignored by the NC100 */
switch(l)
  {
  /* RTC */
  case 0xd0: case 0xd1: case 0xd2: case 0xd3:
  case 0xd4: case 0xd5: case 0xd6: case 0xd7:
  case 0xd8: case 0xd9: case 0xda: case 0xdb:
  case 0xdc: case 0xdd: case 0xde: case 0xdf:
    switch(l-0xd0)
      {
      case 0:	return(ts|('0'+rtc_tm->tm_sec%10));
      case 1:	return(ts|('0'+rtc_tm->tm_sec/10));
      case 2:	return(ts|('0'+rtc_tm->tm_min%10));
      case 3:	return(ts|('0'+rtc_tm->tm_min/10));
      case 4:	return(ts|('0'+rtc_tm->tm_hour%10));
      case 5:	return(ts|('0'+rtc_tm->tm_hour/10));
      
      case 6:	return(ts);
      
      case 7:	return(ts|('0'+rtc_tm->tm_mday%10));
      case 8:	return(ts|('0'+rtc_tm->tm_mday/10));
      case 9:	return(ts|('0'+(rtc_tm->tm_mon+1)%10));
      case 10:	return(ts|('0'+(rtc_tm->tm_mon+1)/10));
      case 11:	return(ts|('0'+(rtc_tm->tm_year-90)%10));
      case 12:	return(ts|('0'+(rtc_tm->tm_year-90)/10));
      
      case 13:
      case 14:
      case 15:
        return(ts);
      
      default:
        printf("can't happen\n");
      }
    return(ts|0);
  
  
#ifndef PTY_SERIAL
  case 0xc0: case 0xc1: return(ts|0);
#else
  /* UART */
  case 0xc0:	/* this reads from the serial port */
    if(serial_input_pending()) return(get_serial_byte());
    return(ts|0);
  
  case 0xc1:    /* returns bit 0=1 if can send a byte */
    /* we assume it's always possible, though (XXX) bad things
     * may happen if there isn't a process reading the pty...?
     */
    return(ts|1);
#endif
    
  /* keyboard */
  case 0xb0: case 0xb1: case 0xb2: case 0xb3: case 0xb4:
  case 0xb5: case 0xb6: case 0xb7: case 0xb8: case 0xb9:
    /* reading 0xb9 also sets bit 3 of irq_status */
    if(l==0xb9) irq_status|=8;
    return(ts|keyports[l-0xb0]);
  
  case 0xa0:	/* card etc. status */
    return(ts|card_status);
  
  case 0x90:	/* IRQ status */
    return(ts|irq_status);
  
  /* memory paging */
  case 0x10: return(ts|check_paging(0));
  case 0x11: return(ts|check_paging(1));
  case 0x12: return(ts|check_paging(2));
  case 0x13: return(ts|check_paging(3));
  }

/* otherwise... */
#if 0
fprintf(stderr,"in l=%02X\n",l);
#endif
return(ts|255);
}


unsigned int out(h,l,a)
int h,l,a;
{
static int ts=13;	/* num. t-states for this out */
time_t timet;

/* h is ignored by the NC100 */
switch(l)
  {
  /* RTC */
  case 0xdd:
    /* this one locks output (I think) when bit 3 is 1 */
    /* get time */
    timet=time(NULL);
    rtc_tm=localtime(&timet);
    return(ts);
      
  case 0xd0: case 0xd1: case 0xd2: case 0xd3:
  case 0xd4: case 0xd5: case 0xd6: case 0xd7:
  case 0xd8: case 0xd9: case 0xda: case 0xdb:
  case 0xdc:            case 0xde: case 0xdf:
    /* ignored, don't want to let them set the time!
     * (would need to be root anyway)
     */
    return(ts);
  
#ifndef PTY_SERIAL
  case 0xc0: case 0xc1: return(ts);
#else
  /* UART */
  case 0xc0:	/* this writes to the serial port */
    put_serial_byte(a);
    return(ts);
  
  case 0xc1:    /* sets up various serial parms which we ignore */
    return(ts);
#endif
  
  case 0x90:	/* IRQ status */
    /* when a zero is written to a bit, it should re-enable that IRQ line */
    irq_status|=(a^255);
    return(ts);
  
  case 0x70:	/* power on/off */
    /* if bit 0 is 0, turn off */
    if((a&1)==0) dontpanic_nmi();
    return(ts);
  
  case 0x60:	/* set irq mask */
    irq_mask=a;
    return(ts);
  
  case 0x50: case 0x51: case 0x52: case 0x53:
    /* speaker frequency, ignored */
    return(ts);
  
  case 0x40:	/* parallel port data, ignored */
    return(ts);
  
  case 0x30:	/* baud rate etc., ignored */
    return(ts);
  
  case 0x20:	/* card wait control, ignored */
    return(ts);
  
  /* memory paging */
  case 0x10: do_paging(0,a); return(ts);
  case 0x11: do_paging(1,a); return(ts);
  case 0x12: do_paging(2,a); return(ts);
  case 0x13: do_paging(3,a); return(ts);
  
  case 0:	/* set screen addr */
    scrnaddr=(a&0xf0)*256;
    return(ts);
  }

/* otherwise... */
#if 0
fprintf(stderr,"out l=%02X,a=%02X\n",l,a);
#endif
return(ts);
}


#ifndef TEXT_VER
/* redraw the screen */
update_scrn()
{
static int count=0;
unsigned char *pageptr;
int f;

/* only do it every 1/Nth */
count++;
if(count<scrn_freq) return(0); else count=0;

/* this should work... */
pageptr=mem+RAM_START+scrnaddr;
for(f=0;f<64;f++)
  vga_drawscansegment(pageptr+f*64,SCRN_XOFS,SCRN_YOFS+f,60);
}


update_kybd()
{
int y;

for(y=0;y<5;y++) scan_keyboard();

if(is_key_pressed(FUNC_KEY(10)))
  {
  while(is_key_pressed(FUNC_KEY(10))) { usleep(20000); scan_keyboard(); };
  dontpanic();	/* F10 = quit */
  /* XXX want this to merely issue NMI when it's all working...? */
  }

if(is_key_pressed(FUNC_KEY(5)))
  {
  while(is_key_pressed(FUNC_KEY(5))) { usleep(20000); scan_keyboard(); };
  do_nmi=1;	/* F5 = NMI */
  }

for(y=0;y<10;y++) keyports[y]=0;

/* this is a bit messy, but librawkey isn't too good at this sort
 * of thing - it wasn't really intended to be used like this :-)
 */
if(is_key_pressed(ENTER_KEY)) 		keyports[0]|=0x10;
if(is_key_pressed(CURSOR_LEFT))		keyports[0]|=0x08;
if(is_key_pressed(RIGHT_SHIFT))		keyports[0]|=0x02;
if(is_key_pressed(LEFT_SHIFT))		keyports[0]|=0x01;
if(is_key_pressed(scancode['5']))	keyports[1]|=0x40;
if(is_key_pressed(scancode[' ']))	keyports[1]|=0x08;
if(is_key_pressed(ESCAPE_KEY))		keyports[1]|=0x04;
if(is_key_pressed(LEFT_CTRL))		keyports[1]|=0x02;
if(is_key_pressed(INSERT_KEY))		keyports[1]|=0x01; /* 'function' */
if(is_key_pressed(TAB_KEY))		keyports[2]|=0x08;
if(is_key_pressed(scancode['1']))	keyports[2]|=0x04;
if(is_key_pressed(LEFT_ALT))		keyports[2]|=0x02; /* 'symbol' */
if(is_key_pressed(CAPS_LOCK))		keyports[2]|=0x01;
if(is_key_pressed(scancode['d']))	keyports[3]|=0x80;
if(is_key_pressed(scancode['s']))	keyports[3]|=0x40;
if(is_key_pressed(scancode['e']))	keyports[3]|=0x10;
if(is_key_pressed(scancode['w']))	keyports[3]|=0x08;
if(is_key_pressed(scancode['q']))	keyports[3]|=0x04;
if(is_key_pressed(scancode['2']))	keyports[3]|=0x02;
if(is_key_pressed(scancode['3']))	keyports[3]|=0x01;
if(is_key_pressed(scancode['f']))	keyports[4]|=0x80;
if(is_key_pressed(scancode['r']))	keyports[4]|=0x40;
if(is_key_pressed(scancode['a']))	keyports[4]|=0x10;
if(is_key_pressed(scancode['x']))	keyports[4]|=0x08;
if(is_key_pressed(scancode['z']))	keyports[4]|=0x04;
if(is_key_pressed(scancode['4']))	keyports[4]|=0x01;
if(is_key_pressed(scancode['c']))	keyports[5]|=0x80;
if(is_key_pressed(scancode['g']))	keyports[5]|=0x40;
if(is_key_pressed(scancode['y']))	keyports[5]|=0x20;
if(is_key_pressed(scancode['t']))	keyports[5]|=0x10;
if(is_key_pressed(scancode['v']))	keyports[5]|=0x08;
if(is_key_pressed(scancode['b']))	keyports[5]|=0x04;
if(is_key_pressed(scancode['n']))	keyports[6]|=0x80;
if(is_key_pressed(scancode['h']))	keyports[6]|=0x40;
if(is_key_pressed(scancode['/']))	keyports[6]|=0x20;
if(is_key_pressed(scancode['#']))	keyports[6]|=0x10;
if(is_key_pressed(CURSOR_RIGHT))	keyports[6]|=0x08;
if(is_key_pressed(DELETE_KEY))		keyports[6]|=0x04;
if(is_key_pressed(CURSOR_DOWN))		keyports[6]|=0x02;
if(is_key_pressed(scancode['6']))	keyports[6]|=0x01;
if(is_key_pressed(scancode['k']))	keyports[7]|=0x80;
if(is_key_pressed(scancode['m']))	keyports[7]|=0x40;
if(is_key_pressed(scancode['u']))	keyports[7]|=0x20;
if(is_key_pressed(scancode['`']))	keyports[7]|=0x10;
if(is_key_pressed(CURSOR_UP))		keyports[7]|=0x08;
if(is_key_pressed(scancode['\\']))	keyports[7]|=0x04;
if(is_key_pressed(scancode['7']))	keyports[7]|=0x02;
if(is_key_pressed(scancode['=']))	keyports[7]|=0x01;
if(is_key_pressed(scancode[',']))	keyports[8]|=0x80;
if(is_key_pressed(scancode['j']))	keyports[8]|=0x40;
if(is_key_pressed(scancode['i']))	keyports[8]|=0x20;
if(is_key_pressed(scancode['\'']))	keyports[8]|=0x10;
if(is_key_pressed(scancode['[']))	keyports[8]|=0x08;
if(is_key_pressed(scancode[']']))	keyports[8]|=0x04;
if(is_key_pressed(scancode['-']))	keyports[8]|=0x02;
if(is_key_pressed(scancode['8']))	keyports[8]|=0x01;
if(is_key_pressed(scancode['.']))	keyports[9]|=0x80;
if(is_key_pressed(scancode['o']))	keyports[9]|=0x40;
if(is_key_pressed(scancode['l']))	keyports[9]|=0x20;
if(is_key_pressed(scancode[';']))	keyports[9]|=0x10;
if(is_key_pressed(scancode['p']))	keyports[9]|=0x08;
if(is_key_pressed(BACKSPACE))		keyports[9]|=0x04;
if(is_key_pressed(scancode['9']))	keyports[9]|=0x02;
if(is_key_pressed(scancode['0']))	keyports[9]|=0x01;
}
#endif		/* !TEXT_VER */


fix_tstates()
{
tstates=0;
#ifdef NO_SPEED_CONTROL
interrupted=1;
#else
pause();
#endif
}


do_interrupt()
{
static int scount=0;
static int autokey_pos=0;

#ifndef TEXT_VER
update_scrn();
update_kybd();
#else
if(autokey_pos>=0)
  {
  unpress_autokeys();
  autokey_pos++;
  if(autokeys[autokey_pos].mask1==0)
    autokey_pos=-1;
  else
    press_autokeys(autokey_pos);
  }
#endif

#ifdef PTY_SERIAL
scount++;
if(scount==4) scount=0,serout_flush();
#endif

interrupted=0;
}


#ifdef PTY_SERIAL

serial_init()
{
/* XXX should do this properly... */
/* mind you this is probably more portable than termios :-) */
char buf[1024];

#ifndef SERIAL_MOUSE
/* the -echo is really for tnc100em, but doesn't hurt anyway */
sprintf(buf,"stty raw -echo <%s",PTY_NAME);
#else
sprintf(buf,"stty 1200 cs7 raw <%s",PTY_NAME);
#endif
system(buf);

if((pty_fd=open(PTY_NAME,O_RDWR|O_NONBLOCK))<0)
  {
  printf("Couldn't open pty!\n");
  exit(1);
  }
}


int serial_input_pending()
{
struct timeval tv;
fd_set fds;

tv.tv_sec=0; tv.tv_usec=0;
FD_ZERO(&fds); FD_SET(pty_fd,&fds);
select(pty_fd+1,&fds,NULL,NULL,&tv);

return(FD_ISSET(pty_fd,&fds));
}


int serial_output_allowed()
{
#if 1
/* assume it always is. select() takes yonks in our terms. */
return(1);
#else
struct timeval tv;
fd_set fds;

tv.tv_sec=0; tv.tv_usec=0;
FD_ZERO(&fds); FD_SET(pty_fd,&fds);
select(pty_fd+1,NULL,&fds,NULL,&tv);

return(FD_ISSET(pty_fd,&fds));
#endif
}

#include <errno.h>

int get_serial_byte()
{
int res;
unsigned char c=0;

if((res=read(pty_fd,&c,1))<=0)
  {
  if(res<0 && errno==EAGAIN) return(0);
  /* it probably closed it; re-open and try again */
  close(pty_fd);
  if((pty_fd=open(PTY_NAME,O_RDWR|O_NONBLOCK))<0)
    {
    printf("Couldn't open pty!\n");
    dontpanic();
    }
  read(pty_fd,&c,1);
  }

return((int)c);
}


put_serial_byte(n)
int n;
{
unsigned char c=n;

#if 0
if(serial_output_allowed())
#endif
  seroutbuf[seroutpos++]=c;
}

serout_flush()
{
if(seroutpos>0)
  write(pty_fd,seroutbuf,seroutpos);
seroutpos=0;
}
#endif /* PTY_SERIAL */
