/* tuneconv - convert pseudo-csf to tune.z data. based on grokfile.c

   Adjusted very slightly by Nicholas Campbell so that the frequencies match
   the clock pulse of the AY-3-8912 on the Amstrad CPC rather than the
   Spectrum */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <math.h>


#define DEFAULTVOLUME	15
#define OCTAVEFIX	3


static int freqtable[4096];
static int num_freqs=1;		/* zero implicit */

static int curline=0;

int bsize=64;


/* (some) function prototypes */
void makecsffilename(char *new,char *org);
int getline(char *buf,FILE *in);
void check_linelen(char *buf,int bsize,int has_comma);
void processcsf(FILE *in);
void addline(int num,int mode,
	char *notes,char *octaves,char *sample,
        char *stereo1,char *stereo2,char *relvol,
	int sod_time,int sod_down,int sod_pos);
void padline(char *str,int n);


int main(void)
{
static char buf[256];
FILE *in;

if((in=fopen("tmpfreq.dat","r"))!=NULL)
  {
  /* read existing frequency table */
  
  while(fgets(buf,sizeof(buf),in)!=NULL)
    freqtable[num_freqs++]=atoi(buf);

  fclose(in);
  }
else
  printf(";auto-generated by tuneconv, edits will be lost!\n\n");

processcsf(stdin);
exit(0);
}


int getline(char *buf,FILE *in)
{
char *ptr;

buf[0]=0;
do
  curline++,ptr=fgets(buf,256,in);
while(buf[0]=='#' && !feof(in));

if(ptr!=NULL)
  {
  if(buf[0]!=0 && buf[strlen(buf)-1]=='\n') buf[strlen(buf)-1]=0;
  if(buf[0]!=0 && buf[strlen(buf)-1]=='\r') buf[strlen(buf)-1]=0;
  }
else
  buf[0]=0;
return((ptr==NULL)?-1:0);
}


/* check line len, and fix it if too long (giving warning) */
void check_linelen(char *buf,int bsize,int has_comma)
{
char *cptr=NULL;

/* if we allow a comma, see if we have one or not. */
if(has_comma)
  cptr=strchr(buf,',');

if((cptr==NULL && strlen(buf)>bsize) ||
   (cptr!=NULL && (cptr-buf)>bsize))
  {
  buf[bsize]=0;
  fprintf(stderr,"sod2: line %d too long.\n",curline);
  exit(1);
  }
}


void processcsf(FILE *in)
{
char buf[256],buf2[256],buf3[256],vbuf[256],sbuf[256],sbuf2[256],*ptr;
int f,g,i;
int sod_time,sod_down,sod_pos;
int tempo=125;

/* look for '*samples' (and possibly '*tempo', '*bsize' before that) */
getline(buf,in);
if(!strncmp(buf,"*tempo",6))
  {
  tempo=atoi(buf+6);
  getline(buf,in);
  }

if(tempo<100) tempo=100;
if(tempo>240) tempo=240;

if(!strncmp(buf,"*bsize",6))
  {
  /* ignored */
  getline(buf,in);
  }

if(strcmp(buf,"*samples"))
  {
  fprintf(stderr,"sod2: missing `*samples' block (must be first).\n");
  exit(1);
  }

/* add all the sample descriptions */
getline(buf,in);
while(strcmp(buf,"*blocklist"))
  {
  if(getline(buf,in)==-1)
    {
    fprintf(stderr,"sod2: missing `*blocklist' section.\n");
    exit(1);
    }
  }

printf("blocklist:\n");

g=1;
getline(buf,in);
while(strcmp(buf,"*blocks"))
  {
  ptr=strrchr(buf,',');
  f=1;
  if(ptr!=NULL)
    {
    do
      {
      f++;
      *ptr++=0;
      if(f<=2)
        printf("defb %s\n",ptr);
      }
    while((ptr=strrchr(buf,','))!=NULL);
    }
  printf("defb %s\n",buf);
  if(f>2)
    fprintf(stderr,
            "tuneconv: warning: "
            "%d: (%d) attempt to use %d channels\n",curline,g,f);
  for(;f<2;f++)
    printf("defb 0\n");
  
  g++;    
  if(getline(buf,in)==-1)
    {
    fprintf(stderr,"sod2: missing `*blocks' section.\n");
    exit(1);
    }
  }

if(g==0)
  {
  fprintf(stderr,"sod2: must be some patterns in the block list!\n");
  exit(1);
  }

printf("defb 255\n\nblocks:\n");


/* now interpret the patterns, and write them as we go along. */
f=1;
while(getline(buf,in)!=-1)
  {
  printf(";p%d\n",f);
  check_linelen(buf,bsize,0);
  
  for(i=0;i<bsize;i++) vbuf[i]='9';	/* max relative vol. */
  vbuf[bsize]=0;
  for(i=0;i<bsize;i++) sbuf[i]=' ';
  sbuf[bsize]=0;
  for(i=0;i<bsize;i++) sbuf2[i]=' ';	/* space=use sod_pos */
  sbuf2[bsize]=0;
  
  sod_time=10000; sod_down=10; sod_pos=0;
  while(buf[0]=='*')
    {
    switch(buf[1])
      {
      case 'v':
        getline(vbuf,in);
        check_linelen(vbuf,bsize,0);
        padline(vbuf,bsize);
        break;
      }
    getline(buf,in);
    check_linelen(buf,bsize,0);
    }

  padline(buf ,bsize);
  getline(buf2,in); check_linelen(buf2,bsize,0); padline(buf2,bsize);
  getline(buf3,in); check_linelen(buf3,bsize,1);
  addline(f,0,buf,buf2,buf3,sbuf,sbuf2,vbuf,sod_time,sod_down,sod_pos);
  f++;
  }

/* freqency table - starts at one, as idx zero is implicitly zero */
for(f=1;f<num_freqs;f++)
  fprintf(stderr,"%d\n",freqtable[f]);
}


void addline(int num,int mode,
	char *notes,char *octaves,char *sample,
        char *stereo1,char *stereo2,char *relvol,
	int sod_time,int sod_down,int sod_pos)
{
int note=0,f,g,gotnote,vol,tmp,noisebit;
char *ptr;
double etime=0.;
int samplenum=-1;

ptr=strchr(sample,',');
if(ptr==NULL)
  vol=DEFAULTVOLUME;
else
  {
  *ptr=0;
  ptr++;
  vol=atoi(ptr);
  }

if(strlen(sample)>10)
  {
  samplenum=-1;
  }
else
  {
  tmp=sample[0];
  if(tmp>'Z') tmp-=32;
  tmp-='0';
  if(tmp>9) tmp-=7;
  
  samplenum=tmp;
  }

if(strlen(notes)==bsize)
  {
  int freq;
  
  for(f=0;f<bsize;f++)
    {
    gotnote=1;
    switch(notes[f])
      {
      case 'c': note= 0; break;
      case 'C': note= 1; break;
      case 'd': note= 2; break;
      case 'D': note= 3; break;
      case 'e': note= 4; break;
      case 'f': note= 5; break;
      case 'F': note= 6; break;
      case 'g': note= 7; break;
      case 'G': note= 8; break;
      case 'a': note= 9; break;
      case 'A': note=10; break;
      case 'b': note=11; break;
      default:  gotnote=0;
      }

    noisebit=0;
    if(gotnote)
      {
      etime=0.;
      freq=(int)(440.*pow(2.,octaves[f]-'0'-OCTAVEFIX+(note-9.)/12.));
      
      if(samplenum==-1)
        {
        tmp=sample[f];
        if(tmp>'Z') tmp-=32;
        tmp-='0';
        if(tmp>9) tmp-=7;
        }
      else
        tmp=samplenum;

      /* noise stuff, based on sample number */
      switch(tmp)
        {
        case 3: noisebit=8+4; break;	/* bass */
        case 4: noisebit=8+2; break;	/* snare */
        case 5:
        case 6: noisebit=8+1; break;	/* hihat */
        }
      }
    else
      freq=0;

    freq/=2;
    
    /* On a Spectrum, the clock pulse is 1773400Hz, but on a CPC, it is
       1000000Hz */

    if(freq) freq=1000000/(16*freq);

    for(g=0;g<num_freqs;g++)
      {
      if(freqtable[g]==freq)
        {
        freq=g;
        break;
        }
      }
    
    if(g==num_freqs)
      {
      freqtable[num_freqs++]=freq;
      freq=g;
      }
    
    printf("%s%d%c",((f&15)==0)?"defb ":"",
           ((relvol[f]=='9' || !freq)?freq:freq+128),
           ((f&15)==15)?'\n':',');
    }
  }
}


/* well, it works :) */
void padline(char *str,int n)
{
while(strlen(str)<n)
  strcat(str," ");
}
