/* Convert sprite and font data from a CPC screen file and write it to two
   separate files (one for the sprites, one for the font) in the form of defb
   statements, so that it can be used by an assembler

   Program written by Nicholas Campbell
   Started 1st May 2005
   Finished 7th May 2005
   
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>

#define AMSDOS_HEADER_SIZE 128
#define CPC_SCREEN_SIZE 0x4000

unsigned char amsdos_header[AMSDOS_HEADER_SIZE];
  /* Memory reserved for AMSDOS header */
unsigned char cpc_screen_mem[CPC_SCREEN_SIZE];
  /* Memory reserved for the CPC screen */
int i;  /* Variable used in loops */


/* ---------
   FUNCTIONS
   --------- */

/* check_amsdos_header

   Check if there is an AMSDOS header at the start of a file
  
   An AMSDOS header is 128 bytes long. A 16-bit checksum is stored at bytes 67
   and 68 of the file, and it is calculated by adding bytes 0-66 together. If
   the calculated checksum does not match the value stored at bytes 67 and 68,
   it is assumed that there is no AMSDOS header at the start of the file
   
   Parameters:
   filename - a string containing the name of the file to check
   
   Returns:
   0 if the file does not contain an AMSDOS header, or 1 if it does
   
*/

int check_amsdos_header(unsigned char *filename) {

  /* Declare variables */
  
  short unsigned int checksum_in_amsdos_header;
    /* Checksum at bytes 67 and 68 of file */
  short unsigned int checksum_calc = 0;  /* Calculated checksum */

  /* Allocate memory to store the AMSDOS header, then read the first 128 bytes
     of the file */

  FILE *file_in = fopen(filename,"rb");
  fread(amsdos_header,1,AMSDOS_HEADER_SIZE,file_in);
  fclose(file_in);
    
  /* Calculate the checksum */
      
  checksum_in_amsdos_header = amsdos_header[67] + (amsdos_header[68]*256);
  for (i = 0; i <= 66; i++) { checksum_calc += amsdos_header[i]; }
    
  /* Return 0 if the file does not contain an AMSDOS header (i.e. the checksums
     did not match), or 1 if it does */

  if (checksum_calc != checksum_in_amsdos_header) {
	return(0);
  } else { return(1); }
  		
}


/* write_sprite_to_file

   Read sprite data from a CPC screen and write to a file in the form of defb
   statements for use with an assembler
   
   Note that although the CPC screen is 80 bytes x 200 lines in size, this
   routine treats the screen as a 40 x 25 grid when specifying sprite
   coordinates and sizes
   
   Parameters:
   file_out - a pointer to a file stream
   sprite_x - the x-coordinate of the sprite on the CPC screen (0-39)
   sprite_y - the y-coordinate of the sprite on the CPC screen (0-24)
   width - the width of the sprite in columns (not bytes)
   height - the height of the sprite in rows (not lines)
   sprite_name - a string containing the name of the sprite. This is used as a
    label for the assembler and is written before the block of defb statements.
    If it is left blank, no label will be written
        
   Returns:
   Nothing
   
*/

void write_sprite_to_file(FILE *file_out, unsigned int sprite_x,
  unsigned int sprite_y, unsigned int width, unsigned int height,
  unsigned char *sprite_name) {

  unsigned int screen_x, screen_y, screen_y2;  /* Variables used in loops */

  /* If a name has been given for the sprite, write it and follow it with a
     colon */
    
  if (strlen(sprite_name) != 0) { fprintf(file_out,"%s:\n",sprite_name); }

  /* Write the width and height of the sprite in bytes, not in rows and
     columns
     
     Each column consists of 2 bytes, and each row consists of 8 bytes */
  
  fprintf(file_out,"defb %d,%d\n",width*2,height*8);

  /* Start reading the sprite data from the CPC screen and writing it to
     defb statements for use with the assembler */
     
  for (screen_y = 0; screen_y < height; screen_y++) {

	/* Each row of the screen (screen_y) consists of eight lines (screen_y2) */
	  
    for (screen_y2 = 0; screen_y2 < 8; screen_y2++) {
      fprintf(file_out,"defb ");

      /* On a CPC screen, each line is 80 bytes long; however, this routine
         treats the screen as having 40 columns
         
         The formula for calculating which byte of the screen to read, given a
         set of x- and y-coordinates, is:

         byte = (y*80) + (y2 * 0x800) + x
         
         where x = 0-79 (column), y = 0-24 (row), y2 = 0-7 (line in row) */
      
      for (screen_x = 0; screen_x < width*2; screen_x++) {
        fprintf(file_out,"&%02x",cpc_screen_mem[
          ((sprite_y+screen_y)*80)+(screen_y2*0x800)+(sprite_x*2)+screen_x]);

        if (screen_x < (width*2)-1) { fprintf(file_out, ","); }
        else { fprintf(file_out,"\n"); }
      }

    }

  }

  /* Write a newline at the end of the block of defb statements, to separate
     it from the next block of sprite data */
    
  fprintf(file_out,"\n");
  	
}


/* write_font_to_file

   Read font data from a CPC screen in mode 0, convert it to a bitwise
   representation where each pixel uses only 1 bit, and write to a file in the
   form of defb statements for use with an assembler

   Note that although the CPC screen is 80 bytes x 200 lines in size, this
   routine treats the screen as a 40 x 25 grid when specifying sprite
   coordinates and sizes

   Each character is 2 x 8 bytes in size and can be drawn in any ink number from
   1-15

   Parameters:
   file_out - a pointer to a file stream
   char_x - the x-coordinate of the first character on the CPC screen (0-39)
   char_y - the y-coordinate of the first character on the CPC screen (0-24)
        
   Returns:
   Nothing
   
*/
       
void write_font_to_file(FILE *file_out, unsigned int char_x,
  unsigned int char_y) {

  unsigned int char_ascii, screen_y;  /* Variables used in loops */
  signed int screen_x;
  unsigned char char_byte;   /* The character byte for the current line */
  unsigned char screen_byte;   /* The byte that is read from the CPC screen */

  fprintf(file_out,"char_table:\n");

  /* Start at ASCII character 32 (space) and finish at ASCII character 127
     (delete) */
  
  for (char_ascii = 32; char_ascii < 128; char_ascii++) {

	fprintf(file_out,"defb ");

	/* Each character consists of 8 lines */
		  
	for (screen_y = 0; screen_y < 8; screen_y++) {
	  char_byte = 0;   /* Reset the character byte for the current line */

	  /* Each line in a character consists of 2 bytes
	  
	     This routine works by reading each pixel of the line. Each byte
	     consists of 2 pixels in CPC screen mode 0. The pixels are read and
	     stored in bits 4-7 of char_byte; bit 7 is the left-most pixel, and bit
	     4 is the right-most pixel. The bits are added to char_byte one at a
	     time, and char_byte is shifted 1 bit to the right. This means that the
	     right-most pixel must be read first, so the routine reads the CPC
	     screen bytes from right to left */
	  	  
	  for (screen_x = 1; screen_x >= 0; screen_x--) {

        /* On a CPC screen, each line is 80 bytes long; however, this routine
           treats the screen as having 40 columns

           The formula for calculating which byte of the screen to read, given a
           set of x- and y-coordinates, is:

           byte = (y*80) + (y2 * 0x800) + x
         
           where x = 0-79 (column), y = 0-24 (row), y2 = 0-7 (line in row) */
 
        screen_byte = cpc_screen_mem[
          (char_y*80)+(screen_y*0x800)+(char_x*2)+screen_x];

        /* The ink number of the right-most pixel is stored in bits 6, 4, 2 and
           0 of the screen byte. ANDing 0x55 (= 01010101 in binary) retains
           these bits and removes information about the left-most pixel
           
           If the ink number of the right-most pixel is anything other than 0,
           set the appropriate bit in the character byte */

        if ((screen_byte & 0x55) != 0) { char_byte = char_byte | 0x80; }
        char_byte >>= 1;   /* Shift the character byte 1 bit to the right */
        
        /* The ink number of the left-most pixel is stored in bits 7, 5, 3 and
           1 of the screen byte. ANDing 0xaa (= 10101010 in binary) retains
           these bits and removes information about the right-most pixel */

        if ((screen_byte & 0xaa) != 0) { char_byte = char_byte | 0x80; }

        /* Do not shift the character byte to the right if this is the last
           pixel to place in the byte */

        if (screen_x > 0) { char_byte >>= 1; }
        
      }

      /* Write the character byte to the file */
      
      fprintf(file_out,"&%02x",char_byte);
      if (screen_y < 7) { fprintf(file_out,","); }
      else { fprintf(file_out,"\n"); }
      
    }

    /* Go to the next character, and set the x- and y-coordinates */
    
    char_x++;
    if (char_x > 39) { char_x = 0; char_y++; }
    	  	  
  }

  /* Write a newline at the end of the block of defb statements, to separate
     it from the next block of sprite data */
    
  fprintf(file_out,"\n");  

}


/* convert_pixels_to_screen_byte

   Convert two pixels (left and right) to the corresponding byte that represents
   these pixels on a CPC screen in MODE 0
   
   The screen display on a CPC in MODE 0 is rather strange. If the bits of the
   left pixel are labelled L3 L2 L1 L0, and the bits of the right pixel are
   labelled R3 R2 R1 R0, the corresponding screen byte is
   L0 R0 L2 R2 L1 R1 L3 R3
   
   Parameters:
   left_pixel - the ink number of the left pixel (0-15)
   right_pixel - the ink number of the right pixel (0-15)
   
   Returns:
   screen_byte - the byte representing the left and right pixels on a CPC screen
   in MODE 0
   
*/

unsigned char convert_pixels_to_screen_byte(unsigned char left_pixel,
  unsigned char right_pixel) {
	  
  unsigned char screen_byte = 0;  /* Reset the screen byte */

  /* Firstly, read bits 3-0 of the left pixel one at a time in that order by
     resetting all the other bits, move the bit to its correct location in the
     screen byte, then OR it with the current screen byte for all the other
     bits that have been read so far */
  
  screen_byte = (left_pixel & 0x08)>>2;
  screen_byte |= (left_pixel & 0x04)<<3;
  screen_byte |= (left_pixel & 0x02)<<2;
  screen_byte |= (left_pixel & 0x01)<<7;

  /* Do the same for bits 3-0 of the right pixel */
  
  screen_byte |= (right_pixel & 0x08)>>3;
  screen_byte |= (right_pixel & 0x04)<<2;
  screen_byte |= (right_pixel & 0x02)<<1;
  screen_byte |= (right_pixel & 0x01)<<6;  

  /* Return the corresponding CPC screen byte */
    
  return(screen_byte);
  		
}


/* ------------
   MAIN PROGRAM
   ------------ */

int main(int argc, char *argv[]) {
	
  /* Declare variables */
	
  struct stat file_info;  /* Buffer to store file information; the stat
                             structure is defined in sys/types.h */
  unsigned char *cpc_screen_filename;  /* Name of CPC file containing sprite data */
  unsigned char *sprite_data_filename;  /* Name of output file to write sprite data */
  unsigned char *font_data_filename;  /* Name of output file to write font data */
  unsigned int cpc_screen_size;  /* Size of input file containing sprite data */
  int amsdos_header_exists;  /* Flag to determine if a file contains an AMSDOS
    header */
  FILE *file_in, *file_out;  /* Streams for reading and writing files */
  unsigned char char_style[9][2] = { {0,8},{0,9},{0,10},{0,11},{0,12},{0,13},
    {0,14},{0,15},{15,9} };  /* Array to store the background and foreground
    ink numbers */
  unsigned char screen_byte;  /* Temporary variable used to store conversions of
    background and foreground ink numbers to the corresponding CPC screen byte
    */


  /* Check the command line arguments
  
     If there are no filenames specified, print an error message */
  
  if (argc < 4) {
    fputs("USAGE: sprtodat input_filename sprite_output_filename font_output_filename\n",stderr);
    exit(1);
  }
	
  /* Get the filename of the CPC screen from the command line and store it in
     another string */
  
  cpc_screen_filename = (char *)malloc(strlen(argv[1])+1);
  strcpy(cpc_screen_filename,argv[1]);
  
  sprite_data_filename = (char *)malloc(strlen(argv[2])+1);
  strcpy(sprite_data_filename,argv[2]);

  font_data_filename = (char *)malloc(strlen(argv[3])+1);
  strcpy(font_data_filename,argv[3]);

  /* If the CPC screen file does not exist, print an error message */
  
  if (stat(cpc_screen_filename,&file_info)) {
	fprintf(stderr, "ERROR: Unable to open %s!\n",cpc_screen_filename);
	exit(1);
  }

  /* Get information about the CPC screen file and store it in an array */
    
  else {
	cpc_screen_size = file_info.st_size;

	/* A standard-sized CPC screen is 0x4000 bytes long, or 0x4080 bytes if the
       AMSDOS header is included. If the file does not match either of these
       lengths, assume that it is not a CPC screen file and print an error
       message */
		
	if ((cpc_screen_size != CPC_SCREEN_SIZE) &&
	  (cpc_screen_size != AMSDOS_HEADER_SIZE + CPC_SCREEN_SIZE)) {
	  fprintf(stderr,"ERROR: %s is not a CPC screen file!\n",
	    cpc_screen_filename);
	  exit(1);
    }
  }
  
  /* Check if the CPC screen file has an AMSDOS header before reading it */

  amsdos_header_exists = check_amsdos_header(cpc_screen_filename); 
    
  /* Open the CPC screen file for reading */

  file_in = fopen(cpc_screen_filename,"rb");

  /* If there is an AMSDOS header at the start of the file, read it into the
     memory reserved for the CPC screen; it will be overwritten, and it is no
     longer required */

  if (amsdos_header_exists) {
	fread(cpc_screen_mem,1,AMSDOS_HEADER_SIZE,file_in);
  }

  /* Read the CPC screen into memory, close the file and free memory allocated
     for the CPC screen filename */
  
  fread(cpc_screen_mem,1,CPC_SCREEN_SIZE,file_in);
  fclose(file_in);
  free(cpc_screen_filename);
    
  /* Open the sprite data file */
 
  file_out = fopen(sprite_data_filename,"w");

  /* Read the sprite for the player's shot */
  
  write_sprite_to_file(file_out,0,6,1,1,"plyrshot_dat");

  /* Read the sprites for the enemies' shots (there are 4 of them, each
     representing a different colour) */

  fprintf(file_out,"enemyshot_dat:\n");
  for (i = 0; i < 4; i++) {
    write_sprite_to_file(file_out,i,8,1,1,"");
  }
  
  /* Read the sprites for baddie types 1, 3 and 4 */
  
  fprintf(file_out,"type134_dat:\n");
  for (i = 0; i < 3; i++) {
    write_sprite_to_file(file_out,(i*2)+1,6,2,2,"");
  }

  /* Read the sprites for the player's ship, with the shield activated and
     deactivated */
    
  fprintf(file_out,"player_dat:\n");
  for (i = 0; i < 2; i++) {
    write_sprite_to_file(file_out,7,6+(i*2),3,2,"");
  }

  /* Read the sprites for baddie types 5, 10 and 11 */
    
  write_sprite_to_file(file_out,0,9,3,3,"type5_dat");
  write_sprite_to_file(file_out,0,12,4,5,"type10_dat");
  write_sprite_to_file(file_out,4,12,6,6,"type11_dat");

  /* Read the sprites for the debris (there are 7 of them, each representing a
     different colour) */
    
  fprintf(file_out,"debris_dat:\n");
  for (i = 0; i < 7; i++) {
    write_sprite_to_file(file_out,i+3,10,1,1,"");
  }

  /* Close the sprite data file */
  
  fclose(file_out);

  /* Open the font data file */
 
  file_out = fopen(font_data_filename,"w");

  /* Convert the font sprites to a bitwise representation and write them to the
     file */
    
  write_font_to_file(file_out,0,0);

  /* Text can be displayed on the CPC screen in any one of nine styles. Each
     style consists of a background ink number (0-15) and a foreground ink
     number (0-15) which the text will be written in
     
     The background and foreground numbers for each style are stored in a two-
     dimensional array. Index x,0 of the array contains the background ink
     number, and index x,1 contains the foreground ink number */
  
  fprintf(file_out,"char_ink_table_temp:\n");
  
  for (i = 0; i < 9; i++) {
  
    /* Each screen byte on the CPC in MODE 0 consists of 2 pixels, so if only
       2 colours are being used, as is the case when writing text, there are
       four possible combinations. Each possible combination is converted and
       the resulting screen byte is written to the file */

    screen_byte = convert_pixels_to_screen_byte(char_style[i][0],
      char_style[i][0]);
    fprintf(file_out,"defb &%02x,",screen_byte);

    screen_byte = convert_pixels_to_screen_byte(char_style[i][0],
      char_style[i][1]);
    fprintf(file_out,"&%02x,",screen_byte);

    screen_byte = convert_pixels_to_screen_byte(char_style[i][1],
      char_style[i][0]);
    fprintf(file_out,"&%02x,",screen_byte);

    screen_byte = convert_pixels_to_screen_byte(char_style[i][1],
      char_style[i][1]);
    fprintf(file_out,"&%02x\n",screen_byte);

  }

  fprintf(file_out,"char_ink_table_temp_end:\n");

  /* Close the font data file */
      
  fclose(file_out);
  
  return(0);
    
}
