/* Convert sprite and font data from a CPC screen file and write it to two
   separate files (one for the sprites, one for the font) in the form of defb
   statements, so that it can be used by an assembler

   Program written by Nicholas Campbell
   Started 1st May 2005
   Finished 7th May 2005

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>

#define AMSDOS_HEADER_SIZE 128
#define CPC_SCREEN_SIZE 0x4000

unsigned char amsdos_header[AMSDOS_HEADER_SIZE];
  /* Memory reserved for AMSDOS header */
unsigned char cpc_screen_mem[CPC_SCREEN_SIZE];
  /* Memory reserved for the CPC screen */
int i;  /* Variable used in loops */


/* ---------
   FUNCTIONS
   --------- */

/* check_amsdos_header

   Check if there is an AMSDOS header at the start of a file

   An AMSDOS header is 128 bytes long. A 16-bit checksum is stored at bytes 67
   and 68 of the file, and it is calculated by adding bytes 0-66 together. If
   the calculated checksum does not match the value stored at bytes 67 and 68,
   it is assumed that there is no AMSDOS header at the start of the file

   Parameters:
   filename - a string containing the name of the file to check

   Returns:
   0 if the file does not contain an AMSDOS header, or 1 if it does

*/

int check_amsdos_header(char *filename) {

  /* Declare variables */

  short unsigned int checksum_in_amsdos_header;
    /* Checksum at bytes 67 and 68 of file */
  short unsigned int checksum_calc = 0;  /* Calculated checksum */

  /* Allocate memory to store the AMSDOS header, then read the first 128 bytes
     of the file */

  FILE *file_in = fopen(filename,"rb");
  fread(amsdos_header,1,AMSDOS_HEADER_SIZE,file_in);
  fclose(file_in);

  /* Calculate the checksum */

  checksum_in_amsdos_header = amsdos_header[67] + (amsdos_header[68]*256);
  for (i = 0; i <= 66; i++) { checksum_calc += amsdos_header[i]; }

  /* Return 0 if the file does not contain an AMSDOS header (i.e. the checksums
     did not match or the calculated checksum was 0, which means that the
     AMSDOS header consists entirely of zeroes), or 1 if it does */

  if ((checksum_calc != checksum_in_amsdos_header) || (checksum_calc == 0)) {
	return(0);
  } else { return(1); }

}


/* convert_pixels

   Convert a set of four MODE 1 pixels from the format used to write pixels to
   the screen, and store them in a more sensible way to make it easier to
   rotate them

   The screen display on a CPC in MODE 1 is rather strange. If the leftmost
   pixel is assigned as (1,0 0,0) and the rightmost pixel is assigned as
   (1,3 0,3), the corresponding screen byte is:
   0,0 0,1 0,2 0,3 1,0 1,1 1,2 1,3
   This routine converts it to the following:
   1,0 0,0 1,1 0,1 1,2 0,2 1,3 0,3

   Parameters:
   screen_byte - the byte representing the four pixels, in the format used to
   write pixels to the screen

   Returns:
   mem_byte - the byte representing the four pixels, in the format used by the
   program's sprite display routine (which must allow for rotating the
   sprites)

*/

unsigned char convert_pixels(unsigned char screen_byte) {

  unsigned char mem_byte = 0;   /* Reset the memory byte */

  /* Read the leftmost pixel (pixel 0) */

  mem_byte = ((screen_byte & 0x08)<<4) | ((screen_byte & 0x80)>>1);
  mem_byte |= ((screen_byte & 0x04)<<3) | ((screen_byte & 0x40)>>2);
  mem_byte |= ((screen_byte & 0x02)<<2) | ((screen_byte & 0x20)>>3);
  mem_byte |= ((screen_byte & 0x01)<<1) | ((screen_byte & 0x10)>>4);

  /* Return the corresponding CPC screen byte */

  return(mem_byte);

}


/* write_sprite_to_file

   Read sprite data from a CPC screen and write to a file in the form of defb
   statements for use with an assembler

   Note that although the CPC screen is 80 bytes x 200 lines in size, this
   routine treats the screen as a 40 x 25 grid when specifying sprite
   coordinates and sizes

   Parameters:
   file_out - a pointer to a file stream
   sprite_x - the x-coordinate of the sprite on the CPC screen (0-39)
   sprite_y - the y-coordinate of the sprite on the CPC screen (0-24)
   width - the width of the sprite in columns (not bytes)
   height - the height of the sprite in rows (not lines)
   sprite_name - a string containing the name of the sprite. This is used as a
    label for the assembler and is written before the block of defb statements.
    If it is left blank, no label will be written

   Returns:
   Nothing

*/

void write_sprite_to_file(FILE *file_out, unsigned int sprite_x,
  unsigned int sprite_y, unsigned int width, unsigned int height,
  char *sprite_name, unsigned int rotated) {

  unsigned int screen_x, screen_y, screen_y2;  /* Variables used in loops */
  unsigned int byte_counter = 0;
  unsigned int line_counter = 0;
  unsigned char pixel;   /* Pixel read from the screen containing the
                            sprites */

  /* If a name has been given for the sprite, write it and follow it with a
     colon */

  if (strlen(sprite_name) != 0) { fprintf(file_out,"%s",sprite_name); }
  else { fprintf(file_out,"      "); }

  /* Start reading the sprite data from the CPC screen and writing it to
     defb statements for use with the assembler */

  for (screen_y = 0; screen_y < height; screen_y++) {

	/* Each row of the screen (screen_y) consists of eight lines (screen_y2) */

    for (screen_y2 = 0; screen_y2 < 8; screen_y2++) {

      /* On a CPC screen, each line is 80 bytes long; however, this routine
         treats the screen as having 40 columns

         The formula for calculating which byte of the screen to read, given a
         set of x- and y-coordinates, is:

         byte = (y*80) + (y2 * 0x800) + x

         where x = 0-79 (column), y = 0-24 (row), y2 = 0-7 (line in row) */

      for (screen_x = 0; screen_x < width*2; screen_x++) {

        if (byte_counter == 0) {
          if (line_counter != 0) { fprintf(file_out,"      "); }
          fprintf(file_out," defb ");
        }

        pixel = cpc_screen_mem[
          ((sprite_y+screen_y)*80)+(screen_y2*0x800)+(sprite_x*2)+screen_x];
        if (rotated) { fprintf(file_out,"&%02x",convert_pixels(pixel)); }
        else { fprintf(file_out,"&%02x",pixel); }
        byte_counter++;

        if (byte_counter < 8) { fprintf(file_out, ","); }
        else { fprintf(file_out,"\n"); byte_counter = 0; line_counter++; }
      }

    }

  }

  /* Write a newline at the end of the block of defb statements, to separate
     it from the next block of sprite data */

  fprintf(file_out,"\n");

}


/* ------------
   MAIN PROGRAM
   ------------ */

int main(int argc, char *argv[]) {

  /* Declare variables */

  struct stat file_info;  /* Buffer to store file information; the stat
                             structure is defined in sys/types.h */
  char *cpc_screen_filename;  /* Name of CPC file containing sprite data */
  char *sprite_data_filename;  /* Name of output file to write sprite data */
  unsigned int cpc_screen_size;  /* Size of input file containing sprite data */
  int amsdos_header_exists;  /* Flag to determine if a file contains an AMSDOS
    header */
  FILE *file_in, *file_out;  /* Streams for reading and writing files */


  /* Check the command line arguments

     If there are no filenames specified, print an error message */

  if (argc < 3) {
    fputs("USAGE: sprtodat input_filename sprite_output_filename\n",stderr);
    exit(1);
  }

  /* Get the filename of the CPC screen from the command line and store it in
     another string */

  cpc_screen_filename = (char *)malloc(strlen(argv[1])+1);
  strcpy(cpc_screen_filename,argv[1]);

  sprite_data_filename = (char *)malloc(strlen(argv[2])+1);
  strcpy(sprite_data_filename,argv[2]);

  /* If the CPC screen file does not exist, print an error message */

  if (stat(cpc_screen_filename,&file_info)) {
	fprintf(stderr, "ERROR: Unable to open %s!\n",cpc_screen_filename);
	exit(1);
  }

  /* Get information about the CPC screen file and store it in an array */

  else {
	cpc_screen_size = file_info.st_size;

	/* A standard-sized CPC screen is 0x4000 bytes long, or 0x4080 bytes if the
       AMSDOS header is included. If the file does not match either of these
       lengths, assume that it is not a CPC screen file and print an error
       message */

	if ((cpc_screen_size != CPC_SCREEN_SIZE) &&
	  (cpc_screen_size != AMSDOS_HEADER_SIZE + CPC_SCREEN_SIZE)) {
	  fprintf(stderr,"ERROR: %s is not a CPC screen file!\n",
	    cpc_screen_filename);
	  exit(1);
    }
  }

  /* Check if the CPC screen file has an AMSDOS header before reading it */

  amsdos_header_exists = check_amsdos_header(cpc_screen_filename);

  /* Open the CPC screen file for reading */

  file_in = fopen(cpc_screen_filename,"rb");

  /* If there is an AMSDOS header at the start of the file, read it into the
     memory reserved for the CPC screen; it will be overwritten, and it is no
     longer required */

  if (amsdos_header_exists) {
	fread(cpc_screen_mem,1,AMSDOS_HEADER_SIZE,file_in);
  }

  /* Read the CPC screen into memory, close the file and free memory allocated
     for the CPC screen filename */

  fread(cpc_screen_mem,1,CPC_SCREEN_SIZE,file_in);
  fclose(file_in);
  free(cpc_screen_filename);

  /* Open the sprite data file */

  file_out = fopen(sprite_data_filename,"w");

  /* Read the sprites for Fizzog */

  write_sprite_to_file(file_out,0,0,2,2,"playr0",1);
  write_sprite_to_file(file_out,2,0,2,2,"playr1",1);
  write_sprite_to_file(file_out,4,0,2,2,"playr2",1);

  write_sprite_to_file(file_out,7,0,2,2,"playl0",1);
  write_sprite_to_file(file_out,9,0,2,2,"playl1",1);
  write_sprite_to_file(file_out,11,0,2,2,"playl2",1);

  write_sprite_to_file(file_out,0,4,2,2,"cart0 ",1);
  write_sprite_to_file(file_out,2,4,2,2,"cart1 ",1);
  write_sprite_to_file(file_out,4,4,2,2,"cart2 ",1);

  write_sprite_to_file(file_out,7,4,2,2,"frog0 ",1);
  write_sprite_to_file(file_out,9,4,2,2,"frog1 ",1);
  write_sprite_to_file(file_out,11,4,2,2,"frog2 ",1);

  write_sprite_to_file(file_out,14,4,2,2,"spin0 ",1);
  write_sprite_to_file(file_out,16,4,2,2,"spin1 ",1);
  write_sprite_to_file(file_out,18,4,2,2,"spin2 ",1);
  write_sprite_to_file(file_out,20,4,2,2,"spin3 ",1);

  write_sprite_to_file(file_out,23,4,2,2,"comp0 ",1);
  write_sprite_to_file(file_out,25,4,2,2,"comp2 ",1);
  write_sprite_to_file(file_out,27,4,2,2,"comp1 ",1);

  write_sprite_to_file(file_out,0,8,1,1,"blkgfx",0);
  write_sprite_to_file(file_out,2,8,1,1,"",0);
  write_sprite_to_file(file_out,4,8,1,1,"",0);
  write_sprite_to_file(file_out,6,8,1,1,"",0);
  write_sprite_to_file(file_out,8,8,1,1,"",0);
  write_sprite_to_file(file_out,10,8,1,1,"",0);
  write_sprite_to_file(file_out,12,8,1,1,"",0);
  write_sprite_to_file(file_out,14,8,1,1,"",0);
  write_sprite_to_file(file_out,16,8,1,1,"",0);
  write_sprite_to_file(file_out,18,8,1,1,"",0);


  /* Close the sprite data file */

  fclose(file_out);

  return(0);

}
