/************************************************************************

    JOYCE v2.1.0 - Amstrad PCW emulator

    Copyright (C) 1996, 2001-2  John Elliott <jce@seasip.demon.co.uk>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*************************************************************************/

/** A vaguely BDOS-like interface - C=function and          **/
/** DE=parameter. It can also take values in B and HL.      **/
/** Returns HL=0 on failure, other values on success.       **/

#include "Pcw.hxx"
#include "PcwBdos.hxx"
#include "inline.hxx"

static FILE *handles[16];	/* Can open up to 15 files; handles[0] is 
				 * reserved */
static char pathname[162];	/* DOS pathname passed by Z80 program */

#define BCDIFY(x) (((x % 10)+16*(x/10)) & 0xFF)


void jdos_time(Z80 *R, int secflag)	/* Read the time into buffer at DE */
{
	time_t secs;			/* If secflag=1, include seconds */
	struct tm *ltm;
	int day, hour, min, sec, lyear;

	time(&secs);
					/* << v1.36: Use local time, 
					 *    rather than UTC. */
	ltm  = localtime(&secs);

	/* Work out the day count. */
	lyear = (ltm->tm_year - 76);	/* Years since 1976 */

	day = (lyear / 4) * 1461;	/* No. of days to start of 
					 * last leapyear */
	lyear = (lyear % 4);
	if (lyear)    { day += 366; --lyear; }
	while (lyear) { day += 365; --lyear; }
	/* Day = day count from 1976 to start of this year */
	day -= (365 + 366 - 1);	
	/* The extra 1 is because the CP/M day count is 1 based */
	day += ltm->tm_yday;

	hour = ltm->tm_hour;
	min  = ltm->tm_min;
	sec  = ltm->tm_sec;	
				/* >> v1.36 */
	if (!R->getDE()) return;
	R->storeB(R->getDE(), day & 0xFF);
	R->storeB(R->getDE() + 1, (day >> 8) & 0xFF);
	R->storeB(R->getDE() + 2, BCDIFY(hour));
	R->storeB(R->getDE() + 3, BCDIFY(min));
	if (secflag) R->storeB(R->getDE() + 4, BCDIFY(sec));
	R->a = BCDIFY(sec);

}
#undef BCDIFY


void txt2fcb(Z80 *R, word addr, char *s)	/* Parse ASCII filename to FCB */
{
	char curch;
	int fcnt = 0;

	while(fcnt < 11) switch(*s)
	{
		case 0:	while (fcnt < 11)
			{
				R->storeB(addr + fcnt, ' ');
				fcnt++;
			}
			return;

		case '.': while (fcnt < 8)
			{
				R->storeB(addr + fcnt, ' ');
				fcnt++;			
			}
			++s;
			break;

		case ':': 
		case '/':	
		case '\\':
			fcnt = 0;
			++s;
			break;

		default:
			/* v1.11: Deal with W95 or Unix filenames which
			 * may have lowercase letters in them */
			curch = (*s) & 0x7F; 
			if (islower(curch)) curch = toupper(curch);
			R->storeB(addr + fcnt, curch);
			++s;
			++fcnt;
			break;

	}
}


word jdos_creat(Z80 *R)	/* Create a file, DE->name */
{
	int n, m;
	char c;

	for (n = 1; n <16; n++)	/* Find a free handle */
	{
		if (!handles[n]) break;
	}
	if (n == 16) return 0;	/* No free handles */
	
	m = 0;
	do {
		c = R->fetchB(R->getDE() + m);	/* Extract pathname */
		pathname[m] = c;
		++m;
	} while (c);

	handles[n] = fopen(pathname, "w+b");	/* Create the file */

	if (!handles[n]) return 0;

	return n;
}

word jdos_open(Z80 *R)	/* Open a file, DE->name */
{
	int n, m;
	char c;

	for (n = 1; n <16; n++)	/* Find a free handle */
	{
		if (!handles[n]) break;
	}
	if (n == 16) return 0;	/* No free handles */
	
	m = 0;
	do {
		c = R->fetchB(R->getDE() + m);	/* Extract pathname */
		pathname[m] = c;
		++m;
	} while (c);

	handles[n] = fopen(pathname, "r+b");	/* Open the file */

	if (!handles[n]) /* Read/write failed. Try read-only */
		handles[n] = fopen(pathname, "rb");
	if (!handles[n]) return 0;

	return n;
}



word jdos_putc(Z80 *R)	/* Write byte to file */
{
	int n = R->getDE();

	if (!handles[n]) return 0;	/* Not open */

	if (fputc(R->b, handles[n]) == EOF) return 0;

	return 1;
}


word jdos_getc(Z80 *R)	/* Read a byte from a file into B */
{
	int n = R->getDE();
	int m;

	if (!handles[n]) return 0;	/* Not open */

	m = fgetc(handles[n]);
	if (m == EOF) return 0;		/* EOF */
	R->b = (m & 0xFF);
	return 1;
}

word jdos_close(Z80 *R)	/* Close file */
{
	int n = R->getDE();
	int m;

	if (!handles[n]) return 0;

	m = fclose(handles[n]);

	handles[n] = NULL;

	return(!m);	
}


void jdos(Z80 *R)
{
	switch(R->c)
	{
		case 0x0F:	/* Open */
		R->setHL( jdos_open(R) );
		break;

		case 0x10:	/* Close */
		R->setHL( jdos_close(R) );
		break;

		case 0x14:	/* Get byte from stream */
		R->setHL( jdos_getc(R) );
		break;

		case 0x15:	/* Write byte to stream */
		R->setHL( jdos_putc(R) );
		break;

		case 0x16:
		R->setHL( jdos_creat(R) ); /* Create file */
		break;

		case 105:
		jdos_time(R, 0);	/* Get the time */
		break;


		case 0x91:		/* Wildcard expansion */
		R->setHL( jdos_wildex(R) );
		break;

		default:
		joyce_dprintf("Unknown call to API function 10: "
			"C=%02x DE=%04x PC=%04x\n",
			R->c, R->getDE(), R->pc);
		break;
	}
}

