/*
    RTC Flash - Firmware updater for the M3002 RTC replacement 'Re3002'.
    Copyright (C) 2013  Oliver Tscherwitschke <oliver@tscherwitschke.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/


#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "main.h"
#include "ihex.h"
#include "rtc.h"


#define DO_WAIT 1
#define DONT_WAIT 0

#define DUMMY_WAIT 0

typedef union
{
    unsigned char data[16];

    struct {
        unsigned char sig_l;
        unsigned char sig_h;
        unsigned char ver_l;
        unsigned char ver_h;
        unsigned char cmd;
        unsigned char arg[4];
        unsigned char dummy[6];
        unsigned char status;
    } bl;
} BLDATA;


typedef union
{
    unsigned char uc[2];
    unsigned short us[1];
} FW_BUFFER;


// ---------------------------------------------------
// Command definitions

#define CMD_READ_SIG    0x81
#define CMD_SET_ADDR    0x82
#define CMD_WRITE_WORD  0x83
#define CMD_READ_WORD   0x84
#define CMD_PAGE_ERASE  0x85
#define CMD_PAGE_WRITE  0x86
#define CMD_START_FW    0x99

#define CMD_START_BOOT  0xBB




unsigned char blCommand(BLDATA* bldata, int wait)
{
    int i;
    unsigned char data;
    unsigned char rc;

    // putchar('0');

    rtcSync();

    for (i = 0; i < 4; i++)
    {
        rtcWrite(5 + i, bldata->bl.arg[i]);
    }

    // putchar('1');

    // issue command
    rtcWrite(4, bldata->bl.cmd);

    // putchar('2');

    // Wait for command execution
    if (wait)
    {
        do
        {
            rc = rtcRead(4, &data);
            // printf("%u %02X\n", rc, data);
        } while (rc != OK || data != 0);
    }
    // putchar('3');

    // read answer
    for (i = 0; i < 4; i++)
    {
        rtcRead(5 + i, &bldata->bl.arg[i]);
    }

    // putchar('4');

    return OK;
}


/////////////////////////////////////////////////////////////////////////////
int fwActive(unsigned char* ver_h, unsigned char* ver_l)
{
    unsigned char sig[2];
    unsigned char ver[2];
    unsigned char rc;
    unsigned char ctrl;
    int ret = 0;

    // Set to test mode
    rtcRead(15, &ctrl);        
    rtcWrite(15, ctrl | 0x80);

    rc = rtcRead(0, &sig[0]);
    if (rc == OK)
        rc = rtcRead(1, &sig[1]);

    if (rc == OK)
        rc = rtcRead(2, &ver[0]);

    if (rc == OK)
        rc = rtcRead(3, &ver[1]);

    if (rc == OK)
    {
        if (ver_l)
            *ver_l = ver[0];

        if (ver_h)
            *ver_h = ver[1];

        if (sig[0] == 'f' && sig[1] == 'w')
            ret = 1;
    }
    
    // End test mode
    rtcWrite(15, ctrl);

    return ret;
}

/////////////////////////////////////////////////////////////////////////////
int blActive(unsigned char* ver_h, unsigned char* ver_l)
{
    unsigned char sig[2];
    unsigned char ver[2];
    unsigned char rc;

    rc = rtcRead(0, &sig[0]);
    if (rc != OK)
        return 0;

    rc = rtcRead(1, &sig[1]);
    if (rc != OK)
        return 0;

    rc = rtcRead(2, &ver[0]);
    if (rc != OK)
        return 0;

    rc = rtcRead(3, &ver[1]);
    if (rc != OK)
        return 0;

    if (ver_l)
        *ver_l = ver[0];

    if (ver_h)
        *ver_h = ver[1];

    if (sig[0] == 'b' && sig[1] == 'l')
    {
        return 1;
    }

    return 0;
}

/////////////////////////////////////////////////////////////////////////////
int startBootloader(unsigned char* ver_h, unsigned char* ver_l)
{
    BLDATA bldata = {{0}};
    unsigned char rc;
    unsigned char ctrl;

    // Set to test mode
    rtcRead(15, &ctrl);        
    rtcWrite(15, ctrl | 0x80);

    // Execute "Start bootloader" command
    bldata.bl.cmd = CMD_START_BOOT;
    bldata.bl.arg[0] = 0x42;
    rc = blCommand(&bldata, DO_WAIT);
    if (rc != OK)
        return ERROR;

    if (blActive(ver_h, ver_l))
        return OK;

    return ERROR;
}


/////////////////////////////////////////////////////////////////////////////
void printHelp(void)
{
    printf("Syntax: rtcflash [OPTIONS] filename\n");
    printf("  OPTIONS:\n");
    printf("    -p IOPORT   :   Set I/O Address of RTC to IOPORT (hex)\n");
}

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
int main(int argc, char *argv[])
{
    BLDATA bldata = {{0}};

    char filename[16] = "";
    long rc;
    FILE* infile;
    unsigned int addr;
    unsigned int word_count;
    unsigned long max_addr;
    unsigned char ver_h, ver_l;
    int i;
    FW_BUFFER* buffer = 0;
    int ret = 0;

    printf("RTCFLASH %u.%u. Re3002 firmware update utility.\n", VERSION_MAJOR, VERSION_MINOR);
    printf("%s\n\n", COPYRIGHT);

    if (argc == 1)
    {
        printHelp();
        return 1;
    }

    ////////////////////////////////////////////////////////////////////////////
    // Parse arguments
    for (i = 1; i < argc; i++)
    {
        if (strcmp(argv[i], "-h") == 0)
        {
            printHelp();
            return 0;
        }
        else if (strcmp(argv[i], "-p") == 0)
        {
            if (i < (argc - 1)) // another arg available?
            {
                rtc_io_port = (unsigned int) strtoul(argv[i + 1], NULL, 16);
            }
            else
            {
                printf("Error: argument missing after '-p'\n");
                return 1;
            }
        }
        else // assume filename
        {
            strncpy(filename, argv[i], sizeof filename - 1);
        }
    }


    ////////////////////////////////////////////////////////////////////////////
    // Allocate momory
    buffer = (FW_BUFFER*) malloc(MAX_FW_SIZE);
    if (buffer == NULL)
    {
        printf("Error: can't allocate memory (%u bytes)\n", MAX_FW_SIZE);
        return 1;
    }


    ////////////////////////////////////////////////////////////////////////////
    // read input file
    infile = fopen(filename, "rt");
    if (infile == NULL)
    {
        printf("Error: can't open input file '%s'\n", filename);
        if (buffer)
            free(buffer);
        return 1;
    }

    rc = read_hex_file(buffer->uc, MAX_FW_SIZE, infile);
    if (rc == -1)
    {
        printf("Error: can't read input file\n");
        if (buffer)
            free(buffer);
        return 2;
    }
    max_addr = rc;


    ////////////////////////////////////////////////////////////////////////////
    // Detect bootloader
    rtcSync();
    if (fwActive(&ver_h, &ver_l))
    {
        printf("Re3002 Firmware %u.%u detected.\n", ver_h, ver_l);
        printf("Starting bootloader... ");
        if (startBootloader(&ver_h, &ver_l) == OK)
            printf("OK, version %u.%u\n", ver_h, ver_l);
        else
            printf("ERROR\n");
    }
    else if (blActive(&ver_h, &ver_l))
    {
        printf("Re3002 Bootloader %u.%u detected.\n", ver_h, ver_l);
    }
    else
    {
        printf("Error: Can't communicate with Re3002\n");
        if (buffer)
            free(buffer);
        return 1;
    }


    ////////////////////////////////////////////////////////////////////////////
    // Flash firmware
    printf("Writing...\n");
    addr = 0;
    word_count = 0;
    for (addr = 0; addr < max_addr; addr += 2)
    {
        if (word_count == 0)
        {
            // set address
            bldata.bl.cmd = CMD_SET_ADDR;
            bldata.bl.arg[0] = addr & 0x00FF;
            bldata.bl.arg[1] = addr >> 8;

            rc = blCommand(&bldata, DO_WAIT);
            printf("\nADDR: %04X (%u)  ", addr, rc);

            // page erase
            bldata.bl.cmd = CMD_PAGE_ERASE;
            rc = blCommand(&bldata, DO_WAIT);
            if (rc == OK)
                putchar('e');
            else
                putchar('E');
        }

        // write word
        bldata.bl.cmd = CMD_WRITE_WORD;
        bldata.bl.arg[0] = buffer->us[addr / 2] & 0x00FF;
        bldata.bl.arg[1] = buffer->us[addr / 2] >> 8;

        // printf("%02X%02X", bldata.bl.arg[1], bldata.bl.arg[0]);

        rc = blCommand(&bldata, DO_WAIT);
        if (rc == OK)
            putchar('W');
        else
            putchar('E');


        word_count++;
        if (word_count == 32)
        {
            // page write
            bldata.bl.cmd = CMD_PAGE_WRITE;
            rc = blCommand(&bldata, DO_WAIT);

            if (rc == OK)
                putchar('P');
            else
                putchar('E');

            word_count = 0;
        }
    }

    if (word_count != 0)
    {
        // page write
        bldata.bl.cmd = CMD_PAGE_WRITE;
        rc = blCommand(&bldata, DO_WAIT);

        if (rc == OK)
            putchar('P');
        else
            putchar('E');
    }


    ////////////////////////////////////////////////////////////////////////////
    // Verify firmware
    printf("\n\nVerifying...\n");
    addr = 0;
    word_count = 0;
    for (addr = 0; addr < max_addr; addr += 2)
    {
        unsigned short verify_data;

        if (word_count == 0)
        {
            // set address
            bldata.bl.cmd = CMD_SET_ADDR;
            bldata.bl.arg[0] = addr & 0x00FF;
            bldata.bl.arg[1] = addr >> 8;

            rc = blCommand(&bldata, DO_WAIT);
            printf("\nADDR: %04X (%u)  ", addr, rc);
        }

        // read word
        bldata.bl.cmd = CMD_READ_WORD;
        rc = blCommand(&bldata, DO_WAIT);
        if (rc == OK)
            putchar('R');
        else
            putchar('E');

        verify_data = (bldata.bl.arg[1] << 8) | bldata.bl.arg[0];
        if (buffer->us[addr / 2] != verify_data)
        {
            printf("Error: Verify failed at %04X. Should be %04X, read %04X\n", addr, buffer->us[addr / 2], verify_data);
            ret = 1;
            break;
        }

        word_count++;
        if (word_count == 32)
            word_count = 0;
    }


    ////////////////////////////////////////////////////////////////////////////
    // Start firmware
    if (ret == 0)
    {
        printf("\n\nStarting Firmware... ");
        bldata.bl.cmd = CMD_START_FW;
        rc = blCommand(&bldata, DONT_WAIT);
        if (rc == OK)
            printf("OK\n");
        else
            printf("Error\n");
    }

    if (buffer)
        free(buffer);

	return ret;
}

