/*
    Re3002 - An M3002 RTC replacement
    Copyright (C) 2013  Oliver Tscherwitschke <oliver@tscherwitschke.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/


.include <m88PAdef.inc>

.EQU FW_VERSION     = 0x0100
.EQU FW_SIGNATURE   = 0x7766

.EQU F_CPU          = 20000000

.EQU BOOT_RST_ADDR  = 0xC00

.EQU USE_DEBUG_PINS = 0

.EQU DEBUG_OUTPUT   = 1

.EQU INPORT         = PINC            ; input port for address/data bus (bits 3..0)
.EQU OUTPORT        = PORTD           ; output port for data (bits 7..4)


.MACRO BUSY_PIN_H      
    sbi PORTB, 2    ; set /BUSY pin to high level (inacative)
.ENDMACRO

.MACRO BUSY_PIN_L      
    cbi PORTB, 2    ; set /BUSY pin to low level (acative)
.ENDMACRO


.MACRO BUSY
    ldi temp, 0xF0
    out OUTPORT, temp
    BUSY_PIN_L
.ENDMACRO

.MACRO NOT_BUSY        
    ldi temp, 0x00
    out OUTPORT, temp
    BUSY_PIN_H
.ENDMACRO



.IF USE_DEBUG_PINS
    .MACRO TESTPIN_H   
        sbi PORTB, 5
    .ENDMACRO

    .MACRO TESTPIN_L   
        cbi PORTB, 5
    .ENDMACRO
.ELSE
    .MACRO TESTPIN_H   
    .ENDMACRO

    .MACRO TESTPIN_L   
    .ENDMACRO
.ENDIF



.EQU T1_OCR_1       = 19530     ; 3 x this value and...
.EQU T1_OCR_2       = 19531     ; 1 x this value results in exacly 1 Hz in average at 20 MHz.




; RV8564 control reg 1 bit definitions:
.EQU RV8564_STOP    = 5   ; STOP bit in control reg 1 of RV8564

.EQU RV8564_WEEKDAY_ADDR = 6


; M3002 Control reg bit definitions:
.EQU M3002_RUN      = 0   ; RUN bit in control reg of M3002
.EQU M3002_TEST     = 7   ; TEST bit in control reg of M3002

.EQU M3002_CONTROL_ADDR = 15


; ---------------------------------------------------
; Command definitions

.EQU CMD_START_BL   = 0xBB



; ---------------------------------------------------
; Register definitions
.DEF zero_reg   = R0            // the code relies on this register beeing always zero

.DEF temp       = R18
.DEF temp2      = R19


; Pointer registers
; .DEF XL   = R26
; .DEF XH   = R27

; .DEF ZL   = R30
; .DEF ZH   = R31



; -----------------------------------------------------------------------
; These regs are exclusively reserved for the ISRs, don't use elsewhere!
.DEF bus_data   = R14
.DEF sreg_buf   = R15
.DEF itemp      = R16
.DEF bus_state  = R17

; .DEF YL   = R28
; .DEF YH   = R29
; -----------------------------------------------------------------------


;==========================================================
.ESEG
ee_rtc_data:    .DB 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x01


;==========================================================
.DSEG

.org 0x100
; ---------------------------------------------------------
rtc_data:           ; the 16 bytes of the simulated RTC
;--- Watch ---
rtc_sec:            .BYTE 1
rtc_min:            .BYTE 1
rtc_hour:           .BYTE 1
rtc_day:            .BYTE 1
rtc_month:          .BYTE 1
rtc_year:           .BYTE 1
rtc_weekday:        .BYTE 1
rtc_weekno:         .BYTE 1
;--- Alarm ---
rtc_a_sec:          .BYTE 1     ; EuroPC uses this as config byte
rtc_a_min:          .BYTE 1     ; EuroPC uses this as config byte
rtc_a_hour:         .BYTE 1     ; EuroPC uses this as config byte
rtc_a_data:         .BYTE 1     ; EuroPC uses this as config byte
;--- Timer ---
rtc_t_sec:          .BYTE 1     ; EuroPC uses this as config byte
rtc_t_min:          .BYTE 1     ; Checksum of the above config bytes (calculated by EuroPC), Low-Byte
rtc_t_hour:         .BYTE 1     ; Checksum of the above config bytes (calculated by EuroPC), High-Byte
;--- Status ---
rtc_status:         .BYTE 1     ; Status and Control byte
; Bit defs for status:
.EQU WATCH_RUN = 0


; ---------------------------------------------------------
; a flag for each RTC byte. Bit 0 set if it should be written to EEPROM, bit 1 set if it should be written to RTC chip.
rtc_dirty:
rtc_dirty_sec:      .BYTE 1
rtc_dirty_min:      .BYTE 1
rtc_dirty_hour:     .BYTE 1
rtc_dirty_day:      .BYTE 1
rtc_dirty_month:    .BYTE 1
rtc_dirty_year:     .BYTE 1
rtc_dirty_weekday:  .BYTE 1
rtc_dirty_weekno:   .BYTE 1
                    .BYTE 8



; ---------------------------------------------------------
; other variables (all variables up to 'data_end' will be initialized to zero on start-up):


time_update_pending:    .BYTE 1 ; = 1 if time update was delayed because of a data transfer

t1_cycle:               .BYTE 1 ; timer 1 cycle counter

test_mode:              .BYTE 1 ; = 1 if test mode active


; ---------------------------------------------------------
.org rtc_data + 0x100   ; status_data and _dirty must be exactly 0x100 away from rtc_data and _dirty
test_data:
test_fw_sig_l:          .BYTE 1
test_fw_sig_h:          .BYTE 1
test_fw_ver_l:          .BYTE 1
test_fw_ver_h:          .BYTE 1
test_cmd:               .BYTE 1
test_arg:               .BYTE 1
test_dummy:             .BYTE 9
test_status:            .BYTE 1


test_dirty:             .BYTE 16


data_end:






;==========================================================
; Flash
.CSEG
.org 0
rjmp RESET          ; Reset Handler
rjmp EXT_INT0       ; IRQ0 Handler
rjmp EXT_INT1       ; IRQ1 Handler
rjmp RESET          ; PCINT0 Handler
rjmp RESET          ; PCINT1 Handler
rjmp RESET          ; PCINT2 Handler
rjmp RESET          ; Watchdog Timer Handler
rjmp RESET          ; TIM2_COMPA ; Timer2 Compare A Handler
rjmp RESET          ; TIM2_COMPB ; Timer2 Compare B Handler
rjmp RESET          ; TIM2_OVF   ; Timer2 Overflow Handler
rjmp RESET          ; TIM1_CAPT  ; Timer1 Capture Handler
rjmp RESET          ; TIM1_COMPA ; Timer1 Compare A Handler
rjmp RESET          ; TIM1_COMPB ; Timer1 Compare B Handler
rjmp RESET          ; TIM1_OVF   ; Timer1 Overflow Handler
rjmp RESET          ; TIM0_COMPA ; Timer0 Compare A Handler
rjmp RESET          ; TIM0_COMPB ; Timer0 Compare B Handler
rjmp RESET          ; TIM0_OVF   ; Timer0 Overflow Handler
rjmp RESET          ; SPI Transfer Complete Handler
rjmp RESET          ; USART, RX Complete Handler
rjmp RESET          ; USART, UDR Empty Handler
rjmp RESET          ; USART, TX Complete Handler
rjmp RESET          ; ADC Conversion Complete Handler
rjmp RESET          ; EEPROM Ready Handler
rjmp RESET          ; Analog Comparator Handler
rjmp RESET          ; 2-wire Serial Interface Handler
rjmp RESET          ; Store Program Memory Ready Handler





;==========================================================
; Main program start
;
RESET:     
    ; Set Stack Pointer to top of RAM
    ldi temp, high(RAMEND)
    out SPH, temp 
    ldi temp, low(RAMEND)
    out SPL, temp
 
    rcall init_system       ; Init I/O ports, timers, interrupts, clear RAM...
    rcall load_eeprom       ; Load RTC state from EEPROM
    rcall read_rtc			; Read time and date from RTC. Overrides EEPROM data.

    .IF DEBUG_OUTPUT
    ldi R24, '_'
    rcall putchar
    .ENDIF

    sei


;==========================================================
; The main loop that runs continuously
;
main_loop:
    

    ; -----------------------------------------
    ; if timer 1 compare match A
    in temp, TIFR1
    sbrs temp, OCF1A
    rjmp main0

        ; clear int flag
        sbi TIFR1, OCF1A

        ; inc t1_cycle
        lds temp2, t1_cycle
        inc temp2
        sts t1_cycle, temp2

        ; if (t1_cycle & 0x03) < 3
        ;     OCR1A = T1_OCR_1
        ; else
        ;     OCR1A = T1_OCR_2
        mov temp, temp2
        andi temp, 0x0F
        cpi temp, 3
        brlo main1

        ldi temp, high(T1_OCR_2)
        sts OCR1AH, temp
        ldi temp, low(T1_OCR_2)
        sts OCR1AL, temp
        rjmp main2

    main1:
        ldi temp, high(T1_OCR_1)
        sts OCR1AH, temp
        ldi temp, low(T1_OCR_1)
        sts OCR1AL, temp
    main2:


        rcall timer_1hz_handler 
    main0:


    ; if time_update_pending != 0 and bus_state == 0, increment time
    lds temp, time_update_pending
    cpi temp, 0
    breq main3
        cpi bus_state, 0
        brne main4
            BUSY
            rcall update
            NOT_BUSY
            clr temp
            sts time_update_pending, temp
        main4:
    main3:



    ; -----------------------------------------
    ; Handle test mode

    ; Only handle test_mode if bus_state == 0
    cpi bus_state, 0
    brne main7

    ; -----------------------------------------
	; Bit 7 of status byte switches test mode on and off

    ; if (test_mode == 0)
    lds temp, test_mode
    cpi temp, 0
    brne main6
           
    ; if (test bit in rtc_status is set)
    lds temp, rtc_status
    sbrs temp, M3002_TEST
    rjmp main5
            
    ; switch to test mode
    rcall start_test_mode
    rjmp main5

main6:
    ; test_mode == 1
    ; if test bit in rtc_status is clear
    lds temp, test_status
    sbrc temp, M3002_TEST
    rjmp main5

    ; end test mode
    rcall end_test_mode

main5:

    ; -----------------------------------------
	; Jump to bootloader?
    lds temp, test_mode
    cpi temp, 0
    breq main7

    lds temp, test_cmd
    cpi temp, CMD_START_BL
    brne main7

    lds temp, test_arg
    cpi temp, 0x42
    brne main7

    ; now jump to bootloader
    .IF DEBUG_OUTPUT
    ldi R24, 'B'
    rcall putchar
    ldi R24, 'L'
    rcall putchar
    ldi R24, ' '
    rcall putchar
    .ENDIF

    cli
    rjmp BOOT_RST_ADDR




main7:
    ; -----------------------------------------
    ; Place for doing other things in the main loop
    ; ...




    rjmp main_loop



;==================================================================
; Init I/O ports, timers, interrupts etc
;
init_system:
    ; Init register variables
    clr bus_state
    clr zero_reg

    ldi YH, high(rtc_data)  ; preset YH with address high byte, the IRQ code only changes YL
    
    ; move vector table to beginning of flash
    ldi temp, (1 << IVCE)
    out MCUCR, temp
    out MCUCR, zero_reg
    
    ; Init Ports
    ldi temp, 0x2C
    out DDRB, temp
    ldi temp, 0xFF
    out PORTB, temp

    ldi temp, 0x00
    out DDRC, temp
    ldi temp, 0x3F
    out PORTC, temp

    ldi temp, 0xF0
    out DDRD, temp
    ldi temp, 0xFF
    out PORTD, temp


    ; Init RAM
    ldi XL, low(rtc_data)
    ldi XH, high(rtc_data)
    clr temp
clear_ram:
    st X+, temp
    cpi XL, low(data_end)
    brne clear_ram
    cpi XH, high(data_end)
    brne clear_ram


	; Init variables
	ldi temp, high(FW_SIGNATURE)
	sts test_fw_sig_h, temp
	ldi temp, low(FW_SIGNATURE)
	sts test_fw_sig_l, temp

	ldi temp, high(FW_VERSION)
	sts test_fw_ver_h, temp
	ldi temp, low(FW_VERSION)
	sts test_fw_ver_l, temp


    ; Init External INTs
    ldi temp, (1 << ISC11) | (1 << ISC10) | (1 << ISC01)      ; trigger on falling edge on INT0 (/WR) and rising edge on INT1 (/RD)
    sts EICRA, temp

    ldi temp, (1 << INT1) | (1 << INT0)
    out EIMSK, temp


    ; Init Timer 1 for 1 Hz interval
    ldi temp, (1 << WGM12) | 5 << CS10   ; CTC mode, CLK / 1024
    sts TCCR1B, temp 

    ldi temp, high(T1_OCR_1)
    sts OCR1AH, temp
    ldi temp, low(T1_OCR_1)
    sts OCR1AL, temp

    ; Init UART
    ldi temp, high(10)
    sts UBRR0H, temp
    ldi temp, low(10)
    sts UBRR0L, temp

    ldi temp, (1 << RXEN0) | (1 << TXEN0)
    sts UCSR0B, temp


    ; Init TWI
    ldi temp, 92 * 2
    sts TWBR, temp

    ldi temp, (1 << TWEN)
    sts TWCR, temp

    ; Init RTC control regs
    ldi R24, 0
    ldi R22, 0
    rcall rtc_write

    ldi R24, 1
    ldi R22, 0
    rcall rtc_write

    ;ldi temp, (1 << OCIE1A);
    ;out TIMSK, temp

    ret



;==================================================================
; Handle the 1 Hz timer event
;
timer_1hz_handler:

    ; Test if WATCH_RUN is set. If not, don't update time
    lds temp, rtc_status
    sbrs temp, WATCH_RUN
    rjmp tim1s0
    
    .IF DEBUG_OUTPUT
    ldi R24, '.'
    rcall putchar
    .ENDIF


    ; if bus_state == 0 (idle): increment time
    cpi bus_state, 0
    brne tim1s1
    BUSY
    rcall inc_time
    NOT_BUSY
    rjmp tim1s0

tim1s1:
    ; else (data transfer in progress): if time_update_pending == 0, set time_update_pending = 1
    lds temp, time_update_pending
    cpi temp, 0
    brne tim1s2
    ldi temp, 1
    sts time_update_pending, temp
    rjmp tim1s0

tim1s2:
    ; else (time_update_pending is already 1): so we abort the current data transfer 
    ; and increment the time by two seconds
    clr bus_state

    BUSY
    rcall inc_time
    rcall inc_time
    NOT_BUSY

    clr temp
    sts time_update_pending, temp
   
tim1s0:

    ; do other things at 1 Hz...
    ; ...
    rcall update
    ret


;==================================================================
; Update the time and if necessary write unwritten data to EEPROM and/or RTC Chip
;
update:
    push R6
    push R20
    push R22
    push R24
    push R25
    push XL
    push XH
    push ZL
    push ZH


    ; scan through the RTC data and write the 'dirty' (unwritten) bytes to EEPROM
    ldi ZH, high(rtc_data)              ; pointer to RTC data
    ldi ZL, low(rtc_data);   

    ldi XH, high(ee_rtc_data)           ; pointer to EEPROM
    ldi XL, low(ee_rtc_data)

    ldi R20, 0                          ; loop counter

update_loop:   
    ldd R6, Z+(rtc_dirty - rtc_data)    ; read dirty flag byte
    ldi temp, 3
    and R6, temp                        ; test if bit 0 or 1 is set
    breq update0


    .IF DEBUG_OUTPUT
    ldi R24, '0'
    add R24, R20
    rcall putchar
    .ENDIF
    
    ; dirty flag bit 0 set: write byte to EEPROM, clear dirty flag and return
    sbrs R6, 0
    rjmp update1

    ; write data to EEPROM
    mov R25, XH
    mov R24, XL
    ld R22, Z

    ; on control byte mask only run-bit
    cpi R20, M3002_CONTROL_ADDR
    brne update5
    andi R22, 0x01

update5:
    rcall ee_write
    
update1:     
    ; write to RTC chip if data is time or date and dirty flag bit 1 is set
    sbrs R6, 1
    rjmp update2

    mov R24, R20
    rcall addr_m3002_to_rv8564
    cpi R22, 0
    breq update2

    ld R22, Z                              ; data to write
    
    ; if writing weekday correct for different base (M3001 = 1..7 / RV-8564 = 0..6)
    cpi R24, RV8564_WEEKDAY_ADDR
    brne update4
    dec R22
update4:
    rcall rtc_write
    rjmp update3

update2:
    cpi R20, 0x0F   // status byte?
    brne update3

    ldi R24, 0
    ldi R22, 0
    lds temp, rtc_status
    sbrs temp, M3002_RUN
    ldi R22, (1 << RV8564_STOP)
    rcall rtc_write

update3:
    clr temp
    std Z+(rtc_dirty - rtc_data), temp

    ;rjmp update_end

update0:
    adiw ZL, 1              ; inc RTC_DATA pointer
    adiw XL, 1              ; inc EEPROM pointer

    inc R20                 ; inc loop counter
    cpi R20, 16
    brne update_loop

update_end:
    pop ZH
    pop ZL
    pop XH
    pop XL
    pop R25
    pop R24
    pop R22
    pop R20
    pop R6


    ret


;==================================================================
; Increase time by one second
;
inc_time:
    push R24
    push R22

    ldi R22, 0x60  ; max for sec and min

    ; inc seconds
    lds R24, rtc_sec
    rcall bcd_inc
    sts rtc_sec, R24
    breq inc_time5          ; seconds wrap-around to 0
        rjmp inc_time0
    inc_time5:
        lds temp, rtc_dirty_sec
        ori temp, 1
        sts rtc_dirty_sec, temp

        lds temp, rtc_dirty_min
        ori temp, 1
        sts rtc_dirty_min, temp

        ; inc minutes
        lds R24, rtc_min
        rcall bcd_inc
        sts rtc_min, R24
        breq inc_time7              ; minutes wrap-around
            rjmp inc_time1
        inc_time7:

            lds temp, rtc_dirty_hour
            ori temp, 1
            sts rtc_dirty_hour, temp


            ; inc hours
            ldi R22, 0x24
            lds R24, rtc_hour
            rcall bcd_inc
            sts rtc_hour, R24
            breq inc_time8          ; hours wrap-around
                rjmp inc_time2
            inc_time8:

                lds temp, rtc_dirty_day
                ori temp, 1
                sts rtc_dirty_day, temp

                lds temp, rtc_dirty_weekday
                ori temp, 1
                sts rtc_dirty_weekday, temp
                
                ; inc weekday
                lds R24, rtc_weekday
                ldi R22, 8
                rcall bcd_inc
                sts rtc_weekday, R24
                brne inc_time6      ; weekday wrap-around
                
                    lds temp, rtc_dirty_weekno
                    ori temp, 1
                    sts rtc_dirty_weekno, R24

                    ; inc weekno
                    lds R24, rtc_weekno
                    ldi R22, 0x54
                    rcall bcd_inc
                    sts rtc_weekno, R24
                
                inc_time6:              
                
                ; inc day
                rcall get_days_in_month
                mov R22, R24

                lds R24, rtc_day
                rcall bcd_inc
                sts rtc_day, R24
                brne inc_time3          ; day wrap-around

                    ldi R24, 1          ; day = 1
                    sts rtc_day, R24

                    lds temp, rtc_dirty_month
                    ori temp, 1
                    sts rtc_dirty_month, temp


                    ; inc month
                    ldi R22, 0x13
                    lds R24, rtc_month
                    rcall bcd_inc
                    sts rtc_month, R24
                    brne inc_time4          ; month wrap-around

                        lds temp, rtc_dirty_year
                        ori temp, 1
                        sts rtc_dirty_year, temp

                        ldi R24, 1             ; month = 1
                        sts rtc_month, R24

                        ; inc year
                        ldi R22, 0xA0
                        lds R24, rtc_year
                        rcall bcd_inc
                        sts rtc_year, R24

                    inc_time4:
                inc_time3:
            inc_time2:
        inc_time1:
    inc_time0:

    pop R22
    pop R24
    ret


;==================================================================
; load EEPROM contents to rtc_data
load_eeprom:
    ldi ZH, high(rtc_data)          ; pointer to RTC data
    ldi ZL, low(rtc_data);   

    ldi R25, high(ee_rtc_data)      ; pointer to EEPROM
    ldi R24, low(ee_rtc_data)

    ldi R20, 0                      ; loop counter

load_loop:
    push R24
    rcall ee_read
    
    cpi R20, 8
    brsh load1
    rcall bcd_correct               ; fix bytes 0..7 if bcd format is invalid
load1:
    st Z+, R24
    pop R24
        
    adiw R24, 1                     ; inc EEPROM pointer

    inc R20
    cpi R20, 16
    brne load_loop

    ret


;==================================================================
; Load time and date from RTC to rtc_data
read_rtc:
    push R2
    push R20

    ldi R20, 0          ; loop counter

rtc1:
    mov R24, R20
    rcall addr_m3002_to_rv8564      ; R24 = rv8564 addr
    mov R2, R24                     ; remember rv8564 addr
    cpi R22, 0
    breq rtc2                       ; not time/date? goto rtc2

    rcall rtc_read                  ; R24 = data, R22 = bit-mask
    and R24, R22

    ldi temp, RV8564_WEEKDAY_ADDR
    cp R2, temp
    brne rtc3
    ldi R22, 8
    rcall bcd_inc 

rtc3:
    ; store data R24 to rtc_data[R20]    
    ldi XL, low(rtc_data)
    ldi XH, high(rtc_data)
    add XL, R20
    adc XH, zero_reg
    st X, R24


    ; Debug output    
    .IF DEBUG_OUTPUT
    push R24
    andi R24, 0xF0
    swap R24
    subi R24, -'0'
    rcall putchar
    pop R24
    andi R24, 0x0F
    subi R24, -'0'
    rcall putchar
    .ENDIF

rtc2:
    inc R20
    cpi R20, 16
    brlo rtc1

    
    ; read status byte and copy STOP bit from RTC to RUN bit (negated)
    ldi R24, 0
    rcall rtc_read
    
    lds temp, rtc_status
    andi temp, ~(1 << M3002_RUN)
    sbrs R24, RV8564_STOP
    ori temp, (1 << M3002_RUN)
    sts rtc_status, temp    
    
    pop R20
    pop R2
    ret



;==================================================================
; Inc BCD in R24 by one and compare result with R22
; in/out: R24
; in:     R22
; Zero flag set if wrap around from (R22 - 1) to 0
bcd_inc:
    rcall bcd_correct

    subi R24, -7
    brhc bcd_inc0
    subi R24, 6
bcd_inc0:
    cp R24, R22
    brlo bcd_inc1
    clr R24
bcd_inc1:
    ret



;==================================================================
; Check BCD in R24, if it is no valid BCD number, set it to zero
; in/out: R24
bcd_correct:
    mov temp, R24
    andi temp, 0x0F
    cpi temp, 0x0A
    brsh bcd_corr_err

    mov temp, R24
    andi temp, 0xF0
    cpi temp, 0xA0
    brsh bcd_corr_err

bcd_corr_end:
    ret

bcd_corr_err:
    ldi R24, 0
    rjmp bcd_corr_end
    

;==================================================================
; return: R24 = days+1 in current month in BCD format
;
get_days_in_month: 
    push ZL
    push ZH
     
    lds R24, RTC_MONTH
    rcall bcd_to_byte
    cpi R24, 2
    breq get_days_feb
    ldi ZH, high(days_in_month * 2)
    ldi ZL, low(days_in_month * 2)
    dec R24
    add ZL, R24
    adc ZH, zero_reg
    lpm temp, Z
    rjmp get_days_end

days_in_month:  .db 0x32, 0x29, 0x32, 0x31, 0x32, 0x31, 0x32, 0x32, 0x31, 0x32, 0x31, 0x32

get_days_feb:
    ldi temp, 0x29
    lds R24, RTC_YEAR
    rcall bcd_to_byte
    andi R24, 0x03
    brne get_days_end
    ldi temp, 0x30

get_days_end:
    mov R24, temp

    pop ZH
    pop ZL
    ret



;==================================================================
; in/out: R24
bcd_to_byte:
    push R0
    push R1

    mov temp, R24
    swap temp
    andi temp, 0x0F
    andi R24, 0x0F
    
    ldi temp2, 10
    mul temp, temp2

    mov temp, R0
    add temp, R24
    mov R24, temp

    pop R1
    pop R0
    ret



;==================================================================
; Convert address from M3002 to RV-8564
; in:  R24 = M3002 address
; out: R24 = RV-8564 address
;      R22 = Bit mask of valid bits in RV-8564, zero if address is not for time or date
;      
;
;M3002   RV-8564   Bit-mask
; 0  ->   2         0x7F        sec
; 1  ->   3         0x7F        min
; 2  ->   4         0x3F        hour
; 3  ->   5         0x3F        day
; 4  ->   7         0x1F        month
; 5  ->   8         0xFF        year
; 6  ->   6         0x07        weekday (M3001 = 1..7 / RV-8564 = 0..6)
;
addr_m3002_to_rv8564:
    push ZH
    push ZL

    cpi R24, 7
    brlo addr1
    ldi R22, 0
    rjmp addr_end

rv8564_mask:
    .db 0x7F, 0x7F, 0x3F, 0x3F, 0x1F, 0xFF, 0x07, 0x00

addr1:
    ldi ZL, low(rv8564_mask * 2)
    ldi ZH, high(rv8564_mask * 2)
    add ZL, R24
    adc ZH, zero_reg
    lpm R22, Z

    cpi R24, 6
    breq addr_end

    cpi R24, 4
    brlo addr2
    subi R24, -3
    rjmp addr_end

addr2:
    subi R24, -2  
    
addr_end:
    pop ZL
    pop ZH    
    ret


;==================================================================
; IN: R24 character to be printed
putchar:
    lds temp, UCSR0A
    sbrs temp, UDRE0
    rjmp putchar

    sts UDR0, R24
    ret



;==================================================================
start_test_mode:

    ldi YH, high(test_data)  ; this lets Y point to test_data instead of rtc_data

    lds temp, rtc_status
    sts test_status, temp

    ldi temp, 1
    sts test_mode, temp
    ret


;==================================================================
end_test_mode:

    ldi YH, high(rtc_data)  ; change Y back to rtc_data

    lds temp, test_status
    sts rtc_status, temp

    ldi temp, 0
    sts test_mode, temp
    ret


;==================================================================
; Write ISR. Gets called, when /CS and RD-/WR are getting pulled low, 
; i.e. at the start of a write cycle to read the address or data.
;
EXT_INT0:
    in sreg_buf, SREG

    ; Read address or data from I/O bus
    in itemp, INPORT


    TESTPIN_L

    ; -----------------------------------------
    ; State 0 (address)
    cpi bus_state, 0
    brne wr_st1

    mov YL, itemp
    andi YL, 0x0F
.if low(rtc_data) <= 63
    ldd bus_data, Y+low(rtc_data)       ; read data from rtc_data[address], in case a read cycle follows
.else
    subi YL, -low(rtc_data)
    ld bus_data, Y
.endif

    ; output the high nibble, in case a read cycle follows
    out OUTPORT, bus_data

    inc bus_state
    rjmp wr_end


wr_st1:
    ; -----------------------------------------
    ; State 1 (high nibble)
    cpi bus_state, 1
    brne wr_st2

    andi itemp, 0x0F
    swap itemp
    mov bus_data, itemp

    inc bus_state
    rjmp wr_end


wr_st2:
    ; -----------------------------------------
    ; State 2 (low nibble)
    cpi bus_state, 2
    brne wr_end

    andi itemp, 0x0F
    or bus_data, itemp

    st Y, bus_data                              ; write data to rtc_data[address]
    ldi itemp, 3
    std Y+(rtc_dirty - rtc_data), itemp         ; set bits 0 and 1 in the corresponding dirty byte


    NOT_BUSY
    clr bus_state

wr_end:
    TESTPIN_H
    out SREG, sreg_buf
    reti



;==================================================================
; Read ISR. Gets called at the end of a read cycle to prepare the next cycle.
;
EXT_INT1:
    in sreg_buf, SREG
    TESTPIN_L

    ; -----------------------------------------
    ; State 0 (busy state was just read) 
    cpi bus_state, 0
    brne rd_st1
    
    ;out OUTPORT, busy
    rjmp rd_end
   
rd_st1:
    ; -----------------------------------------
    ; State 1 (high nibble was just read) 
    cpi bus_state, 1
    brne rd_st2

    swap bus_data           ; prepare low nibble
    out OUTPORT, bus_data
    inc bus_state
    rjmp rd_end  

rd_st2:
    ; -----------------------------------------
    ; State 2  (low nibble was just read)
    cpi bus_state, 2
    brne rd_end
    
    NOT_BUSY
    clr bus_state


rd_end:
    TESTPIN_H
    out SREG, sreg_buf
    reti









;==================================================================
; Include other source files
;
.include "eeprom.asm"
.include "twi.asm"
.include "rtc.asm"






