<?xml version="1.0" encoding="UTF-8"?>
<xsl:stylesheet version="1.0"
				xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
				xmlns:fn="http://www.w3.org/2005/02/xpath-functions">

    <!--
	    Enclosure Front View blade event listener policy

	    1. A bladeStatus event in which presence is not affected:
		    Update the individual blade, but do not redraw the entire enclosure.

	    2. A bladeInserted event in which bladeInfo or bladeStatus is updated
		    and presence is changed to a value of PRESENT:
		    Remove all event listeners from the enclosure bays.
		    Redraw the entire enclosure.

	    3. A bladeRemoved event in which bladeInfo or bladeStatus is updated
		    and presence is changed to a value of ABSENT:
		    Remove all event listeners from the enclosure bays.
		    Redraw the entire enclosure.

	    Listeners will not be added to bays with a presence value of SUBSUMED.
	    This assumes that blade listeners will be added to blades at the time
	    the enclosure is drawn.
    -->

    <!-- Include the individual bay template -->


    <!--
	    Front view template containing the enclosure bays and power supply bays.
	    Blade and storage bays call the bay template (bay.xsl) to render each bay.
	    Power supply bays call the powerSupply template (powerSupply.xsl) to render
	    power supply information. Template matches all nodes because actual XML
	    information is taken from XML document fragment parameters declared above.
    -->
    <xsl:template name="cClassEnclFrontView">
       <xsl:param name="enclSerialNum"/> 
       <xsl:param name="typeOfDisplay"/> 

					<xsl:variable name="heightOfEncl">
						<xsl:choose>
							<xsl:when test="./@type=$E_CLASS">
								 <xsl:value-of select="$E_ENCL_HEIGHT"/>
						 </xsl:when>
						 <xsl:otherwise>
								 <xsl:value-of select="$PC_ENCL_HEIGHT"/>
						 </xsl:otherwise>
						</xsl:choose>
					</xsl:variable>

					<xsl:variable name="widthOfEncl">
						<xsl:choose>
							<xsl:when test="./@type=$E_CLASS">
								 <xsl:value-of select="$E_ENCL_WIDTH"/>
						 </xsl:when>
							<xsl:when test="./@type=$C_CLASS">
								 <xsl:value-of select="$C_ENCL_WIDTH"/>
						 </xsl:when>
						 <xsl:otherwise>
								 <xsl:value-of select="$P_ENCL_WIDTH"/>
						 </xsl:otherwise>
						</xsl:choose>
					</xsl:variable>
		
					<!-- Front view table container -->
					<table border="0" cellspacing="0" cellpadding="0" class="systemStatusTable" id="frontView" >
						<xsl:attribute name="width">
								<xsl:value-of select="$widthOfEncl"/>
						</xsl:attribute>
						<xsl:attribute name="height">
								<xsl:value-of select="$heightOfEncl"/>
						</xsl:attribute>
						<xsl:attribute name="powerZone">
								<xsl:value-of select="./Common/PowerZone"/>
						</xsl:attribute>
						<tr>
							<td valign="top"> 
								<table border="0" cellspacing="0" cellpadding="0" class="systemStatusTable">
									<xsl:call-template name="renderServers">
										<xsl:with-param name="typeOfDisplay" select="$typeOfDisplay"/>
									</xsl:call-template>
									<!-- Only display the power supplies if we are not in the wizard -->
									<!-- Currently commenting out as i dont know wat the purpose of this wizard <xsl:if test="$typeOfDisplay = 'false'"> -->
									<xsl:if test="./@type=$C_CLASS">
										<xsl:call-template name="renderPowerUnits">
											<xsl:with-param name="enclSerialNum" select="$enclSerialNum"/>
										</xsl:call-template>	
									</xsl:if>
								</table>	
							</td>
						</tr>
					</table>
		
	</xsl:template>

    <!--
		Bay table cell template Sets up the bay cell for the device by setting the table cell attributes and adding the renderBlade or renderSwitch 
		template to the cell.
	-->
    <xsl:template name="constructBayCell">

        <xsl:param name="bayNumber" />
        <xsl:param name="typeOfDisplay" />

        <xsl:variable name="presence">
			<xsl:value-of select="./hasDevice"/>
        </xsl:variable> 

		<xsl:variable name="bayType">
			<xsl:value-of select="./@bayType"/>
		</xsl:variable>

		
            <!-- Create an <td> for the bay -->
            <td>
				<xsl:if test="../@type=$C_CLASS">
					<xsl:attribute name="style">
						 <xsl:text>border: 1px solid #333;</xsl:text>
					</xsl:attribute>
				</xsl:if>
				<!-- Set the class that has to applied to this <td>
					If a device is present @ this slot then set the class as deviceCell
					else set it as deviceCellUnselectable
				-->
				
                <xsl:choose>
                    <xsl:when test="$presence=$PRESENT and $typeOfDisplay=$BPV">
                        <xsl:attribute name="class">deviceCell</xsl:attribute>
                    </xsl:when>
                    <xsl:otherwise>
                        <xsl:attribute name="class">deviceCellUnselectable</xsl:attribute>
                    </xsl:otherwise>
                </xsl:choose>

				<!-- Setting the id for this <td> 
					Format: <BayType>_Bay_<SlotNumber>
					For example for a server @ slot 2: Server_Bay_2
				-->
                <xsl:attribute name="id">
                    <xsl:value-of select="concat($bayType, '_','Bay','_', $bayNumber)" />
                </xsl:attribute>

                <!--
					    Determine the row span value for the blade table cell.
					    Compares the blade height to the standard blade half
					    height value.
				-->
                <xsl:variable name="rowSpan">
                    <xsl:value-of select="./SlotsUsed/YDirection" />
                </xsl:variable>

            	<!--
				    Determine the column span value for the blade table cell.
				    Compares the blade width to the standard blade width value.
			    -->
                <xsl:variable name="colSpan">
                    <xsl:value-of select="./SlotsUsed/XDirection" />
                </xsl:variable>

			
				<!-- Set the title /tooltip/flyover that has to be displayed when the mouse pointer hovers over the device 
					 If device is present 
							Format: <Localised Bay> <Start_Slot>-<End_Slot> , <Name of the device> , <Product Model of the device >
					else 
							<Localised Empty Bay>
				-->
<!--
            <xsl:attribute name="onmouseover">try{loadDeviceInfoTip(event,this,160);}catch(e){}</xsl:attribute>
            <xsl:attribute name="onmousemove">try{removeDeviceInfoTip(event,this,false);}catch(e){}</xsl:attribute>
            <xsl:attribute name="onmousedown">try{removeDeviceInfoTip(event,this,true);}catch(e){}</xsl:attribute>
            <xsl:attribute name="onmouseout">try{removeDeviceInfoTip(event,this,true);}catch(e){}</xsl:attribute>
-->
                <xsl:attribute name="title">
                    <xsl:choose>
                        <xsl:when test="$presence=$PRESENT">
							<xsl:value-of select="//I18N/L10NBay"/>:<xsl:text> </xsl:text><xsl:call-template name="computeTitleBay"><xsl:with-param name="slotNumber" select="./SlotNumber"/><xsl:with-param name="slotsXDirect" select="$colSpan"/></xsl:call-template><xsl:text>, </xsl:text><xsl:value-of select="//I18N/L10NProductName"/>:<xsl:text> </xsl:text><xsl:value-of select="./Common/Name"/><xsl:text>, </xsl:text><xsl:value-of select="//I18N/L10NProductModel"/>:<xsl:text> </xsl:text><xsl:value-of select="./Common/Product"/>
                        </xsl:when>
                        <xsl:otherwise>
                            <xsl:value-of select="//I18N/L10NEmptyBay"/>
                        </xsl:otherwise>
                    </xsl:choose>
                </xsl:attribute>
		

                <!--
					    Sets the row span attribute of the table cell from
					    the calculated span value.
				    -->
                <xsl:attribute name="rowspan">
                    <xsl:value-of select="$rowSpan" />
                </xsl:attribute>

                <!--
					    Sets the column span attribute of the table cell from
					    the calculated span value.
				    -->
                <xsl:attribute name="colspan">
                    <xsl:value-of select="$colSpan" />
                </xsl:attribute>

				<!-- Set the height attribute of this cell -->	
                <xsl:attribute name="height">
						<xsl:choose>
							<xsl:when test="../@type=$E_CLASS"><xsl:value-of select="$E_HALF_BAY_HEIGHT * $rowSpan" /></xsl:when>
							<xsl:otherwise><xsl:value-of select="$HALF_BAY_HEIGHT * $rowSpan" /></xsl:otherwise>
						</xsl:choose>
                </xsl:attribute>

				<!-- Set the width attribute of this cell 
					  HALF_BAY_WIDTH * number of slots occupied in X-Direction by the server	
				-->	
				<xsl:choose>
					<xsl:when test="../@type=$C_CLASS">
						<xsl:attribute name="width">
									<xsl:value-of select="$C_HALF_BAY_WIDTH * $colSpan" />
						</xsl:attribute>
					</xsl:when>
					<xsl:otherwise>
                <xsl:attribute name="width">
									<xsl:value-of select="$PE_HALF_BAY_WIDTH * $colSpan" />
                </xsl:attribute>
					</xsl:otherwise>
				</xsl:choose>
				
				<xsl:choose>
					<xsl:when test="$typeOfDisplay=$BPV and $presence=$PRESENT">
						<!-- drilldownURL attribute will point to URL when the user drill downs this device -->
						<xsl:attribute name="drilldownURL">
							<xsl:value-of select="./Common/DrillDownURL" />
						</xsl:attribute>
					</xsl:when>
			    </xsl:choose>

				<!-- This places the checkboxes and status icon inverted in case the bay number is > 8 if set to bottom-->
                <xsl:attribute name="valign">
					    <xsl:choose>
						    <xsl:when test="number($bayNumber) &lt;= number($BLADES_PER_ROW)">top</xsl:when>
						    <xsl:otherwise>top</xsl:otherwise>
					    </xsl:choose>
                </xsl:attribute>

                <!--
					    Determine background attribute of table cell based on
					    the width and height of the blade in the bay.
				    -->
		
                <xsl:attribute name="background">
                    <xsl:choose>
                        <xsl:when test="$presence=$PRESENT">
									<xsl:if test="$bayType=$SERVER">
										<xsl:choose>
											<xsl:when test="../@type=$P_CLASS">
												<xsl:value-of select="$P_SERVER_FRNT"/><xsl:text>server_</xsl:text><xsl:value-of select="$colSpan"/><xsl:text>X_</xsl:text><xsl:value-of select="$rowSpan"/><xsl:text>Y.gif</xsl:text>
											</xsl:when>
											<xsl:when test="../@type=$C_CLASS">
												<xsl:value-of select="$C_SERVER_FRNT"/><xsl:text>server_</xsl:text><xsl:value-of select="$colSpan"/><xsl:text>X_</xsl:text><xsl:value-of select="$rowSpan"/><xsl:text>Y.gif</xsl:text>
											</xsl:when>
											<xsl:when test="../@type=$E_CLASS">
												<xsl:value-of select="$E_SERVER_FRNT"/><xsl:text>server_</xsl:text><xsl:value-of select="$colSpan"/><xsl:text>X_</xsl:text><xsl:value-of select="$rowSpan"/><xsl:text>Y.gif</xsl:text>
											</xsl:when>
										</xsl:choose>
									</xsl:if>
									<xsl:if test="$bayType=$SWITCH">
										<xsl:variable name="modelName" select="./Common/Product"/>
										<xsl:choose>
											<xsl:when test="contains($modelName,$GBE_MODEL_PATTERN)">
												<xsl:value-of select="$P_SWTCH_FRNT"/><xsl:text>gbe2_switch.gif</xsl:text>
											</xsl:when>
											<xsl:when test="contains($modelName,$CISCO_MODEL_PATTERN)">
												<xsl:value-of select="$P_SWTCH_FRNT"/><xsl:text>cisco_switch.gif</xsl:text>
											</xsl:when>
										</xsl:choose>
									</xsl:if>
                        </xsl:when>
                        <xsl:otherwise>
							<xsl:if test="$bayType=$SERVER">
								<xsl:choose>
									<xsl:when test="../@type=$P_CLASS">
										<xsl:value-of select="$P_SERVER_FRNT"/><xsl:value-of select="$BLADE_ABSENT_IMAGE" />
									</xsl:when>
									<xsl:when test="../@type=$C_CLASS">
										<xsl:value-of select="$C_SERVER_FRNT"/><xsl:value-of select="$BLADE_ABSENT_IMAGE" />
									</xsl:when>
									<xsl:when test="../@type=$E_CLASS">
										<xsl:value-of select="$E_SERVER_FRNT"/><xsl:value-of select="$BLADE_ABSENT_IMAGE" />
									</xsl:when>
								</xsl:choose>
							</xsl:if>
							<xsl:if test="$bayType=$SWITCH">
									<xsl:if test="../@type=$P_CLASS">
										<xsl:value-of select="$P_SWTCH_FRNT"/><xsl:value-of select="$SWITCH_ABSENT_IMAGE" />
									</xsl:if>
							</xsl:if>
                        </xsl:otherwise>
                    </xsl:choose>
                </xsl:attribute>
                
                <!--
                    Add blade cell template information to the table cell.
                    TODO: Give this element a standardized programattic id
                    so that we can swap out its contents when an event is received.
                -->
                <xsl:element name="DIV">
                    <xsl:attribute name="style">text-align:center; display:block;</xsl:attribute>
							<xsl:call-template name="renderStatusAndCheckBox">
									<xsl:with-param name="presence" select="$presence" />
									<xsl:with-param name="status" select="./Status/UID" />
									<xsl:with-param name="typeOfDisplay" select="$typeOfDisplay" />
									<xsl:with-param name="amStatus" select="./AutoManager/Status" />
									<xsl:with-param name="devHomeURL" select="./Common/HomePageURL" />
									<xsl:with-param name="amURL" select="./AutoManager/StatusURL" />
									<xsl:with-param name="drillDownURL" select="./Common/DrillDownURL"/>
									<xsl:with-param name="hwtitleStatus" select="./Status/Title"/>
									<xsl:with-param name="amTitleStatus" select="./AutoManager/StatusTitle"/>
							</xsl:call-template>		
                </xsl:element>
                
            </td>

    </xsl:template>
	




	
	<xsl:template name="renderServers">
	<xsl:param name="typeOfDisplay"/>
			<tr>
				<td >
					<xsl:attribute name="width">
						<xsl:choose>
							<xsl:when test="./@type=$E_CLASS">
								 <xsl:value-of select="$E_ENCL_WIDTH"/>
						 </xsl:when>
							<xsl:when test="./@type=$C_CLASS">
								 <xsl:value-of select="$C_ENCL_WIDTH"/>
						 </xsl:when>
						 <xsl:otherwise>
								 <xsl:value-of select="$P_ENCL_WIDTH"/>
						 </xsl:otherwise>
						</xsl:choose>
					</xsl:attribute>
				
					<!-- Bay table container. Holds top and bottom rows of enclosure bays. -->
					<table   border="0" cellspacing="0" cellpadding="0" align="center">	
						<xsl:choose>
							<xsl:when test="./@type=$E_CLASS">
								<tr id="bladesTopRow">
									<!--
										Select bladeInfo elements for the top row
									-->
									<xsl:for-each select="./Device[@bayType='Server']">
										<xsl:call-template name="constructBayCell">
											<xsl:with-param name="bayNumber" select="./SlotNumber" />
											<xsl:with-param name="typeOfDisplay" select="$typeOfDisplay" />
										</xsl:call-template>
									</xsl:for-each>
								</tr>	
							</xsl:when>
							<xsl:otherwise>
								<!-- Render top row bays -->
								<tr id="bladesTopRow">
									<!--
										Select bladeInfo elements for the top row
									-->
									<xsl:for-each select="./Device[@bayType='Switch' and @location='Front']">
										<xsl:if test="./SlotNumber=0">
										<xsl:call-template name="constructBayCell">
											<xsl:with-param name="bayNumber" select="./SlotNumber" />
											<xsl:with-param name="typeOfDisplay" select="$typeOfDisplay" />
										</xsl:call-template>
										</xsl:if>
									</xsl:for-each>
									<xsl:for-each select="./Device[@bayType='Server']">
										<xsl:if test="./SlotNumber&lt;=8">
										<xsl:call-template name="constructBayCell">
											<xsl:with-param name="bayNumber" select="./SlotNumber" />
											<xsl:with-param name="typeOfDisplay" select="$typeOfDisplay" />
										</xsl:call-template>
										</xsl:if>
									</xsl:for-each>
		
									<xsl:for-each select="./Device[@bayType='Switch' and @location='Front']">
										<xsl:if test="./SlotNumber=9999">
										<xsl:call-template name="constructBayCell">
											<xsl:with-param name="bayNumber" select="./SlotNumber" />
											<xsl:with-param name="typeOfDisplay" select="$typeOfDisplay" />
										</xsl:call-template>
										</xsl:if>
									</xsl:for-each>
								</tr>
								<!-- Render bottom row bays -->
								<tr id="bladesBottomRow">
									<xsl:for-each select="./Device[@bayType='Server']">
										<xsl:if test="./SlotNumber&gt;8">
										<xsl:call-template name="constructBayCell">
											<xsl:with-param name="bayNumber" select="./SlotNumber" />
											<xsl:with-param name="typeOfDisplay" select="$typeOfDisplay" />
										</xsl:call-template>
										</xsl:if>
									</xsl:for-each>
								</tr>
							</xsl:otherwise>						
						</xsl:choose>	
					</table>
				</td>
			</tr>
	</xsl:template>
	

	
	<xsl:template name="renderPowerUnits">
		<xsl:param name="enclSerialNum"/>
				<tr>
					<td>
					<xsl:attribute name="width">
						<xsl:choose>
							<xsl:when test="./@type=$C_CLASS">
								 <xsl:value-of select="$C_ENCL_WIDTH"/>
							 </xsl:when>
							 <xsl:otherwise>
								 <xsl:value-of select="$P_ENCL_WIDTH"/>
							 </xsl:otherwise>
						</xsl:choose>
					</xsl:attribute>
					<xsl:attribute name="height">
						<xsl:value-of select="$ONEU_HEIGHT"/>
					</xsl:attribute>
						<!-- Front view power supply container -->
						<table border="0" cellspacing="0" cellpadding="0" align="center">
							<xsl:attribute name="width">
								<xsl:choose>
									<xsl:when test="./@type=$C_CLASS">
										 <xsl:value-of select="$C_ENCL_WIDTH"/>
								 </xsl:when>
								 <xsl:otherwise>
										 <xsl:value-of select="$P_ENCL_WIDTH"/>
								 </xsl:otherwise>
								</xsl:choose>

							</xsl:attribute>
							<tr>
								<!--
									Loop through all of the value elements in the power supply status document
									(one value element for each power supply).
								-->
								<xsl:for-each select="./PowerSupplyUnit">
									<xsl:sort data-type="number" select="BayNumber"/>
									<!-- Get the power supply number for the current power supply. -->
                                    <xsl:variable name="psNumber">
										<xsl:value-of select="./BayNumber"/>
                                    </xsl:variable>
									
									<!-- Retrieve the presence value -->
                                    <xsl:variable name="presence">
										<xsl:value-of select="./hasDevice"/>
                                    </xsl:variable>
									

									<!-- Retrieve the status value -->
                                    <xsl:variable name="status" select="Status/UID"/>
									<!--
										Draw a table cell container for each power supply.  NOTE: These table
										cells will only line up with the above blade bays if there are six power
										supplies. It is currently assumed that there are six power supplies.								
									-->
									<xsl:element name="td">
										<!-- Set up the table cell's attributes -->
										<xsl:attribute name="width">35</xsl:attribute>
										<xsl:attribute name="height"><xsl:value-of select="$ONEU_HEIGHT"/></xsl:attribute>
										<xsl:attribute name="align">center</xsl:attribute>
										<xsl:attribute name="valign">middle</xsl:attribute>
                                        <xsl:attribute name="style">border: 1px solid #333;</xsl:attribute>

                                        <!-- As we cannot drilldown into an powerSupply we will set the class to deviceCellUnselectable -->
                                        <!--<xsl:choose>
                                            <xsl:when test="$presence=$PRESENT">
                                                <xsl:attribute name="class">deviceCell</xsl:attribute>
                                            </xsl:when>
                                            <xsl:otherwise>
                                                <xsl:attribute name="class">deviceCellUnselectable</xsl:attribute>
                                            </xsl:otherwise>
                                        </xsl:choose>
										-->
										<xsl:attribute name="class">deviceCellUnselectable</xsl:attribute>
										<!-- Flyover details to be show over the powerUnit-->
										<xsl:attribute name="title">
											<xsl:choose>
												<xsl:when test="$presence=$PRESENT">
													<xsl:value-of select="//I18N/L10NPOWERSUPPLY"/><xsl:text> </xsl:text><xsl:value-of select="$psNumber"/>
												</xsl:when>
												<xsl:otherwise>
													<xsl:value-of select="//I18N/L10NEmptyBay"/>
												</xsl:otherwise>
											</xsl:choose>
										</xsl:attribute>
										<!--
											Set the table cell's background image based on the presence
											of the power supply in the current bay.
										-->
										<xsl:attribute name="background">
											
											<!-- Set the background image location based on presence -->
											<xsl:choose>
												<xsl:when test="$presence=$PRESENT">
														<xsl:value-of select="$PS_PRESENT_IMAGE" />
												</xsl:when>
												<xsl:otherwise>
													<xsl:value-of select="$PS_ABSENT_IMAGE" />
												</xsl:otherwise>
											</xsl:choose>
										</xsl:attribute>
										
										<!-- TODO: Get correct enclosure number -->
										<xsl:attribute name="id">
											<xsl:value-of select="concat('enc', $enclSerialNum, 'ps', $psNumber, 'Select')" />
										</xsl:attribute>
										
										<xsl:attribute name="devNum"><xsl:value-of select="$psNumber" /></xsl:attribute>
										<xsl:attribute name="enclSerialNum"><xsl:value-of select="$enclSerialNum" /></xsl:attribute>
										<xsl:attribute name="devType">ps</xsl:attribute>
										
										<!-- Power supply information container. Holds the power supply's status icon. -->
										<xsl:element name="div">
											<xsl:attribute name="style">display:block; width:100%;</xsl:attribute>
											<!--
												Set the ID of the power supply information container. The ID can be used to
												retrieve the container and update the innerHTML when a power supply event
												occurs.
											-->
											<xsl:attribute name="id">
												<xsl:value-of select="concat('psContainer', $psNumber)" />
											</xsl:attribute>
											
											<!-- Only render the status icon if the power supply is present. -->
                                            <xsl:choose>
                                                <xsl:when test="$presence=$PRESENT">
                                                    <!--
												        Call the power supply template to set up the current power
												        supply's status icon. Pass as a parameter the power supply's
												        current status.
											        -->
													<xsl:call-template name="statusIcon">
														<xsl:with-param name="statusCode" select="$status" />
														<xsl:with-param name="titleMsg" select="Status/Title" />
													</xsl:call-template>
                                                </xsl:when>
                                                <xsl:otherwise>
                                                    &#160;
                                                </xsl:otherwise>
                                            </xsl:choose>

										</xsl:element>
									</xsl:element>
								</xsl:for-each>
							</tr>
						</table>
					</td>
				</tr>
	</xsl:template>
	
	
	<!-- Main rear enclosure view template -->
	<xsl:template name="OvrlrdEnclRearView">

    <xsl:param name="enclSerialNum" />
    <xsl:param name="typeOfDisplay"/>

    <!-- Table container for rear enclosure view. Contains fans, patch panels, and enclosure managers. -->
    <table border="0" cellpadding="0" cellspacing="0" class="systemStatusTable" id="rearView">
    <!-- Used to display the power zone border around the servere enclosure-->
		<xsl:attribute name="powerZone">
				<xsl:value-of select="./Common/PowerZone"/>
		</xsl:attribute>
    
    <tr>
				<td>
					<table border="0" cellspacing="0" cellpadding="0" class="systemStatusTable" id="frontView">
						<!-- Top row of fans -->
						<tr>
							<td  height="26">
							<xsl:attribute name="width">
								<xsl:choose>
									<xsl:when test="./@type=$C_CLASS">
										 <xsl:value-of select="$C_ENCL_WIDTH"/>
									 </xsl:when>
									 <xsl:otherwise>
										 <xsl:value-of select="$P_ENCL_WIDTH"/>
									 </xsl:otherwise>
								</xsl:choose>
							</xsl:attribute>
								<!-- Table container for rear view top fans -->
								<table height="26" border="0" cellspacing="0" cellpadding="0">
								<xsl:attribute name="width">
									<xsl:choose>
										<xsl:when test="./@type=$C_CLASS">
											 <xsl:value-of select="$C_ENCL_WIDTH"/>
										 </xsl:when>
										 <xsl:otherwise>
											 <xsl:value-of select="$P_ENCL_WIDTH"/>
										 </xsl:otherwise>
									</xsl:choose>
								</xsl:attribute>

									<!-- Top row of fans (1-5) -->
									<tr id="fanTopRow">
										
										<!--
											Loop through all of the fan objects and render the top
											row if the fan number is less than or equal to the number
											of fans per row in the enclosure.
										-->
										<xsl:for-each select="./FANS">
											<xsl:sort data-type="number" select="BayNumber"/>
											<xsl:variable name="bayNumber" select="./BayNumber">
											
											</xsl:variable>	
											<xsl:if test="$bayNumber&lt;=5">
												<xsl:call-template name="fanCell">
													<xsl:with-param name="fanNumber" select="$bayNumber" />
													<xsl:with-param name="enclSerialNum" select="$enclSerialNum" />
												</xsl:call-template>
											</xsl:if>
										</xsl:for-each>
									</tr>
								</table>
							</td>
						</tr>
							
						<!-- Interconnect bays -->
						<tr>
							<td height="78">
							<xsl:attribute name="width">
								<xsl:choose>
									<xsl:when test="./@type=$C_CLASS">
										 <xsl:value-of select="$C_ENCL_WIDTH"/>
									 </xsl:when>
									 <xsl:otherwise>
										 <xsl:value-of select="$P_ENCL_WIDTH"/>
									 </xsl:otherwise>
								</xsl:choose>
							</xsl:attribute>

								<!-- Interconnect bay table container -->
								<table  height="78" border="0" cellspacing="0" cellpadding="0">
								<xsl:attribute name="width">
										 <xsl:value-of select="$C_ENCL_WIDTH"/>
								</xsl:attribute>
	
									<xsl:for-each select="./Device[@bayType='Switch']">
										<xsl:if test="./SlotNumber mod 2 != 0 ">
											<xsl:element name="tr">
												<xsl:call-template name="interConnectBayCell">
													<xsl:with-param name="trayNumber" select="./SlotNumber" /> <!-- Change it to trayNumber -->
													<xsl:with-param name="enclSerialNum" select="$enclSerialNum" />
													<xsl:with-param name="typeOfDisplay" select="$typeOfDisplay" />
												</xsl:call-template>
												
												<xsl:call-template name="interConnectBayCell"> 
													<xsl:with-param name="trayNumber" select="./SlotNumber+1" /> <!-- Change it to trayNumber -->
													<xsl:with-param name="enclSerialNum" select="$enclSerialNum" />
													<xsl:with-param name="typeOfDisplay" select="$typeOfDisplay" />
												</xsl:call-template>
											
											</xsl:element>
										</xsl:if>
									</xsl:for-each>
			
								</table>
							</td>
						</tr>
		
						<!-- Enclosure managers -->
						<tr>
							<td height="19">
							<xsl:attribute name="width">
								<xsl:choose>
									<xsl:when test="./@type=$C_CLASS">
										 <xsl:value-of select="$C_ENCL_WIDTH"/>
									 </xsl:when>
									 <xsl:otherwise>
										 <xsl:value-of select="$P_ENCL_WIDTH"/>
									 </xsl:otherwise>
								</xsl:choose>
							</xsl:attribute>

								<table height="19" border="0" cellspacing="0" cellpadding="0">
								<xsl:attribute name="width">
								<xsl:choose>
									<xsl:when test="./@type=$C_CLASS">
										 <xsl:value-of select="$C_ENCL_WIDTH"/>
									 </xsl:when>
									 <xsl:otherwise>
										 <xsl:value-of select="$P_ENCL_WIDTH"/>
									 </xsl:otherwise>
								</xsl:choose>
								</xsl:attribute>

									<tr>
										<!-- Active Enclosure Module -->
										<xsl:element name="td">
											<xsl:attribute name="width">92</xsl:attribute>
											<xsl:attribute name="style">border: 1px solid #333;</xsl:attribute>
											<xsl:call-template name="renderEM">
												<xsl:with-param name="bayNumber"><xsl:text>1</xsl:text></xsl:with-param>	
												<xsl:with-param name="enclSerialNum" select="$enclSerialNum"/>
												<xsl:with-param name="typeOfDisplay" select="$typeOfDisplay"/>
											</xsl:call-template>
										</xsl:element>
										
										<!-- Link NICs -->
										<xsl:element name="td">
											<xsl:attribute name="width">34</xsl:attribute>
												<xsl:attribute name="background">
													<xsl:value-of select="$EM_LINKNICS_IMAGE" />
												</xsl:attribute>
											</xsl:element>
										
										<!-- Standby Enclosure Module -->
										<xsl:element name="td">
											<xsl:attribute name="width">92</xsl:attribute>
												<xsl:call-template name="renderEM">
													<xsl:with-param name="bayNumber"><xsl:text>2</xsl:text></xsl:with-param>
													<xsl:with-param name="enclSerialNum" select="$enclSerialNum"/>		
													<xsl:with-param name="typeOfDisplay" select="$typeOfDisplay"/>								
												</xsl:call-template>
											</xsl:element>
										</tr>
									</table>
								</td>
							</tr>
					
							<!-- Bottom row of fans -->
							<tr>
								<td height="26">
								<xsl:attribute name="width">
								<xsl:choose>
									<xsl:when test="./@type=$C_CLASS">
										 <xsl:value-of select="$C_ENCL_WIDTH"/>
									 </xsl:when>
									 <xsl:otherwise>
										 <xsl:value-of select="$P_ENCL_WIDTH"/>
									 </xsl:otherwise>
								</xsl:choose>
								</xsl:attribute>

									<!-- Table container for rear view top fans -->
									<table height="26" border="0" cellspacing="0" cellpadding="0">
										<xsl:attribute name="width">
											<xsl:choose>
												<xsl:when test="./@type=$C_CLASS">
													 <xsl:value-of select="$C_ENCL_WIDTH"/>
												 </xsl:when>
												 <xsl:otherwise>
													 <xsl:value-of select="$P_ENCL_WIDTH"/>
												 </xsl:otherwise>
											</xsl:choose>
										</xsl:attribute>
									
										<!-- Bottom row of fans (6-10) -->
										<tr id="fanBottomRow">
											
											<!--
												Loop through all of the fan objects and render the bottom
												row if the fan number is greater than the number of fans
												per row in the enclosure.
											-->
				
											<xsl:for-each select="./FANS">
												<xsl:sort data-type="number" select="BayNumber"/>
												<xsl:variable name="bayNumber" select="./BayNumber"/>
												<xsl:if test="$bayNumber&gt;=6">                            
													<xsl:call-template name="fanCell">
														<xsl:with-param name="fanNumber" select="$bayNumber" />
														<xsl:with-param name="enclSerialNum" select="$enclSerialNum" />
													</xsl:call-template>
												</xsl:if>
											</xsl:for-each>
										</tr>
									</table>
								</td>
							</tr>
		
							<!-- Rear of power supplies -->
							<tr>
								<xsl:element name="td">
									
									<xsl:attribute name="height">25</xsl:attribute>
									<xsl:attribute name="width">210</xsl:attribute>
									<xsl:attribute name="id">
										<xsl:value-of select="concat('enc', $enclSerialNum, 'pim1Select')" />
									</xsl:attribute>
				
									<xsl:attribute name="devNum">
										<xsl:value-of select="1" />
									</xsl:attribute>
									<xsl:attribute name="enclSerialNum">
										<xsl:value-of select="$enclSerialNum" />
									</xsl:attribute>
									<xsl:attribute name="devType">pim</xsl:attribute>
									<xsl:attribute name="title"><xsl:value-of select="//I18N/L10NPIM"/></xsl:attribute>
									<xsl:attribute name="class">deviceCellUnselectable</xsl:attribute>
				
									<xsl:attribute name="background">
										<xsl:value-of select="$PS_REAR_IMAGE" />
									</xsl:attribute>
									<!-- Put an empty space in the cell to ensure that it is drawn. -->
									&#160;
								</xsl:element>
							</tr>

					</table>	
				</td>
			</tr>
			
		</table>
		
	</xsl:template>
	

		<!-- Interconnect bay template -->
	<xsl:template name="interConnectBayCell">
	
		<xsl:param name="typeOfDisplay" />
		<xsl:param name="trayNumber" />		
		<xsl:param name="enclSerialNum" />
		<!--
			Determine cell alignment based on cell number. Cells on the left hand side
			of the table (1, 3, 5, etc.) are aligned left.  Cells on the right hand side
			(2, 4, 6, etc.) are aligned right. NOTE: This must change if bay numbering
			scheme is changed.
		-->
		<xsl:variable name="align">
			<xsl:choose>
				<xsl:when test="number($trayNumber) mod 2 != 0">left</xsl:when>
				<xsl:otherwise>right</xsl:otherwise>
			</xsl:choose>
		</xsl:variable>
		
		<xsl:variable name="status">
			<xsl:for-each select="../Device[@bayType='Switch']">
					<xsl:if test="./SlotNumber = $trayNumber">
						<xsl:value-of select="./Status/UID"/>
					</xsl:if>
			</xsl:for-each>
		</xsl:variable>

		<xsl:variable name="statusTitle">
			<xsl:for-each select="../Device[@bayType='Switch']">
					<xsl:if test="./SlotNumber = $trayNumber">
						<xsl:value-of select="./Status/Title"/>
					</xsl:if>
			</xsl:for-each>
		</xsl:variable>

		<xsl:variable name="Name">
			<xsl:for-each select="../Device[@bayType='Switch'] ">
					<xsl:if test="./SlotNumber = $trayNumber">
						<xsl:value-of select="./Common/Name"/>
					</xsl:if>
			</xsl:for-each>
		</xsl:variable>


		<xsl:variable name="GUID">
			<xsl:for-each select="../Device[@bayType='Switch'] ">
					<xsl:if test="./SlotNumber = $trayNumber">
						<xsl:value-of select="./Common/NodeId"/>
					</xsl:if>
			</xsl:for-each>
		</xsl:variable>


		<xsl:variable name="productName">
			<xsl:for-each select="../Device[@bayType='Switch'] ">
					<xsl:if test="./SlotNumber = $trayNumber">
						<xsl:value-of select="./Common/Product"/>
					</xsl:if>
			</xsl:for-each>
		</xsl:variable>


		<xsl:variable name="drillDownURL">
			<xsl:for-each select="../Device[@bayType='Switch'] ">
					<xsl:if test="./SlotNumber = $trayNumber">
						<xsl:value-of select="./Common/DrillDownURL"/>
					</xsl:if>
			</xsl:for-each>
		</xsl:variable>
		
		<xsl:variable name="presence">
			<xsl:for-each select="../Device[@bayType='Switch'] ">
					<xsl:if test="./SlotNumber = $trayNumber">
						<xsl:value-of select="./hasDevice"/>
					</xsl:if>
			</xsl:for-each>
		</xsl:variable>
		<xsl:variable name="nextPresence">
			<xsl:for-each select="../Device[@bayType='Switch'] ">
					<xsl:if test="./SlotNumber = ($trayNumber+1)">
						<xsl:value-of select="./hasDevice"/>
					</xsl:if>
			</xsl:for-each>
		</xsl:variable>
		<xsl:variable name="colspan">
			
			<xsl:choose>
				
				<xsl:when test="number($trayNumber) mod 2 != 0">
					
					<xsl:choose>
						<xsl:when test="$nextPresence = $SUBSUMED">2</xsl:when>
						<xsl:otherwise>1</xsl:otherwise>
					</xsl:choose>
					
				</xsl:when>
				
				<xsl:otherwise>1</xsl:otherwise>
				
			</xsl:choose>
			
		</xsl:variable>
		
		<xsl:if test="$presence != $SUBSUMED">
			<td style="border: 1px solid #333;">
				<xsl:attribute name="colspan"><xsl:value-of select="$colspan" /></xsl:attribute>				
				<xsl:attribute name="width">111</xsl:attribute>
				<xsl:attribute name="height">18</xsl:attribute>
				<xsl:attribute name="valign">middle</xsl:attribute>
				
                <xsl:choose>
                    <xsl:when test="$presence=$PRESENT and $typeOfDisplay=$BPV">
                        <xsl:attribute name="class">deviceCell</xsl:attribute>
                        <xsl:attribute name="drilldownURL"><xsl:value-of select="$drillDownURL" /></xsl:attribute>
                    </xsl:when>
                    <xsl:otherwise>
                        <xsl:attribute name="class">deviceCellUnselectable</xsl:attribute>
                    </xsl:otherwise>
                </xsl:choose>
				
				<xsl:attribute name="id">
					<xsl:value-of select="concat('interconnect', $trayNumber, 'Select')" />
				</xsl:attribute>
				
				<xsl:attribute name="devNum"><xsl:value-of select="$trayNumber" /></xsl:attribute>
				<xsl:attribute name="enclSerialNum"><xsl:value-of select="$enclSerialNum" /></xsl:attribute>
				<xsl:attribute name="devType">interconnect</xsl:attribute>

				<xsl:attribute name="background">
					<xsl:choose>
						<xsl:when test="$presence=$PRESENT">
								<xsl:choose>
									<xsl:when test="contains($productName,$PASS_MODEL_PATTERN)">
										<xsl:value-of select="$C_SWITCH_PATH"/><xsl:text>io_pass.gif</xsl:text>
									</xsl:when>
									<xsl:when test="contains($productName,$CISCO_MODEL_PATTERN) ">
										<xsl:value-of select="$C_SWITCH_PATH"/><xsl:text>io_cisco.gif</xsl:text>
									</xsl:when>
									<xsl:when test="contains($productName,$SAN_MODEL_PATTERN)">
										<xsl:value-of select="$C_SWITCH_PATH"/><xsl:text>io_fc.gif</xsl:text>
									</xsl:when>
									<xsl:when test="contains($productName,$GBE_MODEL_PATTERN)">
										<xsl:value-of select="$C_SWITCH_PATH"/><xsl:text>io_gbe.gif</xsl:text>
									</xsl:when>
									<xsl:otherwise>
										<xsl:value-of select="$C_SWITCH_PATH"/><xsl:text>io_pass.gif</xsl:text>
									</xsl:otherwise>
								</xsl:choose>
						</xsl:when>
						<xsl:otherwise>
							<xsl:value-of select="$IB_ABSENT_IMAGE" />
						</xsl:otherwise>
					</xsl:choose>
				</xsl:attribute>
				
				
                <xsl:attribute name="title">
                    <xsl:choose>
                        <xsl:when test="$presence=$PRESENT">
							<xsl:value-of select="//I18N/L10NBay"/>:<xsl:text> </xsl:text> <xsl:value-of select="$trayNumber"/><xsl:text>, </xsl:text><xsl:value-of select="//I18N/L10NProductName"/>:<xsl:text> </xsl:text><xsl:value-of select="$Name"/><xsl:text>, </xsl:text><xsl:value-of select="//I18N/L10NProductModel"/>:<xsl:text> </xsl:text><xsl:value-of select="$productName"/>
                        </xsl:when>
                        <xsl:otherwise>
                            <xsl:value-of select="//I18N/L10NEmptyBay"/>
                        </xsl:otherwise>
                    </xsl:choose>
                </xsl:attribute>
				
				<xsl:attribute name="align"><xsl:value-of select="$align" /></xsl:attribute>
				
				<!-- Interconnect bay container -->
					<xsl:call-template name="interconnectBay">
						<xsl:with-param name="typeOfDisplay" select="$typeOfDisplay" />
						<xsl:with-param name="trayNumber" select="$trayNumber" />
						<xsl:with-param name="presence" select="$presence" />
						<xsl:with-param name="status" select="$status" />
						<xsl:with-param name="statusTitle" select="$statusTitle" />
						<xsl:with-param name="drillDownURL" select="$drillDownURL" />
						<xsl:with-param name="GUID" select="$GUID"/>
					</xsl:call-template>
			</td>
		</xsl:if>
	</xsl:template>
	
	
	<!-- Fan cell template -->
	<xsl:template name="fanCell">
		
		<xsl:param name="enclSerialNum" />
		<xsl:param name="fanNumber" />
		<xsl:element name="td">
			<xsl:attribute name="style">border: 1px solid #333;</xsl:attribute>
			<xsl:variable name="presence">
				<xsl:value-of select="./hasDevice"/>
			</xsl:variable>
			
			

			<!--<xsl:attribute name="height">30</xsl:attribute>-->
			<xsl:attribute name="height">26</xsl:attribute>
			<xsl:attribute name="width">42</xsl:attribute>
			<xsl:attribute name="align">center</xsl:attribute>
			<xsl:attribute name="valign">middle</xsl:attribute>
            
            <!--
                TODO: This may always need to be selectable for fans in case
                an e-keying error has occurred and the user needs to know where
                to move fans.
            -->
              <xsl:attribute name="class">deviceCellUnselectable</xsl:attribute>

			<xsl:attribute name="id">
				<xsl:value-of select="concat('enc', $enclSerialNum, 'fan', $fanNumber, 'Select')" />
			</xsl:attribute>
			
			<xsl:attribute name="devNum"><xsl:value-of select="$fanNumber" /></xsl:attribute>
			<xsl:attribute name="enclSerialNum"><xsl:value-of select="$enclSerialNum" /></xsl:attribute>
			<xsl:attribute name="devType">fan</xsl:attribute>

			<xsl:attribute name="title">
				<xsl:choose>
					<xsl:when test="$presence=$PRESENT">
							<xsl:value-of select="//I18N/L10NFAN"/><xsl:text> </xsl:text><xsl:value-of select="$fanNumber"/>
					</xsl:when>
					<xsl:otherwise>
						<xsl:value-of select="//I18N/L10NEmptyBay"/>
					</xsl:otherwise>
				</xsl:choose>
			</xsl:attribute>
			
			<xsl:attribute name="background">

				<xsl:choose>
					
					<xsl:when test="$presence=$PRESENT">
						<xsl:choose>
							<xsl:when test="$fanNumber &lt;= $FANS_PER_ROW">
								<xsl:value-of select="$FAN_PRESENT_TOP_IMAGE" />
							</xsl:when>
							<xsl:otherwise>
								<xsl:value-of select="$FAN_PRESENT_BOT_IMAGE" />
							</xsl:otherwise>
						</xsl:choose>
						
					</xsl:when>
					
					<xsl:otherwise>
                        <!-- TODO: Make sure to change this back -->
						<xsl:value-of select="$FAN_ABSENT_IMAGE" />
					</xsl:otherwise>
					
				</xsl:choose>
				
			</xsl:attribute>
			
			<xsl:element name="div" >
			
				<xsl:attribute name="style">display:block; width:100%; align:center;</xsl:attribute>
				
				<xsl:attribute name="id"><xsl:value-of select="concat('fanContainer', $fanNumber)" /></xsl:attribute>
				
				<!-- Only render the status icon if the fan is present -->
                <xsl:choose>
                    <xsl:when test="$presence=$PRESENT">
						<xsl:call-template name="statusIcon">
							<xsl:with-param name="statusCode" select="Status/UID" />
							<xsl:with-param name="titleMsg" select="Status/Title" />
						</xsl:call-template>
                      </xsl:when>
                    <xsl:otherwise>
                        &#160;
                    </xsl:otherwise>
                </xsl:choose>
				
			</xsl:element>
			
		</xsl:element>
		
	</xsl:template>


<xsl:template name="renderEM">
			<xsl:param name="bayNumber"/>
			<xsl:param name="enclSerialNum"/>
			<xsl:param name="typeOfDisplay"/>
			
			<xsl:attribute name="id"><xsl:value-of select="concat('enc', $enclSerialNum, 'em1Select')" /></xsl:attribute>
			<xsl:attribute name="devType">em</xsl:attribute>
             <xsl:attribute name="valign">middle</xsl:attribute>
             <xsl:attribute name="style">border: 1px solid #333;</xsl:attribute>

			<xsl:attribute  name="align">
				<xsl:choose>
					<xsl:when test="number($bayNumber) mod 2 != 0">left</xsl:when>
					<xsl:otherwise>right</xsl:otherwise>
				</xsl:choose>
			</xsl:attribute>
             
			<xsl:for-each select="./Device[@bayType='MgmtProc']">
				<xsl:if test="./SlotNumber  = $bayNumber">
						<!-- TODO: Get EM Health value -->
						<xsl:variable name="encHealth">
							<xsl:value-of select="./Status/UID"/>
						</xsl:variable>
	
						<xsl:variable name="isPrimary">
							<xsl:value-of select="./isPrimary"/>
						</xsl:variable>
						
						<xsl:variable name="presence">
							<xsl:value-of select="./hasDevice"/>
						</xsl:variable>
						
			
						<!-- Set the class that has to applied to this <td>
							If a device is present @ this slot then set the class as deviceCell
							else set it as deviceCellUnselectable
						-->
						
						
						<xsl:choose>
							<xsl:when test="$presence=$PRESENT and $typeOfDisplay=$BPV ">
								<xsl:attribute name="class">deviceCell</xsl:attribute>
							</xsl:when>
							<xsl:otherwise>
								<xsl:attribute name="class">deviceCellUnselectable</xsl:attribute>
							</xsl:otherwise>
						</xsl:choose>
			
			
						<xsl:attribute name="background">
							<xsl:choose>
								<xsl:when test="$presence=$PRESENT">
									<xsl:value-of select="$EM_PRESENT_IMAGE" />
								</xsl:when>
								<xsl:otherwise>
									<xsl:value-of select="$EM_ABSENT_IMAGE"/>
								</xsl:otherwise>
							</xsl:choose>
						</xsl:attribute>


						<xsl:choose>
							<xsl:when test="$typeOfDisplay=$BPV and ./hasDevice=$PRESENT">
								<!-- drilldownURL attribute will point to URL when the user drill downs this device -->
								<xsl:attribute name="drilldownURL">
									<xsl:value-of select="./Common/DrillDownURL" />
								</xsl:attribute>
							</xsl:when>
						</xsl:choose>
					
						<xsl:variable name="oaRole">
							<xsl:choose>
								<xsl:when test="$isPrimary='true'"><xsl:value-of select="//I18N/L10NACTIVEOA"/></xsl:when>
								<xsl:otherwise><xsl:value-of select="//I18N/L10NSTANDBYOA"/></xsl:otherwise>
							</xsl:choose>
						</xsl:variable>
					
						<xsl:attribute name="title">
							<xsl:choose>
								<xsl:when test="$presence=$PRESENT">
								<xsl:value-of select="//I18N/L10NBay"/>:<xsl:text> </xsl:text><xsl:value-of select="./SlotNumber"/><xsl:text>, </xsl:text><xsl:value-of select="//I18N/L10NProductName"/>:<xsl:text> </xsl:text><xsl:value-of select="./Common/Name"/><xsl:text>, </xsl:text><xsl:value-of select="//I18N/L10NProductModel"/>:<xsl:text> </xsl:text><xsl:value-of select="./Common/Product"/><xsl:text>, </xsl:text><xsl:value-of select="//I18N/L10NRole"/>:<xsl:text> </xsl:text><xsl:value-of select="$oaRole"/>
								</xsl:when>
								<xsl:otherwise>
									<xsl:value-of select="//I18N/L10NEmptyBay"/>
								</xsl:otherwise>
							</xsl:choose>
						</xsl:attribute>
						
 					    <xsl:call-template name="emBay">
							<xsl:with-param name="emNumber" select="$bayNumber" />
							<xsl:with-param name="status" select="$encHealth" />
							<xsl:with-param name="typeOfDisplay" select="$typeOfDisplay" />
							<xsl:with-param name="enclSerialNum" select="$enclSerialNum" />
							<xsl:with-param name="presence" select="$presence" />
						</xsl:call-template>
						

				</xsl:if>
			</xsl:for-each>

</xsl:template>



	<!--
		Individual interconnect bay template. Contains status icon and 
		checkbox for interconnect bays.
	-->
	<xsl:template name="emBay">
		<xsl:param name="emNumber" />
		<xsl:param name="status" />
		<xsl:param name="typeOfDisplay"/>
		<xsl:param name="presence"/>
		<xsl:variable name="hwtoolTip" select="//I18N/HWStatusToolTip"/>
		<xsl:variable name="clkMsg" select="//I18N/DetailsToolTip"/>
		<xsl:variable name="statTitle" select="./Status/Title"/>
		
		<xsl:choose>
			<!-- OA is present in bay -->
            <xsl:when test="$presence=$PRESENT">
			<table cellpadding="0" cellspacing="0" border="0">
				<tr>
					<xsl:choose>	
					<!--
						Determine whether or not the bay is on the left or right side.
						Render the checkbox and status icon accordingly.
					-->
						<xsl:when test="$emNumber mod 2 != 0">
							<!-- Align left -->
								<td valign="top">
									<div style="margin-top:-2px">
										<input type="checkbox" rowselector="yes" id="oaCheckBox" >
											<xsl:if test="$typeOfDisplay!=$BPV">
												<xsl:attribute name="DISABLED"/>
											</xsl:if>
											<xsl:attribute name="guid">
												<xsl:value-of select="./Common/NodeId" />
											</xsl:attribute>
										</input>
									</div>
								</td>

							<td valign="middle">
								  <xsl:choose>
										<xsl:when test="$typeOfDisplay=$BPV">
											<xsl:call-template name="statusIconLink">
												<xsl:with-param name="statusCode" select="$status" />
												<xsl:with-param name="fallBackURL" select="./Common/DrillDownURL"/>
												<xsl:with-param name="titleMsg" select="concat($hwtoolTip,': ',$statTitle,'. ',$clkMsg,'.')"/>
											</xsl:call-template>
										</xsl:when>
										<xsl:otherwise>
											<xsl:call-template name="statusIcon">
												<xsl:with-param name="statusCode" select="$status" />
												<xsl:with-param name="titleMsg" select="concat($hwtoolTip,': ',$statTitle,'. ',$clkMsg,'.')"/>
											</xsl:call-template>
										</xsl:otherwise>
								</xsl:choose>
							</td>
						</xsl:when>
						<xsl:otherwise>
							<!-- Align right -->
							<td valign="middle">
								  <xsl:choose>
										<xsl:when test="$typeOfDisplay=$BPV">
											<xsl:call-template name="statusIconLink">
												<xsl:with-param name="statusCode" select="$status" />
												<xsl:with-param name="fallBackURL" select="./Common/DrillDownURL"/>
												<xsl:with-param name="titleMsg" select="concat($hwtoolTip,': ',$statTitle,'. ',$clkMsg,'.')"/>
											</xsl:call-template>
										</xsl:when>
										<xsl:otherwise>
											<xsl:call-template name="statusIcon">
												<xsl:with-param name="statusCode" select="$status" />
												<xsl:with-param name="titleMsg" select="concat($hwtoolTip,': ',$statTitle,'. ',$clkMsg,'.')"/>
											</xsl:call-template>
										</xsl:otherwise>
								</xsl:choose>
							</td>

								<td valign="top">
									<div style="margin-top:-2px">
										<input type="checkbox" rowselector="yes" id="oaCheckBox" >
											<xsl:if test="$typeOfDisplay!=$BPV">
												<xsl:attribute name="DISABLED"/>
											</xsl:if>
											<xsl:attribute name="guid">
												<xsl:value-of select="./Common/NodeId" />
											</xsl:attribute>
										</input>
									</div>
								</td>

						</xsl:otherwise>
					</xsl:choose>
				</tr>
			</table>
			<!-- No OA is present in bay (absent) -->
			</xsl:when>
			<xsl:otherwise>
				&#160;
			</xsl:otherwise>
		</xsl:choose>
	</xsl:template>
	
	<!--
		Individual interconnect bay template. Contains status icon and 
		checkbox for interconnect bays.
	-->
	<xsl:template name="interconnectBay">
		<xsl:param name="trayNumber" />
		<xsl:param name="presence" />
		<xsl:param name="status" />
		<xsl:param name="statusTitle" />
		<xsl:param name="drillDownURL" />
		<xsl:param name="typeOfDisplay" />
		<xsl:param name="GUID" />
		<xsl:variable name="hwtoolTip" select="//I18N/HWStatusToolTip"/>
		<xsl:variable name="clkMsg" select="//I18N/DetailsToolTip"/>
		
		<xsl:choose>
			<xsl:when test="$presence != $ABSENT">
			<table cellpadding="0" cellspacing="0" border="0">
				<tr>
				<xsl:choose>	
					<!--
						Determine whether or not the bay is on the left or right side.
						Render the checkbox and status icon accordingly. NOTE: This test
						will have to be updated with changes in interconnect bay numbering.
					-->
					<xsl:when test="$trayNumber mod 2 != 0">
						<!-- Align left -->
							<td valign="top">
								<div style="margin-top:-2px">
									<input type="checkbox" rowselector="yes" id="switchCheckBox">
											<xsl:if test="$typeOfDisplay!=$BPV">
												<xsl:attribute name="DISABLED"/>
											</xsl:if>
											<xsl:attribute name="guid">
												<xsl:value-of select="$GUID" />
											</xsl:attribute>
									</input>
								</div>
							</td>
						<td valign="middle">
							  <xsl:choose>
									<xsl:when test="$typeOfDisplay=$BPV">
										<xsl:call-template name="statusIconLink">
											<xsl:with-param name="statusCode" select="$status" />
											<xsl:with-param name="fallBackURL" select="$drillDownURL"/>
											<xsl:with-param name="titleMsg" select="concat($hwtoolTip,': ',$statusTitle,'. ',$clkMsg,'.')"/>
										</xsl:call-template>
									</xsl:when>
									<xsl:otherwise>
										<xsl:call-template name="statusIcon">
											<xsl:with-param name="statusCode" select="$status" />
											<xsl:with-param name="titleMsg" select="concat($hwtoolTip,': ',$statusTitle,'. ',$clkMsg,'.')"/>
										</xsl:call-template>
									</xsl:otherwise>
							</xsl:choose>
						</td>
					</xsl:when>
					<xsl:otherwise>
						<!-- Align right -->
						<td valign="middle">
							  <xsl:choose>
									<xsl:when test="$typeOfDisplay=$BPV">
										<xsl:call-template name="statusIconLink">
											<xsl:with-param name="statusCode" select="$status" />
											<xsl:with-param name="fallBackURL" select="$drillDownURL"/>
											<xsl:with-param name="titleMsg" select="concat($hwtoolTip,': ',$statusTitle,'. ',$clkMsg,'.')"/>
										</xsl:call-template>
									</xsl:when>
									<xsl:otherwise>
										<xsl:call-template name="statusIcon">
											<xsl:with-param name="statusCode" select="$status" />
											<xsl:with-param name="titleMsg" select="concat($hwtoolTip,': ',$statusTitle,'. ',$clkMsg,'.')"/>
										</xsl:call-template>
									</xsl:otherwise>
							</xsl:choose>
						</td>
							<td valign="top">
								<div style="margin-top:-2px">
									<input type="checkbox" rowselector="yes" id="switchCheckBox" >
										<xsl:if test="$typeOfDisplay!=$BPV">
											<xsl:attribute name="DISABLED"/>
										</xsl:if>
										<xsl:attribute name="guid">
											<xsl:value-of select="$GUID" />
										</xsl:attribute>
									</input>
								</div>
							</td>
					</xsl:otherwise>
				</xsl:choose>
			</tr>
		</table>
	</xsl:when>
			<xsl:otherwise>
				&#160;
			</xsl:otherwise>
		</xsl:choose>
</xsl:template>
	
</xsl:stylesheet>