/*
  This software is licensed under the terms of the GNU
  General Public License (GPL) Version 2, available at
  <http://www.fsf.org/copyleft/gpl.html>

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
  MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
  NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
  BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
  ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
  CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.

  Copyright (c) 2004 Topspin Communications.  All rights reserved.

  $Id$
*/


#ifdef RCSID
static const char rcsid[]= "$Id$";
#endif

#ifdef W2K_OS // Vipul
#include <ntddk.h>
#include "include/ts_kernel_services.h"
#include "include/ts_kernel_trace.h"

#include "all/common/include/module_names.h"

#ifdef W2K_OS
/* From NTDDK.H */
HANDLE
PsGetCurrentThreadId( VOID );
/* From NTDDK.H */
#endif
typedef struct tTRACE_INFO_STRUCT {
  tTS_TRACE_LEVEL tracelevel;
  uint32_t flow_mask;
} tTRACE_INFO_STRUCT;

static tTRACE_INFO_STRUCT trace_table[MOD_MAX + 1];

static char big_trace_buf[4096];
static KSPIN_LOCK big_trace_lock;

static void *_tsKernelTraceSeqStart(
                                    struct seq_file *file,
                                    loff_t *pos,
				    void *usr_arg
                                    ) {
  return (void *) (int) (*pos + 1);
}

static void *_tsKernelTraceSeqNext(
                                   struct seq_file *file,
                                   void *arg,
                                   loff_t *pos,
				   void *usr_arg
                                   ) {
  int num = (int) arg;

  (*pos)++;
  return (num <= MOD_MAX) ? (void *) (num + 1) : NULL;
}

static void _tsKernelTraceSeqStop(
                                  struct seq_file *file,
                                  void *arg,
				  void *usr_arg
                                  ) {
  /* nothing for now */
}

static int _tsKernelTraceSeqShow(
                                 struct seq_file *file,
                                 void *arg,
				 void *usr_arg
                                 ) {
  int num = (int) arg;

  if (num == 1) {
    seq_printf(file,
               "module      id  tracelevel   flowmask\n");
  } else {
    num -= 2;

    if (num < MOD_MAX) {
      seq_printf(file,
                 "%-12s%2d  %6d      0x%08x\n",
                 ModuleName[num],
                 (int) num,
                 trace_table[num].tracelevel,
                 trace_table[num].flow_mask);
    }
  }

  return 0;
}

static struct seq_operations trace_seq_operations = {
  _tsKernelTraceSeqStart,
  _tsKernelTraceSeqStop,
  _tsKernelTraceSeqNext,
  _tsKernelTraceSeqShow
};

NTSTATUS
DriverCreate(IN PDEVICE_OBJECT pdo, IN PIRP pIrp)
{
  PIO_STACK_LOCATION      irpStack;
  PFILE_OBJECT            pFileObject;
  NTSTATUS                status = STATUS_SUCCESS;

  irpStack = IoGetCurrentIrpStackLocation(pIrp);
  pFileObject = irpStack->FileObject;

  if (seq_open(&pFileObject->FsContext, &trace_seq_operations, 0) < 0) {
    status = STATUS_NO_MEMORY;
  }

  pIrp->IoStatus.Status = status;
  pIrp->IoStatus.Information = 0;
  IoCompleteRequest(pIrp, IO_NO_INCREMENT);

  return status;
}

NTSTATUS
DriverClose(IN PDEVICE_OBJECT pdo, IN PIRP pIrp)
{
  PIO_STACK_LOCATION      irpStack;
  PFILE_OBJECT            pFileObject;

  irpStack = IoGetCurrentIrpStackLocation(pIrp);
  pFileObject = irpStack->FileObject;

  seq_release(pFileObject->FsContext);

  pIrp->IoStatus.Status = STATUS_SUCCESS;
  pIrp->IoStatus.Information = 0;
  IoCompleteRequest(pIrp, IO_NO_INCREMENT);

  return STATUS_SUCCESS;
}

NTSTATUS
DriverRead(IN PDEVICE_OBJECT pdo, IN PIRP pIrp)
{
  PIO_STACK_LOCATION     irpStack;
  PFILE_OBJECT           pFileObject;
  PVOID                  userBuf;
  NTSTATUS               status;
  ssize_t                count;

  irpStack = IoGetCurrentIrpStackLocation(pIrp);
  pFileObject = irpStack->FileObject;
  userBuf = pIrp->AssociatedIrp.SystemBuffer;
  count = irpStack->Parameters.Read.Length;

  count = seq_read(pFileObject->FsContext, userBuf, count);
  status = count>=0? STATUS_SUCCESS: STATUS_INSUFFICIENT_RESOURCES;

  pIrp->IoStatus.Status = status;
  pIrp->IoStatus.Information = count;
  IoCompleteRequest(pIrp, IO_NO_INCREMENT);

  return status;
}

NTSTATUS
DriverWrite(IN PDEVICE_OBJECT pdo, IN PIRP pIrp)
{
  PIO_STACK_LOCATION      irpStack;
  PFILE_OBJECT            pFileObject;
  PVOID                   userBuf;
  ULONG                   count;
  char                    kernel_buf[256];
  int                     mod;
  int                     level;
  uint32_t                flow_mask;

  irpStack = IoGetCurrentIrpStackLocation(pIrp);
  pFileObject = irpStack->FileObject;
  userBuf = pIrp->AssociatedIrp.SystemBuffer;
  count = irpStack->Parameters.Write.Length;

  if (count > sizeof kernel_buf) {
    count = sizeof kernel_buf;
  }

  memcpy(kernel_buf, userBuf, count);

  kernel_buf[count - 1] = '\0';

  if (sscanf(kernel_buf, "%d %d %i",
             &mod, &level, &flow_mask)
      != 3) {
    goto out;
  }

  if (mod >= 0 && mod < MOD_MAX) {
    level = min(level, T_MAX - 1);
    level = max(level, 0);
    trace_table[mod].tracelevel = level;
    trace_table[mod].flow_mask = flow_mask;
  }

out:
  pIrp->IoStatus.Status = STATUS_SUCCESS;
  pIrp->IoStatus.Information = count;
  IoCompleteRequest(pIrp, IO_NO_INCREMENT);

  return STATUS_SUCCESS;
}

void tsKernelTrace(
                   const char *file,
                   int line,
                   const char *function,
                   tMODULE_ID mod,
                   tTS_TRACE_LEVEL level,
                   uint32_t flow_mask,
                   const char *format,
                   ...
                   ) {
  va_list ap;
  va_start(ap, format);
  _tsKernelTrace(file, line, function, mod, level, flow_mask, format, ap);
  va_end(ap);
}

void _tsKernelTrace(
                   const char *file,
                   int line,
                   const char *function,
                   tMODULE_ID mod,
                   tTS_TRACE_LEVEL level,
                   uint32_t flow_mask,
                   const char *format,
                   va_list args
                   ) {
  KIRQL old_irql;

  if (mod < 0 || mod >= MOD_MAX) {
    return;
  }

  if (trace_table[mod].tracelevel >= level
      && trace_table[mod].flow_mask & flow_mask) {

    int sec, usec;
    LARGE_INTEGER t, f;

    t = KeQueryPerformanceCounter(&f);
    sec  = (int) (t.QuadPart / f.QuadPart);
    usec = (int) ((t.QuadPart % f.QuadPart) * 1000000 / f.QuadPart);

    KeAcquireSpinLock(&big_trace_lock, &old_irql);

    vsnprintf(big_trace_buf, sizeof big_trace_buf, format, args);
    DbgPrint("[%d.%06d][%d][%s][%s:%d]%s\n",
		 sec, usec,
	     (int)PsGetCurrentThreadId(),
	     ModuleName[mod],
	     file,
	     line,
	     big_trace_buf);

    KeReleaseSpinLock(&big_trace_lock, old_irql);
  }
}

void tsKernelTraceLevelSet(
                           tMODULE_ID mod,
                           tTS_TRACE_LEVEL level
                           ) {
  if (mod >= 0 && mod < MOD_MAX
      && level >= T_NO_DISPLAY && level < T_MAX) {
    trace_table[mod].tracelevel = level;
  }
}

void tsKernelTraceFlowMaskSet(
                              tMODULE_ID mod,
                              uint32_t flow_mask
                              ) {
  if (mod >= 0 && mod < MOD_MAX) {
    trace_table[mod].flow_mask = flow_mask;
  }
}

int tsKernelTraceInit(
                      void
                      ) {
  int i;

  KeInitializeSpinLock(&big_trace_lock);

  for (i = 0; i <= MOD_MAX; ++i) {
    trace_table[i].tracelevel = TS_TRACE_LEVEL_KERNEL_DEFVAL;
    trace_table[i].flow_mask  = TS_TRACE_MASK_KERNEL_DEFVAL;
  }

  return 0;
}


void tsKernelTraceCleanup(
                          void
                          ) {
}

#else // W2K_OS 

#ifndef __KERNEL__
#  define __KERNEL__
#endif
#ifndef MODULE
#  define MODULE
#endif

#include "include/ts_kernel_services.h"
#include "include/ts_kernel_trace.h"

#include <linux/module.h>

#include <linux/spinlock.h>
#include <linux/proc_fs.h>

#include <asm/uaccess.h>

#include "all/common/include/module_names.h"

static const char proc_entry_name[] = "tracelevel";
static struct proc_dir_entry *proc_entry;


typedef struct tTRACE_INFO_STRUCT {
  tTS_TRACE_LEVEL tracelevel;
  uint32_t flow_mask;
} tTRACE_INFO_STRUCT;

static tTRACE_INFO_STRUCT trace_table[MOD_MAX + 1] = {
  [0 ... MOD_MAX] = {
    .tracelevel = TS_TRACE_LEVEL_KERNEL_DEFVAL,
    .flow_mask  = TS_TRACE_MASK_KERNEL_DEFVAL
  }
};

static char big_trace_buf[4096];
static spinlock_t big_trace_lock = SPIN_LOCK_UNLOCKED;

static void *_tsKernelTraceSeqStart(
                                    struct seq_file *file,
                                    loff_t *pos
                                    ) {
  return (void *) (long) (*pos + 1);
}

static void *_tsKernelTraceSeqNext(
                                   struct seq_file *file,
                                   void *arg,
                                   loff_t *pos
                                   ) {
  int num = (long) arg;

  (*pos)++;
  return (num <= MOD_MAX) ? (void *) (long) (num + 1) : NULL;
}

static void _tsKernelTraceSeqStop(
                                  struct seq_file *file,
                                  void *arg
                                  ) {
  /* nothing for now */
}

static int _tsKernelTraceSeqShow(
                                 struct seq_file *file,
                                 void *arg
                                 ) {
  int num = (long) arg;

  if (num == 1) {
    seq_printf(file,
               "module      id  tracelevel   flowmask\n");
  } else {
    num -= 2;

    if (num < MOD_MAX) {
      seq_printf(file,
                 "%-12s%2d  %6d      0x%08x\n",
                 ModuleName[num],
                 (int) num,
                 trace_table[num].tracelevel,
                 trace_table[num].flow_mask);
    }
  }

  return 0;
}

static ssize_t _tsKernelTraceProcWrite(
                                       struct file *file,
                                       const char *buffer,
                                       size_t count,
                                       loff_t *pos
                                       ) {
  char kernel_buf[256];
  int ret;
  int mod;
  int level;
  uint32_t flow_mask;

  if (count > sizeof kernel_buf) {
    count = sizeof kernel_buf;
  }

  if (copy_from_user(kernel_buf, buffer, count)) {
    return -EFAULT;
  }

  ret = count;

  kernel_buf[count - 1] = '\0';

  if (sscanf(kernel_buf, "%d %d %i",
             &mod, &level, &flow_mask)
      != 3) {
    return ret;
  }

  if (mod >= 0 && mod < MOD_MAX) {
    level = min(level, T_MAX - 1);
    level = max(level, 0);
    trace_table[mod].tracelevel = level;
    trace_table[mod].flow_mask = flow_mask;
  }

  return ret;
}

static struct seq_operations trace_seq_operations = {
  .start = _tsKernelTraceSeqStart,
  .next  = _tsKernelTraceSeqNext,
  .stop  = _tsKernelTraceSeqStop,
  .show  = _tsKernelTraceSeqShow
};

static int _tsKernelTraceProcOpen(
                                  struct inode *inode,
                                  struct file *file
                                  ) {
  return seq_open(file, &trace_seq_operations);
}

void tsKernelTrace(
                   const char *file,
                   int line,
                   const char *function,
                   tMODULE_ID mod,
                   tTS_TRACE_LEVEL level,
                   uint32_t flow_mask,
                   const char *format,
                   ...
                   ) {
  unsigned long flags;

  if (mod < 0 || mod >= MOD_MAX) {
    return;
  }

  if (trace_table[mod].tracelevel >= level
      && trace_table[mod].flow_mask & flow_mask) {

    va_list args;
    va_start(args, format);

    spin_lock_irqsave(&big_trace_lock, flags);

    vsnprintf(big_trace_buf, sizeof big_trace_buf, format, args);
    /* XXX add time printout? */
    printk("%s[%s][%s][%s:%d]%s\n",
           KERN_CRIT,
           ModuleName[mod],
           function,
           file,
           line,
           big_trace_buf);

    spin_unlock_irqrestore(&big_trace_lock, flags);

    va_end(args);
  }
}

void tsKernelTraceLevelSet(
                           tMODULE_ID mod,
                           tTS_TRACE_LEVEL level
                           ) {
  if (mod >= 0 && mod < MOD_MAX
      && level >= T_NO_DISPLAY && level < T_MAX) {
    trace_table[mod].tracelevel = level;
  }
}

void tsKernelTraceFlowMaskSet(
                              tMODULE_ID mod,
                              uint32_t flow_mask
                              ) {
  if (mod >= 0 && mod < MOD_MAX) {
    trace_table[mod].flow_mask = flow_mask;
  }
}

static struct file_operations trace_proc_operations = {
  .owner   = THIS_MODULE,
  .open    = _tsKernelTraceProcOpen,
  .read    = seq_read,
  .write   = _tsKernelTraceProcWrite,
  .llseek  = seq_lseek,
  .release = seq_release
};

int tsKernelTraceInit(
                      void
                      ) {
  proc_entry = create_proc_entry(proc_entry_name,
                                 S_IRUGO | S_IWUGO,
                                 tsKernelProcDirGet());
  if (!proc_entry) {
    printk(KERN_ERR "Can't create %s in Topspin proc directory\n",
           proc_entry_name);
    return -ENOMEM;
  }

  proc_entry->proc_fops = &trace_proc_operations;

  return 0;
}


void tsKernelTraceCleanup(
                          void
                          ) {
  if (proc_entry) {
    remove_proc_entry(proc_entry_name, tsKernelProcDirGet());
  }
}
#endif
