/*
  This software is licensed under the terms of the GNU
  General Public License (GPL) Version 2, available at
  <http://www.fsf.org/copyleft/gpl.html>

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
  MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
  NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
  BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
  ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
  CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.

  Copyright (c) 2004 Topspin Communications.  All rights reserved.

  $Id$
*/


#ifdef RCSID
static const char rcsid[]= "$Id$";
#endif

#ifdef W2K_OS // Vipul
#include <ntddk.h>
#include "include/ts_kernel_thread.h"

#define THREAD_NAME_SIZE 16

struct tTS_KERNEL_THREAD_STRUCT {
  int                        pid;
  HANDLE                     handle;
  PKTHREAD                   thread;
  KEVENT                     done;
  char                       name[THREAD_NAME_SIZE];
  tTS_KERNEL_THREAD_FUNCTION function;
  void *                     arg;
};

/*
 * These prototypes are not included in ntddk.h so we
 * must include them here.
 */
void KeInitializeApc(PKAPC,
                     PKTHREAD,
                     CCHAR,
                     PKKERNEL_ROUTINE,
                     PKRUNDOWN_ROUTINE,
                     PKNORMAL_ROUTINE,
                     KPROCESSOR_MODE,
                     PVOID);
void KeInsertQueueApc(PKAPC,
                      PVOID,
                      PVOID,
                      UCHAR);

static int _tsKernelThreadStart(
                                void *thread_ptr
                                ) {
  tTS_KERNEL_THREAD thread = thread_ptr;

  thread->thread = KeGetCurrentThread();
  thread->function(thread->arg);

  KeSetEvent(&thread->done, IO_NO_INCREMENT, FALSE);

  return STATUS_SUCCESS;
}

int tsKernelThreadStart(
                        const char *name,
                        tTS_KERNEL_THREAD_FUNCTION function,
                        void *arg,
                        tTS_KERNEL_THREAD *thread
                        ) {
  NTSTATUS status;

  *thread = ExAllocatePool(NonPagedPool, sizeof **thread);
  if (!*thread) {
    return STATUS_NO_MEMORY;
  }

  strncpy((*thread)->name, name, sizeof (*thread)->name);
  (*thread)->name[sizeof (*thread)->name - 1] = '\0';

  KeInitializeEvent(&(*thread)->done, NotificationEvent, FALSE);

  (*thread)->function = function;
  (*thread)->arg      = arg;

  status = PsCreateSystemThread(&(*thread)->handle,
                                THREAD_ALL_ACCESS,
                                NULL,
                                NULL,
                                NULL,
                                (PKSTART_ROUTINE)_tsKernelThreadStart,
                                *thread);

  if (!NT_SUCCESS(status)) {
    ExFreePool(*thread);
    *thread = NULL;
    return STATUS_NO_MEMORY;
  }

  return STATUS_SUCCESS;
}

static int
user_apc(tTS_KERNEL_THREAD thread)
{
  KeSetEvent(&thread->done, IO_NO_INCREMENT, FALSE);
  PsTerminateSystemThread(STATUS_SUCCESS);

  /*NOTREACHED*/
  return STATUS_SUCCESS;
}

static void
kern_apc(PKAPC apc,
         PKNORMAL_ROUTINE norm_routine,
         PVOID context, PVOID sys_arg1, PVOID sys_arg2)
{
  ExFreePool(apc);
}

int tsKernelThreadStop(
                       tTS_KERNEL_THREAD thread
                       ) {
  /*
   * To kill the system thread, create an APC and queue it to
   * the thread.  The APC will eventually get called in the
   * thread context.
   */

  PKAPC           apc;
  NTSTATUS        status;

  apc = ExAllocatePool(NonPagedPool, sizeof(KAPC));
  if (!apc) {
      return STATUS_NO_MEMORY;
  }
  KeInitializeApc(apc,
                  thread->thread,
                  0,
                  (PKKERNEL_ROUTINE)kern_apc,
                  0,
                  (PKNORMAL_ROUTINE)user_apc,
                  KernelMode,
                  thread);
  KeInsertQueueApc(apc, 0, 0, 0);

  KeWaitForSingleObject(&thread->done, Executive, KernelMode, FALSE, NULL);
  ExFreePool(thread);

  return STATUS_SUCCESS;
}

#else //W2K_OS

#ifndef __KERNEL__
#  define __KERNEL__
#endif
#ifndef MODULE
#  define MODULE
#endif

#include "include/ts_kernel_thread.h"
#include "include/ts_kernel_services.h"
#include "include/ts_kernel_trace.h"

#include <linux/slab.h>
#include <linux/timer.h>
#include <linux/smp_lock.h>

#include <asm/semaphore.h>

typedef struct tTS_KERNEL_QUEUE_ENTRY_STRUCT tTS_KERNEL_QUEUE_ENTRY_STRUCT,
  *tTS_KERNEL_QUEUE_ENTRY;

struct tTS_KERNEL_THREAD_STRUCT {
  int                        pid;
  struct semaphore           init_sem;
  struct completion          done;
  char                       name[sizeof ((struct task_struct *) 0)->comm];
  tTS_KERNEL_THREAD_FUNCTION function;
  void *                     arg;
};

struct tTS_KERNEL_QUEUE_THREAD_STRUCT {
  tTS_KERNEL_THREAD                thread;
  spinlock_t                       lock;
  struct list_head                 list;
  wait_queue_head_t                wait;
  tTS_KERNEL_QUEUE_THREAD_FUNCTION function;
  void                            *arg;
};

static void _tsKernelThreadTimeout(
                                   unsigned long arg
                                   ) {
  tTS_KERNEL_THREAD thread = (tTS_KERNEL_THREAD) arg;

  TS_REPORT_WARN(MOD_SYS,
                 "wait_for_completion (thread %s) timed out",
                 thread->name);
}

static int _tsKernelThreadStart(
                                void *thread_ptr
                                ) {
  tTS_KERNEL_THREAD thread = thread_ptr;

  lock_kernel();
#ifdef TS_KERNEL_2_6
  daemonize(thread->name);
#else
  daemonize();
  strncpy(current->comm, thread->name, sizeof current->comm);
  current->comm[sizeof current->comm - 1] = '\0';
#endif
  unlock_kernel();

#ifndef TS_KERNEL_2_6
  reparent_to_init();
#endif

#if defined(INIT_SIGHAND) || defined(TS_KERNEL_2_6)
  /* 2.4.20-8/9 kernels */
  spin_lock_irq(&current->sighand->siglock);
  siginitsetinv(&current->blocked, sigmask(SIGUSR1));
  recalc_sigpending();
  spin_unlock_irq(&current->sighand->siglock);

#else
  spin_lock_irq(&current->sigmask_lock);
  siginitsetinv(&current->blocked, sigmask(SIGUSR1));
  recalc_sigpending(current);
  spin_unlock_irq(&current->sigmask_lock);
#endif

  /* done initializing, let tsKernelThreadStart return */
  up(&thread->init_sem);

  thread->function(thread->arg);

  complete_and_exit(&thread->done, 0);
}

int tsKernelThreadStart(
                        const char *name,
                        tTS_KERNEL_THREAD_FUNCTION function,
                        void *arg,
                        tTS_KERNEL_THREAD *thread
                        ) {
  *thread = kmalloc(sizeof **thread, GFP_KERNEL);
  if (!*thread) {
    return -ENOMEM;
  }

  strncpy((*thread)->name, name, sizeof (*thread)->name);
  (*thread)->name[sizeof (*thread)->name - 1] = '\0';

  sema_init(&(*thread)->init_sem, 0);
  init_completion(&(*thread)->done);

  (*thread)->function = function;
  (*thread)->arg      = arg;

#ifdef CLONE_KERNEL  
  (*thread)->pid = kernel_thread(_tsKernelThreadStart, *thread, CLONE_KERNEL);
#else
  (*thread)->pid = kernel_thread(_tsKernelThreadStart, *thread,
                                 CLONE_FS | CLONE_FILES | CLONE_SIGHAND);
#endif
  if ((*thread)->pid < 0) {
    int ret = (*thread)->pid;
    kfree(*thread);
    *thread = NULL;
    return ret;
  }

  /* wait for thread to initialize before we return */
  down_interruptible(&(*thread)->init_sem);

  return 0;
}

int tsKernelThreadRunning(
                          tTS_KERNEL_THREAD thread
                          ) {
  return kill_proc(thread->pid, 0, 1) == 0;
}

int tsKernelThreadStop(
                       tTS_KERNEL_THREAD thread
                       ) {
  int ret = 0;
  struct timer_list debug_timer;

  init_timer(&debug_timer);
  /* Use SIGUSR1 so that we can block SIGSTOP/SIGKILL (which are used
     by shutdown scripts etc -- cf. bug 3077) */
  ret = kill_proc(thread->pid, SIGUSR1, 1);
  if (ret) {
    goto out;
  }

  debug_timer.function = _tsKernelThreadTimeout;
  debug_timer.expires  = jiffies + 10 * HZ;
  debug_timer.data     = (unsigned long) thread;
  add_timer(&debug_timer);
  wait_for_completion(&thread->done);
  del_timer_sync(&debug_timer);

 out:
  kfree(thread);
  return ret;
}

static void _tsKernelQueueThread(
                                 void *thread_ptr
                                 ) {
  tTS_KERNEL_QUEUE_THREAD thread = thread_ptr;
  struct list_head       *entry;
  int ret;

  while (!signal_pending(current)) {
    ret = wait_event_interruptible(thread->wait,
                                   !list_empty(&thread->list));
    if (ret) {
      TS_REPORT_CLEANUP(MOD_SYS,
                        "queue thread exiting");
      return;
    }

    spin_lock_irq(&thread->lock);
    entry = thread->list.next;
    list_del(entry);
    spin_unlock_irq(&thread->lock);

    thread->function(entry, thread->arg);
  }
}

int tsKernelQueueThreadStart(
                             const char                      *name,
                             tTS_KERNEL_QUEUE_THREAD_FUNCTION function,
                             void                            *arg,
                             tTS_KERNEL_QUEUE_THREAD         *thread
                             ) {
  int ret;

  *thread = kmalloc(sizeof **thread, GFP_KERNEL);
  if (!*thread) {
    return -ENOMEM;
  }

  spin_lock_init(&(*thread)->lock);
  INIT_LIST_HEAD(&(*thread)->list);
  init_waitqueue_head(&(*thread)->wait);
  (*thread)->function = function;
  (*thread)->arg      = arg;

  ret = tsKernelThreadStart(name, _tsKernelQueueThread, *thread, &(*thread)->thread);
  if (ret) {
    kfree(thread);
  }

  return ret;
}

int tsKernelQueueThreadStop(
                            tTS_KERNEL_QUEUE_THREAD thread
                            ) {
  int ret = tsKernelThreadStop(thread->thread);
  kfree(thread);
  return ret;
}

void tsKernelQueueThreadAdd(
                            tTS_KERNEL_QUEUE_THREAD thread,
                            struct list_head       *entry
                            ) {
  unsigned long flags;

  spin_lock_irqsave(&thread->lock, flags);
  list_add_tail(entry, &thread->list);
  spin_unlock_irqrestore(&thread->lock, flags);

  wake_up_interruptible(&thread->wait);
}

#endif
