/* _NVRM_COPYRIGHT_BEGIN_
 *
 * Copyright 2000-2001 by NVIDIA Corporation.  All rights reserved.  All
 * information contained herein is proprietary and confidential to NVIDIA
 * Corporation.  Any use, reproduction, or disclosure without the written
 * permission of NVIDIA Corporation is prohibited.
 *
 * _NVRM_COPYRIGHT_END_
 */

#define  __NO_VERSION__
#include "nv-misc.h"
#include "os-interface.h"
#include "nv-linux.h"

/*
 * Registry variables.
 *
 * These are defined here and used throughout resman. The default values
 * are used unless the user overrides them when loading the kernel module
 * with insmod or by specifying any given option in the modprobe config
 * file (/etc/modules.conf).
 *
 * Registry entries interesting on Linux are defined in this source file.
 * Any given registry name (windows registry leaf names) are mapped to
 * variables with the following naming convention: NVreg_NAME, where NAME
 * is the registry name with space converted to underscores.
 *
 * Note: much of resman merely probes for the existence of a flag, actual
 *       values are often ignored.
 * 
 * TODO:
 *
 *   persistence
 *
 *   Linux does not currently offer a convenient way to store registry
 *   type data persistently from kernel modules. This functionlity has
 *   been requested by module developers and there has been an effort
 *   to implement it with an extension to the module parameter/modprobe
 *   mechanism, but this experimental implementation is not included in
 *   any current Linux kernel releases. While there are several options
 *   to add persistency ourselves, it would have to be done properly to
 *   work reliable. 
 *
 *   device dependence
 *
 *   The registry functions currently operate on a global namespace.
 *   This could be changed to work on a per-device basis.
 */

/*
 * The 2nd argument to MODULE_PARM is used to verify parameters passed
 * to the module at load time. It should be a string in the following 
 * format:
 *
 *   	[min[-max]]{b,h,i,l,s}
 *
 * The MIN and MAX specifiers delimit the length of the array. If MAX
 * is omitted, it defaults to MIN; if both are omitted, the default is
 * 1. The final character is a type specifier.
 *
 *  b   byte
 *  h   short
 *  i   int
 *  l   long
 *  s   string
 */

/* 
 * Option: VideoMemoryTypeOverride
 * 
 * Description:
 *
 * We normally detect memory type on TNT cards by scanning the embedded
 * BIOS. Unfortunately, we've seen some cases where a TNT card has been
 * flashed with the wrong bios. For example, an SDRAM based TNT has been
 * flashed with an SGRAM bios, and therefore claims to be an SGRAM TNT.
 * We've therefore provided an override here. Make sure to set the value
 * toe the type of memory used on your card.
 *
 * Possible Values:
 *
 * 1: SDRAM
 * 2: SGRAM
 *
 * Note that we can only do so much here. There are border cases where
 * even this fails. For example, if 2 TNT cards are in the same system,
 * one SGRAM, one SDRAM.
 *
 * This option is disabled by default, see below for information on how
 * to enable it.
 */

static int NVreg_VideoMemoryTypeOverride = 1;
MODULE_PARM(NVreg_VideoMemoryTypeOverride, "i");

/*
 * Option: EnableVia4x
 *
 * Description:
 *
 * We've had problems with some Via chipsets in 4x mode, we need force
 * them back down to 2x mode. If you'd like to experiment with retaining
 * 4x mode, you may try setting this value to 1 If that hangs the system,
 * you're stuck with 2x mode; there's nothing we can do about it.
 *
 * Possible Values:
 *
 * 0: disable AGP 4x on Via chipsets (default)
 * 1: enable  AGP 4x on Via chipsets
 */

static int NVreg_EnableVia4x = 0;
MODULE_PARM(NVreg_EnableVia4x, "i");

/*
 * Option: EnableALiAGP
 *
 * Description:
 *
 * Some ALi chipsets (ALi1541, ALi1647) are known to cause severe system
 * stability problems with AGP enabled. To avoid lockups, we disable AGP
 * on systems with these chipsets by default. It appears that updating the
 * system BIOS and using recent versions of the kernel AGP Gart driver can
 * make such systems much more stable. If you own a system with one of the
 * aforementioned chipsets and had it working reasonably well previously,
 * or if you want to experiment with BIOS and AGPGART revisions, you can
 * re-enable AGP support by setting this option to 1.
 *
 * Possible Values:
 *
 *  0: disable AGP on Ali1541 and ALi1647 (default)
 *  1: enable  AGP on Ali1541 and ALi1647
 */

static int NVreg_EnableALiAGP = 0;
MODULE_PARM(NVreg_EnableALiAGP, "i");

/* 
 * Option: ReqAGPRate
 *
 * Description:
 *
 * Normally, the driver will compare speed modes of the chipset & card,
 * picking the highest common rate. This key forces a maximum limit, to
 * limit the driver to lower speeds. The driver will not attempt a speed
 * beyond what the chipset and card claim they are capable of.
 *
 * Make sure you really know what you're doing before you enable this
 * override. By default, AGP drivers will enable the fastest AGP rate
 * your card and motherboard chipset are capable of. Then, in some cases,
 * our driver will force this rate down to work around bugs in both our
 * chipsets, and motherboard chipsets. Using this variable will override
 * our bug fixes. This may be desirable in some cases, but not most.
 *
 * This is completely unsupported!
 *
 * Possible Values:
 *
 *  This option expects a bitmask (7 = 1 | 2 | 4, 3 = 1 | 2, etc)
 *
 * This option is disabled by default, see below for information on how
 * to enable it.
 */

static int NVreg_ReqAGPRate = 7;
MODULE_PARM(NVreg_ReqAGPRate, "i");

/* 
 * Option: UpdateKernelAGP
 *
 * Description:
 *
 * This is a cover-my-tail flag. Basically, NVIDIA has work-arounds for
 * various cards/chipsets that we use to override basic AGP setup. This
 * would be called after agpgart has configured AGP, to force AGP into
 * certain modes to workaround problems. 
 *
 * Previously, these updates weren't called for agpgart, only for our
 * internal agp code. This is why some chipsets were far more stable with
 * our code than agpgart (BX for example). I've now added a call to this
 * update code after agpgart gets set up.
 *
 * This should be a very good thing. However, I haven't been able to fully
 * test this interaction. So, I'm adding this override to disable calling
 * the update code, in case of problems. Using this override will force
 * agpgart behavior back to the 0.94 behavior.
 *
 * Possible Values:
 *
 *   0 = enable  nvidia workarounds
 *   1 = disable nvidia workarounds (default)
 */

static int NVreg_UpdateKernelAGP = 1;
MODULE_PARM(NVreg_UpdateKernelAGP, "i");

/* 
 * Option: EnableAGPSBA
 *
 * Description:
 *
 * For stability reasons, the driver will not Side Band Addressing even if
 * both the host chipset and the AGP card support it. You may override this
 * behaviour with the following registry key.
 *
 * This is completely unsupported!
 *
 * Possible Values:
 *
 *  0 = disable Side Band Addressing (default on x86, see below)
 *  1 = enable  Side Band Addressing (if supported)
 */

#if defined (NVCPU_IA64)
/*
 * The default on ia64 is to enable AGP side band addressing; if you need
 * to turn it off, change the registry key below.
 */
static int NVreg_EnableAGPSBA = 1;
#else
/*
 * The default on x86 is to disable AGP side band addressing; if you want
 * to turn it on, change the registry key below.
 */
static int NVreg_EnableAGPSBA = 0;
#endif

MODULE_PARM(NVreg_EnableAGPSBA, "i");

/*
 * Option: EnableAGPFW
 *
 * Description:
 *
 * Similar to Side Band Addressing, Fast Writes are disabled by default. If
 * you wish to enable them on systems that support them, you can do so with
 * this registry key. Please note that this may render your system unstable
 * with many AGP chipsets.
 *
 * This is completely unsupported!
 *
 * Possible Values:
 *
 *  0 = disable Fast Writes (default)
 *  1 = enable  Fast Writes (if supported)
 *
 * This option is disabled by default, see below for information on how
 * to enable it.
 */

static int NVreg_EnableAGPFW = 0;
MODULE_PARM(NVreg_EnableAGPFW, "i");

/*
 * Option: SoftEDIDs
 *
 * Description:
 *
 * The SoftEDIDs registry key enables dynamic generation of an appropriate
 * EDID for mobile LCD displays from data stored in the video BIOS. If this
 * is turned off, then on mobile systems, a hardcoded EDID will be chosen
 * from a table, based on the value of the Mobile registry key.
 *
 * Possible Values:
 *
 *  0 = disable dynamic EDID generation
 *  1 = enable  dynamic EDID generation (default)
 */

static int NVreg_SoftEDIDs = 1;
MODULE_PARM(NVreg_SoftEDIDs, "i");

/* 
 * Option: Mobile
 *
 * Description:
 *
 * The Mobile registry key should only be needed on mobile systems if
 * SoftEDIDs is disabled (see above), in which case the mobile value
 * will be used to lookup the correct EDID for the mobile LCD.
 *
 * Possible Values:
 *
 *  ~0 = auto detect the correct value (default)
 *   1 = Dell notebooks
 *   2 = non-Compal Toshiba
 *   3 = all other notebooks
 *   4 = Compal/Toshiba
 *   5 = Gateway
 *
 * Make sure to specify the correct value for your notebook.
 */

static int NVreg_Mobile = ~0;
MODULE_PARM(NVreg_Mobile, "i");


static int NVreg_ResmanDebugLevel = ~0;
MODULE_PARM(NVreg_ResmanDebugLevel, "i");

static int NVreg_FlatPanelMode = 0;
MODULE_PARM(NVreg_FlatPanelMode, "i");

/*
 * You can enable any of the registry options disabled by default by
 * editing their respective entries in the table below. Options with
 * two leading underscores ("__EnableAGPSBA") are disabled; you can
 * enable such options by removing the underscores; in order for the
 * changes to take effect, you need to recompile and load the kernel
 * module.
 */

nv_parm_t nv_parms[] = {
    { NULL, "__VideoMemoryTypeOverride",  &NVreg_VideoMemoryTypeOverride  },
    { NULL, "ResmanDebugLevel",           &NVreg_ResmanDebugLevel         },
    { NULL, "EnableVia4x",                &NVreg_EnableVia4x              },
    { NULL, "EnableALiAGP",               &NVreg_EnableALiAGP             },
    { NULL, "UpdateKernelAGP",            &NVreg_UpdateKernelAGP          },
    { NULL, "__ReqAGPRate",               &NVreg_ReqAGPRate               },
    { NULL, "EnableAGPSBA",               &NVreg_EnableAGPSBA             },
    { NULL, "EnableAGPFW",                &NVreg_EnableAGPFW              },
    { NULL, "SoftEDIDs",                  &NVreg_SoftEDIDs                },
    { NULL, "Mobile",                     &NVreg_Mobile                   },
    { NULL, "FlatPanelMode",              &NVreg_FlatPanelMode            },
    { NULL, NULL,                         NULL                            }
};

