/* _NVRM_COPYRIGHT_BEGIN_
 *
 * Copyright 1999-2001 by NVIDIA Corporation.  All rights reserved.  All
 * information contained herein is proprietary and confidential to NVIDIA
 * Corporation.  Any use, reproduction, or disclosure without the written
 * permission of NVIDIA Corporation is prohibited.
 *
 * _NVRM_COPYRIGHT_END_
 */



/******************* Operating System AGP Routines *************************\
*                                                                           *
* Module: os-agp.c                                                          *
*   interface to the operating system's native agp support                  *
*                                                                           *
\***************************************************************************/

#define  __NO_VERSION__
#include "nv-misc.h"

#include "os-interface.h"
#include "nv-linux.h"

/* AGPGART support - mostly by Christian Zander [phoenix@minion.de] */

#ifdef AGPGART

typedef struct {
    agp_memory *ptr;
    int num_pages;
    int offset;
#if defined(NVCPU_IA64)
    struct vm_area_struct *vma;
    void  *handle;
#endif
} agp_priv_data;

typedef struct {
    void *aperture;
#ifdef CONFIG_MTRR
    int  mtrr;
#endif
    int  ready;
} agp_gart;

typedef struct {
    int           (*backend_acquire)(void);
    void          (*backend_release)(void);
    void          (*copy_info)(agp_kern_info *);
    agp_memory *  (*allocate_memory)(size_t, unsigned int);
    void          (*free_memory)(agp_memory *);
    int           (*bind_memory)(agp_memory *, off_t);
    int           (*unbind_memory)(agp_memory *);
    void          (*enable)(unsigned int);
} agp_operations_struct;

agp_operations_struct agp_ops;
agp_kern_info         agpinfo;
agp_gart              gart;
#if !defined (KERNEL_2_2)
const drm_agp_t       *drm_agp_p;
#endif

#if defined (KERNEL_2_2)
    #define GET_AGPGART_SYMBOL(sym, sym_string)                     \
        sym = (void*) GET_MODULE_SYMBOL(0, sym_string);             \
        if (sym == NULL)                                            \
        {                                                           \
            nv_printf(NV_DBG_ERRORS,                                \
                "NVRM: AGPGART: unable to retrieve symbol %s\n",    \
                sym_string);                                        \
            return 1;                                               \
        }

    #define AGP_BACKEND_ACQUIRE_SYM   __MODULE_STRING(agp_backend_acquire)
    #define AGP_BACKEND_RELEASE_SYM   __MODULE_STRING(agp_backend_release)
    #define AGP_COPY_INFO_SYM         __MODULE_STRING(agp_copy_info)
    #define AGP_ALLOCATE_MEMORY_SYM   __MODULE_STRING(agp_allocate_memory)
    #define AGP_FREE_MEMORY_SYM       __MODULE_STRING(agp_free_memory)
    #define AGP_BIND_MEMORY_SYM       __MODULE_STRING(agp_bind_memory)
    #define AGP_UNBIND_MEMORY_SYM     __MODULE_STRING(agp_unbind_memory)
    #define AGP_ENABLE_SYM            __MODULE_STRING(agp_enable)
#endif

#endif /* AGPGART */

BOOL KernInitAGP(
    nv_state_t *nv,
    VOID **ap_phys_base,
    VOID **ap_mapped_base,
    U032 *apsize
)
{
#ifndef AGPGART
    return 1;
#else
    U032  agp_rate;
    U032  agp_sba;
    U032  agp_fw;
    char* chipset;
    VOID *bitmap;
    U032 bitmap_size;

    memset( (void *) &gart, 0, sizeof(agp_gart));

#if !defined (KERNEL_2_2)
    if (!(drm_agp_p = inter_module_get_request("drm_agp", "agpgart")))
    {
        nv_printf(NV_DBG_ERRORS,
            "NVRM: AGPGART: unable to retrieve symbol table\n");
        return 1;
    }
    
    agp_ops.backend_acquire = drm_agp_p->acquire;
    agp_ops.backend_release = drm_agp_p->release;
    agp_ops.allocate_memory = drm_agp_p->allocate_memory;
    agp_ops.free_memory     = drm_agp_p->free_memory;
    agp_ops.bind_memory     = drm_agp_p->bind_memory;
    agp_ops.unbind_memory   = drm_agp_p->unbind_memory;
    agp_ops.enable          = drm_agp_p->enable;

    // looks like some newer kernels (for example mandrake 9.0's 2.4.19-16mdk)
    // have updated copy_info to return an integer value, and of course didn't
    // bother bumping the agpgart revision up. The return value is pretty 
    // harmless (backend_acquire would have already failed and caused us to
    // bail), so cast the function pointer to avoid compiler warnings.
    // we may need to revisit this in the future.
    agp_ops.copy_info       = (void (*)(agp_kern_info *)) drm_agp_p->copy_info;

#else
#if defined(CONFIG_KMOD)
    if ( request_module("agpgart") )
    {
        nv_printf(NV_DBG_ERRORS, "NVRM: AGPGART: not loading agpgart.o\n");
        return 1;
    }
#endif

    GET_AGPGART_SYMBOL(agp_ops.backend_acquire, AGP_BACKEND_ACQUIRE_SYM);
    GET_AGPGART_SYMBOL(agp_ops.backend_release, AGP_BACKEND_RELEASE_SYM);
    GET_AGPGART_SYMBOL(agp_ops.copy_info,       AGP_COPY_INFO_SYM);
    GET_AGPGART_SYMBOL(agp_ops.allocate_memory, AGP_ALLOCATE_MEMORY_SYM);
    GET_AGPGART_SYMBOL(agp_ops.free_memory,     AGP_FREE_MEMORY_SYM);
    GET_AGPGART_SYMBOL(agp_ops.bind_memory,     AGP_BIND_MEMORY_SYM);
    GET_AGPGART_SYMBOL(agp_ops.unbind_memory,   AGP_UNBIND_MEMORY_SYM);
    GET_AGPGART_SYMBOL(agp_ops.enable,          AGP_ENABLE_SYM);
#endif

    /* NOTE: from here down, return an error code of '-1'
     * that indicates that agpgart is loaded, but we failed to use it
     * in some way. This is so we don't try to use nvagp and lock up
     * the memory controller.
     */

    if ( (*(agp_ops.backend_acquire))() )
    {
        nv_printf(NV_DBG_ERRORS, "NVRM: AGPGART: backend in use\n");
        return -1;
    }

    if (rm_read_registry_dword(nv, "NVreg", "ReqAGPRate", &agp_rate) == RM_ERROR)
        agp_rate = 0x00000007;
    agp_rate &= 0x00000007;

    if (rm_read_registry_dword(nv, "NVreg", "EnableAGPSBA", &agp_sba) == RM_ERROR)
        agp_sba = 1;
    agp_sba &= 0x00000001;

#if defined(NVCPU_IA64)
    /* Disable fastwrites on IA64 */
    agp_fw = 0;
#else
    if (rm_read_registry_dword(nv, "NVreg", "EnableAGPFW", &agp_fw) == RM_ERROR)
        agp_fw = 1;
    agp_fw &= 0x00000001;
#endif
    (*(agp_ops.copy_info))(&agpinfo);

    switch ( agpinfo.chipset )
    {
        case INTEL_GENERIC:    chipset = "Intel";            break;
        case INTEL_LX:         chipset = "Intel 440LX";      break;
        case INTEL_BX:         chipset = "Intel 440BX";      break;
        case INTEL_GX:         chipset = "Intel 440GX";      break;
        case INTEL_I810:       chipset = "Intel i810";       break;
        case INTEL_I840:       chipset = "Intel i840";       break;
#if !defined (KERNEL_2_2)
        case INTEL_I815:       chipset = "Intel i815";       break;
#if !defined(__rh_config_h__)
        case INTEL_I850:       chipset = "Intel i850";       break;
#endif
#endif
#if defined(NVCPU_IA64)
        case INTEL_460GX:      chipset = "Intel 460GX";      break;
#endif
        case VIA_GENERIC:      chipset = "VIA";              break;
        case VIA_VP3:          chipset = "VIA VP3";          break;
        case VIA_MVP3:         chipset = "VIA MVP3";         break;
        case VIA_MVP4:         chipset = "VIA MVP4";         break;
#if !defined (KERNEL_2_2)
        case VIA_APOLLO_KX133: chipset = "VIA Apollo KX133"; break;
        case VIA_APOLLO_KT133: chipset = "VIA Apollo KT133"; break;
#endif
        case VIA_APOLLO_PRO:   chipset = "VIA Apollo Pro";   break;
        case SIS_GENERIC:      chipset = "SiS";              break;
        case AMD_GENERIC:      chipset = "AMD";              break;
        case AMD_IRONGATE:     chipset = "AMD Irongate";     break;
        case ALI_M1541:        chipset = "ALi M1541";        break;
        case ALI_GENERIC:      chipset = "ALi";              break;
        case NOT_SUPPORTED:    chipset = "unsupported";      break;
        default:               chipset = "unknown";
    }

    nv_printf(NV_DBG_INFO, "NVRM: AGPGART: %s chipset\n", chipset);

#ifdef CONFIG_MTRR
    if ((gart.mtrr = mtrr_add(agpinfo.aper_base,
                              agpinfo.aper_size * 0x100000,
                              MTRR_TYPE_WRCOMB, 0x0)) < 0)
    {
        /*
         * Failure to set a write-combining range on the AGP aperture may
         * be due to the presence of other memory ranges with conflicting
         * caching  attributes. Play safe and fail AGP initialization.
         */
        nv_printf(NV_DBG_ERRORS, 
            "NVRM: AGPGART: unable to set MTRR write-combining\n");
        return -1;
    }
#endif

    /*
     * Just map a single page for drivers to osGetAddressInfo() will work.
     * This will allow a client to create a DMA context for the entire
     * aperture without mapping the entire aperture.
     */

    if (!(gart.aperture = (void*) ioremap(agpinfo.aper_base, RM_PAGE_SIZE)))
    {
        nv_printf(NV_DBG_ERRORS, "NVRM: AGPGART: unable to remap aperture\n");
        return -1;
    }

    // allocate and set the bitmap for tracking agp allocations
    bitmap_size = (agpinfo.aper_size * 0x100000)/PAGE_SIZE/8;
    if (os_alloc_mem(&bitmap, bitmap_size))
    {
        iounmap(gart.aperture);
        nv_printf(NV_DBG_ERRORS, "NVRM: AGPGART: unable to allocate bitmap\n");
        return -1;
    }

    os_mem_set(bitmap, 0xff, bitmap_size);
    if (rm_set_agp_bitmap(nv, bitmap))
    {
        iounmap(gart.aperture);
        os_free_mem(bitmap);
        nv_printf(NV_DBG_ERRORS, "NVRM: AGPGART: unable to set bitmap\n");
        return -1;
    }

    nv_printf(NV_DBG_INFO,
        "NVRM: AGPGART: aperture: %ldM @ 0x%08lx\n",
        (unsigned long)agpinfo.aper_size,
        (unsigned long)agpinfo.aper_base);

    nv_printf(NV_DBG_INFO, 
        "NVRM: AGPGART: aperture mapped from 0x%08lx to 0x%08lx\n",
        agpinfo.aper_base,
        (unsigned long) gart.aperture);

    if (!agp_sba) agpinfo.mode &= ~0x00000200;
    if (!agp_fw)  agpinfo.mode &= ~0x00000010;

    if (!(agp_rate & 0x00000004)) agpinfo.mode &= ~0x00000004;
    if (!(agp_rate & 0x00000002)) agpinfo.mode &= ~0x00000002;
    
    (*(agp_ops.enable))(agpinfo.mode);
    
    if (agpinfo.mode & 0x00000200)
        nv_printf(NV_DBG_INFO, "NVRM: AGPGART: backend supports sba\n");
    if (agpinfo.mode & 0x00000010)
        nv_printf(NV_DBG_INFO, "NVRM: AGPGART: backend supports fw\n");
    if (agpinfo.mode & 0x00000004)
        nv_printf(NV_DBG_INFO, "NVRM: AGPGART: mode 4x\n");
    else if (agpinfo.mode & 0x00000002)
        nv_printf(NV_DBG_INFO, "NVRM: AGPGART: mode 2x\n");
    else if (agpinfo.mode & 0x00000001)
        nv_printf(NV_DBG_INFO, "NVRM: AGPGART: mode 1x\n");

    *ap_phys_base   = (void*) agpinfo.aper_base;
    *ap_mapped_base = (void*) gart.aperture;
    *apsize         = (agpinfo.aper_size * 0x100000) - 1;

    gart.ready = 1;

    return 0;
#endif /* AGPGART */
}

BOOL KernTeardownAGP(
    nv_state_t *nv
)
{
#ifndef AGPGART
    return 1;
#else
    VOID *bitmap;

    /* sanity check to make sure we should actually be here. */
    if (!gart.ready)
    {
        nv_printf(NV_DBG_ERRORS, 
            "NVRM: AGPGART: not ready so not releasing backend\n");
        return 0;
    }

    if (gart.aperture != NULL)
    {
#ifdef CONFIG_MTRR
        if (gart.mtrr > 0)
            mtrr_del(gart.mtrr, 0, 0);
#endif
        iounmap(gart.aperture);
    }

    (*(agp_ops.backend_release))();

#if !defined (KERNEL_2_2)
    inter_module_put("drm_agp");
#endif

    if (rm_clear_agp_bitmap(nv, &bitmap))
    {
        nv_printf(NV_DBG_WARNINGS, "NVRM: AGPGART: failed to clear bitmap\n");
        return 1;
    }

    os_free_mem(bitmap);

    nv_printf(NV_DBG_INFO, "NVRM: AGPGART: backend released\n");
    return 0;
#endif /* AGPGART */
}


RM_STATUS KernAllocAGPPages(
    nv_state_t *nv,
    VOID **pAddress,
    U032 PageCount,
    VOID **pPriv_data,
    U032 *Offset
)
{
#ifndef AGPGART
    *pAddress = (void*) 0;
    return RM_ERROR;
#else
    agp_memory *ptr;
    int err;
    agp_priv_data *data;
    RM_STATUS status;

    if (!gart.ready)
    {
        *pAddress = (void*) 0;
        nv_printf(NV_DBG_ERRORS, "NVRM: AGPGART: backend not ready\n");
        return RM_ERROR;
    }
    
    if (rm_alloc_agp_bitmap(nv, PageCount, Offset))
    {
        *pAddress = (void*) 0;
        nv_printf(NV_DBG_ERRORS, "NVRM: AGPGART: failed to allocate agp offset\n");
        return RM_ERROR;
    }

    ptr = (*agp_ops.allocate_memory)(PageCount, AGP_NORMAL_MEMORY);
    if (ptr == NULL)
    {
        *pAddress = (void*) 0;
        nv_printf(NV_DBG_ERRORS, "NVRM: AGPGART: no pages available\n");
        return RM_ERR_NO_FREE_MEM;
    }
    
    err = (*(agp_ops.bind_memory))(ptr, *Offset);
    if (err)
    {
        // this happens a lot when the aperture itself fills up..
        // not a big deal, so don't alarm people with an error message
        nv_printf(NV_DBG_INFO, "NVRM: AGPGART: unable to bind %lu pages\n",
            (unsigned long) PageCount);
        goto fail;
    }

    /* return the agp aperture address */ 
    *pAddress = (void *) (agpinfo.aper_base + (*Offset << PAGE_SHIFT));

    status = os_alloc_mem((VOID **)&data, sizeof(agp_priv_data));
    if (status != RM_OK)
    {
        nv_printf(NV_DBG_ERRORS, "NVRM: memory allocation failed\n");
        (*(agp_ops.unbind_memory))(ptr);
        goto fail;
    }

    data->ptr = ptr;
    data->num_pages = PageCount;
    data->offset = *Offset;

    *pPriv_data = data;
    
    nv_printf(NV_DBG_INFO, "NVRM: AGPGART: allocated %lu pages\n",
           (unsigned long) PageCount);
    return RM_OK;

fail:
    (*(agp_ops.free_memory))(ptr);
    *pAddress = (void*) 0;

    return RM_ERROR;
#endif /* AGPGART */
}

#if !defined(NVCPU_IA64) || (LINUX_VERSION_CODE < KERNEL_VERSION(2, 4, 9))

RM_STATUS 
KernMapAGPPages(
    VOID *pvma,
    VOID *priv_data
)
{
#ifndef AGPGART
    return RM_ERROR;
#else
    struct vm_area_struct *vma = (struct vm_area_struct *) pvma;
    agp_priv_data *agp_data = (agp_priv_data *) priv_data;
    unsigned long agp_addr;
    int err;

    if ((agp_data == NULL) || (vma == NULL))
        return RM_ERROR;

    agp_addr = agpinfo.aper_base + (agp_data->offset << PAGE_SHIFT);

    err = remap_page_range(vma->vm_start, (size_t) agp_addr, 
                           agp_data->num_pages << PAGE_SHIFT,
#if defined(NVCPU_IA64)
                           vma->vm_page_prot);
#else
                           PAGE_SHARED);
#endif
        
    if (err)
    {
        nv_printf(NV_DBG_ERRORS, "NVRM: AGPGART: unable to remap %lu pages\n",
            (unsigned long)agp_data->num_pages);
        (*(agp_ops.unbind_memory))(agp_data->ptr);
        goto fail;
    }
    
#if defined(NVCPU_IA64)
    vma->vm_pgoff = agp_addr >> PAGE_SHIFT;
    agp_data->vma = vma;
    agp_register_map(vma);
#endif

    return RM_OK;

fail:
    /* XXX what else needs to happen here? */
    return RM_ERROR;
#endif /* AGPGART */
}
#endif

#if !defined(KERNEL_2_2)

RM_STATUS 
KernMapAGPNopage(
    VOID *address,
    VOID *pvma,
    VOID *priv_data,
    VOID **ppage)
{
#ifndef AGPGART
    return RM_OK;
#else
    struct vm_area_struct *vma = (struct vm_area_struct *) pvma;
    agp_priv_data *agp_data = (agp_priv_data *) priv_data;
    agp_memory *agp_memory_ptr = agp_data->ptr;
    struct page *page_ptr;
    unsigned long phys_addr;
    int idx;

    idx = ((unsigned long)address - vma->vm_start) >> PAGE_SHIFT;
    /* There was an interface change durring the 2.4.18 series.  Earlier
      * 2.4.18 ports patches used the earlier interface but most shipping
      * kernels use the new interface.
      */
#if defined(NVCPU_IA64) && (LINUX_VERSION_CODE >= KERNEL_VERSION(2, 4, 18))
    phys_addr = agp_memory_ptr->memory[idx];
#else
    phys_addr = (agp_memory_ptr->memory[idx] & 0xffffff) << 12;
#endif
    page_ptr = virt_to_page(__va(phys_addr));
    get_page(page_ptr);
    *ppage = page_ptr;

#if defined(NVCPU_IA64)
    {
        /*
         * Reliability is dramatically improved when invalidating the cache 
         * lines prior to using write combined mappings.  The "fc" instruction
         * is specified to affect at least 32 bytes, thus the 32 byte increment
         * below.
         */
        int i;
        char *cp = (char *)__va(phys_addr);
        for (i = 0; i < PAGE_SIZE/32; i++)
        {
            ia64_fc (cp);
            cp += 32;
        }
        ia64_sync_i(); /* paranoia */
    }
#endif

    return RM_OK;
#endif
}

#endif /* !defined(KERNEL_2_2) */


RM_STATUS KernFreeAGPPages(
    nv_state_t *nv,
    VOID **pAddress,
    VOID *priv_data
)
{
#ifndef AGPGART
    return RM_OK;
#else
    agp_memory *ptr;
    agp_priv_data *data = priv_data;
    
#if defined(NVCPU_IA64) && (LINUX_VERSION_CODE < KERNEL_VERSION(2, 4, 9))
    agp_unregister_map(data->vma);
#endif

    rm_free_agp_bitmap(nv, data->num_pages, data->offset);

    ptr = data->ptr;
    os_free_mem(data);

    if (!(ptr == NULL))
    {
        size_t pages = ptr->page_count;

        (*(agp_ops.unbind_memory))(ptr);
        (*(agp_ops.free_memory))(ptr);

        nv_printf(NV_DBG_INFO, "NVRM: AGPGART: freed %ld pages\n",
            (unsigned long)pages);
        return RM_OK;
    }
    return RM_ERROR;
#endif
}


