#!/bin/bash
#
# Name: master.sh
# Copyright: (c)2004 Hewlett-Packard
#
# Description: master build script for 2.6 drivers
#
# Chad Dupuis 05/26/04 Initial Development
# Chad Dupuis 06/03/04 Added ability of accept command line
#                      parameters to select which kernel and
#                      source directory we are going to use

#
# function defs
#

# list the source and module directories that are available
# on this system

list_sm_dir () {
 # we don't want to list kernel source directories with RHEL 4 since
 # they do not come preinstalled and the -s argument does not need to
 # be passed to build_driver.rh

 if ! test -f /etc/redhat-release
 then
	echo "Kernel source directories available:"
	echo "" 
	ls /usr/src | grep linux | grep -v obj | paste -
	echo ""
 fi
 echo "Kernel module directories available:"
 echo ""
 ls /lib/modules | grep "2.6" | paste -
}

# prints command line help

print_help () {
 echo "Usage: $0 [-lh -s source_directory] -m module_directory"
 echo ""
 echo "-l: lists the available kernel source and module directories"
 echo "-s: kernel source directory to use; this switch is not used with"
 echo "    Red Hat Enteprise Linux 4"
 echo "-m: kernel module directory to use"
 echo "-h: prints help"
 echo ""
 echo "example: SLES: $0 -s linux-2.4.21-9.EL -m 2.4.21-9.ELsmp"
 echo "         RHEL: $0 -m 2.4.21-9.ELsmp"
}

#
# Script Main
#

# only process arguments if we have any

if [ $# -gt 0 ]
then
 USEARGS=1

 # command line flag

 LISTFLAG=0 
 SOURCEFLAG=0
 MODULEFLAG=0
 HELPFLAG=0

 # parse command line into arguments

 getopt lm:s:h $* 1>/dev/null 2>/dev/null

 # check result of parsing

 if [ $? != 0 ]
 then
 	echo "Bad argument or missing argument"
 	exit 1
 fi

 set -- `getopt lm:s:h $*`

 while [ $1 != -- ]
 do
 	case $1 in
		-l) LISTFLAG=1;;
		-m) MODULEFLAG=1
		    MODULEDIR=$2
		    shift;;
		-s) SOURCEFLAG=1
		    SOURCEDIR=$2
		    shift;;
		-h) HELPFLAG=1;; 
		*) echo "$1 is an illegal argument"
		   exit 1;;
 	esac
 	shift   # next flag
 done

 shift   # skip --

 if [ $HELPFLAG -eq 1 ]
 then
 	print_help
 	exit 0
 elif [ $LISTFLAG -eq 1 ]
 then
 	list_sm_dir
 	exit 0
 fi
else
 USEARGS=0 # don't use any arguments to scripts
fi

EXEDIR="/opt/hp/src/hp_qla2x00src"
cd $EXEDIR

if [ $? -ne 0 ]
then
 exit 1
fi

# build driver from source code

if test -f /etc/redhat-release
then
 # we are a redhat distribution, run build_driver.rh

 if [ $USEARGS -eq 0 ]
 then
 	./build_driver.rh
 	
	if [ $? -ne 0 ]
	then
		echo ""
		echo "Driver build unsuccessful"
		exit 1
	fi
 else
 	# check to make sure that MODULEDIR is not blank 

        if [ "$MODULEDIR" = "" ]
        then
                echo "No directory supplied with the -m flag"
                exit 1
        fi

        # make sure that the module directory exists

        if test ! -d /lib/modules/${MODULEDIR}
        then
                echo "/lib/modules/${MODULEDIR} does not exist"
                exit 1
        fi

	./build_driver.rh -m $MODULEDIR
 	
	if [ $? -ne 0 ]
        then
                echo "Driver build unsuccessful"
                exit 1
        fi
 fi
elif test -f /etc/SuSE-release
then
 # we are a suse distribution, run build_driver.suse

 if [ $USEARGS -eq 0 ]
 then
        ./build_driver.suse
 	
	if [ $? -ne 0 ]
        then
                echo ""
                echo "Driver build unsuccessful"
                exit 1
        fi

 else
	# check to make sure that both MODULEDIR and SOURCEDIR
 	# are not blank

 	if [ "$MODULEDIR" = "" ]
 	then
        	echo "No directory supplied with the -m flag"
        	exit 1
	fi

	if [ "$SOURCEDIR" = "" ]
 	then
        	echo "No directory supplied with the -s flag"
        	exit 1
 	fi

 	# make sure that kernel source and module directories exist

 	if test ! -d /lib/modules/${MODULEDIR}
 	then
        	echo "/lib/modules/${MODULEDIR} does not exist"
        	exit 1
 	fi

 	if test ! -d /usr/src/${SOURCEDIR}
 	then
        	echo "/usr/src/${SOURCEDIR} doesn't not exist"
        	exit 1
 	fi

        ./build_driver.suse -m $MODULEDIR -s $SOURCEDIR
 
	if [ $? -ne 0 ]
        then
                echo ""
                echo "Driver build unsuccessful"
                exit 1
        fi
 fi
else
 echo "Unknown distrubtion, driver build aborted"
 exit 1
fi
 
# edit necessary configuration files

echo ""
./edit_conf

# create new ramdisk image

echo ""

if [ $USEARGS -eq 0 ]
then
 ./make_initrd
else
 ./make_initrd $MODULEDIR
fi

