/*
 * Copyright 2001 Hewlett-Packard Development Company, L.P.
 * 
 * Compaq and the Compaq logo are trademarks of Compaq Information
 * Technologies Group, L.P. in the U.S. and/or other countries.
 *
 * Confidential computer software. Valid license from Compaq required
 * for possession, use or copying. Consistent with FAR 12.211 and 12.212,
 * Commerical Computer Software, Computer Software Documentation, and
 * Technical Data for Commerical Items are licensed to the
 * U.S. Government under vendor's standard commercial license.
 */

/*
 *   Filename: cmhp.c 
 *   Provides the interfaces for all the standard driver to kernel
 */

#include <linux/version.h>
#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/slab.h>
#include <linux/vmalloc.h>
#include <linux/fs.h>
#include <linux/errno.h>
#include <linux/types.h>
#include <linux/pci.h>
#include <linux/proc_fs.h>
#include <linux/fcntl.h>
#include <linux/smp_lock.h>
#include <linux/spinlock.h>
#include <linux/interrupt.h>
#include <linux/devfs_fs.h>
#include <linux/devfs_fs_kernel.h>

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0) 
#include <asm/softirq.h>
#endif
#include <asm/system.h>
#include <asm/delay.h>
#include <asm/io.h>
#include <asm/uaccess.h>
#include <asm/processor.h>

#include "hptypes.h"
#include "hpproto.h"
#include "unixproto.h"

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,16)
#	define inter_module_get(s) symbol_get(s)
#	define inter_module_put(s) symbol_put(s)
#endif

/* PROTOTYPES */
int _cmhp_ioctl(struct inode *i, struct file *f, unsigned int cmd, 
                unsigned long arg);
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0)
void _cmhp_interrupt(int irq, void *dev_id, struct pt_regs *regs);
#else
irqreturn_t _cmhp_interrupt(int irq, void *dev_id, struct pt_regs *regs);
#endif

VOID cmhpInit_inj_err_mutex(VOID);

/*
 * Module information
 */
#define MODULE_NAME         "cmhp"
//#define MODULE_VERSION      "0.1"

MODULE_AUTHOR("Compaq Technologies Group, L.P.");

#ifdef CASM_DESCRIPTION
MODULE_DESCRIPTION(CASM_DESCRIPTION);
//#else
//MODULE_DESCRIPTION(MODULE_NAME " v." MODULE_VERSION 
//      ".\nThis module provides Compaq Memory Hot Plug Controller Driver support.");
#endif

#if ( LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,9))
MODULE_LICENSE("Proprietary - Compaq Technologies Group, L.P.");
#endif /* LINUX_VERSION_CODE */

/* Major device number for /dev/cmhp device */
static uint cmhp_device_major;

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0)
/* Devfs handle for the CMHP device */
static devfs_handle_t   cmhp_devfs_handle;
#endif

/* File operations permitted on this device */
static struct file_operations cmhp_fops = {   
   owner:   THIS_MODULE,
   ioctl:   _cmhp_ioctl,
};

/* Proctect CMHP global variables */
static   spinlock_t      cmhp_var_slock = SPIN_LOCK_UNLOCKED;
static   spinlock_t      cmhp_iml_slock;

/*
 * Convert the unix layer error codes to this
 * kernels error codes
 */
ULONG cmhpErrUnixtoKern(ULONG ulUnixErr)
{
   ULONG retval = 0;
   
   // convert the cmhp return code to the kernel equiv.
   switch(retval)
   {
      case CMHP_INVAL:
         retval = -EINVAL;
         break;
      case CMHP_NOMEM:
	 retval = -ENOMEM;
	 break;
      case CMHP_FAULT:
	 retval = -EFAULT;
	 break;
      case CMHP_EAGAIN:
	 retval = -EAGAIN;
	 break;
      case CMHP_EIO:
	 retval = -EIO;
	 break;
      default:
         break;
   }

   return(retval);
}

/*
 * Provide ioctl interfase
 */
int _cmhp_ioctl(struct inode *i, struct file *f, unsigned int cmd, 
                unsigned long arg)
{
   int retval = 0;

   //TODO: hack to make this happen
   if(cmd == 0x33) //CMHP_GET_NEXT_IML)
   {
      cmhpIoctlGetIMLRequest((PVOID)arg);
      return 0;
   }
   else
   {
      retval = cmhp_ioctl(cmd, arg);
   return cmhpErrUnixtoKern(retval);
   }

}

/********************************************************************
 *              Event Processing Thread Control                     *
 ********************************************************************/


/* Mutex for syncing events */
static DECLARE_MUTEX_LOCKED(cmhp_unload_sync);
static DECLARE_MUTEX_LOCKED(intr_sem);

struct task_struct *cmhpThreadTask = NULL;

/*
 * kcmhpd daemon that handles all events.
 */
static int cmhpThreadHelper(void *arg)
{

   cmhpThreadTask = current;

   /* Make this thread a daemon */
   lock_kernel();

   sprintf(current->comm, cmhp_thread_name);

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0) 
   daemonize();
#else
   daemonize("cmhpd");
#endif

   siginitsetinv(&current->blocked, sigmask(SIGKILL));

   unlock_kernel();

   cmhpDebug(("%s: up & running. Task id %d \n", cmhp_thread_name, current->pid));

   /* Indicate CMHP driver that we are prepared */
   up(arg);

   // start the thread
   cmhpInterruptProcessingThread(0);

   //make sure we don't get killed again
   cmhpThreadTask = NULL;

   /* Indicate rmmod that we are done */
   up(&cmhp_unload_sync);
   
   return(0);
}

/*****************************************************************
 * Start the kcmhpd event thread.
 *****************************************************************/
LONG cmhp_start_thread(VOID)
{
   /* Sync between kcmhpd and insmod/rmmod */
   DECLARE_MUTEX_LOCKED(cmhp_thread_mutex);

   // Start the kernal thead
   kernel_thread(cmhpThreadHelper, &cmhp_thread_mutex, 0); 

   /* Wait for the thread to start */
   down(&cmhp_thread_mutex);

   cmhpDebug(("%s: Event thread %s started successfully. \n",
         cmhp_modname, cmhp_thread_name));
         
   return(0);
}
/*****************************************************************
 * Kill kcmhpd and prevent queuing of further events.
 *****************************************************************/
static VOID cmhp_end_thread(VOID)
{
   if(cmhpThreadTask)
   {
      send_sig(SIGKILL, cmhpThreadTask, 1);
   
      /* Wait for the cmhpd event thread to exit */
      down(&cmhp_unload_sync);
   }

   cmhpDebug(( "%s: kernel thread %s killed. \n", cmhp_modname, cmhp_thread_name)); 

   return;
}

/**************************************************************** 
 * Mutual exclusion for CMHP global variables 
 ****************************************************************/
void cmhp_var_lock(void)
{
   spin_lock(&cmhp_var_slock);
   return;
}

void cmhp_var_unlock(void)
{
   spin_unlock(&cmhp_var_slock);
   return;
}

void cmhp_iml_lock(void)
{
   spin_lock(&cmhp_iml_slock);
   return;
}

void cmhp_iml_unlock(void)
{
   spin_unlock(&cmhp_iml_slock);
   return;
}


/****************************************************************
 * Register the cmhp device driver.
 * Create devfs entry for cmhp device. If devfs is mounted, 
 * the device appears under the devfs tree. If devfs is not 
 * mounted at boot time, * the /dev/cmhp entry needs to be 
 * created from userland.
 ****************************************************************/
static int cmhp_register_device(void)
{
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0) 
   SET_MODULE_OWNER(&cmhp_fops);
#endif

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0) 
   cmhp_device_major = devfs_register_chrdev(0, cmhp_modname, &cmhp_fops); 
#else
   cmhp_device_major = register_chrdev(0, cmhp_modname, &cmhp_fops); 
#endif

   if (cmhp_device_major < 0) 
   {
      cmhp_device_major = 0;
      return(-EINVAL);
   }

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0) 
   cmhp_devfs_handle = devfs_register(NULL, cmhp_modname, 
               DEVFS_FL_DEFAULT|DEVFS_FL_AUTO_DEVNUM, 
               cmhp_device_major, 0,  
               S_IFCHR | S_IRUSR | S_IWUSR | 
               S_IRGRP | S_IWGRP, 
               &cmhp_fops, NULL);

   /* Devfs mounted at boot time */
   if (cmhp_devfs_handle) {
         devfs_get_maj_min(cmhp_devfs_handle, &cmhp_device_major, NULL);
   }
#endif

   /* We should have a major number by now */
   if (cmhp_device_major == 0) {
      printk(KERN_ERR "%s: Unable to register cmhp device.\n", cmhp_modname);
      return(-EINVAL);
   }

   cmhpDebug(("%s: CMHP device registered, major number: %d, device : 0x%x \n", 
         cmhp_modname, cmhp_device_major, MKDEV(cmhp_device_major, 0)));

   return(0);
}

/****************************************************************
 * Remove devfs entry for this device.
 ****************************************************************/
static int cmhp_unregister_device(void)
{
   if (cmhp_device_major) {
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0)
      if (devfs_unregister_chrdev(cmhp_device_major, cmhp_modname)) {
            printk(KERN_WARNING
               "%s: Error unregistering cmhp device. \n", cmhp_modname);
      }
#else
      if (unregister_chrdev(cmhp_device_major, cmhp_modname)) {
            printk(KERN_WARNING
               "%s: Error unregistering cmhp device. \n", cmhp_modname);
      }
#endif
      cmhp_device_major = 0;
   }

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0) 
   if (cmhp_devfs_handle) {
      devfs_unregister(cmhp_devfs_handle);
      cmhp_devfs_handle = NULL;
   }
#endif

   return(0);
}

/****************************************************************
 * Register the name of this module so we can tell if there is
 * already another instance of it running
 ****************************************************************/
static int cmhp_register_module(void)
{
   cmhp_var_lock();

   if (inter_module_get(cmhp_modname)) 
   {
      inter_module_put(cmhp_modname);
      cmhp_var_unlock();
      printk(KERN_ERR
            "%s: Found another instance of %s module.\n",
            cmhp_modname, cmhp_modname);
      return(1);
   }

   inter_module_register(cmhp_modname, THIS_MODULE, (void *)cmhp_modname);

   cmhp_var_unlock();
   return(0);
}

/****************************************************************
 * Unregister the name so that we can later be loaded again
 ****************************************************************/
static void cmhp_unregister_module(void)
{
   inter_module_unregister(cmhp_modname);
   return;
}

/****************************************************************
 * Ask the kernel to let us service this interrupt
 ****************************************************************/
VOID cmhpRegisterIRQ(ULONG irq, PVOID arg)
{
  if(request_irq(irq, _cmhp_interrupt, SA_INTERRUPT | SA_SHIRQ, cmhp_modname, arg)) 
  {
     printk(KERN_WARNING "%s: request irq %d failed.\n", cmhp_modname, (int)irq);
  }
}

/****************************************************************
 *  Tell the kernel that we don't want to service this 
 *  interrupt any more
 ****************************************************************/
VOID cmhpUnRegisterIRQ(ULONG irq, PVOID arg)
{
   free_irq(irq, arg);
}



/**************************************************************
 * Misc counters and Semaphores
 *************************************************************/
/* Count for number of error injections */
static atomic_t inj_err_count[5 /*MAX_MRM*/];
/* Mutual exclusion for injecting errors on dimms */
static struct semaphore inj_err_mutex[5 /*MAX_MRM*/];


VOID cmhpInit_inj_err_mutex()
{
   int i;
   /* Initialize the inject error semaphores */
   for (i = 0; i < 5 /*MAX_MRM*/; i++) 
   {
      init_MUTEX(&inj_err_mutex[i]);
   }
}

VOID cmph_inc_inj_err_count(ULONG mrm)
{
   atomic_inc(&inj_err_count[mrm]);
}
ULONG cmph_read_inj_err_count(ULONG mrm)
{
   atomic_read(&inj_err_count[mrm]);

   return 0;
}
VOID cmph_dec_inj_err_count(ULONG mrm)
{
   atomic_dec(&inj_err_count[mrm]);
}



/**************************************************************
 * Memory Access
 *************************************************************/
PVOID cmhp_get_free_DMA_page()
{
  return (PVOID)__get_free_page(GFP_DMA); 
}

ULONG cmhp_copy_from_user(PVOID dest, PVOID src, ULONG size)
{
   return copy_from_user(dest, src, size);
}
ULONG cmhp_copy_to_user(PVOID dest, PVOID src, ULONG size)
{
   return copy_to_user(dest, src, size);
}

/***************************************************************
 * cmhpMapPhys                                                 *
 *                                                             *
 * This function accepts a 32 bit physical address             *
 * and returns a virtual address.  The virtual address         *
 * will be in the Uncacheable region of the Kernel.            *
 * This function is mainly used when mapping in                *
 * memoey mapped IO.                                           *
 * return NULL if something goes wrong                         *
 *                                                             *
 ***************************************************************/
PVOID cmhpMapPhys(ULONG ulPhysAddr, ULONG ulLength)
{

   PVOID VirtAddr = NULL;
   
   if(ulPhysAddr)
      VirtAddr = ioremap_nocache(ulPhysAddr, ulLength);
  
   cmhpDebug(("\ncmhpMapPhys: Phys 0x%x -> Virt 0x%x Len 0x%x\n", ulPhysAddr, VirtAddr, ulLength));
   return (VirtAddr);
}

/***************************************************************
 * cmhpUnMapPhys                                               *
 *                                                             *
 * This function unmaps the Virtual address from above         *
 *                                                             *
 ***************************************************************/
VOID cmhpUnMapPhys(PVOID VirtAddr, ULONG size)
{
   if(VirtAddr)
      iounmap((PVOID)VirtAddr);
}

/***************************************************************
 * cmhpMalloc                                                   *
 *                                                             *
 * This function allocates a buffer of size ulSize             *
 * return NULL if something goes wrong                         *
 *                                                             *
 ***************************************************************/
PVOID cmhpMalloc(ULONG ulSize)
{
   PUCHAR p=(PUCHAR)kmalloc(ulSize, GFP_KERNEL);
   ULONG i;

   //go ahead and zero out the memory
   if(p)
   {
      for(i = 0; i < ulSize; i++)
      {
         p[i] = 0;
      }
   }
   
   return (PVOID)p;
}  /* end cmhpMalloc */

/***************************************************************
 * cmphFree                                                    *
 *                                                             *
 * This function frees the memory allocated by cmhpMalloc      *
 ***************************************************************/
VOID cmhpFree(PVOID pVAddr, ULONG ulSize)
{
   if(pVAddr)
      kfree (pVAddr);
}  /* end cmhpMalloc */

ULONG cmhpGetPageSize()
{
   return PAGE_SIZE;
}

/********************************************************
 * PCI interfaces
 ********************************************************/
PPCISTRUCTLIST cmhpFindPCIDevice(ULONG ulVendorID, ULONG ulDeviceID)
{
   struct pci_dev *pdev; 
   PPCISTRUCTLIST pPciHead = NULL;
   PPCISTRUCTLIST pPciTail = NULL;
   PPCISTRUCTLIST pPciTmp = NULL;


   for(pdev = pci_find_device(ulVendorID, ulDeviceID, NULL);
       pdev != NULL; 
       pdev = pci_find_device(ulVendorID, ulDeviceID, pdev))
   {
      pPciTmp = cmhpMalloc(sizeof(PCISTRUCTLIST));
      if(pPciTmp != NULL)
      {
         pPciTmp->pNext=NULL;
	 pPciTmp->sysdata = pdev;          // OS kernel data
         pPciTmp->ulMemAddr=pci_resource_start(pdev, 0); 
	 pPciTmp->ulSize = pci_resource_end(pdev,0) - pPciTmp->ulMemAddr;
	 pPciTmp->pvVirtAddr = cmhpMapPhys(pPciTmp->ulMemAddr, pPciTmp->ulSize);
         pPciTmp->ucBusNumber=pdev->bus->number;
         pPciTmp->ucDeviceNumber=PCI_SLOT(pdev->devfn);
         pPciTmp->ucFunctionNumber=PCI_FUNC(pdev->devfn);
         pPciTmp->ucIrq=pdev->irq;
	 cmhpRegisterIRQ(pdev->irq, pPciTmp->pvVirtAddr);
	  
         //Now add this on to the list
	 if(pPciHead == NULL)
	 {
            //this is the first one found
            pPciHead = pPciTmp;
	    pPciTail = pPciTmp;
	 }
	 else
	 {
            //put it at the end of the list
            pPciTail->pNext = pPciTmp;
	    pPciTail = pPciTmp;
	 }
      }
   }//end for

   cmhpDebug(("MemAddr = 0x%x\n", pPciTmp->ulMemAddr));
   return pPciHead;

}

/***************************************************************************
 * Undo the stuff we did in cmhpFindPCIDevice
 *****************************************************************************/
BOOLEAN cmhpFreeDeviceData(PPCISTRUCTLIST pPciList)
{
   if(pPciList)
   {
      // first free the next one
      cmhpFreeDeviceData(pPciList->pNext);

      // stop servicing this interrupt
      cmhpUnRegisterIRQ(pPciList->ucIrq, pPciList->pvVirtAddr);

      // Free the memory mapping
      cmhpUnMapPhys(pPciList->pvVirtAddr, pPciList->ulSize);

      // Then free this one
      cmhpFree(pPciList, sizeof(PCISTRUCTLIST));
   }

   return TRUE;
}

VOID cmhpNotifyHWNewTopOfMemory(ULONG ulVendorID, ULONG ulDeviceID, ULONG addr, WORD val)
{
   struct pci_dev *pdev; 

   for(pdev = pci_find_device(ulVendorID, ulDeviceID, NULL);
       pdev != NULL; 
       pdev = pci_find_device(ulVendorID, ulDeviceID, pdev))
   {
      if((pci_write_config_word(pdev, addr, val)) != 0) 
      {
         cmhpDebug((KERN_WARNING "%s: pci_write_config_dword failed!", 
                cmhp_modname));
      }
   }
}

/*****************************************************************
 * ISR
 *****************************************************************/
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0)
void _cmhp_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
   cmhpDebug(("Got Interrupt %d\n", irq));
   cmhp_interrupt(irq);
}
#else
irqreturn_t _cmhp_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
   cmhpDebug(("Got Interrupt %d\n", irq));
   return cmhp_interrupt(irq);
}
#endif


ULONG cmhpBlockIntrThread()
{
   ULONG ret = 0;

   down_interruptible(&intr_sem);

   if(signal_pending(current))
      ret = UNLOADING;

   return ret;
}

void cmhpWakeupIntrProcessThread()
{
   up(&intr_sem);
}

/**************
 * Pause      *
 **************/
ULONG cmhpUsecWait(ULONG microsecs)
{
   unsigned long usecs_per_jiffie = ( 1000000 / HZ );

/* Added a function to allow a spin delay in excess of 1 clock tick.  If we use
 * udelay to delay for more than a few milliseconds, there is a danger of overflow
 * on some of the faster processors.  Therfore, we need to use the time_after macro
 * to assure this doesn't happen.  We will allow other interrupts, and processes to
 * proceed during this delay, and prevent the spin from totally occupying the CPU
 * by calling the scheduler. - jgl 11/00
 */

   if( microsecs <= usecs_per_jiffie )
   {
      udelay(microsecs);
   }
   else 
   {
      signed long interval = ( microsecs / usecs_per_jiffie );
      set_current_state(TASK_INTERRUPTIBLE);
      schedule_timeout(interval);
   }
   return( microsecs );
}

void cmhpPrint(ULONG ulSev, PCHAR str, ...)
{
   va_list args;
   char buf[1024];
   int i;

   va_start(args, str);
   i = vsprintf(buf, str, args); /* hopefully i < sizeof(buf)-4 */
   va_end(args);

   switch(ulSev)
   {
      case CMHP_INFO:
      {
         printk(KERN_INFO "%s", buf);
         break;
      }
      case CMHP_WARNING:
      {
         printk(KERN_WARNING "%s", buf);
         break;
      }
      case CMHP_ERR:
      {
         printk(KERN_ERR "%s", buf);
         break;
      }
      case CMHP_DBG:
      {
         printk(KERN_DEBUG "%s", buf);
         break;
      }
      default:
      {
         printk("%s", buf);
      }
   }//switch
}

/*
 * Get the seconds since EPOC
 */
ULONG cmhpGetCurrentTimeInSec()
{
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0)
    return CURRENT_TIME;
#else 
    return CURRENT_TIME.tv_sec;
#endif
}


LONG cmhpHotAdd(ULONG ulStart, ULONG ulSize, UCHAR flags)
{
   //lets only display the message once
   static BOOLEAN bDisplayed = FALSE;
   ULONG64 byte_start = (ULONG64)ulStart << 20; 
   ULONG64 byte_size = (ULONG64)ulSize << 20;


   /* Interface to the HAM Module */
   #define	HAM_HOT_ADD	"ham_hot_add"
   typedef ULONG (*ham_interface_t)(ULONG64, ULONG64, UCHAR);
   ham_interface_t ham_hot_add_func = (ham_interface_t)inter_module_get(HAM_HOT_ADD);
   if (ham_hot_add_func) 
   {
      (*ham_hot_add_func)(byte_start, byte_size, flags);
      inter_module_put(HAM_HOT_ADD);
   }
   else 
   {
      if(!bDisplayed)
      {
         printk(KERN_NOTICE "%s: %s not registered.  HAM module may not be loaded. Hot-add event ignored.\n", cmhp_thread_name, HAM_HOT_ADD);
         bDisplayed = TRUE;
      }
   }

   return 0;
}



#ifdef PRE_SS711
ULONG cmhpEventHandler(ULONG ulAction, ULONG ulMatchcode, ULONG ulEventNumber, PVOID pEvt)
{
   //lets only display the message once
   static BOOLEAN bDisplayed = FALSE;

   #define EVENT_LOG_HANDLER "cpqwEventHandler"
   typedef ULONG (*iml_interface_t)(ULONG, ULONG, ULONG, PVOID);
   iml_interface_t cpqwEventHandler = (iml_interface_t)inter_module_get(EVENT_LOG_HANDLER);

   if (cpqwEventHandler) 
   {
      (*cpqwEventHandler)(ulAction, ulMatchcode, ulEventNumber, pEvt);
      inter_module_put(EVENT_LOG_HANDLER);
   }
   else
   {
      if(!bDisplayed)
      {
         printk(KERN_NOTICE "%s: NOTICE: %s not registered. cpqevt module may not be loaded. Events will be not be logged.\n", cmhp_modname, EVENT_LOG_HANDLER);
         bDisplayed = TRUE;
      }
   }
   return 0;
}
#endif

/********************************************************************
 * LOAD and UNLOAD routines
 ********************************************************************/

// Defines for the start up and exit routines to keep track of 
// what was done and what needs to be undone
#define REG_MOD    0x1
#define DEV_FND    0x2
#define REG_DEV    0x4
#define INIT_FUNC  0x8
#define THR_START  0xa
#define FULLY_LOADED (REG_MOD | DEV_FND | REG_DEV | INIT_FUNC | THR_START)

// Our Device list structure
static PPCISTRUCTLIST pDevices = NULL;

/****************************************************************
 * undo what was done
 ****************************************************************/
static void exit_module(ULONG inited)
{
   if(inited & THR_START)
   {
      cmhp_end_thread();
   }

   if(inited & REG_DEV)
   {
      cmhp_unregister_device();
   }

   if(inited & INIT_FUNC)
   {
      cmhp_function_exit();
   }

   if(inited & DEV_FND)
   {
      cmhpFreeDeviceData(pDevices);
   }

   if(inited & REG_MOD)
   {
      cmhp_unregister_module();
   }
}

/****************************************************************
 * Kernels unload entry to the Driver
 ****************************************************************/
static void __exit cmhp_mod_exit(void)
{

   exit_module(FULLY_LOADED);

   printk(KERN_INFO "%s: Module unloaded.\n", cmhp_modname);

   return;
}


/****************************************************************
 * Kernels load entry  to the Driver                         
 ****************************************************************/
static int __init cmhp_mod_init(void)
{
   LONG rc = 0;
   ULONG inited = 0

   cmhpDebug(("%s: Init Compaq Memory Controller\n", cmhp_modname));

   // make sure we are the only ones
   if(!cmhp_register_module())
   {
      inited = REG_MOD;

      // Get our device Data
      pDevices = cmhpFindPCIDevice(PCI_VENDOR_ID_COMPAQ, PCI_DEVICE_ID_COMPAQ_MEMCNTL);
      if(pDevices)
      {
         inited |= DEV_FND;
   
	 // we need the lock before we register the device node
         spin_lock_init(&cmhp_var_slock);
         spin_lock_init(&cmhp_iml_slock);
   
         if(!cmhp_register_device());
         {
            inited |= REG_DEV;
   
            if(!cmhp_function_init(pDevices)) 
            {
               inited |= INIT_FUNC;
               if (!cmhp_start_thread()) 
	       {
                  inited |= THR_START;
	       }
	    }
	 }
      }
   }
   
   if (inited == FULLY_LOADED) 
   {
      printk(KERN_INFO "%s: hp Proliant Memory Hot Plug Driver (rev %s)\n", cmhp_modname, EFS_VER);
   }
   else
   {
      // something went wrong in the init so lets bail
      rc = -1;
      exit_module(inited);
      printk(KERN_ERR "%s: Unable to load module. code = 0x%lx\n", cmhp_modname, inited);
   }

   return (rc);
}


ULONG cmhpAtomicInc(volatile PULONG int_ptr)
{
   if(int_ptr)
      atomic_inc((atomic_t *)int_ptr);

   return int_ptr ? *int_ptr : 0;
}

ULONG cmhpAtomicDec(volatile PULONG int_ptr)
{
   if(int_ptr)
      atomic_dec((atomic_t *)int_ptr);

   return int_ptr ? *int_ptr : 0;
}

module_init(cmhp_mod_init);
module_exit(cmhp_mod_exit);
