/*
 ***************************************************************************
 * (C) Copyright Compaq Information Technologies Group, L.P. 2001, 2002    *
 ***************************************************************************
 *
 *  Title:    OS Utilities
 *
 *  Module:   cridw_linux.c
 *
 *  Version:  1.0
 *
 *  Date:     12/15/2001
 *
 *  Author:   Patrick Schoeller
 *
 ***************************************************************************
 * Description:
 * 
 * The purpose of this file is to isolate OS specific details from the
 * primary code path, thus enabling an 80% common code base.  Examples of
 * functions to include in here are Physical Memory Mapping, Kernel Memory
 * allocation, device discovery, OS specific entry points, etc.
 *
 ***************************************************************************/
#include "convert.h"
#include "cridc.h"
#include "cridw_linux.h"
#ifdef CRID_X86_64
#include <asm/ioctl32.h>
#endif

#if LINUX_VERSION_CODE >= 0x020600
MODULE_LICENSE("Proprietary");
#endif

#ifdef CPQRID_DESCRIPTION
MODULE_DESCRIPTION(CPQRID_DESCRIPTION);
#endif

EXPORT_NO_SYMBOLS;

PCRIDDATA gpCridInstanceData = NULL;

//behavioral flag bit bucket
//FIXME: should be 0
static unsigned int flags = CRID_DEBUG_ALL;
//static unsigned int flags = CRID_DEBUG_ALL | CRID_NO_SERVER_NAME | CRID_NO_BTRY_STATUS;

//Helper function for init and cleanup
static int crid_init_one(void);
static int crid_remove_one(void);

//standard entry points
static int crid_ioctl(struct inode* inode, struct file* fp, unsigned int cmd, unsigned long arg);
static int crid_open(struct inode* inode, struct file* filep);
static int crid_close(struct inode* inode, struct file* filep);
#ifdef CRID_X86_64
int crid_ioctl32_handler(unsigned int fd, unsigned int cmd, unsigned long arg, struct file *file);
#endif

//interrupt handler
#if LINUX_VERSION_CODE < 0x020600
static void crid_intr(int irq, void* instance, struct pt_regs* regs);
#else
static irqreturn_t crid_intr(int irq, void* instance, struct pt_regs* regs);
#endif


static int crid_major = 0;

static struct file_operations crid_ops = {	
	owner:		THIS_MODULE,
	ioctl:		crid_ioctl,     /* ioctl */		
	open:		crid_open,	/* open */		
	release: 	crid_close,	/* close */		
};						

extern struct proc_dir_entry* crid_entry_print_crid;
extern int crid_read_print_crid(char *Buf, char **start, off_t off, int count, int *eof, void *data, PCRIDDATA pInstance);

static int crid_read_print(char *Buf, char **start, off_t off, int count, int *eof, void *data)
{
	return crid_read_print_crid(Buf, start, off, count, eof, data, gpCridInstanceData);
}
/*********************************************************************

@func cridw_translate_error

Routine Description:
This translates homespun error code into Linux error codes 


Return Value: Linux error code


***********************************************************************/
static int cridw_translate_error(int our_error) 
{
	static int error_code[] = {
		0,
		EFAULT,
		EBUSY,
		EINVAL,
		ENOSPC,
		ETIME,
		ENODEV,
		EIO,
		EPERM,
		ENXIO,
		EINTR
	};

	if ((our_error < 0) || (our_error > sizeof(error_code)/sizeof(int))) return EFAULT;
	return error_code[our_error];
}

/*********************************************************************

@func cridw_get_server_name 

Routine Description:
This loads the server name into the string location provided.


Return Value:


***********************************************************************/

void cridw_get_server_name(char *s1, unsigned int n, PCRIDDATA dummy)
{

   strncpy(s1, system_utsname.nodename, n);

} /* end cridw_get_server_name */


/*********************************************************************

@func cridw_call_idistributed

Routine Description:
Turns irq on or off


Return Value:


***********************************************************************/

void cridw_call_idistributed(ULONG irq_on, PCRIDDATA pInstance)
{

	int temp=0;

	if (irq_on) {
#if LINUX_VERSION_CODE < 0x020600
		temp = request_irq(pInstance->ulIRQ, (void (*)(int, void *, struct pt_regs *)) &crid_intr, SA_SHIRQ, D_NAME, pInstance);
#else
		temp = request_irq(pInstance->ulIRQ, (irqreturn_t (*)(int, void *, struct pt_regs *)) &crid_intr, SA_SHIRQ, D_NAME, pInstance);
#endif
	} else {
		free_irq(pInstance->ulIRQ, pInstance);
	}

} /* end cridw_call_idistributed */


/*********************************************************************

@func  init_module

Routine Description:
This is a Linux specific routine which is called when the module
is loaded by the Linux operating system.


Return Value:


***********************************************************************/
int init_module(void)
{

  	int err_type; 
#ifdef CRID_X86_64
	int ret;

	ret = register_ioctl32_conversion((('Z'<<8)|0), crid_ioctl32_handler);
	ret |= register_ioctl32_conversion((('Z'<<8)|1), crid_ioctl32_handler);
	ret |= register_ioctl32_conversion((('Z'<<8)|2), crid_ioctl32_handler);
	ret |= register_ioctl32_conversion((('Z'<<8)|3), crid_ioctl32_handler);
	ret |= register_ioctl32_conversion((('Z'<<8)|4), crid_ioctl32_handler);
	ret |= register_ioctl32_conversion((('Z'<<8)|5), crid_ioctl32_handler);
	ret |= register_ioctl32_conversion((('Z'<<8)|6), crid_ioctl32_handler);
	ret |= register_ioctl32_conversion((('Z'<<8)|7), crid_ioctl32_handler);
	ret |= register_ioctl32_conversion((('Z'<<8)|8), crid_ioctl32_handler);
	ret |= register_ioctl32_conversion((('Z'<<8)|9), crid_ioctl32_handler);
	ret |= register_ioctl32_conversion((('Z'<<8)|0xA), crid_ioctl32_handler);
	ret |= register_ioctl32_conversion((('Z'<<8)|0xC), crid_ioctl32_handler);
	ret |= register_ioctl32_conversion((('Z'<<8)|0xD), crid_ioctl32_handler);
	if (ret) {
		printk(KERN_WARNING "crid:  Error registering ioctl32\n");
	}
#endif
#ifdef ROMDRV_ENABLED
#define SIGNATURE (('$' << 0) | ('U' << 8) | ('L' << 16) | ('D' << 24))
	err_type = romdrv_init(SIGNATURE, 0, NULL, 0, NULL);
	if (err_type != SUCCESS) {
      		printk(KERN_WARNING "crid:  Error Initializing (%d)\n", err_type);
		return err_type;
	}
#endif

	err_type = crid_init_one();
	if (err_type != SUCCESS) {
      		printk(KERN_WARNING "crid:  Error Initializing (%d)\n", err_type);
		return err_type;
	}

	crid_major = register_chrdev(CRID_MAJOR, "cpqrid", &crid_ops);
   	if (crid_major < 0) {
      		printk(KERN_WARNING "crid:  Not able to get Major Number.  Error (%d)\n", crid_major);
     		return (crid_major);
   	}  /* endif */

   	return (0);

}  /* end init_module */


/*********************************************************************

@func  cleanup_module

Routine Description:
cleanup_module() entry point is called prior to unloading.
This will not be called if any channels are open.


Return Value:


***********************************************************************/
void cleanup_module(void)
{
	//in case of a delayed open, the open will fail
	//because the flag indicates pending shutdown.
        if (gpCridInstanceData) gpCridInstanceData->ulStatus |= CARD_BUSY; 
	//Unregister the driver at once, so that we have 
	//clean break from agents.
	unregister_chrdev(CRID_MAJOR, "cpqrid");
#ifdef CRID_X86_64
	unregister_ioctl32_conversion((('Z'<<8)|0));
	unregister_ioctl32_conversion((('Z'<<8)|1));
	unregister_ioctl32_conversion((('Z'<<8)|2));
	unregister_ioctl32_conversion((('Z'<<8)|3));
	unregister_ioctl32_conversion((('Z'<<8)|4));
	unregister_ioctl32_conversion((('Z'<<8)|5));
	unregister_ioctl32_conversion((('Z'<<8)|6));
	unregister_ioctl32_conversion((('Z'<<8)|7));
	unregister_ioctl32_conversion((('Z'<<8)|8));
	unregister_ioctl32_conversion((('Z'<<8)|9));
	unregister_ioctl32_conversion((('Z'<<8)|0xA));
	unregister_ioctl32_conversion((('Z'<<8)|0xC));
	unregister_ioctl32_conversion((('Z'<<8)|0xD));
#endif
	//common shutdown.
	crid_remove_one();

#ifdef ROMDRV_ENABLED
	romdrv_fini();
#endif

}   /* end cleanup_module */



/*********************************************************************

@func  crid_init_one

Routine Description:
This routine will initalize the device and the device driver.


Return Value:


***********************************************************************/
static int crid_init_one(void)
{
   int            i;
   PCRIDDATA      pInstance = NULL;
   POSDATA        pOsData = NULL;


/*
** We need to initialize our PCI Device Structure and get our basic information.
*/

#ifdef DEBUG_INIT
   printk(KERN_DEBUG "START crid_init_one\n");
#endif

/* Allocate and initialize a new instance structure. */
   gpCridInstanceData = (PCRIDDATA)kmalloc(sizeof (CRIDDATA), GFP_KERNEL);
   if (!gpCridInstanceData) return(-EAGAIN);
   memset(gpCridInstanceData, 0, sizeof(CRIDDATA));

   pInstance = gpCridInstanceData;     /* Set local pointer for convenience */

/* Preset initial condition of the card with the global "flags" */
   pInstance->ulInit = flags;

   pInstance->pOsData = (PVOID)kmalloc(sizeof (OSDATA), GFP_KERNEL);
   if (!pInstance->pOsData) return(-EAGAIN);
   memset(pInstance->pOsData, 0, sizeof(OSDATA));
   pOsData = (POSDATA)pInstance->pOsData;

#ifdef CRID_DEBUG
   pInstance->ulInit |= CRID_DEBUG_ALL;
   printk(KERN_DEBUG "crid_init_one:  spin_lock_init\n");
#endif

/* Initialize locks */
   for (i=0; i<MAX_CRID_LOCK; i++) {
   	spin_lock_init(&pOsData->crid_lock[i].lock);
   }

/* Initialize wait queues */
   for (i=0; i<MAX_SEQ_NUM; i++) {
	init_waitqueue_head(&(pOsData->slot_wq[i]));
   }
   init_waitqueue_head(&(pOsData->get_pkt_wq));

/*
** We need to initialize the number of CPU's on this server.
*/
#if LINUX_VERSION_CODE < 0x020600
   pInstance->ulCpuCount = smp_num_cpus;
#else
   pInstance->ulCpuCount = num_online_cpus();
#endif
#ifdef CRID_DEBUG
   if (pInstance->ulInit & CRID_DEBUG_ALL)
      printk(KERN_DEBUG "Number of CPUs  %d\n", pInstance->ulCpuCount);
#endif

#ifdef DEBUG_INIT
   printk(KERN_DEBUG "cridw_linux.c::  Calling cridc_common_init\n");
#endif
   
   if ( cridc_common_init(pInstance) != SUCCESS) {
      cridc_common_fini(pInstance); 
      return (-EOPNOTSUPP);
   }  /* endif */

   crid_entry_print_crid = create_proc_entry("cpqrid", S_IFREG | S_IRUGO, &proc_root);
   if (crid_entry_print_crid) {
	   crid_entry_print_crid->read_proc = &crid_read_print;
   } else {
	   printk("proc entry creation 'cpqrid' FAILED.\n");
   }
   pInstance->ulInit |= CRID_REGISTER_PROC;

/* Announce our arrival! */
   printk("crid:   hp ProLiant Remote Insight Board Driver (rev %s)\n", EFS_VER);

   return (SUCCESS);

}  /* crid_init_one */


/*********************************************************************

@func  crid_remove_one

Routine Description:
This routine will tear down the driver when the driver is unloaded.


Return Value:


***********************************************************************/
static int crid_remove_one(void)
{
	if (crid_entry_print_crid) {
		remove_proc_entry("cpqrid", &proc_root);
	}
	return cridc_common_fini(gpCridInstanceData);
}   /* end crid_remove_one */



/*********************************************************************

@func  crid_open

Routine Description:
The routine to process "open" requests from application space.


Return Value:


***********************************************************************/
static int crid_open(struct inode* inode, struct file* filep) 
{
   int rc = cridc_common_open(gpCridInstanceData);
   if (rc != 0) return -cridw_translate_error(rc);
   return (0);

} /* END crid_open */



/*********************************************************************

@func  crid_close

Routine Description:
This routine processes "close" requests from application space


Return Value:


***********************************************************************/
static int crid_close(struct inode* inode, struct file* filep) 
{

   int rc = cridc_common_close(gpCridInstanceData);
   if (rc != 0) return -cridw_translate_error(rc);
   return (0);

}  /* END crid_close */


#ifdef CRID_X86_64
/*********************************************************************

@func  crid_ioctl32_handler

Routine Description:
This routine provides the interface between 32-bit IOCTL requests and
the 64-bit kernel module.


Return Value:


***********************************************************************/
int crid_ioctl32_handler(unsigned int fd, unsigned int cmd, unsigned long arg, struct file *file)
{
	int err = 0;
	mm_segment_t old_fs;
	ioctl_arg_type32 *ioctl_arg32 = (ioctl_arg_type32 *)arg;
	ioctl_arg_type ioctl_arg64;

#ifdef CRID_DEBUG
	printk(KERN_DEBUG "Entered crid_ioctl32_handler\n");
#endif

	switch (cmd) {
		case (('Z'<<8)|0):
		case (('Z'<<8)|1):
		case (('Z'<<8)|4):
		case (('Z'<<8)|5):
		case (('Z'<<8)|6):
		case (('Z'<<8)|7):
		case (('Z'<<8)|8):
		case (('Z'<<8)|9):
		case (('Z'<<8)|0xA):
		case (('Z'<<8)|0xC):
		case (('Z'<<8)|0xD):
			err |= get_user(ioctl_arg64.in_cmd, &ioctl_arg32->in_cmd);
			err |= get_user(ioctl_arg64.cmd_length, &ioctl_arg32->cmd_length);
			err |= get_user(ioctl_arg64.resp_cmd, &ioctl_arg32->resp_cmd);
			err |= get_user(ioctl_arg64.resp_length, &ioctl_arg32->resp_length);
			if (err) return -EFAULT;
			old_fs = get_fs();
			set_fs(KERNEL_DS);
			err = sys_ioctl(fd, cmd, (unsigned long)&ioctl_arg64);
			set_fs(old_fs);
			break;
		case (('Z'<<8)|2):
		case (('Z'<<8)|3):
			old_fs = get_fs();
			set_fs(KERNEL_DS);
			err = sys_ioctl(fd, cmd, arg);
			set_fs(old_fs);
			break;
	}
	if(err == 0) {
		switch (cmd) {
			case (('Z'<<8)|0):
			case (('Z'<<8)|1):
			case (('Z'<<8)|4):
			case (('Z'<<8)|5):
			case (('Z'<<8)|6):
			case (('Z'<<8)|7):
			case (('Z'<<8)|8):
			case (('Z'<<8)|9):
			case (('Z'<<8)|0xA):
			case (('Z'<<8)|0xC):
			case (('Z'<<8)|0xD):
				err |= put_user(ioctl_arg64.in_cmd, &ioctl_arg32->in_cmd);
				err |= put_user(ioctl_arg64.cmd_length, &ioctl_arg32->cmd_length);
				err |= put_user(ioctl_arg64.resp_cmd, &ioctl_arg32->resp_cmd);
				err |= put_user(ioctl_arg64.resp_length, &ioctl_arg32->resp_length);
				break;
			case (('Z'<<8)|2):
			case (('Z'<<8)|3):
				break;
		}
	}
#ifdef CRID_DEBUG
	printk(KERN_DEBUG "crid_ioctl32_handler exiting... err = %d\n", err);
#endif
	if (err) return -EFAULT;
	return err;
}
#endif
/*********************************************************************

@func  crid_ioctl

Routine Description:
This routine processes IOCTL requests.


Return Value:


***********************************************************************/
static int crid_ioctl(struct inode* inode, struct file* fp, unsigned int cmd, unsigned long arg)
{
   
   int rc = cridc_common_ioctl(cmd, (void *)arg, 0, gpCridInstanceData);
   if (rc != 0) return -cridw_translate_error(rc);
   return (0);

} /* END crid_ioctl */



/*********************************************************************

@func  cridintr

Routine Description:
This is the main routine.  This will call the
common code interrupt routine for processing.


Return Value:

    ISTAT_ASSERTED
    ISTAT_NONE


***********************************************************************/
#if LINUX_VERSION_CODE < 0x020600
static void crid_intr(int irq, void* instance, struct pt_regs* regs)
{

    PCRIDDATA pInstance = (PCRIDDATA) instance;
    cridc_common_intr(pInstance);
}  /* end crid_intr */
#else
static irqreturn_t crid_intr(int irq, void* instance, struct pt_regs* regs)
{
    int rc;

    PCRIDDATA pInstance = (PCRIDDATA) instance;
    rc = cridc_common_intr(pInstance);
    if (rc == 0) {
	    return IRQ_HANDLED;
    } else {
	    return IRQ_NONE;
    }
}  /* end crid_intr */
#endif

/***************************************************************************
 ****************       START OF "MUST HAVE" FUNCTION    *******************
 ***************************************************************************/

/*********************************************************************
** 
** @func  cridw_map32_bit_phys
** 
** Routine Description:
** This routine is OS specific and will map in a physical address into
** memory space.
** 
** 
** Return Value:
** 
** NULL     : FAILURE
** VirtAddr : Virtual Address mapped in.
***********************************************************************/
void *cridw_map32_bit_phys( ULONG ulPhysicalAddr,
                            ULONG ulLength,
                            PCRIDDATA pInstance )
{
   void   *pVirtAddr = NULL;

   pVirtAddr = (PVOID)ioremap(ulPhysicalAddr, ulLength);

   return (pVirtAddr);

}  /* end cridw_map32_bit_phys */


/*********************************************************************
** 
** @func  cridw_umap32_bit_phys
** 
** Routine Description:
** This routine is OS specific and will unmap a physical address from
** the virtual memory space.  For UnixWare 7, we need to just remove
** the virtual address from our internal array.
** 
** Return Value:
** 
***********************************************************************/
VOID cridw_unmap32_bit_phys ( PVOID pVirtAddr,ULONG ulLength,
                              PCRIDDATA pInstance )
{

    iounmap(pVirtAddr);

}  /* end cridw_unmap32_bit_phys */



/*********************************************************************
** 
** @func  cridw_malloc
** 
** Routine Description:
** This routine will allocate kernel memory for 
** driver use.  This is NOT mapped in memory but
** allocated from the kernel pools.
** 
** Return Value:
** Pointer to allocated memory or NULL if this fails.
** 
***********************************************************************/
PVOID cridw_malloc(ULONG ulSize, ULONG ulInit)
{
   PVOID  pVMem;
   ULONG  ulOrder;

#ifdef DEBUG
   if (gpCridInstanceData->ulInit & CRID_DEBUG_ALL)
     printk(KERN_DEBUG "cridw_malloc:  Asking for %li bytes\n", ulSize);
#endif

   ulOrder = get_order(ulSize);

   if(ulInit) {
	if (ulOrder > 5) {
		pVMem = (PVOID)__get_free_pages(GFP_ATOMIC, ulOrder);
	} else {
		pVMem = (PVOID)kmalloc(ulSize, GFP_ATOMIC);
	}  /* endif */
   } else {
	if (ulOrder > 5) {
		pVMem = (PVOID)__get_free_pages(ulInit, GFP_KERNEL);
	} else {
		pVMem = (PVOID)kmalloc(ulSize, GFP_KERNEL);
	}  /* endif */
   }

#ifdef DEBUG
   if (gpCridInstanceData->ulInit & CRID_DEBUG_ALL)
      printk(KERN_DEBUG "cridw_malloc:  %p (size %li)\n",pVMem, ulSize);
#endif

   return (pVMem);

}  /* end cridw_malloc */



/*********************************************************************
** 
** @func  cridw_free
** 
** Routine Description:
** This routine will free kernel memory previously
** allocated for driver use.  This is NOT mapped in memory but
** allocated from the kernel pools.
** 
** Return Value:
** Nothing needed to return
** 
***********************************************************************/
VOID cridw_free(PVOID pVAddr, ULONG ulSize, ULONG ulInit)
{
   ULONG  ulOrder;
#ifdef CRID_DEBUG
   if (gpCridInstanceData->ulInit & CRID_DEBUG_ALL)
      printk("cridw_free:  %p (size %li)\n",pVAddr, ulSize);
#endif

   ulOrder = get_order(ulSize);
   if (ulOrder > 5) {
      free_pages((ULONG)pVAddr, ulOrder);
   } else {
      kfree (pVAddr);
   }  /* endif */

}  /* end cridw_free */



/*********************************************************************
** 
** @func  cridw_lock
** 
** Routine Description:
** This routine is OS specific and will take a synchronization lock for
** access to the ROM routines.
** 
** Return Value:
** Nothing needed to return
** 
***********************************************************************/
#ifdef CRID_DEBUG
void cridw_lock_debug(ULONG ulIndex, PCRIDDATA pInstance, char *pFileName, ULONG ulLine)
{
   POSDATA     pOsData = (POSDATA)pInstance->pOsData;
	
   //cridw_dbg(CRID_KERN_DEBUG, "spin lock %ld\n", ulIndex);
   spin_lock_irqsave(&pOsData->crid_lock[ulIndex].lock, pOsData->crid_lock[ulIndex].flags);
   pOsData->crid_lock[ulIndex].start=ulLine;
   pOsData->crid_lock[ulIndex].end=0;
   pOsData->crid_lock[ulIndex].file=pFileName;
} /* cridw_lock */
#else
void cridw_lock(ULONG ulIndex, PCRIDDATA pInstance)
{
   POSDATA     pOsData = (POSDATA)pInstance->pOsData;
	
   //cridw_dbg(CRID_KERN_DEBUG, "spin lock %ld\n", ulIndex);
   spin_lock_irqsave(&pOsData->crid_lock[ulIndex].lock, pOsData->crid_lock[ulIndex].flags);
} /* cridw_lock */
#endif


/*********************************************************************
** 
** @func  cridw_unlock
** 
** Routine Description:
** This routine is OS specific and will free a synchronization lock for
** the ROM access point.
** 
** Return Value:
** Nothing needed to return
** 
***********************************************************************/
#ifdef CRID_DEBUG
void cridw_unlock_debug(ULONG ulIndex, PCRIDDATA pInstance, char *pFileName, ULONG ulLine)
{
   POSDATA     pOsData = (POSDATA)pInstance->pOsData;

   //cridw_dbg(CRID_KERN_DEBUG, "spin unlock %ld\n", ulIndex);
   spin_unlock_irqrestore(&pOsData->crid_lock[ulIndex].lock, pOsData->crid_lock[ulIndex].flags);
   pOsData->crid_lock[ulIndex].end=ulLine;
} /* cridw_unlock_debug */
#else
void cridw_unlock(ULONG ulIndex, PCRIDDATA pInstance)
{
   POSDATA     pOsData = (POSDATA)pInstance->pOsData;

   //cridw_dbg(CRID_KERN_DEBUG, "spin unlock %ld\n", ulIndex);
   spin_unlock_irqrestore(&pOsData->crid_lock[ulIndex].lock, pOsData->crid_lock[ulIndex].flags);
} /* cridw_unlock */
#endif


/*********************************************************************
** 
** @func  cridw_wait_for_event
** 
** Routine Description:
** This routine is OS specific and will wait to be notified when something
** is placed on the queue.  When the the routine is released from here,
** one of two things has happened:
** 
** 1.  An Event has been put on the queue.
** 2.  The OS is shutting down.
** 
** If no Event is returned from the queue, the OS is shutting down and
** the application should return.
** 
** NOTE:  The QUEUE lock must be held before calling this routine.
** 
** Return Value:
** Nothing needed to return
** 
***********************************************************************/
void cridw_wait_for_event(ULONG ulIndex, PCRIDDATA pInstance)
{
    POSDATA     pOsData = (POSDATA)pInstance->pOsData;

#if LINUX_VERSION_CODE >= 0x020600
    lock_kernel();
#endif
    
    if (ulIndex == GET_PKT_WQ) {
    	interruptible_sleep_on(&pOsData->get_pkt_wq);
    } else {
	interruptible_sleep_on(&pOsData->slot_wq[ulIndex]);
    }

#if LINUX_VERSION_CODE >= 0x020600
    unlock_kernel();
#endif

} /* cridw_wait_for_event */


/*********************************************************************
** 
** @func  cridw_wait_for_event_timeout
** 
** Routine Description:
** This routine is OS specific and will wait to be notified when something
** is placed on the queue.  When the the routine is released from here,
** one of two things has happened:
** 
** 1.  An Event has been put on the queue.
** 
** If no Event is returned from the queue, the OS is shutting down and
** the application should return.
** 
** NOTE:  The QUEUE lock must be held before calling this routine.
** 
** Return Value:
** Nothing needed to return
** 
***********************************************************************/
ULONG cridw_wait_for_event_timeout(ULONG ulIndex, ULONG ulTime, PCRIDDATA pInstance)
{
    POSDATA     pOsData = (POSDATA)pInstance->pOsData;
    ULONG retval = 0;

#if LINUX_VERSION_CODE >= 0x020600
    lock_kernel();
#endif
    
    if (ulIndex == GET_PKT_WQ) {
    	retval = interruptible_sleep_on_timeout(&pOsData->get_pkt_wq, US_TO_TICKS(ulTime));
    } else {
	retval = interruptible_sleep_on_timeout(&pOsData->slot_wq[ulIndex], US_TO_TICKS(ulTime));
    }

#if LINUX_VERSION_CODE >= 0x020600
    unlock_kernel();
#endif
    return retval;

} /* cridw_wait_for_event_timeout */


/*********************************************************************
**
** @func  cridw_event_notify
** 
** Routine Description:
** This routine is OS specific and will notify the application waiting
** on the queue.
** 
** NOTE:  The QUEUE lock does NOT need to be held for this routine.
** 
** Return Value:
** Nothing needed to return
** 
***********************************************************************/
void cridw_event_notify(ULONG ulIndex, PCRIDDATA pInstance)
{
   POSDATA     pOsData = (POSDATA)pInstance->pOsData;

   if (ulIndex == GET_PKT_WQ) {
   	wake_up_interruptible(&pOsData->get_pkt_wq);
   } else {
	wake_up_interruptible(&pOsData->slot_wq[ulIndex]);
   }
			
} /* cridw_event_notify */


/*********************************************************************
** 
** @func  cridw__event_active
** 
** Routine Description:
** This routine is OS specific and checks if the queue is active.
** 
** Return Value:
** 
***********************************************************************/
ULONG cridw_event_active(ULONG ulIndex, PCRIDDATA pInstance)
{
    ULONG retval;
    POSDATA     pOsData = (POSDATA)pInstance->pOsData;

    if (ulIndex == GET_PKT_WQ) {
    	retval = waitqueue_active(&pOsData->get_pkt_wq);
    } else {

    	retval = waitqueue_active(&pOsData->slot_wq[ulIndex]);
    }

    return retval;

   
} /* cridw_event_active */


/*********************************************************************

@func cridw_signal_pending

Routine Description:
This routine is OS specific. 


Return Value:


***********************************************************************/


ULONG cridw_signal_pending(PCRIDDATA pInstance)
{

	return signal_pending(current);

} /* end cridw_signal_pending */

/***************************************************************************
 **************** OS HEALTH COMMON CODE SUPPORT ROUTINES *******************
 ***************************************************************************/

/*********************************************************************
** 
** @func  cridw_in_byte_mem
** 
** Routine Description:
** This is an OS specific routine which will read and return one byte
** of data from a given offset. This is for memory mapped io.
** 
** 
** Return Value:
** 
** UCHAR   :  The data returned from the call
** 
***********************************************************************/
UCHAR cridw_in_byte_mem(PULONG pulAddr, PCRIDDATA pInstance)
{
   PUCHAR  pByte;
   pByte = (UCHAR *) pulAddr;
   return ( (UCHAR)*pByte );

}  /* end cridw_in_byte_mem */


/*********************************************************************
** 
** @func  cridw_in_byte_word
** 
** Routine Description:
** This is an OS specific routine which will read and return one word
** of data from a given offset. This is for memory mapped IO.
** 
** 
** Return Value:
** 
** USHORT   :  The data returned from the call
** 
***********************************************************************/
USHORT cridw_in_word_mem(PULONG pulAddr, PCRIDDATA pInstance)
{
     USHORT *pWord;

     pWord = (USHORT *)pulAddr;
     return ( (USHORT) *pWord );

}  /* end cridw_in_word_mem */


/*********************************************************************
** 
** @func  cridw_in_dword_mem
** 
** Routine Description:
** This is an OS specific routine which will read and return a double 
** word of data from a given offset. This is for memory mapped IO.
** 
** 
** Return Value:
** 
** ULONG   :  The data returned from the call
** 
***********************************************************************/
ULONG cridw_in_dword_mem(PULONG pulAddr, PCRIDDATA pInstance)
{
     unsigned int *pDword;

     pDword = (unsigned int *)pulAddr;
     return ( (unsigned int)*pDword );

}  /* end cridw_in_dword_mem */


/*********************************************************************
** 
** @func  cridw_out_byte_mem
** 
** Routine Description:
** This is an OS specific routine which will write the specified byte
** to the given offset. This is for memory mapped IO.
** 
** 
** Return Value:
** 
** Nothing
** 
***********************************************************************/
void cridw_out_byte_mem( PULONG pAddr, UCHAR ucData, PCRIDDATA pInstance )
{
     PUCHAR pByte;

     pByte = (PUCHAR) pAddr;
     *pByte = ucData; 

}  /* end cridw_out_byte_mem */



/*********************************************************************
** 
** @func  cridw_out_word_mem
** 
** Routine Description:
** This is an OS specific routine which will write the specified word
** to the given offset. This is for Memory Mapped IO.
** 
** 
** Return Value:
** 
** Nothing
** 
***********************************************************************/
void cridw_out_word_mem( PULONG pAddr, USHORT usData, PCRIDDATA pInstance )
{
   PUSHORT pWord;

   pWord = (PUSHORT) pAddr;
   *pWord = usData; 

}  /* end cridw_out_word_mem */



/*********************************************************************
** 
** @func  cridw_out_dword_mem
** 
** Routine Description:
** This is an OS specific routine which will write the specified byte
** to the given offset. This routine is for Memory Mapped IO.
** 
** 
** Return Value:
** 
** Nothing
** 
***********************************************************************/
void cridw_out_dword_mem( PULONG pAddr, unsigned int ulData, PCRIDDATA pInstance )
{
   unsigned int * pDword;

   pDword = (unsigned int *) pAddr;
   *pDword = ulData; 

}  /* end cridw_out_dword_mem */


/*********************************************************************
** 
** @func  cridw_read_byte
** 
** Routine Description:
** This is an OS specific routine which will read and return one byte
** of data from a given offset. This is for IO (in / out) transfers.
** 
** 
** Return Value:
** 
** UCHAR   :  The data returned from the call
** 
***********************************************************************/
UCHAR cridw_read_byte(ULONG ulAddr, PCRIDDATA pInstance)
{
   return ( (UCHAR)inb(ulAddr) );

}  /* end cridw_read_byte */


/*********************************************************************
** 
** @func  cridw_read_word
** 
** Routine Description:
** This is an OS specific routine which will read and return one word
** of data from a given offset. This is for IO (in / out) transfers.
** 
** 
** Return Value:
** 
** USHORT   :  The data returned from the call
** 
***********************************************************************/
USHORT cridw_read_word(ULONG ulAddr, PCRIDDATA pInstance)
{
   return ( (USHORT)inw(ulAddr) );

}  /* end cridw_read_word */


/*********************************************************************
** 
** @func  cridw_read_dword
** 
** Routine Description:
** This is an OS specific routine which will read and return a double 
** word of data from a given offset. This is for IO (in / out) transfers.
** 
** 
** Return Value:
** 
** ULONG   :  The data returned from the call
** 
***********************************************************************/
ULONG cridw_read_dword(ULONG ulAddr, PCRIDDATA pInstance)
{
   return ( (ULONG)inl(ulAddr) );

}  /* end cridw_read_dword */


/*********************************************************************
** 
** @func  cridw_write_byte
** 
** Routine Description:
** This is an OS specific routine which will write the specified byte
** to the given offset. This is for IO (in / out) transfers.
** 
** 
** Return Value:
** 
** Nothing
** 
***********************************************************************/
void cridw_write_byte(ULONG ulAddr, UCHAR ucData, PCRIDDATA pInstance )
{
   outb(ucData, ulAddr);

}  /* end cridw_write_byte */



/*********************************************************************
** 
** @func  cridw_write_word
** 
** Routine Description:
** This is an OS specific routine which will write the specified word
** to the given offset. This is for IO (in / out) transfers.
** 
** 
** Return Value:
** 
** Nothing
** 
***********************************************************************/
void cridw_write_word( ULONG ulAddr, USHORT usData, PCRIDDATA pInstance )
{
   outw(usData, ulAddr);

}  /* end cridw_write_word */



/*********************************************************************
** 
** @func  cridw_write_dword
** 
** Routine Description:
** This is an OS specific routine which will write the specified byte
** to the given offset. This is for IO (in / out) transfers.
** 
** 
** Return Value:
** 
** Nothing
** 
***********************************************************************/
void cridw_write_dword( ULONG ulAddr, unsigned int ulData, PCRIDDATA pInstance )
{
   outl(ulData, ulAddr);

}  /* end cridw_write_dword */


/*********************************************************************
** 
** @func  cridw_microdelay
** 
** Routine Description:
** This is an OS specific routine which will delay a specific amount of
** time.  The time passed in is in Microseconds.
** 
** 
** Return Value:
** 
** Nothing
** 
***********************************************************************/
void cridw_microdelay( ULONG ulUsec, ULONG nosleep, PCRIDDATA pInstance )
{

	int i; 
	
	if (nosleep) {	
		for(i=0; i<(ulUsec)/10000; i++) {
			udelay(10000);
		}
	} else {
		current->state = TASK_INTERRUPTIBLE;
		schedule_timeout(US_TO_TICKS(ulUsec));
	}
}  /* end cridw_microdelay */


/*********************************************************************
** 
** @func  cridw_timeout
** 
** Routine Description:
** This is an OS specific routine which will delay a specific amount of
** time or if a function is passed in, execute a time delay to run
** that function.  
** 
** NOTE:  If pFunction is equal to NULL, then this routine must ONLY
**        be called in a BLOCKING (i.e. not interrupt) context.  We
**        use "delay" which is not permitted to be called at interrupt
**        context.
** 
** Return Value:
** Nothing
** 
***********************************************************************/

static void cridw_timeout_master_handler(ULONG _arg)
{
   PCRIDTIMER		pTimer = (PCRIDTIMER)_arg;

   void			(*func)(void*);
   void*		arg;

   if (pTimer == NULL) return;

   func = pTimer->pFunction;
   arg  = pTimer->pArg;
  
   memset(pTimer, 0, sizeof(CRIDTIMER));
 
   if (func != NULL) (*func)(arg);
}

  
	
ULONG cridw_timeout( PVOID pFunction, PVOID pArg, ULONG ulUsec, 
                    ULONG ulTimeOutID, PCRIDDATA pInstance )
{
   PCRIDTIMER		pTimer;
   POSDATA             	pOsData = (POSDATA)pInstance->pOsData;
   static spinlock_t	lock = SPIN_LOCK_UNLOCKED;
   unsigned long	iflg;	
   int 			rc = 0;

#ifdef CRID_DEBUG
   if (pInstance->ulInit & CRID_DEBUG_ALL)
      printk(KERN_DEBUG "cridw_timeout:  function %p  delay:  %li\n",
                        pFunction, ulUsec);
#endif
  
   if (!pFunction) {
      cridw_microdelay(ulUsec, 1, pInstance);
   } else {
      pTimer = &pOsData->timer[ulTimeOutID];

      //c.s. on
      spin_lock_irqsave(&lock, iflg);
      //If it is already in use return (without scheduling)
      if (pTimer->used) {
	      rc = -1;
	      goto cridw_timeout_exit_0;
      }

      //mark it used
      pTimer->used = 1;
      //Load the real function and argument
      pTimer->pFunction = (void (*)(void*))pFunction;
      pTimer->pArg = pArg;
      //initialize the linux timer
      init_timer(&(pTimer->entry));
      //Call it with the master timeout handler
      pTimer->entry.function = cridw_timeout_master_handler;
      pTimer->entry.data = (ULONG)pTimer;
      pTimer->entry.expires = jiffies + US_TO_TICKS(ulUsec);
      //And go!
      add_timer(&(pTimer->entry));

cridw_timeout_exit_0:
      //c.s. off
      spin_unlock_irqrestore(&lock, iflg);
   }  /* endif */
     
   return rc;

}  /* end cridw_timeout */


/*********************************************************************
** 
** @func  cridw_cancel_timeout
** 
** Routine Description:
** This is an OS specific routine will cancel a previously scheduled
** timeout.  This function should be called with a lock held.
** 
** 
** Return Value:
** 
** 
***********************************************************************/
VOID cridw_cancel_timeout( ULONG ulTimeOutID, PCRIDDATA pInstance )
{
   PCRIDTIMER		pTimer;
   POSDATA             	pOsData = (POSDATA)pInstance->pOsData;

   pTimer = &pOsData->timer[ulTimeOutID];
   if (pTimer->used) {
	   del_timer_sync(&(pTimer->entry));
	   memset(pTimer, 0, sizeof(CRIDTIMER));
   }

}  /* end cridw_cancel_timeout */


/*********************************************************************
** 
** @func  cridw_halt
** 
** Routine Description:
** This is an OS specific routine will cancel timeouts
** 
** 
** Return Value:
** 
** 
***********************************************************************/
VOID cridw_halt( PCRIDDATA pInstance )
{
   int i;
   for (i=0; i<MAX_CRID_TOID; i++) {
   	cridw_cancel_timeout(i, pInstance);	
   }

}  /* end cridw_cancel_timeout */


/*********************************************************************
** 
** @func  cridw_memcpy
** 
** Routine Description:
** This is an OS specific routine which will copy from Source to 
** destination for a certain amount of bytes.  No Alignment is
** assumed so individual OS's may need to double buffer to assure
** alignment if required.
** 
** Return Value:
** 
** Nothing
** 
***********************************************************************/
void *cridw_memcpy(void *dest, void *source, int length)
{
        return memcpy(dest, source,  length);

}  /* end cridw_memcpy */



/*********************************************************************
** 
** @func  cridw_strcpy
** 
** Routine Description:
** This is an OS specific routine which will copy from Source to 
** destination until a NULL character is received.  No alignment is
** assumed so individual OS's may need to double buffer to assure
** alignment if required.
** 
** Return Value:
** Byte pointer to the Destination
** 
***********************************************************************/
char *cridw_strcpy (char *s1, char *s2)
{

   return ( (char *) strcpy(s1,s2));

} /* end cridw_strcpy */

/*********************************************************************
** 
** @func  cridw_strncpy
** 
** Routine Description:
** This is an OS specific routine which will copy from Source to 
** destination with size n.  
** 
** Return Value:
** Byte pointer to the Destination
** 
***********************************************************************/
char *cridw_strncpy(char *s1, char *s2, unsigned int n)
{

   return ( (char *) strncpy(s1,s2,n));

} /* end cridw_strncpy */


/*********************************************************************
** 
** @func  cridw_strcat
** 
** Routine Description:
** This is an OS specific routine which will concatenate from Source to 
** destination until a NULL character is received.  No alignment is
** assumed so individual OS's may need to double buffer to assure
** alignment if required.
** 
** Return Value:
** 
** Byte pointer to the destination.
** 
***********************************************************************/
char *cridw_strcat (char *s1, char *s2)
{

   return ( (char *) strcat(s1,s2));

}  /* end cridw_strcat */


/*********************************************************************
** 
** @func  cridw_strlen
** 
** Routine Description:
** This is an OS specific routine which return the length of a null
** terminated string. No alignment is assumed so individual OS's may 
** need to double buffer to assure alignment if required.
** 
** Return Value:
** 
** String Length
** 
***********************************************************************/
unsigned int cridw_strlen (char *s1)
{

   return ( (unsigned int) strlen(s1) );

} /* end cridw_strlen */



/*********************************************************************
** 
** @func  cridw_memzero
** 
** Routine Description:
** This OS Specific routine will "zero out" a memory location for the
** length specified.
** No alignment is assumed so individual OS's may need to double buffer 
** to assure alignment if required.
** 
** Return Value:
** 
** VOID pointer to memory addressed passed in.
** 
***********************************************************************/
void *cridw_memzero(void *s, int n)
{
   memset(s,0,n);
   return (s);
}  /* end cridw_memzero */


/*********************************************************************
** 
** @func  cridw_memset
** 
** Routine Description:
** This is an OS Specific routine which will set memory to a particular
** value as passed in. We will use cridw_memzero if the value is zero
** to gain the efficency of the SCO UnixWare library routines.
** No alignment is assumed so individual OS's may need to double buffer 
** to assure alignment if required.
** 
** Return Value:
** VOID pointer to memory addressed passed in.
**
**
***********************************************************************/
void * cridw_memset(  void *s, register int c, register int n)
{
   memset(s,c,n);
   return (s);

}  /* end cridw_memset */

/*********************************************************************
** 
** @func  cridw_strncmp
** 
** Routine Description:
** This is an OS Specific routine which will compare two strings for
** a specified length.  Follows the standard C Library for "strncmp".
** No alignment is assumed so individual OS's may need to double buffer 
** to assure alignment if required.
** 
** Return Value:
** VOID pointer to memory addressed passed in.
** 
***********************************************************************/
int cridw_strncmp(register char *s1, register char *s2, register int length)
{
   char *end;

   for (end = s1 + length - 1; s1 < end && *s1 - *s2 == 0; s1++, s2++);
   return((int)(*s1 - *s2));

} /* end cridw_strncmp */

/*********************************************************************
** 
** @func  cridw_sprintf
** 
** Routine Description:
** This is an OS Specific routine used to copy a string into a buffer
** to format specification.
** 
** Return Value:
** 
** 0   : SUCCESS
** 1   : FAILURE
** 
***********************************************************************/

int cridw_sprintf(char* pString, char* pFormat, ...)
{
	int rc;
	va_list args;

	va_start(args, pFormat);
	rc = vsprintf(pString, pFormat, args);
	va_end(args);

	return rc;
}

/*********************************************************************

@func cridw_printf 

Routine Description:
This routine is not really a function but a proxy to printk
has to be known with the same printk signature


Return Value:


***********************************************************************/

#define _PROXY_JUMP(from, to)\
__asm__("\n"\
".globl " #from " \n"\
__ALIGN_STR"\n"\
#from " :\n\t"\
	"jmp " # to "\n\t")

#define PROXY_JUMP(from, to) _PROXY_JUMP(from, to )

PROXY_JUMP( cridw_printf, printk );

/*********************************************************************

@func cridw_log_evt

Routine Description:
This routine provides a way to log and event 
This depends on the presence of the health driver

Return Value:


***********************************************************************/

int cridw_log_evt(int log, int match, int num, void* entry)
{
   //lets only display the message once
   static int displayed;
   int rc = 0;

   #define EVENT_LOG_HANDLER "cpqwEventHandler"
   typedef int (*iml_interface_t)(int, int, int , void*);
   iml_interface_t cpqwEventHandler = (iml_interface_t)inter_module_get(EVENT_LOG_HANDLER);

   if (!cpqwEventHandler) {
#if 0
	if(!displayed) {
		printk(KERN_NOTICE "NOTICE: %s not registered. cpqevt module may not be loaded. Events will be not be logged.\n", 
				EVENT_LOG_HANDLER);
		displayed = 1;
	}
#endif
   	return 0;
   }

   rc = (*cpqwEventHandler)(log, match, num, entry);
   inter_module_put(EVENT_LOG_HANDLER);
   return rc;
}

/*********************************************************************
** 
** @func  cridw_set_evthnd
** 
** Routine Description:
** This routine installs a callback routing in the health driver
** to get new events. 
** 
** Return Value:
** 
** 0   : SUCCESS
** 1   : FAILURE
** 
***********************************************************************/
int cridw_set_evthnd(void* evthnd)
{
   //lets only display the message once
   static int displayed;

   #define EVENT_CALLBACK "callCridLog"
   void** callback = (void**)inter_module_get(EVENT_CALLBACK);
   if (!callback) {
#if 0
	if(!displayed) {
		printk(KERN_NOTICE "NOTICE: %s not registered. cpqevt module may not be loaded. No events will be received.\n", 
				EVENT_CALLBACK);
		displayed = 1;
	}
#endif
   	return 0;
   }

   *callback = evthnd;
   inter_module_put(EVENT_CALLBACK);
   return 0;
}

/*********************************************************************
** 
** @func  cridw_get_ubuff
** 
** Routine Description:
** This routine transfers memory from user to kernel space 
** 
** 
** Return Value:
** 
** 0   : SUCCESS
** 1   : FAILURE
** 
***********************************************************************/

int cridw_get_ubuff( void *pUserBuff,          /* addr to copy from   */
                     void *pKernelBuff,        /* addr to copy to     */
                     ULONG buffer_len,         /* how much to copy    */
                     PCRIDDATA pInstance )     /* OS Specific Data    */
{
   int            rv = SUCCESS;

   rv = (copy_from_user((void*)(pKernelBuff), (void*)(pUserBuff), (buffer_len))?-1:0);

   return (rv);

}  /* end cridw_get_ubuff */


/*********************************************************************
** 
** @func  cridw_put_ubuff
** 
** Routine Description:
** This is an OS Specific routine used to copy a buffer from kernel space
** to user space.
** 
** Return Value:
** 
** 0   : SUCCESS
** 1   : FAILURE
** 
***********************************************************************/
int cridw_put_ubuff( void *pUserBuff,          /* addr to copy from   */
                     void *pKernelBuff,        /* addr to copy to     */
                     ULONG buffer_len,         /* how much to copy    */
                     PCRIDDATA pInstance )     /* OS Specific Data    */
{
   int            rv = SUCCESS;

   rv = (copy_to_user((void*)(pUserBuff), (void*)(pKernelBuff), (buffer_len))?-1:0);

   return (rv);

}  /* end cridw_put_ubuff */


/*********************************************************************
** 
** @func  cridw_pci_find_dev
** 
** Routine Description:
** Given a vendor and device id, enumerate all pci devices fitting that
** bill. The pCookie passed in can be NULL, in which case the first
** device is discovered. Otherwise the next device is returned.
** 
** Return Value:
** An opaque cookie value (handle) signifying the device. 
** 
***********************************************************************/
void* cridw_pci_find_dev(USHORT vendor, USHORT device, void* pCookie) 
{
	return pci_find_device((unsigned int)vendor, (unsigned int)device, (const struct pci_dev*) pCookie);
}

/*********************************************************************
** 
** @func  cridw_pci_read_irq
** 
** Routine Description:
** Reads out the IRQ level of a PCI device passed in as pCookie
** Return Value:
** 0 SUCCESS 
** != 0 FAILURE 
***********************************************************************/
int   cridw_pci_read_irq(void* pCookie, unsigned int* pResult) 
{
	if (pCookie == NULL || pResult == NULL) return -1;

	*pResult = ((struct pci_dev*)pCookie)->irq;
	return 0;
}

/*********************************************************************
** 
** @func  cridw_pci_read_subsys
** 
** Routine Description:
** read subsystem vendor/device id.
** 
** Return Value:
** 0 SUCCESS 
** != 0 FAILURE 
***********************************************************************/
int   cridw_pci_read_subsys(void* pCookie, USHORT* pSubVend, USHORT* pSubDevID)
{
	if (pCookie == NULL || pSubVend == NULL || pSubDevID == NULL) return -1;

	*pSubVend  = ((struct pci_dev*)pCookie)->subsystem_vendor;
	*pSubDevID = ((struct pci_dev*)pCookie)->subsystem_device;
	return 0;
}
/*********************************************************************
** 
** @func  cridw_pci_read_busdevfn
** 
** Routine Description:
** read bus, dev, and function number from the device 
** Return Value:
** 0 SUCCESS 
** != 0 FAILURE 
***********************************************************************/
int   cridw_pci_read_busdevfn(void* pCookie, UCHAR* pBus, UCHAR* pDev, UCHAR* pFn)
{
	if (pCookie == NULL || pBus == NULL || pDev == NULL || pFn == NULL) return -1;

	*pBus= ((struct pci_dev*)pCookie)->bus->number;
	*pDev= PCI_SLOT(((struct pci_dev*)pCookie)->devfn);
	*pFn= PCI_FUNC(((struct pci_dev*)pCookie)->devfn);
	return 0;
}
/*********************************************************************
** 
** @func  
** 
** Routine Description:
** read the base address registers (bar) 0 through 5 
** Return Value:
** 0 SUCCESS 
** != 0 FAILURE 
***********************************************************************/
int   cridw_pci_read_bar(void* pCookie, int index, ULONG* pBar)
{
	if (pCookie == NULL || pBar == NULL) return -1;
	if (0 > index || index >= 6) return -1;

	*pBar = pci_resource_start((const struct pci_dev*)pCookie, index);
	return 0;
}
/*********************************************************************
** 
** @func  
** 
** Routine Description:
** Read the slot number in which the card is 
** Return Value:
** 0 SUCCESS 
** != 0 FAILURE 
***********************************************************************/
int   cridw_pci_read_slot(void* pCookie, unsigned int* pSlot)
{
	if (pCookie == NULL || pSlot == NULL) return -1;
	//Dummy this up for now:
	*pSlot = 0;
	return 0;
}
/*********************************************************************
** 
** @func  
** 
** Routine Description:
** read a byte from pci config header space 
** Return Value:
** 0 SUCCESS 
** != 0 FAILURE 
***********************************************************************/
int   cridw_pci_read_byte(void* pCookie, int offset,  UCHAR* pResult) 
{
	if (pCookie == NULL || pResult == NULL) return -1;

	return pci_read_config_byte((struct pci_dev*)pCookie, offset, pResult);
}
/*********************************************************************
** 
** @func  
** 
** Routine Description:
** read a word from pci config header space 
** Return Value:
** 0 SUCCESS 
** != 0 FAILURE 
***********************************************************************/
int   cridw_pci_read_word(void* pCookie, int offset,  USHORT* pResult)
{
	if (pCookie == NULL || pResult == NULL) return -1;

	return pci_read_config_word((struct pci_dev*)pCookie, offset, pResult);
}
/*********************************************************************
** 
** @func  
** 
** Routine Description:
** read a dword from pci config header space 
** Return Value:
** 0 SUCCESS 
** != 0 FAILURE 
***********************************************************************/
int   cridw_pci_read_dword(void* pCookie, int offset, ULONG* pResult)
{
	if (pCookie == NULL || pResult == NULL) return -1;

	return pci_read_config_dword((struct pci_dev*)pCookie, offset, (u32*)pResult);
}
/*********************************************************************
** 
** @func  
** 
** Routine Description:
** write a byte into pci configuration space 
** Return Value:
** 0 SUCCESS 
** != 0 FAILURE 
***********************************************************************/
int   cridw_pci_write_byte(void* pCookie, int offset,  UCHAR pResult) 
{
	if (pCookie == NULL) return -1;

	return pci_write_config_byte((struct pci_dev*)pCookie, offset, pResult);
}
/*********************************************************************
** 
** @func  
** 
** Routine Description:
** write a word into pci configuration space 
** Return Value:
** 0 SUCCESS 
** != 0 FAILURE 
***********************************************************************/
int   cridw_pci_write_word(void* pCookie, int offset,  USHORT pResult) 
{
	if (pCookie == NULL) return -1;

	return pci_write_config_word((struct pci_dev*)pCookie, offset, pResult);
}
/*********************************************************************
** 
** @func cridw_pci_write_dword 
** 
** Routine Description:
** write a dword into pci configuration space 
** Return Value:
** 0 SUCCESS 
** != 0 FAILURE 
***********************************************************************/
int   cridw_pci_write_dword(void* pCookie, int offset,  ULONG pResult) 
{
	if (pCookie == NULL) return -1;

	return pci_write_config_dword((struct pci_dev*)pCookie, offset, pResult);
}
