#include "../CpqCiHlx.h"
#include "../CpqCiWrp.h"
#include "../CpqCiTyp.h"
#include "CpqCiSem.h"

void hexdump(void *base, void *data, unsigned len);

//BEGIN Semaphore implementation
//Sem array
static struct semaphore* global_sem = NULL;
//Sem array size
static int global_sem_size = 0;
DECLARE_RWSEM(cpqci_rwlock);

#if LINUX_VERSION_CODE >= 0x020400
//locking 
static spinlock_t semaphore_lock = SPIN_LOCK_UNLOCKED;

int cpqci_down_interruptible(struct semaphore * sem, int ms)
{
	int retval = 0;
	struct task_struct *tsk = current;
	DECLARE_WAITQUEUE(wait, tsk);
	//DBG("*******sem %p, ms %d\n", sem, ms);
	tsk->state = TASK_INTERRUPTIBLE;
	add_wait_queue_exclusive(&sem->wait, &wait);

	spin_lock_irq(&semaphore_lock);
	sem->sleepers ++;
	for (;;) {
		int sleepers = sem->sleepers;

		if (signal_pending(current)) {
			retval = -EINTR;
			sem->sleepers = 0;
			atomic_add(sleepers, &sem->count);
			break;
		}

		if (!atomic_add_negative(sleepers - 1, &sem->count)) {
			sem->sleepers = 0;
			break;
		}

		sem->sleepers = 1;	/* us - see -1 above */
		spin_unlock_irq(&semaphore_lock);

		if (ms >= 0) {
			int rc;
			//DBG("***schedule_timeout %d\n", ms/MS_PER_JIFFY);
			rc = schedule_timeout(ms/MS_PER_JIFFY);
			//DBG("***schedule_timeout returns with %d\n", rc);
			if (rc == 0) {
				sem->sleepers = 0;
				atomic_add(sleepers, &sem->count);
				goto SCHEDULING_DONE;
			}
		} else {
			//DBG("***schedule\n");
			schedule();
			//DBG("***schedule returns\n");
		}
		//DBG("*** task %p continues....\n", current);
		tsk->state = TASK_INTERRUPTIBLE;
		spin_lock_irq(&semaphore_lock);
	}
	spin_unlock_irq(&semaphore_lock);
SCHEDULING_DONE:
	tsk->state = TASK_RUNNING;
	remove_wait_queue(&sem->wait, &wait);
	wake_up(&sem->wait);
	return retval;
}
#else

asmlinkage int __sched cpqci_down_interruptible(struct semaphore * sem, int ms)
{
        int retval = 0;
        struct task_struct *tsk = current;
        DECLARE_WAITQUEUE(wait, tsk);
        unsigned long flags;

	printk("%p Entering cpqci_down_interruptible, ms is %x\n",current,ms);
        tsk->state = TASK_INTERRUPTIBLE;
	printk("%p spin_lock_irqsave (1) lock = %d\n",current, sem->wait.lock.lock);
        spin_lock_irqsave(&sem->wait.lock, flags);
        add_wait_queue_exclusive_locked(&sem->wait, &wait);

        sem->sleepers++;
        for (;;) {
                int sleepers = sem->sleepers;
		printk("%p start for loop, sleepers = %d\n",current, sem->sleepers);

                /*
                 * With signals pending, this turns into
                 * the trylock failure case - we won't be
                 * sleeping, and we* can't get the lock as
                 * it has contention. Just correct the count
                 * and exit.
                 */
                if (signal_pending(current)) {
			printk("%p current signal pending\n",current);
                        retval = -EINTR;
                        sem->sleepers = 0;
                        atomic_add(sleepers, &sem->count);
                        break;
                }

                /*
                 * Add "everybody else" into it. They aren't
                 * playing, because we own the spinlock in
                 * wait_queue_head. The "-1" is because we're
                 * still hoping to get the semaphore.
                 */
                if (!atomic_add_negative(sleepers - 1, &sem->count)) {
                        sem->sleepers = 0;
			printk("%p sleepers is 0\n",current);
                        break;
                }
                sem->sleepers = 1;      /* us - see -1 above */
		printk("%p spin_lock_irqrestore lock = %d\n",current, sem->wait.lock.lock);
                spin_unlock_irqrestore(&sem->wait.lock, flags);
printk("%p ms is %x\n",current,ms);
		if (ms >= 0) {
			int rc;
			printk("%p about to call schedule_timeout\n",current);
			rc = schedule_timeout(ms/MS_PER_JIFFY);
			printk("%p succeeded with schedule_timeout: rc is %d\n",current, rc);
			if (rc == 0) {
				sem->sleepers = 0;
				printk("%p sem_count is %d\n",current, sem->count);
				atomic_add(sleepers, &sem->count);
				printk("%p ***remove_wait_queue\n",current);
				hexdump(sem, sem, sizeof(struct semaphore));
				remove_wait_queue(&sem->wait, &wait);
				printk("%p in our routine, after remove_wait_queue\n",current);
				wake_up(&sem->wait);
				printk("%p in our routine, after wake_up\n",current);
				goto SCHEDULING_DONE;
			}
		} else {
			printk("%p about to call schedule\n",current);
                	schedule();
			printk("%p schedule call succeeded\n",current);
                	//schedule_timeout(1000000/MS_PER_JIFFY);
		}
		printk("%p spin_lock_irqsave (2) lock = %d\n",current, sem->wait.lock.lock);
                spin_lock_irqsave(&sem->wait.lock, flags);
                tsk->state = TASK_INTERRUPTIBLE;
		printk("%p end of loop 2\n",current);
        }
        remove_wait_queue_locked(&sem->wait, &wait);
	printk("%p after remove_wait_queue_locked\n",current);
	printk("%p spin_lock_irqrestore (2) lock = %d\n",current, sem->wait.lock.lock);
        spin_unlock_irqrestore(&sem->wait.lock, flags);
	printk("%p after spin_unlock_irqrestore\n",current);
        wake_up(&sem->wait);
	printk("%p after wake_up\n",current);

SCHEDULING_DONE:
	printk("%p entered SCHEDULING DONE label\n",current);
        tsk->state = TASK_RUNNING;
	printk("%p exit with %d\n",current, retval);
        return retval;
}

#endif

void hexdump(void *base, void *data, unsigned len)
{
	unsigned char *d = data;
	unsigned char pbuf[17];
	unsigned i = 0;
	while (i < len) {
		if (i % 16 == 0) {
			if (i)
				printk("  %s\n%08x  ", pbuf, i+((int)data-(int)base));

			else
				printk("%08x  ", i+((int)data-(int)base));
			memset(pbuf, 0, sizeof(pbuf));
		}
		printk("%02x ", *d);
		pbuf[i % 16] = (*d > 32 && *d < 128) ? *d : '.';
		d++;
		i++;
	}
	printk("  %s\n", pbuf);
}

int cpqci_global_sem_destroy() 
{
	if (global_sem == NULL) return 0;

	//printk("***sem_destroy: b4\n");
	//hexdump(global_sem, global_sem, sizeof(struct semaphore)*global_sem_size);

	//for (i=0; i<global_sem_size; i++) {
		//DBG("***force_signal %d\n", i);
		//cpqci_force_signal(&global_sem[i].wait, SIGKILL);
	//}
	kfree(global_sem);
	global_sem = NULL;
	global_sem_size = 0;
	return 0;
}

int cpqci_global_sem_create(int size) 
{
	int i;

	cpqci_global_sem_destroy(); 
	global_sem = kmalloc(sizeof(struct semaphore) * size, GFP_KERNEL);
	if (!global_sem) return -EFAULT;
	global_sem_size = size;
	for (i=0; i<size; i++) sema_init(global_sem + i, 0);
	return 0;
}

#ifdef CPQCI_X86_64
asm(
	".macro SAVE_ARGS addskip=0,norcx=0\n"
	"subq  $9*8+\\addskip,%rsp\n\t"
	"movq  %rdi,8*8(%rsp)\n\t"
	"movq  %rsi,7*8(%rsp)\n\t"
	"movq  %rdx,6*8(%rsp)\n\t"
	".if \\norcx\n"
	".else\n"
	"movq  %rcx,5*8(%rsp)\n\t"
	".endif\n"
	"movq  %rax,4*8(%rsp)\n\t"
	"movq  %r8,3*8(%rsp)\n\t"
	"movq  %r9,2*8(%rsp)\n\t"
	"movq  %r10,1*8(%rsp)\n\t"
	"movq  %r11,(%rsp)\n\t"
	".endm\n"

	".macro RESTORE_ARGS skiprax=0,addskip=0,skiprcx=0\n"
	"movq (%rsp),%r11\n\t"
	"movq 1*8(%rsp),%r10\n\t"
	"movq 2*8(%rsp),%r9\n\t"
	"movq 3*8(%rsp),%r8\n\t"
	".if \\skiprax\n"
	".else\n"
	"movq 4*8(%rsp),%rax\n\t"
	".endif\n"
	".if \\skiprcx\n"
	".else\n"
	"movq 5*8(%rsp),%rcx\n\t"
	".endif\n"
	"movq 6*8(%rsp),%rdx\n\t"
	"movq 7*8(%rsp),%rsi\n\t"
	"movq 8*8(%rsp),%rdi\n\t"
	".if 9*8+\\addskip > 0\n"
	"addq $9*8+\\addskip,%rsp\n\t"
	".endif\n"
	".endm\n"

	".macro thunk_retrax name,func\n"
	".globl \\name\n"
"\\name:\n"
	"SAVE_ARGS\n\t"
	"call \\func\n\t"
	"jmp restore_norax\n\t"
	".endm\n"

	"thunk_retrax cpqci_down_failed_interruptible,cpqci_down_interruptible\n\t"

"restore_norax:\n"
	"RESTORE_ARGS 1\n\t"
	"ret"
);

int down_timeout(struct semaphore * sem, int ms)
{
	int result;

#if WAITQUEUE_DEBUG
	CHECK_MAGIC(sem->__magic);
#endif

	__asm__ __volatile__(
		"# atomic interruptible down operation\n\t"
		LOCK "decl %1\n\t"     /* --sem->count */
		"js 2f\n\t"
		"xorl %0,%0\n"
		"1:\n"
		LOCK_SECTION_START("")
		"2:\tcall cpqci_down_failed_interruptible\n\t"
		"jmp 1b\n"
		LOCK_SECTION_END
		:"=a" (result), "=m" (sem->count)
		:"c" (sem), "b" (ms)
		:"memory");
	return result;
}
#else
asm(
".align 4\n"
".globl cpqci_down_failed_interruptible\n"
"cpqci_down_failed_interruptible:\n\t"
	"pushl %edx\n\t"
	"pushl %ebx\n\t"
	"pushl %ecx\n\t"
	"call cpqci_down_interruptible\n\t"
	"popl %ecx\n\t"
	"popl %ebx\n\t"
	"popl %edx\n\t"
	"ret"
);
int down_timeout(struct semaphore * sem, int ms)
{
	int result;

#if WAITQUEUE_DEBUG
	CHECK_MAGIC(sem->__magic);
#endif

	__asm__ __volatile__(
		"# atomic interruptible down operation\n\t"
		LOCK "decl %1\n\t"     /* --sem->count */
		"js 2f\n\t"
		"xorl %0,%0\n"
		"1:\n"
		".section .text.lock,\"ax\"\n"
		"2:\tcall cpqci_down_failed_interruptible\n\t"
		"jmp 1b\n"
		".previous"
		:"=a" (result), "=m" (sem->count)
		:"c" (sem), "b" (ms)
		:"memory");
	return result;
}
#endif

//END SEMAPHORE implementation

int sem_ioctl(struct inode *inod, struct file *f, unsigned int cmd, unsigned long arg)
{
	sem_value_pair data;
	struct semaphore* sem = NULL;
	data.rc = 0;

	if (arg) {
		DBG("copy_from_user %p %p\n", &data, arg);
		if (copy_from_user(&data, (sem_value_pair*)arg, sizeof(sem_value_pair)))  return -EFAULT; 
	}

	if(cmd != SEM_DESTROY && cmd != SEM_CREATE) {
		down_read(&cpqci_rwlock);
	} else {
		down_write(&cpqci_rwlock);
	}

	DBG("check if sem is NULL\n");
	if (data.sem == NULL && cmd != SEM_CREATE) return -EFAULT;
	DBG("check if sem passed in %p is global %p\n", data.sem, (int)global_sem);
	if (data.sem != NULL && (int)data.sem != (int)global_sem) return -EFAULT;
	sem = global_sem;

	DBG("sem_ioctl start: %p, value = %x, rc = %x\n", sem, data.value, data.rc);

	switch (cmd) {
	case SEM_CREATE:
		data.rc = cpqci_global_sem_create(data.value);
		data.sem = global_sem;
		//flush_signals(current);
		//recalc_sigpending(current);
		DBG("SEM_CREATE %p\n", data.sem);
		break;
	case SEM_DESTROY:
		cpqci_global_sem_destroy();
		DBG("SEM_DESTROY %p\n", sem);
		break;
	case SEM_UP:
		up(sem + data.value);
		DBG("SEM_UP %p\n", sem + data.value);
		break;
	case SEM_DOWN:
		DBG("SEM_DOWN start %p\n", sem + data.value);
		data.rc = down_timeout(sem + data.value, -1);
		if (data.rc != 0) DBG("SEM_DOWN %d failure rc = %d\n", data.value, data.rc);
		if (signal_pending(current)) {
			DBG("SEM_DOWN signal pending\n");
			//flush_signals(current);
			//recalc_sigpending(current);
			data.rc = -EINTR;
		}
		DBG("SEM_DOWN stop %p\n", sem + data.value, data.rc);
		break;
	case SEM_DOWN_TIMER:
		DBG("SEM_DOWN_TIMER start %p ms %d\n", sem + data.value, data.rc);
		data.rc = down_timeout(sem + data.value, data.rc);
		if (data.rc != 0) DBG("SEM_DOWN_TIMER %d failure rc = %d\n", data.value, data.rc);
		if (signal_pending(current)) {
			DBG("SEM_DOWN_TIMER signal pending\n");
			//flush_signals(current);
			//recalc_sigpending(current);
			data.rc = -EINTR;
		}
		DBG("SEM_DOWN_TIMER stop %p %d\n", sem + data.value, data.rc);
		break;
	case SEM_GET_VALUE:
		data.rc= atomic_read(&((sem + data.value)->count));
		DBG("SEM_GET_VALUE %p %d\n", sem + data.value, data.rc);
		break;
	case SEM_SET_VALUE:
		sema_init(sem + data.value, data.rc);
		DBG("SEM_SET_VALUE %p %d\n", sem + data.value, data.rc);
		break;
	default:
		data.rc= -EFAULT;
	}

	if(cmd != SEM_DESTROY && cmd != SEM_CREATE) {
		up_read(&cpqci_rwlock);
	} else {
		up_write(&cpqci_rwlock);
	}
	

	DBG("sem_ioctl end: %p, value = %x, rc = %x\n", sem, data.value, data.rc);

	if (arg) {
		DBG("copy_to_user %p %p\n", arg, &data);
		if (copy_to_user((sem_value_pair*)arg, &data, sizeof(sem_value_pair)))  return -EFAULT; 
	}
	return 0;
}
