/*++

Copyright (c) 2000,2001 Compaq Computer Corporation

Module Name:

    CpqCiKlx.c

Abstract:

    This module contains the code for a ring-0 Linux driver that
    provides kernel mode services to an application in user space.

Author:

    Tom.Bonola@compaq.com   Mar-2000
    Chris.Frantz@compaq.com Mar-2001 Ported to Linux

Environment:

    Kernel mode

Revision History:

--*/


#include "CpqCiHlx.h"
#ifdef CPQCI_X86_64
#include <asm/ioctl32.h>
//#include <linux/compat.h>
#endif
#include "CpqCiWrp.h"
#include "CpqCiTyp.h"
#include "linux/CpqCiSem.h"
#include "CpqCiKlx.h"
#include "CpqCiUlx.h"

#ifdef CPQCI_DESCRIPTION
MODULE_DESCRIPTION(CPQCI_DESCRIPTION);
#endif
#if LINUX_VERSION_CODE >= 0x020600
MODULE_LICENSE("Proprietary");
#endif

#ifdef  DEBUG 
void DbgDump(struct _CPQCIADAPTEROBJECT* Adapter, CPQCICONNECTION Connection);
#else
#define DbgDump( Adapter, Connection )
#endif

#define CPQCI_CONNECTION_NAME  "CpqCiCO"

#define MS_TO_JIFFIES(ms)	((ms)/(1000/HZ))

#if LINUX_VERSION_CODE < 0x020300
#define virt_to_page(x) (mem_map + (__pa(x) >> PAGE_SHIFT))
#elif defined(MAP_NR)
#define virt_to_page(v) MAP_NR(v)
#endif

#define CPQEFSVER	"7.2.0"

#define STRINGIZE(x)	#x
char kernel_version[] = "2.4.7-10smp";
//
//  Driver support prototypes for this module.
//
static int cpqci_open(struct inode *i, struct file *f);
static int cpqci_flush(struct file *f);
static int cpqci_close(struct inode *i, struct file *f);
static int cpqci_ioctl(struct inode *i, struct file *f, unsigned int cmd,
		       unsigned long arg);
#ifdef CPQCI_X86_64
int cpqci_ioctl32_handler(unsigned int fd, unsigned int cmd, unsigned long arg, struct file *file);
#endif
static int cpqci_mmap(struct file *f, struct vm_area_struct *vma);
#if LINUX_VERSION_CODE < 0x020600
void cpqci_intr(int irq, void *dev_id, struct pt_regs *regs);
#else
irqreturn_t cpqci_intr(int irq, void *dev_id, struct pt_regs *regs);
#endif

void CpqCiDrvOsFree(CPQCICONTEXT Context, CPQCIDRVOSHANDLE MemoryHandle);

CPQCIDRVOSHANDLE CpqCiDrvOsMallocContiguous(CPQCICONTEXT Context, CPQCISIZE AllocationSize);
CPQCIDRVOSHANDLE CpqCiDrvOsMap(CPQCICONTEXT Context, CPQCIPHYSICAL PhysicalAddress, CPQCISIZE MappedSize, 
		CPQCIMODE AddressMode, CPQCICACHETYPE CacheType);
CPQCISTATUS CpqCiOsGetRefCount( CPQCICONTEXT Context, CPQCITYPEQUAD* RefCount);

//
// Externs 
//
extern struct proc_dir_entry* cpqci_entry_print_cpqci;
extern int cpqci_read_print_cpqci(char *Buf, char **start, off_t off, int count, int *eof, void *data);
#ifndef ROMDRV_ENABLED
extern int CpqCiShareContext( CPQCICONTEXT Context, struct _CPQCIADAPTEROBJECT *pao,  CPQCICONNECTION connection);
extern CPQCIDRVOSHANDLE GetMemoryHandle( struct _CPQCIADAPTEROBJECT *pao,  CPQCICONNECTION connection);
#endif

//
// Globals
//
CPQCIADAPTEREXTENSION *CpqCi;

#if LINUX_VERSION_CODE < 0x020300
static unsigned long pci_mem_start = 0xf0000000;
#endif

static struct file_operations cpqci_ops = {
#if LINUX_VERSION_CODE >= 0x020300
	THIS_MODULE,
#endif
	open:		cpqci_open,
	flush:		cpqci_flush,
	release:	cpqci_close,
	ioctl:		cpqci_ioctl,
	mmap:		cpqci_mmap,
};

static struct miscdevice cpqci_miscdev = {
	CPQCI_MINOR,
	"cpqci",
	&cpqci_ops,
};

// VMA operations for shared memory
static void cpqci_vma_open(struct vm_area_struct *vma);
static void cpqci_vma_close(struct vm_area_struct *vma);

static unsigned long cpqci_do_munmap(CPQCICONTEXT Context, CPQCIPHYSICAL VirtualAddress, CPQCISIZE MappedSize);

static struct vm_operations_struct cpqci_vm_ops_regular = {
    open:  cpqci_vma_open,
    close: cpqci_vma_close,
    nopage:cpqci_vma_nopage_regular,
};

static struct vm_operations_struct cpqci_vm_ops_high = {
    open:  cpqci_vma_open,
    close: cpqci_vma_close,
    nopage:cpqci_vma_nopage_high,
};



void cpq_usleep(unsigned long usecs);




//====================================================
//Creating, Deleting and Finding Connection Extensions
//====================================================


struct semaphore connection_list_lock;

static CPQCICONNECTIONEXTENSION* cpqci_new_connection() {

	CPQCICONNECTIONEXTENSION* c = kmalloc(sizeof(CPQCICONNECTIONEXTENSION), GFP_KERNEL);
	DBG("(CONNECTIONEXTENSION)kmalloc(%d) = %p\n", sizeof(CPQCICONNECTIONEXTENSION), c);
	if (!c) return NULL;
	
	memset(c, 0, sizeof(CPQCICONNECTIONEXTENSION));
	if (!CpqCi) return c;

	down(&connection_list_lock);
	c->Next = CpqCi->ConnectionList;
	CpqCi->ConnectionList = c;
	up(&connection_list_lock);
	DBG("Insert CONNECTIONOBJECT into chain and returning %p\n", c);

	return c;
}


static CPQCICONNECTIONEXTENSION** cpqci_find_connection(CPQCICONNECTIONEXTENSION* c) {

	CPQCICONNECTIONEXTENSION** pc;

	if (!CpqCi) return NULL;

	down(&connection_list_lock);
	pc = &CpqCi->ConnectionList;
	while (*pc) {
		DBG("Matching Connection Object %p against %p\n", (*pc), c);
		if ((*pc) == c) break; 
		pc = &((*pc)->Next);
	}
	up(&connection_list_lock);

	return pc;
}

static void cpqci_delete_connection_mem_handle(CPQCICONNECTIONEXTENSION* c) {

        if (!c) return;	
	//If the buffer was used, we gotta clean up
	if (c->MemHandleMapped) {
		DBG("(connection) free MemHandleMapped %p (virt %p, phy %p)\n", 
				c->MemHandleMapped,
				c->MemHandleMapped->VirtualAddress,
				c->MemHandleMapped->PhysicalAddress);
		//Context is not needed !! (the memhandle has it all)
		CpqCiDrvOsFree(NULL, c->MemHandleMapped);
		c->MemHandleMapped = NULL;
	}
	if (c->MemHandleMalloc) {
		DBG("(connection) free MemHandleMalloc %p (virt %p, phy %p)\n", 
				c->MemHandleMalloc,
				c->MemHandleMalloc->VirtualAddress,
				c->MemHandleMalloc->PhysicalAddress);
		//Context is not needed !! (the memhandle has it all)
		CpqCiDrvOsFree(NULL, c->MemHandleMalloc);
		c->MemHandleMalloc= NULL;
	}
}

static void cpqci_new_connection_mem_handle(CPQCICONTEXT Context, CPQCICONNECTIONEXTENSION* c, CPQCISIZE AllocationSize) {

        if (!c) return;	
	//If the buffer was used, we are not going to proceed 
	if (c->MemHandleMapped || c->MemHandleMalloc) return; 
	
	//Allocate kernel memory and map it
	c->MemHandleMalloc = CpqCiDrvOsMallocContiguous(Context, AllocationSize);
	if (c->MemHandleMalloc) {
#ifdef DEBUG
		memset(c->MemHandleMalloc->VirtualAddress, 0xA5, AllocationSize);
#else
		memset(c->MemHandleMalloc->VirtualAddress, 0x0, AllocationSize);
#endif

		DBG("(connection) alloc MemHandleMalloc %p (virt %p, phy %p)\n", 
				c->MemHandleMalloc,
				c->MemHandleMalloc->VirtualAddress,
				c->MemHandleMalloc->PhysicalAddress);
		c->MemHandleMapped = CpqCiDrvOsMap(Context, c->MemHandleMalloc->PhysicalAddress, 
				AllocationSize, CpqCiUserMode, CpqCiCached );
	}
	if (c->MemHandleMapped == NULL) {
		CpqCiDrvOsFree(NULL, c->MemHandleMalloc);
		c->MemHandleMalloc = NULL;
	} else {
		DBG("(connection) alloc MemHandleMapped %p (virt %p, phy %p)\n", 
				c->MemHandleMapped,
				c->MemHandleMapped->VirtualAddress,
				c->MemHandleMapped->PhysicalAddress);
	}
}

static void cpqci_delete_connection(CPQCICONNECTIONEXTENSION* c) {

	CPQCICONNECTIONEXTENSION** pc;
      
        if (!c) return;	
	pc = cpqci_find_connection(c);
	if (!pc) return;

	//Delete it out of the chain
	down(&connection_list_lock);
	if (*pc != NULL) {
		DBG("Delete CONNECTIONOBJECT from chain\n");
		*pc = c->Next;
	}
	up(&connection_list_lock);

	cpqci_delete_connection_mem_handle(c);
	DBG("(connection)kfree(%p)\n", c);
	kfree(c);
}


//====================================================
//Cleanup routines 
//====================================================
int cpqci_close_helper(CPQCICONTEXT Context, CPQCICONNECTIONEXTENSION *connectionExtension)
/*++

Routine Description:

    This routine is called as a result of a user call to CloseHandle() or
    program termination.
    
Arguments:

	connectionExtension -

Return Value:

    The result of the operation.

--*/
{
	CPQCIADAPTEREXTENSION *adapterExtension = CpqCi;
	int connection;
	CPQCICHANNEL channel;
	CPQCISTATUS error;
    	unsigned long long ref;
	struct file* f = (struct file*) Context;

	if (connectionExtension == NULL)
		return 0;

    	CpqCiOsGetRefCount(Context, (CPQCITYPEQUAD*) &ref);
    	DBG(">>> Connection %d ref %ld\n", connectionExtension->Connection, ref);

	if ( ref <= 4) {
		//Right off the bat, delete the MemHandle
		cpqci_delete_connection_mem_handle(connectionExtension);
	}
	
	//  We only want to close a connection.
	if (connectionExtension->Header.ObjectType == CpqCiConnection) {

		connection = connectionExtension->Connection;
		//  Close the channel.
		error = CpqCiDrvCloseChannel(Context, adapterExtension->Adapter,
				connection, &channel);

		//Notify any registered waiter on this channel
		//if ((channel >= 0) && (channel < CPQCI_BASE_CRITICALPRIORITY_CHANNEL)) {
		//	cpqci_force_signal(&adapterExtension->RecvSignal[channel].wait, SIGKILL);
		//}

		//error should be ChannelStillInuse or ConnectionStillInuse or OK.
		if (error != CpqCiErrConnectionStillInuse) {
			if (error != CpqCiErrChannelStillInuse) {
				//  Signal the next waiter on the specified priority queue
				//  that a channel has become available.
				if (channel < CPQCI_BASE_HIGHPRIORITY_CHANNEL)
					up(&adapterExtension->ChannelSem[0]);
				else if (channel < CPQCI_BASE_CRITICALPRIORITY_CHANNEL)
					    up(&adapterExtension->ChannelSem[1]);

				DBG("*** Free Channel %d\n", channel);
			}
		
			//This will close the connection extension object of the OS	
			DBG("File %p private data gets deleted\n", f);
			f->private_data = NULL;	
			DBG("connection extension %p gets deleted\n", connectionExtension);
			cpqci_delete_connection(connectionExtension);

			DBG ("*** Free Connection %d\n", connection);
		} else {
			DBG (">>> Connection %d & Channel %d still in use.\n", connection, channel);
		}

		if (f == NULL) {
			DBG (">>> connection extension %p gets deleted\n", connectionExtension);
			cpqci_delete_connection(connectionExtension);
		}

	}

#ifndef DEBUG
	//MOD_DEC_USE_COUNT;
#endif
	return 0;
}

#if 0
int cpqci_close_helper2(CPQCICONTEXT Context)
{
	struct file* f = (struct file*) Context;
	int c;
	CPQCICONNECTIONEXTENSION *connectionExtension;

	DBG("cpqci_close_helper2 file pointer %p refcount %d\n", f, f ? atomic_read((atomic_t*)&f->f_count):0);
	if (f == NULL) return 0;

	connectionExtension = (CPQCICONNECTIONEXTENSION*) f->private_data;
	if (connectionExtension == NULL) return 0;


	c = atomic_read((atomic_t*)&f->f_count);
	//One refcount for the file still open
	if (c <= 1) {
		DBG("File %p private data gets deleted\n", f);
		f->private_data = NULL;	
		DBG("connection extension %p gets deleted\n", connectionExtension);
		cpqci_delete_connection(connectionExtension);
	}
	return 0;
}
#endif

void cleanup_module_helper(int irq_status, int irq, int register_status, struct _CPQCIADAPTEROBJECT* adapter, CPQCIADAPTEREXTENSION *adapterExtension)
/*
 * This is to clean up a (partially) registered channel driver
 *
 *
 */
{

	CPQCICONNECTIONEXTENSION* ci_conn;

#if LINUX_VERSION_CODE >= 0x020400 && defined(CONFIG_PROC_FS)
	if (cpqci_entry_print_cpqci){
	       	remove_proc_entry("cpqci", &proc_root);
	}
#endif 

	//This is a chunk of memory for semaphore handling
	DBG("global_sem_destroy\n");
	cpqci_global_sem_destroy();

	//  We should remove the interrupt service routine first.  This keeps
	//  the ISR off of the event object.
	if (irq_status == 0)
		free_irq(irq, adapterExtension);

	if (register_status == 0)
		misc_deregister(&cpqci_miscdev);

	// Kill any open connection left
	ci_conn = (adapterExtension == NULL) ? NULL : adapterExtension->ConnectionList; 
	DBG("Delete any remaining open connections....\n");
	//down(&connection_list_lock);
	while (ci_conn != NULL) {
		CPQCICONNECTIONEXTENSION* prev_ci_conn = ci_conn;
		DBG("Deleting connection extension %p\n", prev_ci_conn);
		ci_conn = ci_conn->Next;
		//This "unshares" the connection object (context is NULLED).
		CpqCiShareContext(NULL, adapter, prev_ci_conn->Connection);
		cpqci_close_helper(NULL, prev_ci_conn);
	}
	//up(&connection_list_lock);
	DBG("Done");

	//  Next we delete any adapter resources created.
	if (adapter)
		CpqCiDrvDeleteAdapter(NULL, adapter);

	//  If the adapter device object was created, then we have other 
	//  resources to wack.
	if (adapterExtension) {
		DBG("(adapter)kfree(%p)\n", adapterExtension);
		kfree(adapterExtension);
	}
}


int init_module(void)
/*++

Routine Description:

    This is the initialization routine for this driver.

Arguments:

Return Value:

    The function value is the final status from 
    the initialization operation.

--*/
{
	int status;
	int register_status;
	int irq_status;
	struct pci_dev *pd;

	CPQCICONNECTIONEXTENSION *connectionExtension;
	CPQCIADAPTEREXTENSION *adapterExtension;
	CPQCICONNECTIONS connections = 256;	//FIXME: How many do we need?
	struct _CPQCIADAPTEROBJECT* adapter;
	CPQCISTATUS error;

	CPQCIPHYSICAL mmioreg = 0x0;
	CPQCISIZE mmioregsize;

	CPQCICHANNEL channel;

	//*******************************************************************
	//  Set default values.
	//*******************************************************************

	status = 0;
	irq_status = -1;
	register_status = -1;
	adapterExtension = NULL;
	connectionExtension = NULL;
	error = CpqCiSuccess;
	adapter = NULL;
	sema_init(&connection_list_lock, 1);

#ifdef CPQCI_X86_64
	int ret;
	
	ret = register_ioctl32_conversion(CPQCI_IOCTL_PHYSMEM_ALLOC, cpqci_ioctl32_handler);
	ret |= register_ioctl32_conversion(CPQCI_IOCTL_PHYSMEM_FREE, cpqci_ioctl32_handler);
	ret |= register_ioctl32_conversion(CPQCI_IOCTL_CREATE_CHANNEL, cpqci_ioctl32_handler);
	ret |= register_ioctl32_conversion(CPQCI_IOCTL_OPEN_CHANNEL, cpqci_ioctl32_handler);
	ret |= register_ioctl32_conversion(CPQCI_IOCTL_REGISTER_WAITER, cpqci_ioctl32_handler);
	ret |= register_ioctl32_conversion(CPQCI_IOCTL_GET_INFO, cpqci_ioctl32_handler);
	ret |= register_ioctl32_conversion(SEM_CREATE, cpqci_ioctl32_handler);
	ret |= register_ioctl32_conversion(SEM_DESTROY, cpqci_ioctl32_handler);
	ret |= register_ioctl32_conversion(SEM_UP, cpqci_ioctl32_handler);
	ret |= register_ioctl32_conversion(SEM_DOWN, cpqci_ioctl32_handler);
	ret |= register_ioctl32_conversion(SEM_DOWN_TIMER, cpqci_ioctl32_handler);
	ret |= register_ioctl32_conversion(SEM_GET_VALUE, cpqci_ioctl32_handler);
	ret |= register_ioctl32_conversion(SEM_SET_VALUE, cpqci_ioctl32_handler);
	if (ret)
		printk(KERN_WARNING "cpqci:  Error registering ioctl32\n");
#endif
	//*******************************************************************
	//  Enumerate the device.
	//*******************************************************************

#ifdef ROMDRV_ENABLED
#define SIGNATURE (('$' << 0) | ('I' << 8) | ('L' << 16) | ('O' << 24))
	status = romdrv_init(SIGNATURE, 0, NULL, 0, NULL);
	if (status != 0) {
		printk(KERN_WARNING "cpqci:  Error Initializing (%d)\n", status);
		return status;
	}
#endif


	DBG("Scanning PCI for %08x\n", CPQCI_SIGNATURE);
	pd = pci_find_device(CPQCI_SIGNATURE & 0xFFFF,
			    CPQCI_SIGNATURE >> 16, NULL);
	if (!pd) {
		DBG("CpqCiDrv!init_module: can't find %08x\n",
			    CPQCI_SIGNATURE);
		status = -ENODEV;
		goto bad_exit;
	}

	pci_read_config_dword(pd, CPQCI_REG_MMIOREGBAR, (u32 *) & mmioreg);
	mmioregsize = 0x1000;

	//*******************************************************************
	//  Create a device specific adapter object.
	//*******************************************************************

	error =
	    CpqCiDrvCreateAdapter(NULL, mmioreg, mmioregsize, connections,
				  &adapter);
	if (error != CpqCiSuccess) {
		DBG("CpqCiDrv!init_module: CpqCiDrvCreateAdapter(CPQCISTATUS=%u)\n", error);
		status = -1; //FIXME: More descriptive error code
		goto bad_exit;
	}
	//*******************************************************************
	//  Create the master adapter device object used to track access to
	//  this driver.  The master object does not contain a connection, 
	//  only an adapter object.
	//*******************************************************************

	adapterExtension = kmalloc(sizeof(CPQCIADAPTEREXTENSION), GFP_KERNEL);
	DBG("(adapter)kmalloc(%d) = %p\n", sizeof(CPQCIADAPTEREXTENSION), adapterExtension);
	if (!adapterExtension) {
		status = -1;
		goto bad_exit;
	}
	memset(adapterExtension, 0, sizeof(CPQCIADAPTEREXTENSION));

	//*******************************************************************
	//  Update the adapter object's device extension information.
	//*******************************************************************

	adapterExtension->Header.ObjectType = CpqCiAdapter;
	adapterExtension->Header.AdapterExtension = adapterExtension;
	adapterExtension->Adapter = adapter;
	adapterExtension->ConnectionList = NULL;
	adapterExtension->irq = pd->irq;

	//*******************************************************************
	//*******************************************************************

	sema_init(&adapterExtension->ChannelSem[0], 20);
	sema_init(&adapterExtension->ChannelSem[1], 4);
#if LINUX_VERSION_CODE >= 0x020300
	for (channel = 0; channel < CPQCI_MAX_CHANNELS; channel++) {
		init_waitqueue_head(&adapterExtension->RecvSignal[channel]);
	}
#else
	for (channel = 0; channel < CPQCI_MAX_CHANNELS; channel++) {
		adapterExtension->RecvSignal[channel] = NULL;
	}
#endif

	CpqCi = adapterExtension;
	register_status = misc_register(&cpqci_miscdev);
#if LINUX_VERSION_CODE < 0x020600
	irq_status = request_irq(pd->irq, cpqci_intr, SA_INTERRUPT | SA_SHIRQ,
			"cpqci", adapterExtension);
#else
	irq_status = request_irq(pd->irq, (irqreturn_t (*)(int, void*, struct pt_regs *))cpqci_intr, SA_INTERRUPT | SA_SHIRQ, "cpqci", adapterExtension);
#endif
#if LINUX_VERSION_CODE >= 0x020400 && defined(CONFIG_PROC_FS)
	cpqci_entry_print_cpqci = create_proc_entry("cpqci", S_IFREG | S_IRUGO, &proc_root);
	if (cpqci_entry_print_cpqci) {
		cpqci_entry_print_cpqci->read_proc = &cpqci_read_print_cpqci;
	} else {
		printk("proc entry creation 'cpqci' FAILED.\n");
	}
#endif

	return 0;

      bad_exit:

	cleanup_module_helper(irq_status, (pd == NULL) ? 0 : pd->irq, register_status, adapter, adapterExtension);
	return status;

}

void cleanup_module(void)
{
#ifdef CPQCI_X86_64
	unregister_ioctl32_conversion(CPQCI_IOCTL_PHYSMEM_ALLOC);
	unregister_ioctl32_conversion(CPQCI_IOCTL_PHYSMEM_FREE);
	unregister_ioctl32_conversion(CPQCI_IOCTL_CREATE_CHANNEL);
	unregister_ioctl32_conversion(CPQCI_IOCTL_OPEN_CHANNEL);
	unregister_ioctl32_conversion(CPQCI_IOCTL_REGISTER_WAITER);
	unregister_ioctl32_conversion(CPQCI_IOCTL_GET_INFO);
	unregister_ioctl32_conversion(SEM_CREATE);
	unregister_ioctl32_conversion(SEM_DESTROY);
	unregister_ioctl32_conversion(SEM_UP);
	unregister_ioctl32_conversion(SEM_DOWN);
	unregister_ioctl32_conversion(SEM_DOWN_TIMER);
	unregister_ioctl32_conversion(SEM_GET_VALUE);
	unregister_ioctl32_conversion(SEM_SET_VALUE);
#endif

	cleanup_module_helper(0, CpqCi->irq, 0, CpqCi->Adapter, CpqCi);
#ifdef ROMDRV_ENABLED
	romdrv_fini();
#endif
}

int cpqci_open(struct inode *i, struct file *f)
/*++

Routine Description:

    This routine is called as a result of a user call to CreateFile().
    
Arguments:

    i - inode of open device node
    f - file handle of open device node

Return Value:

    The result of the operation.

--*/
{
#ifndef DEBUG
	//MOD_INC_USE_COUNT;
#endif
	DBG("cpqci_open pid = %d\n", current->pid);
	DBG("cpqci_open file= %p\n", f);
	DBG("cpqci_open file pointer %p refcount %d\n", f, f ? atomic_read((atomic_t*)&f->f_count):0);
	f->private_data = NULL;
	return 0;
}

int cpqci_flush(struct file *f)
/*++

Routine Description:

    This routine is called as a result of a user call to CloseHandle() or
    program termination.
    
Arguments:

    i - inode of open device node
    f - file handle of open device node

Return Value:

    The result of the operation.

--*/
{
	CPQCICONNECTIONEXTENSION *connectionExtension;

	if (!f) return 0;
	DBG("cpqci_flush file= %p\n", f);
	DBG("cpqci_flush pid = %d\n", current->pid);
	DBG("cpqci_flush (private = %p)\n", f->private_data);
	DBG("cpqci_flush file pointer %p refcount %d\n", f, f ? atomic_read((atomic_t*)&f->f_count):0);

	connectionExtension = f->private_data;
	//f->private_data = NULL;
	if (connectionExtension == NULL)
		return 0;

	return cpqci_close_helper(f, connectionExtension);
	//return cpqci_close_helper2(f);

}

int cpqci_close(struct inode *i, struct file *f) {
	CPQCICONNECTIONEXTENSION *connectionExtension;

	if (!f) return 0;
	DBG("cpqci_close file= %p\n", f);
	DBG("cpqci_close pid = %d\n", current->pid);
	DBG("cpqci_close (private = %p)\n", f->private_data);
	DBG("cpqci_close file pointer %p refcount %d\n", f, f ? atomic_read((atomic_t*)&f->f_count):0);

	connectionExtension = f->private_data;
	//f->private_data = NULL;
	if (connectionExtension == NULL)
		return 0;

	return cpqci_close_helper(f, connectionExtension);
	//return cpqci_close_helper2(f);
}

#ifdef CPQCI_X86_64
int cpqci_ioctl32_handler(unsigned int fd, unsigned int cmd, unsigned long arg, struct file *file)
{
	int err = 0;
	int i = 0;
	mm_segment_t old_fs;
	sem_value_pair32 *data32 = (sem_value_pair32 *)arg;
	sem_value_pair data64;
	struct _CPQCIDRVOSMEMHANDLE32 *mh32 = (struct _CPQCIDRVOSMEMHANDLE32*)arg;
	struct _CPQCIDRVOSMEMHANDLE mh64;
	CPQCIDRVHANDLEPARMS32 *ub32 = (CPQCIDRVHANDLEPARMS*) arg;
	CPQCIDRVHANDLEPARMS ub64;
	DBG("Entered cpqci_ioctl32_handler cmd: %p\n", cmd) ;

	switch (cmd) {
	case CPQCI_IOCTL_PHYSMEM_ALLOC:
	case CPQCI_IOCTL_PHYSMEM_FREE:
		err |= get_user(mh64.PhysicalAddress, &mh32->PhysicalAddress);
		err |= get_user(mh64.VirtualAddress, &mh32->VirtualAddress);
		err |= get_user(mh64.MemorySize, &mh32->MemorySize);
		err |= get_user(mh64.Attributes, &mh32->Attributes);
		if (err) return -EFAULT;
		old_fs = get_fs();
		set_fs(KERNEL_DS);
		err = sys_ioctl(fd, cmd, (unsigned long)&mh64);
		set_fs(old_fs);
		break;
	case CPQCI_IOCTL_CREATE_CHANNEL:
	case CPQCI_IOCTL_OPEN_CHANNEL:
		for(i = 0; i < CPQCI_OBJECT_NAME_LENGTH; i++) {
			err |= get_user(ub64.ObjectName[i], &ub32->ObjectName[i]);
		}
		err |= get_user(ub64.Priority, &ub32->Priority);
		err |= get_user(ub64.SendPackets, &ub32->SendPackets);
		err |= get_user(ub64.SendPacketSize, &ub32->SendPacketSize);
		err |= get_user(ub64.RecvPackets, &ub32->RecvPackets);
		err |= get_user(ub64.RecvPacketSize, &ub32->RecvPacketSize);
		err |= get_user(ub64.KeyPresent, &ub32->KeyPresent);
		for(i = 0; i < CPQCI_KEY_SIZE; i++) {
			err |= get_user(ub64.Key[i], &ub32->Key[i]);
			err |= get_user(ub64.Ccb[i], &ub32->Ccb[i]);
		}
		err |= get_user(ub64.Error, &ub32->Error);
		err |= get_user(ub64.TimeOut, &ub32->TimeOut);
		err |= get_user(ub64.Oshandle, &ub32->Oshandle);
		if (err) return -EFAULT;
		old_fs = get_fs();
		set_fs(KERNEL_DS);
		err = sys_ioctl(fd, cmd, (unsigned long)&ub64);
		set_fs(old_fs);
		break;
	case CPQCI_IOCTL_REGISTER_WAITER:
#ifdef IOCTL_GET_INFO
	case CPQCI_IOCTL_GET_INFO:
#endif
		old_fs = get_fs();
		set_fs(KERNEL_DS);
		err = sys_ioctl(fd, cmd, arg);
		set_fs(old_fs);
		break;
	case SEM_CREATE:
	case SEM_DESTROY:
	case SEM_UP:
	case SEM_DOWN:
	case SEM_DOWN_TIMER:
	case SEM_GET_VALUE:
	case SEM_SET_VALUE:
		err |= get_user(data64.sem, &data32->sem);
		err |= get_user(data64.value, &data32->value);
		err |= get_user(data64.rc, &data32->rc);
		if (err) return -EFAULT;
		old_fs = get_fs();
		set_fs(KERNEL_DS);
		err = sys_ioctl(fd, cmd, (unsigned long)&data64);
		set_fs(old_fs);
	}
	if(err == 0) {
		switch(cmd) {
		case CPQCI_IOCTL_PHYSMEM_ALLOC:
		case CPQCI_IOCTL_PHYSMEM_FREE:
			err |= put_user(mh64.PhysicalAddress, &mh32->PhysicalAddress);
			err |= put_user(mh64.VirtualAddress, &mh32->VirtualAddress);
			err |= put_user(mh64.MemorySize, &mh32->MemorySize);
			err |= put_user(mh64.Attributes, &mh32->Attributes);
			break;
		case CPQCI_IOCTL_CREATE_CHANNEL:
		case CPQCI_IOCTL_OPEN_CHANNEL:
			for(i = 0; i < CPQCI_OBJECT_NAME_LENGTH; i++) {
				err |= put_user(ub64.ObjectName[i], &ub32->ObjectName[i]);
			}
			err |= put_user(ub64.Priority, &ub32->Priority);
			err |= put_user(ub64.SendPackets, &ub32->SendPackets);
			err |= put_user(ub64.SendPacketSize, &ub32->SendPacketSize);
			err |= put_user(ub64.RecvPackets, &ub32->RecvPackets);
			err |= put_user(ub64.RecvPacketSize, &ub32->RecvPacketSize);
			err |= put_user(ub64.KeyPresent, &ub32->KeyPresent);
			for(i = 0; i < CPQCI_KEY_SIZE; i++) {
				err |= put_user(ub64.Key[i], &ub32->Key[i]);
				err |= put_user(ub64.Ccb[i], &ub32->Ccb[i]);
			}
			err |= put_user(ub64.Error, &ub32->Error);
			err |= put_user(ub64.TimeOut, &ub32->TimeOut);
			err |= put_user(ub64.Oshandle, &ub32->Oshandle);
			break;
		case CPQCI_IOCTL_REGISTER_WAITER:
#ifdef IOCTL_GET_INFO
		case CPQCI_IOCTL_GET_INFO:
#endif
			break;
		case SEM_CREATE:
		case SEM_DESTROY:
		case SEM_UP:
		case SEM_DOWN:
		case SEM_DOWN_TIMER:
		case SEM_GET_VALUE:
		case SEM_SET_VALUE:
			err |= put_user(data64.sem, &data32->sem);
			err |= put_user(data64.value, &data32->value);
			err |= put_user(data64.rc, &data32->rc);
		}
	}
	DBG("cpqci_ioctl32_handler exiting... err = %d\n", err) ;
	if (err) return -EFAULT;
	return err;
}
#endif

int cpqci_ioctl(struct inode *i, struct file *f, unsigned int cmd,
		unsigned long arg)
/*++

Routine Description:

    This routine is the main dispatch routine.

Arguments:

    i - inode of open device node
    f - file handle of open device node
    cmd - operation user wants us to perform
    arg - extra data for operation

Return Value:

    The function value is the status of the operation.

--*/
{
	int status = 0;
	CPQCIADAPTEREXTENSION *adapterExtension = CpqCi;
	CPQCICONNECTIONEXTENSION *c;

	//one set of parameters
	CPQCIDRVHANDLEPARMS userbuffer;
	CPQCIDRVHANDLEPARMS*ub = &userbuffer;

	CPQCICONNECTION connection;
	CPQCICHANNEL channel;

	module_info_type driver_information;
	struct _CPQCIDRVOSMEMHANDLE mh;

	DBG("cpqci_ioctl file pointer %p refcount %d\n", f, f ? atomic_read((atomic_t*)&f->f_count):0);

	switch (cmd) {
	case CPQCI_IOCTL_PHYSMEM_ALLOC:
		c = (CPQCICONNECTIONEXTENSION*) f->private_data;
		if (copy_from_user(&mh, (void*) arg, sizeof(mh))) {
			status =  -EFAULT;
			goto bad_ioctl_exit;
		}
		if (c == NULL) {
			status =  -EFAULT;
			goto bad_ioctl_exit;
		}

		down(&connection_list_lock);
		cpqci_new_connection_mem_handle((CPQCICONTEXT)f, c, mh.MemorySize);
		up(&connection_list_lock);

		if (c->MemHandleMapped == NULL) {
			status =  -EFAULT;
			goto bad_ioctl_exit;
		}

		DBG("copy_to_user(%p, %p, %x)\n", arg, c->MemHandleMapped, sizeof(struct _CPQCIDRVOSMEMHANDLE));
		if (copy_to_user((void*) arg, c->MemHandleMapped, sizeof(mh))) {
			status =  -EFAULT;
			goto bad_ioctl_exit;
		}
		break;

	case CPQCI_IOCTL_PHYSMEM_FREE:
		c = (CPQCICONNECTIONEXTENSION*) f->private_data;
		if (c == NULL) {
			status =  -EFAULT;
			goto bad_ioctl_exit;
		}
		if (copy_from_user(&mh, (void*) arg, sizeof(mh))) {
			status =  -EFAULT;
			goto bad_ioctl_exit;
		}
		//Sanity check on address passed in
		if (mh.VirtualAddress != c->MemHandleMapped->VirtualAddress) {
			status =  -EFAULT;
			goto bad_ioctl_exit;
		}
		down(&connection_list_lock);
		cpqci_delete_connection_mem_handle(c);
		up(&connection_list_lock);
		break;

	case CPQCI_IOCTL_CREATE_CHANNEL:
		DBG("Entered CREATE_CHANNEL ioctl %p\n", c);
		if (copy_from_user(ub, (CPQCIDRVHANDLEPARMS *) arg,
				   sizeof(CPQCIDRVHANDLEPARMS))) {
			status =  -EFAULT;
			goto bad_ioctl_exit;
		}
		
		c = cpqci_new_connection();
		DBG("New Connection Extension(create) %p\n", c);
		if (c == NULL) {
			status =-ENOMEM;
			goto bad_ioctl_exit;
		}

		if (ub->Priority == CpqCiNormalPriority) {
			if (ub->TimeOut != CPQCIWAITFOREVER) {
				status = down_timeout(&adapterExtension->ChannelSem[0],ub->TimeOut);
			} else {
				status = down_interruptible(&adapterExtension->ChannelSem[0]);
			}
		}
		else if (ub->Priority == CpqCiHighPriority) {
			if (ub->TimeOut != CPQCIWAITFOREVER) {
				status = down_timeout(&adapterExtension->ChannelSem[1],ub->TimeOut);
			} else {
				status = down_interruptible(&adapterExtension->ChannelSem[1]);
			}
		}

		if (status) {
			status =-EBADF;
			goto bad_ioctl_exit;
		}

		f->private_data = c;
		DBG("IOCTL_CREATE pid = %d c= %p\n", current->pid, f->private_data);
		c->Header.ObjectType = CpqCiConnection;
		if (adapterExtension->Header.ObjectType == CpqCiAdapter) {
			ub->Error = CpqCiDrvCreateChannel(f,
				adapterExtension->Adapter,
				ub->SendPackets, ub->SendPacketSize,
				ub->RecvPackets, ub->RecvPacketSize,
				&ub->Ccb, ub->Priority, CpqCiUserMode,
				ub->KeyPresent == CpqCiTrue ? &ub->Key : NULL,
				&connection, &channel);


			//  Construct an OS device name so that the usermode thread can obtain
			//  an OS-aware context to this connection.
			//  (this connection name isn't actually used in linux
			//   and can probably be removed)
			if (ub->Error == CpqCiSuccess) {
				c->Connection = connection;
				c->Channel = channel;
				sprintf(ub->ObjectName, "%s%u", CPQCI_CONNECTION_NAME, (unsigned int)connection);
				
				CpqCiShareContext(f, adapterExtension->Adapter, c->Connection);
//                            DbgDump( adapterExtension->Adapter, connection );
			} else
				//  If there were no more channels, clear the waiter signal
				//  for this priority queue.
			if (ub->Error == CpqCiErrNoMoreChannels) {
				//FIXME: Need logic
			} else
				//  A valid channel was removed from the free list, however, an error
				//  occured sometime after the channel was removed from the list.
				//  Since other callers could have seen an empty list during the time
				//  the channel was removed from the free list, we must signal the
				//  target priority event signal to "wake" any potential waiters if
				//  a channel was temporarily removed from the free list and replaced
				//  because the CpqCiDrvCreateChannel() routine failed.
			if (channel != CPQCI_INVALID_CHANNEL) {
				if (ub->Priority == CpqCiNormalPriority)
					up(&adapterExtension->ChannelSem[0]);
				else if (ub->Priority == CpqCiHighPriority)
					up(&adapterExtension->ChannelSem[1]);
			}
		} else {
			ub->Error = CpqCiErrBadObjectType;
			status = -EBADF;
			goto bad_ioctl_exit;
		}

		if (ub->Error == CpqCiSuccess) {
			CPQCITYPEQUAD ref;
    			CpqCiOsGetRefCount(f, (CPQCITYPEQUAD*) &ref);
			DBG("*** Create Connection %d ref %d channel %d.\n", connection, ref.uint, channel);
		} else {
			DBG("CpqCiDrv!DriverDispatch: connection attempt failed.\n");
			status = -1;
			goto bad_ioctl_exit;
		}
		DBG("copy_to_user(%p, %p, %x)\n", arg, ub, sizeof(CPQCIDRVHANDLEPARMS));
		copy_to_user((CPQCIDRVHANDLEPARMS *) arg, ub,
			     sizeof(CPQCIDRVHANDLEPARMS));

		break;

	/***************************************************************************
        //  This IOCTL is called to open an already existing connection between the
        //  usermode consumer and a channel.  This IOCTL can only operate on
        //  adapter objects.
        ****************************************************************************/

	case CPQCI_IOCTL_OPEN_CHANNEL:
		if (copy_from_user(ub, (CPQCIDRVHANDLEPARMS *) arg,
				   sizeof(CPQCIDRVHANDLEPARMS))) {
			status = -EFAULT;
			goto bad_ioctl_exit;
		}

		c = cpqci_new_connection();
		DBG("New Connection Extension(open) %p\n", c);
		if (c == NULL) {
			status =-ENOMEM;
			goto bad_ioctl_exit;
		}

		f->private_data = c;
		c->Header.ObjectType = CpqCiConnection;
		if (adapterExtension->Header.ObjectType == CpqCiAdapter) {
			ub->Error = CpqCiDrvOpenChannel(f,
					adapterExtension->Adapter,
					CpqCiUserMode, &ub->Ccb,
					&ub->Key, &connection,
					&channel);

			//  Construct an OS device name so that the usermode thread can obtain
			//  an OS-aware context to this connection.
			//  (this connection name isn't actually used in linux
			//   and can probably be removed)
			if (ub->Error == CpqCiSuccess) {
				c->Connection = connection;
				c->Channel = channel;
				sprintf(ub->ObjectName, "%s%u", CPQCI_CONNECTION_NAME, (unsigned int)connection);

				CpqCiShareContext(f, adapterExtension->Adapter, c->Connection);
//                            DbgDump( adapterExtension->Adapter, connection );
			}
		} else {
			ub->Error = CpqCiErrBadObjectType;
			status = -EBADF;
			goto bad_ioctl_exit;
		}

		if (ub->Error == CpqCiSuccess) {
			CPQCITYPEQUAD ref;
    			CpqCiOsGetRefCount(f, (CPQCITYPEQUAD*) &ref);
			DBG("*** Open Connection %d ref %d channel %d.\n", connection, ref.uint, channel);
		} else {
			DBG("CpqCiDrv!DriverDispatch: CpqCiDrvOpenChannel(CPQCISTATUS=%u).\n", ub->Error);
			status = -1;
			goto bad_ioctl_exit;
		}
		copy_to_user((CPQCIDRVHANDLEPARMS *) arg, ub,
			     sizeof(CPQCIDRVHANDLEPARMS));

		break;

	/***************************************************************************
        //  This IOCTL is called to register a thread as a waiter on the arrival
        //  of receive messages.
        ****************************************************************************/

	case CPQCI_IOCTL_REGISTER_WAITER:
		c = f->private_data;

		//  We only want to register a thread already connected to the channel.
		if ((c != NULL) && (c->Header.ObjectType == CpqCiConnection)) {
			CpqCiDrvRegisterWaiter(f,
					       adapterExtension->Adapter,
					       c->Connection);

			status=0;
			DBG("IOCTL_REGISTER_WAITER sleep for %d ms\n", arg);
			interruptible_sleep_on_timeout(
				&adapterExtension->RecvSignal[c->Channel],
				MS_TO_JIFFIES(arg));
			if (signal_pending(current)) {
				DBG("Woken up abruptly\n");
				status = -EINTR;
				//FIXME: Unregister the waiter 
			}
		} else {
			status = -EBADF;
			DBG("CpqCiDrv!DriverDispatch: CpqCiDrvRegisterWaiter(BADCONNECTION).\n");
			// No cleanup necessary
		}

		break;

#ifdef IOCTL_GET_INFO
	case CPQCI_IOCTL_GET_INFO:
		memset(&driver_information, 0, sizeof(module_info_type));

		strcpy(driver_information.name, "cpqci.o");
		strcpy(driver_information.revision_number, CPQEFSVER);
		strcpy(driver_information.purpose, "Compaq iLO Communication Driver for Linux");
		
		driver_information.revision_date.year = BUILD_YEAR;
		driver_information.revision_date.month= BUILD_MONTH;
		driver_information.revision_date.day  = BUILD_DAY;

		if (arg) copy_to_user((module_info_type*) arg, &driver_information, sizeof(module_info_type));

		break;
#endif
	/***************************************************************************
        //  The specified IOCTL is not supported.
        ****************************************************************************/

	default:
		DBG("default ioctl %x\n", cmd);
		//First check for sem ioctl
		if (cmd >= SEM_CREATE && cmd <= SEM_SET_VALUE) {
			status = sem_ioctl(i, f, cmd, arg);
		} else {
			status = -ENOIOCTLCMD;
		}
	}

	DBG("IOCTL cmd %x arg %p returns %d\n", cmd, arg, status);
	return status;

bad_ioctl_exit:

	if (arg && ub) {
		copy_to_user((CPQCIDRVHANDLEPARMS *) arg, ub,
		     sizeof(CPQCIDRVHANDLEPARMS));
	}

	if (c) { 
		DBG("(connection)cpqci_delete_connection(%p)\n", c);
		cpqci_delete_connection(c);
		f->private_data = NULL; //c ??
	}

	return status;
}

/*
** cpqci_mmap
**
** Remaps a physical page range into the virtual range specified by vma.
** The only way we should ever get into this function is as a result of
** a call to do_mmap in CpqCiDrvOsMap.  Any other call (e.g. from mmap
** in user space) will fail.
*/
int cpqci_mmap(struct file *f, struct vm_area_struct *vma)
{
	unsigned long size = vma->vm_end - vma->vm_start;
	CPQCICONNECTIONEXTENSION *e = f->private_data;
	unsigned long pa;
	unsigned int nopage_method = 0;

	DBG("cpqci_mmap file pointer %p refcount %d\n", f, f ? atomic_read((atomic_t*)&f->f_count):0);

	if (e == NULL) return -EINVAL;

	pa = e->paddr;

	if (pa < (unsigned long)pci_mem_start) {
		DBG("Mapping regular memory at %p (high mem %p)\n", pa, pci_mem_start);
		nopage_method = 0;
	} else {       
		DBG("Mapping PCI memory at %p (high mem %p)\n", pa, pci_mem_start);
		nopage_method = 0;
	}

	if ((e->MemHandleMalloc != NULL) && (pa == e->MemHandleMalloc->PhysicalAddress)) {
		DBG("This is the mapping of the additional memory buffer with phys %p ...\n", e->MemHandleMalloc->PhysicalAddress);
		nopage_method = 0;
	}



#if LINUX_VERSION_CODE > 0x020300
	vma->vm_flags |= VM_RESERVED;
	vma->vm_private_data= e;
	DBG("Manipulating vma->vm_private_data = %p\n", vma->vm_private_data);
#endif


	if (nopage_method == 0) {
		vma->vm_flags |= VM_IO;
#if defined(MREMAP_PAGE_RANGE) || LINUX_VERSION_CODE >= 0x020600
               	if (remap_page_range(vma, vma->vm_start, pa, size, vma->vm_page_prot))
		                        return -EAGAIN;
#else
               	if (remap_page_range(vma->vm_start, pa, size, vma->vm_page_prot))
	                        return -EAGAIN;
#endif
    		vma->vm_ops = &cpqci_vm_ops_high;
	}

	if (nopage_method == 1) {
		vma->vm_flags &= ~VM_IO;
    		vma->vm_ops = &cpqci_vm_ops_regular;
	}

	DBG("Protection bits %p\n", vma->vm_page_prot);
	DBG("Flag bits %p\n", vma->vm_flags);

	return 0;
}

/*
** cpqci_intr
**
** Interrupt handler for the channel interface.  This function gets
** a list of channels with pending messages and wakes up any processes
** waiting on those channels.
*/
#if LINUX_VERSION_CODE < 0x020600
void cpqci_intr(int irq, void *dev_id, struct pt_regs *regs)
#else
irqreturn_t cpqci_intr(int irq, void *dev_id, struct pt_regs *regs)
#endif
{
	CPQCICHANNEL channel;
	CPQCICHANNELS channelList = 0;
	CPQCICHANNELS deferredList = 0;
	CPQCIADAPTEREXTENSION *adapterExtension = dev_id;
	int i;
	int rc;

	//This is to ensure that this is really our interrupt
#if LINUX_VERSION_CODE < 0x020600
	if (adapterExtension != CpqCi) return;
#else
	if (adapterExtension != CpqCi) return IRQ_NONE;
#endif

	DBG("Interrupt\n");

	//  Spin down the pending channel list until there are no more
	//  pending channels.  This has the effect of waking up waiters
	//  on the respective channel receive FIFO.
	
	for (i=0; i<32; i++) {
		rc = CpqCiDrvGetPendingChannels(NULL, adapterExtension->Adapter, &channel, &channelList, &deferredList);
		if (rc != CpqCiSuccess) {
			break;
		}

		DBG("PendingChannel = %d\n", channel);
		if (CpqCiDrvUnregisterWaiter (NULL, adapterExtension->Adapter, channel) == CpqCiSuccess) {
			deferredList |= (1 << channel);
		}
#if LINUX_VERSION_CODE >= 0x020300
		wake_up_all(&adapterExtension->RecvSignal[channel]);
#else
		wake_up(&adapterExtension->RecvSignal[channel]);
#endif
	}
	if (rc == CpqCiSuccess) {
		DBG("BUG: Still Pending Channels, specifically Channel %d\n", channel);
	}

#if LINUX_VERSION_CODE >= 0x020600
	return IRQ_HANDLED;
#endif
	DBG("Interrupt over\n");
}


//****************************************************************
//*         VMA OPERATIONS FOR SHARED MEMORY.
//****************************************************************
//
static unsigned long cpqci_do_mmap(CPQCICONTEXT Context, CPQCIPHYSICAL PhysicalAddress, CPQCISIZE MappedSize, unsigned long prot)
{
	unsigned long va;

	struct file *f = (struct file *) Context;
	CPQCICONNECTIONEXTENSION *e = f->private_data;
	DBG("cpqci_do_mmap file pointer %p refcount %d\n", f, f ? atomic_read((atomic_t*)&f->f_count):0);
	e->paddr = PhysicalAddress;
	DBG("do_mmap(%p, %d)\n", PhysicalAddress, MappedSize);
	down_write(&current->mm->mmap_sem);
	va = do_mmap(f, 0, MappedSize, prot, MAP_SHARED, 0);
	up_write(&current->mm->mmap_sem);
	DBG("%p\n", va);
	//The high number would indicate a failure to map properly
	if (va > 0xfffffc00) va = 0;
	return va;
}

static unsigned long cpqci_do_munmap(CPQCICONTEXT Context, CPQCIPHYSICAL VirtualAddress, CPQCISIZE MappedSize)
{
			DBG("munmap(%p, %d)\n", VirtualAddress, MappedSize);
#if LINUX_VERSION_CODE >= 0x020402
#if defined(MUNMAP4)
			if (current->mm) {
				down_write(&current->mm->mmap_sem);
				do_munmap(current->mm, (unsigned long) VirtualAddress, MappedSize, 1);
				up_write(&current->mm->mmap_sem);
			}
#else
			if (current->mm) {
				down_write(&current->mm->mmap_sem);
				do_munmap(current->mm, (unsigned long) VirtualAddress, MappedSize);
				up_write(&current->mm->mmap_sem);
			}
#endif
#endif
}

// **********************************************************************
//VMA operations
// **********************************************************************

/*
 * Compatibility  
 */
static void* vma_private_data(struct vm_area_struct *vma)
{
#if LINUX_VERSION_CODE > 0x020300
    return vma->vm_private_data;
#elif LINUX_VERSION_CODE > 0x020200
    struct file *fp = vma->vm_file;
    if (fp) return fp->private_data;
    return NULL;
#endif
}

#if 0
static void cpqci_vma_open(struct vm_area_struct *vma)
{ 
	int len;
	CPQCIADAPTER adapter;
	CPQCICONNECTIONEXTENSION* e;
	CPQCICONNECTIONEXTENSION** pe;
	CPQCICONNECTION Connection;
	CPQCICHANNEL Channel;

	DBG("cpqci_vma_open start=%p (private %p) \n", vma->vm_start, vma_private_data(vma));

	//MOD_INC_USE_COUNT; 

	if (!CpqCi) return;
	if (!CpqCi->Adapter) return;

	adapter = CpqCi->Adapter;
	len = vma->vm_end - vma->vm_start;

	e = (CPQCICONNECTIONEXTENSION*) vma_private_data(vma);
	if (e == NULL) {
		DBG("vma private data does not match connection extension\n");
		return;
	}

	pe = cpqci_find_connection(e);
	DBG("pe = %p\n");
	if ((pe == NULL) || (*pe == NULL)) {
		DBG("vma private data does not match connection extension\n");
		return;
	}

	Connection = e->Connection;
	Channel = e->Channel;

	if (e->shmaddr == vma->vm_start) {
		//Increase the channel ref count
		DBG("Increasing Channel %d Refcount from %ld by one\n", Channel, adapter->ChannelObject[Channel].ReferenceCount);
		cpqci_atomic_inc((vu64_t*)&adapter->ChannelObject[Channel].ReferenceCount);
		DBG("Channel %d Refcount is now %ld\n", Channel, adapter->ChannelObject[Channel].ReferenceCount);
		//Increase the connection object ref count 
		DBG("Increasing Connection %d Refcount from %ld by one\n", Connection, adapter->ConnectionObject[Connection].ReferenceCount);
		cpqci_atomic_inc((vu64_t*)&adapter->ConnectionObject[Connection].ReferenceCount);
		DBG("Connection %d Refcount is now %ld\n", Connection, adapter->ConnectionObject[Connection].ReferenceCount);
	}
}
#endif

static void cpqci_vma_open(struct vm_area_struct *vma)
{ 
#if LINUX_VERSION_CODE > 0x020300
	DBG("cpqci_vma_open start=%p (private %p) \n", vma->vm_start, vma->vm_private_data);
#endif
}

static void cpqci_vma_close(struct vm_area_struct *vma)
{ 
#if LINUX_VERSION_CODE > 0x020300
	DBG("cpqci_vma_close start=%p (private %p) \n", vma->vm_start, vma->vm_private_data);
#endif
}

#if LINUX_VERSION_CODE > 0x020300
static struct page* cpqci_vma_nopage_regular(struct vm_area_struct *vma, unsigned long address, int write) 
#else
static unsigned long cpqci_vma_nopage_regular(struct vm_area_struct *vma, unsigned long address, int write)
#endif
{
	unsigned long offset;
#if LINUX_VERSION_CODE > 0x020300
	struct page* page = NOPAGE_SIGBUS;
#else
	struct page* page = NULL;
#endif
	CPQCICONNECTIONEXTENSION* e = (CPQCICONNECTIONEXTENSION*) vma_private_data(vma);
	CPQCICONNECTION connection ;
    	struct _CPQCIADAPTEROBJECT *pao;
	CPQCIDRVOSMEMHANDLE mh;
	unsigned long va;
//FIXME: critical section ??
	//
	DBG("nopage: vma_start= %p\n", vma->vm_start);
	DBG("nopage: vma_end= %p\n", vma->vm_end);
	DBG("nopage: address = %p\n", address);
	offset = (address - vma->vm_start);
	DBG("nopage: offset = %p\n", offset);
	if (offset >= (vma->vm_end-vma->vm_start)) goto nopage_regular_out;

	if (e == NULL) goto nopage_regular_out;

	connection = e->Connection;
    	pao = CpqCi->Adapter;
	mh = (CPQCIDRVOSMEMHANDLE) GetMemoryHandle(pao, connection);
	va = (unsigned long)__va(mh->PhysicalAddress + offset);

#ifdef DEBUG
	DBG("nopage: va = %p\n", va);
	for (i=0; i<16; i++) {
		DBG("%2.2x ", ((unsigned char*)va)[i]);
	}
	DBG("\n");
#endif
	page = virt_to_page(va);
	DBG("nopage: page = %p\n", page);
	get_page(page);

nopage_regular_out:

#if LINUX_VERSION_CODE > 0x020300
	//return NULL;
	return page;
#else
	return va;
#endif

}

#if LINUX_VERSION_CODE > 0x020300
static struct page* cpqci_vma_nopage_high(struct vm_area_struct *vma, unsigned long address, int write) 
#else
static unsigned long cpqci_vma_nopage_high(struct vm_area_struct *vma, unsigned long address, int write) 
#endif
{
#if LINUX_VERSION_CODE > 0x020300
	struct page* page = NOPAGE_SIGBUS;
#else
	struct page* page = NULL;
#endif
	DBG("nopage: vma_start= %p\n", vma->vm_start);
	DBG("nopage: vma_end= %p\n", vma->vm_end);
	DBG("nopage: address = %p\n", address);
#if LINUX_VERSION_CODE > 0x020300
	return page;
#else
	return 0;
#endif
}


//****************************************************************
//*         DRIVER MEMORY MANAGEMENT ROUTINES.
//****************************************************************

#define _cached         (0 << 0)
#define _uncached       (1 << 0)

#define _kernelmode     (0 << 1)
#define _usermode       (1 << 1)

#define _allocated      (0 << 2)
#define _mapped         (1 << 2)

#define _noncontiguous  (0 << 3)
#define _contiguous     (1 << 3)

#define _memoryobj      (0 << 4)	// 3-bits of info
#define _objecttype     (7 << 4)	// 3-bits of info

CPQCIDRVOSHANDLE CpqCiDrvOsMallocContiguous(CPQCICONTEXT Context,
					    CPQCISIZE AllocationSize)
/*++

Routine Description:
    This routine allocates physically contiguous memory for the requested size.

Arguments:
    AllocationSize - The size in bytes of the memory to allocate.
    
Return Value:
    A handle representing the memory region allocated.  A value of NULL (0)
    indicates no more resources to fulfill the request.

Caveats:
    The allocated memory MUST have the following attributes:

        CACHED
        NON-PAGED
        CONTIGUOUS

--*/
{
	CPQCIDRVOSMEMHANDLE mh = NULL;
	void *va = NULL;
	int i;

	//  Make sure the requested allocation size is valid.
	if (AllocationSize) {
		//  Allocate a memory handle from non-paged pool.
		mh =
		    kmalloc(sizeof(struct _CPQCIDRVOSMEMHANDLE), GFP_KERNEL);
		DBG("(mh)kmalloc(%d) = %p\n", sizeof(struct _CPQCIDRVOSMEMHANDLE), mh);
		if (mh) {
			//  Allocate the actual memory from non-paged pool, physically contiguous.
			DBG("MallocContiguous(2^%d pages) = ", get_order(AllocationSize));
#ifdef CPQCI_X86_64
			va = (void*) __get_free_pages(GFP_KERNEL | GFP_DMA, get_order(AllocationSize));
#else
			va = (void*) __get_free_pages(GFP_KERNEL, get_order(AllocationSize));
#endif
			DBG("%p\n", va);
			if (va) {
				//This is to prevent junk pointers, etc. 
				memset(va, 0, AllocationSize);
				//  Update the memory handle.
				mh->PhysicalAddress = __pa(va);
				mh->VirtualAddress = va;
				mh->MemorySize = AllocationSize;
				mh->Attributes =
				    _memoryobj | _cached | _kernelmode |
				    _allocated | _contiguous;
				for (i = 0; i < AllocationSize; i+=PAGE_SIZE) {
#if LINUX_VERSION_CODE < 0x020300
					set_bit(PG_reserved, &(virt_to_page(va + i)->flags));
#elif LINUX_VERSION_CODE < 0x020600
					mem_map_reserve(virt_to_page(va + i));
					atomic_inc(&virt_to_page(va + i)->count);
#else
					SetPageReserved(virt_to_page(va + i));
					atomic_inc(&virt_to_page(va + i)->_count);
#endif
					set_bit(PG_locked, &virt_to_page(va + i)->flags);
				}
			} else {
				DBG("(failed malloc contiguous)kfree(%p)\n", mh);
				kfree(mh);
				mh = NULL;
			}
		}
	}

	return mh;
}

CPQCIDRVOSHANDLE CpqCiDrvOsMalloc(CPQCICONTEXT Context,
				  CPQCISIZE AllocationSize)
/*++

Routine Description:
    This routine allocates noncontiguous memory for the requested size.

Arguments:
    AllocationSize - The size in bytes of the memory to allocate.
    
Return Value:
    A handle representing the memory region allocated.  A value of NULL (0)
    indicates no more resources to fulfill the request.

Caveats:
    The allocated memory MUST have the following attributes:

        CACHED
        NON-PAGED
        CONTIGUOUS

--*/
{
	CPQCIDRVOSMEMHANDLE mh = NULL;
	void *va = NULL;

	//  Make sure the requested allocation size is valid.
	if (AllocationSize) {
		//  Allocate a memory handle from non-paged pool.
		mh = kmalloc(sizeof(struct _CPQCIDRVOSMEMHANDLE), GFP_KERNEL);
		DBG("(mh)kmalloc(%d) = %p\n", sizeof(struct _CPQCIDRVOSMEMHANDLE), mh);
		if (mh) {
			//  Allocate the actual memory from non-paged pool, physically contiguous.
			DBG("Malloc(%x) = ", AllocationSize);
			va = kmalloc(AllocationSize, GFP_KERNEL);
			DBG("%p\n", va);
			if (va) {
				//This is to prevent junk pointers, etc. 
				memset(va, 0, AllocationSize);
				//  Update the memory handle.
				mh->PhysicalAddress = __pa(va);
				mh->VirtualAddress = va;
				mh->MemorySize = AllocationSize;
				mh->Attributes =
				    _memoryobj | _cached | _kernelmode |
				    _allocated | _noncontiguous;
			} else {
				DBG("(failed malloc)kfree(%p)\n", mh);
				kfree(mh);
				mh = NULL;
			}
		}
	}

	return mh;
}

void CpqCiDrvOsFree(CPQCICONTEXT Context, CPQCIDRVOSHANDLE MemoryHandle)
{
	CPQCIDRVOSMEMHANDLE mh = MemoryHandle;
	int i;

	//mh needs to be defined, otherwise noop
	if (!mh) return;

	if (mh->VirtualAddress != 0) {
		if (mh->Attributes & _mapped) {
			if (mh->Attributes & _usermode) {
				DBG("cpqci_do_munmap(%p, %d)\n", mh->VirtualAddress, mh->MemorySize);
				cpqci_do_munmap(Context, (unsigned long) mh->VirtualAddress, mh->MemorySize);
			} else {
				DBG("iounmap(%p, %d)\n", mh->VirtualAddress, mh->MemorySize);
				iounmap(mh->VirtualAddress);
			}
		} else {
			if (mh->Attributes & _contiguous) {
				DBG("KfreeContiguous(%p, 2^%d pages)\n", mh->VirtualAddress, get_order(mh->MemorySize));
				if (mh->VirtualAddress != NULL) { 
					for (i = 0; i < mh->MemorySize; i += PAGE_SIZE) {
						clear_bit(PG_locked, &virt_to_page(mh->VirtualAddress + i)->flags);
#if LINUX_VERSION_CODE < 0X020600
						atomic_dec(&virt_to_page(mh->VirtualAddress + i)->count);
#else
						atomic_dec(&virt_to_page(mh->VirtualAddress + i)->_count);
#endif
#if LINUX_VERSION_CODE < 0x020300
						clear_bit(PG_reserved, &(virt_to_page(mh->VirtualAddress + i)->flags));
#elif LINUX_VERSION_CODE < 0X020600
						mem_map_unreserve(virt_to_page(mh->VirtualAddress + i));
#else
						ClearPageReserved(virt_to_page(mh->VirtualAddress + i));
#endif
					}

					free_pages((unsigned long) mh->VirtualAddress, get_order(mh->MemorySize));
				}			
			} else {
				DBG("Kfree(%p, %d)\n", mh->VirtualAddress, mh->MemorySize);
				kfree(mh->VirtualAddress);
			}
		}
	}

	if (mh) {
		DBG("Kfree(mh = %p)\n", mh);
		kfree(mh);
	}

	return;
}

CPQCIDRVOSHANDLE
CpqCiDrvOsDupHandle(CPQCICONTEXT Context, CPQCIDRVOSHANDLE Handle)
{
	CPQCIDRVOSHANDLE mh;

	if (!Handle) return NULL;

	mh = kmalloc(sizeof(struct _CPQCIDRVOSMEMHANDLE), GFP_KERNEL);
	DBG("(OSMEMHANDLE)Kmalloc(%d) = %p\n", sizeof(struct _CPQCIDRVOSMEMHANDLE), mh);
	if (!mh) return mh;

	memcpy(mh, Handle, sizeof(struct _CPQCIDRVOSMEMHANDLE));
	return mh;
}

CPQCIDRVOSHANDLE
CpqCiDrvOsMap(CPQCICONTEXT Context, CPQCIPHYSICAL PhysicalAddress,
	      CPQCISIZE MappedSize,
	      CPQCIMODE AddressMode, CPQCICACHETYPE CacheType)
{
	CPQCIDRVOSMEMHANDLE mh = NULL;
	void *va = NULL;

	//  Make sure the requested allocation size is valid.
	if (MappedSize) {
		//  Allocate a memory handle from non-paged pool.
		mh = kmalloc(sizeof(struct _CPQCIDRVOSMEMHANDLE), GFP_KERNEL);
		DBG("(mh)kmalloc(%d) = %p\n", sizeof(struct _CPQCIDRVOSMEMHANDLE), mh);
		if (mh) {
			if (AddressMode == CpqCiKernelMode) {
				DBG("ioremap(%p, %d) = ", PhysicalAddress, MappedSize);
				//va = ioremap_nocache(PhysicalAddress, MappedSize);
				va = __ioremap(PhysicalAddress, MappedSize, _PAGE_PCD);
				DBG("%p\n", va);
			} else {
				DBG("mmap(%p, %d) = ", PhysicalAddress, MappedSize);
				va = (void*) cpqci_do_mmap(Context, PhysicalAddress, MappedSize, PROT_READ | PROT_WRITE);
				DBG("%p\n", va);
			}
		}
	}
	//  If a valid va was obtained, fill in the memory handle.
	if (va) {
		mh->PhysicalAddress = PhysicalAddress;
		mh->VirtualAddress = va;
		mh->MemorySize = MappedSize;
		mh->Attributes = _memoryobj | _mapped;
		mh->Attributes |=
		    ((AddressMode == CpqCiUserMode) ? _usermode :
		     _kernelmode);
		mh->Attributes |=
		    ((CacheType == CpqCiCached) ? _cached : _uncached);
	} else if (mh) {
		//  If we could not obtain a valid mapping, but we allocated a memory handle,
		//  return the memory handle to the free pool.
		DBG("(failed map)kfree(%p)\n", mh);
		kfree(mh);
		mh = NULL;
	}

	return mh;
}

CPQCIVIRTUAL CpqCiDrvOsGetVA(CPQCICONTEXT Context,
			     CPQCIDRVOSHANDLE MemoryHandle)
{
	if (!MemoryHandle) {
		DBG("CpqCiDrvOSGetVA: MemoryHandle == NULL\n");
		return 0;
	}
	DBG("CpqCiDrvOSGetVA: Physical Address == %p\n", ((CPQCIDRVOSMEMHANDLE)MemoryHandle)->PhysicalAddress);
	DBG("CpqCiDrvOSGetVA: Virtual Address == %p\n", ((CPQCIDRVOSMEMHANDLE)MemoryHandle)->VirtualAddress);
	return (CPQCIVIRTUAL) ((CPQCIDRVOSMEMHANDLE)
			       (MemoryHandle))->VirtualAddress;
}

CPQCIPHYSICAL CpqCiDrvOsGetPA(CPQCICONTEXT Context,
			      CPQCIDRVOSHANDLE MemoryHandle)
{
	if (!MemoryHandle) {
		DBG("CpqCiDrvOSGetVA: MemoryHandle == NULL\n");
		return 0;
	}
	return *((CPQCIPHYSICAL *)
		 (&((CPQCIDRVOSMEMHANDLE) (MemoryHandle))->PhysicalAddress));
}

CPQCISIZE CpqCiDrvOsGetSize(CPQCICONTEXT Context,
			    CPQCIDRVOSHANDLE MemoryHandle)
{
	if (!MemoryHandle) {
		DBG("CpqCiDrvOSGetVA: MemoryHandle == NULL\n");
		return 0;
	}
	return (CPQCISIZE) ((CPQCIDRVOSMEMHANDLE)
			    (MemoryHandle))->MemorySize;
}

CPQCICACHETYPE CpqCiDrvOsGetCacheType(CPQCICONTEXT Context,
				      CPQCIDRVOSHANDLE MemoryHandle)
{
	if (!MemoryHandle) {
		DBG("CpqCiDrvOSGetVA: MemoryHandle == NULL\n");
		return 0;
	}
	return (((CPQCIDRVOSMEMHANDLE) (MemoryHandle))->Attributes &
		_uncached) ? CpqCiUncached : CpqCiCached;
}

CPQCIMODE CpqCiDrvOsGetMode(CPQCICONTEXT Context,
			    CPQCIDRVOSHANDLE MemoryHandle)
{
	if (!MemoryHandle) {
		DBG("CpqCiDrvOSGetVA: MemoryHandle == NULL\n");
		return 0;
	}
	return (((CPQCIDRVOSMEMHANDLE) (MemoryHandle))->Attributes &
		_usermode) ? CpqCiUserMode : CpqCiKernelMode;
}


CPQCIVOID CpqCiDrvOsStall( 	CPQCICONTEXT Context, 
				CPQCITIMEOUT MicroSeconds)
{
	CpqCiUSleep(MicroSeconds);
}

CPQCISTATUS
CpqCiOsGetRefCount(
	CPQCICONTEXT Context, 
	CPQCITYPEQUAD* RefCount
   )
{
	struct file* fp = (struct file*) Context;
	
	memset(RefCount, 0, sizeof(CPQCITYPEQUAD));

	DBG("RefCount of File %p\n", fp);

	if (!fp) return CpqCiSuccess;

	RefCount->uint = atomic_read((atomic_t*)&fp->f_count);
	DBG("RefCount: %p\n", RefCount->uint);

	return CpqCiSuccess;
}

CPQCIVIRTUAL
CpqCiMemSet(
	CPQCIVIRTUAL dest, 
	u8_t t, 
	CPQCISIZE s) 
{
	return memset(dest, t, s);
}

#ifndef SLES7
CPQCISIZE CpqCiSNPrintf(
	CPQCICHAR* buf, 
	CPQCISIZE size, 
	const CPQCICHAR* fmt, 
	...
	) 
{
	va_list args;
	int i;
	va_start(args, fmt);
	i=vsnprintf(buf, size, fmt, args);
	va_end(args);
	return i;
}
#endif 

CPQCISIZE CpqCiSPrintf(
	CPQCICHAR* buf, 
	const CPQCICHAR* fmt, 
	...
	) 
{
	va_list args;
	int i;
	va_start(args, fmt);
	i=vsprintf(buf, fmt, args);
	va_end(args);
	return i;
}

void* 
CpqCiGetCurrentTask()
{
	void* curr = (void*) current;
	return curr;
}

#if 0
CPQCIVOID
CpqCiOsDupConnection(
	CPQCICONTEXT Context,
    	CPQCICONNECTIONOBJECT* DupObject,
	CPQCICONNECTIONOBJECT* ConnectionObject
)
{
    //Copy mapped memory for Mapped Memory 
    DupObject->MappedMemoryHandle = 
	    CpqCiDrvOsDupHandle(Context, ConnectionObject->MappedMemoryHandle);
    DBG("Duped MemoryHandle to %p\n", DupObject->MappedMemoryHandle);
    //Copy mapped memory for Doorbell Register
    DupObject->MappedDoorbellHandle = 
	    CpqCiDrvOsDupHandle(Context, ConnectionObject->MappedDoorbellHandle);
    DBG("Duped DoorbellHandle to %p\n", DupObject->MappedDoorbellHandle);

    //Duplicate the Ccb
    CpqCiCopyBuffer(	&DupObject->MappedCcbCopy, 
                        &ConnectionObject->MappedCcbCopy,
                        sizeof(CPQCICCB) );

}
#endif
