/** 
 * Pointer 
 *   Pointer used to represent a warning or critical disk space usage 
 *   threshold. The manages drawing itself and tracking the percentage 
 *   that it represents. 
 */
import java.awt.*;
// Pointer class
// This draws a pointer to represent a percentage
//
class Pointer 
{
   private int column;
   private double range;         // Pointer range in pixels
   private int percent;	         // Pointer percentage
   private int originalValue;    // Value used to determine if modified
   private int[] yPoints;        // Row array to draw pointer triangle
   private Font font;
   private Color color;
   private Color enabledColor;	// Enabled pointer color
   private Color textColor;      // Text color
   private boolean selected;
   private boolean available;    // If not available threshold not displayed
   private boolean writeAccess;  // If no write access threshold cannot be modified
   private Threshold applet;
   // Constants
   public static final int FONT_SIZE  = 10;
   public static final int ROW        =  1;   // Row or y position
   public static final int WIDTH      = 24;   
   public static final int HEIGHT     = 20;
   public static final int HALF_WIDTH = WIDTH / 2;   
   // Used in calculating 1/2 triangle width at a point y
   public static final double DELTA = (double)WIDTH / ((double)HEIGHT * 2.0);
   public static final Color DISABLED_COLOR = Color.lightGray;
   public static final double MIN_RANGE   = 100.0;
   public static final int    MAX_PERCENT =  100;  // Pointer percentage range
   public static final int    MIN_PERCENT =   6;  
   // Constructor
   public Pointer(int value, Color background, Color foreground, 
                  boolean access, Threshold app) 
   {
      applet = app;
      writeAccess = true;
      percent = -1;
      selected = false;
      available = true;
      enabledColor = background;
      textColor = foreground;
      font = new Font("Arial", Font.PLAIN, FONT_SIZE);
      // Set the pointer range
      range = (double)(applet.APP_WIDTH - WIDTH);
      if (range <= MIN_RANGE) range = MIN_RANGE;
      
      // Make sure we have a valid percent
      if ((value < 0) || (value > MAX_PERCENT)) 
      {
         value = 0;
      }
      setPercent(value);
      originalValue = getPercent();
      writeAccess = access;
      // Initialize row points array
      yPoints = new int[3];
      yPoints[0] = ROW;
      yPoints[1] = ROW;
      yPoints[2] = ROW + HEIGHT;
   }
   // Selection
   public void setSelected(boolean select) 
   {
      selected = select;
   }
   public boolean isSelected() 
   {
      return selected;
   }
   // Is the pointer in a new location
   public boolean isModified()
   {
      return (originalValue != getPercent());
   }
   // Reset pointer to original value
   public void reset()
   {
      setPercent(originalValue);
   }
   // Can the pointer be modified
   public boolean isAvailable() 
   {
      return available;
   }
   public void setAvailable(boolean state) 
   {
      available = state;
   }
   // Is the pointer enabled
   public boolean isEnabled()
   {
      return (available && (percent >= MIN_PERCENT));
   }
   public Dimension getSize()
   {
      return new Dimension(WIDTH, HEIGHT);
   }
   // Percentage
   public void setPercent(int value) 
   {  
      if (available && writeAccess)
      {
         if ((value >= 0) && (value <= MAX_PERCENT) && 
             (percent != value))
         {
            percent = value;
            column = (int)(range * (double)value / 100.0) + HALF_WIDTH;
            // Set pointer display color
            if (percent >= MIN_PERCENT) color = enabledColor;
            else color = DISABLED_COLOR;
            // Threshold synchronization
            applet.notifyChange(percent, this);
         }
      }
   }
   public int getPercent() 
   {
      return percent;
   }
   public int calcPercent(int newX) 
   {
      double value = (double)(newX - HALF_WIDTH) / range;		
      return ((int)(value * 100));
   }
   // Location
   public Point getLocation() 
   {
      return (new Point(column, ROW));
   }
   public void moveTo(int newX) 
   {
      setPercent(calcPercent(newX));
   }
   // Is mouse cursor inside pointer
   public boolean isInside(int xPt, int yPt) 
   {
      // Calculate 1/2 triangle width at row
      int w = HALF_WIDTH - (int)((DELTA * (double)(yPt - ROW) + 0.5));
      return ((yPt >= ROW) && (yPt < (ROW + HEIGHT)) &&
              (xPt <= (column + w)) && (xPt >= (column - w)));
   }
   // Draw the pointer
   public void draw(Graphics g) 
   {
      if (available) 
      {
         // Initialize points for drawing
         int[] xPoints = new int[3];
         xPoints[0] = column - HALF_WIDTH;
         xPoints[1] = column + HALF_WIDTH;
         xPoints[2] = column;
         // Draw triangle
         g.setColor(color);
         g.fillPolygon(xPoints, yPoints, 3);
         g.setColor(Color.lightGray);
         g.drawPolygon(xPoints, yPoints, 3);
         // Put percent text in center of triangle if above minimum
         if (percent >= MIN_PERCENT)
         {
            g.setFont(font);
            g.setColor(textColor);
            String str = "" + getPercent();
            FontMetrics fm = g.getFontMetrics(font);
            g.drawString(str, column - fm.stringWidth(str) / 2, ROW + fm.getAscent());
         }
      }
   }
}
