/** 
 * PtrCanvas
 *   Canvas for drawing the bar graph and threshold pointers. Double-buffering
 *   is used for drawing to remove flicker when pointers are dragged.
 */
import java.awt.*;
import java.awt.event.*;
class PtrCanvas extends Canvas 
{
   private int barPercent;        // Percentage for bar graph 
   private Pointer warningPointer;		    
   private Pointer criticalPointer;		    
   private Pointer clickedPointer;
   private int pointerDelta;
   private Font font;
   private int xOffset;
   private Image imageBuffer;      // Offscreen image
   private Graphics imageGraphics;
   private Image backgroundImage;
   private Dimension pointerDim;
   private int barX;               // Percentage bar position and size 
   private int barY;
   private String percentText;
   private Color barBackgndColor;
   private Color barNormalColor;
   private Color barWarningColor;
   private Color barCriticalColor;
   private Color barTextLtColor;
   private Color barTextDkColor;
   // Constants
   public static final int FONT_SIZE = 14;
   // Constructs a canvas to draw bar graph and threshold pointers
   public PtrCanvas(int barValue, Pointer warningPtr, Pointer criticalPtr, 
                   int ptrDelta, Image background) 
   {
      clickedPointer = null;
      warningPointer = warningPtr;
      criticalPointer = criticalPtr;
      pointerDelta = ptrDelta;
      xOffset = 0;
      // Initialize bar graph
      pointerDim = warningPointer.getSize();
      barX = pointerDim.width / 2;
      barY = pointerDim.height;
      // Initialize percentage text
      if ((barValue < 0) || (barValue > 100)) 
      {
         barValue = 0;
         percentText = "Not Available";
         warningPointer.setAvailable(false);
         criticalPointer.setAvailable(false);
      }
      else  // Format percentage string
      {
         percentText = "" + barValue + "%";
      }
      barPercent = barValue;
      imageBuffer = null;
      imageGraphics = null;
      backgroundImage = background;
      font = new Font("Arial", Font.PLAIN, FONT_SIZE);
      addMouseListener(new Drag());
      addMouseMotionListener(new Move());
   }
   // Set bar graph colors
   public void setBarGraphColors(Color background, Color normal, Color warn, 
                                 Color panic, Color ltText, Color dkText)
   {
      barBackgndColor  = background;
      barNormalColor   = normal;
      barWarningColor  = warn;
      barCriticalColor = panic;
      barTextLtColor   = ltText;
      barTextDkColor   = dkText;
   }
   // Mouse adapter for draging pointers
   class Drag extends MouseAdapter 
   {
      public void mousePressed(MouseEvent e) 
      {
         warningPointer.setSelected(false);
         criticalPointer.setSelected(false);
         Point pt;
         Point mousePt = e.getPoint();
         
         // Did the user click inside a pointer
         if (criticalPointer.isInside(mousePt.x, mousePt.y)) 
         {
            clickedPointer = criticalPointer;
            pt = clickedPointer.getLocation();
            xOffset = pt.x - mousePt.x;
         }
         else if (warningPointer.isInside(mousePt.x, mousePt.y)) 
         {
            clickedPointer = warningPointer;
            pt = clickedPointer.getLocation();
            xOffset = pt.x - mousePt.x;
         }
         else clickedPointer = null;
         repaint();
      }
      public void mouseReleased(MouseEvent e) 
      {
         if (clickedPointer != null) 
         {
            clickedPointer.setSelected(true);
         }
         repaint();
      }
   }
   class Move extends MouseMotionAdapter 
   {
      public void mouseDragged(MouseEvent e) 
      {
         int newX = e.getX() + xOffset;
         // Warning cannot be greater than critical
         if (clickedPointer == warningPointer) 
         {
            if ((warningPointer.calcPercent(newX) + pointerDelta) <= 
                 criticalPointer.getPercent()) 
            {
               clickedPointer.moveTo(newX);
               repaint();
            }
         }
         // Critical cannot be less than warning
         if (clickedPointer == criticalPointer) 
         {
            if (criticalPointer.calcPercent(newX) >= 
               (warningPointer.getPercent() + pointerDelta)) 
            {
               clickedPointer.moveTo(newX);
               repaint();
            }
         }
      }
   }
   // Update screen
   public void update(Graphics g) 
   {
      paint(g);
   }
   // Load entire image before painting
   public boolean imageUpdate(Image image, int flags, int x, int y, int w, int h)
   {
      if ((flags & ALLBITS) != 0) repaint();
      return ((flags & (ALLBITS | ABORT)) == 0);
   }
   // Paints the pointers using double buffering
   public void paint(Graphics g) 
   {
      int imageWidth  = getSize().width;
      int imageHeight = getSize().height;
      if (imageBuffer == null) 
      {
         imageBuffer = createImage(imageWidth, imageHeight);
         if (imageGraphics != null) 
         {
            imageGraphics.dispose();
         }
         imageGraphics = imageBuffer.getGraphics();
      }
      // Draw background image
      imageGraphics.drawImage(backgroundImage, 0, 0, this);
      
      // Draw pointers
      warningPointer.draw(imageGraphics);
      criticalPointer.draw(imageGraphics);
      // Draw bar graph
      int barWidth  = imageWidth - pointerDim.width;
      int barHeight = imageHeight - pointerDim.height;
      int percentWidth = (int)((double)barPercent / 100.0 * (double)barWidth);
      // Set bar and text colors
      Color barColor = barNormalColor;
      Color barTextColor = barTextLtColor;
      if (criticalPointer.isEnabled() &&
          barPercent > criticalPointer.getPercent())
      {
         barColor = barCriticalColor;
      }
      else if (warningPointer.isEnabled() && 
               barPercent > warningPointer.getPercent())
      {
         barColor = barWarningColor;
         barTextColor = barTextDkColor;
      }
      // Draw the bar graph
      imageGraphics.setColor(barColor);
      imageGraphics.fillRect(barX, barY, percentWidth, barHeight);
      imageGraphics.setColor(barBackgndColor);
      imageGraphics.fillRect(barX + percentWidth, barY, 
                             barWidth - percentWidth, barHeight);
      // Draw percent text
      if (percentText.length() > 0)
      {
         imageGraphics.setFont(font);
         FontMetrics fm = imageGraphics.getFontMetrics(font);
         int a = fm.getAscent();
         int textX = percentWidth - fm.stringWidth(percentText) + barX;
         int textY = (barHeight - a) / 2 + a + barY;
         if (barPercent == 0) 
         {
            textX = barX + 5;
            barTextColor = barTextDkColor;
         }
         else if (barPercent < 20) 
         {
            textX = percentWidth + barX + 5;
            barTextColor = barTextDkColor;
         }
         imageGraphics.setColor(barTextColor);
         imageGraphics.drawString(percentText, textX, textY); 
      }
      // Display image
      g.drawImage(imageBuffer, 0, 0, this);
   }
}
