/**
 * Threshold
 *   This applet displays a bar graph for the disk space usage and two 
 *   pointers for setting warning and critical thresholds. 
 */
import java.awt.*;
import java.awt.event.*;
import java.applet.Applet;
public class Threshold extends Applet 
{
   private String diskIndex;         // Parameter values
   private int spaceUsed;            
   private int warningValue;
   private int criticalValue;
   private boolean modified;
   private boolean writeAccess;
   private Pointer warningPointer;	 // Pointer to set warning threshold
   private Pointer criticalPointer;  // Pointer to set critical threshold
   private Pointer selectedPointer;
   private PtrCanvas canvas;	       // Canvas that draws the pointers
   // Applet synchronization variables
   private static Threshold list, listTail;
   private Threshold next, prev;
   private static boolean concurrence;
   private static String indexes;
   private static String warnings;
   private static String criticals;
       
   // Constants
   public static final Color APP_BACKGROUND     = Color.white;
   public static final Color BAR_CRITICAL_COLOR = Color.red;
   public static final Color BAR_WARNING_COLOR  = Color.yellow;
   public static final Color BAR_NORMAL_COLOR   = Color.blue;
   public static final Color BAR_BACKGRND_COLOR = Color.lightGray;
   public static final Color BAR_TEXT_LT_COLOR  = Color.white;   
   public static final Color BAR_TEXT_DK_COLOR  = Color.black;   
   public static final int APP_WIDTH     = 348;
   public static final int APP_HEIGHT    =  50;
   public static final int POINTER_DELTA =   3;  // Minimum difference of pointers
   // Initializes the applet, creating pointers and laying out the panel
   public void init() 
   {
      // Get parameters
      diskIndex = getParameter("Index");
      spaceUsed = atoi(getParameter("SpaceUsed"));
      warningValue = atoi(getParameter("Warning"));
      criticalValue = atoi(getParameter("Critical"));
      String strAccess = getParameter("Access");
      writeAccess = strAccess.equals("true");
      // Threshold synchronization
      concurrence = false;
      //System.out.println("Space used: " + spaceUsed);
      
      // Create warning pointer
      warningPointer = new Pointer(warningValue, BAR_WARNING_COLOR, 
                                   Color.black, writeAccess, this);
      // Make sure critical pointer is greater than warning
      if (criticalValue < (warningPointer.getPercent() + POINTER_DELTA))
      {
         criticalValue = warningPointer.getPercent() + POINTER_DELTA;
      }
      criticalPointer = new Pointer(criticalValue, BAR_CRITICAL_COLOR, 
                                    Color.white, writeAccess, this);
      selectedPointer = null;
      // Create canvas for drawing pointers
      Image backgroundImage = getImage(getDocumentBase(), "backgrnd.jpg");
      canvas = new PtrCanvas(spaceUsed, warningPointer, criticalPointer, 
                             POINTER_DELTA, backgroundImage);
      canvas.setBackground(APP_BACKGROUND);
      canvas.setBarGraphColors(BAR_BACKGRND_COLOR, BAR_NORMAL_COLOR, BAR_WARNING_COLOR, 
                               BAR_CRITICAL_COLOR, BAR_TEXT_LT_COLOR, BAR_TEXT_DK_COLOR);
      setLayout(new BorderLayout());
      add("Center", canvas);	
      resize(APP_WIDTH, APP_HEIGHT);
      modified = false;
   }
   public void start() 
   {
      canvas.repaint();
      Threshold.addToList(this);
   }
   public void stop()
   {
      Threshold.removeFromList(this);
   }
   public Insets Insets() 
   {
      return new Insets(1,1,1,1);
   }
   // Return the disk index
   public String getDiskIndex()
   {
      return diskIndex;
   }
   // Convert a string to an integer
   public int atoi(String str) 
   {  
      try {  
         return Integer.parseInt(str.trim());
      }
      catch(NumberFormatException e) {  
         return 0; 
      }
   }
   // Reset thresholds to original values
   public synchronized void reset()
   {
      setConcurrent(false);
      Threshold current;
      for (current = list; current != null; current = current.next) 
      {
         current.warningPointer.reset();
         current.criticalPointer.reset();
         current.canvas.repaint();
      }
      modified = false;
   }
   // Have any of the thresholds been changed
   public synchronized boolean isModified()
   {
      if (! modified) 
      {
         Threshold current;
         for (current = list; current != null; current = current.next) 
         {
            if (current.warningPointer.isModified() ||
                current.criticalPointer.isModified())
               modified = true;
         }
      }
      return modified;
   }
   // Return the changes
   public synchronized String getIndexes()
   {
      indexes = "";
      Threshold current;
      if (isModified()) {
         // Set all disk to the same value
         if (concurrence) {
            indexes   = "-1";
            for (current = list; current != null; current = current.next) 
            {
               if (current.warningPointer.isAvailable())
               {
                  warnings  = "" + current.warningPointer.getPercent();
                  criticals = "" + current.criticalPointer.getPercent();
                  break;
               }
            }
         }
         else  // List modified indexes
         {
            boolean first = true;
   
            for (current = list; current != null; current = current.next) 
            {
               if (current.warningPointer.isModified() ||
                  current.criticalPointer.isModified()) 
               {
                  if (first)
                  {
                     indexes   = current.getDiskIndex();
                     warnings  = "" + current.warningPointer.getPercent();
                     criticals = "" + current.criticalPointer.getPercent();
                     first = false;
                  }
                  else  // Add a comma 
                  {
                     indexes   += "," + current.getDiskIndex();
                     warnings  += "," + current.warningPointer.getPercent();
                     criticals += "," + current.criticalPointer.getPercent();
                  }
               }
            }
         }
      }
      return indexes;
   }
   public synchronized String getWarnings()
   {
      // Always call getIndexes(); first
      return warnings;
   }
   public synchronized String getCriticals()
   {
      // Always call getIndexes(); first
      return criticals;
   }
   // Applet synchronization methods
   public synchronized void setConcurrent(boolean state)
   {
      concurrence = state;
 
      if (concurrence) 
      {
         // Set all threshold to highest value
         Threshold current;
         int warning,  maxWarning  = 0;
         int critical, maxCritical = 0;
         for (current = list; current != null; current = current.next) 
         {
            // Get maximum threshold values
            critical = current.criticalPointer.getPercent();
            if (critical > maxCritical) maxCritical = critical;
            warning = current.warningPointer.getPercent();
            if (warning > maxWarning) maxWarning = warning;
            if ((maxWarning + POINTER_DELTA) > maxCritical) 
               maxWarning = maxCritical - POINTER_DELTA;
         }
         // Now change all thresholds to the max values
         if (maxCritical != criticalPointer.getPercent())
            criticalPointer.setPercent(maxCritical);
         else notifyChange(maxCritical, criticalPointer);
         if (maxWarning != warningPointer.getPercent())
            warningPointer.setPercent(maxWarning);
         else notifyChange(maxWarning, warningPointer);
      }
   }
   public synchronized boolean isConcurrent()
   {
      return concurrence;
   }
   public static synchronized void addToList(Threshold node)
   {
      if (list == null) {
         list = listTail = node;
         node.next = node.prev = null;
      }
      else {
         node.prev = listTail;
         listTail.next = listTail = node;
         node.next = null;
      }
   }
   public static synchronized void removeFromList(Threshold node)
   {
      Threshold current = list;
      while (current != null && current != node) {
         current = current.next;
      }
      // We found it!
      if (current == node) {     
         if (list == current) {
            list = current.next;
         }
         if (listTail == current) {
            listTail = current.prev;
         }
         if (current.next != null) {
            current.next.prev = current.prev;
         }
         if (current.prev != null) {
            current.prev.next = current.next;
         }
         current.next = current.prev = null;
      }
   }
   public synchronized void notifyChange(int value, Pointer ptr)
   {
      // Called when a threshold changes
      if (! isConcurrent()) return;
      // Set all thresholds to the same value
      synchronized(ptr) {
         Threshold current;
         for (current = list; current != null; current = current.next) 
         {
            if (ptr == warningPointer) 
            {
               if ((value + POINTER_DELTA) <= 
                    current.criticalPointer.getPercent())
               {
                  current.warningPointer.setPercent(value);
                  current.canvas.repaint();
               }
            }
            else if (ptr == criticalPointer) 
            {
               if (value >= (current.warningPointer.getPercent() + 
                   POINTER_DELTA))
               {
                  current.criticalPointer.setPercent(value);
                  current.canvas.repaint();
               }
            }
         }  
      } 
   }
}  
